!+ Source module  "src_conv_tiedtke"
!------------------------------------------------------------------------------

MODULE src_conv_tiedtke

!------------------------------------------------------------------------------
!
! Description:
!   The module "src_conv_tiedtke" performs calculations related to the para-
!   meterization of subgrid-scale moist convection. The present parameteri-
!   zation scheme is a mass-flux scheme according to Tiedke (1989).
!
!   All global variables of the model that are used by the convection routines
!   are imported by USE statements below. The interface of the convection
!   routines and the model is provided by the organizational routine
!   "organize_conv_tiedtke"
!
!   The parameterization package has been provided by B. Ritter in a
!   Plug-compatible Fortran77-Version, which is based on an earlier version   
!   by M. Tiedtke (ECMWF). Some  modifications have been done for the 
!   F90 and the parallel Version:
!   Internal communication by common-blocks is replaced by module parameters,
!   scalars and arrays defined in this module.
!
! Current Code Owner: DWD, Dmitrii Mironov
!  phone:  +49  69  8062 2705
!  fax:    +49  69  8062 3721
!  email:  Dmitrii.Mironov@dwd.de
!
! History:
! Version    Date       Name
! ---------- ---------- ----
! 2.17       2002/05/08 Ulrich Schaettler
!  Initial release (Just a renaming from src_convection.f90 before)
! 2.19       2002/10/24 Jan-Peter Schulz
!  Diagnosis of near-surface gusts generated by convection
!  Correctec a bug in line 2063
! 3.2        2003/02/07 Ulrich Schaettler
!  Moved the communications to lmorg and organize_physics.
! 3.3        2003/04/03 Jan-Peter Schulz
!  Neglected downdraft windspeed (for testing)
! 3.4        2003/06/25 Erdmann Heise
!  Introduce some modifications to suppress 'convective drizzle'
! 3.5        2003/09/02 Ulrich Schaettler
!  Eliminated USE-statements not used any more
! 3.7        2004/02/18 Ulrich Schaettler
!  Store convective cloud water on clw_con
! 3.13       2004/12/03 Ulrich Schaettler
!  Get convective latent heating for LHN (Klaus Stephan)
!  Explicit formulation of relaxation for lateral boundaries (factor rmy)
!                                        (Jochen Foerstner)
! 3.15       2005/03/03 Ulrich Schaettler
!  Replaced FLOAT by REAL
! 3.16       2005/07/22 Erdmann Heise
!  Correction of zvddraf (maximal possible convective gust)
!  Introduction of lconv_inst, to choose whether instantaneous or min/max
!  values of top_con/bas_con are written.
! 3.18       2006/03/03 Klaus Stephan / Jochen Foerstner
!  LHN namelist parameter moved to data_lheat_nudge to avoid to many dependencies
!  Corrections for writing instantaneous values
! V4_4         2008/07/16 Dmitrii Mironov
!  Changes in the cumulus convection scheme related to the treatment of 
!  convective cloud condensate as a mixed water-ice phase  and of detrained 
!  convective cloud condensate
!  Eliminated timing variables which are unused (Ulrich Schaettler)
! V4_5         2008/09/10 Ulrich Schaettler
!  Moved declaration of entr_sc (before: entrscv) to new module data_convection
! V4_8         2009/02/16 Ulrich Schaettler, Guenther Zaengl
!  Only compute convective part vgust_con of maximal wind gust (Uli)
!  Use p0hl (reference pressure at half levels) for full consistency with
!  new reference atmosphere implementation (Guenther)
! V4_10        2009/09/11 Ulrich Schaettler
!  Define cu_evap as ALLOCATABLE field
! V4_12        2010/05/11 Ulrich Schaettler
!  Removed t0(_melt)
! V4_13        2010/05/11 Michael Gertz
!  Adaptions to SVN
! V4_18        2011/05/26 Ulrich Schaettler
!  Introduced conditional compilation for Nudging
!  for COSMO-ART: introduced additional fields for COSMO-ART
! V4_20        2011/08/31 Matthias Raschendorfer
!  Introducing calculation of convective buoyant TKE production 'tket_conv'
!  Using tke-field only, if 'lctke=T'.
! V4_25        2012/09/28 Anne Roches, Oliver Fuhrer
!  Replaced qx-variables by using them from the tracer module
!  Implemented some type conversions (to avoid special compiler warnings)
! V4_26        2012/12/06 Matthias Raschendorfer
!  Allowing only non negative buoyant production terms of TKE
! V4_27        2013/03/19 Astrid Kerkweg, Ulrich Schaettler
!  MESSy interface introduced: get input fields for CVTRANS and SCAV
! V4_28        2013/07/12 KIT, Ulrich Schaettler
!  Changes to adapt COSMO-ART to new tracer module: all dependencies to
!  COSMOART and POLLEN deleted, because this is now handled by the tracer module
!  Use variables for vertical grid from module vgrid_refatm_utils (US)
! V4_29        2013/10/04 Astrid Kerkweg, Ulrich Schaettler
!  Unification of MESSy interfaces and COSMO Tracer structure
!  For the COSMO-Model only use vcoord from vgrid_refatm_utils
!
! Code Description:
! Language: Fortran 90.
! Software Standards: "European Standards for Writing and
! Documenting Exchangeable Fortran 90 Code".
!==============================================================================
!
! Declarations:
!
! Modules used:

USE data_parameters, ONLY :   &
    ireals,    & ! KIND-type parameter for real variables
    iintegers    ! KIND-type parameter for standard integer variables

!------------------------------------------------------------------------------

USE data_convection, ONLY :   &
    entr_sc      ! mean entrainment rate for shallow convection

!------------------------------------------------------------------------------

USE data_modelconfig, ONLY :   &

! 2. horizontal and vertical sizes of the fields and related variables
! --------------------------------------------------------------------
    ie,           & ! number of grid points in zonal direction
    je,           & ! number of grid points in meridional direction
    ke,           & ! number of grid points in vertical direction
    ke1,          & ! ke + 1
    ieke,         & ! ie*ke

! 3. start- and end-indices for the computations in the horizontal layers
! -----------------------------------------------------------------------
!    These variables give the start- and the end-indices of the 
!    forecast for the prognostic variables in a horizontal layer.
!    Note, that the indices for the wind-speeds u and v differ from 
!    the other ones because of the use of the staggered Arakawa-C-grid.
!    
    istart   ,    & !
    iend     ,    & !
    jstart   ,    & !
    jend     ,    & !
    istartpar,    & ! start index for computations in the parallel program
    iendpar,      & ! end index for computations in the parallel program
    jstartpar,    & ! start index for computations in the parallel program
    jendpar,      & ! end index for computations in the parallel program


! 4. constants for the horizontal rotated grid and related variables
! ------------------------------------------------------------------
    edadlat,      & ! 1 / (radius of the earth * dlat)

! 5. variables for the time discretization and related variables
! --------------------------------------------------------------
    dt,           & ! long time-step
    dt2,          & ! dt*2.            

! 8. Organizational variables to handle the COSMO humidity tracers
! ----------------------------------------------------------------
    idt_qv

! end of data_modelconfig

!------------------------------------------------------------------------------

USE data_constants  , ONLY :   &

! 2. physical constants and related variables
! -------------------------------------------

    r_d,          & ! gas constant for dry air    
    r_v,          & ! gas constant of water vapour
    rdv,          & ! r_d / r_v
    rvd_m_o,      & ! r_v/r_d - 1  
    cp_d,         & ! specific heat of dry air at constant pressure
    lh_v,         & ! latent heat of vapourization
    lh_f,         & ! latent heat of fusion         
    lh_s,         & ! latent heat of sublimation    
    g,            & ! acceleration due to gravity

! 3. constants for parametrizations
! ---------------------------------
    b1,           & ! variables for computing the saturation vapour pressure
    b2w,          & ! over water (w) and ice (i)
    b2i,          & !               -- " --
    b3,           & !               -- " --
    b4w,          & !               -- " --
    b4i             !               -- " --

! end of data_constants

!------------------------------------------------------------------------------

USE data_fields     , ONLY :   &

! 1. constant fields for the reference atmosphere                     (unit)
! -----------------------------------------------
    p0         ,    & ! base state pressure                           (Pa) 
    p0hl       ,    & ! base state pressure on half levels            (Pa) 
    dp0        ,    & ! pressure thickness of layer                   (Pa) 
    hhl        ,    & ! geometrical height of half levels             ( m ) 

! 2. external parameter fields                                        (unit)
! ----------------------------
    llandmask  ,    & ! landpoint mask
    rmy        ,    & ! Davis-parameter for boundary relaxation         --


! 3. prognostic variables                                             (unit)
! -----------------------
    u          ,    & ! zonal wind speed                              ( m/s )
    v          ,    & ! meridional wind speed                         ( m/s )
    w          ,    & ! vertical velocity                             ( m/s )
    t          ,    & ! temperature                                   (  k  )
    pp         ,    & ! deviation from the reference pressure         ( pa  )
    tke        ,    & ! SQRT(2 * turbulent kinetic energy)            ( m/s )
                      ! (defined on half levels)

! 5. fields for surface values and soil model variables               (unit )
! -----------------------------------------------------
    ps        ,     & ! surface pressure                              ( pa  )
    t_g       ,     & ! weighted surface temperature                  (  K  )

! 6. fields that are computed in the parametrization and dynamics     (unit )
! ---------------------------------------------------------------
!   fields for convective subgrid-scale precipitation
    clc_con     ,   & ! cloud cover due to convection                   --
!_cdm All "clw" variables now contain the mixed-phase convective cloud condensate.
    clw_con     ,   & ! cloud liquid water due to convection            --
    prr_con     ,   & ! precipitation rate of rain, convective        (kg/m2*s)
    prs_con     ,   & ! precipitation rate of snow, convective        (kg/m2*s)
    prne_con    ,   & ! precipitation rate, no evaporat., convective  (kg/m2*s)
    bas_con     ,   & ! level index of convective cloud base            -- 
    top_con     ,   & ! level index of convective cloud base            --
    tt_conv     ,   & ! temperature tendency due to convection        ( K/s  )
    qvt_conv    ,   & ! humidity    tendency due to convection        ( 1/s  )
    qct_conv    ,   & ! qc-tendency tendency due to convection        ( 1/s  )
    qit_conv    ,   & ! qi-tendency tendency due to convection        ( 1/s  )
    ut_conv     ,   & ! u-tendency due to convection                  ( m/s^2)
    vt_conv     ,   & ! v-tendency due to convection                  ( m/s^2)
    tket_conv   ,   & ! TKE-tendency due to convective buoyancy       ( m2/s3 )
    dqvdt       ,   & ! threedimendional moisture convergence         (1/s)
    qvsflx      ,   & ! surface flux of water vapour                  (kg/m2*s)
    mflx_con    ,   & ! cloud base massflux                           (kg/m2*s)
    cape_con    ,   & ! convective available energy                   (   J/kg)
    qcvg_con    ,   & ! moisture convergence for Kuo-type closure     (    1/s)
    tke_con     ,   & ! convective turbulent energy                   (   J/kg)  !MR: not yet defined

! 7. fields for model output and diagnostics                          (unit )
! ------------------------------------------
    vgust_con         ! maximum convective gust at 10m                ( m/s )

! end of data_fields

!------------------------------------------------------------------------------

USE data_runcontrol , ONLY :   &

! 1. start and end of the forecast
! --------------------------------
    ntstep,       & ! actual time step
                    ! indices for permutation of three time levels
    nold,         & ! corresponds to ntstep - 1
    nnow,         & ! corresponds to ntstep
    ntke,         & ! time step for tke

! 3. controlling the physics
! --------------------------
    lconf_avg,    & ! average convective forcings in case of massflux closure
    lcape,        & ! convection with CAPE closure
    lctke,        & ! convection with turbulent convective energy closure (not
                    ! yet fully implemented, don't use lctke = .TRUE. !)
    lconv_inst,   & ! output of instantaneous values of top_con/bas_con
                    ! instead of min/max for an output interval

! 5. additional control variables
! -------------------------------
    lexpl_lbc,    & ! explicit formulation of relaxation for lateral boundaries
    ltime,        & ! detailed timings of the program are given
    l2tls,        & ! forecast with 2-TL integration scheme
    l_cosmo_art,  & ! enables gases and aerosols
    l_pollen        ! run Pollen component

! end of data_runcontrol 

!------------------------------------------------------------------------------

USE data_parallel , ONLY:    &
  num_compute,     & ! number of compute PEs
  my_cart_id,      & ! rank of this subdomain in the cartesian communicator
  my_cart_pos,     & ! position of this subdomain in the cartesian grid
                     ! in x- and y-direction
  my_cart_neigh,   & ! neighbors of this subdomain in the cartesian grid
  icomm_cart,      & ! communicator for the virtual cartesian topology
  iexch_req,       & ! stores the sends requests for the neighbor-exchange
                     ! that can be used by MPI_WAIT to identify the send
  imp_reals,       & ! determines the correct REAL type used in the model
                     ! for MPI
  nboundlines,     & ! number of boundary lines of the domain for which
                     ! no forecast is computed = overlapping boundary
                     ! lines of the subdomains
  sendbuf,         & ! sending buffer for boundary exchange:
                     ! 1-4 are used for sending, 5-8 are used for receiving
  isendbuflen        ! length of one column of sendbuf

!------------------------------------------------------------------------------

USE src_tracer,    ONLY : trcr_get, trcr_get_ntrcr, trcr_meta_get,            &
                          trcr_errorstr

!------------------------------------------------------------------------------

USE data_tracer,   ONLY : T_CONV_ID, T_CONV_ON

!------------------------------------------------------------------------------

USE environment,   ONLY : model_abort

!------------------------------------------------------------------------------

USE vgrid_refatm_utils, ONLY : vcoord

!------------------------------------------------------------------------------

#ifdef NUDGING
USE data_lheat_nudge, ONLY  :  &
     llhn           ! on/off switch for latent heat nudging (lhn)

!------------------------------------------------------------------------------

USE src_lheating , ONLY :  &
     get_gs_lheating            ! storage of grid scale latent heating for lhn
#endif

!------------------------------------------------------------------------------

#ifdef MESSY
! MESSY/BMIL
USE messy_main_data_bi, ONLY: massfu, massfd, u_entr, u_detr, d_entr, d_detr &
                            , cv_precflx, cv_snowflx, cv_rform, cv_sform     &
                            , cv_lwc, cv_iwc, cu_top, cu_bot, cv_precnew     &
                            , cv_snownew
! MESSY/SMCL
USE messy_main_timer,         ONLY: time_step_len
USE messy_main_constants_mem, ONLY: vtmpc2, cpd=>cp_air, gg => g
#endif

!==============================================================================

IMPLICIT NONE

!==============================================================================

! Declarations

! The following parameters are tunable constants for the cumulus convection
! scheme. 

LOGICAL, PRIVATE ::       &
  lmfmid        = .TRUE. ,&  ! switch for mid-level convection
  lmfdd         = .TRUE. ,&  ! switch for inclusion of downdrafts
  lmfdudv       = .TRUE.     ! switch for cumulus effects on momentum

REAL (KIND = ireals), PARAMETER, PRIVATE  :: &
  entrpen       = 0.00010, & ! mean entrainment rate for deep convection
  entrmid       = 0.00010, & ! mean entrainment rate for mid-level convection
  entrdd        = 0.00020, & ! mean entrainment rate for downdrafts
  entr_pb       = 1.E-04_ireals, & ! constant in extended formulation (DM)
                                   ! turbulent entrainment/detrainment rate

  cmfcmax       = 1.0    , & ! maximum mass flux
  cmfcmin       = 1.E-10 , & ! minimum mass flux  (security)
  cmfctop       = 0.33   , & ! relative mass flux above level of non-buoyancy
  cmfdeps       = 0.3    , & ! relative mass flux at level of free-sinking

  capeconstant  = 0.05   , & ! time-constant cu_base massflux from CAPE
  capemin       = 1.0    , & ! minimum CAPE for cu_base massflux determination
  cmbtke        = 1.0    , & ! time-constant cu_base massflux from ctke
  ctkemin       = 0.05   , & ! mimimum ctke for cu_base massflux determination

  cprcon        = 0.0002     ! conversion rate cloud water to precipitation

!_cdm Declare the following parameters and variables here, not inside SUBROUTINE cu_cond,
!     as they are used in several procedures of T89.
! Set Tmpmin, Tmpmax and exp_mp as in the ECMWF IFS (CY31r1).
REAL (KIND=ireals), PARAMETER, PRIVATE :: &
  Tmpmin = 250.16_ireals       , & ! Minium temperature of the mixed-phase temperature range [K]
  Tmpmax = 273.16_ireals       , & ! Maximum temperature of the mixed-phase temperature range [K]
  exp_mp = 2._ireals               ! Exponent in the interpolation formula
                                   ! for the mixed-phase water fraction [-]
REAL (KIND=ireals) :: &
!_cdm Introduce hl_mp
  lh_mp             , & ! Effective latent heat of evaporation/sublimation
                        ! for the water-ice mixed phase [J kg^{-1}]
  fr_wat            , & ! Water fraction for the water-ice mixed phase [-]
  qs_w              , & ! Saturation specific humidity over water [-]
  qs_i              , & ! Saturation specific humidity over ice [-]
  qs_m              , & ! Saturation specific humidity for mixed phase
  qdvdt_w           , & ! First derivative of the saturation specific humidity over water
                        ! with respect to temperature [K^{-1}]
  qdvdt_i           , & ! First derivative of the saturation specific humidity over ice
                        ! with respect to temperature [K^{-1}]
  qdvdt_m               ! First derivative of the saturation specific humidity
                        ! with respect to temperature for the mixed phase [K^{-1}]

REAL (KIND = ireals), PRIVATE  :: &
  cu_frr                     ! fraction of grid area covered with precipitation

!REAL (KIND = ireals), PRIVATE  :: &
!  cu_evap  (60)              ! factor for evaporation of rain

REAL (KIND = ireals), ALLOCATABLE, PRIVATE  :: &
  cu_evap  (:)               ! factor for evaporation of rain

REAL (KIND = ireals), PRIVATE  :: &
    ctmelt               , & ! tripel point
    cc2                  , & ! definition of utitility constants
    c5hlccp              , & ! for saturation humidity
    c5hlscp              , & !
    chlcdcp              , & !
    chlsdcp                  !

! The logical variable LFIRSTI is used to initialize the parameter cu_frr and
! cu_evap on the first CALL of Subroutine organize_conv_tiedtke
LOGICAL, PRIVATE ::       &
  lfirsti = .TRUE.           ! switch for initialization

#ifdef MESSY
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: massfu_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: massfd_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: uentr_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: udetr_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: dentr_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: ddetr_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cvprecflx_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cvsnowflx_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cvlwc_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cviwc_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cvrform_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cvsform_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cvprecnew_2d
REAL(KIND=ireals), POINTER, DIMENSION(:,:) :: cvsnownew_2d

INTEGER (KIND=iintegers), SAVE :: idx_qv = -99
#endif

!==============================================================================
! Module procedures in src_conv_tiedtke
!==============================================================================

CONTAINS

!==============================================================================
!+ Module procedure for organization
!------------------------------------------------------------------------------

SUBROUTINE organize_conv_tiedtke

!------------------------------------------------------------------------------
!
! Description:
!
!   The module procedure organize_conv_tiedtke is the interface of the model
!   to the parameterization package for moist convection.
!   At present, only one parameterization scheme, the Tiedtke mass-flux scheme,
!   is available.
!
! Externals
!   cu_tied  : Tiedtke mass-flux scheme             
!
!------------------------------------------------------------------------------

! Local scalars and automatic arrays (also to be used in lower level routines):
! -----------------------------------------------------------------------------

  ! Input for the convection routine "cu_tied"
  ! -----------------------------------------
  REAL    (KIND=ireals   )  ::  &
     zt      (ie,ke),    & ! temperature at full levels
     zqv     (ie,ke),    & ! specific humidiy at full levels
     zu      (ie,ke),    & ! zonal wind component
     zv      (ie,ke),    & ! meridional wind component
     zw      (ie,ke),    & ! vertical velocity in z-system
     zfif    (ie,ke  ),  & ! geopotential at full levels
     zfih    (ie,ke1),   & ! geopotential at half levels
     zpf     (ie,ke  ),  & ! pressure at full levels 
     zph     (ie,ke1),   & ! pressure at half levels 
     zt_g    (ie    ),   & ! surface temperature
     zdqvdt  (ie,ke),    & ! moisture tencency
     zqhfl   (ie  ),     & ! surface moisture flux    
     zcumflx (ie),       & ! cu_base massflux
     ztke    (ie,ke),    & ! turbulent kinetic energy
     zcucape (ie),       & ! convective available energy
     zcuconv (ie),       & ! moisture convergence to be used in cu_base massflux
     zdt                   ! actual timestep for 2 or 3 TL-scheme

  REAL (KIND=ireals), ALLOCATABLE ::  &
     ztrcr(:,:,:)         ! tracers at full levels

  LOGICAL   ::  &
     lolp (ie)             ! land-sea indicator

  ! Output from the convection routine "cu_tied"
  ! -----------------------------------------
  REAL    (KIND=ireals   ) ::  &
     zcutke   (ie),     & ! turbulent convective energy !MR: not defined anywhere
     zvddraf  (ie),     & ! maximum possible convective gust
     zmyfac   (ie),     & ! Davies-Relaxation factor
     zdt_con  (ie,ke),  & ! convective tendency of temperature
     zdqv_con (ie,ke),  & ! convective tendency of specific humidity
     zdtke_con(ie,ke),  & ! convective buoyant TKE production on half levels
!cdm Cloud-water and cloud-ice tendencies due to detrained convective cloud condensate,
!    local arrays.
     zqct_con (ie,ke),  & ! convective tendency of cloud water
     zqit_con (ie,ke),  & ! convective tendency of cloud ice
     zdu_con  (ie,ke),  & ! convective tendency of u                 
     zdv_con  (ie,ke),  & ! convective tendency of v                 
     zco_clw  (ie,ke),  & ! convective cloud water
     zpr_con  (ie),     & ! convective precipitation rate of rain
     zps_con  (ie),     & ! convective precipitation rate of snow
     zpre_con (ie)        ! convective precipitation rate without evaporation

  REAL (KIND=ireals), ALLOCATABLE ::  &
     ztrcr_con(:,:,:)     ! convective tendency of tracers

  REAL (KIND=ireals) ::  &
     zdtlhn_con(ie,ke), & ! conv T-tend. due to LH exchanges, without rain evap.
     ttlh_conv(ie,je,ke)

  INTEGER (KIND=iintegers) ::  &
     mbas_con (ie)      , & ! cloud base level index
     mtop_con (ie)          ! cloud top  level index
  LOGICAL                  ::  &
     locum (ie)             ! indicator for convection at gridpoints
  
  ! Other local arrays and scalars  
  REAL    (KIND=ireals   ) ::  &
     zrealdiff,             & !
     zbas, ztop,            &
     zdttop, zdtbot, zdepth_max, zdepth_now

 REAL    (KIND=ireals   ) ::  &  ! Weights for horizontal averaging
     zcent = 0.2500_ireals, & ! centre weight in a nine point stencil
     zside = 0.1250_ireals, & ! weight for side points 
     zedge = 0.0625_ireals    ! weight for edge points

  INTEGER (KIND=iintegers) ::  &
     i, j, k, km1, nx,      & !
     mtop, mbas,            & !
     iztrcr, nztrcr_con,    & !
     isp                      ! number of species (COSMO_ART)

  INTEGER  (KIND=iintegers), ALLOCATABLE :: &
    iztrcr_con(:)             ! Index list of tracers which undergo passive
                              ! transport by convection

  REAL (KIND=ireals), POINTER :: &
    ztrcr_nx(:,:,:)  => NULL(),     &  ! tracer variable at tlev=nx
    ztrcr_tens(:,:,:)=> NULL(),     &  ! tracer tendency
    qv(:,:,:)        => NULL()         ! QV at nx

! For error handling
! ------------------
  INTEGER (KIND=iintegers) ::  &
    izerror,                & !
    izstata                   ! error status at allocation

  CHARACTER (LEN=80)       ::  &
    yzerrmsg

  CHARACTER (LEN=25)       :: yzroutine
!
! End of header
!==============================================================================

  izerror  = 0
  yzerrmsg = '   '
  yzroutine= 'organize_conv_tiedtke'

  ! Select timelevel and timestep of the computation
  IF ( l2tls ) THEN
    nx  = nnow
    zdt = dt
  ELSE
    nx  = nold
    zdt = dt2
  ENDIF

  ! Retrieve the required microphysics tracers
  CALL trcr_get(izerror, idt_qv, ptr_tlev = nx, ptr = qv)
  IF (izerror /= 0) THEN
    yzerrmsg = trcr_errorstr(izerror)
    CALL model_abort(my_cart_id, izerror, yzerrmsg, yzroutine)
  ENDIF

#ifndef MESSY
  ! Get the value (ON/OFF) of the convection metadata for all tracers
  ALLOCATE (iztrcr_con(trcr_get_ntrcr()), STAT=izerror)
  iztrcr_con = 0_iintegers

  CALL trcr_meta_get(izerror, T_CONV_ID, iztrcr_con)
  IF (izerror /= 0_iintegers) THEN
    yzerrmsg = trcr_errorstr(izerror)
    CALL model_abort(my_cart_id, izerror, yzerrmsg, yzroutine)
  ENDIF

  ! Find the tracers which should be transported by convection
  nztrcr_con = 0_iintegers
  DO iztrcr = 1, trcr_get_ntrcr() ! loop over tracers
    IF (iztrcr_con(iztrcr) == T_CONV_ON) THEN
      nztrcr_con = nztrcr_con + 1
      iztrcr_con(nztrcr_con) = iztrcr
    ENDIF
  ENDDO

  ! Allocate local structure containing the convective tendency
  ! of the tracers
  ALLOCATE( ztrcr_con(ie, ke, nztrcr_con), STAT=izerror)
  ztrcr_con = 0.0_ireals
  ALLOCATE( ztrcr    (ie, ke, nztrcr_con), STAT=izerror)
  ztrcr = 0.0_ireals
#else
  nztrcr_con = 1
#endif

  ! Some initializations
  IF (lfirsti) THEN
    ctmelt  = 273.16                   ! tripel point
    cc2     = b1 * rdv                 ! definition of utitility constants
    c5hlccp = b2w*(b3-b4w)*lh_v/cp_d   ! for saturation humidity
    c5hlscp = b2i*(b3-b4i)*lh_s/cp_d   !
    chlcdcp = lh_v/cp_d                !
    chlsdcp = lh_s/cp_d                !

    cu_frr = 2.0 / SQRT ( 0.001/edadlat )
    cu_frr = MIN ( cu_frr, 1.0_ireals )

    ALLOCATE (cu_evap(ke), STAT=izerror)
    cu_evap(:) = 0.0
    DO  k = 1, ke  
      cu_evap(k)   = 1.93E-6_ireals * 261.0_ireals * SQRT( 1000.0_ireals/   &
            (38.3_ireals*0.293_ireals)* SQRT(vcoord%sigm_coord(k)) )*0.5_ireals / g 
    ENDDO
    lfirsti = .FALSE.
  ENDIF

  ! In order to save come CPU-time, the convection scheme in a former version
  ! of the model has been called only for those points, which have been 
  ! identified as convecively unstable in the previous time step. The present
  ! version calls the convection routine only at fixed time increments nincconv
  ! (e.g. every 10 timsteps) and stores the concective tendencies and the 
  ! precipitation rates on global arrays which are held fixed in the 
  ! intermediate steps.  The time increment nincconv can be set on NAMELIST  
  ! input.

  ! Reset the convective tendencies, the precipitation rates and the 
  ! convective cloud cover each time when the convection scheme is called
! tt_conv (:,:,:) = 0.0_ireals
! qvt_conv(:,:,:) = 0.0_ireals
! ut_conv (:,:,:) = 0.0_ireals
! vt_conv (:,:,:) = 0.0_ireals
  clc_con (:,:,:) = 0.0_ireals
! prr_con (:,:  ) = 0.0_ireals
! prs_con (:,:  ) = 0.0_ireals
! prne_con(:,:  ) = 0.0_ireals

#ifdef NUDGING
  IF (llhn) ttlh_conv  (:,:,:) = 0.0_ireals
#endif

#ifdef MESSY
  ! PRESET FIELDS FOR CVTRANS INPUT
  massfu(:,:,:) = 0._ireals
  massfd(:,:,:) = 0._ireals
  u_entr(:,:,:) = 0._ireals
  d_entr(:,:,:) = 0._ireals
  u_detr(:,:,:) = 0._ireals
  d_detr(:,:,:) = 0._ireals
  cv_precflx(:,:,:) = 0._ireals
  cv_snowflx(:,:,:) = 0._ireals
  cv_lwc(:,:,:)     = 0._ireals
  cv_iwc(:,:,:)     = 0._ireals
  cv_rform(:,:,:)   = 0._ireals
  cv_sform(:,:,:)   = 0._ireals
  cu_top(:,:)       = 0.0_ireals
  cu_bot(:,:)       = 0.0_ireals
#endif

  ! Preset the output arrays for the results of the mass-flux scheme
! mflx_con(:,:) = 0.0_ireals
! cape_con(:,:) = 0.0_ireals
! qcvg_con(:,:) = 0.0_ireals
! tke_con (:,:) = 0.0_ireals
 
  ! In the present version of the model, the vertical layer index of the
  ! convective cloud base and cloud top are stored as hourly maximum
  ! values and after each hour of integration these fields are reset
  ! to zero in routine "near_surface".
  ! the arrays top_con and bas_con are reset in routine "near_surface"
  ! or here for instantaneous values.
  IF ( lconv_inst ) THEN
    bas_con (:,:) = 0.0_ireals
    top_con (:,:) = 0.0_ireals
  ENDIF

  !--------------------------------------------------------------
  ! Loop from south to north. The convection scheme is called for
  ! eache j-slice.
  !--------------------------------------------------------------

  DO  j = jstart, jend

!US  is this really necessary?
!US  in cu_tied, the highest model level is NOT set for these variables!
!US    ! Preset the output arrays of the convection scheme
!   zdt_con (:,1) = 0.0_ireals
!   zdqv_con(:,1) = 0.0_ireals
!   zdtke_con(:,1) = 0.0_ireals
!dm Cloud-water and cloud-ice tendencies due to detrained convective cloud condensate,
!   local arrays.
!   zqct_con(:,1) = 0.0_ireals
!   zqit_con(:,1) = 0.0_ireals
!   zdu_con (:,1) = 0.0_ireals
!   zdv_con (:,1) = 0.0_ireals
!   zco_clw (:,1) = 0.0_ireals
!   zpr_con (:)   = 0.0_ireals
!   zps_con (:)   = 0.0_ireals
!   zpre_con(:)   = 0.0_ireals
!   mbas_con(:)   = 0
!   mtop_con(:)   = 0
!   locum   (:)   = .FALSE.

    zdtlhn_con (:,:) = 0.0_ireals

    ! Preset the additional output fields for convection
    zcumflx(:) = 0.0_ireals
    zvddraf(:) = 0.0_ireals
    zcucape(:) = 0.0_ireals
    zcuconv(:) = 0.0_ireals
    zcutke (:) = 0.0_ireals

    ! Prepare the input arrays for the convection scheme
    IF ( lexpl_lbc ) THEN
      DO i = istart, iend
        zmyfac(i) = 1.0_ireals - rmy(i,j,1)
      ENDDO
    ELSE
      DO i = istart, iend
        zmyfac(i) = 1.0_ireals - rmy(i,j,1)/( 1.0_ireals + rmy(i,j,1) )
      ENDDO
    ENDIF

    DO  k = 1, ke 
      km1 = MAX ( 1, k-1 )
      DO i = istart, iend
        zt    (i,k)  = t (i,j,k,nx)
        zqv   (i,k)  = qv(i,j,k   )
        zu    (i,k)  = 0.5_ireals*( u(i,j,k,nx) + u(i-1,j,k,nx) )
        zv    (i,k)  = 0.5_ireals*( v(i,j,k,nx) + v(i,j-1,k,nx) )
        zw    (i,k)  = 0.5_ireals*( w(i,j,k,nx) + w(i,j,k+1,nx) )
        zfif  (i,k)  = 0.5_ireals*g*( hhl(i,j,k) + hhl(i,j,k+1) )
        zfih  (i,k)  = g* hhl(i,j,k) 
        zdqvdt(i,k)  = dqvdt(i,j,k)*zmyfac(i)
        zpf   (i,k)  = p0(i,j,k) + pp(i,j,k,nx)
        zph   (i,k)  = p0hl(i,j,k) + 0.5_ireals*(pp(i,j,k,nx) + pp(i,j,km1,nx))
      ENDDO
    ENDDO

    IF (lctke) THEN !using TKE closure
      DO  k = 1, ke
        DO i = istart, iend
          ztke   (i,k) = tke     (i,j,k,ntke)
        ENDDO
      ENDDO
    ENDIF

#ifndef MESSY
    ! loop over tracers (which undergo convective transport)
    DO iztrcr = 1, nztrcr_con

      ! get pointer to tracer (at timelevel nx)
      CALL trcr_get(izerror, iztrcr_con(iztrcr), ptr_tlev=nx, ptr=ztrcr_nx)
      IF (izerror /= 0_iintegers) THEN
        yzerrmsg = trcr_errorstr(izerror)
        CALL model_abort(my_cart_id, izerror, yzerrmsg, yzroutine)
      ENDIF

      ! copy out block
      DO  k = 1, ke
        DO i = istart, iend
          ztrcr(i,k,iztrcr) = ztrcr_nx(i,j,k)
        ENDDO
      ENDDO

    ENDDO
#endif

    DO i = istart, iend
      zfih  (i,ke1)  = g* hhl(i,j,ke1) 
      zph   (i,ke1)  = ps(i,j,nx)
      zt_g  (i)      = t_g  (i,j,nx)
      lolp  (i)      = llandmask (i,j)
      zqhfl (i)      = qvsflx(i,j)*zmyfac(i)
    ENDDO     

    IF (lconf_avg) THEN ! Replace convective forcings by area average
      DO  k = 1, ke
        km1 = MAX ( 1, k-1 )
        DO i = istart, iend
          zw    (i,k)  = 0.5_ireals * &
                         (  zcent*( w(i,j,k,nx) + w(i,j,k+1,nx)  )           &
                          + zside*( w(i-1,j  ,k  ,nx) + w(i+1,j  ,k  ,nx)    &
                                  + w(i,j-1  ,k  ,nx) + w(i  ,j+1,k  ,nx)  ) &
                          + zedge*( w(i-1,j-1,k  ,nx) + w(i+1,j-1,k  ,nx)    &
                                  + w(i-1,j+1,k  ,nx) + w(i+1,j+1,k  ,nx)  ) &
                          + zside*( w(i-1,j  ,k+1,nx) + w(i+1,j  ,k+1,nx)    &
                                  + w(i  ,j-1,k+1,nx) + w(i  ,j+1,k+1,nx)  ) &
                          + zedge*( w(i-1,j-1,k+1,nx) + w(i+1,j-1,k+1,nx)    &
                                  + w(i-1,j+1,k+1,nx) + w(i+1,j+1,k+1,nx)  ) )
          zdqvdt(i,k)  = zmyfac(i) * ( zcent*dqvdt(i,j,k)                  &
                          + zside*( dqvdt(i-1,j,k  ) + dqvdt(i+1,j,k  )    &
                                  + dqvdt(i,j-1,k  ) + dqvdt(i,j+1,k  )  ) &
                          + zedge*( dqvdt(i-1,j-1,k) + dqvdt(i+1,j-1,k)    &
                                  + dqvdt(i-1,j+1,k) + dqvdt(i+1,j+1,k)  ) )
        ENDDO
      ENDDO
      DO i = istart, iend
        zqhfl (i)      = zmyfac(i) * ( zcent*qvsflx(i,j)                 & 
                          + zside*( qvsflx(i-1,j  ) + qvsflx(i+1,j  )    &
                                  + qvsflx(i,j-1  ) + qvsflx(i,j+1  )  ) &
                          + zedge*( qvsflx(i-1,j-1) + qvsflx(i+1,j-1)    &
                                  + qvsflx(i-1,j+1) + qvsflx(i+1,j+1)  ) )
      ENDDO
    ENDIF
 
#ifdef MESSY
    ! set here 2d Pointer to 2d field as the following subroutine(s) are
    ! only 2D
    massfu_2d => massfu(:,j,:)
    massfd_2d => massfd(:,j,:)
    uentr_2d  => u_entr(:,j,:)
    udetr_2d  => u_detr(:,j,:)
    dentr_2d  => d_entr(:,j,:)
    ddetr_2d  => d_detr(:,j,:)
    cvprecflx_2d => cv_precflx(:,j,:)
    cvsnowflx_2d => cv_snowflx(:,j,:)
    cvlwc_2d     => cv_lwc(:,j,:)
    cviwc_2d     => cv_iwc(:,j,:)
    cvrform_2d   => cv_rform(:,j,:)
    cvsform_2d   => cv_sform(:,j,:)
    cvprecnew_2d => cv_precnew(:,j,:)
    cvsnownew_2d => cv_snownew(:,j,:)
#endif

    ! call to cu_tied with new output argument zdtlhn_con (conv T-tend. due
    ! to LH exchanges, without rain evap.)
    ! Call to cu_tied with new arguments zqct_con, zqit_con: cloud-water and
    ! cloud-ice tendencies due to detrained convective cloud condensate. (DM)
    CALL cu_tied (                                             &
         zt     , zqv     , ztrcr   , zu    , zv    , zw    ,  &
         zfif   , zfih    , zpf     , zph   ,                  &
         zt_g   , zdqvdt  , zqhfl   , lolp  , zdt   ,          &
         ie     , ke      , istart  , iend  , nztrcr_con    ,  &
         zcumflx, zcucape , zcuconv , zcutke, ztke  ,          &
         zvddraf, zco_clw ,                                    &
         zdt_con, zdtlhn_con, zdqv_con, ztrcr_con   , zdu_con ,&
         zdv_con, zqct_con, zqit_con, zdtke_con,               &
         zpr_con, zps_con , mbas_con, mtop_con, zpre_con, locum)

    ! Store the output from the convection scheme on the corresponding
    ! global arrays
 
    DO  k = 1, ke
      DO  i = istart, iend
        IF( locum(i) ) THEN
          tt_conv  (i,j,k) = zdt_con  (i,k)
          qvt_conv (i,j,k) = zdqv_con (i,k)
          qct_conv (i,j,k) = zqct_con (i,k)
          qit_conv (i,j,k) = zqit_con (i,k)
          ut_conv  (i,j,k) = zdu_con  (i,k)
          vt_conv  (i,j,k) = zdv_con  (i,k)
          tket_conv(i,j,k) = zdtke_con(i,k)
          clw_con  (i,j,k) = zco_clw  (i,k)
        ELSE
          tt_conv  (i,j,k) = 0.0_ireals
          qvt_conv (i,j,k) = 0.0_ireals
          qct_conv (i,j,k) = 0.0_ireals
          qit_conv (i,j,k) = 0.0_ireals
          ut_conv  (i,j,k) = 0.0_ireals
          vt_conv  (i,j,k) = 0.0_ireals
          tket_conv(i,j,k) = 0.0_ireals
          clw_con  (i,j,k) = 0.0_ireals
        ENDIF
      ENDDO     
    ENDDO     

#ifndef MESSY
    ! loop over tracers (which undergo convective transport)
    DO iztrcr = 1, nztrcr_con

      ! get pointer to tracer tendency
      CALL trcr_get(izerror, iztrcr_con(iztrcr), ptr_tens=ztrcr_tens)
      IF (izerror /= 0_iintegers) THEN
        yzerrmsg = trcr_errorstr(izerror)
        CALL model_abort(my_cart_id, izerror, yzerrmsg, yzroutine)
      ENDIF

      ! copy back block of tendency
      DO  k = 1, ke
        DO i = istart, iend
          IF (locum(i)) THEN
            ztrcr_tens (i,j,k) = ztrcr_tens (i,j,k) + ztrcr_con (i,k,iztrcr)
          ENDIF
        ENDDO
      ENDDO

    ENDDO
#endif

#ifdef NUDGING
    ! Store conv T-tend. due to LH exchanges, without rain evap.
    IF (llhn) THEN
      DO  k = 1, ke
       DO  i = istart, iend
        IF( locum(i) ) ttlh_conv  (i,j,k) = zdtlhn_con (i,k)
       ENDDO
      ENDDO
    ENDIF
#endif

    DO  i = istart, iend
      IF( locum(i) .AND. mtop_con(i) > 0 ) THEN
        mtop_con (i) = MAX   ( mtop_con(i)-2, 2 )
        prr_con(i,j) = zpr_con(i)
        prs_con(i,j) = zps_con(i)
        prne_con(i,j)= zpre_con(i)
        zdepth_max   = bas_con(i,j) - top_con(i,j)
        zdepth_now   = REAL ( mbas_con(i) - mtop_con(i), ireals )
        IF ( zdepth_max < zdepth_now ) THEN
           top_con(i,j) = REAL  ( mtop_con(i), ireals )
           bas_con(i,j) = REAL  ( mbas_con(i), ireals )
        ENDIF
      ELSE
        prr_con(i,j) = 0.0_ireals
        prs_con(i,j) = 0.0_ireals
        prne_con(i,j)= 0.0_ireals
      ENDIF
    ENDDO     

    ! Calculate the convective cloud cover by a simple empirical
    ! relation (following B.Ritter, FE14). Anvils are assumed for
    ! a temperature increase at top level
    DO  i = istart, iend
      IF( locum(i) .AND. mtop_con(i) > 0 ) THEN
        mtop = mtop_con(i)
        mbas = mbas_con(i)
#ifdef MESSY
        cu_top(i,j) = mtop_con(i)
        cu_bot(i,j) = mbas_con(i)
#endif
        zbas = 0.5_ireals*( hhl(i,j,mbas) + hhl(i,j,mbas+1) )
        ztop = 0.5_ireals*( hhl(i,j,mtop) + hhl(i,j,mtop+1) )
        DO  k = mtop, mbas-1
          clc_con(i,j,k) = 0.35_ireals*(ztop-zbas)/5000.0_ireals 
          IF ( k == mtop ) THEN
            zdtbot = t(i,j,k+1,nx) - t(i,j,k  ,nx)
            zdttop = t(i,j,k  ,nx) - t(i,j,k-1,nx)
            IF ( zdtbot > 0.0_ireals .AND. zdttop <= 0.0_ireals ) THEN
              clc_con(i,j,k) = 2.0_ireals*clc_con(i,j,k)
            ENDIF
          ENDIF
          clc_con(i,j,k) = MIN ( 1.0_ireals, MAX(0.05_ireals, clc_con(i,j,k)) )
        ENDDO        
      ENDIF
    ENDDO        
 
    DO i = istart, iend
      IF (locum(i)) THEN
        tke_con (i,j) = MAX(0.0_ireals,zcutke(i))
        mflx_con(i,j) = zcumflx(i)
        ! correction of zvddraf (max. possible convective gust)
        IF (3600.0_ireals*(zpr_con(i)+zps_con(i)) <= 0.015) zvddraf(i)=0.0
        vgust_con(i,j)= MAX(vgust_con(i,j),zvddraf(i))
        cape_con(i,j) = zcucape(i)
        qcvg_con(i,j) = zcuconv(i)
      ELSE
        tke_con (i,j) = 0.0
        mflx_con(i,j) = 0.0
        cape_con(i,j) = 0.0
        qcvg_con(i,j) = 0.0
      END IF
    END DO

  !--------------------------------------------------------------
  ! End of the loop from south to north. 
  !--------------------------------------------------------------

  ENDDO       

#ifdef NUDGING
  IF (llhn) THEN
    CALL get_gs_lheating('dir',1,ke,ttlh_conv*zdt)
  ENDIF
#endif

!------------------------------------------------------------------------------
! End of the subroutine
!------------------------------------------------------------------------------

END SUBROUTINE organize_conv_tiedtke

!==============================================================================

!+ Module procedure in "Convection" 
!------------------------------------------------------------------------------

! cu_tied with new output argument zdtlhn_con (conv T-tend. due
! to LH exchanges, without rain evap.)
! cu_tied with new arguments for cloud-water and cloud-ice tendencies due to 
! detrained convective cloud condensate (DM)
SUBROUTINE cu_tied (                                             &
           pt     , pqv     , ptrcr   , pu    , pv    , pw    ,  &
           pfif   , pfih    , ppf     , pph   ,                  &
           pt_g   , pdqvdt  , pqhfl   , lolp  , pdt2  ,          &
           idim   , kdim    , isc     , iec   , nztr  ,          &
           pcumflx, pcucape , pcuconv , pcuctke,ptke  ,          &
           pvddraf, pco_clw ,                                    &
           pdt_con, pdtlhn_con, pdqv_con, ptrcr_con   , pdu_con ,&
           pdv_con, pqct_con, pqit_con, pdtke_con,               &
           ppr_con, pps_con , mbas_con, mtop_con, ppne_con, locum)

!------------------------------------------------------------------------------
!
! Description:
!
!   The module procedure cu_tied organizes the massflux cumulus parameter-
!   ization scheme according to M. Tiedke.
!   This routine computes the physical tendencies of the prognostic variables 
!   t, qv, u and v due to convective processes and also cgas, caero and cpollen.
!   These are
!     - convective fluxes due to updrafts and saturated downdrafts
!     - formation of precipitation
!     - evaporation of rain below cloud base
!
!   Input for the scheme are the grid scale values of T, qv, u, v, w, p, fi 
!   and the moisture tendency due to turbulent mixing and 3-d advection 
! 
!   Output of the scheme are 
!     - tendencies of T, qv, u and v 
!     - convective precipitation rates of rain and snow 
!     - cloud base and cloud top model level indices
!
!
! Method:
!
!   The parameterisation is based on a mass flux representation of
!   convective processes and proceeds along the following steps:
!
!     - definition of constants and parameters
!     - specification of half level values and initialization of
!       updraft and downdraft values 
!     - determination of cloud base (if existing) 
!       and specification of cloud base mass flux from PBL moisture
!       budget (lcape = lctke = .false.), or from convective available
!       energy (lcape = .true., lctke = .false.), or from vertically
!       integrated turbulent kinetic energy (lcape = .false., lctke = .true.).
!     - cloud ascent calculations in *cu_asc* in absence of downdrafts
!     - downdraft calculations:
!                        a) values at level of free sinking 
!                        b) determination of moist descent in  *cu_ddraf*
!                        c) recalculation of cloud base mass flux
!                           including the effect of cumulus downdrafts
!     - final cloud ascent in *cu_asc*
!     - final adjustments to convective fluxes  and evaporation
!       in sub-cloud layer
!     - calculation of surface rain/snow and atmospheric tendencies of T and qv 
!     - calculation of atmospheric tendencies of u and v 
!
! Externals
!
!   cu_asc  : CLOUD ASCENT FOR ENTRAINING PLUME
!   cu_ddraf: DOES MOIST DESCENT FOR CUMULUS DOWNDRAFTS
!
! Switches 
!
!   lmfmid = .T.  MIDLEVEL CONVECTION IS SWITCHED ON
!   lmfdd  = .T.  CUMULUS DOWNDRAFTS SWITCHED ON
!   lmfdudv= .T.  CUMULUS FRICTION SWITCHED ON
!
! Model parameters 
!
!   An 'initial' call to cu_param before the first call to the convection
!   scheme is required to set up the parameters of the scheme which are
!   communicated to the various modules via common block *com_cumf*
!
! Reference:
!
!   Paper on massflux scheme by M. Tiedtke, 1989.
!
!------------------------------------------------------------------------------
!
! Declarations:
!
!------------------------------------------------------------------------------

! Subroutine arguments:
! --------------------

! Input data
! ----------
  INTEGER (KIND=iintegers), INTENT (IN) ::  &
     idim ,       & ! array dimension in zonal direction
     kdim ,       & ! array dimension in vertical direction 
     isc  ,       & ! start index for first  array computation
     iec  ,       & ! end   index for first  array computation
     nztr           ! number of tracers for convection

  REAL    (KIND=ireals   ), INTENT (IN) ::  &
     pt      (idim,kdim),     & ! temperature at full levels
     pqv     (idim,kdim),     & ! specific humidiy at full levels
     ptrcr   (idim,kdim,nztr),& ! tracers at full levels
     pu      (idim,kdim),     & ! zonal wind component
     pv      (idim,kdim),     & ! meridional wind component
     pw      (idim,kdim),     & ! vertical velocity in z-system (full levles)
     pfif    (idim,kdim  ),   & ! geopotential at full levels
     pfih    (idim,kdim+1),   & ! geopotential at half levels
     ppf     (idim,kdim  ),   & ! pressure at full levels 
     pph     (idim,kdim+1),   & ! pressure at half levels 
     pt_g    (idim       ),   & ! surface temperature
     pdqvdt  (idim,kdim),     & ! moisture tencency
     pqhfl   (idim     ),     & ! surface moisture flux    
     ptke    (idim,kdim),     & ! turbulent kinetic energy (SQRT(2*TKE))
     pdt2                       ! timestep (*2 for leapfrog integration)

  LOGICAL                 , INTENT (IN) ::  &
     lolp (idim)          ! land-sea indicator
  
! Output data
! -----------
  REAL    (KIND=ireals   ), INTENT (OUT) ::  &
     pdt_con  (idim, kdim),  & ! convective tendency of temperature
     pdqv_con (idim, kdim),  & ! convective tendency of specific humidity
     ptrcr_con(idim,kdim,nztr), & ! convective tendency of tracers
     pdtke_con(idim, kdim),  & ! convective boyant TKE prpoduction on half levels
     pqct_con (idim, kdim),  & ! convective tendency of cloud water
     pqit_con (idim, kdim),  & ! convective tendency of cloud ice
     pdu_con  (idim, kdim),  & ! convective tendency of u                 
     pdv_con  (idim, kdim),  & ! convective tendency of v                 
     pco_clw  (idim, kdim),  & ! convective cloud liquid water
     ppr_con  (idim),        & ! convective precipitation rate of rain
     pps_con  (idim),        & ! convective precipitation rate of snow
     ppne_con (idim),        & ! convective precipitation rate without evaporat.
     pcumflx  (idim),        & ! cu_base massflux
     pcucape  (idim),        & ! convective available energy (J/kg)
     pcuctke  (idim),        & ! convective turbulent energy (J/kg)
     pcuconv  (idim),        & ! moisture convergence used for cu_base massflux
     pdtlhn_con (idim, kdim)   ! conv T-tend. due to LH exchanges/no rain evap.

     ! These output-arrays have to be set to zero on call of this routine !

  REAL    (KIND=ireals   ), INTENT (INOUT) ::  &
     pvddraf  (idim)          ! maximum possible convective gust

  INTEGER (KIND=iintegers), INTENT (OUT) ::  &
     mbas_con (idim)      , & ! cloud base level index
     mtop_con (idim)          ! cloud top  level index

  LOGICAL                 , INTENT (OUT) ::  &
     locum(idim)          ! indicatior for convection at gridpoints

! Local scalars and automatic arrays (also to be used in lower level routines):
! ----------------------------------
  INTEGER (KIND=iintegers) ::  &
    i, k, km1,             & ! loop indices over spatial dimensions
    iztr ,                 & ! tracer looping index
    isp,                   & ! number of chem. types (COSMO_ART)
    mcum,                  & ! counter of convective points
    mtop, mkb, msum,       & ! 
    mtopm2,mbot,           & !
    mlab   (idim,kdim),    & !
    mclab  (idim,kdim),    & !
    mtype  (idim),         & !
    mdtop  (idim),         & !
    mctop0 (idim),         & !
    mlwmin (idim),         & !
    mtmelt (idim),         & ! melting level
    kcptop (idim)            ! highest unstable layer in CAPE-determination

  REAL    (KIND=ireals   ) ::  &
    ztu    (idim,kdim) ,      & !
    zqu    (idim,kdim) ,      & !
    ztru   (idim,kdim,nztr),  & ! tracer updraft
    zuu    (idim,kdim) ,      & !
    zvu    (idim,kdim) ,      & !
    zlu    (idim,kdim) ,      & !
    zlude  (idim,kdim) ,      & !
    zmfu   (idim,kdim) ,      & !
    zmfus  (idim,kdim) ,      & !
    zmfuq  (idim,kdim) ,      & !
    zmful  (idim,kdim) ,      & !
    zdmfup (idim,kdim) ,      & !
    ztd    (idim,kdim) ,      & !
    zqd    (idim,kdim) ,      & !
    ztrd   (idim,kdim,nztr) , & ! tracer downdraft
    zud    (idim,kdim) ,      & !
    zvd    (idim,kdim) ,      & !
    zld    (idim,kdim) ,      & !
    zmfd   (idim,kdim) ,      & !
    zmfds  (idim,kdim) ,      & !
    zmfdq  (idim,kdim) ,      & !
    zdmfdp (idim,kdim) ,      & !
    zcape  (idim)      ,      & ! convective available energy CAPE
    zcptu  (idim,kdim) ,      & ! updraft temperature in CAPE-determination
    zcpqu  (idim,kdim)          ! updraft humidity in CAPE-determination

  REAL    (KIND=ireals   ) ::  &
    zmfutr  (idim,kdim,nztr), &  ! mass flux up: tracers
    zmfuu   (idim,kdim),      &  ! mass flux up: u
    zmfuv   (idim,kdim),      &  ! mass flux up: v
    zmfdtr  (idim,kdim,nztr), &  ! mass flux down: tracers
    zmfdu   (idim,kdim),      &  ! mass flux down: u
    zmfdv   (idim,kdim),      &  ! mass flux down: v

    zqsen  (idim,kdim) ,      & ! saturation specific humidiy at full levels
    ztenh  (idim,kdim) ,      & !
    zqenh  (idim,kdim) ,      & !
    zqsenh (idim,kdim) ,      & !
    zrhgdz (idim,kdim)          ! rho*g*dz

  REAL    (KIND=ireals   ) ::  &
    z1dp      (idim)   ,      & !
    zcond     (idim)   ,      & !
    zqold     (idim)   ,      & !
    zrfl      (idim)   ,      & !
    zrneva    (idim)   ,      & !
    zentr     (idim)   ,      & !
    zhcbase   (idim)   ,      & !
    zmfub     (idim)   ,      & !
    zmfub1    (idim)   ,      & !
    zdqpbl    (idim)   ,      & !
    zdqcv     (idim)   ,      & !
    zwmax     (idim)   ,      & !
    zqenwb (idim,kdim) ,      & !
    ztenwb (idim,kdim) ,      & !

!_cdm Storage variable "zl" does not seem to be needed if "lh_mp" is used (see above).
    zbuo,  zc3, zc4, zc5, zzs,                               & !  
    zlkdcp, z1dwlk, z1dgdt, zttest, zqtest, zmftop,          & !
    zzz, zhsat, zhhat, zgam, zfac, zqumqe, zdqmin,           & !
    zdeltx, zrnew, zrmin, zrfln, zpbmpt, zeps, zdp, zzp, zl, & !
    zztvcu, zztven, zcpfunc,                                 & !
    zcvfl_s, zcvfl_q ! convective flux density or dry static energy and vater vapour

  LOGICAL                  ::  &
  llo1, ll2, ll3,            & !
  llo2   (idim),             & !              
  loddraf(idim),             & !              
  loflag (idim)                !              
 
#ifdef MESSY
  REAL(KIND=ireals) :: xdpevap, xsfl(idim), xrfl(idim), xrsum, xpsubcl(idim) &
       , xrfln, xrnew, xrmin, xcucov, xcons1, xcons2, xsnmlt, xfac, yrfl
  REAL(KIND=ireals),PARAMETER :: alv = 2.5008e6_ireals ! latent heat for vaporisation in J/kg
  REAL(KIND=ireals),PARAMETER :: als = 2.8345e6_ireals ! latent heat for sublimation in J/kg
  REAL(KIND=ireals),PARAMETER :: alf = als - alv
#endif

!------------ End of header ---------------------------------------------------
 
!------------------------------------------------------------------------------
! Begin Subroutine cu_tied             
!------------------------------------------------------------------------------
 
  z1dgdt=1./(g*pdt2)   ! pdt2=1*dt for first model time step
 
  ! the highest level of these variables is not set below!
  pdt_con  (:,1)   = 0.0_ireals
  pdqv_con (:,1)   = 0.0_ireals
#ifndef MESSY
  ptrcr_con(:,1,:) = 0.0_ireals
#endif
  pdtke_con(:,1)   = 0.0_ireals
  pqct_con (:,1)   = 0.0_ireals
  pqit_con (:,1)   = 0.0_ireals
  pdu_con  (:,1)   = 0.0_ireals
  pdv_con  (:,1)   = 0.0_ireals

!******************************************************************************
!*                                                                            *
!* Section 1: Initialisation of some variables and computation of additional  *
!*            half level properties ( former subroutine cu_ini )              *
!*                                                                            *
!******************************************************************************
 
  ! Initialization of arrays needed in the parameterisation of
  ! convection; in particular interpolation of large-scale
  ! (environmental) fields to model half levels and determination
  ! of the level of maximum (upward) vertical velocity
 
  ! 1. Specify saturation specific humidity, large scale parameter at half 
  !    levels, adjust temperature if statically unstable, find level of maximum
  !    vertical velocity.
  !    Half level geopotential set to central value between full layers !
  !    First guess for half level temperature from adiabatic interpolation
  !    of adjacent full layer values (maximum of two values is taken)
  !    First guess for half level saturation humidity is saturation
  !    humidity at upper full level

  DO k = 1, kdim
    DO i = isc, iec
      ! This loop has been modified to account for the water-ice mixed phase.
      ! Water fraction for the mixed water-ice phase as dependent on temperature
      ! (DM)
      IF (pt(i,k).LE.Tmpmin) THEN
        fr_wat = 0._ireals
      ELSE IF (pt(i,k).GE.Tmpmax) THEN
        fr_wat = 1._ireals
      ELSE
        fr_wat = ((pt(i,k)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
      ENDIF
      ! Saturation specific humidity over water and over ice
      qs_w = cc2*EXP( b2w*(pt(i,k)-b3)/(pt(i,k)-b4w) )/ppf(i,k)
      qs_i = cc2*EXP( b2i*(pt(i,k)-b3)/(pt(i,k)-b4i) )/ppf(i,k)
      ! Effective saturation specific humidity over water-ice mixed phase
      qs_m = fr_wat*qs_w + (1._ireals-fr_wat)*qs_i
      zqsen(i,k) = qs_m / (1.0-rvd_m_o*qs_m)
    ENDDO
  ENDDO

  DO k = 2, kdim
    DO i = isc, iec
      ztenh(i,k) = ( MAX (cp_d*pt(i,k-1)+pfif(i,k-1),cp_d*pt(i,k)+pfif(i,k) ) &
                    - pfih(i,k) )/cp_d  
      zqsenh(i,k)= zqsen(i,k-1)
      z1dp  (i)  = 1.0/pph(i,k)
      loflag(i)  = .TRUE.
    END DO
 
    ! adjust temperature and humidity at half level to value for moist
    ! adiabatic interpolation from neighbour levels (i.e. consider 
    ! condensation/evaporation process)

    CALL cu_cond ( ztenh(:,k), zqsenh(:,k), z1dp,              &
                   loflag , .TRUE.  , .TRUE.,                  &
                   idim   , isc     , iec            )
 
    ! interpolation of specific humidity to half levels, following
    ! a moist adiabate from the upper full level and avoiding supersaturation
    DO i = isc, iec
      zqenh(i,k) = MIN( pqv(i,k-1), zqsen(i,k-1) ) + zqsenh(i,k) - zqsen(i,k-1) 
      zqenh(i,k) = MAX( zqenh(i,k), 0.0_ireals )
    END DO
   
  END DO    ! vertical loop
 
  ! lowest layer
  DO i = isc, iec
!   avoid 'convective drizzle' by skipping the next two statements
!   ztenh (i,kdim) = (cp_d*pt(i,kdim) + pfif(i,kdim) - pfih(i,kdim))/cp_d  
!   zqenh (i,kdim) = pqv(i,kdim)
    ztenh (i,1   ) = pt  (i,1)
    zqenh (i,1   ) = pqv (i,1)
    mlwmin(i)      = kdim             ! starting value 
    zwmax (i)      = 0.0              ! starting value 
  END DO

  ! avoid unstable structure in half level temperature profile 
  DO k = kdim-1, 2, -1
    DO i = isc, iec
       zzs = MAX( cp_d*ztenh(i,k)+pfih(i,k), cp_d*ztenh(i,k+1)+pfih(i,k+1) )
       ztenh(i,k) = ( zzs - pfih(i,k) )/cp_d  ! ztenh may become incompatible
    END DO                                    ! with zqenh
  END DO
  DO k = kdim, 1, -1
    DO i = isc, iec
      IF( pw(i,k) > zwmax(i) ) THEN
          zwmax (i) = pw(i,k)
          mlwmin(i) = k
      END IF
    END DO
  END DO
 
  ! Initialize updraft and downdraft values 
  DO k = 1, kdim
    km1 = MAX( k-1, 1 ) 
    DO i = isc, iec
      ztu    (i,k) = ztenh (i,k)
      ztd    (i,k) = ztenh (i,k)
      zqu    (i,k) = zqenh (i,k)
      zqd    (i,k) = zqenh (i,k)
      zlu    (i,k) = 0.
      zuu    (i,k) = pu (i,km1)
      zud    (i,k) = pu (i,km1)
      zvu    (i,k) = pv (i,km1)
      zvd    (i,k) = pv (i,km1)
      zmfu   (i,k) = 0.
      zmfd   (i,k) = 0.
      zmfus  (i,k) = 0.
      zmfds  (i,k) = 0.
      zmfuq  (i,k) = 0.
      zmfdq  (i,k) = 0.
      zdmfup (i,k) = 0.
      zdmfdp (i,k) = 0.
      zlude  (i,k) = 0.
      mlab   (i,k) = 0
      mclab  (i,k) = 0
      zrhgdz (i,k) = ppf(i,k)*( pfih(i,k) - pfih(i,k+1) )/(r_d*pt(i,k))
      zcptu  (i,k) = ztenh(i,k)
      zcpqu  (i,k) = zqenh(i,k)
    END DO
  END DO

#ifndef MESSY
DO iztr = 1, nztr
  DO k = 1, kdim
    km1 = MAX( k-1, 1 ) 
!CDIR NODEP
    DO i = isc, iec
      ztru(i,k,iztr)  = ptrcr(i,km1,iztr)
      ztrd(i,k,iztr)  = ptrcr(i,km1,iztr)
    END DO
  END DO
ENDDO
#endif

!******************************************************************************
!*                                                                            *
!* Section 2: Cloud base calculations ( former subroutine cu_base )           *
!*                                                                            *
!******************************************************************************

  ! a) Determination of cloud-base values
  !     Method
  !       humidity, pressure and geopotential at half levels this
  !       routine computes corresponding cloud base values and the
  !       following flag indices:
  !             mlab=1 as indicator of sub-cloud levels
  !             mlab=2 as indicator of in-cloud levels
  !     - to define a cloud base, surface air is lifted dry-adiaba-
  !       tically up to cloud base (non-entraining plume, i.e. for
  !       constant massflux)
  !     - temperature and humidity are adjusted inside of the cloud
  !       to take into account condensation effects

  ! Initialize lifting level variables
  DO i = isc, iec
    mlab (i,kdim ) = 1        ! lowest layer is below cloud base
    mbas_con(i)    = kdim-1   ! cloud base index
    locum(i)       =.false.
  END DO

  DO k = kdim-1, 2, -1     ! Vertical loop over layers

    DO i = isc, iec
      z1dp (i) = 1./pph(i,k)
      IF ( mlab(i,k+1).EQ.1 ) THEN
        loflag(i) = .TRUE.   ! ascent continues
      ELSE
        loflag(i) = .FALSE.  ! ascent complete
      ENDIF
    END DO

    DO i = isc, iec
      IF( loflag(i) ) THEN        ! cloud base not found yet
        zqu(i,k) = zqu(i,k+1) ! retain parcel's humdity
                      ! parcel temperature after ascent to next layer
        ztu(i,k) = ( cp_d*ztu(i,k+1) + pfih(i,k+1) - pfih(i,k) )/cp_d  

        ! difference between parcel (virtual) temperature and environmental
        ! (virtual) temperature determines buoancy of parcel (0.5K added)
        zbuo =   ztu  (i,k) * ( 1.0 + rvd_m_o*zqu  (i,k) )  &
               - ztenh(i,k) * ( 1.0 + rvd_m_o*zqenh(i,k) ) + 0.5
        IF(zbuo.GT.0.) mlab(i,k) = 1 ! sub-cloud indicator for positve buoyancy
        zqold(i) = zqu(i,k) ! store parcel humidity in local variable
      END IF               
    END DO

    !Check for condensation and adjust parcel temperature and humidity
    !if condensation/sublimation occurs

    CALL cu_cond ( ztu(:,k), zqu(:,k), z1dp  ,                 &
                   loflag , .TRUE.  , .FALSE.,                 &
                   idim   , isc     , iec             )

!     If ascent calculations are still active and parcel humidity
!     changed due to condensation:
!      
    DO i = isc, iec
      IF( loflag(i) .AND. zqu(i,k).NE.zqold(i) ) THEN
        mlab(i,k) = 2 !  set indicator variable to in-cloud value
        zlu(i,k) = zlu(i,k) + zqold(i) - zqu(i,k)
        zbuo =   ztu  (i,k)*( 1.0 + rvd_m_o*zqu  (i,k) )  &
               - ztenh(i,k)*( 1.0 + rvd_m_o*zqenh(i,k) ) + 0.5
        IF(zbuo.GT.0.) THEN ! Test for buoyancy
          mbas_con(i) = k        ! define cloud base index
          locum(i)    = .TRUE.   ! indicate existence of unstable cloud base
          zcptu(i,k)  = ztu(i,k)
          zcpqu(i,k)  = zqu(i,k)
          mclab(i,k)  = 2
        END IF
      END IF
    END DO
 
  END DO   ! Vertical loop

   
  ! b) total moisture convergence and decision on type of convection
 
  DO i = isc, iec
    zdqcv (i) = pdqvdt(i,1)*zrhgdz(i,1)
    zdqpbl(i) = 0.0
    mdtop (i) = 0
  ENDDO

  DO k = 2, kdim
    DO i = isc, iec
      zdqcv(i) = zdqcv(i) + pdqvdt(i,k)*zrhgdz(i,k)
      IF ( k >= mbas_con(i) )  THEN
        zdqpbl(i) = zdqpbl(i) + pdqvdt(i,k)*zrhgdz(i,k)
      ENDIF
    ENDDO
  ENDDO

  DO i = isc, iec
    IF ( zdqcv(i) > MAX(0.0_ireals, -1.1_ireals*pqhfl(i)*g) ) THEN
       mtype(i) = 1     ! penetrative convection
    ELSE
       mtype(i) = 2     ! shallow convection
    ENDIF
  ENDDO
  

  ! c) Determination of convective available potential energy and convective
  !    turbulent energy (vertical mean of tke)

  DO i = isc, iec
    kcptop(i) = 0    ! Initialize k-Index for cloud-top
    zcape(i)  = 0.0  ! Initialize convective available energy (J/kg)
  END DO

  DO k = kdim-2, 2, -1  ! Loop over layers
 
    DO i = isc, iec
      z1dp (i) = 1./pph(i,k)
      IF ( locum(i) .AND. mclab(i,k+1).EQ.2 ) THEN
        loflag(i) = .TRUE.   ! in-cloud parcel ascent continues
      ELSE
        loflag(i) = .FALSE.  ! ascent complete
      ENDIF
    END DO

    DO i = isc, iec
      IF ( loflag(i) ) THEN  !  Unstable cloud level exists below 
        zcpqu(i,k) = zcpqu(i,k+1)               
        zcptu(i,k) = ( cp_d*zcptu(i,k+1) + pfih(i,k+1) - pfih(i,k) )/cp_d
      ENDIF
    ENDDO

    ! Saturation adjustment for moist adiabatic ascent
    CALL cu_cond ( zcptu(:,k), zcpqu(:,k), z1dp  ,  &
                 loflag , .TRUE.  , .FALSE.,        &
                 idim   , isc     , iec             )
    DO i = isc, iec
      IF( loflag(i) ) THEN
        zztvcu  = zcptu(i,k)* (1.0 + rvd_m_o*zcpqu(i,k) )  
        zztven  = ztenh(i,k)* (1.0 + rvd_m_o*zqenh(i,k) ) 
        zbuo    = zztvcu - zztven + 0.5
        IF(zbuo > 0.0) THEN      ! Test for buoyancy
          mclab(i,k)  = 2        ! Set indicator for ascent to continue
          kcptop(i)   = k        ! set cloud top level for ascent
          zcape (i)   = zcape(i) + (zbuo/zztven)*(pfih(i,k) - pfih(i,k+1))
        END IF
        pcucape(i) = zcape(i)
      END IF
    END DO
 
  END DO                     ! End loop over layers

  IF (lctke) THEN
    DO i = isc, iec
      IF (locum(i)) THEN
        ! Determination of turbulent convective energy by vertically
        ! averaging tke (here the field ptke is SQRT(2*tke) in m/s):
        pcuctke(i)   = 0.0
        DO k = mbas_con(i),kcptop(i),-1
          pcuctke(i) = pcuctke(i) + (ppf(i,k) - ppf(i,k-1))*ptke(i,k)**2
        END DO
        pcuctke(i)   = pcuctke(i)*0.5/(ppf(i,mbas_con(i)) - ppf(i,kcptop(i)))
      END IF  ! locum
    END DO
  END IF      ! lctke

!******************************************************************************
!*                                                                            *
!* Section 3: Moisture supply in boundary layer and preliminary cloud base    *
!*            mass flux (excluding downdraft effects)                         *
!*                                                                            *
!******************************************************************************
 
  IF (lcape) THEN                   ! Convective available energy closure
    
    DO i = isc, iec
      IF (locum(i)) THEN ! convective active points only
        mkb       = mbas_con(i)
        llo1      = zcape(i) > capemin
!       Massflux  = capeconstant * rho * sqrt(CAPE)   (kg/s m**2)
        IF (llo1) THEN
          zmfub(i)= capeconstant*SQRT(zcape(i))*(ppf(i,mkb)/(r_d*pt(i,mkb)))
          zmfub(i)= MIN( zmfub(i), cmfcmax)  ! massflux upper limit
        ELSE
          zmfub(i)= 0.0
          locum(i)= .FALSE.
        END IF
        pcumflx(i)= zmfub(i)
        IF (zdqpbl(i).gt.0.0001) THEN
          pcuconv(i) = zdqpbl(i)
        ELSE
          pcuconv(i) = 0.0
        END IF
        IF ( mtype(i).EQ.1 ) THEN            ! penetrative convection
          zentr(i) = entrpen
        ELSE                                 ! shallow convection
          zentr(i) = entr_sc
        ENDIF
      END IF                                 ! Convective active points
    END DO

  ELSE IF (lctke) THEN              ! Convective turbulent energy closure
    
    DO i = isc, iec
      IF (locum(i)) THEN                     ! Convectice active points only
        mkb        = mbas_con(i)
        zqumqe     = zqu(i,mkb) + zlu(i,mkb) - zqenh(i,mkb)
        llo1       = pcuctke(i) > ctkemin 
        IF (llo1) THEN
          zmfub(i) = (pcuctke(i)/cmbtke)*ppf(i,mkb)/(g*r_d*pt(i,mkb))
        ELSE
          zmfub(i) = 0.0
        END IF
        zmfub(i)   = MIN(zmfub(i),cmfcmax)
        pcumflx(i)= zmfub(i)
        IF (zdqpbl(i).gt.0.0001) THEN
          pcuconv(i) = zdqpbl(i)
        ELSE
          pcuconv(i) = 0.0
        END IF
        IF (.NOT. llo1) locum(i) = .FALSE.
        IF ( mtype(i).EQ.1 ) THEN            ! penetrative convection
          zentr(i) = entrpen
        ELSE                                 ! shallow convection
          zentr(i) = entr_sc
        END IF
      END IF                                 ! locum
    END DO

  ELSE                              ! Moisture convergence closure
 
    DO i = isc, iec
      mkb = mbas_con(i)
      zqumqe = zqu(i,mkb) + zlu(i,mkb) - zqenh(i,mkb)
      zdqmin = MAX( 0.01_ireals*zqenh(i,mkb), 1.E-10_ireals )
!     avoid 'convective drizzle' by using a minimum moisture convergence
!     llo1   = zdqpbl(i) > 0.0      & ! positive moisture convergence
      llo1   = zdqpbl(i) > (1.E-3*g*zqumqe) & ! minimum positive moist. conv.
              .AND. zqumqe > zdqmin & ! parcel humidity exceeds env.hum.
              .AND. locum(i)          ! convective grid point
      IF ( llo1 ) THEN
        zmfub(i) = zdqpbl(i) / (g*MAX(zqumqe, zdqmin))
      ELSE
        zmfub(i) = 0.0
        locum(i) = .FALSE.            ! GP switched off
      ENDIF
      zmfub(i) = MIN( zmfub(i), cmfcmax)   !  massflux upper limit
      pcumflx(i)= zmfub(i)
      IF (zdqpbl(i).gt.0.0001) THEN
        pcuconv(i) = zdqpbl(i)
      ELSE
        pcuconv(i) = 0.0
      END IF
      IF ( mtype(i).EQ.1 ) THEN            ! penetrative convection
        zentr(i) = entrpen
      ELSE                                 ! shallow convection
        zentr(i) = entr_sc
      ENDIF
    ENDDO

  END IF   ! lcape/lctke

!******************************************************************************
!*                                                                            *
!* Section 4: Cloud ascent for entraining plume                               *
!*                                                                            *
!******************************************************************************

! a) estimate cloud height for entrainement/detrainement calculations
!    in cu_asc (max.possible cloud height for non-entraining plume,
!    following Arakawa/Schubert, 1974)

  DO i = isc, iec
    mkb        = mbas_con(i)

    ! Modification to account for the water-ice mixed phase
    ! Water fraction and effective latent heat of evaporation/sublimation for the mixed phase
    ! as function of the updraught temperature.
    IF (ztu(i,mkb).LE.Tmpmin) THEN
      fr_wat = 0._ireals
    ELSE IF (ztu(i,mkb).GE.Tmpmax) THEN
      fr_wat = 1._ireals
    ELSE
      fr_wat = ((ztu(i,mkb)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
    ENDIF
    lh_mp = fr_wat*lh_v + (1._ireals-fr_wat)*lh_s
    zhcbase(i) = cp_d*ztu (i,mkb) + pfih(i,mkb) + lh_mp*zqu(i,mkb)

    mctop0 (i) = mbas_con(i) - 1
  ENDDO

  ! Modification to account for the water-ice mixed phase. (DM)
  ! This part of the code (that seemingly stems from the original T89) is too cryptic.
  ! Think of simplifying it.

  DO k = kdim-1, 3, -1  ! vertical loop from bottom to top+2
    DO i = isc, iec
      ! Water fraction as function of temperature of the environment air
      IF (ztenh(i,k).LE.Tmpmin) THEN
        fr_wat = 0._ireals
      ELSE IF (ztenh(i,k).GE.Tmpmax) THEN
        fr_wat = 1._ireals
      ELSE
        fr_wat = ((ztenh(i,k)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
      ENDIF

      ! Effective latent heat of evaporation/sublimation for the mixed phase
      lh_mp = fr_wat*lh_v + (1._ireals-fr_wat)*lh_s

      ! moist static energy at saturation of environment
      zhsat = cp_d*ztenh (i,k) + pfih  (i,k) + lh_mp*zqsenh(i,k)

!_cdm "zgam" seems to be simply "qdvdt_w" that should be replaces with "qdvdt_m"
!     to account for the mixed phase. Notice, however,
!     that the saturation specific humidity "zqsenh" and the temperature "ztenh"
!     of the environment air have aready been adjusted with due regard for the mixed phase
!     (see CALL cu_cond above). Then, it might be sufficient to only replace the constant "b4w"
!     in the following computation of "zgam" with the "effective" value of "b4" characteristic
!     of the mixed phase. For the sake of clarity, we compute "zgam" in the same way as in
!     the modified SUBROUTINE cu_cond.
      qdvdt_w = c5hlccp * zqsenh(i,k)/(1.0-rvd_m_o*zqsenh(i,k)) / (ztenh(i,k)-b4w)**2
      qdvdt_i = c5hlscp * zqsenh(i,k)/(1.0-rvd_m_o*zqsenh(i,k)) / (ztenh(i,k)-b4i)**2
      zgam    = fr_wat*qdvdt_w + (1._ireals-fr_wat)*qdvdt_i
      zzz     = cp_d*ztenh(i,k)*rvd_m_o
      zhhat   = zhsat - ( zzz + zgam*zzz ) / ( 1.0 + zgam*zzz/lh_mp) &
                * MAX( zqsenh(i,k) - zqenh(i,k), 0.0_ireals )

      !compare most static energy of base with environment of present layer
      IF( k < mctop0(i) .AND. zhcbase(i) > zhhat)  mctop0(i) = k
    ENDDO
  ENDDO
 
  ! b) ascent in cu_asc in 'absence' of downdrafts
 
  CALL  cu_asc  (                                                  &
           ppf     , pt      , pqv   , ptrcr   , pu     , pv   ,   &
           zqsen   , pfif    ,                                     &
           pph     , ztenh   , zqenh , pfih ,                      &
           pdqvdt  , pw      , zdqpbl, pcumflx,                    &
           mlwmin  , mtype   ,  mlab , lolp , locum,               &
           idim    , kdim    , isc   , iec  , nztr ,               &
           ztu     , zqu     , ztru  , zlu  , zuu  , zvu  ,        &
           zmfu    , zmfub   , zmfus , zmfuq, zmful, zlude,        &
           zdmfup  , zentr   ,                                     &
           mbas_con, mtop_con, mctop0, mcum ) 

  ! Check for existence of any convective grid points

  IF(mcum.EQ.0) RETURN  ! Exit routine in case of no convection
 
  ! c) check cloud depth and modify entrainement rate accordingly     
  !    calculate precipitation rate (for downdraft calculation)  
 
  DO i=isc,iec
    ! cloud thickness (in Pa)
    zpbmpt = pph(i,mbas_con(i)) - pph(i,mtop_con(i))
    ! 'thin' convective clouds must not be of penetrative type:
    IF( locum(i)                 &  ! convective grid point
        .AND. mtype(i) == 1      &  ! classified as penetrative
        .AND. zpbmpt < 2.0E4 )   &  ! 200 hpa pressure thickness
       mtype(i) = 2                 ! --> type changed to shallow
    ! Correct (if applicable) entrainment rate for 'thin' clouds
    IF( mtype(i) == 2)   zentr(i) = entr_sc
    ! store cloud top in local array
    IF(locum(i))  mctop0(i) = mtop_con(i)
    ! set precipitation rate from top layer mass flux
    zrfl(i) = zdmfup(i,1)
  ENDDO
 

  ! d) Modify precipitation rate as mass flux changes
  DO k = 2, kdim
    DO i = isc, iec
      zrfl(i) = zrfl(i) + zdmfup(i,k)
    ENDDO
  ENDDO
 
!******************************************************************************
!*                                                                            *
!* Section 5: Downdraft calculations                                          *
!*                                                                            *
!******************************************************************************

  IF (lmfdd) THEN     ! downdraft calculations switched on
 
    ! a) determine 'level of free sinking' in cu_dlfs
    !     Method:   - from environmental values of T,q,u,v,p and fi and
    !                 updraft values of of T,q,u and v and also the cloud
    !                 base mass flux and the precipitation rate, the values
    !                 of T,q,u and v and the massflux at LFS are determined
    !               - of equal parts of cloud air and moist environmental
    !                 air are checked for negative buoyancy
    !               - the descending air is adjusted for evaporation of rain

    ! set defaults for downdrafts
    DO i = isc, iec
      loddraf(i) = .false.
      mdtop  (i) = kdim+1
      llo2   (i) = .false.
    ENDDO

    ! determine level of free sinking by scanning from base to top of each 
    ! cumulus cloud and 
    ! a) determine wet bulb environmental T and q
    ! b) mix with cumulus air
    ! c) check for negative buoancy
    !    it is assumed that downdraft air is an equal parts mixture of
    !    cloud air and environmental air at wet bulb temperature (i.e. which 
    !    became saturated due to evaporation of rain and cloud water)
 
    DO  k = 3, kdim - 3  ! begin vertical loop

    ! wet bulb temperature and humidity for environmental air
    DO i = isc, iec
      z1dp  (i  ) = 1.0 / pph(i,k)
      ztenwb(i,k) = ztenh (i,k)
      zqenwb(i,k) = zqenh(i,k)
      llo2  (i)   = locum(i)             & ! active grid point
                   .AND. zrfl(i) > 0.0   & ! precip > 0
                   .AND. .NOT.loddraf(i) & ! down draft switch off
                   .AND. (k<mbas_con(i).AND.k>mtop_con(i))! in cloud
    ENDDO

    ! saturation adjustment for environmental T and q

    CALL cu_cond ( ztenwb(:,k), zqenwb(:,k), z1dp  ,           &
                   llo2   , .FALSE.  , .TRUE.,                 &
                   idim   , isc     , iec            )

    ! mix cloud and environmental air and check for negative buoancy
    ! then set values for downdraft at LFS
    DO i = isc, iec
      IF(llo2(i)) THEN
        zttest = 0.5*( ztu(i,k) + ztenwb(i,k) )
        zqtest = 0.5*( zqu(i,k) + zqenwb(i,k) )
        zbuo   = zttest*(1.+rvd_m_o*zqtest)- ztenh(i,k)*(1.+rvd_m_o*zqenh(i,k))
        zcond(i) = zqenh(i,k) - zqenwb(i,k)
        zmftop   = - cmfdeps*zmfub(i)
        IF( zbuo <0.0 .AND. zrfl(i) > 10.0*zmftop*zcond(i) ) THEN
          mdtop(i)    = k
          loddraf(i)  = .true.
          ztd  (i,k) = zttest
          zqd  (i,k) = zqtest
          zmfd (i,k) = zmftop
          zmfds(i,k) = zmfd(i,k)*( cp_d*ztd(i,k) + pfih(i,k) )
          zmfdq(i,k) = zmfd(i,k)*zqd(i,k)
          zdmfdp(i,k-1) = -0.5*zmfd(i,k)*zcond(i)
          zrfl(i)       = zrfl(i) + zdmfdp(i,k-1)
        ENDIF
      ENDIF
    ENDDO
    IF (lmfdudv) THEN
      DO i = isc, iec
        IF(zmfd(i,k).LT.0.) THEN
          zud(i,k) = 0.5*( zuu(i,k) + pu(i,k-1) )
          zvd(i,k) = 0.5*( zvu(i,k) + pv(i,k-1) )
        ENDIF
      ENDDO
    ENDIF

#ifndef MESSY
    DO iztr = 1, nztr
      DO i=isc, iec
        IF (zmfd(i,k) < 0.0_ireals) THEN
          ztrd(i,k,iztr) = 0.5 * (ztru(i,k,iztr) + ptrcr(i,k-1,iztr))
        ENDIF
      ENDDO
    ENDDO
#endif

    ENDDO            ! end of vertical loop

  ! b) calculate downdraft effects in cu_ddraf

    CALL  cu_ddraf(                                              &
           pu     , pv   , pph  , ztenh , zqenh  , ptrcr , pfih ,&
           pt     ,                                              &
           idim   , kdim , isc  , iec   , nztr   ,               &
           ztd    , zqd  , ztrd , zud   , zvd    , pvddraf,      &
           zmfd   , zmfds, zmfdq, zdmfdp, zrfl   ,               &
           loddraf  )

  ! c) recalculate convective fluxes due to effect of downdrafts
  !    on boundary layer moisture budget

    DO i = isc, iec
      IF(loddraf(i)) THEN     ! if downdraft exists
        mkb  = mbas_con(i)
        llo1 = zmfd(i,mkb) < 0.0  ! test downdraft mass flux at base
        IF ( llo1 ) THEN
          zeps = cmfdeps
        ELSE
          zeps = 0.0
        ENDIF
        ! humidity difference (updraft - environment)
        zqumqe = zqu(i,mkb) + zlu(i,mkb)- zeps*zqd(i,mkb)  &
                            - (1.0 - zeps)*zqenh(i,mkb)
        zdqmin = MAX( 0.01_ireals*zqenh(i,mkb), 1.E-10_ireals )
        llo1   = zdqpbl(i) > 0.0        & ! positive moisture convergence
                 .AND. zqumqe.GT.zdqmin & ! sufficient humidity excess
                 .AND. locum(i)           ! convective grid point
        IF ( llo1 .AND. (.NOT. (lcape .OR. lctke)) ) THEN
          zmfub1(i) = zdqpbl(i) / ( g*MAX( zqumqe, zdqmin ) )
        ELSE
          zmfub1(i) = zmfub(i)
        ENDIF
        IF ( .NOT.(  ( mtype(i) == 1 .OR. mtype(i) == 2 ) .AND.  &
                       ABS( zmfub1(i) - zmfub(i) ) < 0.2*zmfub(i)  ) ) THEN
              zmfub1(i) = zmfub (i)
        ENDIF
      ENDIF
    ENDDO

    DO k = 1, kdim
      DO i = isc, iec
        IF(loddraf(i)) THEN
          zfac        = zmfub1(i) / MAX( zmfub(i), 1.0E-10_ireals )
          zmfd  (i,k) = zmfd  (i,k) * zfac
          zmfds (i,k) = zmfds (i,k) * zfac
          zmfdq (i,k) = zmfdq (i,k) * zfac
          zdmfdp(i,k) = zdmfdp(i,k) * zfac
        ENDIF
      ENDDO
    ENDDO
  
    DO i=isc,iec
      IF(loddraf(i)) zmfub(i) = zmfub1(i)
    ENDDO
 
  ENDIF         ! cumulus downdraft switch on
 
!******************************************************************************
!*                                                                            *
!* Section 6: Determine final cloud ascent for entraining plume               *
!*              penetrative convection (type=1)                               *
!*              shallow     convection (type=2)                               *
!*              mid-level   convection (type=3)                               *
!*                                                                            *
!******************************************************************************

  CALL  cu_asc  (                                                  &
           ppf     , pt      , pqv   , ptrcr, pu   , pv   , zqsen, &
           pfif    ,                                               &
           pph     , ztenh   , zqenh , pfih ,                      &
           pdqvdt  , pw      , zdqpbl, pcumflx,                    &
           mlwmin  , mtype   , mlab , lolp , locum ,               &
           idim    , kdim    , isc  , iec  , nztr  ,               &
           ztu     , zqu     , ztru , zlu  , zuu   , zvu  ,        &
           zmfu    , zmfub   , zmfus , zmfuq, zmful, zlude,        &
           zdmfup  , zentr   ,                                     &
           mbas_con, mtop_con, mctop0, mcum ) 

    DO k = 1, kdim
      DO i = isc, iec
          pco_clw(i,k) = MAX ( 0.0_ireals, MIN(100.0_ireals, zlu(i,k)) )
      ENDDO
    ENDDO

!_cdm After the 2nd call of cu_asc, "zlude" contanis detrained cloud condensate for all levels.
!     It has the same dimensions as "zmful".

!******************************************************************************
!*                                                                            *
!* Section 7: Determine final convective fluxes (former subroutine cu_flux)   *
!*            and conside evaporation of rain in sub-cloud layers             *
!*                                                                            *
!******************************************************************************
  mtop = kdim

  DO i = isc, iec
     mtop = MIN ( mtop, mtop_con(i) )
     IF( .NOT.locum(i) .OR. mdtop(i) < mtop_con(i) ) loddraf(i)=.false.
     IF( .NOT.locum(i))  mtype(i) = 0
  ENDDO

! mtopm2 = mtop - 2
  mtopm2 = 2

  DO k = mtopm2, kdim
    DO i = isc, iec

      IF( locum(i) .AND. k >= mtop_con(i)-1 ) THEN
         zmfus(i,k) = zmfus(i,k) - zmfu (i,k)*(cp_d*ztenh(i,k)+pfih(i,k))
         zmfuq(i,k) = zmfuq(i,k) - zmfu (i,k)*zqenh(i,k)
         IF ( lolp(i) ) THEN   ! land points
            zdp = 3.0E4
         ELSE                  ! sea points
            zdp = 1.5E4
         ENDIF
         ! Test cloud thickness, environmental humidity for change in zdmfup
!_cdm Do not add *lude (detrained liquid water) to *dmfup (updraught precipitation flux).
!     This "cloud depth and environmental humidity control" seems to have its origin in the old
!     version of T89 and is not likely to be consistent with the present-day LM physics.
!     The following line must not be used in case the qc and qi tendencies due to
!     detrained liquid water (cloud condensate) are passed to the microphysics routines.
         IF( loddraf(i) .AND. k >= mdtop(i) ) THEN
            zmfds(i,k) = zmfds(i,k) - zmfd(i,k)*(cp_d*ztenh(i,k)+pfih(i,k))
            zmfdq(i,k) = zmfdq(i,k) - zmfd (i,k)*zqenh(i,k)
         ELSE
            zmfd  (i,k  ) = 0.
            zmfds (i,k  ) = 0.
            zmfdq (i,k  ) = 0.
            zdmfdp(i,k-1) = 0.
         ENDIF
      ELSE              ! above cloud and non-convective points
         zmfu  (i,k  ) = 0.
         zmfd  (i,k  ) = 0.
         zmfus (i,k  ) = 0.
         zmfds (i,k  ) = 0.
         zmfuq (i,k  ) = 0.
         zmfdq (i,k  ) = 0.
         zmful (i,k  ) = 0.
         zdmfup(i,k-1) = 0.
         zdmfdp(i,k-1) = 0.
         zlude (i,k-1) = 0.
      ENDIF
    ENDDO
  ENDDO 

!_cdm Sub-cloud layer. Calculations may have to be modified.
!     The existence of cloud below the cloud lower edge is sort of strange.
  DO k = mtopm2, kdim
    DO i = isc, iec
      IF( locum(i) .AND. k > mbas_con(i) ) THEN
        mkb = mbas_con(i)
        zzp = ( pph(i,kdim+1) - pph(i,k) )/( pph(i,kdim+1) - pph(i,mkb) ) 
        IF ( mtype(i).EQ.3 ) zzp=zzp**2
        zmfus(i,k) = zmfus(i,mkb) * zzp
        zmfuq(i,k) = zmfuq(i,mkb) * zzp
        zmful(i,k) = zmful(i,mkb) * zzp
      ENDIF
    ENDDO
  ENDDO      
 
  ! precipitation at surface
  zrfl(:) = 0.0

#ifndef MESSY
  DO k = mtopm2, kdim
    DO i = isc, iec
      IF( locum(i) ) zrfl(i) = zrfl(i) + zdmfup(i,k)+zdmfdp(i,k)
    ENDDO
  ENDDO  
 
#else

  xrfl(:)=0._ireals
  xsfl(:)=0._ireals
  xcons1 = cpd/(alf*gg*time_step_len)
  xcons2 = 1._ireals/(gg*time_step_len)
  DO k = mtopm2, kdim
    DO i = isc, iec
       IF( locum(i) ) THEN
          zrfl(i) = zrfl(i) + zdmfup(i,k)+zdmfdp(i,k)
          IF(pt(i,k).GT.ctmelt) THEN
              xrfl(i)=xrfl(i)+zdmfup(i,k)+zdmfdp(i,k)
              IF(xsfl(i).GT.0._ireals.AND.pt(i,k).GT.ctmelt+2._ireals) THEN
                 xfac=xcons1*(1._ireals+vtmpc2*pqv(i,k))                   &
                             *(pph(i,k+1)-pph(i,k))
                 xsnmlt=MIN(xsfl(i),xfac*(pt(i,k)-ctmelt+2._ireals))
                 xsfl(i)=xsfl(i)-xsnmlt
                 xrfl(i)=xrfl(i)+xsnmlt
              END IF 
           ELSE
              xsfl(i)=xsfl(i)+zdmfup(i,k)+zdmfdp(i,k)
           END IF
        END IF 
        !mz_ht_20040212+ 
        cvprecflx_2d(i,k) = xrfl(i) ! for convec channel: rain flux
        cvsnowflx_2d(i,k) = xsfl(i) !                     snow flux
        !mz_ht_20040212- 
     END DO
  END DO
  DO i = isc, iec
     xrfl(i)=MAX(xrfl(i),0._ireals)
     xsfl(i)=MAX(xsfl(i),0._ireals)
     xpsubcl(i)=xrfl(i)+xsfl(i)
  END DO
  xcucov = 0.05_ireals
  DO k = mtopm2, kdim
    DO i = isc, iec
       IF(locum(i).AND.k.GE.mbas_con(i).AND.xpsubcl(i).GT.1.e-20_ireals) THEN
           yrfl=xpsubcl(i)
           xrnew=(MAX(0._ireals,SQRT(yrfl/xcucov)-                            &
                        cu_evap(k)*(pph(i,k+1)-pph(i,k))*      &
                        MAX(0._ireals,zqsen(i,k)-pqv(i,k))))**2*xcucov
           xrmin=yrfl-xcucov*MAX(0._ireals,0.8_ireals*zqsen(i,k)-pqv(i,k)) &
                        *xcons2*(pph(i,k+1)-pph(i,k))
           xrnew=MAX(xrnew,xrmin)
           xrfln=MAX(xrnew,0._ireals)
           yrfl=MIN(0._ireals,xrfln-yrfl)
           xpsubcl(i)=xrfln
  
           !mz_ht_20040212+   for convect channel
           xrsum=xrfl(i)+xsfl(i)
           xdpevap=xpsubcl(i)-xrsum
           cvprecflx_2d(i,k) = cvprecflx_2d(i,k) + &
                xdpevap * xrfl(i) * (1._ireals/MAX(1.e-20_ireals,xrsum))
           cvsnowflx_2d(i,k) = cvsnowflx_2d(i,k) + &
                xdpevap * xsfl(i) * (1._ireals/MAX(1.e-20_ireals,xrsum))
           !mz_ht_20040212-
        END IF
     END DO
  END DO
#endif

  ! get LHN profiles before evaporation adjustment below
  ! cloud base (lowest level heating is not taken into account)
  ! Modification to account for water-ice mixed phase. (DM)
  DO k = mtopm2, kdim - 1  ! above lowest model layer
    DO i = isc, iec
      IF(locum(i)) THEN ! for convective grid points only
        ! Water fraction as function of temperature
        IF (pt(i,k).LE.Tmpmin) THEN
          fr_wat = 0._ireals
        ELSE IF (pt(i,k).GE.Tmpmax) THEN
          fr_wat = 1._ireals
        ELSE
          fr_wat = ((pt(i,k)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
        ENDIF
        ! Effective latent heat of evaporation/sublimation for the mixed phase
        lh_mp = fr_wat*lh_v + (1._ireals-fr_wat)*lh_s
        pdtlhn_con(i,k) =    g/zrhgdz(i,k)/cp_d  &
                         * (-1.) * lh_mp*( zmful(i,k+1) - zmful(i,k) &
                                      - ( zdmfup(i,k)+zdmfdp(i,k) ) )
      ENDIF
    ENDDO
  ENDDO

  ! Check for highest base of cumulus clouds with rain > 0
  mbot = kdim
  DO i = isc, iec
    zrfl(i) = MAX( 0.0_ireals, zrfl(i) )
    IF ( locum(i) .AND. zrfl (i) > 0.0 ) THEN
       mbot = MIN ( mbot, mbas_con(i) )
    ENDIF
    zrneva (i) = zrfl(i)
  ENDDO
 
  ! Determine evaporation (if necessary)
  IF (mbot.lt.kdim ) THEN       

    DO k = mbot, kdim      ! from highest cloud base to lowest layer
      DO i=isc,iec
        IF(locum(i) .AND. mbas_con(i) <= k .AND. zrfl(i) > 0.0) THEN  
          ! determine evaporation based on rain rate adjusted to cloud cover,
          ! layer thickness and saturation deficit in environment 
          zrnew = MAX ( 0.0_ireals, SQRT( zrfl(i)/cu_frr) -  &
                                   cu_evap(k)*(pph(i,k+1)-pph(i,k)) &
                                   *MAX(0.0_ireals, zqsen(i,k)-pqv(i,k)) )
          zrnew = (zrnew**2) * cu_frr

          ! allow a maximum evaporation of 95% of humidity in precipitating area
          zrmin = zrfl(i) - cu_frr*MAX( 0.0_ireals,                            &
!  Change the humidity threshold for evaporation of convective precipitation.
!  Use the value of 0.80 as recommended by ECMWF.
               0.80_ireals*zqsen(i,k)-pqv(i,k) ) *z1dgdt*(pph(i,k+1)-pph(i,k))
          zrnew       = MAX ( zrnew, zrmin )
          zrfln       = MAX ( zrnew, 0.0_ireals   )
          zdmfup(i,k) = zdmfup(i,k) + zrfln - zrfl(i)
          zrfl(i)     = zrfln
        ENDIF
      ENDDO
    ENDDO

  ENDIF     ! Test for necessity of evaporation calculations

 
!******************************************************************************
!*                                                                            *
!* Section 8: Compute the final tendencies for grid scale variables T and qv  *
!*            (former subroutine cu_dtdq)                                     *
!*                                                                            *
!******************************************************************************

! preset tendencies of T and qv as well as melting level:
  DO i = isc, iec
    mtmelt(i)= kdim
  ENDDO

!_cdm Notice that "mtomm2=2"
  DO k = mtopm2, kdim - 1  ! above lowest model layer
    DO i = isc, iec
      IF(locum(i)) THEN ! for convective grid points only

        ! Modification to account for the water-ice mixed phase.
        ! Water fraction as function of temperature of the environment air
        IF (pt(i,k).LE.Tmpmin) THEN
          fr_wat = 0._ireals
        ELSE IF (pt(i,k).GE.Tmpmax) THEN
          fr_wat = 1._ireals
        ELSE
          fr_wat = ((pt(i,k)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
        ENDIF
        ! Effective latent heat of evaporation/sublimation for the mixed phase
        lh_mp = fr_wat*lh_v + (1._ireals-fr_wat)*lh_s

        ! Store detrained condensate as tendencies for further use 
        ! by the other schemes.
        ! The fractions of liquid water and of ice in the mixed-phase convective cloud
        ! is determined in a way consistent with the saturation adjustment.
        pqct_con(i,k) = g/zrhgdz(i,k) * fr_wat             * zlude(i,k)
        pqit_con(i,k) = g/zrhgdz(i,k) * (1._ireals-fr_wat) * zlude(i,k)

        pdt_con(i,k) =    g/zrhgdz(i,k)/cp_d  &
                      * (  zmfus (i,k+1) - zmfus (i,k) &
                         + zmfds (i,k+1) - zmfds (i,k) &
                         - lh_mp*( zmful (i,k+1) - zmful(i,k) &
                               -(zdmfup(i,k  ) + zdmfdp(i,k)) ) )

        ! "pdt_con" no longer contains contribution due to detrained condensate.
        pdt_con(i,k) =   pdt_con(i,k) + lh_mp/cp_d*(pqct_con(i,k)+pqit_con(i,k))

        pdqv_con(i,k)=   g/zrhgdz(i,k)                   &
                       *(  zmfuq(i,k+1) - zmfuq(i,k)   &
                         + zmfdq(i,k+1) - zmfdq(i,k)   &
                         + zmful(i,k+1) - zmful(i,k)   &
                         -(zdmfup(i,k) + zdmfdp(i,k)) )

        ! "pdqv_con" no longer contains contribution due to detrained condensate.
        pdqv_con(i,k)=   pdqv_con(i,k) - ( pqct_con(i,k) + pqit_con(i,k) )

        ll2 = pfif(i,k)-pfih(i,kdim+1) < 3000.0 .AND. mtmelt(i)== kdim
        IF ( ll2 ) THEN
          mtmelt(i)=k    ! set level of about 300m above ground
        ENDIF
      END IF
#ifdef MESSY
      IF (pt(i,k).gt.ctmelt) THEN
         cvprecnew_2d(i,k) = zdmfup(i,k)+zdmfdp(i,k)
      ELSE
         cvsnownew_2d(i,k) = zdmfup(i,k)+zdmfdp(i,k)
      END IF
#endif 
    END DO
  ENDDO

  k = kdim      ! special case: lowest model layer

  DO i=isc,iec
    IF(locum(i)) THEN      ! for convective grid points only
      ! Modification to account for the water-ice mixed phase.
      ! Water fraction as function of temperature
      IF (pt(i,k).LE.Tmpmin) THEN
        fr_wat = 0._ireals
      ELSE IF (pt(i,k).GE.Tmpmax) THEN
        fr_wat = 1._ireals
      ELSE
      fr_wat = ((pt(i,k)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
      ENDIF

      ! Effective latent heat of evaporation/sublimation for the mixed phase
      lh_mp = fr_wat*lh_v + (1._ireals-fr_wat)*lh_s

      ! Store detrained condensate as tendencies for further use in the microphyscics scheme.
      ! The fractions of liquid water and of ice in the mixed-phase convective cloud
      ! is determined in a way consistent with the saturation adjustment.
      pqct_con(i,k) = g/zrhgdz(i,k) * fr_wat             * zlude(i,k)
      pqit_con(i,k) = g/zrhgdz(i,k) * (1._ireals-fr_wat) * zlude(i,k)
      pdt_con(i,k) = - g /zrhgdz(i,k)/cp_d  &
                     * ( zmfus(i,k) + zmfds(i,k)    &
                    -lh_mp*(zmful(i,k) + zdmfup(i,k) + zdmfdp(i,k)) )

      ! "pdt_con" no longer contains contribution due to detrained liquid water.
      pdt_con(i,k) =   pdt_con(i,k) + lh_mp/cp_d*(pqct_con(i,k)+pqit_con(i,k))
      pdqv_con(i,k)= - g /zrhgdz(i,k)                &
                     * ( zmfuq(i,k) + zmfdq(i,k)    &
                       + zmful(i,k) + zdmfup(i,k) + zdmfdp(i,k) )
      ! "pdqv_con" no longer contains contribution due to detrained liquid water.
      pdqv_con(i,k)=   pdqv_con(i,k) - ( pqct_con(i,k) + pqit_con(i,k) )
    END IF
#ifdef MESSY
  IF (pt(i,k).gt.ctmelt) THEN
     cvprecnew_2d(i,k) = zdmfup(i,k)+zdmfdp(i,k)
  ELSE
     cvsnownew_2d(i,k) = zdmfup(i,k)+zdmfdp(i,k)
  END IF
#endif
  END DO

  DO k = mtopm2, kdim ! for all model half levels (except the lower model boundary)
    DO i = isc, iec
      IF (locum(i)) THEN ! for convective grid points only
        ! RA: Computation of the convective buoyant TKE production:
        zcvfl_s  =  zmfus (i,k) + zmfds (i,k)
        zcvfl_q  =  zmfuq (i,k) + zmfdq (i,k)

        pdtke_con(i,k) = MAX( 0.0_ireals, g*r_d/pph(i,k) * &
                         ( zcvfl_s/cp_d + &
                           ztenh(i,k)*zcvfl_q*rvd_m_o/(1.0_ireals-rvd_m_o*zqenh(i,k)) ) )
        ! RA
      END IF
    END DO
  END DO

! convective rain/snow at surface
! the distinction between convective rain or snow depends on whether
! the layer 300m above the surface has a temperature warmer than
! 3 degrees below freezing (see definition of mtmelt above and ll2
! below
!_cdm Think if the following criterium is consistet with the use of the water-ice mixed phase.
  DO i = isc, iec
    llo1 = pt_g(i) >= ctmelt
    ll2  = pt(i,mtmelt(i)) > (ctmelt-3.0)
    IF ( lolp(i) ) THEN   ! for land points
      ll3 = llo1.OR.ll2   ! Ts>0 or warm layer above
    ELSE                  ! for sea points
      ll3 = ll2           ! warm layer above
    ENDIF
    IF ( ll3 ) THEN
      ppr_con(i) = zrfl(i)
      pps_con(i) = 0.0
    ELSE
      ppr_con(i) = 0.0
      pps_con(i) = zrfl(i)
    ENDIF
    ppne_con(i) = zrneva(i)
  END DO


!******************************************************************************
!*                                                                            *
!* Section 9: Compute the final tendencies for grid scale variables u and v   *
!*            and cgas, caero and cpolle                                      *
!*            (former subroutine cu_dudv)                                     *
!*                                                                            *
!******************************************************************************

  ! mass fluxes of tracers and corresponding tendencies
#ifndef MESSY
  DO iztr = 1, nztr
    DO k=mtopm2, kdim
!CDIR NODEP
      DO i = isc, iec
        IF( locum(i) ) THEN
          zmfutr(i,k,iztr) =  zmfu(i,k) * ( ztru(i,k  ,iztr)    &
                                         - ptrcr(i,k-1,iztr) )
          zmfdtr(i,k,iztr) =  zmfd(i,k) * ( ztrd(i,k  ,iztr)    &
                                         - ptrcr(i,k-1,iztr) )
        ENDIF
      ENDDO
    ENDDO

    DO k = mtopm2, kdim
!CDIR NODEP
      DO i = isc, iec
        IF( locum(i) .AND. k > mbas_con(i) ) THEN
          mkb = mbas_con(i)
          zzp = (pph(i,kdim+1)-pph(i,k)) / (pph(i,kdim+1)-pph(i,mkb))
          IF ( mtype(i).EQ.3 ) THEN
            zzp = zzp**2
          ENDIF
          zmfutr(i,k,iztr) =  zmfutr(i,mkb,iztr)*zzp
        ENDIF
      ENDDO
    ENDDO

    DO k = mtopm2, kdim-1         ! layers above lowest layer
!CDIR NODEP
      DO i = isc, iec
        IF( locum(i) ) THEN   ! convective grid points only
          ptrcr_con(i,k,iztr) = g/zrhgdz(i,k) * &
                 ( zmfutr(i,k+1,iztr)-zmfutr(i,k,iztr)+zmfdtr(i,k+1,iztr) &
                  -zmfdtr(i,k,iztr))
        ENDIF
      ENDDO
    ENDDO
!CDIR NODEP
    DO i = isc, iec
      IF( locum(i) ) THEN   ! convective grid points only
        ptrcr_con(i,kdim,iztr) = - g/zrhgdz(i,kdim) *                     &
                                 ( zmfutr(i,kdim,iztr) + zmfdtr(i,kdim,iztr) )
      END IF
    ENDDO
  ENDDO
#endif


  IF (lmfdudv) THEN      ! switch for in-/exclusion of cumulus friction


  ! mass fluxes of u and v and corresponding tendencies
    DO k = mtopm2, kdim
      DO i = isc, iec
        IF( locum(i) ) THEN
          zmfuu(i,k) = zmfu(i,k) * ( zuu(i,k) - pu(i,k-1) )
          zmfuv(i,k) = zmfu(i,k) * ( zvu(i,k) - pv(i,k-1) )
          zmfdu(i,k) = zmfd(i,k) * ( zud(i,k) - pu(i,k-1) )
          zmfdv(i,k) = zmfd(i,k) * ( zvd(i,k) - pv(i,k-1) )
        ENDIF
      ENDDO
    ENDDO

    DO k = mtopm2, kdim
      DO i = isc, iec
        IF( locum(i) .AND. k > mbas_con(i) ) THEN
          mkb = mbas_con(i)
          zzp = (pph(i,kdim+1)-pph(i,k)) / (pph(i,kdim+1)-pph(i,mkb))
          IF ( mtype(i).EQ.3 ) THEN
            zzp = zzp**2
          ENDIF
          zmfuu(i,k) = zmfuu(i,mkb)*zzp
          zmfuv(i,k) = zmfuv(i,mkb)*zzp
        ENDIF
      ENDDO
    ENDDO

    DO k = mtopm2, kdim-1         ! layers above lowest layer
      DO i = isc, iec
        IF( locum(i) ) THEN   ! convective grid points only
          pdu_con(i,k) =  g/zrhgdz(i,k) * &
                          ( zmfuu(i,k+1)-zmfuu(i,k)+zmfdu(i,k+1)-zmfdu(i,k) )
          pdv_con(i,k) =  g/zrhgdz(i,k) * &
                          ( zmfuv(i,k+1)-zmfuv(i,k)+zmfdv(i,k+1)-zmfdv(i,k) )
        ENDIF
      ENDDO
    ENDDO

    k = kdim   ! lowest layer
    DO i = isc, iec
      IF( locum(i) ) THEN   ! convective grid points only
        pdu_con(i,k) = - g/zrhgdz(i,k) * ( zmfuu(i,k) + zmfdu(i,k) )
        pdv_con(i,k) = - g/zrhgdz(i,k) * ( zmfuv(i,k) + zmfdv(i,k) )
      END IF
    ENDDO

  ENDIF   ! switch for in-/exclusion of cumulus friction
 
#ifdef MESSY
  ! rescue fluxes for CVTRANS
  DO k = mtopm2, kdim
     DO i = isc, iec
        massfu_2d(i,k) = zmfu(i,k)
        massfd_2d(i,k) = zmfd(i,k)
     ENDDO
  ENDDO
#endif
 
!------------------------------------------------------------------------------
! End of the subroutine
!------------------------------------------------------------------------------
END SUBROUTINE cu_tied

!==============================================================================

!+ Module procedure in "Convection" 
!------------------------------------------------------------------------------

SUBROUTINE cu_asc  (                                              &
           ppf    , pt      , pqv    , ptr   , pu     , pv   ,    &
           pqsen  , pfif    ,                                     &
           pph    , pth     , pqvh   , pfih  ,                    &
           pdqvdt , pw      , pdqpbl , pzumflx,                   &
           klwmin , ktype   , klab   , lolp  , locum  ,           &
           idim   , kdim    , isc    , iec   , nztr   ,           &
           pt_u   , pqv_u   , ptr_u  , plu   , pu_u   , pv_u ,    &
           pmf_u  , pmf_ub  , pmf_us , pmf_uq, pmf_ul , plude,    &
           pdmfup , pentr   ,                                     &
           mbas_con, mtop_con, mtop_con0, kcum )

!------------------------------------------------------------------------------
!
! Description:
!
!   The module procedure cu_asc performs cloud ascent calculations for   
!   the cumulus upddrafts to obtain the vertical in-cloud profiles of
!   T, q, l, u and v as well as the precipitation fluxes.
!
! Method:
!   Surface air is lifted dry-adiabatically to cloud base from there, a moist 
!   ascent is calculated for an entraining/detraining plume; different 
!   entrainement/detrainement rates are specified for different types
!   of convection (deep/shallow) in cases where neither deep nor shallow 
!   convection  occurs, a check for possible existence of mid-level
!   convection is made
!
!------------------------------------------------------------------------------

! Subroutine arguments:
! --------------------

! Input data
! ----------
  INTEGER (KIND=iintegers), INTENT (IN) ::  &
     idim ,       & ! array dimension in zonal direction
     kdim ,       & ! array dimension in vertical direction 
     isc  ,       & ! start index for first  array computation
     iec  ,       & ! end   index for first  array computation
     nztr

  REAL    (KIND=ireals   ), INTENT (IN) ::  &
     pt     (idim,kdim),     & ! temperature at full levels
     pth    (idim,kdim),     & ! temperature at half levels
     pqv    (idim,kdim),     & ! specific humidity at full levels
     pqvh   (idim,kdim),     & ! specific humidity at half levels
     ptr    (idim,kdim,nztr),& ! tracers at full levels
     pu     (idim,kdim),     & ! zonal wind component at full levels
     pv     (idim,kdim),     & ! meridional wind component at full levels
     pw     (idim,kdim),     & ! vertical velocity of z-system
     pph    (idim, kdim+1),  & ! pressure at half levels 
     ppf    (idim,kdim  ),   & ! pressure at full levels 
     pfif   (idim,kdim  ),   & ! geopotential at full levels
     pfih   (idim,kdim  ),   & ! geopotential at half levels
     pqsen  (idim,kdim  ),   & ! sat. specific humidity     
     pdqvdt (idim,kdim  ),   & ! humidity tendency          
     pdqpbl (idim       )      ! pbl moisture convergence

  REAL    (KIND=ireals   ), INTENT (INOUT) ::  &
     pmf_ub (idim     ), &    ! upward mass flux at cloud base
     pmf_u  (idim,kdim), &    ! upward mass flux
     pmf_us (idim,kdim), &    ! upward mass flux *s
     pmf_uq (idim,kdim), &    ! upward   * qv
     pmf_ul (idim,kdim), &    ! upward   * ql
     pzumflx(idim     ), &    !
     plude  (idim,kdim), &    ! 
     pdmfup (idim,kdim), &    !
     pentr  (idim     ), &    ! entrainment 
     pt_u   (idim,kdim), &    !
     pqv_u  (idim,kdim), &    !
     plu    (idim,kdim), &    !
     ptr_u  (idim,kdim,nztr),&!   
     pu_u   (idim,kdim), &    ! 
     pv_u   (idim,kdim)       !

  INTEGER (KIND=iintegers), INTENT (IN) ::  &
     klwmin (idim)        !
  INTEGER (KIND=iintegers), INTENT (INOUT) ::  &
     ktype  (idim),    &  !
     mbas_con(idim),   &  ! indicator for convection (on/off) at gridpoints
     mtop_con(idim),   &  ! indicator for convection (on/off) at gridpoints
     mtop_con0(idim),  &  ! indicator for convection (on/off) at gridpoints
     klab(idim,kdim)      ! indicator for convection (on/off) at gridpoints

  LOGICAL                 , INTENT (IN) ::  &
     lolp (idim)          ! land-sea indicator
  LOGICAL                 , INTENT (INOUT) ::  &
     locum(idim)          ! indicator for convection (on/off) at gridpoints
  
! Output data
! -----------
  INTEGER (KIND=iintegers), INTENT (OUT) ::  &
     kcum                    ! number of convective gridpoints

! Local parameters: 
! ----------------
  REAL    (KIND=ireals   ), PARAMETER ::  &
     ctmelt = 273.16         ! tripel point

! Local scalars and automatic arrays:
! ----------------------------------
  INTEGER (KIND=iintegers) ::  &
    i, k , iztr,           & ! loop indices over spatial dimensions
    msum, iklwmin,         & !
    isp                      ! number of chem. types (COSMO_ART)

  REAL    (KIND=ireals   ) ::  &
    zdmfen (idim),         & !
    zdmfde (idim),         & !
    zmfutr (idim,nztr),    & !
    zmfuu  (idim),         & !
    zmfuv  (idim),         & !
    zpbase (idim),         & !
    zptop  (idim),         & !
    zqold  (idim),         & !
    zcond  (idim),         & !
    z1dp   (idim),         & !
    zrho   (idim),         & !
    zdmfmax,zzzmb,zdprho,zentr,zpmid,zqeen,zseen,zscde,zqude, & !
    zmfusk,zmfuqk,zmfulk,     & !
    zbuo,zprcon,zlnew,        & !
    zzdmf,zdnoprc, pentr_pb     !

  LOGICAL                  ::  &
  llo1,llo2,                   & !
  loflag (idim)
 
!------------ End of header ---------------------------------------------------

!------------------------------------------------------------------------------
! Begin Subroutine cu_asc              
!------------------------------------------------------------------------------
 
! Security parameter
  zdmfmax = cmfcmax / REAL (kdim/4, ireals)
 
! Set defaults

  DO i = isc, iec
    IF( .NOT.locum(i) ) ktype(i) = 0   ! type of convection
  ENDDO

  DO k = 1, kdim
    DO i = isc, iec
      plu    (i,k) = 0.
      pmf_u  (i,k) = 0.
      pmf_us (i,k) = 0.
      pmf_uq (i,k) = 0.
      pmf_ul (i,k) = 0.
      ! Detrained cloud condensate is set to zero at all levels
      plude  (i,k) = 0.
      pdmfup (i,k) = 0.
    ENDDO
  ENDDO
 
  DO i = isc, iec
    IF (.NOT.locum(i) .OR. ktype(i) == 3 ) THEN
      DO k = 1, kdim 
        klab(i,k) = 0
      ENDDO
    ENDIF
    IF (.NOT.locum(i)) THEN
      DO k = 1, kdim 
       IF( pph(i,k) < 4.0E4 ) mtop_con0(i) = k
      ENDDO
    ENDIF
  ENDDO
 
! Initialize values at lifting level
 
  DO i = isc, iec
    mtop_con(i) = kdim - 1
    IF( .NOT.locum(i) ) THEN
      mbas_con(i)   = kdim-1
      pmf_ub(i)     = 0.
      pqv_u(i,kdim) = 0.
    ENDIF
    pmf_u (i,kdim) = pmf_ub(i)
    pmf_us(i,kdim) = pmf_ub(i)*( cp_d*pt_u(i,kdim) + pfih(i,kdim) )
    pmf_uq(i,kdim) = pmf_ub(i)*pqv_u(i,kdim)
    IF(lmfdudv) THEN
      zmfuu(i) = pmf_ub(i)*pu_u(i,kdim)
      zmfuv(i) = pmf_ub(i)*pv_u(i,kdim)
    ENDIF
  ENDDO

#ifndef MESSY
  DO iztr = 1, nztr
    DO i = isc, iec
      zmfutr(i,iztr)    = pmf_ub(i)* ptr_u(i,kdim,iztr)
    ENDDO
  ENDDO
#endif

  DO i = isc, iec
    locum(i) = .FALSE.    ! reset convection switch
  ENDDO
 
  ! Perform ascent:  a) dry adiabatic lifting and 
  !                  b) allowing for condensation
  !                  c) check buoancy and set flags
  !                     (klab=1 :sub-cloud layer, klab=2 :cloud layer)

  !_cdm Begin loop over vertical levels
  DO k = kdim-1, 2, -1
    ! for model layers in the 'lower' atmosphere:
    ! if neither shallow nor deep convection exists, specify cloud base
    ! values for mid-level convection

    IF( lmfmid .AND. k < kdim-1 .AND. k > kdim/2 ) THEN
      DO i = isc, iec
        IF( .NOT. locum(i) .AND. klab(i,k+1) == 0  & !'inactive grid point'
            .AND. pqv(i,k) > 0.90*pqsen(i,k)       & !'moist environment'
!           avoid 'convective drizzle' by using a minimum grid-scale lifting  
!           .AND. pw(i,k) > 0.0) THEN                !'grid scale lifting'
            .AND. pw(i,k) > (1.E-3*r_d*pt(i,k)/ppf(i,k))) THEN
          pt_u  (i,k+1) = (cp_d*pt(i,k) + pfif(i,k) - pfih(i,k+1) )/cp_d  
          pqv_u (i,k+1) = pqv(i,k)
          plu   (i,k+1) = 0.0
          zzzmb = MAX( cmfcmin, pw(i,k)*ppf(i,k)/(r_d*pt(i,k)) )
          zzzmb = MIN( zzzmb, cmfcmax )
          pzumflx(i)     = zzzmb !Massflux of midlevel convection for global field
          pmf_ub (i)     = zzzmb
          pmf_u  (i,k+1) = pmf_ub(i)
          pmf_us (i,k+1) = pmf_ub(i) *(cp_d*pt_u(i,k+1)+pfih(i,k+1))
          pmf_uq (i,k+1) = pmf_ub(i)*pqv_u(i,k+1)
          pmf_ul (i,k+1) = 0.0
          pdmfup(i,k+1)  = 0.0
          mbas_con (i)   = k
          klab  (i,k+1)  = 1
          ktype (i)      = 3
          pentr (i)      = entrmid

          IF(lmfdudv) THEN                     
            pu_u  (i,k+1) = pu(i,k)
            pv_u  (i,k+1) = pv(i,k)
            zmfuu(i)      = pmf_ub(i)*pu_u(i,k+1)
            zmfuv(i)      = pmf_ub(i)*pv_u(i,k+1)
          ENDIF
        ENDIF       ! 'suitability' test for mid-level conv.
      ENDDO

#ifndef MESSY 
      DO iztr = 1, nztr
!CDIR NODEP
        DO i = isc, iec
          IF( .NOT. locum(i) .AND. klab(i,k+1) == 0  & !'inactive grid point'
              .AND. pqv(i,k) > 0.90*pqsen(i,k)       & !'moist environment'
!             avoid 'convective drizzle' by using a minimum grid-scale lifting  
!             .AND. pw(i,k) > 0.0) THEN                !'grid scale lifting'
              .AND. pw(i,k) > (1.E-3*r_d*pt(i,k)/ppf(i,k))) THEN
            ptr_u (i,k+1,iztr)= ptr(i,k,iztr)
            zmfutr(i,iztr)    = pmf_ub(i)*ptr_u(i,k+1,iztr)
          ENDIF
        ENDDO
      ENDDO
#endif
    ENDIF      ! 'lower' atmosphere test
 
    DO i = isc, iec
      IF( klab(i,k+1) == 0 ) klab(i,k) = 0
      IF( klab(i,k+1) >  0 ) THEN
        loflag(i)= .TRUE.   ! active grid point                            
      ELSE
        loflag(i)= .FALSE.  ! inactive grid point                          
      ENDIF
    ENDDO

    ! calculation of entrainement/detrainement rates
    DO i = isc, iec
      zdmfen(i) = 0.
      zdmfde(i) = 0.
      zrho  (i) = pph(i,k+1)/(r_d*pth(i,k+1))
      zpbase(i) = pph(i,mbas_con(i))
      zptop (i) = pph(i,mtop_con0(i))
    ENDDO
 
    DO i = isc, iec
      IF( locum(i) ) THEN
        zdprho = ( pfih(i,k) - pfih(i,k+1) ) / g
!DM+AS> Incorporate the new dependence of turbulent D/E
!    on mass flux and temperature difference,
!!        ! ... tuning constant times buoyancy param.
!!        pentr_pb = entr_pb*g/(0.5_ireals*(pth(i,k+1)+pt_u(i,k+1))) &
!!        ! ... times updraught-environment temperature difference
!!          &      * (pt_u(i,k+1)-pth(i,k+1))                        &
!!        ! ... divided by square of mass flux over density
!!          &      / (MAX(cmfcmin, pmf_u(i,k+1))/zrho(i))**2
!!        ! ... take the correction term to be non-negative
!!        pentr_pb = MAX(0._ireals, pentr_pb)
!!        ! ... add correction term to "pentr"
!!        zentr = (pentr(i)+pentr_pb)*pmf_u(i,k+1)*zdprho
        zentr  = pentr(i)*pmf_u(i,k+1)*zdprho
!<DM+AS
        llo1   = k < mbas_con(i)
        IF(llo1) zdmfde(i) = zentr
        zpmid  = 0.5*( zpbase(i) + zptop(i) )
        llo2  = llo1 .AND. ktype(i)==2 .AND. &
                ( zpbase(i)-pph(i,k) < 0.2E5 .OR. pph(i,k) > zpmid )
        IF(llo2) zdmfen(i) = zentr
        iklwmin = MAX( klwmin(i), mtop_con0(i)+2 )
        llo2 = llo1 .AND. ( ktype(i)==1 .OR. ktype(i)==3 ) .AND. &
               ( k>=iklwmin .OR. ppf(i,k) > zpmid )
        IF(llo2) zdmfen(i) = zentr
        IF(llo2 .AND. pqvh(i,k+1) > 1.E-5) THEN
          zdmfen(i) = zentr + MAX(pdqvdt(i,k),0.0_ireals) /      &
                                  pqvh(i,k+1)*zrho(i)*zdprho
        ENDIF
      ENDIF
    ENDDO
 
    ! adiabatic ascent for entraining/detraining plume

    DO i = isc, iec
      z1dp(i)    = 1.0/pph(i,k)
      IF(loflag(i)) THEN
        zdmfen(i) = MIN( zdmfen(i), zdmfmax )
        zdmfde(i) = MIN( zdmfde(i), 0.75_ireals*pmf_u(i,k+1) )
        pmf_u(i,k)= pmf_u(i,k+1) + zdmfen(i) - zdmfde(i)
        zqeen = pqvh(i,k+1)*zdmfen(i)
        zseen = ( cp_d*pth (i,k+1) + pfih(i,k+1) )*zdmfen(i)
        zscde = ( cp_d*pt_u(i,k+1) + pfih(i,k+1) )*zdmfde(i)
        zqude = pqv_u(i,k+1)*zdmfde(i)
        ! Detrained cloud condensate stored in "plude"
        plude(i,k) = plu(i,k+1)*zdmfde(i)
        zmfusk = pmf_us(i,k+1) + zseen - zscde
        zmfuqk = pmf_uq(i,k+1) + zqeen - zqude
        zmfulk = pmf_uL(i,k+1) - plude(i,k)
        plu  (i,k) =  zmfulk*( 1./MAX(cmfcmin,pmf_u(i,k)) )
        pqv_u(i,k) =  zmfuqk*( 1./MAX(cmfcmin,pmf_u(i,k)) )
        pt_u (i,k) = (zmfusk*( 1./MAX(cmfcmin,pmf_u(i,k)) ) -      &
                                  pfih(i,k))/cp_d  
        pt_u(i,k)  = MAX( 100.0_ireals, pt_u(i,k) )
        pt_u(i,k)  = MIN( 400.0_ireals, pt_u(i,k) )
        zqold(i)   = pqv_u(i,k)
      ENDIF
    ENDDO
 
#ifdef MESSY
    ! rescue fluxes for CVTRANS
    DO i = isc, iec
       !massfu_2d(:,k) = pmf_u(:,k)
       udetr_2d(i,k)  = zdmfde(i)
       uentr_2d(i,k)  = zdmfen(i)
    ENDDO
#endif

    ! corrections for moist ascent by adjusting T,q and l
    ! calulation of condensation and corresponding adjustment of T and q

    ! CALL of cu_cond with "condensation only flag = TRUE"
    CALL cu_cond ( pt_u(:,k), pqv_u(:,k), z1dp  ,              &
                   loflag , .TRUE.  , .FALSE.,                 &
                   idim   , isc     , iec            )

    DO i = isc, iec
      IF( loflag(i)) THEN
        IF( pqv_u(i,k).NE.zqold(i) ) THEN
          klab(i,k) = 2
          plu (i,k) = plu(i,k) + zqold(i) - pqv_u(i,k)
#ifdef MESSY
          IF (pt(i,k) > ctmelt) THEN
             cvlwc_2d(i,k) = plu (i,k)
          ELSE
             cviwc_2d(i,k) = plu (i,k)
          ENDIF
#endif
          zbuo      =  pt_u(i,k)*(1.+rvd_m_o*pqv_u(i,k))  &
                      - pth(i,k)*(1.+rvd_m_o*pqvh(i,k))
          IF( klab(i,k+1) == 1 ) zbuo = zbuo + 0.5
          IF( zbuo > 0.0 .AND. pmf_u(i,k) >= 0.1*pmf_ub(i) ) THEN
            mtop_con(i) = k
            locum(i)    = .TRUE.
!           avoid 'convective drizzle' by changing the precipitation computation
!           zdnoprc     = 1.0E4
!           IF ( zpbase(i)-pph(i,k) < zdnoprc ) THEN
            zdnoprc     = 2.0E4
            IF ( zpbase(i)-zptop(i) < zdnoprc ) THEN
              zprcon= 0.0
            ELSE
              zprcon= cprcon
            ENDIF
            zlnew = plu(i,k)/( 1.0 + zprcon*(pfih(i,k)-pfih(i,k+1)) )
            pdmfup(i,k) = MAX( 0.0_ireals, (plu(i,k)-zlnew)*pmf_u(i,k) )
#ifdef MESSY
            IF (pt(i,k) > ctmelt) THEN
               cvrform_2d(i,k) = MAX(0._ireals,(plu(i,k)-zlnew))
            ELSE
               cvsform_2d(i,k) = MAX(0._ireals,(plu(i,k)-zlnew))
            ENDIF
#endif
            plu(i,k)    = zlnew
          ELSE
            klab(i,k)  = 0
            pmf_u(i,k) = 0.0
          ENDIF
        ENDIF
      ENDIF
    ENDDO

    DO i = isc, iec
      IF( loflag(i) ) THEN
        pmf_ul(i,k) = plu(i,k)*pmf_u(i,k)
        pmf_us(i,k) = ( cp_d*pt_u(i,k) + pfih(i,k) )*pmf_u(i,k)
        pmf_uq(i,k) = pqv_u(i,k)*pmf_u(i,k)
      ENDIF
    ENDDO

#ifndef MESSY
    DO iztr = 1, nztr
!CDIR NODEP
      DO i = isc, iec
        IF( loflag(i) ) THEN
          zmfutr(i,iztr) = zmfutr(i,iztr) + zdmfen(i)*ptr (i,k,iztr)  &
                                    - zdmfde(i)*ptr_u (i,k+1,iztr)
          IF( pmf_u(i,k) > 0.0 ) THEN
            ptr_u(i,k,iztr) = zmfutr(i,iztr)*(1./pmf_u(i,k))
          ENDIF
        ENDIF
      ENDDO
    ENDDO
#endif

    IF(lmfdudv) THEN
      DO i = isc, iec
        IF( loflag(i) ) THEN
          zmfuu(i) = zmfuu (i) + zdmfen(i)*pu(i,k)  &
                               - zdmfde(i)*pu_u (i,k+1)
          zmfuv(i)= zmfuv (i)  + zdmfen(i)*pv(i,k)  &
                               - zdmfde(i)*pv_u(i,k+1)
          IF( pmf_u(i,k) > 0.0 ) THEN
            pu_u(i,k) = zmfuu(i)*(1./pmf_u(i,k))
            pv_u(i,k) = zmfuv(i)*(1./pmf_u(i,k))
          ENDIF
        ENDIF
      ENDDO
    ENDIF
 
  ENDDO         ! vertical loop
 
!     convective fluxes above non-buoancy level
!     
!        (NOTE: CLOUD VARIABLES LIKE T,Q and L ARE NOT
!               AFFECTED BY DETRAINMENT and ARE ALREADY KNOWN
!               FROM PREVIOUS CALCULATIONS ABOVE)
 
  DO i = isc, iec 
    IF( mtop_con(i) == kdim-1 ) locum(i) = .FALSE.  
    mbas_con(i) = MAX( mbas_con(i), mtop_con(i) )
  ENDDO

  msum = 1  !! Must be set to one, otherwise errors in parallel version
  DO i = isc, iec
    IF ( locum(i) ) msum = msum + 1
  ENDDO

  kcum = msum             ! store number of convective grid points
  ! Can "msum=0" ever occur with the above "msum=1 parallele version fix" ?
  IF ( msum == 0 ) RETURN ! Exit the subroutine in case of no convection

  DO i = isc, iec
    IF( locum(i) ) THEN
      k          = mtop_con(i) - 1
      zzdmf      = cmfctop
      zdmfde(i)  = ( 1.0 - zzdmf )*pmf_u(i,k+1)
      plude(i,k) = zdmfde(i)*plu(i,k+1)
      pmf_u(i,k) = pmf_u(i,k+1) - zdmfde(i)
      pmf_us(i,k)  = ( cp_d*pt_u(i,k) + pfih(i,k) )*pmf_u(i,k)
      pmf_uq(i,k)  = pqv_u(i,k)*pmf_u(i,k)
      pmf_ul(i,k)  = plu(i,k)*pmf_u(i,k)
      ! Detrained cloud condensate stored in "plude"
      plude(i,k-1) = pmf_ul(i,k)
    ENDIF
  ENDDO

#ifndef MESSY
  DO iztr = 1, nztr
!CDIR NODEP
    DO i = isc, iec
      IF( locum(i) ) THEN
        k            = mtop_con(i) - 1
        ptr_u(i,k,iztr) = ptr_u(i,k+1,iztr)
      ENDIF
    ENDDO
  ENDDO
#endif

  IF(lmfdudv) THEN
    DO i = isc, iec
      IF( locum(i) ) THEN
        k = mtop_con(i) - 1
        pu_u(i,k) = pu_u(i,k+1)
        pv_u(i,k) = pv_u(i,k+1)
      ENDIF
    ENDDO
  ENDIF

!_cdm At this point, "plude" contanis detrained cloud condensate for all levels.
!     It has the same dimensions as "pmf_ul".

!------------------------------------------------------------------------------
! End of the subroutine
!------------------------------------------------------------------------------
END SUBROUTINE cu_asc 

!==============================================================================

!+ Module procedure in "Convection" 
!------------------------------------------------------------------------------

SUBROUTINE cu_ddraf (                                                 &
           pu     , pv      , pph    , pth    , pqvh   , ptr  , pfih ,&
           pt     ,                                                   &
           idim   , kdim    , isc    , iec    , nztr   ,              &
           pt_d   , pqv_d   , ptr_d  , pu_d   , pv_d   , pvddraf,     &
           pmf_d  , pmf_ds  , pmf_dq , pdmfdp, prfl   ,               &
           loddraf  )

!------------------------------------------------------------------------------
!
! Description:
!
!   The module procedure cu_ddraf performs downdraft calculations
!
!   Method:    a moist descent for an entraining/detraining plume is
!              calculated by
!              a) moving air dry-adiabatically to the next level below
!              b) correcting for evaporation to obtain a saturated state
!                 the output of the routine are fluxes of s, q and the
!                 evaporation rate and u,v at levels where downdrafts occur.
!
!------------------------------------------------------------------------------
!
! Declarations:
!
!------------------------------------------------------------------------------
! Subroutine arguments:
! --------------------

! Input data
! ----------
  INTEGER (KIND=iintegers), INTENT (IN) ::  &
     idim ,       & ! array dimension in zonal direction
     kdim ,       & ! array dimension in vertical direction 
     isc  ,       & ! start index for first  array computation
     iec  ,       & ! end   index for first  array computation
     nztr           ! number of tracers for convection

  REAL    (KIND=ireals   ), INTENT (IN) ::  &
     pu     (idim,kdim),     & ! zonal wind component
     pv     (idim,kdim),     & ! meridional wind component
     pph    (idim,kdim+1),   & ! pressure at half levels 
     pth    (idim,kdim),     & ! temperature at half levels
     pqvh   (idim,kdim),     & ! specific humidity at half levels
     ptr    (idim,kdim,nztr),& ! tracers at full levels
     pfih   (idim,kdim),     & ! geopotential at half levels
     pt     (idim,kdim)        ! temperature at full levels

  LOGICAL                 , INTENT (INOUT) ::  &
     loddraf(idim)          ! switch for downdrafts
  
! Output data
! -----------
  REAL    (KIND=ireals   ), INTENT (INOUT) ::  &
     pt_d   (idim, kdim),       & ! downdraft temperature
     pqv_d  (idim, kdim),       & ! downdraft specific humidity
     ptr_d  (idim, kdim, nztr), & ! downdraft tracers
     pu_d   (idim, kdim),       & ! downdraft u-velocity
     pv_d   (idim, kdim),       & ! downdraft v-velocity
     pvddraf(idim      ),       & ! maximum possible convective gust
     prfl   (idim      ),       & ! total precipitation flux
                                  !     various mass fluxes
     pmf_d   (idim, kdim),      & !
     pmf_ds  (idim, kdim),      & ! 
     pmf_dq  (idim, kdim),      & !
     pdmfdp  (idim, kdim)         !

! Local scalars and automatic arrays:
! ----------------------------------
  INTEGER (KIND=iintegers) ::  &
    i, k , iztr,           & ! loop indices over spatial dimensions
    msum, mtopde,          & ! melting level
    isp                      ! number of chem types (COSMO_ART)

  REAL    (KIND=ireals   ) ::  &
    zdmfen (idim),         & !
    zdmfde (idim),         & !
    zcond  (idim),         & !
    z1dp   (idim),         & !
    zentr, zseen, zqeen, zsdde, zqdde, zmfdsk, zmfdqk, & ! 
    zbuo, zmfduk, zmfdvk, zdmfdp, ztddraf, zqprec,     & !
    zmfdtrk(nztr)

  REAL    (KIND=ireals   ) ::  &
    zvbuo  (idim)            ! gusts generated by buoyancy forces

  LOGICAL                  ::  &
  llo1,                    & !
  llo2 (idim)                !
 
!------------ End of header ---------------------------------------------------

!------------------------------------------------------------------------------
! Begin Subroutine cu_ddraf            
!------------------------------------------------------------------------------

! Initialization:

  zvbuo(:) = 0.0_ireals

! Moist descent: 
! a) entrainement rates assuming linear decrease of mass flux in PBL
! b) consider evaporative cooling and moisting of descending air
! c) check negative buoancy and specify final T,q,u,v and downward fluxes

  DO  k = 3, kdim   ! Vertical loop

    ! check occurrence of downdrafts
    msum = 0
    DO i = isc, iec
      llo2(i) = loddraf(i) .AND. pmf_d(i,k-1) < 0.0
      IF ( llo2(i) ) THEN
        msum =msum + 1
      ENDIF
    ENDDO
    IF (msum.EQ.0) CYCLE   ! no grid point with downdraft found

    DO i = isc, iec
      IF( llo2(i) ) THEN
        zentr = entrdd*pmf_d(i,k-1)*( pfih(i,k-1) - pfih(i,k) )/g
        zdmfen(i) = zentr
        zdmfde(i) = zentr
      ENDIF
    ENDDO

    mtopde = kdim - 2
    IF( k > mtopde ) THEN
      DO i = isc, iec
        IF( llo2(i) ) THEN
          zdmfen(i) = 0.
          zdmfde(i) = pmf_d(i,mtopde)*( pph(i,k) - pph(i,k-1) ) &
                            / ( pph(i,kdim+1) - pph(i,mtopde) )
        ENDIF
      ENDDO
    ENDIF
 
    DO i = isc, iec
      z1dp(i) = 1.0 / pph(i,k)
      IF( llo2(i) ) THEN
        pmf_d(i,k) = pmf_d(i,k-1) + zdmfen(i) - zdmfde(i)
        zseen = ( cp_d*pth(i,k-1) + pfih(i,k-1) )*zdmfen(i)
        zqeen = pqvh(i,k-1)*zdmfen(i)
        zsdde = ( cp_d*pt_d(i,k-1) + pfih(i,k-1) )*zdmfde(i)
        zqdde = pqv_d(i,k-1)*zdmfde(i)
        zmfdsk = pmf_ds(i,k-1) + zseen - zsdde
        zmfdqk = pmf_dq(i,k-1) + zqeen - zqdde
        pqv_d(i,k) =   zmfdqk*(1.0/MIN( -cmfcmin, pmf_d(i,k) ))
        pt_d (i,k) = ( zmfdsk*(1.0/MIN( -cmfcmin, pmf_d(i,k) )) &
                      - pfih(i,k) ) / cp_d  
        pt_d(i,k)  = MIN( 400.0_ireals, pt_d(i,k) )          ! security
        pt_d(i,k)  = MAX( 100.0_ireals, pt_d(i,k) )          ! security
        zcond(i)   = pqv_d(i,k)
      ENDIF
    ENDDO

    CALL cu_cond ( pt_d(:,k), pqv_d(:,k), z1dp  ,              &
                   llo2   , .FALSE.  , .TRUE.,                 &
                   idim   , isc     , iec       )

    DO i = isc, iec
      IF( llo2(i) ) THEN
        zcond(i) = zcond(i) - pqv_d(i,k)
        zbuo =   pt_d(i,k)*( 1.0 + rvd_m_o*pqv_d(i,k) )  &
               - pth (i,k)*( 1.0 + rvd_m_o*pqvh (i,k) )
        llo1 = zbuo < 0.0   & ! negative bouancy in downdraft
               .AND.        & ! condensation < precipitation
               (prfl(i) - pmf_d(i,k)*zcond(i) > 0.0)
        IF (.NOT. llo1 ) pmf_d(i,k) = 0.0
        pmf_ds(i,k) = pmf_d(i,k) * ( cp_d*pt_d(i,k) + pfih(i,k) )
        pmf_dq(i,k) = pqv_d(i,k)*pmf_d(i,k)
        zdmfdp      = - pmf_d(i,k)*zcond(i)
        pdmfdp(i,k-1) = zdmfdp
        prfl(i)       = prfl(i) + zdmfdp
!
! Calculate gusts generated by buoyancy forces
!
        IF (llo1) THEN
          ztddraf    = 0.5_ireals*(pt_d(i,k-1) + pt_d(i,k))
!          zqprec     = prfl(i) / MAX(cmfcmin,-pmf_d(i,k-1))
          zqprec     = 0.0_ireals              ! Neglect rain water, it gives
                                               ! unrealistic high values
          zvbuo(i)   = zvbuo(i) + 2.0_ireals                         &
                     *((pt(i,k-1)-ztddraf   )/pt(i,k-1) + zqprec   ) &
                     *(pfih(i,k-1) - pfih(i,k))
        ENDIF
!
      ENDIF
    ENDDO

#ifndef MESSY
    DO iztr = 1, nztr
!CDIR NODEP
      DO i = isc, iec
        IF( llo2(i) .AND. pmf_d(i,k) < 0.0 ) THEN
          zmfdtrk(iztr)= pmf_d(i,k-1)*ptr_d(i,k-1,iztr) &
                       + zdmfen(i)*ptr(i,k-1,iztr) - zdmfde(i)*ptr_d(i,k-1,iztr)
          ptr_d(i,k,iztr) = zmfdtrk(iztr) / MIN (-cmfcmin, pmf_d(i,k))
        ENDIF
      ENDDO
    ENDDO
#endif

    IF ( lmfdudv ) THEN
      DO i = isc, iec
        IF( llo2(i) .AND. pmf_d(i,k) < 0.0 ) THEN
          zmfduk =   pmf_d(i,k-1)*pu_d(i,k-1) &
                   + zdmfen(i)*pu(i,k-1) - zdmfde(i)*pu_d(i,k-1)
          zmfdvk =   pmf_d(i,k-1)*pv_d(i,k-1) &
                   + zdmfen(i)*pv(i,k-1) - zdmfde(i)*pv_d(i,k-1)
          pu_d(i,k) = zmfduk / MIN ( -cmfcmin, pmf_d(i,k) )
          pv_d(i,k) = zmfdvk / MIN ( -cmfcmin, pmf_d(i,k) )
!
! Calculate downdraft windspeed
! (Neglected for testing)
!
!         IF (k.EQ. kdim) pvddraf(i) = pu_d(i,k)**2 + pv_d(i,k)**2
!
        ENDIF
      ENDDO
    ENDIF

#ifdef MESSY
    ! rescue fluxes for CVTRANS
    DO i = isc, iec
      IF( llo2(i) ) THEN
         dentr_2d(i,k)  = zdmfen(i)
         ddetr_2d(i,k)  = zdmfde(i)
      ENDIF
       !massfd_draf(jl,jk,jrow) = pmfd(jl,jk)
    ENDDO
#endif

  ENDDO   ! end of vertical loop
!
! Calculate the maximum possible convective gust
!
  DO i=isc,iec
    pvddraf(i) = SQRT(0.2_ireals*MAX(zvbuo(i),0.0_ireals) + pvddraf(i))
    pvddraf(i) = MIN(pvddraf(i),30.0_ireals)   ! But do not allow convective
                                               ! gusts higher than 30 m/s
  ENDDO


!------------------------------------------------------------------------------
! End of the subroutine
!------------------------------------------------------------------------------
END SUBROUTINE cu_ddraf

!==============================================================================
!+ Module procedure in "Convection"
!------------------------------------------------------------------------------

SUBROUTINE cu_cond (                                               &
           pt     , pqv     , p1op   , plflag , pcflag , peflag,   &
           idim   , isc     , iec                                  )

!-----------------------------------------------------------------------------!
! Description:
!
!   The module procedure cu_cond does a saturation adjustment for
!   temperature and specific humidity.
!
!   Method:    Thermodynamic adjustment by instantaneous condensation
!              at constant pressure using a double iteration method.
!              Release of latent heat of condendation and of deposition
!              is considered depending on temperature.
!
!------------------------------------------------------------------------------
!
! Declarations:
!
!------------------------------------------------------------------------------
! Subroutine arguments:
! --------------------
! Input data
! ----------
  INTEGER (KIND=iintegers), INTENT (IN) ::  &
     idim ,        & ! array dimension in zonal direction
     isc  ,        & ! start index for first  array computation
     iec             ! end   index for first  array computation

  REAL     (KIND=ireals   ), INTENT (IN) ::  &
     p1op  (idim)    ! reciprocal of pressure, 1.0/p

  LOGICAL                 ,  INTENT (IN) ::  &
     plflag (idim),& ! switch for points where adjustment shall be made
     pcflag,       & ! condensation only (.TRUE)
     peflag          ! evaporation only  (.TRUE)

! Input/Output data
! -----------
  REAL     (KIND=ireals   ), INTENT (INOUT) ::  &
     pt    (idim), & ! temperature on input, adjusted on output
     pqv   (idim)    ! specific humidity on input, adjusted on ouput

! Local scalars and automatic arrays:
! ----------------------------------
  INTEGER (KIND=iintegers) ::  &
    i                ! loop indix

  REAL    (KIND=ireals   ) ::  &
    zcond(idim)      ! condensation amount

  REAL    (KIND=ireals   ) ::  &
    zc3, zc4, zc5, zhldcp, zqsat, zcor, zcond1, zfacc, zface  ! local storage

!DM+AS>
!_nu
!_nu ! Local Parameters
!_nu REAL (KIND=ireals), PARAMETER :: &
!_nu   Tmpmin = 236.15_ireals       , & ! Minium temperature of the mixed-phase temperature range [K]
!_nu   Tmpmax = 267.15_ireals       , & ! Maximum temperature of the mixed-phase temperature range [K]
!_nu   exp_mp = 1._ireals               ! Exponent in the interpolation formula
!_nu                                    ! for the mixed-phase water fraction [-]
!_nu ! Local Scalars
!_nu REAL (KIND=ireals) :: &
!_nu   fr_wat            , & ! Water fraction for the water-ice mixed phase [-]
!_nu   qs_w              , & ! Saturation specific humidity over water [-]
!_nu   qs_i              , & ! Saturation specific humidity over ice [-]
!_nu   qs_m              , & ! Saturation specific humidity for mixed phase
!_nu   qdvdt_w           , & ! First derivative of the saturation specific humidity over water
!_nu                         ! with respect to temperature [K^{-1}]
!_nu   qdvdt_i           , & ! First derivative of the saturation specific humidity over ice
!_nu                         ! with respect to temperature [K^{-1}]
!_nu   qdvdt_m               ! First derivative of the saturation specific humidity
!_nu                         ! with respect to temperature for the mixed phase [K^{-1}]
!_nu
!<DM+AS

!------------ End of header ---------------------------------------------------

!------------------------------------------------------------------------------
! Begin Subroutine cu_cond
!------------------------------------------------------------------------------
    zfacc = 0.0_ireals
    zface = 0.0_ireals
    IF (pcflag) zfacc = 1.0_ireals
    IF (peflag) zface = 1.0_ireals

    DO i = isc, iec
      zcond(i) = 0.0_ireals     ! Initialize condensation variable
      IF(plflag(i)) THEN        ! only, if ascent still continues
!DM+AS>
        ! Water fraction for the mixed water-ice phase as dependent on temperature
        IF (pT(i).LE.Tmpmin) THEN
          fr_wat = 0._ireals
        ELSE IF (pT(i).GE.Tmpmax) THEN
          fr_wat = 1._ireals
        ELSE
          fr_wat = ((pT(i)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
        ENDIF
        ! saturation over water and ice
        qs_w = cc2*EXP( b2w*(pt(i)-b3)/(pt(i)-b4w) )*p1op(i)
        qs_i = cc2*EXP( b2i*(pt(i)-b3)/(pt(i)-b4i) )*p1op(i)
        ! Effective saturation for mixed phase region
        qs_m = fr_wat*qs_w + (1._ireals-fr_wat)*qs_i
        qs_w = MIN( 0.5_ireals, qs_w )
        qs_i = MIN( 0.5_ireals, qs_i )
        qs_m = MIN( 0.5_ireals, qs_m )
        qs_w = qs_w / (1.0-rvd_m_o*qs_w)
        qs_i = qs_i / (1.0-rvd_m_o*qs_i)
        qs_m = qs_m / (1.0-rvd_m_o*qs_m)
        ! Effective latent heat of evaporation/sublimation for the mixed phase
        zhldcp = fr_wat*chlcdcp + (1._ireals-fr_wat)*chlsdcp
        ! The amount of condensate resulting from the saturation adjustment
        qdvdt_w = c5hlccp * qs_w/(1.0-rvd_m_o*qs_w) / (pt(i)-b4w)**2
        qdvdt_i = c5hlscp * qs_i/(1.0-rvd_m_o*qs_i) / (pt(i)-b4i)**2
        qdvdt_m = fr_wat*qdvdt_w + (1._ireals-fr_wat)*qdvdt_i
        zcond1  = (pqv(i)-qs_m)/(1._ireals+qdvdt_m)
        ! switches for evaporation vs condensation
        zcond(i) = zfacc*MAX( zcond1, 0.0_ireals )  + &
                   zface*MIN( zcond1, 0.0_ireals )
        ! integrate T and qv
        pt(i)    = pt(i) + zhldcp*zcond(i)
        pqv(i)   = pqv(i) - zcond(i)
!<DM+AS
      END IF
    END DO
    !Second iteration
    DO i = isc, iec
      IF( plflag(i) .AND. zcond(i).NE.0.0) THEN  !saturation adjustment
!DM+AS>
        ! Water fraction for the mixed water-ice phase as dependent on temperature
        IF (pT(i).LE.Tmpmin) THEN
          fr_wat = 0._ireals
        ELSE IF (pT(i).GE.Tmpmax) THEN
          fr_wat = 1._ireals
        ELSE
          fr_wat = ((pT(i)-Tmpmin)/(Tmpmax-Tmpmin))**exp_mp
        ENDIF
        ! saturation over water and ice
        qs_w = cc2*EXP( b2w*(pt(i)-b3)/(pt(i)-b4w) )*p1op(i)
        qs_i = cc2*EXP( b2i*(pt(i)-b3)/(pt(i)-b4i) )*p1op(i)
        ! Effective saturation for mixed phase region
        qs_m = fr_wat*qs_w + (1._ireals-fr_wat)*qs_i
        qs_w = MIN( 0.5_ireals, qs_w )
        qs_i = MIN( 0.5_ireals, qs_i )
        qs_m = MIN( 0.5_ireals, qs_m )
        qs_w = qs_w / (1.0-rvd_m_o*qs_w)
        qs_i = qs_i / (1.0-rvd_m_o*qs_i)
        qs_m = qs_m / (1.0-rvd_m_o*qs_m)
        ! Effective latent heat of evaporation/sublimation for the mixed phase
        zhldcp = fr_wat*chlcdcp + (1._ireals-fr_wat)*chlsdcp
        ! The amount of condensate resulting from the saturation adjustment
        qdvdt_w = c5hlccp * qs_w/(1.0-rvd_m_o*qs_w) / (pt(i)-b4w)**2
        qdvdt_i = c5hlscp * qs_i/(1.0-rvd_m_o*qs_i) / (pt(i)-b4i)**2
        qdvdt_m = fr_wat*qdvdt_w + (1._ireals-fr_wat)*qdvdt_i
!_cdm "pcflag" and "peflag" are not used during the 2nd iteration. Is that OK?
        zcond1  = (pqv(i)-qs_m)/(1._ireals+qdvdt_m)
        ! integrate T and qv
        pt(i)    = pt(i) + zhldcp*zcond1
        pqv(i)   = pqv(i) - zcond1
!<DM+AS
      END IF
    END DO

!-----------------------------------------------------------------------------
! End of the subroutine
!------------------------------------------------------------------------------
END SUBROUTINE cu_cond

!==============================================================================


!------------------------------------------------------------------------------
! End of module src_conv_tiedtke
!------------------------------------------------------------------------------
END MODULE src_conv_tiedtke
