#ifdef OASIS_IOW_ESM
SUBROUTINE send_fld

!!---------------------------------------------------------------------
!!              ***  ROUTINE send_fld  ***
!!
!! ** Purpose : Prepare and send coupling fields to OASIS
!!      
!!----------------------------------------------------------------------

USE oas_cos_vardef

USE data_runcontrol,  ONLY : &
  ntstep,        &
  nnow,          &
  nstart,        &
  ltime

USE data_modelconfig, ONLY : &
  ke, ie, je,    &
  dt,            &
  pollon,        &
  pollat,        &
  idt_qv,        &
  dlon, dlat

USE data_parameters,  ONLY : &
  ireals,        &
  iintegers

USE data_parallel,      ONLY :  &
  my_cart_id     ! rank of this subdomain in the cartesian communicator

USE data_fields,      ONLY : &
  ! 1. constant fields for the reference atmosphere                     (unit)
  ! -----------------------------------------------
  p0        ,    & ! base state pressure                           (Pa)
  ! 3. prognostic variables                                             (unit)
  ! -----------------------
  u         ,    & ! zonal wind speed                              ( m/s )
  v         ,    & ! meridional wind speed                         ( m/s )
  t         ,    & ! temperature                                   (  k  )
  pp        ,    & ! deviation from the reference pressure         ( pa  )
  ! 6. fields that are computed in the parametrization and dynamics     (unit )
  ! ---------------------------------------------------------------
  !   fields of convective and grid-scale precipitation
  prr_con    ,   & ! precipitation rate of rain, convective        (kg/m2*s)
  prs_con    ,   & ! precipitation rate of snow, convective        (kg/m2*s)
  prr_gsp    ,   & ! precipitation rate of rain, grid-scale        (kg/m2*s)
  prs_gsp    ,   & ! precipitation rate of snow, grid-scale        (kg/m2*s)
  prg_gsp    ,   & ! precipitation rate of graupel, grid-scale     (kg/m2*s)
  !   fields of the radiation
  swdir_s     ,   & ! direct shortwave downward radiation           ( W/m2)
  swdifd_s    ,   & ! diffuse shortwave downward radiation          ( W/m2) 
  swdifu_s    ,   & ! diffuse shortwave upward radiation            ( W/m2) 
  lwu_s       ,   & ! thermal upward radiation at the ground        ( W/m2)
  lwd_s       ,   & ! thermal downward radiation at the ground      ( W/m2)
  
  qvsflx      ,   & ! moisture flux 
  
  lhfl_s      ,   & ! latent heat flux (surface)                    ( W/m2)
  shfl_s      ,   & ! sensible heat flux (surface)                  ( W/m2)
  !   momentum
  umfl_s      ,   & ! u-momentum flux (surface)                     ( N/m2)
  vmfl_s      ,   & ! v-momentum flux (surface)                     ( N/m2)

  ps          ,   & ! surface pressure
 
  rlon        ,   & ! geographical longitude
  rlat        ,   & ! geographical latitude

  u_10m       ,   & ! zonal wind at 10m                      ( m/s )
  v_10m       ,   & ! meridional wind at 10m                 ( m/s )
  
  tch         ,   & ! turbulent transfer coefficient for heat       (1)
  tcm         ,   & ! turbulent transfer coefficient for momentum   (1)
  zalso_flc, & ! shortwave albedo from flux calculator
  llandmask


USE netcdf

USE time_utilities,   ONLY : & 
  get_timings,       & 
  i_cpl_put

USE src_tracer,      ONLY : trcr_get, trcr_errorstr

USE environment,      ONLY : model_abort

USE utilities,  ONLY : & 
  uvrot2uv_vec

IMPLICIT NONE

INTEGER(KIND=iintegers) :: &
  isec,            &
  kinfo,           &
  ierror,          &
  jn,      &
  i, j, k,         &
  ii, jj,          &
  izerror


INTEGER(KIND=iintegers) :: &
  im1, ip1, jm1, jp1,      &
  left, right, bottom, top,&
  istatus,                 &
  ncfileid, ncvarid,       &
  nrcvinfo(nfld_rcv_tot)  ! OASIS info argument
  
INTEGER(KIND=4)         :: &
  mask(nlei-nldi+1,nlej-nldj+1), & ! mask array
  mask_u(nlei-nldi+1,nlej-nldj+1), & ! mask array
  mask_v(nlei-nldi+1,nlej-nldj+1) ! mask array

REAL(KIND=ireals)      :: &
  ! Shepard interpolation for sent fields tcm and tch
  c, &     ! sum of weights
  w        ! weight for nighbor cells for Shepard interoplation

REAL(KIND=ireals)      :: &
  ztmp1  (ie,je),         &
  ztmp2  (ie,je)

CHARACTER (LEN=80)      :: &
  yzerrmsg

CHARACTER (LEN=25)       :: yzroutine

REAL (KIND=ireals), POINTER :: &
  qv_now     (:,:,:)   => NULL()! , &     ! QV at nx

!------------------------------------------------------------------------------
!- End of header
!------------------------------------------------------------------------------

  izerror = 0_iintegers   
  yzerrmsg = ''
  yzroutine= 'send_fld'

 ! Retrieve the required microphysics tracers
  CALL trcr_get(izerror, idt_qv,  ptr_tlev = nnow, ptr = qv_now)
  IF (izerror /= 0) THEN
    yzerrmsg = trcr_errorstr(izerror)
    CALL model_abort(my_cart_id, izerror, yzerrmsg, yzroutine)
  ENDIF

!----------------------------------------------------------------------------
! 1 coupling-specific handling of sent fields
!----------------------------------------------------------------------------

! Coupling only on PE with at least one unmasked grid point
IF ( lpe_cpl ) THEN

  isec = nint( (ntstep - nstart ) * dt )

  IF ( debug_oasis > 15 .AND. my_cart_id == 0 ) THEN
    WRITE(nulout,*) 'in send_fld: ntstep=',ntstep,' isec=',isec
  ENDIF

!----------------------------------------------------------------------------
! 1.6 IOW_ESM-specific handling of sent fields
!----------------------------------------------------------------------------
  IF ( ytype_oce == 'flxcl' ) THEN 
    ! We will first send variables to the flux calculator and then read in the calculated fluxes.
    ! First send...
    jn = 0
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)=t(:,:,ke,nnow)
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Atmposheric temperature in lowest cell ASTATM00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)=pp(:, :, ke, nnow) + p0(:, :, ke)        
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Atmospheric pressure in lowest cell ASPATM00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)=ps(:,:,nnow)          
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Surface pressure ASPSUR00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:) = -(prr_con(:,:) + prr_gsp(:,:))
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Massflux rain ASMRAI00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:) = -(prs_con(:,:) + prs_gsp(:,:))                   
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Massflux snow ASMSNO00 
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:) = -lwd_s(:,:)          
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Radiation longwave downward ASRLWD00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:) = lwu_s(:,:)                    
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Radiation longwave upward ASRLWU00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)=-swdir_s(:,:)          
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Radiation shortwave direct downward ASRSDD00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)= - swdifd_s(:,:) !swdifu_s(:,:) - swdifd_s(:,:)          
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)   ! Radiation shortwave indirect net flux ASRSIN00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:) = u_10m(:,:)
      ztmp2(:,:) = v_10m(:,:)
      CALL uvrot2uv_vec(ztmp1,ztmp2,rlat,rlon,pollat,pollon,ie,je)
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Eastward 10m wind ASU10M00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      WHERE (zmask /= 0) ztmp2(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp2, kinfo)  ! Northward 10m wind ASV10M00
    ENDIF
    
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)=tch(:,:)
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals

      ! process coefficient before sending
      DO j = nldj,nlej
        DO i = nldi,nlei
          ! if there is a coupled cell that is also markes as land, we don't to want the coefficient directly,
          ! we want to interpolate it from neighboring coupled but non(!)-land cells
          ! otherwise a exchange grid cell over water will have turbulent coefficient of land point
          IF ((zmask(i,j) == 0) .AND. llandmask(i,j)) THEN

            ztmp1(i,j) = 0.0_ireals
            c = 0.0_ireals

            ! interpolate from the closest two shells at maximum
            DO k = 1, 3
              ! get indeces of shell borders
              left = max(i-k,1)
              right = min(i+k, ie)
              bottom = max(j-k,1)
              top = min(j+k, je)

              DO jj = bottom, top
                ! go over left neighbors from bottom to top
                IF ((zmask(left,jj) == 0) .AND. (.NOT. llandmask(left,jj))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((left - i) * dlon)**2 + ((jj - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tch(left,jj)
                  c = c + w
                ENDIF
                ! go over right neighbors from bottom to top
                IF ((zmask(right,jj) == 0) .AND. (.NOT. llandmask(right,jj))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((right - i) * dlon)**2 + ((jj - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tch(right,jj)
                  c = c + w
                ENDIF
              ENDDO
              DO ii = left, right
                ! go over bottom neighbors from left to right
                IF ((zmask(ii,bottom) == 0) .AND. (.NOT. llandmask(ii,bottom))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((ii - i) * dlon)**2 + ((bottom - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tch(ii,bottom)
                  c = c + w
                ENDIF
                ! go over top neighbors from left to right
                IF ((zmask(ii,top) == 0) .AND. (.NOT. llandmask(ii,top))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((ii - i) * dlon)**2 + ((top - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tch(ii,top)
                  c = c + w
                ENDIF   
              ENDDO      
              
              IF (c > 0.0_ireals) THEN
                ! if we found valid points for interpolation in this shell, we stop
                ztmp1(i,j) = ztmp1(i,j) / c
                EXIT
                ! otherwise we go to the next shell
              ENDIF 
            ENDDO

            ! if there are no valid point for interpolation in any shell, we take the value as is
            IF (c == 0.0_ireals) THEN
              ztmp1(i,j) = tch(i,j)
            ENDIF

          ENDIF
        ENDDO
      ENDDO      
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)   ! Atmospheric diffusion coeff. for mosisture ASAMOI00
    ENDIF
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)=tcm(:,:)
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals

      DO j = nldj,nlej
        DO i = nldi,nlei
          ! if there is a coupled cell that is also markes as land, we don't to want the coefficient directly,
          ! we want to interpolate it from neighboring coupled but non(!)-land cells
          ! otherwise a exchange grid cell over water will have turbulent coefficient of land point
          IF ((zmask(i,j) == 0) .AND. llandmask(i,j)) THEN

            ztmp1(i,j) = 0.0_ireals
            c = 0.0_ireals

            ! interpolate from the closest two shells at maximum
            DO k = 1, 3
              ! get indeces of shell borders
              left = max(i-k,1)
              right = min(i+k, ie)
              bottom = max(j-k,1)
              top = min(j+k, je)

              DO jj = bottom, top
                ! go over left neighbors from bottom to top
                IF ((zmask(left,jj) == 0) .AND. (.NOT. llandmask(left,jj))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((left - i) * dlon)**2 + ((jj - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tcm(left,jj)
                  c = c + w
                ENDIF
                ! go over right neighbors from bottom to top
                IF ((zmask(right,jj) == 0) .AND. (.NOT. llandmask(right,jj))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((right - i) * dlon)**2 + ((jj - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tcm(right,jj)
                  c = c + w
                ENDIF
              ENDDO
              DO ii = left, right
                ! go over bottom neighbors from left to right
                IF ((zmask(ii,bottom) == 0) .AND. (.NOT. llandmask(ii,bottom))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((ii - i) * dlon)**2 + ((bottom - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tcm(ii,bottom)
                  c = c + w
                ENDIF
                ! go over top neighbors from left to right
                IF ((zmask(ii,top) == 0) .AND. (.NOT. llandmask(ii,top))) THEN
                  ! use Shepard interpolation with power parameter 4, weight is inverse distance**4 (in units of dlon)
                  w = (dlon ** 4) / (((ii - i) * dlon)**2 + ((top - j) * dlat)**2) ** 2 
                  ztmp1(i,j) = ztmp1(i,j) + w  * tcm(ii,top)
                  c = c + w
                ENDIF   
              ENDDO      
              
              IF (c > 0.0_ireals) THEN
                ! if we found valid points for interpolation in this shell, we stop
                ztmp1(i,j) = ztmp1(i,j) / c
                EXIT
                ! otherwise we go to the next shell
              ENDIF 
            ENDDO

            ! if there are no valid point for interpolation in any shell, we take the value as is
            IF (c == 0.0_ireals) THEN
              ztmp1(i,j) = tcm(i,j)
            ENDIF

          ENDIF
        ENDDO
      ENDDO

      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)   ! Atmospheric diffusion coeff. for momentum ASAMOM00
    ENDIF
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:) = qv_now(:,:,ke)
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Specific vapor content ASQATM00
    ENDIF
    jn = jn + 1
    ztmp1(:,:) = 0.0
    ztmp2(:,:) = 0.0
    DO j = nldj,nlej
      jm1 = max(j-1,1)
      jp1 = min(j+1,je)
      DO i = nldi,nlei
        im1 = max(i-1,1)
        ip1 = min(i+1,ie)
        ztmp1(i,j) = 0.5*(u(i,j,ke,nnow) + u(im1,j,ke,nnow))
        ztmp2(i,j) = 0.5*(v(i,j,ke,nnow) + v(i,jm1,ke,nnow))
      ENDDO
    ENDDO
    CALL uvrot2uv_vec(ztmp1,ztmp2,rlat,rlon,pollat,pollon,ie,je)
    IF( ssnd(jn)%laction ) THEN
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! Eastward wind in lowest atmos level ASUATM00
    ENDIF
    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      WHERE (zmask /= 0) ztmp2(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp2, kinfo)  ! Eastward wind in lowest atmos level ASVATM00
    ENDIF

    jn = jn + 1
    IF( ssnd(jn)%laction ) THEN
      ztmp1(:,:)=zalso_flc(:,:)
      WHERE (zmask /= 0) ztmp1(:,:)=0.0_ireals
      CALL oas_cos_snd (jn, isec, ztmp1, kinfo)  ! shortwave albedo ASALBE00
    ENDIF

    ! Then receive...

    ! get integer mask within halo limits
    mask = INT(zmask(nldi:nlei,nldj:nlej))
    mask_u = INT(zmask_u(nldi:nlei,nldj:nlej))
    mask_v = INT(zmask_v(nldi:nlei,nldj:nlej))
    
    nrcvinfo (:) = OASIS_idle
    ztmp1 (:,:) = 0.0_ireals

    DO jn = nfld_rcv_flc_early + 1, nfld_rcv_flc
#ifdef IOW_ESM_DEBUG
      WRITE(*,*) 'calling oas_cos_rcv for field jn=',jn,' from flux calculator.'
#endif
      CALL oas_cos_rcv( jn, isec, ztmp1(:,:), nrcvinfo(jn) )   
      IF( nrcvinfo(jn) == OASIS_Rcv ) THEN
        frcv(:,:,jn)=ztmp1(:,:) 
#ifdef IOW_ESM_DEBUG
        WRITE(*,*) 'CCLM received field jn=',jn,' from flux calculator.'
#endif
      ENDIF
#ifdef IOW_ESM_DEBUG
      WRITE(*,*) 'finished calling oas_cos_rcv for field jn=',jn,' from flux calculator.'
#endif
    ENDDO

    jn = nfld_rcv_flc_early
    jn = jn + 1   ! evaporation mass flux MEVA
    IF( srcv(jn)%laction ) THEN 
      IF(CPL_FLG .gt. 0) THEN   
        WHERE (mask == 0) 
          qvsflx(nldi:nlei,nldj:nlej) = -frcv(nldi:nlei,nldj:nlej,jn)
        ELSEWHERE 
          qvsflx(nldi:nlei,nldj:nlej) = 0.0
        ENDWHERE
      ENDIF
    ENDIF
    jn = jn + 1   ! latent heat flux HLAT
    IF( srcv(jn)%laction ) THEN 
      IF(CPL_FLG .gt. 0) THEN    
        WHERE (mask == 0) 
          lhfl_s(nldi:nlei,nldj:nlej) = -frcv(nldi:nlei,nldj:nlej,jn)
        ELSEWHERE
          lhfl_s(nldi:nlei,nldj:nlej) = 0.0
        ENDWHERE
      ENDIF
    ENDIF
    jn = jn + 1   ! sensible heat flux HSEN
    IF( srcv(jn)%laction ) THEN 
      IF(CPL_FLG .gt. 0) THEN    
        WHERE (mask == 0) 
          shfl_s(nldi:nlei,nldj:nlej) = -frcv(nldi:nlei,nldj:nlej,jn)
        ELSEWHERE
          shfl_s(nldi:nlei,nldj:nlej) = 0.0
        ENDWHERE
      ENDIF
    ENDIF
    jn = jn + 1   ! upward flux of eastward momentum UMOM
    IF( srcv(jn)%laction ) THEN 
      IF(CPL_FLG .gt. 0) THEN    
        WHERE (mask_u == 0) 
          umfl_s(nldi:nlei,nldj:nlej) = -frcv(nldi:nlei,nldj:nlej,jn)
        ELSEWHERE
          umfl_s(nldi:nlei,nldj:nlej) = 0.0
        ENDWHERE
      ENDIF
    ENDIF
    jn = jn + 1   ! upward flux of northward momentum VMOM
    IF( srcv(jn)%laction ) THEN 
      IF(CPL_FLG .gt. 0) THEN    
        WHERE (mask_v == 0)
          vmfl_s(nldi:nlei,nldj:nlej) = -frcv(nldi:nlei,nldj:nlej,jn)
        ELSEWHERE
          vmfl_s(nldi:nlei,nldj:nlej) = 0.0
        ENDWHERE
      ENDIF
    ENDIF

  ENDIF ! ytype_oce
ENDIF ! lpe_cpl

IF (ltime) CALL get_timings (i_cpl_put, ntstep, dt, ierror)

END SUBROUTINE send_fld

#endif ! OASIS_IOW_ESM