#IFDEF OASIS_IOW_ESM
SUBROUTINE receive_fld

!!---------------------------------------------------------------------
!!              ***  ROUTINE receive_fld  ***
!!
!! ** Purpose : - Prepare and receive coupling fields to OASIS
!!    
!!----------------------------------------------------------------------

USE oas_cos_vardef

USE data_modelconfig, ONLY : &
  dt,              &
  ie, je,          & 
  idt_qv

USE data_parameters,  ONLY : &
  ireals,          &
  iintegers

USE data_parallel,    ONLY : & 
  my_cart_id

USE environment,      ONLY : &
  model_abort

USE data_runcontrol , ONLY : &
  nnow  ,         & ! corresponds to ntstep
  nnew  ,         & ! corresponds to ntstep + 1
  ntstep,         & !
  nstart,         & ! first time step of the forecast
  ltime

USE data_constants,   ONLY : &
  sigma             ! Boltzmann-constant

USE data_soil,        ONLY : &
  ctalb       ! thermal albedo (1-emissivity) ?!

USE data_fields,      ONLY : &                        
  zalso_flc  ,    & ! averaged surface albedo for shortwave radiation from flux calculator ( )
  !    fields for surface values and soil model variables               (unit )
  ! -----------------------------------------------------
  t_g       ,     & ! weighted surface temperature                  (  K  )
  t_g_rad   ,     & ! weighted surface temperature for radiation    (  K  )
  t_s       ,     & ! temperature of the ground surface             (  k  )
  fr_ice            ! ice fraction 

USE MPI, only: MPI_BARRIER, MPI_COMM_WORLD

USE time_utilities,   ONLY : &
  get_timings,      & 
  i_cpl_add_comp,   &
  i_cpl_get

USE src_tracer,      ONLY : trcr_get, trcr_errorstr

IMPLICIT NONE

!
! local parameters, variables and arrays
!

INTEGER(KIND=iintegers) :: &
  isec,               &
  jn,                 &
  i, j, k, n, dummy,  &
  izerror

INTEGER(KIND=iintegers) :: &
  nrcvinfo(nfld_rcv_tot)  ! OASIS info argument

INTEGER(KIND=4)         :: &
  mask(nlei-nldi+1,nlej-nldj+1) ! mask array

REAL(KIND=ireals)      :: &
  ztmp1  (ie,je)
  
CHARACTER(LEN=200)      :: &
  yzerrmsg               ! error message for error handling

CHARACTER (LEN=25)       :: & 
  yzroutine

REAL (KIND=ireals), POINTER :: &
    qv_now     (:,:,:)   => NULL() !, &     ! QV at nx

!------------------------------------------------------------------------------
!- End of header
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! 1. Attribute fields to received fields (independent of the specific coupling) 
!-------------------------------------------------------------------------------


   izerror  = 0_iintegers
   yzerrmsg = ''
   yzroutine= 'receive_fld'

   CALL trcr_get(izerror, idt_qv, ptr_tlev = nnow, ptr = qv_now)
   IF (izerror /= 0) THEN
      yzerrmsg = trcr_errorstr(izerror)
      CALL model_abort(my_cart_id, izerror, yzerrmsg, yzroutine)
   ENDIF

! Coupling only with PEs that contain at least one land point
IF ( lpe_cpl ) THEN

  nrcvinfo (:) = OASIS_idle
  ztmp1 (:,:) = 0.0_ireals

  isec = nint ( ( ntstep - nstart ) * dt )

  IF (  debug_oasis > 15 .AND. my_cart_id == 0 ) THEN
    WRITE(nulout,*) 'in receive_fld: ntstep=',ntstep,' isec=',isec
  ENDIF

  IF (ltime) CALL get_timings (i_cpl_add_comp, ntstep, dt, izerror)

  IF (MOD(isec,dt_cp)==0) THEN  ! coupling time step
    CALL MPI_BARRIER(MPI_COMM_WORLD,dummy)
  ENDIF

!------------------------------------------------------------------------------
! 2. Receive all coupling fields (independent of the specific coupling) 
!-------------------------------------------------------------------------------
  IF( ytype_oce == 'flxcl' .AND. lpe_cpl ) THEN
    ! 1) Receive early fields from flux calculator
    DO jn = 1, nfld_rcv_flc_early
      
#ifdef IOW_ESM_DEBUG
      WRITE(*,*) 'calling oas_cos_rcv for field jn=',jn,' from flux calculator.'
#endif
      CALL oas_cos_rcv( jn, isec, ztmp1(:,:), nrcvinfo(jn) )
      IF( nrcvinfo(jn) == OASIS_Rcv ) THEN
        frcv(:,:,jn)=ztmp1(:,:)
#ifdef IOW_ESM_DEBUG
        WRITE(*,*) 'CCLM received field jn=',jn,' from flux calculator.'
#endif
      ENDIF
#ifdef IOW_ESM_DEBUG
      WRITE(*,*) 'finished calling oas_cos_rcv for field jn=',jn,' from flux calculator.'
#endif
    ENDDO
  ENDIF

ENDIF ! lpe_cpl

IF (ltime) CALL get_timings (i_cpl_get, ntstep, dt, izerror)

IF( lpe_cpl ) THEN
  IF( ytype_oce == 'flxcl') THEN
  
    ! 1) Process early fields from flux calculator
    mask = INT(zmask(nldi:nlei,nldj:nlej))

    jn = 0

    ! Update non-masked t_s with frcv array
    ! SST index
    jn = jn + 1   ! surface temperature
    IF( srcv(jn)%laction ) THEN 
    IF(CPL_FLG .gt. 0) THEN    
      WHERE (mask == 0) t_s(nldi:nlei,nldj:nlej,nnow) = (1.0_ireals - fmask(nldi:nlei,nldj:nlej))*frcv(nldi:nlei,nldj:nlej,jn) + t_s(nldi:nlei,nldj:nlej,nnow)*fmask(nldi:nlei,nldj:nlej)
      WHERE (mask == 0) t_s(nldi:nlei,nldj:nlej,nnew) = (1.0_ireals - fmask(nldi:nlei,nldj:nlej))*frcv(nldi:nlei,nldj:nlej,jn) + t_s(nldi:nlei,nldj:nlej,nnew)*fmask(nldi:nlei,nldj:nlej)
    ENDIF
    ENDIF

    jn = jn + 1   ! blackbody radiation => convert back to radiative temperature
    IF( srcv(jn)%laction ) THEN 
    IF(CPL_FLG .gt. 0) THEN   
      WHERE (mask == 0) 
        t_g_rad(nldi:nlei,nldj:nlej,nnow) = (1.0_ireals - fmask(nldi:nlei,nldj:nlej))*frcv(nldi:nlei,nldj:nlej,jn) + fmask(nldi:nlei,nldj:nlej) * sigma * t_g(nldi:nlei,nldj:nlej,nnow)**4 
        t_g_rad(nldi:nlei,nldj:nlej,nnew) = (1.0_ireals - fmask(nldi:nlei,nldj:nlej))*frcv(nldi:nlei,nldj:nlej,jn) + fmask(nldi:nlei,nldj:nlej) * sigma * t_g(nldi:nlei,nldj:nlej,nnew)**4 
        t_g_rad(nldi:nlei,nldj:nlej,nnow) = (t_g_rad(nldi:nlei,nldj:nlej,nnow) / sigma)**0.25 ! TODO: think about using zalth (thermal albedo of surface) here
        t_g_rad(nldi:nlei,nldj:nlej,nnew) = (t_g_rad(nldi:nlei,nldj:nlej,nnew) / sigma)**0.25
      ELSEWHERE
        t_g_rad(nldi:nlei,nldj:nlej,nnow) = t_g(nldi:nlei,nldj:nlej,nnow)
        t_g_rad(nldi:nlei,nldj:nlej,nnew) = t_g(nldi:nlei,nldj:nlej,nnew)
      ENDWHERE
    ENDIF
    ENDIF

    jn = jn + 1 ! FICE fraction of ice
    IF( srcv(jn)%laction ) THEN
    IF(CPL_FLG .gt. 0) THEN
      WHERE (mask == 0) fr_ice(nldi:nlei,nldj:nlej) = frcv(nldi:nlei,nldj:nlej,jn)
    ENDIF
    ENDIF

    jn = jn + 1 ! ALBE averaged surface albedo for shortwave radiation
    IF( srcv(jn)%laction ) THEN
    IF(CPL_FLG .gt. 0) THEN
      WHERE (mask == 0) 
        zalso_flc(nldi:nlei,nldj:nlej) = frcv(nldi:nlei,nldj:nlej,jn)
      ELSEWHERE
        zalso_flc(nldi:nlei,nldj:nlej) = -1.0
      ENDWHERE
    ENDIF
    ENDIF
  
  ENDIF ! ytype_oce == 'flxcl'
ENDIF ! lpe_cpl

IF (ltime) CALL get_timings (i_cpl_add_comp, ntstep, dt, izerror)

END SUBROUTINE receive_fld
#endif
