#ifdef OASIS_IOW_ESM
SUBROUTINE oas_cos_define
!
!**** *oas_cos_define*  - Define grids etc for OASIS
!
!     Purpose. 
!     --------
!        Define grid and field information for atmosphere
!        exchange between AGCM, OGCM and COUPLER. (OASIS3 software)
!
!**   Interface.
!     ----------
!       *CALL*  *oas_cos_define*
!
!     Input:
!     -----
!
!     Output:
!     ------
!
!     Method:
!     ------
!        
!     Externals:
!     ---------
!       prism_init, prism_init_comp, prism_get_localcomm : prism library
!
!     Author:
!     -------
!       R. Redler, NEC Laboratories Europe
!       K. Mogensen, ECMWF
!
!     Modifications.
!     --------------
!       E. Maisonnave : adapted from NEMO to COSMO
!       A. Dobler: adapted to OASIS3
!       E. Maisonnave : adapted to COSMO
!       2011-11 J. Brauch: calculate area
!       2012-08: unified interface for OASIS - COSMO/CLM, Will/Davin/Brauch/Weiher
!       2013-06: extension for staggered variables for ECHAM coupling, Weiher
!       2013-08-05: unified OASIS interface (UOI), Will/Byrne

USE data_parameters,  ONLY : &
  ireals,                 &
  iintegers

USE data_runcontrol,  ONLY : &
  nuspecif

USE oas_cos_vardef

USE mod_oasis_namcouple    ! OASIS3-MCT namcouple variables: e.g. coupling time step

USE data_modelconfig, ONLY : &
  ie,                     &
  je,                     &
  ie_tot,                 &
  je_tot,                 &
  dt

USE data_parallel,    ONLY : &
  my_cart_id,             & ! rank of this subdomain in the cartesian communicator
  isubpos,                & ! positions of the subdomains in the total domain. Given
                            ! are the i- and the j-indices of the lower left and the
                            ! upper right grid point in the order
                            !                  i_ll, j_ll, i_ur, j_ur.
                            ! Only the interior of the domains are considered, not
                            ! the boundary lines.
  nboundlines,            & ! number of boundary lines of the domain for which
                            ! no forecast is computed = overlapping boundary
                            ! lines of the subdomains
  intbuf,                 & !
  charbuf
                            
USE parallel_utilities, ONLY : &
  distribute_field

USE environment,        ONLY : &
  get_free_unit

USE netcdf


IMPLICIT NONE

!
! local variables

INTEGER(KIND=iintegers) :: &
  ji, i, j,          & !
  id_part,                 & !
  paral(5),                & ! OASIS3 box partition
  var_nodims(2),           & ! 
  ishape(2,2),             & ! shape of arrays passed to PSMILe
  ierrstat,                & !
  izerrstat,               & !
  istatus,                 & ! NetCDF status
  ncfileid, ncvarid          ! NetCDF IDs

!------------------------------------------------------------------------------
!- End of header
!------------------------------------------------------------------------------

      
! -----------------------------------------------------------------
! ... Setup the OASIS interface
! ----------------------------------------------------------------
 
IF ( my_cart_id == 0 .AND. debug_oasis > 15 ) THEN
  WRITE(nulout,'(A)') ' *****************************************************'
  WRITE(nulout,'(A)') ' *     CCLM: Setting up OASIS3(-MCT) interface       *'
  WRITE(nulout,'(A)') ' *****************************************************'
  CALL flush(nulout)
ENDIF
      
! Subdomain size without halo (remove the halo, halo width is "nboundlines")
nldi = 1  + nboundlines
nldj = 1  + nboundlines
nlei = ie - nboundlines
nlej = je - nboundlines
      
iboundstart = 0
jboundstart = 0
      
! Subdomain size without halo BUT with boundlines
! (For outer domains, the halo has been falsely removed since there is no halo, add it back)
IF ( isubpos(my_cart_id,1) == 1 + nboundlines ) THEN
  nldi = 1
  iboundstart = nboundlines
ENDIF
IF ( isubpos(my_cart_id,3) == ie_tot - nboundlines ) THEN
  nlei = ie
ENDIF
IF ( isubpos(my_cart_id,2) == 1 + nboundlines ) THEN
  nldj = 1
  jboundstart = nboundlines
ENDIF
IF ( isubpos(my_cart_id,4) == je_tot - nboundlines ) THEN
  nlej = je
ENDIF
      
! Local extent
jih = nlei - nldi + 1
jjh = nlej - nldj + 1

IF ( my_cart_id == 0 .AND. debug_oasis > 15 ) THEN
  WRITE(nulout,*) '++++ CCLM sub-domain: my_cart_id=',my_cart_id, &
    ' nldi,nlei,nldj,nlej=',nldi,nlei,nldj,nlej
  CALL flush(nulout)
ENDIF

ALLOCATE (zmask_tot(ie_tot,je_tot), STAT=ierrstat )
zmask_tot(:,:) = 0.0_ireals

IF ( ytype_oce=='flxcl') THEN 
  ALLOCATE ( zmask(ie,je), zmask_u(ie,je), zmask_v(ie,je), STAT=ierrstat )
  zmask(:,:) = 0.0_ireals
  zmask_u(:,:) = 0.0_ireals
  zmask_v(:,:) = 0.0_ireals

  !--------------------------------------
  ! Read in masks from mapping files
  !--------------------------------------

  ! 1. Get gridcells that are coupled (zmask = 0.0) or not (zmask = 1.0)
  !--------------------------------------
  ! 1. a t-grid
  istatus=nf90_open('mappings/maskfile.nc', NF90_NOWRITE, ncfileid)
  IF (istatus /= nf90_noerr) THEN
    WRITE (*,*) "ERROR no mappings/maskfile.nc found. Abort."
    istatus = nf90_close(ncfileid)
    RETURN
  ENDIF
  istatus=nf90_inq_varid(ncfileid, 'frac_coupled' , ncvarid)
  istatus=nf90_get_var(ncfileid, ncvarid, zmask_tot, &
        (/ 1, 1 /),      &
          (/ ie_tot, je_tot /))
  istatus=nf90_close(ncfileid)
  WHERE (zmask_tot > 0.0_ireals)  zmask_tot = 1.0_ireals 
  WHERE (zmask_tot <= 0.0_ireals)  zmask_tot = 0.0_ireals 
  zmask_tot = 1.0_ireals - zmask_tot

  CALL distribute_field(zmask_tot, ie_tot, je_tot, zmask, ie, je, 0, nerror)
  
  !--------------------------------------
  ! 1. b u-grid
  istatus=nf90_open('mappings/maskfile_u_grid.nc', NF90_NOWRITE, ncfileid)
  IF (istatus /= nf90_noerr) THEN
    WRITE (*,*) "WARNING no mappings/maskfile_u_grid.nc found. Use mappings/maskfile.nc instead"
    istatus = nf90_close(ncfileid)
  ELSE
    istatus=nf90_inq_varid(ncfileid, 'frac_coupled' , ncvarid)
    istatus=nf90_get_var(ncfileid, ncvarid, zmask_tot, &
          (/ 1, 1 /),      &
            (/ ie_tot, je_tot /))
    istatus=nf90_close(ncfileid)
    WHERE (zmask_tot > 0.0_ireals)  zmask_tot = 1.0_ireals 
    WHERE (zmask_tot <= 0.0_ireals)  zmask_tot = 0.0_ireals 
    zmask_tot = 1.0_ireals - zmask_tot
  ENDIF

  CALL distribute_field(zmask_tot, ie_tot, je_tot, zmask_u, ie, je, 0, nerror)

  !--------------------------------------
  ! 1. c v-grid
  istatus=nf90_open('mappings/maskfile_v_grid.nc', NF90_NOWRITE, ncfileid)
  IF (istatus /= nf90_noerr) THEN
    WRITE (*,*) "WARNING no mappings/maskfile_v_grid.nc found. Use mappings/maskfile.nc instead"
    istatus = nf90_close(ncfileid)
  ELSE
    istatus=nf90_inq_varid(ncfileid, 'frac_coupled' , ncvarid)
    !read into zmask here, later this will be a separate variable
    istatus=nf90_get_var(ncfileid, ncvarid, zmask_tot, &
          (/ 1, 1 /),      &
            (/ ie_tot, je_tot /))
    istatus=nf90_close(ncfileid)
    WHERE (zmask_tot > 0.0_ireals)  zmask_tot = 1.0_ireals 
    WHERE (zmask_tot <= 0.0_ireals)  zmask_tot = 0.0_ireals 
    zmask_tot = 1.0_ireals - zmask_tot
  ENDIF

  CALL distribute_field(zmask_tot, ie_tot, je_tot, zmask_v, ie, je, 0, nerror)

  ! 2. Get fraction of gridcells that are coupled (fmask < 1.0) or not (fmask = 1.0)
  !--------------------------------------
  ! 2. a t-grid
  ALLOCATE (fmask (ie,je), fmask_u(ie,je), fmask_v(ie,je), fmask_tot(ie_tot,je_tot), STAT=ierrstat) 
  istatus=nf90_open('mappings/maskfile.nc', NF90_NOWRITE, ncfileid)
  IF (istatus /= nf90_noerr) THEN
    WRITE (*,*) "ERROR no mappings/maskfile.nc found. Abort."
    istatus = nf90_close(ncfileid)
    RETURN
  ENDIF
  istatus=nf90_inq_varid(ncfileid, 'frac_coupled' , ncvarid)
  istatus=nf90_get_var(ncfileid, ncvarid, fmask_tot, &
        (/ 1, 1 /),      &
          (/ ie_tot, je_tot /))
  istatus=nf90_close(ncfileid)
  ! Uncomment the following two lines to avoid means values at border cells
  ! WHERE (fmask_tot > 0.0_ireals)  fmask_tot = 1.0_ireals 
  ! WHERE (fmask_tot <= 0.0_ireals)  fmask_tot = 0.0_ireals 
  fmask_tot = 1.0_ireals - fmask_tot 

  CALL distribute_field(fmask_tot, ie_tot, je_tot, fmask, ie, je, 0, nerror)
  
  !--------------------------------------
  ! 2. b u-grid
  istatus=nf90_open('mappings/maskfile_u_grid.nc', NF90_NOWRITE, ncfileid)
  IF (istatus /= nf90_noerr) THEN
    WRITE (*,*) "WARNING no mappings/maskfile_u_grid.nc found. Use mappings/maskfile.nc instead"
    istatus = nf90_close(ncfileid)
  ELSE
    fmask_tot = 0.0_ireals
    istatus=nf90_inq_varid(ncfileid, 'frac_coupled' , ncvarid)
    istatus=nf90_get_var(ncfileid, ncvarid, fmask_tot, &
          (/ 1, 1 /),      &
            (/ ie_tot, je_tot /))
    istatus=nf90_close(ncfileid)
    ! Uncomment the following two lines to avoid means values at border cells
    ! WHERE (fmask_tot > 0.0_ireals)  fmask_tot = 1.0_ireals 
    ! WHERE (fmask_tot <= 0.0_ireals)  fmask_tot = 0.0_ireals 
    fmask_tot = 1.0_ireals - fmask_tot 
  ENDIF

  CALL distribute_field(fmask_tot, ie_tot, je_tot, fmask_u, ie, je, 0, nerror)

  !--------------------------------------
  ! 2. c v-grid
  istatus=nf90_open('mappings/maskfile_v_grid.nc', NF90_NOWRITE, ncfileid)
  IF (istatus /= nf90_noerr) THEN
    WRITE (*,*) "WARNING no mappings/maskfile_v_grid.nc found. Use mappings/maskfile.nc instead"
    istatus = nf90_close(ncfileid)
  ELSE
    fmask_tot = 0.0_ireals
    istatus=nf90_inq_varid(ncfileid, 'frac_coupled' , ncvarid)
    istatus=nf90_get_var(ncfileid, ncvarid, fmask_tot, &
          (/ 1, 1 /),      &
            (/ ie_tot, je_tot /))
    istatus=nf90_close(ncfileid)
    ! Uncomment the following two lines to avoid means values at border cells
    ! WHERE (fmask_tot > 0.0_ireals)  fmask_tot = 1.0_ireals 
    ! WHERE (fmask_tot <= 0.0_ireals)  fmask_tot = 0.0_ireals 
    fmask_tot = 1.0_ireals - fmask_tot
  ENDIF

  CALL distribute_field(fmask_tot, ie_tot, je_tot, fmask_v, ie, je, 0, nerror)

  DEALLOCATE(fmask_tot)
ENDIF

! With OASIS3, all domains involved; for now, this will also be valid for 
! OASIS3-MCT (coupling only on a subset of subdomains becomes important if
! many processors are used)
lpe_cpl = .TRUE.

IF ( debug_oasis > 15 .AND. my_cart_id == 0) THEN
  WRITE(nulout,*) ' prism_terminate_grids_writing '
  CALL flush(nulout)
ENDIF

IF ( lpe_cpl ) THEN

! -----------------------------------------------------------------
! ... Define the partition
! -----------------------------------------------------------------

  ! whole domain decomposition can be represented by a rectangle
  paral(1) = 2  ! 2 means box partitioning
  ! Global extent in x
  paral(5) = ie_tot
  ! Upper left corner global offset
  paral(2) = isubpos(my_cart_id,1)-1-iboundstart + ( isubpos(my_cart_id,2)-1-jboundstart) * paral(5)
  ! Local extent in x
  paral(3) = jih
  ! Local extent in y
  paral(4) = jjh

!  write(*,*) "il_paral cclm",  paral(1),paral(2), paral(3), paral(4), paral(5)
  CALL prism_def_partition_proto( id_part, paral, nerror )


  IF ( nerror /= PRISM_Success ) THEN
    CALL prism_abort_proto (ncomp_id, 'oas_cos_define', 'Failure in prism_def_partition' )
  ENDIF

! -----------------------------------------------------------------
! ... Initialize some variables
! ----------------------------------------------------------------

  nfld_snd_flc = 0

  nfld_rcv_flc = 0
  nfld_rcv_flc_early = 0
  nfld_snd_tot = 0
  nfld_rcv_tot = 0

! -----------------------------------------------------------------
! ... Define list of SENT variables per coupling
! ----------------------------------------------------------------

  nfld = 0

  IF (ytype_oce == 'flxcl') THEN
    WRITE(nulout,*) 'Use coupling time step dt_cp=', dt_cp
    IF ( debug_oasis > 15 .AND. my_cart_id == 0) THEN        
      WRITE(nulout,*) ' coupling time step dt_cp=', dt_cp
      CALL flush(nulout)
    ENDIF
    
    nfld_snd_flc = 17
    ALLOCATE ( nlev_snd_flc (nfld_snd_flc), STAT=ierrstat )
    ALLOCATE ( nam_snd_flc  (nfld_snd_flc), STAT=ierrstat )
    nlev_snd_flc = (/1,1,1,1,1,1,1,1,1,1, &
                    1,1,1,1,1,1,1/)

    nam_snd_flc( 1) =  'ASTATM00'
    nam_snd_flc( 2) =  'ASPATM00'
    nam_snd_flc( 3) =  'ASPSUR00'
    nam_snd_flc( 4) =  'ASMRAI00'
    nam_snd_flc( 5) =  'ASMSNO00'
    nam_snd_flc( 6) =  'ASRLWD00'
    nam_snd_flc( 7) =  'ASRLWU00'
    nam_snd_flc( 8) =  'ASRSDD00'
    nam_snd_flc( 9) =  'ASRSIN00'
    nam_snd_flc(10) =  'ASU10M00'
    nam_snd_flc(11) =  'ASV10M00'
    nam_snd_flc(12) =  'ASAMOI00'
    nam_snd_flc(13) =  'ASAMOM00'
    nam_snd_flc(14) =  'ASQATM00'
    nam_snd_flc(15) =  'ASUATM00'
    nam_snd_flc(16) =  'ASVATM00'
    nam_snd_flc(17) =  'ASALBA00'
    
    DO ji = 1, nfld_snd_flc
      nfld = nfld + nlev_snd_flc(ji)
    ENDDO
  ENDIF

  ! total number of fields to be sent:
  nfld_snd_tot = nfld

! -----------------------------------------------------------------
! ... Define list of RECEIVED variables per coupling
! ----------------------------------------------------------------

  nfld = 0

  IF ( ytype_oce == 'flxcl' ) THEN
    nfld_rcv_flc = 9
    nfld_rcv_flc_early = 4
    ALLOCATE ( nlev_rcv_flc (nfld_rcv_flc), STAT=ierrstat )
    ALLOCATE ( nam_rcv_flc  (nfld_rcv_flc), STAT=ierrstat )
    nlev_rcv_flc = (/1,1,1,1,1,1,1,1,1/)

    nam_rcv_flc(1) =  'ARTSUR00'
    nam_rcv_flc(2) =  'ARRBBR00'
    nam_rcv_flc(3) =  'ARFICE00'
    nam_rcv_flc(4) =  'ARALBE00'
    nam_rcv_flc(5) =  'ARMEVA00'
    nam_rcv_flc(6) =  'ARHLAT00'
    nam_rcv_flc(7) =  'ARHSEN00'
    nam_rcv_flc(8) =  'ARUMOM00'
    nam_rcv_flc(9) =  'ARVMOM00'
    
    
    DO ji = 1, nfld_rcv_flc
      nfld = nfld + nlev_rcv_flc(ji)
    ENDDO
  ENDIF

  ! total number of fields to be received:
  nfld_rcv_tot = nfld

  IF ( debug_oasis > 15 .AND. my_cart_id == 0) THEN
    WRITE(nulout,*) ' nfld_snd_tot=',nfld_snd_tot,' nfld_rcv_tot=',nfld_rcv_tot
    CALL flush(nulout)
  ENDIF

! -----------------------------------------------------------------
! ... write variables names for all active couplings to one structure
! ----------------------------------------------------------------

  ! Allocate memory for data exchange:
  ALLOCATE( ssnd(nfld_snd_tot), stat = nerror )
  ALLOCATE( srcv(nfld_rcv_tot), stat = nerror )
  ALLOCATE( exfld1(ie, je), stat = nerror )

  IF ( nerror > 0 ) THEN
    CALL prism_abort_proto( ncomp_id, 'oas_cos_define', 'Failure in allocating exfld, ssnd or srcv' )
    RETURN
  ENDIF

  ! fill ssnd with names of fields to be sent
  nfld = 0  

  IF ( ytype_oce == 'flxcl') THEN 
    DO ji = 1, nfld_snd_flc
      nfld = nfld + 1
      ssnd(nfld)%clname = TRIM(nam_snd_flc(ji))
    ENDDO
  ENDIF

  ! fill srcv with names of fields to be received:
  nfld = 0

  IF ( ytype_oce == 'flxcl') THEN 
    DO ji = 1, nfld_rcv_flc
      nfld = nfld + 1
      srcv(nfld)%clname = TRIM(nam_rcv_flc(ji))
    ENDDO
  ENDIF

! ----------------------------------------------------------------------------
! ... Variable selection
! ----------------------------------------------------------------------------

  ! This is still preliminary: set laction for all (this way no subsets of
  ! variables can be selected; this has to be changed yet)
  ssnd(:)%laction = .TRUE.
  srcv(:)%laction = .TRUE.

  var_nodims(1) = 2   ! Dimension number of exchanged arrays
  var_nodims(2) = 1   ! number of bundles (always 1 for OASIS3)

! -----------------------------------------------------------------
! ... Define the shape of the valid region without the halo and overlaps between CPUs
! -----------------------------------------------------------------

  ishape(1,1) = 1
  ishape(2,1) = jih
  ishape(1,2) = 1
  ishape(2,2) = jjh

! -----------------------------------------------------------------
! ... Announce variables to be sent and to be received
! -----------------------------------------------------------------

  IF ( debug_oasis > 15 .AND. my_cart_id == 0) THEN
    WRITE(nulout,*) ' ishape=',ishape,' id_part=',id_part
    CALL flush(nulout)
  ENDIF

  ! Announce variables to be sent:
  DO ji = 1, nfld_snd_tot
    CALL prism_def_var_proto( ssnd(ji)%nid, ssnd(ji)%clname, id_part, &
      var_nodims, PRISM_Out, ishape, PRISM_REAL, nerror )
    IF ( nerror /= PRISM_Success ) CALL prism_abort_proto( ssnd(ji)%nid, &
      'oas_cos_define', 'Failure in prism_def_var for '//TRIM(ssnd(ji)%clname) )
  ENDDO
      
  ! Announce variables to be received:
  DO ji = 1, nfld_rcv_tot
    CALL prism_def_var_proto( srcv(ji)%nid, srcv(ji)%clname, id_part, &
      var_nodims, PRISM_In, ishape, PRISM_REAL, nerror )
    IF ( nerror /= PRISM_Success ) CALL prism_abort_proto( srcv(ji)%nid, &
      'oas_cos_define', 'Failure in prism_def_var for '//TRIM(srcv(ji)%clname) )
  ENDDO

  ! Allocate array to store received fields between two coupling steps
  ALLOCATE( frcv(ie, je, nfld_rcv_tot), stat = nerror )
  IF ( nerror > 0 ) THEN
    CALL prism_abort_proto( ncomp_id, 'oas_cos_define', 'Failure in allocating frcv' )
    RETURN
  ENDIF   

! ------------------------------------------------------------------
! ... End of definition phase (must be called by all processes including 
!     the PEs not involved in the coupling)
! ------------------------------------------------------------------
       
!write(*,*), "finished oasis var definition cclm"
 
  CALL prism_enddef_proto( nerror )

  IF ( nerror /= PRISM_Success ) CALL prism_abort_proto ( ncomp_id, &
    'oas_cos_define', 'Failure in prism_enddef')

  IF ( debug_oasis > 15 .AND. my_cart_id == 0) THEN
    WRITE(nulout,*) ' oas_cos_define: end of definition phase'
    CALL flush(nulout)
  ENDIF
  !write(*,*), "finished oasis_def cclm"

! -----------------------------------------------------------------
! ... Deallocate arrays that are not needed anymore
! -----------------------------------------------------------------

  
  IF (ALLOCATED (nlev_snd_flc)) THEN
    DEALLOCATE ( nlev_snd_flc, STAT=ierrstat )
    DEALLOCATE ( nam_snd_flc , STAT=ierrstat )
  ENDIF
  
  IF (ALLOCATED (nlev_rcv_flc)) THEN
    DEALLOCATE ( nlev_rcv_flc, STAT=ierrstat )
    DEALLOCATE ( nam_rcv_flc , STAT=ierrstat )
  ENDIF

ENDIF ! lpe_cpl 
      
CALL MPI_Barrier(kl_comm, nerror)

END SUBROUTINE oas_cos_define

!==============================================================================
!+ reads the namelist for OASIS-/coupling-specific parameters
!------------------------------------------------------------------------------

SUBROUTINE read_namelist_oasis (ierror, yerrmsg)

USE data_parallel,    ONLY : my_cart_id
USE data_parameters,  ONLY : ireals, iintegers
USE environment,      ONLY : model_abort, get_free_unit
USE oas_cos_vardef,   ONLY : debug_oasis
USE data_runcontrol,  ONLY : nuspecif

IMPLICIT NONE


INTEGER (KIND=iintegers), INTENT(OUT)           ::                      &
  ierror       ! error status

CHARACTER (LEN=  *),      INTENT(OUT)           ::                      &
  yerrmsg      ! error message

!------------------------------------------------------------------------------
! local variables:
INTEGER (KIND=iintegers)   ::   &
  nuin,                         &
  izerrstat

CHARACTER (LEN=11)         ::   &
  yinput             ! Namelist INPUT file

!------------------------------------------------------------------------------
!- End of header
!------------------------------------------------------------------------------
 
!------------------------------------------------------------------------------
!- Begin Subroutine read_namelist_oasis
!------------------------------------------------------------------------------

ierror    = 0_iintegers
izerrstat = 0_iintegers

! -----------------------------------------------------------------
! 1 Open NAMELIST-INPUT file
! ----------------------------------------------------------------
    
IF (my_cart_id == 0) THEN

   IF ( debug_oasis > 15 ) &
     PRINT *,'    INPUT OF THE NAMELIST FOR COUPLING WITH OASIS3(-MCT)'

  yinput   = 'INPUT_OASIS'
  CALL get_free_unit (nuin)
  OPEN (nuin, FILE=yinput, FORM='FORMATTED', STATUS='UNKNOWN',  &
    IOSTAT=izerrstat)
  IF (izerrstat /= 0) THEN
    yerrmsg  = ' ERROR    *** Error while opening file INPUT_OASIS *** '
    ierror   = 2
    RETURN
  ENDIF
      
ENDIF
    
! -----------------------------------------------------------------
! 2 read the NAMELIST group oasisctl
! ----------------------------------------------------------------

CALL input_oasisctl (nuspecif, nuin, izerrstat)

IF (izerrstat > 0) THEN
  yerrmsg  = ' ERROR *** Wrong values occured in NAMELIST group /OASISCTL/ *** '
  ierror   = 3
  RETURN
ELSEIF (izerrstat < 0) THEN
  PRINT *, ' ERROR while reading NAMELIST group /OASISCTL/ in input_oasisctl '
  ierror   = 4
  RETURN
ENDIF

! -----------------------------------------------------------------
! 3 Close NAMELIST-INPUT file
! ----------------------------------------------------------------

IF (my_cart_id == 0) THEN

  CLOSE (nuin, STATUS='KEEP', IOSTAT=izerrstat)
  IF (izerrstat /= 0) THEN
    yerrmsg = ' ERROR *** while closing file INPUT_OASIS *** '
    ierror  = 5
  ENDIF

  IF ( debug_oasis > 15 ) PRINT *, 'After closing INPUT_OASIS'
ENDIF
    
!------------------------------------------------------------------------------
!- End of the Subroutine
!------------------------------------------------------------------------------

END SUBROUTINE read_namelist_oasis
  
!==============================================================================
!+ input of NAMELIST oasisctl
!------------------------------------------------------------------------------

SUBROUTINE input_oasisctl (nuspecif, nuin, ierrstat)

!------------------------------------------------------------------------------
!
! Description:
!
! Method:
!   All variables are initialized with default values and then read in from
!   the file INPUT. The input values are checked for errors and for
!   consistency. If wrong input values are detected the program prints
!   an error message. The program is not stopped in this routine but an
!   error code is returned to the calling routine that aborts the program after
!   reading in all other namelists.
!   In parallel mode, the variables are distributed to all nodes with the
!   environment-routine distribute_values.
!   Both, default and input values are written to the file YUSPECIF
!   (specification of the run).
!
!------------------------------------------------------------------------------

USE data_parallel,    ONLY : &
  my_cart_id,      &
  charbuf,         &            
  intbuf,          &
  logbuf,          &
  realbuf,         &
  nproc,           & ! total number of processors: nprocx * nprocy
  imp_integers,    & ! determines the correct INTEGER type used in the
                     ! model for MPI
  imp_character,   & ! determines the correct CHARACTER type used in the
                     ! model for MPI
  imp_logical,     & ! determines the correct LOGICAL type used in the
                     ! model for MPI
  imp_reals,       & ! determines the correct REAL type used in the
                     ! model for MPI
  icomm_world        ! communicator that belongs to igroup_world, i.e.
                     ! = MPI_COMM_WORLD

USE data_parameters,  ONLY : &
  ireals,          &
  iintegers

USE environment,      ONLY : &
  model_abort

USE parallel_utilities, ONLY: &
  distribute_values

USE data_modelconfig, ONLY : &
  ke,               &
  ie_tot,           &
  je_tot

USE oas_cos_vardef, ONLY : &
  debug_oasis,      &
  ytype_lsm,        &
  ytype_oce,        &
  CPL_FLG,          &
  dt_cp

IMPLICIT NONE

! Parameter list:
  INTEGER (KIND=iintegers), INTENT (IN)      ::        &
    nuspecif,     & ! Unit number for protocolling the task
    nuin            ! Unit number for Namelist INPUT file

  INTEGER (KIND=iintegers), INTENT (INOUT)   ::        &
    ierrstat        ! error status variable

  CHARACTER (LEN=80)        ::                       &
    yerrmsg            ! error message
    
! Local variables: 
  INTEGER (KIND=iintegers)   ::  &
    i, iz_err
  
  INTEGER (KIND=iintegers)   ::  &
    debug_oasis_d,          &
    dt_cp_d

  REAL (KIND=ireals) :: &
    CPL_FLG_d

  CHARACTER (LEN=5) ::  &
    ytype_lsm_d,  & ! type of Land Surface Model
    ytype_oce_d     ! type of OCEAN Model
! Define the namelist group
  NAMELIST /oasisctl/ ytype_lsm, ytype_oce, debug_oasis, CPL_FLG, dt_cp
                      
!------------------------------------------------------------------------------
!- End of header -
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
!- Begin SUBROUTINE input_oasisctl
!------------------------------------------------------------------------------

ierrstat = 0_iintegers
iz_err   = 0_iintegers

IF ( ierrstat /= 0 ) THEN
  PRINT *, ' ERROR    *** Error while allocating variables in routine input_oasisctl *** '
  ierrstat   = 2
  RETURN
ENDIF

!------------------------------------------------------------------------------
!- Section 1: Initialize the default variables
!------------------------------------------------------------------------------

IF (my_cart_id == 0) THEN

  ! general namelist parameters
  ytype_lsm_d = 'terra'
  ytype_oce_d = 'nooce'
  debug_oasis_d = 0          ! no debug output per default
  CPL_FLG_d = 0              ! no coupling
  dt_cp_d = 600              ! default coupling time step

                          
!------------------------------------------------------------------------------
!- Section 2: Initialize variables with defaults
!------------------------------------------------------------------------------

  ytype_lsm = ytype_lsm_d
  ytype_oce = ytype_oce_d
  debug_oasis = debug_oasis_d
  CPL_FLG = CPL_FLG_d
    dt_cp = dt_cp_d 

!------------------------------------------------------------------------------
!- Section 3: Input of the namelist values
!------------------------------------------------------------------------------

  READ (nuin, oasisctl, IOSTAT=iz_err)

ENDIF ! my_cart_id

IF (nproc > 1) THEN
  ! distribute error status to all processors
  CALL distribute_values  (iz_err, 1, 0, imp_integers, icomm_world, ierrstat)
ENDIF

IF (iz_err /= 0) THEN
  ierrstat = -1
  RETURN
ENDIF

IF (my_cart_id == 0) THEN

!------------------------------------------------------------------------------
!- Section 4: Check values for errors and consistency
!------------------------------------------------------------------------------

  IF ( ytype_lsm /= 'terra' ) THEN
    PRINT *, ' ERROR  *** Wrong type of Land Surface Model *** :', ytype_lsm
    ierrstat = 1002 
  ENDIF
  
  IF ( ytype_oce /= 'flxcl' .AND. ytype_oce /= 'nooce' ) THEN
    PRINT *, ' ERROR  *** Wrong type of ocean Model *** :', ytype_oce
    ierrstat = 1002 
  ENDIF

  IF ( debug_oasis < 0 .OR. debug_oasis > 30 ) THEN
    PRINT *, ' Warning  *** debug output for OASIS only for values between 16 and 30 *** '
    ierrstat = 1002
  ENDIF
  

  IF ( ierrstat /= 0 ) THEN
    PRINT *, ' ERROR  *** Error while checking values of the namelist oasisctl *** '
    RETURN
  ENDIF

ENDIF ! my_cart_id

!------------------------------------------------------------------------------
!- Section 5: Distribute variables to all nodes
!------------------------------------------------------------------------------

IF (nproc > 1) THEN

  IF (my_cart_id == 0) THEN
  
    charbuf( 1) = ytype_lsm
    charbuf( 2) = ytype_oce  

    intbuf ( 1) = debug_oasis
    intbuf ( 2) = CPL_FLG 
    intbuf ( 3) = dt_cp 

      
  ENDIF ! my_cart_id == 0
  
  CALL distribute_values (charbuf , 2, 0, imp_character, icomm_world, ierrstat)
  CALL distribute_values (intbuf  , 3, 0, imp_integers , icomm_world, ierrstat)

  IF (my_cart_id /= 0) THEN

    ytype_lsm = charbuf( 1)
    ytype_oce = charbuf( 2)  

    debug_oasis = intbuf ( 1)
    CPL_FLG = intbuf (2)
    dt_cp = intbuf (3)
  
  ENDIF ! my_cart_id == 0

  IF ( ierrstat /= 0 ) THEN
    PRINT *, ' ERROR *** in distributing buffers *** '
    RETURN
  ENDIF

ENDIF ! nproc

!------------------------------------------------------------------------------
!- Section 6: Output of the namelist variables and their default values
!------------------------------------------------------------------------------

IF ( debug_oasis > 15 .AND. my_cart_id == 0) THEN
  PRINT *, 'Writing NAMELIST to file'

  WRITE (nuspecif, '(A2)')  '  '
  WRITE (nuspecif, '(A25)') '     NAMELIST:  oasisctl'
  WRITE (nuspecif, '(A25)') '     -------------------'
  WRITE (nuspecif, '(A2)')  '  '
  WRITE (nuspecif, '(T7,A,T26,A,T44,A,T63,A)')                               &
    'Variable', 'Actual Value', 'Default Value', 'Format'

  WRITE (nuspecif, '(T8,A, T27,A, T45,A, T64,A3)')                           &
    'ytype_lsm', ytype_lsm, ytype_lsm_d, ' C ' 
  WRITE (nuspecif, '(T8,A, T27,A, T45,A, T64,A3)')                           &
    'ytype_oce', ytype_oce, ytype_oce_d, ' C ' 

  WRITE (nuspecif, '(T8,A,T27,I12  ,T45,I12  ,T64,A3)')                      &                      
    'debug_oasis', debug_oasis, debug_oasis_d, ' I '


  WRITE (nuspecif, '(T8,A,T27,I12  ,T45,L12  ,T64,A3)')                      &                      
    'CPL_FLG', CPL_FLG, CPL_FLG_d, ' L '

  WRITE (nuspecif, '(T8,A,T27,I12  ,T45,L12  ,T64,A3)')                      &                      
    'dt_cp', dt_cp, dt_cp_d, ' I '

ENDIF ! my_cart_id


!------------------------------------------------------------------------------
!- End of the Subroutine
!------------------------------------------------------------------------------

END SUBROUTINE input_oasisctl 

#endif
