!+ Data module for the variables only used to compute the local information
!-------------------------------------------------------------------------------

MODULE data_obs_qc_limits

!-------------------------------------------------------------------------------
!
! Description:
!   This module contains the parameters used to derive the limits (thresholds)
!   for the quality control ('QC') of conventional observations.
!   This includes variance tables deduced from observation errors, which are
!   used as variances of the background departure (for pressure dependent
!   quality control thresholds), and which may vary from the observation error
!   variances given in module 'data_obs_cdfin'.
!    
!   Note: This module is part of the 'COSMO data assimilation library 2'
!         for observation operators including quality control.
!         It is used commonly by the COSMO model and the 3DVAR program package !
!
! Current Code Owner (for COSMO and for DWD 3DVAR):
!  DWD, Christoph Schraff  
!  phone:  +49  69  8062 2725
!  fax:    +49  69  8062 3721
!  email:  christoph.schraff@dwd.de
!
! History:
! Version    Date       Name
! ---------- ---------- ----
! V4_28        2013/07/12 Christoph Schraff
!  Initial release, by extraction from 'data_nudge_local.f90'.
!  Previous milestones in 'data_nudge_local.f90' related to QC thresholds:
!   - 1998 : Initial release.
!   - 2000 : Radiosonde height thresholds reduced to values commonly used for
!            US-network.   Height error correlation matrix added.
!   - 2001 : Variance / threshold tables for wind adjusted to values used at
!            ECMWF.   Introduction of several additional factors to thresholds
!            for selected observations following ECMWF, and to surface pressure
!            thresholds used for the spatial consistency check.
!   - 2004-2006 : New limits / errors / factors for revised optional humidity
!                 thresholds.   Revised basic humidity QC thresholds (= new
!                 GME-OI values).   Introduction of analysis layers and standard
!                 layers for revised optional multi-level check.
!
! CAUTION: This module is used commonly by the 3DVAR and COSMO main programs.!!!
!!!        Therefore, anybody wanting to introduce a modification to this    !!!
!!!        module in the context of either of these programs must consult    !!!
!!!        the 'current code owner' of this module for the other program,    !!!
!!!        in order to allow for checking that the modification will comply  !!!
!!!        with both program packages. This must be done before the          !!!
!!!        modification is put into the Version Control System (VCS).        !!!
!
! Code Description:
! Language: Fortran 90.
! Software Standards: "European Standards for Writing and
! Documenting Exchangeable Fortran 90 Code".
!===============================================================================
!
! Modules used:
!
!-------------------------------------------------------------------------------

USE data_parameters, ONLY :   &
    ireals,    & ! KIND-type parameter for real variables
    iintegers    ! KIND-type parameter for standard integer variables

!-------------------------------------------------------------------------------

IMPLICIT NONE

!===============================================================================

! Local Declarations:

!-------------------------------------------------------------------------------
! Global (i.e. public) Declarations:
!-------------------------------------------------------------------------------


!-------------------------------------------------------------------------------
! Section 1:  Parameters used for the threshold quality control of observations
!-------------------------------------------------------------------------------

!      1.1  Length and pressure levels used for the variance tables
!           -------------------------------------------------------

  INTEGER (KIND=iintegers) , PARAMETER  :: &
    nqclev  =  15    ! number of levels in the quality control threshold tables

  REAL (KIND=ireals)        ::    &
    tabqclp(nqclev)    ! ln(tabqcp(15))

  REAL (KIND=ireals)       , PARAMETER  :: &
    tabqcp (nqclev)  & ! levels of the error / threshold tables
              = (/ 100000._ireals, 85000._ireals, 70000._ireals, 50000._ireals,&
                    40000._ireals, 30000._ireals, 25000._ireals, 20000._ireals,&
                    15000._ireals, 10000._ireals,  7000._ireals,  5000._ireals,&
                     3000._ireals,  2000._ireals,  1000._ireals /)

  INTEGER (KIND=iintegers) , PARAMETER  :: &
    nqcalv  =  25    ! number of analysis layers for multi-level check plus 1

  REAL (KIND=ireals)       , PARAMETER  :: &
    tabanp (nqcalv)  & ! pressure limits of analysis layers for multi-lev. check
              = (/ 109900._ireals,100000._ireals, 97500._ireals, 95000._ireals,&
                    92500._ireals, 90000._ireals, 87500._ireals, 85000._ireals,&
                    82500._ireals, 80000._ireals, 77500._ireals, 72500._ireals,&
                    60000._ireals, 45000._ireals, 35000._ireals, 27500._ireals,&
                    22500._ireals, 17500._ireals, 12500._ireals,  8500._ireals,&
                     6000._ireals,  4000._ireals,  2500._ireals,  1500._ireals,&
                        1._ireals  /)

  INTEGER (KIND=iintegers) , PARAMETER  :: &
    nanstd (nqclev)  & ! number of analysis layers within the standard layers
                       ! [ps,925], (925,775], (775,600], (600,450], (450,p-next]
              = (/ 4, 6, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 /)


!      1.2  Tables of square roots of background departure variances
!      -------------------------------------------------------------

!           defined at the standard levels tabqcp(1:nqclev)

  REAL (KIND=ireals)       , PARAMETER  :: &
                     ! (root of) radiosonde (TEMP, PILOT) wind error variance
                     !           -----------------------------
    qcvsond (nqclev) = (/ 2.3_ireals,   2.3_ireals,   2.5_ireals,   3.0_ireals,&
                          3.5_ireals,   3.7_ireals,   3.5_ireals,   3.5_ireals,&
                          3.4_ireals,   3.3_ireals,   3.2_ireals,   3.2_ireals,&
                          3.3_ireals,   3.6_ireals,   4.5_ireals /)           ,&
                     ! (values in O.I.: 2.0, 2.4, 2.5, 3.4, 3.6, 3.8, 3.2, 3.2
                     !                  2.4, 2.2, 2.0, 2.0, 2.5, 3.0, 4.0 )
                     !
                     ! (root of) radiosonde (TEMP, PILOT) height error variance
                     !           -------------------------------
    qczsond (nqclev) = (/ 4.3_ireals,   4.4_ireals,   5.2_ireals,   8.4_ireals,&
                          9.8_ireals,  10.7_ireals,  11.8_ireals,  13.2_ireals,&
                         15.2_ireals,  18.1_ireals,  19.5_ireals,  22.5_ireals,&
                         25.0_ireals,  32.0_ireals,  40.0_ireals /)           ,&
                     ! ( values for European Radiosondes in O.I.: 1.5*table )
                     !
                     ! (root of) radiosonde temperature error variance
                     !           ----------------------
    qctsond (nqclev) = (/ 1.2_ireals,   1.0_ireals,   0.7_ireals,   0.4_ireals,&
                          0.4_ireals,   0.5_ireals,   0.5_ireals,   0.6_ireals,&
                          0.7_ireals,   0.8_ireals,   0.8_ireals,   0.9_ireals,&
                          0.9_ireals,   1.0_ireals,   1.2_ireals /)           ,&
                     ! ( ECMWF: 1.7, 1.5, 1.3, 1.2, 1.2, 1.4, 1.5, 1.5, 
                     !          1.6, 1.7, 1.8, 1.9, 2.0, 2.2, 2.5 )
                     !
                     ! (root of) AIREP wind error variance
                     !           ----------
    qcvairp (nqclev) = (/ 2.5_ireals,   2.5_ireals,   3.0_ireals,   3.5_ireals,&
                          4.0_ireals,   4.0_ireals,   4.0_ireals,   4.0_ireals,&
                          4.0_ireals,   4.0_ireals,   4.0_ireals,   4.0_ireals,&
                          4.0_ireals,   4.0_ireals,   4.0_ireals /)           ,&
                     ! (values in the past: 3.0 below 500 hPa, 4.0 above 300hPa)
                     !
                     ! (root of) AIREP temperature error variance
                     !           -----------------
    qctairp (nqclev) = (/ 1.2_ireals,   1.0_ireals,   0.7_ireals,   0.5_ireals,&
                          0.5_ireals,   0.6_ireals,   0.6_ireals,   0.7_ireals,&
                          0.8_ireals,   0.9_ireals,   1.0_ireals,   1.1_ireals,&
                          1.1_ireals,   1.2_ireals,   1.4_ireals /)           ,&
                     ! ( ECMWF: 1.4, 1.3, 1.2, 1.2, 1.2, 1.3, 1.3, 1.4, 
                     !          1.4, 1.4, 1.5, 1.6, 1.8, 2.0, 2.2 )
                     !
                     ! (root of) SYNOP wind error variance (not used for quality
                     !           ----------                 control thresholds)
    qcvsynp (nqclev) = (/ 3.6_ireals,   3.6_ireals,   5.8_ireals,   6.8_ireals,&
                          7.8_ireals,   9.8_ireals,  11.0_ireals,  11.8_ireals,&
                         11.8_ireals,  11.8_ireals,  11.8_ireals,  11.8_ireals,&
                         11.8_ireals,  11.8_ireals,  11.8_ireals /)           ,&
                     ! ( ECMWF: 3.0, 3.0, 3.0, 3.4, 3.6, 3.8, 3.2, 3.2,
                     !          2.4, 2.2, 2.0, 2.0, 2.0, 2.5, 3.0
                     !          used for sea stations only )
                     !
                     ! (root of) SYNOP height error variance (not used for qual-
                     !           ------------                 ity ctrl thresh.)
    qczsynp (nqclev) = (/ 7.0_ireals,   8.0_ireals,   8.6_ireals,  12.1_ireals,&
                         14.9_ireals,  18.8_ireals,  25.4_ireals,  27.7_ireals,&
                         32.4_ireals,  39.4_ireals,  50.3_ireals,  59.3_ireals,&
                         69.8_ireals,  96.0_ireals, 114.2_ireals /)           ,&
                     !
    qcvdrib =  5.4_ireals  ,& ! DRIBU wind    \   (root of) error variances
    qczdrib = 14.0_ireals  ,& ! DRIBU height   >    (not used for quality
    qczship = 14.0_ireals     ! SHIP height   /      control thresholds)


!      1.3  Error correlations used for the height and thickness check
!      ---------------------------------------------------------------

!           defined at the standard levels tabqcp(1:nqclev)

  REAL (KIND=ireals)        ::    &
    qczcorl(nqclev,nqclev)      ! radiosonde height error correlation matrix

  INTEGER (KIND=iintegers) , PARAMETER  :: &
    nzcorl (nqclev,nqclev) =  & ! radiosonde height error correlation matrix
                                ! ----------------------------------- [*1000]
                             RESHAPE(                                          &
 (/1000, 716, 276,  29,   5,   1,   0,   0,   0,   0,   0,   0,   0,   0,   0, &
    716,1000, 733, 183,  55,   9,   3,   1,   0,   0,   0,   0,   0,   0,   0, &
    276, 733,1000, 573, 268,  77,  31,  10,   2,   0,   0,   0,   0,   0,   0, &
     29, 183, 573,1000, 851, 480, 288, 138,  49,  11,   3,   1,   0,   0,   0, &
      5,  55, 268, 851,1000, 814, 601, 364, 167,  51,  19,   7,   1,   0,   0, &
      1,   9,  77, 480, 814,1000, 935, 738, 458, 200,  93,  44,  11,   2,   0, &
      0,   3,  31, 288, 601, 935,1000, 919, 678, 361, 194, 104,  31,   7,   0, &
      0,   1,  10, 138, 364, 738, 919,1000, 895, 597, 375, 229,  84,  24,   2, &
      0,   0,   2,  49, 167, 458, 678, 895,1000, 861, 649, 460, 214,  78,   9, &
      0,   0,   0,  11,  51, 200, 361, 597, 861,1000, 929, 782, 480, 230,  42, &
      0,   0,   0,   3,  19,  93, 194, 375, 649, 929,1000, 951, 710, 412, 103, &
      0,   0,   0,   1,   7,  44, 104, 229, 460, 782, 951,1000, 878, 598, 192, &
      0,   0,   0,   0,   1,  11,  31,  84, 214, 480, 710, 878,1000, 881, 426, &
      0,   0,   0,   0,   0,   2,   7,  24,  78, 230, 412, 598, 881,1000, 725, &
      0,   0,   0,   0,   0,   0,   0,   2,   9,  42, 103, 192, 426, 725,1000/)&
                                                           , (/nqclev,nqclev/) )


!      1.4  Further parameters used for quality control (QC)
!      ------------------------------------------------

  REAL (KIND=ireals)                    :: &
                   ! time factors (upper-air / surface-level) for QC thresholds
    qctf  (4) = (/ 0.2_ireals, 0.2_ireals, 0.2_ireals, 0.2_ireals/),& ! upperair
    qctfsu(4) = (/ 0.2_ireals, 0.2_ireals, 0.2_ireals, 0.2_ireals/)   ! surface-
                   ! fraction of the QC threshold at the obs. time,   !   level
                   ! which is added to the threshold for each hour
                   ! of difference between observation and model time;
                   ! indices: 1: horiz. wind, 2: 'surface' pressure,
                   !          3: temperature, 4: (relative) humidity
                   !

  REAL (KIND=ireals)       , PARAMETER  :: &
                   !    QC check multiples for height thresholds
    qcvfz (2) = (/ 5.48_ireals , 7.75_ireals /) 
                   ! [  threshold  =  qcvfz(jflag) * err(obs)
                   !  = SQRT(err(f.g.)**2 + err(obs)**2) * SQRT(errlim(jflag))
                   !  ~ SQRT(3) *err(obs) * SQRT(errlim(jflag))
                   !  If the deviation exceeds the threshold resulting from
                   !    errlim(1) = 10 , then: 'probably good', check thickness,
                   !                           use as limit for thickness check
                   !    errlim(2) = 20 , then: 'probably bad', reject obs.
                   !    (cf. OI code, var. 'RMULGF') ]
                   !
!   qcvfdz         ! QC check multiple for thickness threshold
!                  ! [  errlim = 5.0625  (cf. OI code, var. 'RMULTF') ]

  REAL (KIND=ireals)       , PARAMETER  :: &
    qcvairl =  5.0_ireals  ,& ! limit of background wind speed for rejection of
                              ! observed AIREP zero winds
                   ! additional factors used for the following QC thresholds:
    qcvairf =  0.8_ireals  ,& ! (ECMWF: 0.5)  AIREP wind        thresholds
    qctairf =  1.0_ireals  ,& ! (ECMWF: 1.6)  AIREP temperature thresholds
    qcvpilf =  0.9_ireals  ,& ! (ECMWF: 0.8)  PILOT wind        thresholds
    qcvdrbf =  0.5_ireals  ,& !               DRIBU wind        thresholds
    qczdrbf =  0.9_ireals  ,& !               DRIBU pressure    thresholds
                   ! factors to derive surface pressure QC thresholds
                   ! (only) if surface pressure tendency is observed:
    qcftend =  0.6_ireals  ,& ! fraction of observed tendency to be added
                              ! to pressure QC threshold
    qcfctnd =  0.8_ireals  ,& ! reduction factor to the pressure threshold
                              ! as specified by namelist input
                   ! factors to derive surface pressure thresholds used for the
                   ! spatial consistency check (SSC):
    qcfgood =  0.7_ireals  ,& ! factor to the main threshold to distinguish
                              ! 'good' from 'probably good' obs.
                              ! (all except 'good' obs. wiil be SSC-checked) 
    qcfpbad =  1.5_ireals  ,& ! factor to the main threshold to distinguish
                              ! 'probably bad' from 'bad' obs.
                              ! (all except 'bad' obs. will be used for the SSC)
    qcfspat =  0.8_ireals  ,& ! reduction factor to the threshold used for
                              ! the SSC step itself
    qcftsiv =  0.8_ireals  ,& ! reduction factor to the threshold used for
                              ! the IWV-SSC step itself
    qcfbias =  0.5_ireals  ,& ! fraction of ABS( bias ) to be added to the
                              ! threshold used for the SSC step itself
    dtchkps =  2.0_ireals  ,& ! time radius of influence for the linearly
                              ! decreasing temporal weights used in the SSC
    qcqinvf =  0.2_ireals  ,& ! humidity first guess error enhancement per [K]
                              ! of observed inversion (--> inversion term)
    qcqlapf =  0.25_ireals ,& ! maximum humidity first guess error enhancement
                              ! if obs has zero lapse rate ('stability term')
    qcqlapl = -0.0065_ireals,&! lapse rate limit: if obs profile is more stable
                              ! then humidity first guess error is enhanced
    qcqladz =  1.0_ireals  ,& ! maximum stability factor enhancement
    qcqlapi =  0.05_ireals ,& ! scaling factor of positive lapse rate for
                              ! enhancing inversion term of f.g. enhancement
    qcqfge1 =  0.10_ireals ,& ! first guess humidity error at latitude > qcqlatl
    qcqfge2 =  0.15_ireals ,& ! first guess humidity error at latitude < qcqlatl
    qcqlatl = 30.00_ireals ,& ! latitude limit (for first guess humidity error)
!   qcqfgef (3) = (/ 0.667_ireals,   1.278_ireals,   1.944_ireals /)
!   qcqfgef (3) = (/ 0.972_ireals,   1.458_ireals,   1.944_ireals /)
    qcqfgef (3) = (/ 1.800_ireals,   2.500_ireals,   3.100_ireals /)
                              ! factors for first guess humidity errors

!-------------------------------------------------------------------------------

END MODULE data_obs_qc_limits
