#ifdef OASIS_IOW_ESM
MODULE bias_corrections

USE netcdf, ONLY : &
  nf90_open,               &
  nf90_close,              &
  nf90_get_var, &
  nf90_inq_varid, &
  nf90_get_att, &
  NF90_NOWRITE, &
  NF90_NOERR

USE data_parameters,  ONLY : &
  ireals,          &
  iintegers
  
USE data_modelconfig, ONLY : &
  ie,                     &
  je,                     &
  ie_tot,                 &
  je_tot

USE data_parallel,    ONLY : &
  my_cart_id,      &
  logbuf,          &
  nproc,           & ! total number of processors: nprocx * nprocy
  imp_logical,     & ! determines the correct LOGICAL type used in the
  imp_integers,    &
  icomm_world        ! communicator that belongs to igroup_world, i.e.
                     ! = MPI_COMM_WORLD

USE environment,      ONLY : &
  model_abort, get_free_unit

USE parallel_utilities, ONLY: &
  distribute_values, distribute_field 
  
IMPLICIT NONE

PUBLIC initialize_bias_corrections

ENUM, BIND(C)
    ENUMERATOR :: E_ASWDIR_S_CORRECTION = 1
    ENUMERATOR :: E_ALWD_S_CORRECTION = 2
    ENUMERATOR :: E_N_CORRECTIONS = 2
ENDENUM

! initializes names for corrextion -> corresponds to out variable that is corrected
CHARACTER(len=8), PARAMETER, DIMENSION(E_N_CORRECTIONS) :: corrections_names = [ &
    'ASWD_S', &
    'ALWD_S' &
    ] 

REAL(KIND=ireals), ALLOCATABLE :: &
    corrections(:,:,:,:)

LOGICAL  ::  &
    lcorrections(E_N_CORRECTIONS) = .FALSE., &
    lat_night = .FALSE.
CONTAINS

SUBROUTINE process_input_corrections (nuin, ierrstat)
    
    ! Parameter list:
      INTEGER (KIND=iintegers), INTENT (IN)      ::        &
        nuin            ! Unit number for Namelist INPUT file
    
      INTEGER (KIND=iintegers), INTENT (INOUT)   ::        &
        ierrstat        ! error status variable
    
      CHARACTER (LEN=80)        ::                       &
        yerrmsg            ! error message
        
    ! Local variables: 
      INTEGER (KIND=iintegers)   ::  &
        i, iz_err

    ! Define the namelist group
      NAMELIST /correctionsctl/ lcorrections, lat_night
                          
    !------------------------------------------------------------------------------
    !- End of header -
    !------------------------------------------------------------------------------
    
    !------------------------------------------------------------------------------
    !- Begin SUBROUTINE input_oasisctl
    !------------------------------------------------------------------------------
    
    ierrstat = 0_iintegers
    iz_err   = 0_iintegers
    
    IF ( ierrstat /= 0 ) THEN
      WRITE(*,*) ' ERROR    *** Error while allocating variables in routine process_input_corrections  *** '
      ierrstat   = 2
      RETURN
    ENDIF
    
    !------------------------------------------------------------------------------
    !- Section 3: Input of the namelist values
    !------------------------------------------------------------------------------
    IF (my_cart_id == 0) THEN

      READ (nuin, correctionsctl, IOSTAT=iz_err)

    ENDIF

    IF (nproc > 1) THEN
      ! distribute error status to all processors
      CALL distribute_values  (iz_err, 1, 0, imp_integers, icomm_world, ierrstat)
    ENDIF

    IF (iz_err /= 0) THEN
      WRITE(*,*) "Could not read INPUT_BIAS, set lcorrections to default: ", lcorrections 
      ierrstat = -1
      RETURN
    ENDIF
    

    !------------------------------------------------------------------------------
    !- Section 4: Check values for errors and consistency
    !------------------------------------------------------------------------------

    IF (my_cart_id == 0) THEN
    
      DO i = 1, E_N_CORRECTIONS
          IF ( lcorrections(i) /= .FALSE. .and.  lcorrections(i) /= .TRUE.) THEN
              WRITE (*,*) ' ERROR  *** Wrong value ', lcorrections(i) , ' for correction ', i
              ierrstat = 1002 
                ierrstat = 1002 
              ierrstat = 1002 
          ENDIF
      ENDDO

      IF ( ierrstat /= 0 ) THEN
          WRITE (*,*) ' ERROR  *** Error while checking values of the namelist oasisctl *** '
          RETURN
      ENDIF
        
    
    ENDIF ! my_cart_id
    
    !------------------------------------------------------------------------------
    !- Section 5: Distribute variables to all nodes
    !------------------------------------------------------------------------------
    
    IF (nproc > 1) THEN
    
      IF (my_cart_id == 0) THEN
        DO i = 1, E_N_CORRECTIONS
            logbuf(i) = lcorrections(i)
        ENDDO

        logbuf(E_N_CORRECTIONS + 1) = lat_night 
    
          
      ENDIF ! my_cart_id == 0
      
      CALL distribute_values (logbuf ,E_N_CORRECTIONS, 0, imp_logical, icomm_world, ierrstat)
    
      IF (my_cart_id /= 0) THEN
        DO i = 1, E_N_CORRECTIONS
            lcorrections(i) = logbuf(i) 
        ENDDO

        lat_night = logbuf(E_N_CORRECTIONS + 1)

      ENDIF ! my_cart_id == 0
    
      IF ( ierrstat /= 0 ) THEN
        WRITE (*,*) ' ERROR *** in distributing buffers *** '
        RETURN
      ENDIF
    
    ENDIF ! nproc
    
    !------------------------------------------------------------------------------
    !- Section 6: Output of the namelist variables and their default values
    !------------------------------------------------------------------------------
    
    WRITE (*,*) 'Apply bias corrections for:'

    DO i = 1, E_N_CORRECTIONS
        WRITE (*,*)  corrections_names(i), lcorrections(i)
    ENDDO
    !------------------------------------------------------------------------------
    !- End of the Subroutine
    !------------------------------------------------------------------------------
    
END SUBROUTINE process_input_corrections 

SUBROUTINE read_namelist_corrections(ierror, yerrmsg)
    
    INTEGER (KIND=iintegers), INTENT(OUT)           ::                      &
      ierror       ! error status
    
    CHARACTER (LEN=  *),      INTENT(OUT)           ::                      &
      yerrmsg      ! error message
    
    !------------------------------------------------------------------------------
    ! local variables:
    INTEGER (KIND=iintegers)   ::   &
      nuin,                         &
      izerrstat
    
    CHARACTER (LEN=10)         ::   &
      yinput = 'INPUT_BIAS'
    
    !------------------------------------------------------------------------------
    !- End of header
    !------------------------------------------------------------------------------
     
    !------------------------------------------------------------------------------
    !- Begin Subroutine read_namelist_oasis
    !------------------------------------------------------------------------------
    
    ierror    = 0_iintegers
    izerrstat = 0_iintegers
    
    ! -----------------------------------------------------------------
    ! 1 Open NAMELIST-INPUT file
    ! ----------------------------------------------------------------
        
    IF (my_cart_id == 0) THEN
    
      WRITE  (*,*) '    INPUT OF THE NAMELIST FOR BIAS CORRECTIONS'

      CALL get_free_unit (nuin)
      OPEN (nuin, FILE=yinput, FORM='FORMATTED', STATUS='UNKNOWN',  &
        IOSTAT=izerrstat)
      IF (izerrstat /= 0) THEN
        yerrmsg = ' ERROR    *** Error while opening file '//yinput//' *** '
        ierror   = 2
        RETURN
      ENDIF
          
    ENDIF
        
    ! -----------------------------------------------------------------
    ! 2 read the NAMELIST group oasisctl
    ! ----------------------------------------------------------------
    
    CALL process_input_corrections (nuin, izerrstat)
    
    IF (izerrstat > 0) THEN
      yerrmsg  = ' ERROR *** Wrong values occured in NAMELIST group /correctionsctl/ *** '
      ierror   = 3
      RETURN
    ELSEIF (izerrstat < 0) THEN
      WRITE (*,*) ' ERROR while reading NAMELIST group /correctionsctl/ in process_input_corrections '
      ierror   = 4
      RETURN
    ENDIF
    
    ! -----------------------------------------------------------------
    ! 3 Close NAMELIST-INPUT file
    ! ----------------------------------------------------------------
    
    IF (my_cart_id == 0) THEN
    
      CLOSE (nuin, STATUS='KEEP', IOSTAT=izerrstat)
      IF (izerrstat /= 0) THEN
        yerrmsg = ' ERROR *** while closing file '//yinput//'*** '
        ierror  = 5
      ENDIF
    ENDIF
        
    !------------------------------------------------------------------------------
    !- End of the Subroutine
    !------------------------------------------------------------------------------
    
END SUBROUTINE read_namelist_corrections

SUBROUTINE initialize_bias_corrections(ierror, yerrmsg)

    INTEGER (KIND=iintegers), INTENT(OUT)           ::                      &
        ierror       ! error status
  
    CHARACTER (LEN=  *),      INTENT(OUT)           ::                      &
        yerrmsg      ! error message

    INTEGER(KIND=iintegers) :: &
      istatus,                 & ! NetCDF status
      ncfileid, ncvarid,       & ! NetCDF IDs
      nerror, &
      i, j

    REAL(KIND=ireals) :: &
      fillvalue     
    
    REAL(KIND=ireals), ALLOCATABLE :: &
      corrections_tot(:,:,:,:)     ! Temporary buffer for receiving

    CHARACTER(LEN=2) :: yj

    CHARACTER(LEN=128) :: filename

    ! Read namelist for corrections
    CALL read_namelist_corrections(ierror, yerrmsg)

    IF (ierror /= 0) THEN
      WRITE(*, *) yerrmsg
      WRITE(*, *) "Use defaults instead: ", lcorrections
      RETURN
    ENDIF 

    ! allocate resources for each process
    ALLOCATE(corrections(E_N_CORRECTIONS, 12, ie, je))

    ! root process allocates temporal storage for reading in the correction
    ALLOCATE(corrections_tot(E_N_CORRECTIONS, 12, ie_tot, je_tot))
    corrections_tot(:, :, :, :) = 0.0_ireals
    
    ! Read in correction for each month
    DO i = 1, E_N_CORRECTIONS
        IF (.NOT. lcorrections(i)) THEN
            CYCLE
        ENDIF

        DO j = 1, 12
            WRITE (yj,'(I2.2)') j 
            yj = ADJUSTL(yj)

            filename = 'corrections/'//TRIM(corrections_names(i))//'-'//TRIM(yj)//'.nc'

            istatus = nf90_open(TRIM(filename), NF90_NOWRITE, ncfileid)
            IF (istatus /= NF90_NOERR) THEN
                WRITE(*,*) 'Could not open ', TRIM(filename), ' for bias correction. Unset correction.'
                CYCLE
            ENDIF

            istatus = nf90_inq_varid(ncfileid, TRIM(corrections_names(i)) , ncvarid)
            IF (istatus /= NF90_NOERR) THEN
                WRITE(*,*) 'Could not get varid for variable '//TRIM(corrections_names(i))//'. Unset correction.'
                CYCLE
            ENDIF

            istatus = nf90_get_var(ncfileid, ncvarid, corrections_tot(i,j,:,:), &
                                (/ 1, 1 /),      &
                                (/ ie_tot, je_tot/))
            IF (istatus /= NF90_NOERR) THEN
                WRITE(*,*) 'Could not get variable '//TRIM(corrections_names(i))//'. Unset correction.'
                corrections_tot(i,j,:,:) = 0.0_ireals
                CYCLE
            ENDIF    

            istatus = nf90_get_att(ncfileid, ncvarid, "_FillValue", fillvalue)
            IF (istatus /= NF90_NOERR) THEN
                WRITE(*,*) 'Could not get fill value. Unset correction.'
                corrections_tot(i,j,:,:) = 0.0_ireals
                CYCLE
            ENDIF             

            istatus = nf90_close(ncfileid)
            IF (istatus /= NF90_NOERR) THEN
                WRITE(*,*) 'Could not close ', TRIM(filename), 'for bias correction.'
                CYCLE
            ENDIF

            WHERE (corrections_tot(i,j,:,:) == fillvalue) corrections_tot(i,j,:,:) = 0.0

            CALL distribute_field(corrections_tot(i,j,:,:), ie_tot, je_tot, corrections(i,j,:,:), ie, je, 0, nerror)
        ENDDO

    ENDDO

    DEALLOCATE(corrections_tot)
    WRITE (*,*) "Read in bias correction fields."

END SUBROUTINE initialize_bias_corrections

SUBROUTINE finalize_bias_corrections

    DEALLOCATE(corrections)

END SUBROUTINE finalize_bias_corrections

END MODULE bias_corrections
#endif