SUBROUTINE CUDDRAFN &
 & ( KIDIA,    KFDIA,    KLON,     KTDIA,    KLEV,&
 & LDDRAF,&
 & PTENH,    PQENH,    PUEN,     PVEN,&
 & PGEO,     PGEOH,    PAPH,     PRFL,&
 & PTD,      PQD,      PMFU,&
 & PMFD,     PMFDS,    PMFDQ,    PDMFDP,&
 & PDMFDE,   PMFDDE_RATE,        PKINED )  

!          THIS ROUTINE CALCULATES CUMULUS DOWNDRAFT DESCENT

!          M.TIEDTKE         E.C.M.W.F.    12/86 MODIF. 12/89

!          PURPOSE.
!          --------
!          TO PRODUCE THE VERTICAL PROFILES FOR CUMULUS DOWNDRAFTS
!          (I.E. T,Q,U AND V AND FLUXES)

!          INTERFACE
!          ---------

!          THIS ROUTINE IS CALLED FROM *CUMASTR*.
!          INPUT IS T,Q,P,PHI,U,V AT HALF LEVELS.
!          IT RETURNS FLUXES OF S,Q AND EVAPORATION RATE
!          AND U,V AT LEVELS WHERE DOWNDRAFT OCCURS

!          METHOD.
!          --------
!          CALCULATE MOIST DESCENT FOR ENTRAINING/DETRAINING PLUME BY
!          A) MOVING AIR DRY-ADIABATICALLY TO NEXT LEVEL BELOW AND
!          B) CORRECTING FOR EVAPORATION TO OBTAIN SATURATED STATE.

!     PARAMETER     DESCRIPTION                                   UNITS
!     ---------     -----------                                   -----
!     INPUT PARAMETERS (INTEGER):

!    *KIDIA*        START POINT
!    *KFDIA*        END POINT
!    *KLON*         NUMBER OF GRID POINTS PER PACKET
!    *KTDIA*        START OF THE VERTICAL LOOP
!    *KLEV*         NUMBER OF LEVELS

!    INPUT PARAMETERS (LOGICAL):

!    *LDDRAF*       .TRUE. IF DOWNDRAFTS EXIST

!    INPUT PARAMETERS (REAL):

!    *PTENH*        ENV. TEMPERATURE (T+1) ON HALF LEVELS          K
!    *PQENH*        ENV. SPEC. HUMIDITY (T+1) ON HALF LEVELS     KG/KG
!    *PUEN*         PROVISIONAL ENVIRONMENT U-VELOCITY (T+1)      M/S
!    *PVEN*         PROVISIONAL ENVIRONMENT V-VELOCITY (T+1)      M/S
!    *PGEO*         GEOPOTENTIAL                                  M2/S2
!    *PGEOH*        GEOPOTENTIAL ON HALF LEVELS                  M2/S2
!    *PAPH*         PROVISIONAL PRESSURE ON HALF LEVELS           PA
!    *PMFU*         MASSFLUX UPDRAFTS                           KG/(M2*S)

!    UPDATED PARAMETERS (REAL):

!    *PRFL*         PRECIPITATION RATE                           KG/(M2*S)

!    OUTPUT PARAMETERS (REAL):

!    *PTD*          TEMPERATURE IN DOWNDRAFTS                      K
!    *PQD*          SPEC. HUMIDITY IN DOWNDRAFTS                 KG/KG
!    *PMFD*         MASSFLUX IN DOWNDRAFTS                       KG/(M2*S)
!    *PMFDS*        FLUX OF DRY STATIC ENERGY IN DOWNDRAFTS       J/(M2*S)
!    *PMFDQ*        FLUX OF SPEC. HUMIDITY IN DOWNDRAFTS         KG/(M2*S)
!    *PDMFDP*       FLUX DIFFERENCE OF PRECIP. IN DOWNDRAFTS     KG/(M2*S)
!    *PMFDDE_RATE*  DOWNDRAFT DETRAINMENT RATE                   KG/(M2*S)
!    *PKINED*       DOWNDRAFT KINETIC ENERGY                     M2/S2

!          EXTERNALS
!          ---------
!          *CUADJTQ* FOR ADJUSTING T AND Q DUE TO EVAPORATION IN
!          SATURATED DESCENT

!          REFERENCE
!          ---------
!          (TIEDTKE,1989)

!          MODIFICATIONS
!          -------------
!             92-09-21 : Update to Cy44      J.-J. MORCRETTE
!             03-08-28 : Clean-up detrainment rates   P. BECHTOLD
!        M.Hamrud      01-Oct-2003 CY28 Cleaning

!----------------------------------------------------------------------

USE PARKIND1  ,ONLY : JPIM     ,JPRB
USE YOMHOOK   ,ONLY : LHOOK,   DR_HOOK

USE YOMCST   , ONLY : RG       ,RCPD     ,RETV
USE YOECUMF  , ONLY : ENTRDD   ,RMFCMIN  ,NJKT3

IMPLICIT NONE

INTEGER(KIND=JPIM),INTENT(IN)    :: KLON 
INTEGER(KIND=JPIM),INTENT(IN)    :: KLEV 
INTEGER(KIND=JPIM),INTENT(IN)    :: KIDIA 
INTEGER(KIND=JPIM),INTENT(IN)    :: KFDIA 
INTEGER(KIND=JPIM)               :: KTDIA ! Argument NOT used
LOGICAL           ,INTENT(IN)    :: LDDRAF(KLON) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PTENH(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PQENH(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PUEN(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PVEN(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PGEO(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PGEOH(KLON,KLEV+1) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PAPH(KLON,KLEV+1) 
REAL(KIND=JPRB)   ,INTENT(INOUT) :: PRFL(KLON) 
REAL(KIND=JPRB)   ,INTENT(INOUT) :: PTD(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(INOUT) :: PQD(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(IN)    :: PMFU(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(INOUT) :: PMFD(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(INOUT) :: PMFDS(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(INOUT) :: PMFDQ(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(OUT)   :: PDMFDP(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(OUT)   :: PDMFDE(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(OUT)   :: PMFDDE_RATE(KLON,KLEV) 
REAL(KIND=JPRB)   ,INTENT(OUT)   :: PKINED(KLON,KLEV)
REAL(KIND=JPRB) ::     ZDMFEN(KLON),           ZDMFDE(KLON),&
 & ZCOND(KLON),            ZOENTR(KLON),&
 & ZBUOY(KLON)  
REAL(KIND=JPRB) ::     ZPH(KLON)
LOGICAL ::  LLO2(KLON)
INTEGER(KIND=JPIM) :: ICALL, IK, IS, ITOPDE, JK, JL

REAL(KIND=JPRB) :: ZBUO, ZBUOYZ, ZBUOYV, ZDMFDP, ZDZ, ZENTR, ZMFDQK,&
 & ZMFDSK, ZQDDE, ZQEEN, ZRAIN, &
 & ZSDDE, ZSEEN, ZZENTR, ZRG, ZFACBUO, Z_CWDRAG , ZDKBUO, ZDKEN
REAL(KIND=JPRB) :: ZHOOK_HANDLE

#include "cuadjtq.intfb.h"

IF (LHOOK) CALL DR_HOOK('CUDDRAFN',0,ZHOOK_HANDLE)
ITOPDE=NJKT3
ZRG=1.0_JPRB/RG
ZFACBUO=0.5_JPRB/(1.0_JPRB+0.5_JPRB)
Z_CWDRAG=(3._JPRB/8._JPRB)*0.506_JPRB/0.2_JPRB/RG
!----------------------------------------------------------------------

!     1.           CALCULATE MOIST DESCENT FOR CUMULUS DOWNDRAFT BY
!                     (A) CALCULATING ENTRAINMENT/DETRAINMENT RATES, 
!                         INCLUDING ORGANIZED ENTRAINMENT DEPENDENT ON
!                         NEGATIVE BUOYANCY AND ASSUMING
!                         LINEAR DECREASE OF MASSFLUX IN PBL
!                     (B) DOING MOIST DESCENT - EVAPORATIVE COOLING
!                         AND MOISTENING IS CALCULATED IN *CUADJTQ*
!                     (C) CHECKING FOR NEGATIVE BUOYANCY AND
!                         SPECIFYING FINAL T,Q,U,V AND DOWNWARD FLUXES
!                    -------------------------------------------------

DO JL=KIDIA,KFDIA
  ZOENTR(JL)=0.0_JPRB
  ZBUOY(JL)=0.0_JPRB
  ZDMFEN(JL)=0.0_JPRB
  ZDMFDE(JL)=0.0_JPRB
  PDMFDE(JL,:)=0.0_JPRB
  PMFDDE_RATE(JL,:)=0.0_JPRB
  PKINED(JL,:)=0.0_JPRB
ENDDO

DO JK=3,KLEV
  IS=0
  DO JL=KIDIA,KFDIA
    ZPH(JL)=PAPH(JL,JK)
    LLO2(JL)=LDDRAF(JL).AND.PMFD(JL,JK-1) < 0.0_JPRB
    IF(LLO2(JL)) THEN
      IS=IS+1
    ENDIF
  ENDDO
  IF(IS == 0) CYCLE

  DO JL=KIDIA,KFDIA
    IF(LLO2(JL)) THEN
      ZENTR=ENTRDD*PMFD(JL,JK-1)*(PGEOH(JL,JK-1)-PGEOH(JL,JK))*ZRG
      ZDMFEN(JL)=ZENTR
      ZDMFDE(JL)=ZENTR
    ENDIF
  ENDDO

  IF(JK > ITOPDE) THEN
    DO JL=KIDIA,KFDIA
      IF(LLO2(JL)) THEN
        ZDMFEN(JL)=0.0_JPRB
        ZDMFDE(JL)=PMFD(JL,ITOPDE)*&
         & (PAPH(JL,JK)-PAPH(JL,JK-1))/&
         & (PAPH(JL,KLEV+1)-PAPH(JL,ITOPDE))  
      ENDIF
    ENDDO
  ENDIF

  IF(JK <= ITOPDE) THEN
    DO JL=KIDIA,KFDIA
      IF(LLO2(JL)) THEN
        ZDZ=-(PGEOH(JL,JK-1)-PGEOH(JL,JK))*ZRG
        ZZENTR=ZOENTR(JL)*ZDZ*PMFD(JL,JK-1)
        ZDMFEN(JL)=ZDMFEN(JL)+ZZENTR
        ZDMFEN(JL)=MAX(ZDMFEN(JL),0.3_JPRB*PMFD(JL,JK-1))
        ZDMFEN(JL)=MAX(ZDMFEN(JL),-0.75_JPRB*PMFU(JL,JK)-&
         & (PMFD(JL,JK-1)-ZDMFDE(JL)))  
        ZDMFEN(JL)=MIN(ZDMFEN(JL),0.0_JPRB)
      ENDIF

      PDMFDE(JL,JK)=ZDMFEN(JL)-ZDMFDE(JL)

    ENDDO
  ENDIF
  DO JL=KIDIA,KFDIA
    IF(LLO2(JL)) THEN
      PMFD(JL,JK)=PMFD(JL,JK-1)+ZDMFEN(JL)-ZDMFDE(JL)
      ZSEEN=(RCPD*PTENH(JL,JK-1)+PGEOH(JL,JK-1))*ZDMFEN(JL)
      ZQEEN=PQENH(JL,JK-1)*ZDMFEN(JL)
      ZSDDE=(RCPD*PTD(JL,JK-1)+PGEOH(JL,JK-1))*ZDMFDE(JL)
      ZQDDE=PQD(JL,JK-1)*ZDMFDE(JL)
      ZMFDSK=PMFDS(JL,JK-1)+ZSEEN-ZSDDE
      ZMFDQK=PMFDQ(JL,JK-1)+ZQEEN-ZQDDE
      PQD(JL,JK)=ZMFDQK*(1.0_JPRB/MIN(-RMFCMIN,PMFD(JL,JK)))
      PTD(JL,JK)=(ZMFDSK*(1.0_JPRB/MIN(-RMFCMIN,PMFD(JL,JK)))-PGEOH(JL,JK))/RCPD
      PTD(JL,JK)=MIN(400._JPRB,PTD(JL,JK))
      PTD(JL,JK)=MAX(100._JPRB,PTD(JL,JK))
      ZCOND(JL)=PQD(JL,JK)
    ENDIF
  ENDDO

  IK=JK
  ICALL=2
  CALL CUADJTQ &
   & ( KIDIA,    KFDIA,    KLON,     KTDIA,    KLEV,&
   & IK,&
   & ZPH,      PTD,      PQD,      LLO2,     ICALL )  

  DO JL=KIDIA,KFDIA
    IF(LLO2(JL)) THEN
      ZCOND(JL)=ZCOND(JL)-PQD(JL,JK)
      ZBUO=PTD(JL,JK)*(1.0_JPRB+RETV  *PQD(JL,JK))-&
       & PTENH(JL,JK)*(1.0_JPRB+RETV  *PQENH(JL,JK))  
      IF(PRFL(JL) > 0.0_JPRB.AND.PMFU(JL,JK) > 0.0_JPRB) THEN
        ZRAIN=PRFL(JL)/PMFU(JL,JK)
        ZBUO=ZBUO-PTD(JL,JK)*ZRAIN
      ENDIF
      IF(ZBUO >= 0.0_JPRB.OR.PRFL(JL) <= (PMFD(JL,JK)*ZCOND(JL))) THEN
        PMFD(JL,JK)=0.0_JPRB
        ZBUO=0.0_JPRB
      ENDIF
      PMFDS(JL,JK)=(RCPD*PTD(JL,JK)+PGEOH(JL,JK))*PMFD(JL,JK)
      PMFDQ(JL,JK)=PQD(JL,JK)*PMFD(JL,JK)
      ZDMFDP=-PMFD(JL,JK)*ZCOND(JL)
      PDMFDP(JL,JK-1)=ZDMFDP
      PRFL(JL)=PRFL(JL)+ZDMFDP

! COMPUTE ORGANIZED ENTRAINMENT FOR USE AT NEXT LEVEL

      ZBUOYZ=ZBUO/PTENH(JL,JK)
      ZBUOYV=ZBUOYZ
      ZBUOYZ=MIN(ZBUOYZ,0.0_JPRB)
      ZDZ=-(PGEO(JL,JK-1)-PGEO(JL,JK))
      ZBUOY(JL)=ZBUOY(JL)+ZBUOYZ*ZDZ
      ZOENTR(JL)=RG*ZBUOYZ*0.5_JPRB/(1.0_JPRB+ZBUOY(JL))

! STORE DOWNDRAUGHT DETRAINMENT RATES

      PMFDDE_RATE(JL,JK)=-ZDMFDE(JL)

! COMPUTE KINETIC ENERGY

      ZDKBUO=ZDZ*ZBUOYV*ZFACBUO
      IF(ZDMFEN(JL) < 0.0_JPRB)THEN
        ZDKEN=MIN(1.0_JPRB,(1 + RG*Z_CWDRAG)*&
         & ZDMFEN(JL)/MIN(-RMFCMIN,PMFD(JL,JK-1)))  
      ELSE
        ZDKEN=MIN(1.0_JPRB,(1 + RG*Z_CWDRAG)*&
         & ZDMFDE(JL)/MIN(-RMFCMIN,PMFD(JL,JK-1)))  
      ENDIF
      PKINED(JL,JK)=MAX(0.0_JPRB,(PKINED(JL,JK-1)*(1-ZDKEN)+ZDKBUO)/(1+ZDKEN))

    ENDIF
  ENDDO
  
ENDDO

IF (LHOOK) CALL DR_HOOK('CUDDRAFN',1,ZHOOK_HANDLE)
END SUBROUTINE CUDDRAFN
