SUBROUTINE check_files (ystartdate, yenddate, hinchours, yprefix1, yprefix2, ysuffix, inpdir, lastoutput, itype_calendar)

!
! Purpose: checks whether all necessary data files are available of the following file syntax
!          <prefix>yyyymmddhh<suffix>
!
!          Output on standard device: 0 = all files found
!                                     1 = missing files exist
!          Output on file "check_files.log" : listing of files
!

  IMPLICIT NONE

! Input Parameter list:
! ---------------------

  CHARACTER (LEN=*), INTENT(IN) :: &
    inpdir

  INTEGER , INTENT(IN) :: &
    itype_calendar  ! type of calendar 
                    ! 0 = standard calendar
                    ! 1 = climatological calendar (360 days per year)
                    ! 2 = climatological calendar (365 days per year)

  LOGICAL, INTENT(IN) :: &
    lastoutput      ! = true, if the last output file should be checked

  CHARACTER (LEN=10) :: &
    ystartdate,     &
    yenddate

  CHARACTER (LEN=*)  , INTENT(IN) :: &
    yprefix1,        &
    yprefix2,        &
    ysuffix

  INTEGER :: &
    hinchours

! Local variables:
! ------------------------------------

  CHARACTER (LEN=200) :: &
    bndfile1, bndfile2

  CHARACTER (LEN=10) :: &
    yactdate

  CHARACTER (LEN=2) :: &
    yhh

  INTEGER :: &
    iargc,        &
    hh_start,     &
    hh_end,       &
    hh_add,       &
    hours,        &
    endhours

  LOGICAL :: &
    lexist,   &
    lok=.TRUE.


  OPEN (1, FILE='check_files.log',form='formatted')

  yhh = ystartdate(9:10)
  READ(yhh,'(I2)') hh_start
  hh_start = hh_start - MOD(hh_start,hinchours)
  WRITE(yhh,'(I2.2)') hh_start
  ystartdate(9:10) = yhh
  yhh = yenddate(9:10)
  READ(yhh,'(I2)') hh_end
  hh_add = hinchours - MOD(hh_end,hinchours)
  IF (hh_add == hinchours) hh_add = 0
  
  IF (hh_add /= hinchours) THEN
    CALL calc_utc_date (yenddate, REAL(hh_add), itype_calendar, yactdate)
    yenddate = yactdate
  ENDIF
  
  CALL calc_hours (ystartdate, yenddate, itype_calendar, endhours)

  hours = 0
  
  IF (.NOT. lastoutput) endhours = endhours - hinchours

  DO WHILE (hours <= endhours)

    CALL calc_utc_date (ystartdate, REAL(hours), itype_calendar, yactdate)
    bndfile1 = TRIM(inpdir)//'/'//TRIM(yprefix1)//yactdate//TRIM(ysuffix)
    !... check if boundary file exists
    INQUIRE(file=bndfile1,exist=lexist)
    IF (.NOT. lexist) THEN
      IF ( TRIM(yprefix1) == TRIM(yprefix2)) THEN
        WRITE (1,*) TRIM(bndfile1),' does not exist'
        lok = .FALSE.
      ELSE
        bndfile2 = TRIM(inpdir)//'/'//TRIM(yprefix2)//yactdate//TRIM(ysuffix)
        INQUIRE(file=bndfile2,exist=lexist)
        IF (.NOT. lexist) THEN
          WRITE (1,*) TRIM(bndfile1),' does not exist'
          WRITE (1,*) TRIM(bndfile2),' does not exist'
          lok = .FALSE.
        ELSE
          WRITE (1,*) TRIM(bndfile1),' found'
        ENDIF
     ENDIF
!      PRINT *, 'missing file: ',TRIM(bndfile)
    ELSE
      WRITE (1,*) TRIM(bndfile1),' found'
    ENDIF
 
    hours = hours + hinchours
  ENDDO

  CLOSE (1)

  IF (lok) THEN
    write (*,'(A1)') '0'
  ELSE
    write (*,'(A1)') '1'
  ENDIF


END SUBROUTINE check_files

!------------------------------------------------------------------------------
!------------------------------------------------------------------------------

SUBROUTINE calc_utc_date (ystartdate, hadd, itype_calendar, yactdate)

!-------------------------------------------------------------------------------
!
! Description:
!   This routine determines the actual date of this forecast step.
!
! Method:
!   Using the date of the forecast-start, the number of time steps 
!   already performed and the length of the time steps, the actual
!   date is calculated taking leap-years into consideration.
!   The date is given in three different formats.
!
! Modules used:    NONE
!
!-------------------------------------------------------------------------------

  IMPLICIT NONE

!
! Input Parameter list:
! ---------------------

  CHARACTER (LEN=10), INTENT(IN)          ::                &
    ystartdate ! start date of the forecast

  REAL, INTENT(IN)       ::   &
    hadd       ! hours to be added to ystartdate

  INTEGER, INTENT(IN)          ::                &
    itype_calendar ! type of calendar


! Output Parameter list:
! ----------------------

  CHARACTER (LEN=10) , INTENT(OUT)         ::               &
    yactdate  ! actual date in the form   yyyymmddhh

! Local variables:

INTEGER      ::                                       &
  month(12), monthsum(13), ileap, iweek, iy, m,                       &
  idd, imm, iyy, ihh, iday, imonth, iyear, ihour, immhours, iyyhours
  CHARACTER (LEN=3)            :: yweek(7)

!------------ End of header ----------------------------------------------------

! Begin subroutine calc_utc_date

DATA         month  / 31 ,  28 ,  31 ,  30 ,  31 ,  30 ,       &
                      31 ,  31 ,  30 ,  31 ,  30 ,  31 /
DATA         yweek  /'MON', 'TUE', 'WED', 'THU', 'FRI', 'SAT', 'SUN' /


! Statementfunction: ileap(yy) = 0:  no leap year, 
!                    ileap(yy) = 1:  leap year
  ileap (iy) = IABS( MOD(iy,  4) -   4) /   4  & ! every       4 years is a leapyear
              -IABS( MOD(iy,100) - 100) / 100  & ! every     100 years is no leapyear
              +IABS( MOD(iy,400) - 400) / 400    ! but every 400 years is a leapyear

! Divide ystartdate in day, month, year and hour
! and calculate the sums of days from the beginning of the year to the 
! end of the months
  READ ( ystartdate, '(I4,3I2)' ) iyy, imm, idd, ihh

  SELECT CASE (itype_calendar)
  
  CASE (0)   !  Standard year
  
  month (2)    = 28 + ileap (iyy)
  monthsum(1) =  0
  DO m =  2 , 13
    monthsum(m) =  monthsum(m-1) + month(m-1)
  enddo

! Determine how many hours have passed in this year
  iyyhours = (idd*24) + monthsum(imm)*24 + (ihh-24)
  iyyhours = iyyhours + hadd
! Take turning of the year into account
    IF (iyyhours < 0) THEN
      iyear    = iyy
      DO WHILE (iyyhours >= (8760+ileap(iyear)*24))
       iyyhours = iyyhours - (8760+ileap(iyear)*24)
       iyear=iyear+1
       month (2)    = 28 + ileap (iyear)
       monthsum(1) =  0
       DO m =  2 , 13
           monthsum(m) =  monthsum(m-1) + month(m-1)
       ENDDO
      ENDDO
    ELSE IF (iyyhours >= (8760+ileap(iyy)*24)) THEN
      ! Take also into account if the run lasts
      ! for several years
      iyear    = iyy
      DO WHILE (iyyhours >= (8760+ileap(iyear)*24))
       iyyhours = iyyhours - (8760+ileap(iyear)*24)
       iyear=iyear+1
       month (2)    = 28 + ileap (iyear)
       monthsum(1) =  0
       DO m =  2 , 13
           monthsum(m) =  monthsum(m-1) + month(m-1)
       ENDDO
      ENDDO
    ELSE
      iyear    =   iyy
    ENDIF

! Determine the actual date from iyyhours
  m        = 1
  immhours = iyyhours
  DO WHILE (immhours >= 0)
    m        = m+1
    immhours = iyyhours - monthsum(m) * 24
  ENDDO
  imonth   = m-1
  immhours = iyyhours - monthsum(imonth)*24
  iday     = immhours/24 + 1
  ihour    = MOD(immhours,24)
  iweek    = MOD(monthsum(imonth) + iday + (iyear-1981)+(iyear-1981)/4+2 , 7) + 1

  WRITE ( yactdate(1:4) , '(I4.4)' ) iyear
  WRITE ( yactdate(5:6) , '(I2.2)' ) imonth
  WRITE ( yactdate(7:8) , '(I2.2)' ) iday
  WRITE ( yactdate(9:10), '(I2.2)' ) ihour

  CASE (1)   !  360 days per year

  monthsum(1) =  0
  DO m =  2 , 13
    monthsum(m) =  monthsum(m-1) + 30
  enddo

! Determine how many hours have passed in this year
  iyyhours = (idd*24) + monthsum(imm)*24 + (ihh-24)
  iyyhours = iyyhours + hadd

! Take turning of the year into account
  IF (iyyhours < 0) THEN
    iyear    = iyy-1
    iyyhours = 8640  + iyyhours
  ELSE IF (iyyhours >= 8640) THEN
      iyear    = iyy
    DO WHILE (iyyhours >= 8640)
      iyear    = iyear+1
      iyyhours = iyyhours - 8640
    ENDDO
  ELSE
    iyear    =   iyy
  ENDIF

! Determine the actual date from iyyhours
  m        = 1
  immhours = iyyhours
  DO WHILE (immhours >= 0)
    m        = m+1
    immhours = iyyhours - monthsum(m) * 24
  ENDDO
  imonth   = m-1
  immhours = iyyhours - monthsum(imonth)*24
  iday     = immhours/24 + 1
  ihour    = MOD(immhours,24)
  iweek    = MOD(monthsum(imonth) + iday + (iyear-1981) , 7) + 1

  WRITE ( yactdate(1:4) , '(I4.4)' ) iyear
  WRITE ( yactdate(5:6) , '(I2.2)' ) imonth
  WRITE ( yactdate(7:8) , '(I2.2)' ) iday
  WRITE ( yactdate(9:10), '(I2.2)' ) ihour

  
  CASE (2)   !  365 days per year
  

  monthsum(1) =  0
  DO m =  2 , 13
    monthsum(m) =  monthsum(m-1) + month(m-1)
  enddo

! Determine how many hours have passed in this year
  iyyhours = (idd*24) + monthsum(imm)*24 + (ihh-24)
  iyyhours = iyyhours + hadd
! Take turning of the year into account
    IF (iyyhours < 0) THEN
      iyear    = iyy
      DO WHILE (iyyhours >= 8760)
       iyyhours = iyyhours - 8760
       iyear=iyear+1
       monthsum(1) =  0
       DO m =  2 , 13
           monthsum(m) =  monthsum(m-1) + month(m-1)
       ENDDO
      ENDDO
    ELSE IF (iyyhours >= 8760) THEN
      ! Take also into account if the run lasts
      ! for several years
      iyear    = iyy
      DO WHILE (iyyhours >= 8760)
       iyyhours = iyyhours - 8760
       iyear=iyear+1
       monthsum(1) =  0
       DO m =  2 , 13
           monthsum(m) =  monthsum(m-1) + month(m-1)
       ENDDO
      ENDDO
    ELSE
      iyear    =   iyy
    ENDIF

! Determine the actual date from iyyhours
  m        = 1
  immhours = iyyhours
  DO WHILE (immhours >= 0)
    m        = m+1
    immhours = iyyhours - monthsum(m) * 24
  ENDDO
  imonth   = m-1
  immhours = iyyhours - monthsum(imonth)*24
  iday     = immhours/24 + 1
  ihour    = MOD(immhours,24)
  iweek    = MOD(monthsum(imonth) + iday + (iyear-1981)+(iyear-1981)/4+2 , 7) + 1

  WRITE ( yactdate(1:4) , '(I4.4)' ) iyear
  WRITE ( yactdate(5:6) , '(I2.2)' ) imonth
  WRITE ( yactdate(7:8) , '(I2.2)' ) iday
  WRITE ( yactdate(9:10), '(I2.2)' ) ihour

  END SELECT

END SUBROUTINE calc_utc_date

!------------------------------------------------------------------------------
!------------------------------------------------------------------------------

SUBROUTINE calc_hours (ystartdate, yenddate, itype_calendar, hours)

!-------------------------------------------------------------------------------
!
! Description:
!   This routine determines the number of hours between two dates
!
!
! Modules used:    NONE
!
!-------------------------------------------------------------------------------

  IMPLICIT NONE
!
! Input Parameter list:
! ---------------------

  CHARACTER (LEN=10)           ::                           &
    ystartdate, & ! start date of the forecast
    yenddate      ! end date of the forecast

  INTEGER, INTENT(IN)          ::                &
    itype_calendar ! type of calendar

! Output Parameter list:
! ---------------------

  INTEGER :: &
    hours

! Local variables:
  INTEGER      ::                                       &
    i, iy, &
    month(12), ileap, year_s, month_s, day_s, hour_s, year_e, month_e, day_e, hour_e, &
    full_years, full_months, full_days

!------------ End of header ----------------------------------------------------

! Begin subroutine calc_hours

DATA         month  / 31 ,  28 ,  31 ,  30 ,  31 ,  30 ,       &
                      31 ,  31 ,  30 ,  31 ,  30 ,  31 /

! Statementfunction: ileap(yy) = 0:  no leap year, 
!                    ileap(yy) = 1:  leap year
  ileap (iy) = IABS( MOD(iy,  4) -   4) /   4  & ! every       4 years is a leapyear
              -IABS( MOD(iy,100) - 100) / 100  & ! every     100 years is no leapyear
              +IABS( MOD(iy,400) - 400) / 400    ! but every 400 years is a leapyear


! Divide ystartdate in day, month, year and hour
! and calculate the sums of days from the beginning of the year to the 
! end of the months
  READ ( ystartdate, '(I4,3I2)' ) year_s, month_s, day_s, hour_s
  READ ( yenddate, '(I4,3I2)' ) year_e, month_e, day_e, hour_e

  hours = 0
  full_years = year_e - year_s
  
  SELECT CASE (itype_calendar)
  
  CASE (0)   !  Standard year
  
! year_s == year_e
    IF (year_s == year_e) THEN
      month(2) = 28 + ileap(year_s)
      IF (month_e > month_s + 1) THEN
        DO i = month_s+1, month_e-1
          hours = hours + month(i) * 24
        ENDDO
      ENDIF
      IF (month_e > month_s) THEN
        full_days = month(month_s) - day_s
        IF (full_days > 1) THEN
          hours = hours + full_days * 24
        ENDIF
        hours = hours + 24 - hour_s
        full_days = day_e - 1
        IF (full_days > 0) THEN
          hours = hours + full_days * 24
        ENDIF
        hours = hours + hour_e
      ELSE
        full_days = day_e - day_s - 1
        IF (full_days > 0) THEN
          hours = hours + full_days * 24
        ENDIF
        IF (day_e > day_s) THEN
          hours = hours + 24 - hour_s
          hours = hours + hour_e
        ELSE
          hours = hours + hour_e - hour_s
        ENDIF
      ENDIF

    ELSE
  ! year_s < year_e
    ! full years
      IF (full_years > 1) THEN
        DO i = year_s+1, year_e-1
          hours = hours + (365 + ileap(i)) * 24
        ENDDO
      ENDIF
    ! start year
      month(2) = 28 + ileap(year_s)
    ! full months
      full_months = 12 - month_s
      IF (full_months > 1) THEN
        DO i = month_s+1, 12
          hours = hours + month(i) * 24
        ENDDO
      ENDIF
    ! full days
      full_days = month(month_s) - day_s
      IF (full_days > 0) THEN
        hours = hours + full_days * 24
      ENDIF
    ! hours left
      hours = hours + 24 - hour_s

    ! end year
    ! full months
      IF (month_e > 1) THEN
        month(2) = 28 + ileap(year_e)
        DO i = 1, month_e-1
          hours = hours + month(i) * 24
        ENDDO
      ENDIF
    ! full days
      IF (day_e > 1) THEN
        hours = hours + (day_e - 1) * 24
      ENDIF
    ! hours left
      hours = hours + hour_e
      
    ENDIF

  CASE (1)   !  360 days per year

! year_s == year_e
    IF (year_s == year_e) THEN
      IF (month_e > month_s + 1) THEN
        hours = hours + (month_e - month_s - 1) * 720
      ENDIF
      IF (month_e > month_s) THEN
        full_days = 30 - day_s
        IF (full_days > 1) THEN
          hours = hours + full_days * 24
        ENDIF
        hours = hours + 24 - hour_s
        full_days = day_e - 1
        IF (full_days > 0) THEN
          hours = hours + full_days * 24
        ENDIF
        hours = hours + hour_e
      ELSE
        full_days = day_e - day_s - 1
        IF (full_days > 0) THEN
          hours = hours + full_days * 24
        ENDIF
        IF (day_e > day_s) THEN
          hours = hours + 24 - hour_s
          hours = hours + hour_e
        ELSE
          hours = hours + hour_e - hour_s
        ENDIF
      ENDIF

    ELSE
    
    ! year_s < year_e
    ! full years
      IF (full_years > 1) THEN
        hours = hours + (full_years-1) * 8640
      ENDIF
    ! start year
    ! full months
      full_months = 12 - month_s
      IF (full_months > 0) THEN
        hours = hours + full_months * 720
      ENDIF
    ! full days
      full_days = 30 - day_s
      IF (full_days > 0) THEN
        hours = hours + full_days * 24
      ENDIF
    ! hours left
        hours = hours + 24 - hour_s
    ! end year
    ! full_months
      IF (month_e > 1) THEN
        hours = hours + (month_e - 1) * 720
      ENDIF
    ! full_days
      IF (day_e > 1) THEN
        hours = hours + (day_e - 1) * 24
      ENDIF
    ! hours left
      hours = hours + hour_e
      
    ENDIF
  
  CASE (2)   !  365 days per year
  
! year_s == year_e
    IF (year_s == year_e) THEN
      IF (month_e > month_s + 1) THEN
        DO i = month_s+1, month_e-1
          hours = hours + month(i) * 24
        ENDDO
      ENDIF
      IF (month_e > month_s) THEN
        full_days = month(month_s) - day_s
        IF (full_days > 1) THEN
          hours = hours + full_days * 24
        ENDIF
        hours = hours + 24 - hour_s
        full_days = day_e - 1
        IF (full_days > 0) THEN
          hours = hours + full_days * 24
        ENDIF
        hours = hours + hour_e
      ELSE
        full_days = day_e - day_s - 1
        IF (full_days > 0) THEN
          hours = hours + full_days * 24
        ENDIF
        IF (day_e > day_s) THEN
          hours = hours + 24 - hour_s
          hours = hours + hour_e
        ELSE
          hours = hours + hour_e - hour_s
        ENDIF
      ENDIF

    ELSE
  ! year_s < year_e
    ! full years
      IF (full_years > 1) THEN
        DO i = year_s+1, year_e-1
          hours = hours + 8760
        ENDDO
      ENDIF
    ! start year
    ! full months
      full_months = 12 - month_s
      IF (full_months > 1) THEN
        DO i = month_s+1, 12
          hours = hours + month(i) * 24
        ENDDO
      ENDIF
    ! full days
      full_days = month(month_s) - day_s
      IF (full_days > 0) THEN
        hours = hours + full_days * 24
      ENDIF
    ! hours left
      hours = hours + 24 - hour_s

    ! end year
    ! full months
      IF (month_e > 1) THEN
        DO i = 1, month_e-1
          hours = hours + month(i) * 24
        ENDDO
      ENDIF
    ! full days
      IF (day_e > 1) THEN
        hours = hours + (day_e - 1) * 24
      ENDIF
    ! hours left
      hours = hours + hour_e
      
    ENDIF
    
 END SELECT
  
 
  
END SUBROUTINE calc_hours
