!+ Source module for meteorological utility routines
!==============================================================================

MODULE  meteo_utilities

!==============================================================================
!
! Description:
!   This module provides service utilities for meteorological calculations.
!     - no routine uses other modules, except the declarations for the
!       KIND-type parameter; the data access is by parameter list only
!     - no routine allocates dynamic memory; work space needed is
!       provided via the parameter list
!     - no derived data types are used
!
!   Routines (module procedures) currently contained:
!
!     - calrho
!       Computes the air density for time-level nx
!
!     - calrho_densities
!       Computes the air density for time-level nx
!
!     - calrho_tp_pp
!       Computes the air density for time-level nx
!
!     - calps
!       Computes the surface pressure ps from the lowest full model level.
!
!     - cloud_diag
!       computes the cloud water and - cover including subscale clouds
!
!     - moist_split
!       Splits the generalized relative humidity into vapor, cloud water
!       and cloud ice, according to current temperature and pressure
!       (mixed phase cloud)
!
!     - satad
!       Corrects the temperature, the specific humidity and the cloud water
!       content for condensation/evaporation.
!
!     - psat_w
!       Saturation water vapour pressure
!
!     - qsat
!       Specific humidity at saturation pressure
!
!     - tgcom
!       Computes the ground temperature.
!
!
! Current Code Owner: DWD, Ulrich Schaettler
!  phone:  +49  69  8062 2739
!  fax:    +49  69  8062 3721
!  email:  ulrich.schaettler@dwd.de
!
! History:
! Version    Date       Name
! ---------- ---------- ----
! 1.1        2005/04/11 Ulrich Schaettler
!  Initial release for INT2LM
! 1.2        2005/07/22 Jochen Foerstner, Ulrich Schaettler
!  Included new SR for computing calrho_densities
!  Modification in SR satad, to "save" the start values for the temperature
! V1_5         2007/07/09 Ulrich Schaettler
!  Eliminated unused statement functions; more adaptations to COSMO-Model
! V1_9         2009/09/03 Guenther Zaengl
!  Additional subroutine for new reference atmosphere;
!  Adaptations to COSMO-Model version
!  New subroutine moist_split for splitting generalized relative humidity
! V1_11        2010/04/23 Michael Gertz
!  Adaptions to SVN
! V1_12        2010/06/14 Oliver Fuhrer
!  Modified version of SR moist_split, to enable vectorization
! V1_14        2010/11/19 Ulrich Schaettler
!  Update to latest COSMO-Model version
! V1_19        2012/06/06 Ulrich Schaettler
!  Update to latest COSMO-Model version 4.23
! V1_20        2012/09/03 Michael Baldauf
!  Changes in subroutines for reference_atmospheres (as in COSMO-Model 4.24):
!    introduced new argument t0hl and its computation
!    removed computations of k-indices for special pressure levels
!  Introduced new subroutine k_index_of_pressure_levels for this task
! V1_22        2013/07/11 Ulrich Schaettler
!  Moved subroutines for reference atmospheres to module vgrid_refatm_utils
!
! Code Description:
! Language: Fortran 90.
! Software Standards: "European Standards for Writing and
! Documenting Exchangeable Fortran 90 Code".
!==============================================================================
!
! Declarations:
!
! Modules used:
!------------------------------------------------------------------------------

USE data_parameters , ONLY :   &
  ireals,    & ! KIND-type parameters for real variables
  iintegers    ! KIND-type parameter for standard integer variables

!------------------------------------------------------------------------------

IMPLICIT NONE

!------------------------------------------------------------------------------

CONTAINS

!==============================================================================

!------------------------------------------------------------------------------

SUBROUTINE calrho  ( t, pp, qv, qc, qrs, p0, rho, ie, je, ke,  &
                     r_d, rvd_m_o )

!------------------------------------------------------------------------------
! Description:
!   This routine computes the air density for present time level.
!
! Method:
!   Application of the gas-law, rho = p/(r_d*tv)
!
!------------------------------------------------------------------------------
!
! Parameter list:
INTEGER (KIND=iintegers), INTENT (IN)    ::    &
  ie, je, ke          ! dimensions of the fields

REAL (KIND=ireals), INTENT (IN)          ::    &
  t    (ie,je,ke),   & ! temperature
  pp   (ie,je,ke),   & ! pressure perturbation
  qv   (ie,je,ke),   & ! specific humidity
  qc   (ie,je,ke),   & ! specific cloud water content
  qrs  (ie,je,ke),   & ! specific rain and snow content
  p0   (ie,je,ke),   & ! base-state pressure
  r_d            ,   & ! gas constant of dry air
  rvd_m_o              ! r_v/r_d - 1

REAL (KIND=ireals), INTENT (OUT)       ::    &
  rho  (ie,je,ke)      ! air density

!-------------------------------------------------------------------------------
! Begin subroutine calrho

  rho = ( p0 + pp ) / ( r_d*t*(1.0 + rvd_m_o*qv - qc - qrs) )

END SUBROUTINE calrho

!==============================================================================
!==============================================================================

!------------------------------------------------------------------------------

SUBROUTINE calrho_densities( t, pp, rho_v, rho_c, rho_rs, p0, rho,        &
                             ie, je, ke, r_d, rvd_m_o )

!------------------------------------------------------------------------------
! Description:
!   This routine computes the air density for present time level.
!
! Method:
!   Application of the gas-law, but input quantitities are densities
!
!------------------------------------------------------------------------------
!
! Parameter list:
INTEGER (KIND=iintegers), INTENT (IN)    ::    &
  ie, je, ke            ! dimensions of the fields

REAL (KIND=ireals), INTENT (IN)          ::    &
  t     (ie,je,ke),   & ! temperature
  pp    (ie,je,ke),   & ! pressure perturbation
  rho_v (ie,je,ke),   & ! density humidity
  rho_c (ie,je,ke),   & ! density cloud water content
  rho_rs(ie,je,ke),   & ! density rain and snow content
  p0    (ie,je,ke),   & ! base-state pressure
  r_d             ,   & ! gas constant of dry air
  rvd_m_o               ! r_v/r_d - 1

REAL (KIND=ireals), INTENT (OUT)       ::    &
  rho  (ie,je,ke)       ! air density

!------------------------------------------------------------------------------
! Begin subroutine calrho_densities

  rho = ( p0 + pp ) / ( r_d * t ) - ( rvd_m_o*rho_v - rho_c - rho_rs )

END SUBROUTINE calrho_densities

!==============================================================================
!==============================================================================

!------------------------------------------------------------------------------

SUBROUTINE calrho_tp_pp( tp, pp, qv, qc, qrs, t0, p0, rho,    &
  ie, je, ke, r_d, rvd_m_o )

!------------------------------------------------------------------------------
! Description:
!   This routine computes the air density for present time level.
!
! Method:
!   Application of the gas-law, rho = p/(r_d*tv)
!
!------------------------------------------------------------------------------
!
! Parameter list:
INTEGER (KIND=iintegers), INTENT (IN)    ::    &
  ie, je, ke           ! dimensions of the fields

REAL (KIND=ireals), INTENT (IN)          ::    &
  tp   (ie,je,ke),   & ! temperature perturbation
  pp   (ie,je,ke),   & ! pressure perturbation
  qv   (ie,je,ke),   & ! specific humidity
  qc   (ie,je,ke),   & ! specific cloud water content
  qrs  (ie,je,ke),   & ! specific rain and snow content
  t0   (ie,je,ke),   & ! base-state temperature
  p0   (ie,je,ke),   & ! base-state pressure
  r_d            ,   & ! gas constant of dry air
  rvd_m_o              ! r_v/r_d - 1

REAL (KIND=ireals), INTENT (OUT)       ::    &
  rho  (ie,je,ke)      ! air density

!------------------------------------------------------------------------------
! Begin subroutine calrho_tp_pp

  rho = ( p0 + pp ) / ( r_d*( t0 + tp ) * (1.0 + rvd_m_o*qv - qc - qrs) )

END SUBROUTINE calrho_tp_pp

!==============================================================================
!==============================================================================

!------------------------------------------------------------------------------

SUBROUTINE calps ( ps, pp, t, qv, qc, qrs, rho0, p0, dp0,          &
                   ie, je, rvd_m_o, r_d, istart, iend, jstart, jend )

!------------------------------------------------------------------------------
!
! Description:
!   This routine computes the surface pressure ps by extrapolating the
!   nonhydrostatic pressure of the lowest full model level in a
!   hydrostatic manner.
!
!   The fields are passed twodimensional, that means the calling procedure has
!   to choose the proper level and time level.
!
! Method:
!   Formula to be described in the documentation
!
!-------------------------------------------------------------------------------

! Parameter list:

INTEGER (KIND=iintegers), INTENT (IN)    ::    &
  ie, je,                     & ! dimensions of the fields
  istart, iend, jstart, jend    ! start and end-indices of the computation

REAL (KIND=ireals), INTENT (IN)          ::    &
  pp   (ie,je)    , & ! perturbation pressure
  t    (ie,je)    , & ! temperature
  qv   (ie,je)    , & ! specific water vapor content
  qc   (ie,je)    , & ! specific cloud water content
  qrs  (ie,je)    , & ! precipitation water (water loading)
  p0   (ie,je)    , & ! reference pressure at full levels
  dp0  (ie,je)    , & ! full level pressure thickness
  rho0 (ie,je)        ! reference density at full levels

REAL (KIND=ireals), INTENT (INOUT)       ::    &
  ps   (ie,je)        ! surface pressure

REAL (KIND=ireals), INTENT (IN)          ::    &
  rvd_m_o,          & ! r_v/r_d - 1
  r_d                 ! gas constant for dry air

!-------------------------------------------------------------------------------

! Begin subroutine calps

  ps (istart:iend,jstart:jend) =                                              &
       ( p0(istart:iend,jstart:jend) + pp(istart:iend,jstart:jend) )          &
   *EXP (  0.5*dp0(istart:iend,jstart:jend)  /                                &
         ( t(istart:iend,jstart:jend)                                         &
           * (1.0 + rvd_m_o*qv(istart:iend,jstart:jend)                       &
              - qc(istart:iend,jstart:jend) - qrs(istart:iend,jstart:jend))   &
           * r_d * rho0(istart:iend,jstart:jend)) )

END SUBROUTINE calps

!==============================================================================
!==============================================================================

!------------------------------------------------------------------------------

SUBROUTINE cloud_diag ( clc, clwc,                         &
                        iis, iie, ijs, ije, iks, ike,      &
                        ids, ide, jds, jde, kds, kde,      &
                        ie , je , ke , ke1,                &
                        rdv, o_m_rdv, rvd_m_o, lhocp, t0,  &
                        b1, b2w, b3, b4w, b234w, b2i, b4i, &
                        uc1, uc2, ucl, clc_diag, q_crit,   &
                        t, qv, qc, p_tot,  rcld, ps,       &
                        itype_wcld )

!------------------------------------------------------------------------------
!
! Description:
!
!     This routine calculates the area fraction of a grid box covered
!     by stratiform (non-convective) clouds.
!     If subgrid-scale condensation is required, an additional
!     saturation adjustment is done.
!
! Method:
!
!     itype_wcld = 1 :
!     The fractional cloud cover clc is determined empirically from
!     relative humidity. Also, an in-cloud water content of sugrid-scale
!     clouds is determined as a fraction of the saturation specific
!     humidity. Both quantities define the grid-volume mean cloud water
!     content.
!     itype_wcld=2:
!     A Gaussion distribution is assumed for the saturation deficit
!     dq = qt - qs where qt = qv + ql is the total water content and
!     qs is the saturation specific humidity. Using the standard deviation
!     rcld of this distribution (on input) and the conservative grid-scale
!     quantities qt and tl (liquid water temperature), a corrected liquid
!     water content is determined which contains alse the contributions from
!     subgrid-scale clouds. A corresponding cloudiness is also calculated.
!
!------------------------------------------------------------------------------

! Subroutine arguments
!----------------------

! Scalar arguments with intent(in):

INTEGER (KIND=iintegers), INTENT (IN) :: &  ! dimensions and run indices
  ie, je, ke, ke1, itype_wcld,   & !
  iis, ijs, iks, iie, ije, ike,  & !
  ids, jds, kds, ide, jde, kde

REAL (KIND=ireals), INTENT (IN)       :: &  ! Some physical constants
  rdv, o_m_rdv, rvd_m_o,  & ! Rd/Rv, 1-Rd/Rv, Rv/Rd-1
  lhocp, t0,              & ! Lev/Cpd, freezing temperature
  b1, b3, b2w, b4w,       & ! Constants to calculate saturation vapour pressure
  b2i, b4i, b234w,        & !  "
  uc1,uc2,ucl,            & ! empirical constants to calculate cloud cover
  clc_diag,               & ! cloud cover at saturation
  q_crit                    ! critical value for normalized over-saturation

! Array arguments with intent(in):

REAL (KIND=ireals), INTENT (IN)        :: & !
  t    (ie,je,ke ),    &    ! temperature (main levels)
  qv   (ie,je,ke ),    &    ! water vapour (")
  qc   (ie,je,ke ),    &    ! cloud water  (")
  p_tot(ie,je,ke ),    &    ! full pressure (")
  rcld (ie,je,ke1),    &    ! standard deviation of saturation deficit
  ps   (ie,je)              ! surface pressure

! Array arguments with intent(out):

REAL (KIND=ireals), INTENT (OUT)        :: &
  clc (ids:ide,jds:jde,kds:kde),  & ! stratiform subgrid-scale cloud cover
  clwc(ids:ide,jds:jde,kds:kde)     ! liquid water content of ""

! Local variables and constants
! -----------------------------

INTEGER (KIND=iintegers) :: &
  i,j,k,n                   ! loop indices

REAL (KIND=ireals), PARAMETER :: &
  zsig_max = 1.0E-3_ireals,  & ! max. standard deviation of saturation deficit
  zclwfak  = 0.005_ireals,   & ! fraction of saturation specific humidity
  zuc      = 0.95_ireals       ! constant for critical relative humidity

REAL (KIND=ireals)       :: &
  temp, pres, ql, qt, qs, tl, dq, & !
  gam, q, sig, uc,                & !
  zsigma, zclc1, zq_max             !

REAL (KIND=ireals)       :: &
  zpsat_w, zqvap, zdqsdt,       & !statement functions and
  zpvap, zqsat, ztemp, zpres      !their formal arguments

!------------ End of header ---------------------------------------------------

! Definition of statement functions:

! saturation vapour pressure over water (zpsat_w) and over ice (zpsat_i):
  zpsat_w(ztemp) = b1 * EXP( b2w*(ztemp-b3)/(ztemp-b4w) )
! zpsat_i(ztemp) = b1 * EXP( b2i*(ztemp-b3)/(ztemp-b4i) )

! specific humidity:
  zqvap(zpvap,zpres) = rdv * zpvap / ( zpres - o_m_rdv*zpvap )

! Derivation of zqsat with respect to temperature:
  zdqsdt(ztemp,zqsat) = b234w * ( 1.0_ireals + rvd_m_o*zqsat ) * zqsat &
                             / (ztemp-b4w)**2

! Begin Subroutine cloud_diag
! ---------------------------

  zq_max   = q_crit*(1.0_ireals/clc_diag - 1.0_ireals)

  DO k = iks, ike
    DO j = ijs, ije
      DO i = iis, iie

        ql   = qc(i,j,k)               ! cloud water content
        qt   = ql + qv(i,j,k)          ! total water content
        pres = p_tot(i,j,k)            ! pressure
        temp = t(i,j,k)                ! temperature
        tl   = temp - lhocp*ql         ! liquid water temperature
        qs   = zqvap(zpsat_w(tl),pres) ! saturation mixing ratio
        dq   = qt - qs                 ! saturation deficit
        gam  = 1.0_ireals / ( 1.0_ireals + lhocp*zdqsdt(tl,qs) )

        IF ( itype_wcld == 1 ) THEN

        ! Calculation of cloud cover and cloud water content
        ! using an empirical relative humidity criterion

          zsigma = pres / ps(i,j)

          ! critical relative humidity
          uc     = zuc - uc1 * zsigma * ( 1.0_ireals - zsigma )  &
                   * ( 1.0_ireals + uc2*(zsigma-0.5_ireals) )

          ! cloud cover
          clc(i,j,k) = MAX( 0.0_ireals,  &
                       MIN( 1.0_ireals, clc_diag * ((qt/qs-uc)/(ucl-uc))) )**2

          ! in-cloud water content
          ql = qs * zclwfak

          ! grid-volume water content
          IF ( dq > 0.0_ireals ) THEN
            zclc1 = clc_diag * ( (1.0_ireals-uc)/(ucl-uc) )**2
            ql    = ql + (gam*dq-ql)*(clc(i,j,k)-zclc1)/(1.0_ireals-zclc1)
          END IF
          ql = clc(i,j,k) * ql

        ELSEIF ( itype_wcld == 2 ) THEN

        ! Statistical calculation of cloud cover and cloud water content
        ! using the standard deviation of the saturation deficit

          sig = MIN ( zsig_max, rcld(i,j,k) )

          ! in case of sig=0, the method is similar to grid-scale
          ! saturation adjustment. Otherwise, a fractional cloud cover
          ! is diagnosed.
          IF ( sig <= 0.0_ireals ) THEN
            clc(i,j,k) = ABS ( (SIGN(1.0_ireals,dq)+1.0_ireals)*0.5_ireals )
            ql         = clc(i,j,k) * gam * dq
          ELSE
            q          = dq / sig
            clc(i,j,k) = MIN ( 1.0_ireals, MAX ( 0.0_ireals, &
                                        clc_diag * (1.0_ireals+q/q_crit) ) )
            IF ( q <= - q_crit ) THEN
              ql = 0.0_ireals
            ELSEIF ( q >= zq_max ) THEN
              ql = gam * dq
            ELSE
              ql = gam * sig * (q+q_crit) * (q+zq_max) / (2_ireals*(q_crit+zq_max))
            ENDIF
          ENDIF

        ENDIF

        clwc(i,j,k) = ql

     ENDDO
   ENDDO
 ENDDO

END SUBROUTINE cloud_diag

!==============================================================================
!==============================================================================

SUBROUTINE moist_split (t, p, grh, qvmin, qcmin, qimin, pi,                 &
                        b1, b2_w, b2_i, b3, b4_w, b4_i, Rdv, O_m_rdv,       &
                        qv, qc, qi, ie, je)

!-------------------------------------------------------------------------------
!
! Description:
!   This routine splits the generalized relative humidity into vapor, liquid 
!   water and ice, accordingly to the temperature and to the pressure.
!
! Method:
!   Tests based on the temperature, allowing representation of mixed phase 
!   clouds, using saturation functions above water and above ice.
!
!-------------------------------------------------------------------------------

! Subroutine arguments:
! --------------------

 INTEGER (KIND=iintegers), INTENT (IN)    ::  &
    ie, je

 REAL    (KIND=ireals),    INTENT (IN)    ::  &
    t(ie,je),                       & ! Current temperature
    p(ie,je),                       & ! Current pressure
    grh(ie,je),                     & ! Generalized relative humidity
    qvmin, qcmin, qimin,            & ! Lowest limit values
    b1, b2_w, b2_i, b3, b4_w, b4_i, & ! physical parameters
    Rdv, O_m_rdv, pi                  ! physical parameters

 REAL    (KIND=ireals),    INTENT (OUT)   ::  &
    qv(ie,je),       & ! Specific water vapor content (specific humidity)
    qc(ie,je),       & ! Specific cloud water content
    qi(ie,je)          ! Specific cloud ice content

!------------------------------------------------------------------------------

! Local scalars:
! -------------
  REAL    (KIND=ireals   ) ::  &
    T1, T2, Thom, Tzero,                     & ! Threshold temperatures
    alpha(ie,je), beta(ie,je),               & ! Cloud characteristic functions
    zaq, zaqi, zbq(ie,je), zbqi(ie,je),      & ! Saturation functions
    zot1mt2, zotzmt1, ztmp

  INTEGER (KIND=iintegers) :: &
    i,j

! Definition of statement functions:
! ---------------------------------

REAL    (KIND=ireals)    ::   sf_psat_w, sf_psat_i,sf_qsat, x, y, z, zi, v, w, wi

sf_psat_w (x,y,z,v,w)  = y * EXP (z*(x-v)/(x-w))              ! Saturation pressure above water

sf_psat_i (x,y,zi,v,wi)= y * EXP(zi*(x-v)/(x-wi))             ! Saturation pressure above ice

sf_qsat    (x,y,z,v)   = z * x / MAX( (y-v*x), 1.0_ireals)    ! Saturation specific humidity


!------------ End of header ---------------------------------------------------

!------------------------------------------------------------------------------
! Section 1: Initializations and definitions
!------------------------------------------------------------------------------

! 1.1. Threshold temperature values

Tzero   = 271.15_ireals ! threshold for heterogeneous freezing of raindrops (or +/- zero Celsius)
T1      = 248.15_ireals ! threshold for het.nucleation of cloud ice (water-saturation above)
Thom    = 236.15_ireals ! homogeneous ice nucleation
T2      = 236.15_ireals ! threshold for hom. freezing (ice-saturation under)

zot1mt2 = 1.0_ireals/(T1-T2)
zotzmt1 = 1.0_ireals/(Tzero-T1)

DO i = 1, ie
  DO j = 1, je

    ! 1.2. Saturation functions

    zaq  = sf_psat_w (t(i,j), b1, b2_w, b3, b4_w)
    zaqi = sf_psat_i (t(i,j), b1, b2_i, b3, b4_i)
    zbq(i,j)  = sf_qsat   (zaq,  p(i,j), Rdv, O_m_rdv)
    zbqi(i,j) = sf_qsat   (zaqi, p(i,j), Rdv, O_m_rdv)

    ! 1.3. Alpha function (= characteristic relative humidity, function of cloud temperature)
    !      Note: Alpha varies linearly between 1.0 and psati/psatl for temperature from T2 to T1

    ztmp = MIN(1.0_ireals,MAX(0.0_ireals,(T1-t(i,j))*zot1mt2))
    alpha(i,j) = 1.0_ireals-(1.0_ireals-zaqi/zaq)*ztmp

    ! 1.4. Beta function (= characteristic ratio between ice and (ice+water) in clouds)

    beta(i,j) = 1.0_ireals
    ztmp = MIN(1.0_ireals,MAX(0.0_ireals,(t(i,j)-T1)*zotzmt1))
    beta(i,j) = 0.5_ireals*(COS(pi*ztmp)+1.0_ireals)
    IF ((t(i,j) < Tzero) .AND. alpha(i,j) < 1.0_ireals) THEN
      beta(i,j) = 1.0_ireals
    ENDIF

  END DO
END DO

!------------------------------------------------------------------------------
! Section 2: Split accordingly to temperature and pressure
!------------------------------------------------------------------------------

  !********************************************
  !                                           *
  ! Adaptation based on an original idea      *
  !         of  Ulrike Wacker                 *
  !                                           *
  !********************************************

DO i = 1, ie
  DO j = 1, je

    ! CASE A: Saturation impossible (grh <= alpha)

    qv(i,j) = grh(i,j) * zbq(i,j)
    qc(i,j) = 0.0_ireals
    qi(i,j) = 0.0_ireals

    ! CASE B: Saturation possible (grh > alpha)

    ! Case B1: T >= 0 , thus no ice
    IF ((grh(i,j) > alpha(i,j)) .AND. (t(i,j) >= Tzero)) THEN
      qv(i,j) = MIN(1.0_ireals, grh(i,j)) * zbq(i,j)
      qc(i,j) = MAX(0.0_ireals, grh(i,j)-1.0_ireals) * zbq(i,j)
      qi(i,j) = 0.0_ireals
    END IF

    ! Case B2a: Thom < T < Tzero , thus mixed phase possible, saturation over water
    IF ( (grh(i,j) > alpha(i,j)) .AND. (t(i,j) < Tzero) .AND. (t(i,j) > Thom) .AND. (alpha(i,j) == 1.0_ireals)) THEN
      qv(i,j) = MIN(1.0_ireals, grh(i,j)) * zbq(i,j)
      qc(i,j) = MAX(0.0_ireals, grh(i,j)-1.0_ireals) * zbq(i,j) * (1.0_ireals-beta(i,j))
      qi(i,j)=  MAX(0.0_ireals, grh(i,j)-1.0_ireals) * zbq(i,j) * beta(i,j)
    END IF

    ! Case B2b: Thom < T < Tzero , thus mixed phase possible, saturation over ice
    IF ( (grh(i,j) > alpha(i,j)) .AND. (t(i,j) < Tzero) .AND. (t(i,j) > Thom) .AND. (alpha(i,j) < 1.0_ireals)) THEN
      qv(i,j) = MIN(1.0_ireals, grh(i,j)) * zbqi(i,j)
      qc(i,j) = 0.0_ireals
      qi(i,j)=  MAX(0.0_ireals, grh(i,j)-1.0_ireals) * zbqi(i,j)
    ENDIF

    ! Case B3: T < Thom , thus no water
    IF ( (grh(i,j) > alpha(i,j)) .AND. (t(i,j) <= Thom)) THEN
      qv(i,j) = MIN(1.0_ireals, grh(i,j)) * zbqi(i,j)
      qc(i,j) = 0.0_ireals
      qi(i,j)=  MAX(0.0_ireals, grh(i,j)-1.0_ireals) * zbqi(i,j)
    ENDIF

  END DO
END DO

!------------------------------------------------------------------------------
! Section 3: Final values check and lower limit restriction (security)
!------------------------------------------------------------------------------

 qv(:,:) = MAX(qv(:,:),qvmin)
 qc(:,:) = MAX(qc(:,:),0.0_ireals)
 qi(:,:) = MAX(qi(:,:),0.0_ireals)

!------------ End of the subroutine -------------------------------------------

END SUBROUTINE moist_split

!==============================================================================
!==============================================================================

SUBROUTINE SATAD ( kitera, te, qve, qce, tstart, phfe,                        &
                   zdqd  , zqdwe, zh   , ztg0  , ztgn, zdqdt0, zgqd0, zphe ,  &
                   b1, b2w, b3, b4w, b234w, rdrd, emrdrd, rddrm1, lh_v, cpdr, &
                   cp_d, idim, jdim, ilo, iup, jlo, jup )

!-------------------------------------------------------------------------------
!
! Description:
!   This routine corrects the temperature (te), the specific humidity (qve)
!   and the cloud water content (qce) for condensation/evaporation.
!
! Method:
!   Saturation adjustment, i.e. reversible condensation/evaporation at
!   constant pressure by assuming chemical equilibrium of water and vapor
!
!-------------------------------------------------------------------------------

! Subroutine arguments:
! --------------------
  INTEGER (KIND=iintegers), INTENT (IN)    ::  &
    kitera,              & !  Numver of iterations in the numerical scheme
    idim, jdim,          & !  Dimension of I/O-fields
    ilo, iup, jlo, jup     !  start- and end-indices for the computation

  REAL    (KIND=ireals),    INTENT (IN)    ::  &
    tstart  (idim,jdim), & ! Start temperature for iteration
    phfe    (idim,jdim)  ! Pressure (input)

  REAL    (KIND=ireals),    INTENT (INOUT) ::  &
    te      (idim,jdim), & ! Temperature on input/ouput
    qve     (idim,jdim), & ! Specific humidity on input/output
    qce     (idim,jdim), & ! Specific cloud water content on input/output
    zdqd    (idim,jdim), & !
    zqdwe   (idim,jdim), & !
    zh      (idim,jdim), & !
    ztg0    (idim,jdim), & !
    ztgn    (idim,jdim), & !
    zdqdt0  (idim,jdim), & !
    zgqd0   (idim,jdim), & !
    zphe    (idim,jdim)    !

  REAL    (KIND=ireals),    INTENT (IN)    ::  &
    b1, b2w, b3, b4w, b234w, rdrd, emrdrd, rddrm1, lh_v, cpdr, cp_d

! Local parameters: None
! ----------------
! Local scalars:
! -------------
  INTEGER (KIND=iintegers) ::  &
    i, j,                & !  Loop indices
    nzit,                & !  Loop for iterations
    nsat,                & !  Number of saturated gridpoints
    iwrk(idim*jdim),     & !  i-index of saturated gridpoints
    jwrk(idim*jdim),     & !  j-index of saturated gridpoints
    indx                   !  loop index

  REAL    (KIND=ireals   ) ::  &
    zgeu  ,              & !
    zgqdu ,              & !
    zgew  ,              & !
    zqwmin,              & ! Minimum cloud water content for adjustment
    fgew  ,              & ! Name of satement function
    fgqd  ,              & ! ...
    fdqdt ,              & ! ...
    zt    ,              & ! Dummy argument for statement functions
    zge   ,              & ! ...
    zp    ,              & ! ...
    zgqd                   ! ...

  REAL    (KIND=ireals   ) ::  &
    minzdqd

!------------ End of header ----------------------------------------------------

!-------------------------------------------------------------------------------
! Begin Subroutine satad
!-------------------------------------------------------------------------------

! STATEMENT FUNCTIONS

fgew(zt)       = b1*EXP( b2w*(zt-b3)/(zt-b4w) )
fgqd(zge,zp)   = rdrd*zge/( zp - emrdrd*zge )
fdqdt(zt,zgqd) = b234w*( 1.0 + rddrm1*zgqd )*zgqd/( zt-b4w )**2

  zqwmin = 1.0E-20_ireals

  nsat = 0

  minzdqd= 1.0_ireals

  DO j = jlo , jup
    DO i = ilo , iup

      ! "save" the start values for the temperature
      ztg0 (i,j) = tstart(i,j)

      ! correction for negative values of qv and qc
      qve (i,j) = MAX( qve(i,j), 0.0_ireals )
      qce (i,j) = MAX( qce(i,j), 0.0_ireals )

      ! assume first subsaturation
      zqdwe(i,j)= qve(i,j) + qce(i,j)
      te (i,j)  = te(i,j) - lh_v*qce(i,j)*cpdr
      qve(i,j)  = zqdwe(i,j)
      qce(i,j)  = 0.0_ireals
      zgeu      = fgew(te(i,j))
      zgqdu     = fgqd(zgeu,phfe(i,j))
      zdqd(i,j) = zgqdu - zqdwe(i,j)
      minzdqd   = MIN(minzdqd,zdqd(i,j))

    ENDDO
  ENDDO

!NEC_CB if zdqd>=0, then for sure no points are found
  IF ( minzdqd >= 0.0_ireals ) RETURN

  DO j = jlo , jup
    DO i = ilo , iup

      IF (zdqd(i,j) < 0.0_ireals ) THEN
        nsat       = nsat+1
        iwrk(nsat) = i
        jwrk(nsat) = j
      ENDIF

    ENDDO
  ENDDO

  IF (nsat == 0) RETURN

! Do saturation adjustments for saturated gridpoints
! --------------------------------------------------

!cdir nodep
  DO indx = 1, nsat
     i = iwrk(indx)
     j = jwrk(indx)
     zh   (i,j) = cp_d*te(i,j) + lh_v*qve(i,j)
     zphe (i,j) = phfe(i,j)
     zgew       = fgew(ztg0(i,j))
     zgqd0(i,j) = fgqd(zgew,zphe(i,j))
  ENDDO

  IF ( kitera > 1 ) THEN
    DO  nzit  = 1 , kitera-1

!cdir nodep
      DO indx = 1, nsat
        i = iwrk(indx)
        j = jwrk(indx)
        zdqdt0(i,j) = fdqdt(ztg0(i,j),zgqd0(i,j))
        ztg0(i,j)   = (zh(i,j) - lh_v*(zgqd0(i,j)-zdqdt0(i,j)*ztg0(i,j)))/ &
                      ( cp_d + lh_v*zdqdt0(i,j) )
        zgew        = fgew(ztg0(i,j))
        zgqd0(i,j)  = fgqd(zgew,zphe(i,j))
      ENDDO
    ENDDO
  ENDIF

!-------------------------------------------------------------------------------

!cdir nodep
  DO indx = 1, nsat
      i = iwrk(indx)
      j = jwrk(indx)
      zdqdt0(i,j) = fdqdt(ztg0(i,j),zgqd0(i,j))
      ztgn(i,j)   = ( zh(i,j) - lh_v*(zgqd0(i,j)-zdqdt0(i,j)*ztg0(i,j)) ) / &
                    ( cp_d + lh_v*zdqdt0(i,j) )
      zgqd0(i,j)  = zgqd0(i,j) + zdqdt0(i,j)*( ztgn(i,j)-ztg0(i,j) )
  ENDDO

! Distribute the result on gridpoints
! -----------------------------------

!cdir nodep
  DO indx = 1, nsat
      i = iwrk(indx)
      j = jwrk(indx)
      te (i,j) =  ztgn(i,j)
      qve(i,j) = zgqd0(i,j)
      qce(i,j) = MAX( zqdwe(i,j) - zgqd0(i,j), zqwmin )
  ENDDO

! End of the subroutine

END SUBROUTINE satad

!*******************************************************************************

FUNCTION psat_w(tx, b1, b2_w, b3, b4_w)

!-------------------------------------------------------------------------------
!
! Description:
!   Saturation water vapour pressure (with respect to water,
!   depending on the temperature "tx")
!
!-------------------------------------------------------------------------------

  REAL (KIND=ireals), INTENT(IN) :: tx, b1, b2_w, b3, b4_w

  REAL (KIND=ireals) :: psat_w

  psat_w     = B1*EXP(B2_w*(tx - B3)/(tx - B4_w))

END FUNCTION psat_w

!==============================================================================
!==============================================================================

FUNCTION qsat(psatx, px, Rdv, O_m_rdv)

!-------------------------------------------------------------------------------
!
! Description:
!   Specific humidity at saturation pressure (depending on the
!   saturation water vapour pressure "psatx" and the air pressure "px")
!
!-------------------------------------------------------------------------------

  REAL (KIND=ireals), INTENT(IN) :: psatx, px, Rdv, O_m_rdv

  REAL (KIND=ireals) :: qsat

!US changed by Davide, because of problems with ECMWF data high up in the sky
  qsat = Rdv*psatx/MAX((px-O_m_rdv*psatx),1.0_ireals)

!US old version:
! qsat = Rdv*psatx/(px-O_m_rdv*psatx)

END FUNCTION qsat

!==============================================================================
!==============================================================================

SUBROUTINE tgcom (tg, ts, tb, ws, llp, ie, je, cf_snow,                 &
                  istart, iend, jstart, jend)

!-------------------------------------------------------------------------------
!
! Description:
!   Computation of the temperature tg at the boundary layer between the ground
!   and the atmosphere. Only 2-dimensional arrays can be passed to tgcom. It
!   must be called using the desired time level.
!
! Method:
!   For grid points above water and for grid points on land that are not
!   covered with snow:   tg = ground surface temperature tb
!   For snow covered land points, tg is a function of the temperature of the
!   the snow surface ts and the ground surface temperature tb:
!       tg = ts + exp( -rhde*ws ) * (tb-ts)
!   from Version 2.18 on replaced by
!       tg = ts + ( 1. - MIN(1.,ws/cf_snow)) * (tb -ts)
!
!-------------------------------------------------------------------------------

! Parameter list:

INTEGER (KIND=iintegers), INTENT (IN)    ::    &
  ie, je,                     & ! dimensions of the fields
  istart, iend, jstart, jend    ! start and end-indices of the computation

REAL (KIND=ireals), INTENT (INOUT)       ::    &
  tg (ie,je)    ! temperature at the boundary between ground and atmosphere

REAL (KIND=ireals), INTENT (IN)          ::    &
  ts (ie,je), & ! temperature of the snow surface
  tb (ie,je), & ! temperature of the ground surface
  ws (ie,je)    ! water content of snow

LOGICAL,  INTENT (IN)                    ::    &
  llp (ie,je)   ! pattern of land- and sea-points

REAL (KIND=ireals), INTENT (IN)          ::    &
  cf_snow       ! factor for the computation

!-------------------------------------------------------------------------------

! Begin subroutine tgcom

  WHERE ( (llp(istart:iend,jstart:jend) .EQV. .TRUE.) .AND.                   &
                                       (ws(istart:iend,jstart:jend) > 0.0) )
      tg(istart:iend,jstart:jend) = ts(istart:iend,jstart:jend) +             &
           (1.0_ireals - MIN(1.0_ireals,ws(istart:iend,jstart:jend)/cf_snow)) &
             * (tb(istart:iend,jstart:jend) - ts(istart:iend,jstart:jend))
  ELSEWHERE
      tg(istart:iend,jstart:jend) =   tb(istart:iend,jstart:jend)
  END WHERE

END SUBROUTINE tgcom

!==============================================================================

END MODULE meteo_utilities
