#!/usr/bin/env python

import matplotlib.pyplot as plt
import matplotlib.colors as mcolors
import matplotlib.path as mpath
import numpy as np
import numpy.ma as ma
import netCDF4
import cartopy
import cartopy.crs as ccrs
import string

# Figure 11: ENSO
#==================
# Load datasets (Monthly intantaneous snapshots)
f1 = netCDF4.Dataset('Ts_ENSO_yrs1600-1950_GMDplot.nc')

# Grid info
lons = f1.variables['longitude'][:]
lats = f1.variables['latitude'][:]

# De-seasoned SST over tropical Pacific
tsp  = f1.variables['Ts'][:,:,:]

n3p4 = np.mean(np.mean(tsp[:,9:13,28:46],axis=1),axis=1)
#==================
def ENSOFglobarr( ax, inarray, cmin=None, cmax=None, pal='seismic', clev=6 , clab=6):

#    ax.set_global()
    ax.set_extent([111, 288, -30, 30], crs=ccrs.PlateCarree())
    ax.coastlines()
    ax.gridlines()
    ax.add_feature(cartopy.feature.LAND, zorder=0)

    cs = ax.pcolormesh(lons ,lats, inarray, cmap=pal, vmin=cmin, vmax=cmax, transform=ccrs.PlateCarree())
    cbar = plt.colorbar(mappable=cs,orientation='horizontal',ax=ax,shrink=.6,pad=0.02, extend='both')
    cbar.ax.tick_params(labelsize=14) 
    plt.tight_layout()
#==================

# Define composites for El Nino and La Nina states, where nino3.4 anomalies exceed +/- 1 std
npstd = np.std(n3p4)
idxp = np.where(n3p4 > n3p4*0+npstd)
idxn = np.where(n3p4 < n3p4*0-npstd)
npm = np.mean(tsp[idxp[0],:,:],axis=0)
nnm = np.mean(tsp[idxn[0],:,:],axis=0)

clev=6
clab=6

fig = plt.figure(figsize=(18,12))

cy = np.linspace(0,4200,4200)/12

ax0 = plt.subplot2grid((10, 10), (0, 0),colspan=10,rowspan=4, 
                       projection=ccrs.Mercator(central_longitude=-180.0, min_latitude=-30.0, max_latitude=30.0, 
                                                globe=None, latitude_true_scale=0.0))
ENSOFglobarr( ax0, npm, cmin=-1.5, cmax=1.5, pal="RdBu_r", clev=clev, clab=clab)
ax0.text(-0.05, 1.05, 'a', transform=ax0.transAxes, size=20, weight='bold')

ax3 = plt.subplot2grid((10, 10), (4, 0),colspan=10,rowspan=4, 
                       projection=ccrs.Mercator(central_longitude=-180.0, min_latitude=-30.0, max_latitude=30.0, 
                                                globe=None, latitude_true_scale=0.0))
ENSOFglobarr( ax3, nnm, cmin=-1.5, cmax=1.5, pal="RdBu_r", clev=clev, clab=clab)
ax3.text(-0.05, 1.05, 'b', transform=ax3.transAxes, size=20, weight='bold')

ax1 = plt.subplot2grid((10, 10), (8, 0),colspan=8,rowspan=2)
npstd = np.std(n3p4)
ax1.plot(cy,n3p4)
ax1.plot(cy,n3p4*0+npstd)
ax1.plot(cy,n3p4*0-npstd)
ax1.set_xlabel("Time (years)", fontsize=14)
ax1.set_ylabel("$\Delta$T ($^o$C)", fontsize=14)
ax1.set_ylim(-3,3)
ax1.text(-0.05, 1.05, 'c', transform=ax1.transAxes, size=20, weight='bold')

ax2 = plt.subplot2grid((10, 10), (8, 8),colspan=2,rowspan=2)
ax2.hist(n3p4,bins=31,range=[-3,3],orientation='horizontal',density=True)
ax2.set_yticks([])
ax2.text(-0.05, 1.05, 'd', transform=ax2.transAxes, size=20, weight='bold')

plt.savefig("Figures/FORTE2_Figure11.png", dpi=300, transparent=True, bbox_inches='tight', pad_inches=0.1)


#Figure 13: NAO
#==================
# Load datasets (Daily data)
f1 = netCDF4.Dataset('FORTE_NAO_GMDplot.nc')
f2 = netCDF4.Dataset('C20th_NAO_GMDplot.nc')

# Grid info
lon  = f1.variables['longitude'][:]
lat  = f1.variables['latitude'][:]
pcs  = f1.variables['pcs'][:]
eofs = f1.variables['eofs'][:]
evs  = f1.variables['evs'][:]


olon  = f2.variables['longitude'][:]
olat  = f2.variables['latitude'][:]
opcs  = f2.variables['pcs'][:]
oeofs = f2.variables['eofs'][:]
oevs  = f2.variables['evs'][:]

print("FORTE EOF variances",evs[:5]*100)
print("Obs   EOF variances",oevs[:5]*100)

# Create figure
#==================
def Fglobarr( ax, ilon, ilat, inarray, cmin=None, cmax=None, pal='seismic', clev=6 , clab=6):

    ax.set_global()
    ax.set_extent([-90, 40, 20, 80], crs=ccrs.PlateCarree())
    ax.coastlines()
    ax.gridlines()
    ax.add_feature(cartopy.feature.LAND, zorder=0)
    # Make a boundary path in PlateCarree projection, I choose to start in
    # the bottom left and go round anticlockwise, creating a boundary point
    # every 1 degree so that the result is smooth:
    vertices = [(lon, 20) for lon in range(-90, 41, 1)] +                [(lon, 80) for lon in range(40, -81, -1)]
    boundary = mpath.Path(vertices)
    ax.set_boundary(boundary, transform=ccrs.PlateCarree())

    cs = ax.pcolormesh(ilon ,ilat, inarray, cmap=pal, vmin=cmin, vmax=cmax, transform=ccrs.PlateCarree())
    cbar = plt.colorbar(mappable=cs,orientation='horizontal',ax=ax,shrink=.6,pad=0.05, extend='both')
    cbar.ax.tick_params(labelsize=20) 
#==================

fig = plt.figure(figsize=(20,10))

comp=0
# Figure 1a
clev = [-2,2,6,10,14,18,22,26]
clab = [2,10,18,26]
ax1 = plt.subplot2grid((3, 4), (0, 0),colspan=2,rowspan=2, projection=ccrs.AlbersEqualArea(central_longitude=-20.0, central_latitude=50.0, 
                                                                false_easting=0.0, false_northing=0.0, 
                                                                standard_parallels=(20.0, 50.0), globe=None))
Fglobarr( ax1, lon, lat, eofs[comp,:,:], cmin=-4, cmax=4, pal="RdBu_r", clev=clev, clab=clab)
ax1.text(-0.05, 1.05, 'a', transform=ax1.transAxes, size=20, weight='bold')
ax1.text(0.5, 1.0, "    {:.2f}%".format(evs[comp]*100), transform=ax1.transAxes, size=20, ha='center')

# Figure 1b
ax2 = plt.subplot2grid((3, 4), (2, 0),colspan=2)
plt.plot(pcs[:,comp]*0,color='k')
plt.plot(pcs[:,comp],color='b')
ax2.tick_params(labelsize=18)
ax2.set_xlabel("Time (years)", fontsize=20)
ax2.set_ylabel("NAO Index", fontsize=20)
ax2.set_ylim(-3,3)
ax2.text(-0.05, 1.05, 'b', transform=ax2.transAxes, size=20, weight='bold')

# Figure 1c
clev = [-2,2,6,10,14,18,22,26]
clab = [2,10,18,26]
ax1 = plt.subplot2grid((3, 4), (0, 2),colspan=2,rowspan=2, projection=ccrs.AlbersEqualArea(central_longitude=-20.0, central_latitude=50.0, 
                                                                false_easting=0.0, false_northing=0.0, 
                                                                standard_parallels=(20.0, 50.0), globe=None))
Fglobarr( ax1, olon, olat, oeofs[comp,:,:], cmin=-4, cmax=4, pal="RdBu_r", clev=clev, clab=clab)
ax1.text(-0.05, 1.05, 'c', transform=ax1.transAxes, size=20, weight='bold')
ax1.text(0.5, 1.0, "    {:.2f}%".format(oevs[comp]*100), transform=ax1.transAxes, size=20, ha='center')

# Figure 1d
ax2 = plt.subplot2grid((3, 4), (2, 2),colspan=2)
plt.plot(opcs[:,comp]*0,color='k')
plt.plot(opcs[:,comp],color='b')
ax2.tick_params(labelsize=18)
ax2.set_xlabel("Time (years)", fontsize=20)
ax2.set_ylabel("NAO Index", fontsize=20)
ax2.set_ylim(-3,3)
ax2.text(-0.05, 1.05, 'd', transform=ax2.transAxes, size=20, weight='bold')

plt.tight_layout()
plt.savefig("Figures/FORTE2_Figure13.png", dpi=300, transparent=True, bbox_inches='tight', pad_inches=0.1)


# Figure 12: AO (Daily data)
#==================
# Load datasets (Daily data)
f1 = netCDF4.Dataset('FORTE_AO_GMDplot.nc')
f2 = netCDF4.Dataset('C20th_AO_GMDplot.nc')

# Grid info
lon  = f1.variables['longitude'][:]
lat  = f1.variables['latitude'][:]
pcs  = f1.variables['pcs'][:]
eofs = f1.variables['eofs'][:]
evs  = f1.variables['evs'][:]

olon  = f2.variables['longitude'][:]
olat  = f2.variables['latitude'][:]
opcs  = f2.variables['pcs'][:]
oeofs = f2.variables['eofs'][:]
oevs  = f2.variables['evs'][:]

print("FORTE EOF variances",evs[:5]*100)
print("Obs   EOF variances",oevs[:5]*100)

# Create figure
#==================
def Fglobarr( ax, ilon, ilat, inarray, cmin=None, cmax=None, pal='seismic', clev=6 , clab=6):

    from cartopy.util import add_cyclic_point

    ax.set_global()
    ax.coastlines()
    ax.gridlines()
    ax.add_feature(cartopy.feature.LAND, zorder=0)
    ilonsc = add_cyclic_point(ilon)
    inarrayc = add_cyclic_point(inarray)

    cs = ax.pcolormesh(ilonsc ,ilat, inarrayc, cmap=pal, vmin=cmin, vmax=cmax, transform=ccrs.PlateCarree())
    cbar = plt.colorbar(mappable=cs,orientation='horizontal',ax=ax,shrink=.6,pad=0.02, extend='both')
    cbar.ax.tick_params(labelsize=20) 
#==================

fig = plt.figure(figsize=(20,15))

comp=0
# Figure 1a
clev = [-2,2,6,10,14,18,22,26]
clab = [2,10,18,26]
ax1 = plt.subplot2grid((5, 4), (0, 0),colspan=2,rowspan=4, projection=ccrs.Orthographic(central_latitude=90.0, central_longitude=-20.0))
Fglobarr( ax1, lon, lat, eofs[comp,:,:], cmin=-4, cmax=4, pal="RdBu_r", clev=clev, clab=clab)
ax1.text(-0.05, 1.05, 'a', transform=ax1.transAxes, size=20, weight='bold')
ax1.text(0.5, 1.02, " {:.2f}%".format(evs[comp]*100), transform=ax1.transAxes, size=20, ha='center')

# Figure 1b
ax2 = plt.subplot2grid((5, 4), (4, 0),colspan=2)
plt.plot(pcs[:,comp]*0,color='k')
plt.plot(pcs[:,comp],color='b')
ax2.tick_params(labelsize=18)
ax2.set_xlabel("Time (years)", fontsize=20)
ax2.set_ylabel("AO Index", fontsize=20)
ax2.set_ylim(-3,3)

# Figure 1c
clev = [-2,2,6,10,14,18,22,26]
clab = [2,10,18,26]
ax1 = plt.subplot2grid((5, 4), (0, 2),colspan=2,rowspan=4, projection=ccrs.Orthographic(central_latitude=90.0, central_longitude=-20.0))
Fglobarr( ax1, olon, olat, oeofs[comp,:,:], cmin=-4, cmax=4, pal="RdBu_r", clev=clev, clab=clab)
ax1.text(-0.05, 1.05, 'c', transform=ax1.transAxes, size=20, weight='bold')
ax1.text(0.5, 1.02, " {:.2f}%".format(oevs[comp]*100), transform=ax1.transAxes, size=20, ha='center')

# Figure 1d
ax2 = plt.subplot2grid((5, 4), (4, 2),colspan=2)
plt.plot(opcs[:,comp]*0,color='k')
plt.plot(opcs[:,comp],color='b')
ax2.tick_params(labelsize=18)
ax2.set_xlabel("Time (years)", fontsize=20)
ax2.set_ylabel("AO Index", fontsize=20)
ax2.set_ylim(-3,3)
ax2.text(-0.05, 1.05, 'd', transform=ax2.transAxes, size=20, weight='bold')

plt.tight_layout()
plt.savefig("Figures/FORTE2_Figure12.png", dpi=300, transparent=True, bbox_inches='tight', pad_inches=0.1)


# Figure 14: SAM (Daily data)
#==================
# Load datasets (Daily data)
f1 = netCDF4.Dataset('FORTE_SAM_GMDplot.nc')
f2 = netCDF4.Dataset('C20th_SAM_GMDplot.nc')

# Grid info
lon  = f1.variables['longitude'][:]
lat  = f1.variables['latitude'][:]
pcs  = f1.variables['pcs'][:]
eofs = f1.variables['eofs'][:]
evs  = f1.variables['evs'][:]

olon  = f2.variables['longitude'][:]
olat  = f2.variables['latitude'][:]
opcs  = f2.variables['pcs'][:]
oeofs = f2.variables['eofs'][:]
oevs  = f2.variables['evs'][:]

print("FORTE EOF variances",evs[:5]*100)
print("Obs   EOF variances",oevs[:5]*100)

# Create figure: FORTE SAM
#==================
def Fglobarr( ax, ilon, ilat, inarray, cmin=None, cmax=None, pal='seismic', clev=6 , clab=6):

    from cartopy.util import add_cyclic_point

    ax.set_global()
    ax.coastlines()
    ax.gridlines()
    ax.add_feature(cartopy.feature.LAND, zorder=0)
    ilonsc = add_cyclic_point(ilon)
    inarrayc = add_cyclic_point(inarray)

    cs = ax.pcolormesh(ilonsc ,ilat, inarrayc, cmap=pal, vmin=cmin, vmax=cmax, transform=ccrs.PlateCarree())
    cbar = plt.colorbar(mappable=cs,orientation='horizontal',ax=ax,shrink=.6,pad=0.02, extend='both')
    cbar.ax.tick_params(labelsize=20) 
#==================

fig = plt.figure(figsize=(20,15))

comp=0
# Figure 1a
clev = [-2,2,6,10,14,18,22,26]
clab = [2,10,18,26]
ax1 = plt.subplot2grid((5, 4), (0, 0),colspan=2,rowspan=4, projection=ccrs.Orthographic(central_latitude=-90.0, central_longitude=-20.0))
Fglobarr( ax1, lon, lat, eofs[comp,:,:]*-1, cmin=-5, cmax=5, pal="RdBu_r", clev=clev, clab=clab)
ax1.text(-0.05, 1.05, 'a', transform=ax1.transAxes, size=20, weight='bold')
ax1.text(0.5, 1.02, " {:.2f}%".format(evs[comp]*100), transform=ax1.transAxes, size=20, ha='center')

# Figure 1b
ax2 = plt.subplot2grid((5, 4), (4, 0),colspan=2)
plt.plot(pcs[:,comp]*0,color='k')
plt.plot(pcs[:,comp]*-1,color='b')
ax2.tick_params(labelsize=18)
ax2.set_xlabel("Time (months)", fontsize=20)
ax2.set_ylabel("SAM Index", fontsize=20)
ax2.text(-0.05, 1.05, 'b', transform=ax2.transAxes, size=20, weight='bold')

# Figure 1c
clev = [-2,2,6,10,14,18,22,26]
clab = [2,10,18,26]
ax1 = plt.subplot2grid((5, 4), (0, 2),colspan=2,rowspan=4, projection=ccrs.Orthographic(central_latitude=-90.0, central_longitude=-20.0))
Fglobarr( ax1, olon, olat, oeofs[comp,:,:]*-1, cmin=-5, cmax=5, pal="RdBu_r", clev=clev, clab=clab)
ax1.text(-0.05, 1.05, 'c', transform=ax1.transAxes, size=20, weight='bold')
ax1.text(0.5, 1.02, " {:.2f}%".format(oevs[comp]*100), transform=ax1.transAxes, size=20, ha='center')

# Figure 1d
ax2 = plt.subplot2grid((5, 4), (4, 2),colspan=2)
plt.plot(opcs[:,comp]*0,color='k')
plt.plot(opcs[:,comp]*-1,color='b')
ax2.tick_params(labelsize=18)
ax2.set_xlabel("Time (months)", fontsize=20)
ax2.set_ylabel("SAM Index", fontsize=20)
ax2.text(-0.05, 1.05, 'd', transform=ax2.transAxes, size=20, weight='bold')

plt.tight_layout()
plt.savefig("Figures/FORTE2_Figure14.png", dpi=300, transparent=True, bbox_inches='tight', pad_inches=0.1)
