#
# ###############################################################
# Load station data from CAPITOUL campaign
# The data are averaged over the hour before the full hour
# in order to be consistant with time averaged model output
# ###############################################################
#
seload_capcnrm_stations=function(path,filesave) {
  #
  # The time period of interest for the evaluation against CAPITOUL stations
  #
  xmin=as.Date(paste(2004,"/",03,"/",01,sep=""))
  xmax=as.Date(paste(2005,"/",02,"/",28,sep=""))
  montbegvec=seq(xmin,xmax,"month")
  #
  xmin=as.Date(paste(2004,"/",04,"/",01,sep=""))
  xmax=as.Date(paste(2005,"/",03,"/",31,sep=""))
  montnexvec=seq(xmin,xmax,"month")
  #
  # #######################################
  # Load meta data
  # #######################################
  #
  filemeta=paste(path,"lat_lon_cnrm_stations",sep="")
  data=read.table(filemeta,header=FALSE,skip=0)
  DIM=dim(data)
  if (DIM[2]!=6) stop("Error in read of metadata")
  #
  name_cnrm_cap = as.vector(data$V1)
  numb_cnrm_cap = as.vector(data$V3)
  lats_cnrm_cap = as.vector(data$V4)
  lons_cnrm_cap = as.vector(data$V5)
  heig_cnrm_cap = as.vector(data$V6)
  #
  if ((max(lats_cnrm_cap)> 090.01)||(min(lats_cnrm_cap) < (-0.01))) stop("Error in station latitude")
  if ((max(lons_cnrm_cap)> 180.01)||(min(lons_cnrm_cap) < (-0.01))) stop("Error in station longitude")
  if ((max(heig_cnrm_cap)> 280.01)||(min(heig_cnrm_cap) < (130)  )) stop("Error in station height")
  #
  # Define the short name as a function of the zone
  #
  shortname=array(NA,31)
  shortname[24]="Mo"
  shortname[25]="Tg"
  shortname[26]="Tm"
  shortname[27]="Az"
  shortname[29]="La"
  shortname[30]="fa"
  shortname[31]="Cs"
  #
  # #######################################
  # Load data
  # #######################################
  #
  # The time of day axis
  #
  minutes=seq(60,1440,60)/1440
  #
  # Determine the total length of the time series 
  # and allocate the output field
  #
  nyear=substr(montbegvec[1],1,4)
  nmont=substr(montbegvec[1],6,7)
  juldaybeg=myjulday(as.integer(nmont),1,as.integer(nyear))
  #
  nextyear=substr(montnexvec[length(montnexvec)],1,4)
  nextmont=substr(montnexvec[length(montnexvec)],6,7)
  juldaynex=myjulday(as.integer(nextmont),1,as.integer(nextyear))-1
  #
  lenfullts=length(minutes)*(juldaynex-juldaybeg+1)
  #
  # Check whether it is one year (for CAPITOUL)
  #
  if (lenfullts!=(365*24)) stop("Observations should be for one year")
  #
  # Allocate the fields
  #
  name = array(NA,length(numb_cnrm_cap))
  numb = array(NA,length(numb_cnrm_cap))
  lats = array(NA,length(numb_cnrm_cap))
  lons = array(NA,length(numb_cnrm_cap))
  heig = array(NA,length(numb_cnrm_cap))
  #
  juld_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  temp_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  relh_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  qspe_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  pabs_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  rrat_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  widd_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  wiff_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  frvd_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  frvm_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  fird_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  firm_fullts = array(NA,c(length(numb_cnrm_cap),lenfullts))
  #
  # Loop over stations and time periods
  #
  ind_zone = 1
  #
  for (zone in numb_cnrm_cap) {
    #
    name[ind_zone] = name_cnrm_cap[ind_zone]
    numb[ind_zone] = numb_cnrm_cap[ind_zone]
    lats[ind_zone] = lats_cnrm_cap[ind_zone]
    lons[ind_zone] = lons_cnrm_cap[ind_zone]
    heig[ind_zone] = heig_cnrm_cap[ind_zone]
    #
    time_index = 1
    #
    for (tt in 1:length(montbegvec)) {
      #
      nyear=substr(montbegvec[tt],1,4)
      nmont=substr(montbegvec[tt],6,7)
      #
      juldaybeg=myjulday(as.integer(nmont),1,as.integer(nyear))
      #
      nextyear=substr(montnexvec[tt],1,4)
      nextmont=substr(montnexvec[tt],6,7)
      #
      juldaynex=myjulday(as.integer(nextmont),1,as.integer(nextyear))-1
      #
      juldayfrall=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      temp=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      relh=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      qspe=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      pabs=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      rrat=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      widd=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      wiff=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      frvd=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      frvm=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      fird=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      firm=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      #
      for (day in juldaybeg:juldaynex) {
        #
        ind1=1+length(minutes)*(day-juldaybeg)
        ind2=  length(minutes)*(1+day-juldaybeg)
        #
        juldayfrall[ind1:ind2]=day
        juldayfrall[ind1:ind2]=juldayfrall[ind1:ind2]+minutes
        #
      }
      #
      filein=paste(path,shortname[zone],"_BDD_",nmont,"-",nyear,".asc",sep="")
      #
      print(paste("Read ",filein,sep=""))
      #
      if (file.exists(filein)) {
        #
        system(paste("sed -i 's/,/./g' ",filein,sep=""))
        #
        data=read.table(filein,header=FALSE,skip=15,sep="")
        #
        dateread = as.vector(data$V1)
        timeread = as.vector(data$V2)
        tempread = as.numeric(as.vector(data$V3))
        pabsread = as.numeric(as.vector(data$V4))
        relhread = as.numeric(as.vector(data$V5))
        rratread = as.numeric(as.vector(data$V6))
        widdread = as.numeric(as.vector(data$V7))
        wiffread = as.numeric(as.vector(data$V8))
        frvdread = as.numeric(as.vector(data$V9))
        frvmread = as.numeric(as.vector(data$V10))
        firdread = as.numeric(as.vector(data$V11))
        firmread = as.numeric(as.vector(data$V12))
        #
        if (length(which(is.na(tempread)))>0.5) stop("NA generated in read of temp")
        if (length(which(is.na(pabsread)))>0.5) stop("NA generated in read of pabs")
        if (length(which(is.na(relhread)))>0.5) stop("NA generated in read of relh")
        if (length(which(is.na(rratread)))>0.5) stop("NA generated in read of rrat")
        if (length(which(is.na(widdread)))>0.5) stop("NA generated in read of widd")
        if (length(which(is.na(wiffread)))>0.5) stop("NA generated in read of wiff")
        if (length(which(is.na(frvdread)))>0.5) stop("NA generated in read of frvd")
        if (length(which(is.na(frvmread)))>0.5) stop("NA generated in read of frvm")
        if (length(which(is.na(firdread)))>0.5) stop("NA generated in read of fird")
        if (length(which(is.na(firmread)))>0.5) stop("NA generated in read of firm")
        #
        # Values that are 9999 are set to NA
        #
        tempread[which(tempread==9999)]=NA
        pabsread[which(pabsread==9999)]=NA
        relhread[which(relhread==9999)]=NA
        rratread[which(rratread==9999)]=NA
        widdread[which(widdread==9999)]=NA
        wiffread[which(wiffread==9999)]=NA
        frvdread[which(frvdread==9999)]=NA
        frvmread[which(frvmread==9999)]=NA
        firdread[which(firdread==9999)]=NA
        firmread[which(firmread==9999)]=NA
        #
        # Plausibility checks
        #
        if (length(which(!is.na(tempread)))>0.5) {
          if (max(tempread,na.rm=TRUE)>(+45)) {print(max(tempread,na.rm=TRUE));stop("Unplausible temperature")}
          if (min(tempread,na.rm=TRUE)<(-20)) {print(min(tempread,na.rm=TRUE));stop("Unplausible temperature")}
        }
        #
        if (length(which(!is.na(pabsread)))>0.5) {
          if (max(pabsread,na.rm=TRUE)>(+1050)) {print(max(pabsread,na.rm=TRUE));stop("Unplausible pressure")}
          if (min(pabsread,na.rm=TRUE)<(+0960)) {print(min(pabsread,na.rm=TRUE));stop("Unplausible pressure")}
        }
        #
        if (length(which(!is.na(relhread)))>0.5) {
          if (max(relhread,na.rm=TRUE)>(+105)) {print(max(relhread,na.rm=TRUE));stop("Unplausible relative humidity")}
          if (min(relhread,na.rm=TRUE)<(+010)) {print(min(relhread,na.rm=TRUE));stop("Unplausible relative humidity")}
        }
        #
        if (length(which(!is.na(rratread)))>0.5) {
          if (max(rratread,na.rm=TRUE)>(+200   )) {print(max(rratread,na.rm=TRUE));stop("Unplausible rain rate")}
          if (min(rratread,na.rm=TRUE)<(-1.0e-6)) {print(min(rratread,na.rm=TRUE));stop("Unplausible rain rate")}
        }
        #
        if (length(which(!is.na(widdread)))>0.5) {
          if (max(widdread,na.rm=TRUE)>(+360   )) {print(max(widdread,na.rm=TRUE));stop("Unplausible wind direction")}
          if (min(widdread,na.rm=TRUE)<(-1.0e-6)) {print(min(widdread,na.rm=TRUE));stop("Unplausible wind direction")}
        }
        #
        if (length(which(!is.na(wiffread)))>0.5) {
          if (max(wiffread,na.rm=TRUE)>(+30))     {print(max(wiffread,na.rm=TRUE));stop("Unplausible wind speed")}
          if (min(wiffread,na.rm=TRUE)<(-1.0e-6)) {print(min(wiffread,na.rm=TRUE));stop("Unplausible wind speed")}
        }
        #
        if (length(which(!is.na(frvdread)))>0.5) {
          frvdread[which(frvdread<0)]=0.0
          if (max(frvdread,na.rm=TRUE)>(+2000))   {print(max(frvdread,na.rm=TRUE));stop("Unplausible downward shortwave radiation")}
          if (min(frvdread,na.rm=TRUE)<(-1.0e-6)) {print(min(frvdread,na.rm=TRUE));stop("Unplausible downward shortwave radiation")}
        }
        #
        if (length(which(!is.na(frvmread)))>0.5) {
          frvmread[which(frvmread<0)]=0.0
          frvmread[which(frvmread>350)]=NA
          if (max(frvmread,na.rm=TRUE)>(+350)) {print(max(frvmread,na.rm=TRUE));stop("Unplausible upward shortwave radiation")}
          if (min(frvmread,na.rm=TRUE)<(-10))  {print(min(frvmread,na.rm=TRUE));stop("Unplausible upward shortwave radiation")}
        }
        #
        firdread[which(firdread<100)]=NA
        firdread[which(firdread>500)]=NA
        if (length(which(!is.na(firdread)))>0.5) {
          if (max(firdread,na.rm=TRUE)>(+500)) {print(max(firdread,na.rm=TRUE));stop("Unplausible downward infrared radiation")}
          if (min(firdread,na.rm=TRUE)<(+100)) {print(min(firdread,na.rm=TRUE));stop("Unplausible downward infrared radiation")}
        }
        #
        firmread[which(firmread<100)]=NA
        firmread[which(firmread>600)]=NA
        if (length(which(!is.na(firmread)))>0.5) {
          if (max(firmread,na.rm=TRUE)>(+600)) {print(max(firmread,na.rm=TRUE));stop("Unplausible upward infrared radiation")}
          if (min(firmread,na.rm=TRUE)<(+100)) {print(min(firmread,na.rm=TRUE));stop("Unplausible upward infrared radiation")}
        }
        #
        # Convert date string into year,month,day
        #
        dayyread=as.integer(substr(dateread,01,02))
        montread=as.integer(substr(dateread,04,05))
        yearread=as.integer(substr(dateread,07,10))
        #
        hourread=floor(timeread/10000)
        timeread_rem=timeread-10000*hourread
        #
        minuread=floor(timeread_rem/100)
        #
        if (length(which(is.na(dayyread)))>0.5) stop("NA generated in read of dayy")
        if (length(which(is.na(montread)))>0.5) stop("NA generated in read of mont")
        if (length(which(is.na(yearread)))>0.5) stop("NA generated in read of year")
        if (length(which(is.na(hourread)))>0.5) stop("NA generated in read of hour")
        if (length(which(is.na(minuread)))>0.5) stop("NA generated in read of minu")
        #
        # Calculate the fractional Julian date of the observed data
        #
        juldayintobs=myjulday(montread,dayyread,yearread)
        juldayfraobs=juldayintobs+(3600*hourread + 60*minuread)/86400
        #
        # Attribute the meteorological data to the correct time interval
        # The average over the hour before the output time is taken
        #
        count = 0
        #
        for (ii in 1:length(juldayfrall)) {
          #
          tdiff=juldayfrall[ii]-juldayfraobs
          ind=which((tdiff<(59.5/1440.0))&(tdiff>=0.0))
          #
          if (length(ind)>0.5) {
            #
            temp[ii]=mean(tempread[ind],na.rm=TRUE)
            relh[ii]=mean(relhread[ind],na.rm=TRUE)
            pabs[ii]=mean(pabsread[ind],na.rm=TRUE)
            rrat[ii]=mean(rratread[ind],na.rm=TRUE)
            widd[ii]=mean(widdread[ind],na.rm=TRUE)
            wiff[ii]=mean(wiffread[ind],na.rm=TRUE)
            frvd[ii]=mean(frvdread[ind],na.rm=TRUE)
            frvm[ii]=mean(frvmread[ind],na.rm=TRUE)
            fird[ii]=mean(firdread[ind],na.rm=TRUE)
            firm[ii]=mean(firmread[ind],na.rm=TRUE)
            #
            # Estimate the specific humidity [g/kg]
            #
            ZFOES    = psat(tempread[ind]+XTT)
            ZWORK1   = ZFOES/(100.0*pabsread[ind])
            ZWORK2   = XRD/XRV
            qsat     = ZWORK2*ZWORK1/(1.+(ZWORK2-1.)*ZWORK1)
            qspe_vec = 1.0e+3 * 0.01 * relhread[ind]*qsat
            qspe[ii] = mean(qspe_vec,na.rm=TRUE)
            #
            if (!is.na(qspe[ii])) {
              if ((qspe[ii]<0.0)||(qspe[ii]>100.0)) {
                print(tempread[ind])
                print(pabsread[ind])
                print(relhread[ind])
                print(qspe[ii])
                stop("Unplausible value for specific humidity")
              }
            }
            #
          } else {
            count = count+1
          }
          #
        }
        #
        print(paste("For, ",nyear,nmont," ",100*count/length(juldayfrall),"% of data are missing"))
        #
      } else {
        #
        print(paste("For, ",nyear,nmont," the entire month is missing"))
        #
      }
      #
      # Write the data on the output fields
      #
      timebeg=time_index
      timeend=time_index+length(juldayfrall)-1
      #
      juld_fullts[ind_zone,timebeg:timeend] = juldayfrall
      temp_fullts[ind_zone,timebeg:timeend] = temp
      pabs_fullts[ind_zone,timebeg:timeend] = pabs
      relh_fullts[ind_zone,timebeg:timeend] = relh
      qspe_fullts[ind_zone,timebeg:timeend] = qspe
      rrat_fullts[ind_zone,timebeg:timeend] = rrat
      widd_fullts[ind_zone,timebeg:timeend] = widd
      wiff_fullts[ind_zone,timebeg:timeend] = wiff
      frvd_fullts[ind_zone,timebeg:timeend] = frvd
      frvm_fullts[ind_zone,timebeg:timeend] = frvm
      fird_fullts[ind_zone,timebeg:timeend] = fird
      firm_fullts[ind_zone,timebeg:timeend] = firm
      #
      time_index=time_index+length(juldayfrall)
      #
    }
    #
    ind_zone=ind_zone+1
    #
    if ((time_index-1)!=lenfullts) stop("Wrong time index")
    #
  }
  #
  if ((ind_zone-1)!=7) stop("Wrong number of CAPITOUL CNRM surface stations treated")
  #
  # Save output
  #
  LST=list(lats=lats,lons=lons,heig=heig,name=name,numb=numb,juld_fullts=juld_fullts,
           temp_fullts=temp_fullts,relh_fullts=relh_fullts,pabs_fullts=pabs_fullts,
           rrat_fullts=rrat_fullts,widd_fullts=widd_fullts,wiff_fullts=wiff_fullts,
           frvd_fullts=frvd_fullts,frvm_fullts=frvm_fullts,fird_fullts=fird_fullts,
           firm_fullts=firm_fullts,qspe_fullts=qspe_fullts)
  #
  save(LST,file=filesave)
  #
}
#
# ######################################################################
# Load station data from CAPITOUL campaign
# The data are averaged over the hour before the full hour
# in order to be consistant with the model averaged diagnostics
# ######################################################################
#
seload_capsurf_stations=function(path,filesave) {
  #
  # The time period of interest for the evaluation against CAPITOUL stations
  #
  xmin=as.Date(paste(2004,"/",03,"/",01,sep=""))
  xmax=as.Date(paste(2005,"/",02,"/",28,sep=""))
  montbegvec=seq(xmin,xmax,"month")
  #
  xmin=as.Date(paste(2004,"/",04,"/",01,sep=""))
  xmax=as.Date(paste(2005,"/",03,"/",31,sep=""))
  montnexvec=seq(xmin,xmax,"month")
  #
  # #######################################
  # Load meta data
  # #######################################
  #
  filemeta=paste(path,"lat_lon_capitoul_stations",sep="")
  data=read.table(filemeta,header=FALSE,skip=0)
  DIM=dim(data)
  if (DIM[2]!=6) stop("Error in read of metadata")
  #
  name_surf_cap = as.vector(data$V1)
  numb_surf_cap = as.vector(data$V3)
  lats_surf_cap = as.vector(data$V4)
  lons_surf_cap = as.vector(data$V5)
  heig_surf_cap = as.vector(data$V6)
  #
  if ((max(lats_surf_cap)> 090.01)||(min(lats_surf_cap) < (-90.01 ))) stop("Error in station latitude")
  if ((max(lons_surf_cap)> 180.01)||(min(lons_surf_cap) < (-180.01))) stop("Error in station longitude")
  if ((max(heig_surf_cap)> 280.01)||(min(heig_surf_cap) < (130)   ))  stop("Error in station height")
  # 
  # #######################################
  # Load data
  # #######################################
  #
  # The time of day axis (One value each 12 minutes)
  #
  minutes=seq(0,1439,12)/1440
  #
  # Determine the total length of the time series 
  # and allocate the output field
  #
  nyear=substr(montbegvec[1],1,4)
  nmont=substr(montbegvec[1],6,7)
  juldaybeg=myjulday(as.integer(nmont),1,as.integer(nyear))
  #
  nextyear=substr(montnexvec[length(montnexvec)],1,4)
  nextmont=substr(montnexvec[length(montnexvec)],6,7)
  juldaynex=myjulday(as.integer(nextmont),1,as.integer(nextyear))-1
  #
  lenfullts=length(minutes)*(juldaynex-juldaybeg+1)
  #
  # Check whether it is one year (for CAPITOUL)
  #
  if (lenfullts!=(365*24*5)) stop("Observations should be for one year")
  #
  # Allocate the fields
  #
  name = array(NA,length(numb_surf_cap))
  numb = array(NA,length(numb_surf_cap))
  lats = array(NA,length(numb_surf_cap))
  lons = array(NA,length(numb_surf_cap))
  heig = array(NA,length(numb_surf_cap))
  #
  juld_fullts = array(NA,c(length(numb_surf_cap),lenfullts))
  temp_fullts = array(NA,c(length(numb_surf_cap),lenfullts))
  relh_fullts = array(NA,c(length(numb_surf_cap),lenfullts))
  qspe_fullts = array(NA,c(length(numb_surf_cap),lenfullts))
  #
  # Loop over stations and time periods
  #
  ind_zone = 1
  #
  for (zone in numb_surf_cap) {
    #
    print(" ")
    print(paste("zone :",zone))
    print(paste("Load station ",name_surf_cap[ind_zone]))
    #
    name[ind_zone] = name_surf_cap[ind_zone]
    numb[ind_zone] = numb_surf_cap[ind_zone]
    lats[ind_zone] = lats_surf_cap[ind_zone]
    lons[ind_zone] = lons_surf_cap[ind_zone]
    heig[ind_zone] = heig_surf_cap[ind_zone]
    #
    time_index = 1
    #
    for (tt in 1:length(montbegvec)) {
      #
      nyear=substr(montbegvec[tt],1,4)
      nmont=substr(montbegvec[tt],6,7)
      #
      juldaybeg=myjulday(as.integer(nmont),1,as.integer(nyear))
      #
      nextyear=substr(montnexvec[tt],1,4)
      nextmont=substr(montnexvec[tt],6,7)
      #
      juldaynex=myjulday(as.integer(nextmont),1,as.integer(nextyear))-1
      #
      juldayfrall=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      temp=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      relh=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      qspe=array(NA,length(minutes)*(juldaynex-juldaybeg+1))
      #
      for (day in juldaybeg:juldaynex) {
        #
        ind1=1+length(minutes)*(day-juldaybeg)
        ind2=  length(minutes)*(1+day-juldaybeg)
        #
        juldayfrall[ind1:ind2]=day
        juldayfrall[ind1:ind2]=juldayfrall[ind1:ind2]+minutes
        #
      }
      #
      nzone=sprintf("%02d",zone)
      #
      filein=paste(path,nyear,nmont,"_zone",nzone,".tuc",sep="")
      #
      if (file.exists(filein)) {
        #
        system(paste("sed -i 's/,/./g' ",filein,sep=""))
        #
        data=read.table(filein,header=FALSE,skip=41,sep=";")
        #
        dateread = as.vector(data$V2)
        timeread = as.vector(data$V3)
        tempread = as.numeric(as.vector(data$V4))
        relhread = as.numeric(as.vector(data$V5))
        #
        if (length(which(is.na(tempread)))>0.5) stop("NA generated in read of temp")
        if (length(which(is.na(relhread)))>0.5) stop("NA generated in read of relh")
        #
        # Plausibility checks
        #
        if (length(which(!is.na(tempread)))>0.5) {
          if (max(tempread,na.rm=TRUE)>(+45)) {print(max(tempread,na.rm=TRUE));stop("Unplausible temperature")}
          if (min(tempread,na.rm=TRUE)<(-20)) {print(min(tempread,na.rm=TRUE));stop("Unplausible temperature")}
        }
        #
        if (length(which(!is.na(relhread)))>0.5) {
          if (max(relhread,na.rm=TRUE)>(+105)) {print(max(relhread,na.rm=TRUE));stop("Unplausible relative humidity")}
          if (min(relhread,na.rm=TRUE)<(+010)) {print(min(relhread,na.rm=TRUE));stop("Unplausible relative humidity")}
        }
        #
        dayyread=as.integer(substr(dateread,02,03))
        montread=as.integer(substr(dateread,05,06))
        yearread=as.integer(substr(dateread,08,11))
        hourread=as.integer(substr(timeread,07,08))
        minuread=as.integer(substr(timeread,10,11))
        #
        if (length(which(is.na(dayyread)))>0.5) stop("NA generated in read of dayy")
        if (length(which(is.na(montread)))>0.5) stop("NA generated in read of mont")
        if (length(which(is.na(yearread)))>0.5) stop("NA generated in read of year")
        if (length(which(is.na(hourread)))>0.5) stop("NA generated in read of hour")
        if (length(which(is.na(minuread)))>0.5) stop("NA generated in read of minu")
        #
        # Calculate the fractional Julian date of the observed data
        #
        juldayintobs=myjulday(montread,dayyread,yearread)
        juldayfraobs=juldayintobs+(3600*hourread + 60*minuread)/86400
        #
	# Attribute the correct time period
	#
        count = 0
        #
        for (ii in 1:length(juldayfrall)) {
          #
          tdiff=juldayfrall[ii]-juldayfraobs
          ind=which((tdiff<(11.5/1440.0))&(tdiff>=0.0))
          #
          if (length(ind)>=2) {
	    #
            print(juldayfrall[ii])
            print("More than 1 observation for one 12-minute period found")
            print("This is not possible with 12 minute data")
            stop("Program halted")
	    #
          } else if ((length(ind)>0)&&(length(ind)<=1)) {
            #
            temp[ii]=tempread[ind]
            relh[ii]=relhread[ind]
            #
            # Estimate the specific humidity [g/kg]
            # The pressure is assumed to be 101325 Pa
            #
            ZFOES  = psat(tempread[ind]+XTT)
            ZWORK1 = ZFOES/101325
            ZWORK2 = XRD/XRV
            qsat = ZWORK2*ZWORK1/(1.+(ZWORK2-1.)*ZWORK1)
            qspe_vec=1.0e+3 * 0.01*relhread[ind]*qsat
            qspe[ii]=qspe_vec
            #
          } else {
            count = count+1
          }
          #
        }
        #
        print(paste("For, ",nyear,nmont," ",100*count/length(juldayfrall),"% of data are missing"))
        #
      } else {
        print(paste("For, ",nyear,nmont," the entire month is missing"))
      }
      #
      # Write the data on the output fields
      #
      timebeg=time_index
      timeend=time_index+length(juldayfrall)-1
      #
      juld_fullts[ind_zone,timebeg:timeend] = juldayfrall
      temp_fullts[ind_zone,timebeg:timeend] = temp
      relh_fullts[ind_zone,timebeg:timeend] = relh
      qspe_fullts[ind_zone,timebeg:timeend] = qspe
      #
      time_index=time_index+length(juldayfrall)
      #
    }
    #
    ind_zone=ind_zone+1
    #
    if ((time_index-1)!=lenfullts) stop("Wrong time index")
    #
  }
  #
  if ((ind_zone-1)!=20) stop("Wrong number of CAPITOUL stations treated")
  #
  # Save output
  #
  LST=list(lats=lats,lons=lons,heig=heig,name=name,numb=numb,juld_fullts=juld_fullts,
           temp_fullts=temp_fullts,relh_fullts=relh_fullts,qspe_fullts=qspe_fullts)
  #
  save(LST,file=filesave)
  #
}
#
# ###############################################
# Load station data from Dijon campaign
# ###############################################
#
seload_dijsurf_stations=function(path,filesave) {
  #
  print("   ")
  print("Preparation of Dijon surface stations")
  print("   ")
  #
  # The number of stations to read
  #
  nstat=60
  #
  # Load meta data
  #
  filemeta=paste(path,"coordsICU.csv",sep="")
  data=read.table(filemeta,header=FALSE,skip=1,sep=",")
  DIM=dim(data)
  if (DIM[1]!=nstat) stop("Error in read of metadata")
  if (DIM[2]!=03) stop("Error in read of metadata")
  #
  name_surf_dij = as.vector(data$V1)
  #
  # Determine the station number
  #
  numb_surf_dij = array(NA,length(name_surf_dij))
  for (ii in 1:length(name_surf_dij)) {
    name=name_surf_dij[ii]
    numb_surf_dij[ii]=as.numeric(substring(name,2))
  }
  #
  # Convert latitude and longitude into dd.dddd
  #
  lats_surf_dij_org = as.vector(data$V2)
  lons_surf_dij_org = as.vector(data$V3)
  #
  lats_surf_dij = array(NA,length(name_surf_dij))
  lons_surf_dij = array(NA,length(name_surf_dij))
  #
  for (ii in 1:length(lats_surf_dij_org)) {
    #
    lats_string=lats_surf_dij_org[ii]
    lons_string=lons_surf_dij_org[ii]
    #
    aux_lat=substring(lats_string,1,2)
    aux_lon=substring(lons_string,1,2)
    #
    lats_surf_dij[ii]=as.numeric(aux_lat)
    lons_surf_dij[ii]=as.numeric(aux_lon)
    #
    aux_lat=substring(lats_string,4,5)
    aux_lon=substring(lons_string,4,5)
    #
    lats_surf_dij[ii]=lats_surf_dij[ii]+as.numeric(aux_lat)/60.0
    lons_surf_dij[ii]=lons_surf_dij[ii]+as.numeric(aux_lon)/60.0
    #
    aux_lat=substring(lats_string,7,8)
    aux_lon=substring(lons_string,7,8)
    #
    lats_surf_dij[ii]=lats_surf_dij[ii]+as.numeric(aux_lat)/3600.0
    lons_surf_dij[ii]=lons_surf_dij[ii]+as.numeric(aux_lon)/3600.0
    #
  }
  #
  # Attribute the height from the table send by colleagues
  #
  data=read.csv("/home/schoetterr/Bureau/SURFEX/DIJON_OBS/Classification_LCZ_WUDAPT_60.csv")
  heig_surf_dij = as.numeric(data$ALTITUDE)
  #
  if ((max(lats_surf_dij)>090.01)||(min(lats_surf_dij)<(-0.01))) stop("Error in station latitude")
  if ((max(lons_surf_dij)>180.01)||(min(lons_surf_dij)<(-0.01))) stop("Error in station longitude")
  #
  lats = lats_surf_dij
  lons = lons_surf_dij
  heig = heig_surf_dij
  name = name_surf_dij
  numb = numb_surf_dij
  #
  # Load data
  #
  # The time of day axis (only hourly values are taken)
  #
  hours=seq(1,24,1)/24
  #
  # Determine the total length of the time series 
  # and allocate the output field
  #
  juldaybeg=myjulday(06,01,2014)
  juldayend=myjulday(12,08,2016)
  #
  # The length of the full time series
  # 
  lenfullts=length(hours)*(juldayend-juldaybeg+1)
  #
  # Allocate output fields
  # 
  juldayfrall = array(NA,lenfullts)
  juld_fullts = array(NA,lenfullts)
  temp_fullts = array(NA,c(nstat,lenfullts))
  relh_fullts = array(NA,c(nstat,lenfullts))
  qspe_fullts = array(NA,c(nstat,lenfullts))
  #
  for (day in juldaybeg:juldayend) {
    #
    ind1=1+length(hours)*(  day-juldaybeg)
    ind2=  length(hours)*(1+day-juldaybeg)
    #
    juldayfrall[ind1:ind2]=day
    juldayfrall[ind1:ind2]=juldayfrall[ind1:ind2]+hours
    #
  }
  #
  # Read air temperature and relative humidity
  #
  flist=c("T_ICU14-16.csv","HR_ICU14-16.csv")
  #
  for (nfile in 1:length(flist)) { 
    #
    data=read.table(paste(path,flist[nfile],sep=""),header=FALSE,skip=1,sep=",")
    DIM=dim(data)
    if (DIM[2]!=(nstat+1)) stop("Wrong number of stations read")
    #
    if (nfile==1) {
      datetime_temp = as.vector(data[,1])
    } else if (nfile==2) {
      datetime_relh = as.vector(data[,1])
    } else {
      stop("No rule for this variable")
    }
    #
    valsstat = array(NA,c(length(datetime_temp),nstat))
    for (tt in 1:nstat) {
      auxdat=as.vector(as.numeric(data[,(1+tt)]))
      valsstat[,tt]=auxdat
    }
    #
    if (nfile==1) {
      tempread = valsstat
    } else if (nfile==2) {
      relhread = valsstat
    } else {
      stop("No rule for this variable")
    }
    #
  }
  #
  # Plausibility checks for meteorological variables
  #
  amax=max(tempread,na.rm=TRUE)
  amin=min(tempread,na.rm=TRUE)
  #
  if (amax>40.0)    {print(amax);stop("Unplausible value for air temperature")}
  if (amin<(-20.0)) {print(amin);stop("Unplausible value for air temperature")}
  #
  amax=max(relhread,na.rm=TRUE)
  amin=min(relhread,na.rm=TRUE)
  #
  if (amax>100.1) {print(amax);stop("Unplausible value for relative humidity")}
  if (amin<10.0 ) {print(amin);stop("Unplausible value for relative humidity")}
  #
  # Determine the julian day of the observations
  #
  yearread_temp = as.integer(substr(datetime_temp,01,04))
  montread_temp = as.integer(substr(datetime_temp,06,07))
  dayyread_temp = as.integer(substr(datetime_temp,09,10))
  hourread_temp = as.integer(substr(datetime_temp,12,13))
  minuread_temp = as.integer(substr(datetime_temp,15,16))
  secnread_temp = as.integer(substr(datetime_temp,18,19))
  #
  yearread_relh = as.integer(substr(datetime_relh,01,04))
  montread_relh = as.integer(substr(datetime_relh,06,07))
  dayyread_relh = as.integer(substr(datetime_relh,09,10))
  hourread_relh = as.integer(substr(datetime_relh,12,13))
  minuread_relh = as.integer(substr(datetime_relh,15,16))
  secnread_relh = as.integer(substr(datetime_relh,18,19))
  #
  # Calculate the fractional Julian date of the observed data
  #
  juldayintobs_temp = myjulday(montread_temp,dayyread_temp,yearread_temp)
  juldayintobs_relh = myjulday(montread_relh,dayyread_relh,yearread_relh)
  #
  juldayfraobs_temp = juldayintobs_temp + (3600*hourread_temp + 60 * minuread_temp + secnread_temp)/86400
  juldayfraobs_relh = juldayintobs_relh + (3600*hourread_relh + 60 * minuread_relh + secnread_relh)/86400
  #
  adiff=abs(juldayfraobs_relh-juldayfraobs_temp)
  if (max(adiff)>1.0e-6) stop("Inconsistant time steps")
  #
  juldayfraobs = juldayfraobs_temp
  #
  # Attribute the meteorological data to the correct time interval
  # The average over the hour preceeding the target hour is taken.
  #
  count = 0
  #
  for (ii in 1:length(juldayfrall)) {
    #
    tdiff=juldayfrall[ii]-juldayfraobs
    ind=which((tdiff<(59.5/1440.0))&(tdiff>=0.0))
    #
    if (length(ind)>1.5) {
      print(ind)
      print("More than one value per hour found in observations")
      print("However, Dijon observations should be hourly")
      stop("Program halted")
    } else if (length(ind)>0.5) {
      #
      temp_fullts[,ii]=tempread[ind,]
      relh_fullts[,ii]=relhread[ind,]
      #
      # Estimate the specific humidity [g/kg]
      # The pressure is assumed to be 101325 Pa
      #
      ZFOES  = psat(tempread[ind,]+XTT)
      ZWORK1 = ZFOES/101325
      ZWORK2 = XRD/XRV
      qsat = ZWORK2*ZWORK1/(1.+(ZWORK2-1.)*ZWORK1)
      qspe_fullts[,ii]=1.0e+3 * 0.01 * relhread[ind,] * qsat
      #
    } else {
      count = count+1
    }
    #
  }
  #
  print(paste("A total of ",100*count/length(juldayfrall),"% of data are missing"))
  #
  # Write the data on the output fields
  #
  juld_fullts = juldayfrall
  #
  # Save output
  #
  LST=list(lats=lats,lons=lons,heig=heig,name=name,numb=numb,juld_fullts=juld_fullts,
           temp_fullts=temp_fullts,relh_fullts=relh_fullts,qspe_fullts=qspe_fullts)
  #
  save(LST,file=filesave)
  #
}
#
# ###############################################
# Load station data from Hong Kong observatory
# ###############################################
#
seload_hko_stations=function(path,filesave) {
  #
  print("                                             ")
  print("Preparation of Hong Kong observatory stations")
  print("                                             ")
  #
  # A: Define meta data
  #
  # The names of the files to load
  #
  name_hko_full = list.files(path = path, pattern = NULL, all.files = FALSE,
                             full.names = FALSE, recursive = FALSE,
                             ignore.case = FALSE, include.dirs = FALSE)
  #
  # The station names
  #
  name_hko = c()
  for (tt in 1:length(name_hko_full)) name_hko = c(name_hko,gsub("_","",substr(name_hko_full[tt],1,3)))
  print(name_hko)
  #
  # Hardcoded meta data for each station
  #
  numb_hko = array(NA,length(name_hko))  # Station number 
  lats_hko = array(NA,length(name_hko))  # Station latitude  [dd.dddd]
  lons_hko = array(NA,length(name_hko))  # Station longitude [dd.dddd]
  heig_hko = array(NA,length(name_hko))  # Station ground elevation [m]
  hiff_hko = array(NA,length(name_hko))  # Height of wind anemometer above ground [m]
  hite_hko = array(NA,length(name_hko))  # Height of temperature anemometer above ground [m]
  town_hko = array(NA,length(name_hko))  # Fraction of TOWN tile for evaluation [1]
  natu_hko = array(NA,length(name_hko))  # Fraction of NATURE tile for evaluation [1]
  watr_hko = array(NA,length(name_hko))  # Fraction of WATER tile for evaluation [1]
  seaf_hko = array(NA,length(name_hko))  # Fraction of SEA tile for evaluation [1]
  roof_hko = array(NA,length(name_hko))  # Flag indicating a rooftop station [0 or 1]
  #
  # Bluff head
  #
  ind = which(name_hko=="BHD")
  if (length(ind)!=1) stop("Station BHD not found")
  #
  numb_hko[ind] = 1
  lats_hko[ind] = 22.1975
  lons_hko[ind] = 114.2119
  heig_hko[ind] = 94
  hiff_hko[ind] = 9
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Beas river
  #
  ind = which(name_hko=="BR1")
  if (length(ind)!=1) stop("Station BR1 not found")
  #
  numb_hko[ind] = 2
  lats_hko[ind] = 22.4933
  lons_hko[ind] = 114.105
  heig_hko[ind] = 11
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Cheng Chau
  #
  ind = which(name_hko=="CCH")
  if (length(ind)!=1) stop("Station CCH not found")
  #
  numb_hko[ind] = 3
  lats_hko[ind] = 22.201111
  lons_hko[ind] = 114.0267
  heig_hko[ind] = 72
  hiff_hko[ind] = 27
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Central Pier
  #
  ind = which(name_hko=="CP1")
  if (length(ind)!=1) stop("Station CP1 not found")
  #
  numb_hko[ind] = 4
  lats_hko[ind] = 22.28889
  lons_hko[ind] = 114.1558
  heig_hko[ind] = 3
  hiff_hko[ind] = 27
  hite_hko[ind] = 0
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.0
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.5
  roof_hko[ind] = 0
  #
  # Ching Pak House
  #
  ind = which(name_hko=="CPH")
  if (length(ind)!=1) stop("Station CPH not found")
  #
  numb_hko[ind] = 5
  lats_hko[ind] = 22.34806
  lons_hko[ind] = 114.1092
  heig_hko[ind] = 61
  hiff_hko[ind] = 75
  hite_hko[ind] = 62
  town_hko[ind] = 1
  natu_hko[ind] = 0
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 1
  #
  # Ping Chau
  #
  ind = which(name_hko=="EPC")
  if (length(ind)!=1) stop("Station EPC not found")
  #
  numb_hko[ind] = 6
  lats_hko[ind] = 22.54667
  lons_hko[ind] = 114.4283
  heig_hko[ind] = 29
  hiff_hko[ind] = 10
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Green island
  #
  ind = which(name_hko=="GI")
  if (length(ind)!=1) stop("Station GI not found")
  #
  numb_hko[ind] = 7
  lats_hko[ind] = 22.285
  lons_hko[ind] = 114.1128
  heig_hko[ind] = 88
  hiff_hko[ind] = 19
  hite_hko[ind] = 0
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Hong Kong International Airport
  #
  ind = which(name_hko=="HKA")
  if (length(ind)!=1) stop("Station HKA not found")
  #
  numb_hko[ind] = 8
  lats_hko[ind] = 22.30944
  lons_hko[ind] = 113.9219
  heig_hko[ind] = 6
  hiff_hko[ind] = 8
  hite_hko[ind] = 1
  town_hko[ind] = 1.0
  natu_hko[ind] = 0.0
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.0
  roof_hko[ind] = 0
  #
  # Hong Kong Observatory
  #
  ind = which(name_hko=="HKO")
  if (length(ind)!=1) stop("Station HKO not found")
  #
  numb_hko[ind] = 9
  lats_hko[ind] = 22.30194
  lons_hko[ind] = 114.1742
  heig_hko[ind] = 32
  hiff_hko[ind] = 42
  hite_hko[ind] = 1
  town_hko[ind] = 0.0
  natu_hko[ind] = 1.0
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.0
  roof_hko[ind] = 0
  #
  # Hong Kong Park
  #
  ind = which(name_hko=="HKP")
  if (length(ind)!=1) stop("Station HKP not found")
  #
  numb_hko[ind] = 10
  lats_hko[ind] = 22.27833
  lons_hko[ind] = 114.1622
  heig_hko[ind] = 26
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Wong Chuk Hang
  #
  ind = which(name_hko=="HKS")
  if (length(ind)!=1) stop("Station HKS not found")
  #
  numb_hko[ind] = 11
  lats_hko[ind] = 22.24778
  lons_hko[ind] = 114.1736
  heig_hko[ind] = 5
  hiff_hko[ind] = 25
  hite_hko[ind] = 1
  town_hko[ind] = 0.4
  natu_hko[ind] = 0.6
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Happy Valley
  #
  ind = which(name_hko=="HPV")
  if (length(ind)!=1) stop("Station HPV not found")
  #
  numb_hko[ind] = 12
  lats_hko[ind] = 22.27056
  lons_hko[ind] = 114.1836
  heig_hko[ind] = 5
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.1
  natu_hko[ind] = 0.9
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tseung Kwan O
  #
  ind = which(name_hko=="JKB")
  if (length(ind)!=1) stop("Station JKB not found")
  #
  numb_hko[ind] = 13
  lats_hko[ind] = 22.31583
  lons_hko[ind] = 114.2556
  heig_hko[ind] = 38
  hiff_hko[ind] = 14
  hite_hko[ind] = 1
  town_hko[ind] = 0.15
  natu_hko[ind] = 0.85
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Kat O
  #
  ind = which(name_hko=="KAT")
  if (length(ind)!=1) stop("Station KAT not found")
  #
  numb_hko[ind] = 14
  lats_hko[ind] = 22.5363
  lons_hko[ind] = 114.3019
  heig_hko[ind] = 10
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.5
  roof_hko[ind] = 0
  #
  # Kadoorie farm and botanic garden
  #
  ind = which(name_hko=="KFB")
  if (length(ind)!=1) stop("Station KFB not found")
  #
  numb_hko[ind] = 15
  lats_hko[ind] = 22.43278
  lons_hko[ind] = 114.1208
  heig_hko[ind] = 307
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Kowloon city
  #
  ind = which(name_hko=="KLT")
  if (length(ind)!=1) stop("Station KLT not found")
  #
  numb_hko[ind] = 16
  lats_hko[ind] = 22.335
  lons_hko[ind] = 114.1847
  heig_hko[ind] = 92
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Kings Park
  #
  ind = which(name_hko=="KP")
  if (length(ind)!=1) stop("Station KP not found")
  #
  numb_hko[ind] = 17
  lats_hko[ind] = 22.31194
  lons_hko[ind] = 114.1728
  heig_hko[ind] = 65
  hiff_hko[ind] = 25
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Kau Sau Chau
  #
  ind = which(name_hko=="KSC")
  if (length(ind)!=1) stop("Station KSC not found")
  #
  numb_hko[ind] = 18
  lats_hko[ind] = 22.37028
  lons_hko[ind] = 114.3125
  heig_hko[ind] = 39
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Kwun Tong
  #
  ind = which(name_hko=="KTG")
  if (length(ind)!=1) stop("Station KTG not found")
  #
  numb_hko[ind] = 19
  lats_hko[ind] = 22.31861
  lons_hko[ind] = 114.2247
  heig_hko[ind] = 90
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Lamma Island
  #
  ind = which(name_hko=="LAM")
  if (length(ind)!=1) stop("Station LAM not found")
  #
  numb_hko[ind] = 20
  lats_hko[ind] = 22.22611
  lons_hko[ind] = 114.1086
  heig_hko[ind] = 7
  hiff_hko[ind] = 10
  hite_hko[ind] = 0
  town_hko[ind] = 0
  natu_hko[ind] = 0
  watr_hko[ind] = 0
  seaf_hko[ind] = 1
  roof_hko[ind] = 0
  #
  # Lau Fau Shan
  #
  ind = which(name_hko=="LFS")
  if (length(ind)!=1) stop("Station LFS not found")
  #
  numb_hko[ind] = 21
  lats_hko[ind] = 22.46889
  lons_hko[ind] = 113.9836
  heig_hko[ind] = 31
  hiff_hko[ind] = 19
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Ngong Ping
  #
  ind = which(name_hko=="NGP")
  if (length(ind)!=1) stop("Station NGP not found")
  #
  numb_hko[ind] = 22
  lats_hko[ind] = 22.25861
  lons_hko[ind] = 113.9128
  heig_hko[ind] = 593
  hiff_hko[ind] = 14
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Nei Lak Shan
  #
  ind = which(name_hko=="NLS")
  if (length(ind)!=1) stop("Station NLS not found")
  #
  numb_hko[ind] = 23
  lats_hko[ind] = 22.26333
  lons_hko[ind] = 113.91111
  heig_hko[ind] = 747
  hiff_hko[ind] = 10
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Peng Chau
  #
  ind = which(name_hko=="PEN")
  if (length(ind)!=1) stop("Station PEN not found")
  #
  numb_hko[ind] = 24
  lats_hko[ind] = 22.29111
  lons_hko[ind] = 114.0433
  heig_hko[ind] = 34
  hiff_hko[ind] = 13
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tai Mei Tuk
  #
  ind = which(name_hko=="PLC")
  if (length(ind)!=1) stop("Station PLC not found")
  #
  numb_hko[ind] = 25
  lats_hko[ind] = 22.47528
  lons_hko[ind] = 114.2375
  heig_hko[ind] = 51
  hiff_hko[ind] = 20
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Kai Tak
  #
  ind = which(name_hko=="SE")
  if (length(ind)!=1) stop("Station SE not found")
  #
  numb_hko[ind] = 26
  lats_hko[ind] = 22.30972
  lons_hko[ind] = 114.2133
  heig_hko[ind] = 1
  hiff_hko[ind] = 15
  hite_hko[ind] = 0
  town_hko[ind] = 0.0
  natu_hko[ind] = 0.4
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.6
  roof_hko[ind] = 0
  #
  # Kai Tak Runway Park
  #
  ind = which(name_hko=="SE1")
  if (length(ind)!=1) stop("Station SE1 not found")
  #
  numb_hko[ind] = 27
  lats_hko[ind] = 22.30472
  lons_hko[ind] = 114.2172
  heig_hko[ind] = 4
  hiff_hko[ind] = 15.7
  hite_hko[ind] = 1
  town_hko[ind] = 0.0
  natu_hko[ind] = 0.4
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.6
  roof_hko[ind] = 0
  #
  # Shek Kong
  #
  ind = which(name_hko=="SEK")
  if (length(ind)!=1) stop("Station SEK not found")
  #
  numb_hko[ind] = 28
  lats_hko[ind] = 22.43611
  lons_hko[ind] = 114.0847
  heig_hko[ind] = 16
  hiff_hko[ind] = 10
  hite_hko[ind] = 1
  town_hko[ind] = 1
  natu_hko[ind] = 0
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Sha Tin
  #
  ind = which(name_hko=="SHA")
  if (length(ind)!=1) stop("Station SHA not found")
  #
  numb_hko[ind] = 29
  lats_hko[ind] = 22.4025
  lons_hko[ind] = 114.21
  heig_hko[ind] = 6
  hiff_hko[ind] = 10
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Sai Kung
  #
  ind = which(name_hko=="SKG")
  if (length(ind)!=1) stop("Station SKG not found")
  #
  numb_hko[ind] = 30
  lats_hko[ind] = 22.37556
  lons_hko[ind] = 114.2744
  heig_hko[ind] = 4
  hiff_hko[ind] = 28
  hite_hko[ind] = 1
  town_hko[ind] = 0.6
  natu_hko[ind] = 0.0
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.4
  roof_hko[ind] = 0
  #
  # Shau Kei Wan
  #
  ind = which(name_hko=="SKW")
  if (length(ind)!=1) stop("Station SKW not found")
  #
  numb_hko[ind] = 31
  lats_hko[ind] = 22.28167
  lons_hko[ind] = 114.2361
  heig_hko[ind] = 53
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Sha Lo Wan
  #
  ind = which(name_hko=="SLW")
  if (length(ind)!=1) stop("Station not found")
  #
  numb_hko[ind] = 32
  lats_hko[ind] = 22.29111
  lons_hko[ind] = 113.9069
  heig_hko[ind] = 61
  hiff_hko[ind] = 10
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Sheung Shui
  #
  ind = which(name_hko=="SSH")
  if (length(ind)!=1) stop("Station SSH not found")
  #
  numb_hko[ind] = 33
  lats_hko[ind] = 22.50194
  lons_hko[ind] = 114.1111
  heig_hko[ind] = 10
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Sham Shui Po
  #
  ind = which(name_hko=="SSP")
  if (length(ind)!=1) stop("Station SSP not found")
  #
  numb_hko[ind] = 34
  lats_hko[ind] = 22.33583
  lons_hko[ind] = 114.1369
  heig_hko[ind] = 11
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Stanley
  #
  ind = which(name_hko=="STY")
  if (length(ind)!=1) stop("Station STY not found")
  #
  numb_hko[ind] = 35
  lats_hko[ind] = 22.21417
  lons_hko[ind] = 114.2186
  heig_hko[ind] = 31
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tap Mun
  #
  ind = which(name_hko=="TAP")
  if (length(ind)!=1) stop("Station TAP not found")
  #
  numb_hko[ind] = 36
  lats_hko[ind] = 22.47139
  lons_hko[ind] = 114.2138
  heig_hko[ind] = 15
  hiff_hko[ind] = 20
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tais Caim
  #
  ind = which(name_hko=="TC")
  if (length(ind)!=1) stop("Station TC not found")
  #
  numb_hko[ind] = 37
  lats_hko[ind] = 22.35778
  lons_hko[ind] = 114.2178
  heig_hko[ind] = 572
  hiff_hko[ind] = 15
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Ta Kwu Ling
  #
  ind = which(name_hko=="TKL")
  if (length(ind)!=1) stop("Station TKL not found")
  #
  numb_hko[ind] = 38
  lats_hko[ind] = 22.52861
  lons_hko[ind] = 114.1567
  heig_hko[ind] = 15
  hiff_hko[ind] = 13
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tai Mo Shan
  #
  ind = which(name_hko=="TMS")
  if (length(ind)!=1) stop("Station TMS not found")
  #
  numb_hko[ind] = 39
  lats_hko[ind] = 22.41056
  lons_hko[ind] = 114.1217
  heig_hko[ind] = 955
  hiff_hko[ind] = 11
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tai Po
  #
  ind = which(name_hko=="TP")
  if (length(ind)!=1) stop("Station TP not found")
  #
  numb_hko[ind] = 40
  lats_hko[ind] = 22.44611
  lons_hko[ind] = 114.1789
  heig_hko[ind] = 15
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.0
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.5
  roof_hko[ind] = 0
  #
  # Tuen Mun Children and Juvenile home
  #
  ind = which(name_hko=="TU1")
  if (length(ind)!=1) stop("Station TU1 not found")
  #
  numb_hko[ind] = 41
  lats_hko[ind] = 22.38583
  lons_hko[ind] = 113.9642
  heig_hko[ind] = 28
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tsuen Wan Shing Mun
  #
  ind = which(name_hko=="TW")
  if (length(ind)!=1) stop("Station TW not found")
  #
  numb_hko[ind] = 42
  lats_hko[ind] = 22.37556
  lons_hko[ind] = 114.1267
  heig_hko[ind] = 35
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Tsuen Wan
  #
  ind = which(name_hko=="TWN")
  if (length(ind)!=1) stop("Station TWN not found")
  #
  numb_hko[ind] = 43
  lats_hko[ind] = 22.38361
  lons_hko[ind] = 114.1078
  heig_hko[ind] = 142
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # New Tsing Yi Station
  #
  ind = which(name_hko=="TY1")
  if (length(ind)!=1) stop("Station TY1 not found")
  #
  numb_hko[ind] = 44
  lats_hko[ind] = 22.34417
  lons_hko[ind] = 114.11
  heig_hko[ind] = 8
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Pak Tam Chung 
  #
  ind = which(name_hko=="TYW")
  if (length(ind)!=1) stop("Station TYW not found")
  #
  numb_hko[ind] = 45
  lats_hko[ind] = 22.40278
  lons_hko[ind] = 114.3231
  heig_hko[ind] = 5
  hiff_hko[ind] = 18
  hite_hko[ind] = 1
  town_hko[ind] = 0
  natu_hko[ind] = 1
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # The Peak
  #
  ind = which(name_hko=="VP1")
  if (length(ind)!=1) stop("Station VP1 not found")
  #
  numb_hko[ind] = 46
  lats_hko[ind] = 22.26417
  lons_hko[ind] = 114.155
  heig_hko[ind] = 406
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Waglan Island
  #
  ind = which(name_hko=="WGL")
  if (length(ind)!=1) stop("Station WGL not found")
  #
  numb_hko[ind] = 47
  lats_hko[ind] = 22.18222
  lons_hko[ind] = 114.3033
  heig_hko[ind] = 56
  hiff_hko[ind] = 27
  hite_hko[ind] = 1
  town_hko[ind] = 0.0
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0.0
  seaf_hko[ind] = 0.5
  roof_hko[ind] = 0
  #
  # Wetland Park
  #
  ind = which(name_hko=="WLP")
  if (length(ind)!=1) stop("Station WLP not found")
  #
  numb_hko[ind] = 48
  lats_hko[ind] = 22.46666
  lons_hko[ind] = 114.0089
  heig_hko[ind] = 4
  hiff_hko[ind] = 11
  hite_hko[ind] = 1
  town_hko[ind] = 0.0
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0.5
  seaf_hko[ind] = 0.0
  roof_hko[ind] = 0
  #
  # Wong Tai Sin
  #
  ind = which(name_hko=="WTS")
  if (length(ind)!=1) stop("Station WTS not found")
  #
  numb_hko[ind] = 49
  lats_hko[ind] = 22.33944
  lons_hko[ind] = 114.2053
  heig_hko[ind] = 21
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Yuen Long Park
  #
  ind = which(name_hko=="YLP")
  if (length(ind)!=1) stop("Station YLP not found")
  #
  numb_hko[ind] = 50
  lats_hko[ind] = 22.44083
  lons_hko[ind] = 114.0183
  heig_hko[ind] = 8
  hiff_hko[ind] = 0
  hite_hko[ind] = 1
  town_hko[ind] = 0.5
  natu_hko[ind] = 0.5
  watr_hko[ind] = 0
  seaf_hko[ind] = 0
  roof_hko[ind] = 0
  #
  # Check metadata (individual)
  # 
  if ((min(numb_hko)<1)|(max(numb_hko)>50)) stop("Wrong station number")
  if ((min(lats_hko)<21)|(max(lats_hko)>24)) stop("Wrong latitude")
  if ((min(lons_hko)<112)|(max(lons_hko)>116)) stop("Wrong longitude")
  if ((min(heig_hko)<0)|(max(heig_hko)>1000)) stop("Wrong station height")
  if ((min(hiff_hko)<0)|(max(hiff_hko)>100)) stop("Wrong wind anemometer height")
  if ((min(hite_hko)<0)|(max(hite_hko)>100)) stop("Wrong temperature anemometer height")
  if ((min(town_hko)<0)|(max(town_hko)>1)) stop("Wrong town fraction")
  if ((min(natu_hko)<0)|(max(natu_hko)>1)) stop("Wrong nature fraction")
  if ((min(watr_hko)<0)|(max(watr_hko)>1)) stop("Wrong water fraction")
  if ((min(seaf_hko)<0)|(max(seaf_hko)>1)) stop("Wrong sea fraction")
  if ((min(roof_hko)<0)|(max(roof_hko)>1)) stop("Wrong roof station identifier")
  #
  # Check metadata (consistency)
  #
  asum=town_hko+natu_hko+watr_hko+seaf_hko
  if (max(abs(asum-1.0))>1.0e-4) stop("Wrong sum of surfex tile fractions")
  #
  # Load meteorological parameters
  #
  # The number of stations
  #
  nstat = length(name_hko_full)
  #
  # The time of day axis (only hourly values are taken)
  #
  hours=seq(1,24,1)/24
  #
  # Determine the total length of the time series 
  # and allocate the output field
  #
  juldaybeg=myjulday(01,01,1970)
  juldayend=myjulday(01,01,2019)
  #
  # The length of the full time series
  #
  lenfullts=length(hours)*(juldayend-juldaybeg+1)
  #
  juldayfrall = array(NA,lenfullts)
  juld_fullts = array(NA,lenfullts)
  #
  for (day in juldaybeg:juldayend) {
    #
    ind1=1+length(hours)*(  day-juldaybeg)
    ind2=  length(hours)*(1+day-juldaybeg)
    #
    juldayfrall[ind1:ind2]=day
    juldayfrall[ind1:ind2]=juldayfrall[ind1:ind2]+hours
    #
  }
  #
  # Allocate output fields
  #
  widd_fullts = array(NA,c(nstat,lenfullts))
  wiff_fullts = array(NA,c(nstat,lenfullts))
  temp_fullts = array(NA,c(nstat,lenfullts))
  relh_fullts = array(NA,c(nstat,lenfullts))
  qspe_fullts = array(NA,c(nstat,lenfullts)) 
  mslp_fullts = array(NA,c(nstat,lenfullts))
  rrat_fullts = array(NA,c(nstat,lenfullts))
  radg_fullts = array(NA,c(nstat,lenfullts))  
  #
  # Loop over stations
  #
  for (stat in 1:length(name_hko_full)) {
    #
    print(paste("Treating",name_hko_full[stat]))
    #
    data=read.table(paste(path,name_hko_full[stat],sep=""),header=TRUE,skip=0,sep=",")
    DIM=dim(data)
    #
    yearread = as.numeric(as.vector(data$yyyy))
    montread = as.numeric(as.vector(data$mm))
    dayyread = as.numeric(as.vector(data$dd))
    hourread = as.numeric(as.vector(data$hour))
    widdread = as.numeric(as.vector(data$dir))
    wiffread = as.numeric(as.vector(data$spd))
    tempread = as.numeric(as.vector(data$temp))
    relhread = as.numeric(as.vector(data$rh))
    mslpread = as.numeric(as.vector(data$mslp))
    rratread = as.numeric(as.vector(data$rf))
    radgread = as.numeric(as.vector(data$radG))
    #
    if (length(radgread)==0) {
      radgread=array(NA,length(rratread))
    }
    #
    # Plausibility checks for dates
    #
    if (length(which(!is.na(yearread)))==0) stop("No information on year")
    if (max(yearread)>2019) stop("Unplausible value for year")
    if (min(yearread)<1970) stop("Unplausible value for year")
    #
    if (length(which(!is.na(montread)))==0) stop("No information on month")
    if (max(montread)>12) stop("Unplausible value for month")
    if (min(montread)<01) stop("Unplausible value for month")
    #
    if (length(which(!is.na(dayyread)))==0) stop("No information on day")
    if (max(dayyread)>31) stop("Unplausible value for day")
    if (min(dayyread)<01) stop("Unplausible value for day")
    #
    if (length(which(!is.na(hourread)))==0) stop("No information on hour")
    if (max(hourread)>24) stop("Unplausible value for hour")
    if (min(hourread)<01) stop("Unplausible value for hour")
    #
    # Plausibility checks for meteorological variables
    #
    if (length(which(!is.na(widdread)))>0.5) {        
      if (max(widdread,na.rm=TRUE)>360.0) stop("Unplausible value for wind direction")
      if (min(widdread,na.rm=TRUE)<(0.0)) stop("Unplausible value for wind direction")
    }
    #
    if (length(which(!is.na(wiffread)))>0.5) {
      if (max(wiffread,na.rm=TRUE)> 50.0) stop("Unplausible value for wind speed")
      if (min(wiffread,na.rm=TRUE)<(0.0)) stop("Unplausible value for wind speed")
    }
    #
    if (length(which(!is.na(radgread)))>0.5) {
      #
      # Convert total downwelling solar radiation from MJ/m2 to W/m2
      #
      radgread = 1.0e+6*radgread/3600
      #
      if (max(radgread,na.rm=TRUE)> 1500.0) stop("Unplausible value for total downwelling solar radiation")
      if (min(radgread,na.rm=TRUE)<(0.0)) stop("Unplausible value for total downwelling solar radiation")
    }
    #
    if (length(which(!is.na(tempread)))>0.5) {
      if (max(tempread,na.rm=TRUE)>  40.0)  stop("Unplausible value for air temperature")
      if (min(tempread,na.rm=TRUE)<(-20.0)) stop("Unplausible value for air temperature")
    }
    #
    if (length(which(!is.na(relhread)))>0.5) {
      if (max(relhread,na.rm=TRUE)>100 ) {print(max(relhread,na.rm=TRUE));stop("Unplausible value for relative humidity")}
      if (min(relhread,na.rm=TRUE)<0.0) {print(min(relhread,na.rm=TRUE));stop("Unplausible value for relative humidity")}
    }
    #
    if (length(which(!is.na(rratread)))>0.5) {
      if (max(rratread,na.rm=TRUE)>200.0) {print(max(rratread,na.rm=TRUE));stop("Unplausible value for hourly precipitation")}
      if (min(rratread,na.rm=TRUE)<0.0 ) {print(min(rratread,na.rm=TRUE));stop("Unplausible value for hourly precipitation")}
    }
    #
    # Pressure must be converted to Pa
    #
    mslpread=100.0*mslpread
    #
    if (length(which(!is.na(mslpread)))>0.5) {
      if (max(mslpread,na.rm=TRUE)>110000) stop("Unplausible value for pressure")
      if (min(mslpread,na.rm=TRUE)<090000) stop("Unplausible value for pressure")
    }
    #
    # Test coherence between meta data and observations
    # All data must be NA, in the case the sensor height is 0
    #
    if (hiff_hko[stat]==0) {
      #
      if (length(which(!is.na(widdread)))>0.5) {
        print("No wind anemometer height given, but non-NA data have been found for WIDD")
        print("The values are set to NA")
	widdread=array(NA,length(widdread))
      }
      #
      if (length(which(!is.na(wiffread)))>0.5) {
        print("No wind anemometer height given, but non-NA data have been found for WIFF")
        print("The values are set to NA")
	wiffread=array(NA,length(wiffread))
      }
      #
    }
    #
    if (hite_hko[stat]==0) {
      #
      if (length(which(!is.na(tempread)))>0.5) {
        print("No temperature measurement height given, but non-NA data have been found for TEMP")
	print("The values are set to NA")
	tempread=array(NA,length(tempread))
      }
      #
      if (length(which(!is.na(relhread)))>0.5) {
        print("No temperature measurement height given, but non-NA data have been found for RELH")
	print("The values are set to NA")
	relhread=array(NA,length(relhread))
      }
      #
    }
    #
    # Calculate the fractional Julian date of the observed data
    #
    juldayintobs = myjulday(montread,dayyread,yearread)
    juldayfraobs = juldayintobs + 3600*hourread/86400
    #
    # Attribute the meteorological data to the correct time interval
    # The average over the hour preceeding the target hour is taken.
    #
    # Check whether observations are consecutive in one-hour intervals
    #
    adiff=juldayfraobs[2:length(juldayfraobs)]-juldayfraobs[1:(length(juldayfraobs)-1)]
    #
    if ( max(abs(adiff-1/24)) < 1.0e-4 ) {
      #
      # Data are exaclty hourly, only the first point of the observations is searched
      #
      tdiff=abs(juldayfrall-juldayfraobs[1])
      ind=which(tdiff < 0.25/24 )
      #
      if (length(ind)>1.5) {
        #
        print(ind)
        print("More than one value per hour found in observations")
        print("However, Hong Kong observations should be hourly")
        stop("Program halted")
	#
      } else if (length(ind)>0.5) {
        #
        temp_fullts[stat,ind:(ind+length(tempread)-1)]=tempread
        relh_fullts[stat,ind:(ind+length(relhread)-1)]=relhread
        widd_fullts[stat,ind:(ind+length(widdread)-1)]=widdread
        wiff_fullts[stat,ind:(ind+length(wiffread)-1)]=wiffread
        radg_fullts[stat,ind:(ind+length(radgread)-1)]=radgread	
        mslp_fullts[stat,ind:(ind+length(mslpread)-1)]=mslpread
        rrat_fullts[stat,ind:(ind+length(rratread)-1)]=rratread  
        #
        # Estimate the specific humidity [g/kg]
	#
        # The observed pressure is taken, otherwise a very simple
	# correction for altitude is made assuming mslp = 101325.
        #
	pabsread=mslpread
	pabsread[which(is.na(pabsread))] = 101325.0
	#
	tempall=array(NA,c(length(tempread)))
	tempall[which(is.na(tempall))] = XTT + 25.0
	#
        pabsread = pabsread - heig_hko[stat] * XG * (pabsread/(XRD*tempall))
        #
	if (length(which(is.na(pabsread)))>0) stop("NA in estimated pressure at station")
	#
        ZFOES  = psat(tempread+XTT)
        ZWORK1 = ZFOES/pabsread
        ZWORK2 = XRD/XRV
        qsat = ZWORK2*ZWORK1/(1.+(ZWORK2-1.)*ZWORK1)
        qspe_fullts[stat,ind:(ind+length(tempread)-1)]=1.0e+3 * 0.01 * relhread * qsat
        #
      } else {
        stop("Observation time not found")
      }
      #
    } else {
      #
      # Non hourly data: the observation point must be searched
      #
      for (ii in 1:length(juldayfraobs)) {
        #
        tdiff=abs(juldayfrall-juldayfraobs[ii])
        ind=which(tdiff < 0.25/24 )
        #
        if (length(ind)>1.5) {
          print(ind)
          print("More than one value per hour found in observations")
          print("However, Hong Kong observations should be hourly")
          stop("Program halted")
        } else if (length(ind)>0.5) {
          #
          temp_fullts[stat,ind]=tempread[ii]
          relh_fullts[stat,ind]=relhread[ii]
          widd_fullts[stat,ind]=widdread[ii]
          wiff_fullts[stat,ind]=wiffread[ii]
          radg_fullts[stat,ind]=radgread[ii]	  
          mslp_fullts[stat,ind]=mslpread[ii]
          rrat_fullts[stat,ind]=rratread[ii]
          #
          # Estimate the specific humidity [g/kg]
          # The pressure is assumed to be 101325 Pa
          #
          ZFOES  = psat(tempread[ii]+XTT)
          ZWORK1 = ZFOES/101325
          ZWORK2 = XRD/XRV
          qsat = ZWORK2*ZWORK1/(1.+(ZWORK2-1.)*ZWORK1)
          qspe_fullts[stat,ind]=1.0e+3 * 0.01 * relhread[ii] * qsat
          #
        }
        #
      }
    }
    #
    # End loop over stations
    #
  }
  #
  # Write the data on the output fields
  #
  juld_fullts = juldayfrall
  #
  LST=list(name_hko=name_hko,numb_hko=numb_hko,lats_hko=lats_hko,lons_hko=lons_hko,heig_hko=heig_hko,
           hiff_hko=hiff_hko,hite_hko=hite_hko,town_hko=town_hko,natu_hko=natu_hko,
	   watr_hko=watr_hko,seaf_hko=seaf_hko,roof_hko=roof_hko,juld_fullts=juld_fullts,
	   widd_fullts=widd_fullts,wiff_fullts=wiff_fullts,temp_fullts=temp_fullts,
	   relh_fullts=relh_fullts,qspe_fullts=qspe_fullts,mslp_fullts=mslp_fullts,
	   rrat_fullts=rrat_fullts,radg_fullts=radg_fullts)
  #
  save(LST,file=filesave)
  #
}
#
# ##############################################
# Prepare the CAPITOUL mast observations
# ##############################################
#
seprepmast=function(fobs) {
#
# Read fluxes prepared by Marine Goret
#
load(file="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/CO2_Rfiles/mast_fluxes_ok.Rdata")
#
date_co2=data_flx$date
flux_co2=as.numeric(as.vector(data_flx$flx_co2))
conc_co2=as.numeric(as.vector(data_flx$conc_co2))
stat_co2=as.numeric(as.vector(data_flx$stat_wco2))
#
# Conversion mg -> umol
#
cov=1.0e+3*(1/44)
flux_co2=cov*flux_co2
#
# Calculate Julian day of CO2 flux observations
#
dateyea=as.numeric(substr(date_co2,01,04))
datemon=as.numeric(substr(date_co2,06,07))
dateday=as.numeric(substr(date_co2,09,10))
datehou=as.numeric(substr(date_co2,12,13))
datemin=as.numeric(substr(date_co2,14,15))
#
datejul_co2 = myjulday(datemon,dateday,dateyea)
#
# Add fraction of day
#
datejul_co2 = datejul_co2 + datehou/24.0 + datemin/24.0/60.0
#
# Set CO2 fluxes to NA when flag indicates that the
# conditions for flux measurements are not fulfilled.
#
ind_co2 = which((stat_co2>=6)&(stat_co2<=9))
#
print(paste(100*length(ind_co2)/length(stat_co2),"% of CO2 observations are removed"))
print("since the conditions for the flux measurements are not fulfilled")
print(" ")
#
flux_co2[ind_co2]=NA
#
# Read surface energy budget from ASCII files
#
pfin="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/Ta/FLUX/V8/mensuel/"
#
filelist=c()
filelist=c(filelist,"Ta200402_flx1800-dat.txt")
filelist=c(filelist,"Ta200403_flx1800-dat.txt")
filelist=c(filelist,"Ta200404_flx1800-dat.txt")
filelist=c(filelist,"Ta200405_flx1800-dat.txt")
filelist=c(filelist,"Ta200406_flx1800-dat.txt")
filelist=c(filelist,"Ta200407_flx1800-dat.txt")
filelist=c(filelist,"Ta200408_flx1800-dat.txt")
filelist=c(filelist,"Ta200409_flx1800-dat.txt")
filelist=c(filelist,"Ta200410_flx1800-dat.txt")
filelist=c(filelist,"Ta200411_flx1800-dat.txt")
filelist=c(filelist,"Ta200412_flx1800-dat.txt")
filelist=c(filelist,"Ta200501_flx1800-dat.txt")
filelist=c(filelist,"Ta200502_flx1800-dat.txt")
#
# The number of half-hourly time steps
#
nstep = (myjulday(2,28,2005) - myjulday(2,15,2004) + 1) * 2 * 24
#
dayvecmerge=array(NA,nstep)
houvecmerge=array(NA,nstep)
senvecmerge=array(NA,nstep)
latvecmerge=array(NA,nstep)
radvecmerge=array(NA,nstep)
#
count=1
#
for (tt in 1:length(filelist)) {
  #
  # Remove the header of the file
  #
  filein=paste(pfin,filelist[tt],sep="")
  fileau=paste(pfin,filelist[tt],"_aux",sep="")
  #
  system(paste("tail -n +14 ",filein," > ",fileau,sep=""))
  #
  # Scan the data
  #
  dataobshalf=scan(fileau,what="character",quiet=TRUE)
  #
  len=length(dataobshalf)
  #
  posday=seq(01,len,10)
  poshou=seq(02,len,10)
  possen=seq(04,len,10)
  poslat=seq(06,len,10)
  posrad=seq(10,len,10)
  #
  dayvec=dataobshalf[posday]
  houvec=dataobshalf[poshou]
  #
  senvec=as.numeric(dataobshalf[possen])
  latvec=as.numeric(dataobshalf[poslat])
  radvec=as.numeric(dataobshalf[posrad])
  #
  senvec[which(senvec > +998)]=NA
  latvec[which(latvec > +998)]=NA
  radvec[which(radvec > +998)]=NA
  #
  senvec[which(senvec < -998)]=NA
  latvec[which(latvec < -998)]=NA
  radvec[which(radvec < -998)]=NA
  #
  senvec[which(senvec == -9.999999e-99)]=NA
  latvec[which(latvec == -9.999999e-99)]=NA
  radvec[which(radvec == -9.999999e-99)]=NA
  #
  len=length(dayvec)
  #
  dayvecmerge[count:(count+len-1)]=dayvec
  houvecmerge[count:(count+len-1)]=houvec
  senvecmerge[count:(count+len-1)]=senvec
  latvecmerge[count:(count+len-1)]=latvec
  radvecmerge[count:(count+len-1)]=radvec
  #
  count=count+len
  #
}
#
if ((count-1)!=nstep) {print(count-1);print(nstep);stop("Wrong number of time steps read from ASCII files")}
#
# Write the half-hourly values into ASCII-Files
#
write.table(dayvecmerge,file=paste(pdata,"MAST_DAY" ,sep=""),append=FALSE)
write.table(houvecmerge,file=paste(pdata,"MAST_TIME",sep=""),append=FALSE)
write.table(senvecmerge,file=paste(pdata,"MAST_SEN" ,sep=""),append=FALSE)
write.table(latvecmerge,file=paste(pdata,"MAST_LAT" ,sep=""),append=FALSE)
write.table(radvecmerge,file=paste(pdata,"MAST_RAD" ,sep=""),append=FALSE)
#
# Convert date
#
dateday=array(NA,length(dayvecmerge))
datemon=array(NA,length(dayvecmerge))
dateyea=array(NA,length(dayvecmerge))
datehou=array(NA,length(dayvecmerge))
datemin=array(NA,length(dayvecmerge))
datejul=array(NA,length(dayvecmerge))
#
dateday=as.vector(as.numeric(substr(dayvecmerge,1,02)))
datemon=as.vector(as.numeric(substr(dayvecmerge,4,05)))
dateyea=as.vector(as.numeric(substr(dayvecmerge,7,10)))
#
datejul=myjulday(datemon,dateday,dateyea)
#
# Add fraction of day
#
datehou=as.vector(as.numeric(substr(houvecmerge,1,2)))
datemin=as.vector(as.numeric(substr(houvecmerge,3,4))) + 1
#
datejul=datejul+datehou/24.0+datemin/24.0/60.0
#
# Extract time series starting with 20/02/2004 00:30:00
#
datejulstart = myjulday(2,20,2004) + 30/60/24
indstart     = which(abs(datejul-datejulstart)<1.0e-4)
#
if (length(indstart)!=1) stop("Start date not found")
LEN=length(dayvecmerge)
#
dayvecmerge = dayvecmerge [indstart:LEN]
datejul     = datejul     [indstart:LEN]
senvec      = senvecmerge [indstart:LEN]
latvec      = latvecmerge [indstart:LEN]
radvec      = radvecmerge [indstart:LEN]
#
# Check whether dates are consistant between
# CO2 fluxes and mast observations
#
if (length(datejul)!=length(datejul_co2)) stop("Inconsistant length of flux vectors")
#
adiff=datejul_co2-datejul
if (max(abs(adiff))>1.0e-4) {print(max(abs(adiff)));stop("Inconsistant dates of flux vectors")}
#
# Check percentage of missing data
#
print("                                   ")
print("Percentage of missing data for mast")
print(paste("Sensible heat flux : ",100*length(which(is.na(senvec)))  /length(senvec)  ))
print(paste("Latent   heat flux : ",100*length(which(is.na(latvec)))  /length(latvec)  ))
print(paste("Net radiation      : ",100*length(which(is.na(radvec)))  /length(radvec)  ))
print(paste("CO2 concentration  : ",100*length(which(is.na(conc_co2)))/length(conc_co2)))
print(paste("CO2 flux           : ",100*length(which(is.na(flux_co2)))/length(flux_co2)))
#
# Calculate the average diurnal cycle during the DJF, MAM, JJA and SON seasons
#
# Create vector with season
#
auxjul=floor(datejul)
LST=mycaldat(auxjul)
monvec=LST$m
seavec=array(NA,length(monvec))
#
seavec[which(((monvec<=02)|(monvec>=12)))]=1
seavec[which(((monvec>=03)&(monvec<=05)))]=2
seavec[which(((monvec>=06)&(monvec<=08)))]=3
seavec[which(((monvec>=09)&(monvec<=11)))]=4
#
# Create vector with the hour of the day
#
houvec=24*(datejul-floor(datejul))
#
# Find indices belonging to hour and season
#
inddayseas=array(NA,c(4,48,150))
#
for (seas in 1:4) for (hou in 0:47) {
  aux=which((seavec==seas)&(abs(houvec-hou/2)<0.0001))
  inddayseas[seas,(hou+1),1:length(aux)]=aux
}
#
# Average diurnal cycle per season
#
sendcyc      = array(NA,c(4,48))
latdcyc      = array(NA,c(4,48))
raddcyc      = array(NA,c(4,48))
co2_concdcyc = array(NA,c(4,48))
co2_fluxdcyc = array(NA,c(4,48))
#
for (seas in 1:4) for (hou in 1:48) {
  sendcyc     [seas,hou]=mean(senvec  [inddayseas[seas,hou,]],na.rm=TRUE)
  latdcyc     [seas,hou]=mean(latvec  [inddayseas[seas,hou,]],na.rm=TRUE)
  raddcyc     [seas,hou]=mean(radvec  [inddayseas[seas,hou,]],na.rm=TRUE)
  co2_concdcyc[seas,hou]=mean(conc_co2[inddayseas[seas,hou,]],na.rm=TRUE)
  co2_fluxdcyc[seas,hou]=mean(flux_co2[inddayseas[seas,hou,]],na.rm=TRUE)
}
#
# Calculate the diurnal averages assuming half-hourly values
#
nday=length(datejul)/48
if (floor(nday)!=nday) stop("No integer number of days")
#
# Get indices
#
auxdatejul=datejul-datejul[1]
indav=array(NA,c(nday,48))
for (day in 1:nday) {
  aux=which(floor(auxdatejul)==(day-1))
  if (length(aux)<46) {print(aux);stop("Wrong indices")}
  indav[day,1:length(aux)]=aux
}
#
# Calculation of the daily averages
#
datdayav      = array(NA,nday)
juldayav      = array(NA,nday)
sendayav      = array(NA,nday)
latdayav      = array(NA,nday)
raddayav      = array(NA,nday)
co2_concdayav = array(NA,nday)
co2_fluxdayav = array(NA,nday)
#
for (day in 1:nday) {
  datdayav     [day]=dayvecmerge[indav[day,1]]
  juldayav     [day]=datejul[indav[day,1]]
  sendayav     [day]=mean(senvec[indav[day,]],na.rm=TRUE)
  latdayav     [day]=mean(latvec[indav[day,]],na.rm=TRUE)
  raddayav     [day]=mean(radvec[indav[day,]],na.rm=TRUE)
  co2_concdayav[day]=mean(conc_co2[indav[day,]],na.rm=TRUE)
  co2_fluxdayav[day]=mean(flux_co2[indav[day,]],na.rm=TRUE)
}
#
# Calculation of the average weekly cycle
# during the heating and cooling season
# HEAT: 01/10/2004 to 28/02/2005
# COOL: 01/06/2004 to 31/08/2004
#
datebeg_heat=myjulday(10,01,2004)
dateend_heat=myjulday(02,28,2005)
#
INDHEAT=which((juldayav>=datebeg_heat)&(juldayav<=dateend_heat))
#
datebeg_cool=myjulday(06,01,2004)
dateend_cool=myjulday(08,31,2004)
#
INDCOOL=which((juldayav>=datebeg_cool)&(juldayav<=dateend_cool))
#
# Determine the day of week
#
aux=as.Date(datdayav,format="%d/%m/%Y")
weekd=weekdays(aux)
#
# Select data
#
sencool      = sendayav[INDCOOL]
latcool      = latdayav[INDCOOL]
radcool      = raddayav[INDCOOL]
co2_conccool = co2_concdayav[INDCOOL]
co2_fluxcool = co2_fluxdayav[INDCOOL]
weekdcool    = weekd   [INDCOOL]
#
senheat      = sendayav[INDHEAT]
latheat      = latdayav[INDHEAT]
radheat      = raddayav[INDHEAT]
co2_concheat = co2_concdayav[INDHEAT]
co2_fluxheat = co2_fluxdayav[INDHEAT]
weekdheat    = weekd   [INDHEAT]
#
dayofweek=array(NA,7)
#
sencoolwd      = array(NA,c(2,7))
latcoolwd      = array(NA,c(2,7))
radcoolwd      = array(NA,c(2,7))
co2_conccoolwd = array(NA,c(2,7))
co2_fluxcoolwd = array(NA,c(2,7))
#
senheatwd      = array(NA,c(2,7))
latheatwd      = array(NA,c(2,7))
radheatwd      = array(NA,c(2,7))
co2_concheatwd = array(NA,c(2,7))
co2_fluxheatwd = array(NA,c(2,7))
#
for (wd in 1:7) {
  #
  if      (wd==1) named="lundi"
  else if (wd==2) named="mardi"
  else if (wd==3) named="mercredi"
  else if (wd==4) named="jeudi"
  else if (wd==5) named="vendredi"
  else if (wd==6) named="samedi"
  else if (wd==7) named="dimanche"
  else stop("This option is not implemented")
  #
  dayofweek[wd]=named
  #
  lcool=length(which(!is.na(sencool[which(weekdcool==named)])))
  lheat=length(which(!is.na(senheat[which(weekdheat==named)])))
  #
  sencoolwd     [1,wd] = mean(sencool     [which(weekdcool==named)],na.rm=TRUE)
  latcoolwd     [1,wd] = mean(latcool     [which(weekdcool==named)],na.rm=TRUE)
  radcoolwd     [1,wd] = mean(radcool     [which(weekdcool==named)],na.rm=TRUE)
  co2_conccoolwd[1,wd] = mean(co2_conccool[which(weekdcool==named)],na.rm=TRUE)
  co2_fluxcoolwd[1,wd] = mean(co2_fluxcool[which(weekdcool==named)],na.rm=TRUE)
  #
  senheatwd     [1,wd] = mean(sencool     [which(weekdheat==named)],na.rm=TRUE)
  latheatwd     [1,wd] = mean(latcool     [which(weekdheat==named)],na.rm=TRUE)
  radheatwd     [1,wd] = mean(radcool     [which(weekdheat==named)],na.rm=TRUE)
  co2_concheatwd[1,wd] = mean(co2_concheat[which(weekdheat==named)],na.rm=TRUE)
  co2_fluxheatwd[1,wd] = mean(co2_fluxheat[which(weekdheat==named)],na.rm=TRUE)
  #
  sencoolwd     [2,wd] = sd(sencool     [which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  latcoolwd     [2,wd] = sd(latcool     [which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  radcoolwd     [2,wd] = sd(radcool     [which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  co2_conccoolwd[2,wd] = sd(co2_conccool[which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  co2_fluxcoolwd[2,wd] = sd(co2_fluxcool[which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  #
  senheatwd     [2,wd] = sd(sencool     [which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  latheatwd     [2,wd] = sd(latcool     [which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  radheatwd     [2,wd] = sd(radcool     [which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  co2_concheatwd[2,wd] = sd(co2_concheat[which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  co2_fluxheatwd[2,wd] = sd(co2_fluxheat[which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  #
}
#
# Save the mast observation based data
#
LST=list(sencoolwd=sencoolwd,latcoolwd=latcoolwd,radcoolwd=radcoolwd,
         co2_conccoolwd=co2_conccoolwd,co2_fluxcoolwd=co2_fluxcoolwd,
         senheatwd=senheatwd,latheatwd=latheatwd,radheatwd=radheatwd,
	 co2_concheatwd=co2_concheatwd,co2_fluxheatwd=co2_fluxheatwd,
         datdayav=datdayav,juldayav=juldayav,sendayav=sendayav,
	 latdayav=latdayav,raddayav=raddayav,co2_concdayav=co2_concdayav,
	 co2_fluxdayav=co2_fluxdayav,sendcyc=sendcyc,latdcyc=latdcyc,
	 raddcyc=raddcyc,co2_concdcyc=co2_concdcyc,co2_fluxdcyc=co2_fluxdcyc)
#
save(LST,file=fobs)
#
}
#
# ###############################################################
# Load data from routine observation station
# ###############################################################
#
seload_rod_station=function(path,filesave,juldaybeg,juldayend,namestat,numbstat) {
  #
  print("                                       ")
  print("Preparation of routine observation data")
  print("                                       ")
  #
  # Load meta data
  #
  filemeta=paste(path,"Meta.txt",sep="")
  data=read.table(filemeta,header=FALSE,skip=0,sep=" ")
  LEN=length(data)
  if (LEN!=03) stop("Error in read of metadata")
  #
  lats = data$V1
  lons = data$V2
  heig = data$V3
  #
  if ((lats>090.01)||(lats<(-090.01))) stop("Error in station latitude")
  if ((lons>180.01)||(lons<(-180.01))) stop("Error in station longitude")
  if ((heig  < 0.0)||(heig>(1000.0 ))) stop("Error in station height")
  #
  # Load data
  #
  # The time of day axis (hourly values are taken)
  #
  hours=seq(1,24,1)/24
  #
  # The length of the full time series
  #
  lenfullts=length(hours)*(juldayend-juldaybeg+1)
  #
  # Allocate output fields
  # 
  juldayfrall = array(NA,lenfullts)
  juld_fullts = array(NA,lenfullts)
  temp_fullts = array(NA,lenfullts)
  relh_fullts = array(NA,lenfullts)
  qspe_fullts = array(NA,lenfullts)
  pabs_fullts = array(NA,lenfullts)
  rrat_fullts = array(NA,lenfullts)
  widd_fullts = array(NA,lenfullts)
  wiff_fullts = array(NA,lenfullts)
  glob_fullts = array(NA,lenfullts)
  #
  for (day in juldaybeg:juldayend) {
    #
    ind1=1+length(hours)*(  day-juldaybeg)
    ind2=  length(hours)*(1+day-juldaybeg)
    #
    juldayfrall[ind1:ind2]=day
    juldayfrall[ind1:ind2]=juldayfrall[ind1:ind2]+hours
    #
  }
  #
  # Load data
  #
  data=read.table(paste(path,"Data.txt",sep=""),header=TRUE,skip=0,sep=";")
  #
  timeread = as.vector(data$DATE)
  rratread = as.numeric(as.vector(data$RR1))
  tempread = as.numeric(as.vector(data$T))
  tdewread = as.numeric(as.vector(data$TD))
  pabsread = as.numeric(as.vector(data$PSTAT))
  wiffread = as.numeric(as.vector(data$FF))
  widdread = as.numeric(as.vector(data$DD))
  relhread = as.numeric(as.vector(data$U))
  globread = as.numeric(as.vector(data$GLO))
  #
  # Convert J/cm² (hour) to W/m²
  #
  globread=globread*(1.0/0.36)
  #
  # Plausibility checks for meteorological variables
  #
  amax=max(tempread,na.rm=TRUE)
  amin=min(tempread,na.rm=TRUE)
  if (amax>50.0)    {print(amax);stop("Unplausible value for air temperature")}
  if (amin<(-50.0)) {print(amin);stop("Unplausible value for air temperature")}
  #
  amax=max(relhread,na.rm=TRUE)
  amin=min(relhread,na.rm=TRUE)
  if (amax>100.1) {print(amax);stop("Unplausible value for relative humidity")}
  if (amin<10.0 ) {print(amin);stop("Unplausible value for relative humidity")}
  #
  amax=max(rratread,na.rm=TRUE)
  amin=min(rratread,na.rm=TRUE)
  if (amax>100.1) {print(amax);stop("Unplausible value for precipitation rate")}
  if (amin<0.0 )  {print(amin);stop("Unplausible value for precipitation rate")}
  #
  amax=max(tdewread,na.rm=TRUE)
  amin=min(tdewread,na.rm=TRUE)
  if (amax>50.1)    {print(amax);stop("Unplausible value for dew point")}
  if (amin<(-50.0)) {print(amin);stop("Unplausible value for dew point")}
  #
  amax=max(pabsread,na.rm=TRUE)
  amin=min(pabsread,na.rm=TRUE)
  if (amax>1100.1) {print(amax);stop("Unplausible value for pressure")}
  if (amin<900.0 ) {print(amin);stop("Unplausible value for pressure")}
  #
  amax=max(wiffread,na.rm=TRUE)
  amin=min(wiffread,na.rm=TRUE)
  if (amax>100.1) {print(amax);stop("Unplausible value for wind speed")}
  if (amin<0.0 ) {print(amin);stop("Unplausible value for wind speed")}
  #
  amax=max(widdread,na.rm=TRUE)
  amin=min(widdread,na.rm=TRUE)
  if (amax>360.1) {print(amax);stop("Unplausible value for wind direction")}
  if (amin<0.0 ) {print(amin);stop("Unplausible value for wind direction")}
  #
  amax=max(globread,na.rm=TRUE)
  amin=min(globread,na.rm=TRUE)
  if (amax>2000.1) {print(amax);stop("Unplausible value for total shortwave radiation")}
  if (amin<0.0 )   {print(amin);stop("Unplausible value for total shortwave radiation")}
  #
  # Determine the julian day of the observations
  #
  yearread = as.integer(substr(timeread,01,04))
  montread = as.integer(substr(timeread,05,06))
  dayyread = as.integer(substr(timeread,07,08))
  hourread = as.integer(substr(timeread,09,10))
  #
  # Calculate the fractional Julian date of the observed data
  #
  juldayintobs = myjulday(montread,dayyread,yearread)
  juldayfraobs = juldayintobs + (3600*hourread)/86400
  #
  # Attribute the meteorological data to the correct time interval
  # The average over the hour preceeding the target hour is taken.
  #
  count = 0
  #
  for (ii in 1:length(juldayfrall)) {
    #
    tdiff=juldayfrall[ii]-juldayfraobs
    ind=which((tdiff<(59.5/1440.0))&(tdiff>=0.0))
    #
    if (length(ind)>1.5) {
      print(ind)
      print("More than one value per hour found in observations")
      print("However, ROD observations should be hourly")
      stop("Program halted")
    } else if (length(ind)>0.5) {
      #
      # Estimate the specific humidity [g/kg]
      #
      ZFOES  = psat(tempread[ind]+XTT)
      ZWORK1 = ZFOES/(100.0*pabsread[ind])
      ZWORK2 = XRD/XRV
      qsat = ZWORK2*ZWORK1/(1.+(ZWORK2-1.)*ZWORK1)
      qspe_fullts[ii]=1.0e+3 * 0.01 * relhread[ind] * qsat
      #
      temp_fullts[ii]=tempread[ind]
      relh_fullts[ii]=relhread[ind]
      pabs_fullts[ii]=pabsread[ind]
      rrat_fullts[ii]=rratread[ind]
      widd_fullts[ii]=widdread[ind]
      wiff_fullts[ii]=wiffread[ind]
      glob_fullts[ii]=globread[ind]
      #
    } else {
      count = count+1
    }
    #
  }
  #
  print(paste("A total of ",100*count/length(juldayfrall),"% of requested time steps are missing"))
  #
  # Write the data on the output fields
  #
  juld_fullts = juldayfrall
  #
  # Save output
  #
  print(dim(rrat_fullts))
  stop("Check")
  #
  LST=list(lats=lats,lons=lons,heig=heig,name=namestat,
           numb=numbstat,juld_fullts=juld_fullts,
           temp_fullts=temp_fullts,relh_fullts=relh_fullts,
           qspe_fullts=qspe_fullts,pabs_fullts=pabs_fullts,
           rrat_fullts=rrat_fullts,widd_fullts=widd_fullts,
           wiff_fullts=wiff_fullts,glob_fullts=glob_fullts)
  #
  save(LST,file=filesave)
  #
}
#
# ##################################################################
# Prepare observations of anthropogenic heat from CAPITOUL campaign
# ##################################################################
#
seprepobs=function(fobs,timedayout,timedayfor) {
#
# A: Daily values
#
pfin="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/qf/Processed_data/Center/qf.dat"
#
dataobs=scan(pfin,what="character",quiet=TRUE)
#
lenobs=length(dataobs)
#
posdate=seq(1,lenobs,5)
poshant=seq(2,lenobs,5)
poshele=seq(3,lenobs,5)
poshtra=seq(4,lenobs,5)
poshgas=seq(5,lenobs,5)
#
dateobs=dataobs[posdate]
hantobs=dataobs[poshant]
heleobs=dataobs[poshele]
htraobs=dataobs[poshtra]
hgasobs=dataobs[poshgas]
#
# Convert observation date
#
dateobsday=array(NA,length(dateobs))
dateobsmon=array(NA,length(dateobs))
dateobsyea=array(NA,length(dateobs))
datejulobs=array(NA,length(dateobs))
#
dateobsold=dateobs
#
for (ii in 1:length(dateobs)) {
  #
  aux=dateobs[ii]
  #
  dateobsday[ii]=as.numeric(substr(aux,1,02))
  dateobsmon[ii]=as.numeric(substr(aux,4,05))
  dateobsyea[ii]=as.numeric(substr(aux,7,10))
  #
  datejulobs[ii]=myjulday(dateobsmon[ii],dateobsday[ii],dateobsyea[ii])
  #
}
#
# Omit the first 4 time steps of the observational record
#
dateobs=datejulobs[5:length(datejulobs)]
dateobsold=dateobsold[5:length(dateobsold)]
#
hantobs=as.numeric(hantobs[5:length(datejulobs)])
heleobs=as.numeric(heleobs[5:length(datejulobs)])
htraobs=as.numeric(htraobs[5:length(datejulobs)])
hgasobs=as.numeric(hgasobs[5:length(datejulobs)])
#
# Check equal dates
#
diff=abs(dateobs-timedayout)
diff=abs(dateobs-timedayfor)
#
if (max(diff)>0.01) stop("Dates must be equal")
#
# Calculate the average weekly cycle during heating
# and cooling period.
# HEAT: 01/02/2004 to 01/04/2004 and 01/11/2004 to 01/03/2005
# COOL: 01/06/2004 to 01/09/2004
#
date1beg=myjulday(02,01,2004)
date1end=myjulday(04,01,2004)
date2beg=myjulday(11,01,2004)
date2end=myjulday(04,01,2005)
#
INDHEAT=which( ((dateobs>=date1beg)&(dateobs<=date1end)) |
               ((dateobs>=date2beg)&(dateobs<=date2end)) )
#
# Days during the cooling season 01/06/2004 to 01/09/2005 
#
date1beg=myjulday(06,01,2004)
date1end=myjulday(09,01,2004)
#
INDCOOL=which((dateobs>=date1beg)&(dateobs<=date1end))
#
# Determine the day of week
#
aux=as.Date(dateobsold,format="%d/%m/%Y")
weekd=weekdays(aux)
#
# Select data
#
hantobscool=hantobs[INDCOOL]
heleobscool=heleobs[INDCOOL]
htraobscool=htraobs[INDCOOL]
hgasobscool=hgasobs[INDCOOL]
weekdcool=weekd[INDCOOL]
#
hantobsheat=hantobs[INDHEAT]
heleobsheat=heleobs[INDHEAT]
htraobsheat=htraobs[INDHEAT]
hgasobsheat=hgasobs[INDHEAT]
weekdheat=weekd[INDHEAT]
#
dayofweek=array(NA,7)
#
hantobscoolwd=array(NA,c(2,7))
heleobscoolwd=array(NA,c(2,7))
htraobscoolwd=array(NA,c(2,7))
hgasobscoolwd=array(NA,c(2,7))
#
hantobsheatwd=array(NA,c(2,7))
heleobsheatwd=array(NA,c(2,7))
htraobsheatwd=array(NA,c(2,7))
hgasobsheatwd=array(NA,c(2,7))
#
for (wd in 1:7) {
  #
  if      (wd==1) named="lundi"
  else if (wd==2) named="mardi"
  else if (wd==3) named="mercredi"
  else if (wd==4) named="jeudi"
  else if (wd==5) named="vendredi"
  else if (wd==6) named="samedi"
  else if (wd==7) named="dimanche"
  else stop("This option is not implemented")
  #
  dayofweek[wd]=named
  #
  lcool=length(na.omit(hantobscool[which(weekdcool==named)]))
  lheat=length(na.omit(hantobsheat[which(weekdheat==named)]))
  #
  hantobscoolwd[1,wd]=mean(hantobscool[which(weekdcool==named)],na.rm=TRUE)
  heleobscoolwd[1,wd]=mean(heleobscool[which(weekdcool==named)],na.rm=TRUE)
  htraobscoolwd[1,wd]=mean(htraobscool[which(weekdcool==named)],na.rm=TRUE)
  hgasobscoolwd[1,wd]=mean(hgasobscool[which(weekdcool==named)],na.rm=TRUE)
  #
  hantobsheatwd[1,wd]=mean(hantobsheat[which(weekdheat==named)],na.rm=TRUE)
  heleobsheatwd[1,wd]=mean(heleobsheat[which(weekdheat==named)],na.rm=TRUE)
  htraobsheatwd[1,wd]=mean(htraobsheat[which(weekdheat==named)],na.rm=TRUE)
  hgasobsheatwd[1,wd]=mean(hgasobsheat[which(weekdheat==named)],na.rm=TRUE)
  #
  hantobscoolwd[2,wd]=sd(hantobscool[which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  heleobscoolwd[2,wd]=sd(heleobscool[which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  htraobscoolwd[2,wd]=sd(htraobscool[which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  hgasobscoolwd[2,wd]=sd(hgasobscool[which(weekdcool==named)],na.rm=TRUE)/sqrt(lcool)
  #
  hantobsheatwd[2,wd]=sd(hantobsheat[which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  heleobsheatwd[2,wd]=sd(heleobsheat[which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  htraobsheatwd[2,wd]=sd(htraobsheat[which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  hgasobsheatwd[2,wd]=sd(hgasobsheat[which(weekdheat==named)],na.rm=TRUE)/sqrt(lheat)
  #
}
#
# B: Half hourly values
#
pfin="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/qf/Processed_data/Center/qf_%1800.dat"
#
dataobshalf=scan(pfin,what="character",quiet=TRUE)
#
lenobshalf=length(dataobshalf)
#
posdatehalf=seq(1,lenobshalf,5)
poshourhalf=seq(2,lenobshalf,5)
poshelehalf=seq(3,lenobshalf,5)
poshtrahalf=seq(4,lenobshalf,5)
poshgashalf=seq(5,lenobshalf,5)
#
dateobshalf=dataobshalf[posdatehalf]
hourobshalf=dataobshalf[poshourhalf]
heleobshalf=as.numeric(dataobshalf[poshelehalf])
htraobshalf=as.numeric(dataobshalf[poshtrahalf])
hgasobshalf=as.numeric(dataobshalf[poshgashalf])
#
# Convert observation date
#
dateobsdayhalf=array(NA,length(dateobshalf))
dateobsmonhalf=array(NA,length(dateobshalf))
dateobsyeahalf=array(NA,length(dateobshalf))
dateobshouhalf=array(NA,length(dateobshalf))
dateobsminhalf=array(NA,length(dateobshalf))
datejulobshalf=array(NA,length(dateobshalf))
#
for (ii in 1:length(dateobshalf)) {
  #
  aux=dateobshalf[ii]
  #
  dateobsdayhalf[ii]=as.numeric(substr(aux,1,02))
  dateobsmonhalf[ii]=as.numeric(substr(aux,4,05))
  dateobsyeahalf[ii]=as.numeric(substr(aux,7,10))
  #
  datejulobshalf[ii]=myjulday(dateobsmonhalf[ii],dateobsdayhalf[ii],dateobsyeahalf[ii])
  #
  # Add fraction of day
  #
  aux=hourobshalf[ii]
  #
  dateobshouhalf[ii]=as.numeric(substr(aux,1,2))
  dateobsminhalf[ii]=as.numeric(substr(aux,4,5))
  #
  datejulobshalf[ii]=datejulobshalf[ii]+dateobshouhalf[ii]/24.0+dateobsminhalf[ii]/24.0/60.0
  #
}
#
dateobshalf=datejulobshalf
#
# Calculate the average diurnal cycle during 
# the DJF, MAM, JJA and SON seasons
#
# Create vector with season
#
auxjul=floor(dateobshalf)
LST=mycaldat(auxjul)
monvec=LST$m
seavec=array(NA,length(monvec))
#
seavec[which(((monvec<=02)|(monvec>=12)))]=1
seavec[which(((monvec>=03)&(monvec<=05)))]=2
seavec[which(((monvec>=06)&(monvec<=08)))]=3
seavec[which(((monvec>=09)&(monvec<=11)))]=4
#
# Create vector with the hour of the day
#
houvec=24*(dateobshalf-floor(dateobshalf))
#
# Find indices belonging to hour and season
#
inddayseas=array(NA,c(4,48,150))
#
for (seas in 1:4) for (hou in 0:47) {
  aux=which((seavec==seas)&(abs(houvec-hou/2)<0.0001))
  inddayseas[seas,(hou+1),1:length(aux)]=aux
}
#
# Average diurnal cycle per seaon
#
heleobsdcyc=array(NA,c(4,48))
htraobsdcyc=array(NA,c(4,48))
hgasobsdcyc=array(NA,c(4,48))
#
for (seas in 1:4) for (hou in 1:48) {
  heleobsdcyc[seas,hou]=mean(heleobshalf[inddayseas[seas,hou,]],na.rm=TRUE)
  htraobsdcyc[seas,hou]=mean(htraobshalf[inddayseas[seas,hou,]],na.rm=TRUE)
  hgasobsdcyc[seas,hou]=mean(hgasobshalf[inddayseas[seas,hou,]],na.rm=TRUE)
}
#
# Save all observation based data
#
LST=list(dateobs=dateobs,hantobs=hantobs,heleobs=heleobs,
         hantobscoolwd=hantobscoolwd,heleobscoolwd=heleobscoolwd,
         htraobscoolwd=htraobscoolwd,hgasobscoolwd=hgasobscoolwd,
         hantobsheatwd=hantobsheatwd,heleobsheatwd=heleobsheatwd,
         htraobsheatwd=htraobsheatwd,hgasobsheatwd=hgasobsheatwd,
         hantobscoolwd=hantobscoolwd,heleobscoolwd=heleobscoolwd,
         htraobscoolwd=htraobscoolwd,hgasobscoolwd=hgasobscoolwd,
         hantobsheatwd=hantobsheatwd,heleobsheatwd=heleobsheatwd,
         htraobsheatwd=htraobsheatwd,hgasobsheatwd=hgasobsheatwd,
         dayofweek,heleobsdcyc=heleobsdcyc,
         htraobsdcyc=htraobsdcyc,hgasobsdcyc=hgasobsdcyc,
         htraobs=htraobs,hgasobs=hgasobs,dateobshalf=dateobshalf,
         heleobshalf=heleobshalf,htraobshalf=htraobshalf,
         hgasobshalf=hgasobshalf)
#
save(LST,file=fobs)
#
}
