#!/usr/bin/Rscript
#
library(sp)
library(RNetCDF)
library(MASS)
library(gplots)
library(raster)
library(pracma)
library(foreign)
library(ggplot2)
library(circular)
library(geosphere)
library(RColorBrewer)
#
source("sub.R")
source("input.R")
source("subplot.R")
#
warnings()
#
# ####################################################
# User input section
# ####################################################
#
# The name of the simulation
#
name="HK_HWSep2009_Final"
name="HK_HWMay2018_Final"
#
# The variables that shall be analysed
#
varlist=c("UHOR","WW","TH","TK","RV")
#
# The number of regions
#
nreg=7
#
# The path for saving the budgets 
#
psave="/home/robert/Bureau/SURFEX/ANALYSIS/BUDGETS/"
#
# The names of the regions
#
name_reg=c("KLT","KP","HKO","HKP","KOWLOON","HKIW","HKIE")
#
# Shift between UTC and local time
#
time_delta = 8
#
# The limit for plotting the contribution
#
limper = 0.04
#
# ####################################################
# End user input section
# ####################################################
#
for (var in varlist) {
  #
  if (var=="UHOR") {
     proc_name=c("INIF_BU_RU","ENDF_BU_RU","AVEF_BU_RU","Grid nesting","Curvature",
                 "Coriolis force","Vegetation drag","Building drag",
		 "Vertical turbulence","Horizontal turbulence","Advection",
		 "Pressure gradient")
     unit="m/s/h"
     cfl = 3600.0
  } else if (var=="WW") {
     proc_name=c("INIF_BU_RW","ENDF_BU_RW","AVEF_BU_RW","Grid nesting","Curvature",
                 "Coriolis force","Vertical turbulence","Horizontal turbulence",
		 "Gravity","Advection","Pressure gradient")
     unit="m/s/h"
     cfl = 3600.0
  } else if (var=="TH") {
     proc_name=c("INIF_BU_RTH","ENDF_BU_RTH","AVEF_BU_RTH","Grid nesting","Radiation",
                 "Building heat flux","Vertical turbulence","Horizontal turbulence",
		 "Dissipation","Advection","NEGA_BU_RTH","HENU_BU_RTH","HON_BU_RTH",
		 "SFR_BU_RTH","DEPS_BU_RTH","DEPG_BU_RTH","REVA_BU_RTH","RIM_BU_RTH",
		 "ACC_BU_RTH","CFRZ_BU_RTH","WETG_BU_RTH","DRYG_BU_RTH","GMLT_BU_RTH",
		 "IMLT_BU_RTH","BERFI_BU_RTH","CDEPI_BU_RTH")
    unit="Air temperature tendency [K/h]"
    cfl = 3600.0
  } else if (var=="TK") {
    proc_name=c("INIF_BU_RTKE","ENDF_BU_RTKE","AVEF_BU_RTKE","Vegetation drag",
    "Building drag","Dynamical production","Thermal production","Dissipation",
    "Turbulent transport","Advection")
    unit="TKE tendency [m²/s²/h]"
    cfl = 3600.0
  } else if (var=="RV") {
    proc_name=c("INIF_BU_RRV","ENDF_BU_RRV","AVEF_BU_RRV","Grid nesting",
    "Building moisture flux","Vertical turbulence","Horizontal turbulence",
    "Advection","NEGA_BU_RRV","HENU_BU_RRV","DEPS_BU_RRV","DEPG_BU_RRV",
    "REVA_BU_RRV","CDEPI_BU_RRV")
    unit="Mixing ratio tendency [g/kg/h]"
    cfl = 1000*3600.0
  } else {
    print(var)
    stop("No rule implemented for this variable")
  }
  #
  for (reg in 1:nreg) {
    #
    nameload = paste0(psave,name,"Budget_",var,"_",name_reg[reg],".csv")
    data = cfl*read.csv(nameload,header=FALSE,sep = ",", quote = "\"",dec = ".")
    #
    if (reg==1) {
      DIM=dim(data)
      data_allreg=array(NA,c(DIM[1],DIM[2],nreg))
    }
    #
    for (ii in 1:DIM[1]) for (jj in 1:DIM[2]) {
      data_allreg[ii,jj,reg]=data[ii,jj]
    }
    #
  }
  #
  for (reg in 1:nreg) {  
    #
    data=data_allreg[,,reg]
    #
    # Tendency
    #
    tend=(data[2,]-data[1,])/cfl
    #
    # Sum of all process tendencies
    #
    sum_proc=array(0,DIM[2])
    for (proc in 4:length(proc_name)) {
      sum_proc=sum_proc+data[proc,]
    }
    #
    nameplot=paste0(psave,name,"Budget_",var,"_",name_reg[reg],".pdf")
    print(nameplot)
    #
    # Define xaxis and labels
    #
    xaxis=c(seq(1+time_delta,24,1),seq(1,time_delta,1))
    xlim=c(0.5,24.5)
    xaxis_lab=xaxis
    #
    # Define yaxis
    #
    LST=sescale(data_allreg[4:length(proc_name),,])
    ymin=LST$amin;ymax=LST$amax;yinc=LST$inc
    yaxis=seq(ymin,ymax,yinc)
    ylim=c(ymin,ymax)
    yaxis_lab=yaxis
    #
    # Create plot
    #
    pdf(nameplot,width=8.32,height=7)
    par(mar=c(5,6,1.0,1))
    par(oma=c(0,0,0,0))
    par(xaxs="i",yaxs="i")
    #
    cexall=1.8
    #
    plot0(axes=FALSE,ylim=ylim,xlim=xlim,xlab=NA,ylab=NA,frame.plot=T)
    #
    seginc=0.05
    seglwd=1.5
    #
    legendvec=c()
    colors=c()
    #
    max_allreg=max(data_allreg[4:length(proc_name),,])
    min_allreg=min(data_allreg[4:length(proc_name),,])
    dif_allreg=max_allreg-min_allreg
    #
    res = array(0.0,length(data[1,]))
    #
    for (proc in 4:length(proc_name)) {
      #
      vec_allreg = data_allreg[proc,,]      
      #
      # Only plot those contributions with at least one value
      # larger than 1% of the largest in one of the regions
      #
      if ( max(abs(vec_allreg)) > limper*dif_allreg ) {
        #
        res = res + data[proc,]
	#
        # Attribute the color according to the process
        #
	if (proc_name[proc]=="Vegetation drag") {
	  col_sel="darkgreen"
        } else if ( (proc_name[proc]=="Building drag") ||
	            (proc_name[proc]=="Building heat flux") ||
                    (proc_name[proc]=="Building moisture flux") ) {
	  col_sel="red"
	} else if ((proc_name[proc]=="Vertical turbulence")||(proc_name[proc]=="Dynamical production")) {
	  col_sel="blue"
    	} else if ((proc_name[proc]=="Horizontal turbulence")||(proc_name[proc]=="Thermal production")) {
	  col_sel="darkblue"
    	} else if ((proc_name[proc]=="Advection")||(proc_name[proc]=="Turbulent transport")) {
	  col_sel="green"
    	} else if (proc_name[proc]=="Pressure gradient") {
	  col_sel="magenta"
    	} else if (proc_name[proc]=="Coriolis force") {
	  col_sel="brown"
    	} else if (proc_name[proc]=="Gravity") {
	  col_sel="orange"
    	} else if (proc_name[proc]=="Radiation") {
	  col_sel="darkred"
    	} else if (proc_name[proc]=="Dissipation") {
	  col_sel="brown4"
        } else {
          print(proc_name[proc])
	  print("Please define a color for this physical process")
	  stop("Check")
        }
        #
        points(xaxis,data[proc,],lty=1,pch=16,col=col_sel,cex=0.8)
	legendvec=c(legendvec,proc_name[proc])
	colors=c(colors,col_sel)
	#
      }
      #
    }
    #
    # Plot the sum of tendencies
    #
    # points(xaxis,res,lty=1,pch=16,col="black",cex=0.8)
    # legendvec=c(legendvec,"Sum of tendencies")
    # colors=c(colors,"black")
    #
    abline(h = 0, col="black", lwd=3, lty=1)
    #
    # Write axes
    #
    axis(1,at=xaxis,lab=xaxis_lab,las=1,cex.axis=1.3)
    axis(2,at=yaxis,lab=yaxis_lab,las=2,cex.axis=cexall)
    #
    mtext("Local time [h]",side=1,line=3,cex=cexall,col="black")
    mtext(unit            ,side=2,line=4,cex=cexall,col="black")
    #
    legend("topleft",legendvec,lty=1,col=colors,lwd=2.0,cex=1.3)
    #
    garbage <- dev.off()
    #
  }
}
#
print("                                ")
print("All budgets plotted successfully")
print("                                ")
#
