#!/usr/bin/Rscript
#
# Plot results of a coupled Meso-NH simulation
# that has been performed on a supercomputer
#
library(sp)
library(RNetCDF)
library(MASS)
library(gplots)
library(raster)
library(pracma)
library(foreign)
library(ggplot2)
library(circular)
library(geosphere)
library(RColorBrewer)
#
source("sub.R")
source("input.R")
source("subplot.R")
source("sereadpgd.R")
source("subloadobs.R")
#
options(warn=1)
#
# ####################################################
# User input section
# ####################################################
#
# Shall data be newly downloaded?
#
lnew=0
#
# The name of the simulation
#
name="HK_May2018"
padd="/RUNDIRS_HongKong/"
#
# Vector with nests to treat and their grid size [m]
#
nest_vec=c(4,5)
xdx=c(250,125)
#
# The height above ground of the first w model levels
#
zlvl_w=c(0.0,2.0,4.3,6.94,9.98,13.48,17.50,22.13,27.45,33.57,40.60,48.69,58.00,68.70,81.0,95.16,111.43,130.15,151.67)
#
# Interpolation to scalar levels
#
zlvl=0.5*(zlvl_w[1:(length(zlvl_w)-1)]+zlvl_w[2:(length(zlvl_w))])
#
# The name of the scenarios
#
scen=c("REFER")
#
# Execute loop over rundays 
#
lloop=1
#
# The date of initialisation
#
yearini=2018
montini=5
dayyini=1
#
# The number of simulation days to investigate
#
nrunday=31
#
# Start and end day of time series to investigate
#
ndaybeg_ts=1
ndayend_ts=31
#
# Boundaries of domain to extract for time series
#
latmin_ts=47.2534
latmax_ts=47.3860
lonmin_ts=04.8782
lonmax_ts=05.1524
#
# The output time steps (hours in day)
#
nout_time=seq(1,24,1)
#
# Flag for evaluation against station network observations
#
flag_capitoul=0.0
flag_dijonobs=0.0
flag_hkongobs=1.0
#
# The supercomputer where the simulations have been performed
#
supcmp="beaufix"
#
# The seasons to investigate
#
seasvec=c("MAY2018")
julday_seasons=array(NA,c(length(seasvec),nrunday))
julday_seasons[1,1:length(seq(myjulday(5,1,2018),myjulday(5,31,2018),1))]=seq(myjulday(5,1,2018),myjulday(5,31,2018),1)
#
# The time period of the day to average
#
perio_vec=c("DAILY")
hourbeg=c(01)
hourend=c(24)
#
perio_vec=c(perio_vec,"NOON")
hourbeg=c(hourbeg,03)
hourend=c(hourend,08)
#
perio_vec=c(perio_vec,"EVEN")
hourbeg=c(hourbeg,11)
hourend=c(hourend,13)
#
perio_vec=c(perio_vec,"NIGT")
hourbeg=c(hourbeg,17)
hourend=c(hourend,22)
#
# The height above ground in which the wind vectors shall be plotted
#
hagl_wivec=c(10,30,60,100)
#
# Latitude and longitude of zoom zones for wind fields
#
name_zoom=c("KOWLOON","HKI","SHATIN")
latsmin_zoom=c(022.2748,022.2479,022.3452)
latsmax_zoom=c(022.3520,022.3008,022.4387)
lonsmin_zoom=c(114.1270,114.1166,114.1387)
lonsmax_zoom=c(114.2194,114.2189,114.2614)
#
# ####################################################
# Create a vector with the segment names
# Number of nests must be below 10
# ####################################################
#
cbeg="MNHOU."
cint=".SEGM0."
#
segallnm=c()
for (nest in nest_vec) {
  cnes=sprintf("%01d",nest)
  segallnm=c(segallnm,paste(cbeg,cnes,cint,sep=""))
}
#
clvl0="_LVL_0_SPLIT_"
clvl1="_LVL_1_SPLIT_"
clvl2="_LVL_2_SPLIT_"
#
# Get maximum number of days per season
#
DIM=dim(julday_seasons)
nmaxday=DIM[2]
#
# ####################################################
# Dowload and plot results from PGD
# ####################################################
#
# Download PGD data from supercomputer
#
if (lnew==1) {
  #
  pout=paste(pdsup,name,"/",sep="")
  dir.create(pout,showWarnings=FALSE)
  #
  for (sce in 1:length(scen)) {
    #
    pout=paste(pdsup,name,"/",scen[sce],sep="")
    dir.create(pout,showWarnings=FALSE)
    #
    pout=paste(pdsup,name,"/",scen[sce],"/PGD",sep="")
    dir.create(pout,showWarnings=FALSE)
    #
    print("    ")
    print(paste("Downloading physiographic fields for scenario ",scen[sce]))
    #
    for (nest in nest_vec) {
      #
      print("                                                     ")
      print(paste("Downloading physiographic fields for nest",nest))
      #
      pout=paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,sep="")
      dir.create(pout,showWarnings=FALSE)
      #
      pathin=paste(pscra,padd,name,"/",scen[sce],"/PREP_PGD_NEST_",nest,"/DIAG/*",sep="")
      system(paste("scp schoetterr@",supcmp,":",pathin," ",pout," >> OUTBIN 2>> OUTBIN",sep=""))
      #
    }
    #
  }
} else {
  print("                                             ")
  print("CAUTION: old physiographic fields are used   ")
}
#
print("                                                    ")
print("Fields with physiographic data have been downloaded ")
print("                                                    ")
#
# Load fractions of SURFEX tiles (required for later conversions)
# Only the subdomain is used since this will also be used for the time
# dependent variables
#
# Get field dimensions for all nests
#
lenmask=c();lentown=c()
for (nest in nest_vec) {
  LST=myno_map(paste(pdsup,name,"/",scen[1],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_FRAC_TOWN.nc4",sep=""))
  lenmask=c(lenmask,length(LST$data_map))
  filein=paste(pdsup,name,"/",scen[1],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_BLDFRA.nc4",sep="")
  if (file.exists(filein)) {
    LST=myno_map(filein)
    lentown=c(lentown,length(LST$data_map))
  } else {
    lentown=c(lentown,0)
  }
}
#
count_nest=0
#
for (nest in nest_vec) {
  #
  count_nest=count_nest+1
  #
  for (sce in 1:length(scen)) {
  #
  # Allocation of fields at first call
  #
  if ((sce==1)&&(nest==nest_vec[1])) {
    #
    LONS = array(NA,c(max(lenmask),length(nest_vec),length(scen)))
    LATS = array(NA,c(max(lenmask),length(nest_vec),length(scen)))
    HEIG = array(NA,c(max(lenmask),length(nest_vec),length(scen)))
    #
    FRAC_TOWN   = array(NA ,c(max(lenmask),length(nest_vec),length(scen)))
    FRAC_SEA    = array(NA ,c(max(lenmask),length(nest_vec),length(scen)))
    FRAC_WATER  = array(NA ,c(max(lenmask),length(nest_vec),length(scen)))
    FRAC_NATURE = array(NA ,c(max(lenmask),length(nest_vec),length(scen)))
    FRAC_FOREST = array(0.0,c(max(lenmask),length(nest_vec),length(scen)))
    #
  }
  #
  # Creation of directory for constant fields
  #
  pconst=paste(pdsup,name,"/",scen[sce],"/PGD/RCONST_Nest_",nest,sep="")
  system(paste("mkdir -p",pconst))
  #
  LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_LON.nc4",sep=""))
  LONS[1:lenmask[count_nest],count_nest,sce]=LST$data_map
  #
  LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_LAT.nc4",sep=""))
  LATS[1:lenmask[count_nest],count_nest,sce]=LST$data_map
  #
  LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_ZS.nc4",sep=""))
  HEIG[1:lenmask[count_nest],count_nest,sce]=LST$data_map
  #
  LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_FRAC_TOWN.nc4",sep=""))
  FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]=LST$data_map
  #
  LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_FRAC_SEA.nc4",sep=""))
  FRAC_SEA[1:lenmask[count_nest],count_nest,sce]=LST$data_map
  #
  LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_FRAC_WATER.nc4",sep=""))
  FRAC_WATER[1:lenmask[count_nest],count_nest,sce]=LST$data_map
  #
  LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_FRAC_NATURE.nc4",sep=""))
  FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]=LST$data_map
  #
  ctl=FRAC_TOWN [1:lenmask[count_nest],count_nest,sce]+FRAC_SEA   [1:lenmask[count_nest],count_nest,sce]+
      FRAC_WATER[1:lenmask[count_nest],count_nest,sce]+FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]
  #
  if (max(abs(ctl-1.0)) > 1.0E-6) stop("Wrong fractions")
  #
  # Write longitude, latitude and elevation as R-fields
  #
  LST=list(vals=LONS[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/LONS_Nest_",nest,sep=""))
  #
  LST=list(vals=LATS[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/LATS_Nest_",nest,sep=""))
  #
  LST=list(vals=HEIG[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/HEIG_Nest_",nest,sep=""))
  #
  # Write the fractions as R-fields
  #
  LST=list(vals=FRAC_TOWN[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/FRAC_TOWN_Nest_",nest,sep=""))
  #
  LST=list(vals=FRAC_SEA[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/FRAC_SEA_Nest_",nest,sep=""))
  #
  LST=list(vals=FRAC_WATER[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/FRAC_WATER_Nest_",nest,sep=""))
  #
  LST=list(vals=FRAC_NATURE[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/FRAC_NATURE_Nest_",nest,sep=""))
  #
  # Calculate the sum of the forest covers
  #
  cforest=0
  for (cover in c(seq(301,343,1),seq(10,33,1),seq(198,219,1))) {
    filein=paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_COVER",cover,".nc4",sep="")
    if (file.exists(filein)) {
      cforest=cforest+1
      LST=myno_map(filein)
      aux=LST$data_map
      FRAC_FOREST[1:lenmask[count_nest],count_nest,sce]=FRAC_FOREST[1:lenmask[count_nest],count_nest,sce]+aux
    }
  }
  #
  if (cforest==0) {
    print("   ")
    print("CAUTION: No forest covers found")
    print("This might be due to the use of ECOCLIMAP-I in Meso-NH")
    print("UTCI and TRAD in rural areas will be in sun only")
    # stop("")
  }
  #
  amin=min(FRAC_FOREST[1:lenmask[count_nest],count_nest,sce])
  amax=max(FRAC_FOREST[1:lenmask[count_nest],count_nest,sce])
  #
  if (amin<(-1.e-6))     stop("Wrong forest fraction")
  if (amax>(1.0+1.0e-6)) stop("Wrong forest fraction")
  #
  LST=list(vals=FRAC_FOREST[1:lenmask[count_nest],count_nest,sce])
  save(LST,file=paste(pconst,"/FRAC_FOREST_Nest_",nest,sep=""))
  #
  # Load building and garden fraction (required for later conversions)
  # These are only available when TEB is used
  #
  if (file.exists(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_BLDFRA.nc4",sep=""))) {
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_BLDFRA.nc4",sep=""))
    #
    # Allocation of fields at first call
    #
    if ((sce==1)&&(nest==nest_vec[1])) {
      #
      BLDFRA = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      VEGFRA = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      RODFRA = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      BLDHEI = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      WALLOH = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      BLDTYP = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      BLDUSE = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      BLDAGE = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      BLDTER = array(NA,c(max(lentown),length(nest_vec),length(scen)))
      #
    }
    #
    BLDFRA[1:lentown[count_nest],count_nest,sce]=LST$data_map
    BLDFRA[which(is.na(BLDFRA))]=0.0
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_GARDEN.nc4",sep=""))
    VEGFRA[1:lentown[count_nest],count_nest,sce]=LST$data_map
    VEGFRA[which(is.na(VEGFRA))]=1.0
    #
    # Diagnose road fraction
    #
    RODFRA[1:lentown[count_nest],count_nest,sce]=1.0-BLDFRA[1:lentown[count_nest],count_nest,sce]-VEGFRA[1:lentown[count_nest],count_nest,sce]
    #
    # Save the data
    #
    LST=list(vals=BLDFRA[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/BLDFRA_Nest_",nest,sep=""))
    #
    LST=list(vals=VEGFRA[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/VEGFRA_Nest_",nest,sep=""))
    #
    LST=list(vals=RODFRA[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/RODFRA_Nest_",nest,sep=""))
    #
    # Other diagnostics
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_BLD_HEIG.nc4",sep=""))
    height=LST$data_map
    BLDHEI[1:lentown[count_nest],count_nest,sce]=height
    BLDHEI[which(is.na(BLDHEI))]=0.0
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_WALL_O_H.nc4",sep=""))
    walloh=LST$data_map
    WALLOH[1:lentown[count_nest],count_nest,sce]=walloh
    WALLOH[which(is.na(WALLOH))]=0.0
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_BLDTYPE.nc4",sep=""))
    bldtyp=LST$data_map
    BLDTYP[1:lentown[count_nest],count_nest,sce]=bldtyp
    BLDTYP[which(is.na(BLDTYP))]=0
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_USETYPE.nc4",sep=""))
    usetyp=LST$data_map
    BLDUSE[1:lentown[count_nest],count_nest,sce]=usetyp
    BLDUSE[which(is.na(BLDUSE))]=0
    #
    # Combined information on building age
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_IBLDAGE.nc4",sep=""))
    indage=LST$data_map
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_CBLDAGE.nc4",sep=""))
    merage=LST$data_map
    merage[which(BLDUSE[1:lentown[count_nest],count_nest,sce]==5)]=indage[which(BLDUSE[1:lentown[count_nest],count_nest,sce]==5)]
    #
    BLDAGE[1:lentown[count_nest],count_nest,sce]=merage
    BLDAGE[which(is.na(BLDAGE))]=0
    #
    # Combined information on building construction materials territory
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_P1TERR.nc4",sep=""))
    terp1=LST$data_map
    #
    LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_D_PXTERR.nc4",sep=""))
    merter=LST$data_map
    merter[which(BLDAGE[1:lentown[count_nest],count_nest,sce]==1)]=terp1[which(BLDAGE[1:lentown[count_nest],count_nest,sce]==1)]
    #
    BLDTER[1:lentown[count_nest],count_nest,sce]=merter
    BLDTER[which(is.na(BLDTER))]=0
    #
    LST=list(vals=BLDHEI[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/BLDHEIGHT_Nest_",nest,sep=""))
    #
    LST=list(vals=WALLOH[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/WALL_O_HOR_Nest_",nest,sep=""))
    #
    LST=list(vals=BLDTYP[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/BUILD_TYPE_Nest_",nest,sep=""))
    #
    LST=list(vals=BLDUSE[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/BUILD_USE_Nest_",nest,sep=""))
    #
    LST=list(vals=BLDAGE[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/BUILD_AGE_Nest_",nest,sep=""))
    #
    LST=list(vals=BLDTER[1:lentown[count_nest],count_nest,sce])
    save(LST,file=paste(pconst,"/BUILD_TER_Nest_",nest,sep=""))
    #
  }
}
}
#
# ################################################
# Plot PGD data for each scenario
# ################################################
#
plotbase=paste(pmaps,name,sep="")
system(paste("rm -Rf",plotbase))
dir.create(plotbase,showWarnings=FALSE)
#
for (sce in 1:length(scen)) {
  #
  plotdir=paste(pmaps,name,"/",scen[sce],sep="")
  dir.create(plotdir,showWarnings=FALSE)
  #
  count_nest=0
  #
  for (nest in nest_vec) {
    #
    count_nest=count_nest+1
    #
    plotdir=paste(pmaps,name,"/",scen[sce],"/PREP_PGD_NEST_",nest,sep="")
    dir.create(plotdir,showWarnings=FALSE)
    #
    for (var in 1:length(pgdsupvec)) {
      #
      filein=paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_",pgdsupvec[var],".nc4",sep="")
      #
      if (file.exists(filein)) {
        #
        LST=myno_map(filein)
        fieldplot=LST$data_map
        #
        # Multiply with TOWN fraction
        #
        if ( (pgdsupvec[var]=="D_BLDFRA")||(pgdsupvec[var]=="D_WALL_O_H")||(pgdsupvec[var]=="D_NB_POP") ) {
          #
          LST=myno_map(paste(pdsup,name,"/",scen[sce],"/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_FRAC_TOWN.nc4",sep=""))
          town=LST$data_map
          #
          if (length(town)!=length(fieldplot)) stop("Inconsistant length of entries")
          fieldplot=fieldplot*town
	  #
        }
	#
        nameplot=paste(pmaps,name,"/",scen[sce],"/PREP_PGD_NEST_",nest,"/",pgdsupvec[var],".pdf",sep="")
        #
        seplotonline(LONS[1:lenmask[count_nest],count_nest,1],LATS[1:lenmask[count_nest],count_nest,1],fieldplot,nameplot,
                     varname=pgdsupvec[var],nametitle=pgdtitlevec[var],flaginc=0)
        #
      }
    }
  }
}
#
# ####################################################
# Prepare CAPITOUL station observations
# ####################################################
#
if (flag_capitoul>0.5) {
  #
  print("     ")
  print("Model results will be evaluated against the CAPITOUL surface stations")
  print("     ")
  #
  # SURF network: Only temperature and relative humidity
  #
  filesave="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/SURF_STATIONS/RDATA/SAVE_CAPITOUL_SURFSTATIONS"
  if (!file.exists(filesave)) {
    seload_capsurf_stations("/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/SURF_STATIONS/DATA/",filesave)
  }
  #
  load(file=filesave)
  lats_surfstat=LST$lats;lons_surfstat=LST$lons;name_surfstat=LST$name;numb_surfstat=LST$numb;heig_surfstat=LST$heig
  juld_surfstat=LST$juld_fullts[1,];temp_surfstat=LST$temp_fullts
  relh_surfstat=LST$relh_fullts;qspe_surfstat=LST$qspe_fullts
  surf_zs=6.0
  #
  # Correction for dailight saving time
  #
  LST=mycaldat(juld_surfstat)
  mont_surfstat=LST$m;dayy_surfstat=LST$d;year_surfstat=LST$y
  #
  ind_nodailight=which((mont_surfstat>=11)|(mont_surfstat<=3))
  ind_isdailight=which((mont_surfstat< 11)&(mont_surfstat> 3))
  #
  juld_surfstat[ind_nodailight]=juld_surfstat[ind_nodailight]+(2.0/24.0)
  juld_surfstat[ind_isdailight]=juld_surfstat[ind_isdailight]+(1.0/24.0)
  #
  # Correction for MNH one-hour averages
  #
  juld_surfstat=juld_surfstat-(0.5/24)
  #
  # CNRM station network: More variables
  #
  filesave="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/SURF_STATIONS/RDATA/SAVE_CAPITOUL_CNRMSTATIONS"
  if (!file.exists(filesave)) {
    seload_capcnrm_stations("/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/MTO_CNRM/",filesave)
  }
  #
  load(file=filesave)
  lats_cnrmstat=LST$lats;lons_cnrmstat=LST$lons;heig_cnrmstat=LST$heig;name_cnrmstat=LST$name;numb_cnrmstat=LST$numb
  juld_cnrmstat=LST$juld_fullts[1,];temp_cnrmstat=LST$temp_fullts;relh_cnrmstat=LST$relh_fullts
  pabs_cnrmstat=LST$pabs_fullts;rrat_cnrmstat=LST$rrat_fullts;widd_cnrmstat=LST$widd_fullts
  wiff_cnrmstat=LST$wiff_fullts;glob_cnrmstat=LST$frvd_fullts;qspe_cnrmstat=LST$qspe_fullts
  lwdw_cnrmstat=LST$fird_fullts
  cnrm_zs=2.0 
  #
  # Routine observation station at Blagnac airport
  #
  filesave="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/ROD_Blagnac/SAVE_ROD_BLAGNAC"
  if (!file.exists(filesave)) {
    juldayrodbeg=myjulday(03,01,2004)
    juldayrodend=myjulday(02,28,2005)
    seload_rod_station("/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/ROD_Blagnac/",
      filesave,juldayrodbeg,juldayrodend,"Blagnac",22)
  }
  #
  load(file=filesave)
  lats_rodstat=LST$lats;lons_rodstat=LST$lons;heig_rodstat=LST$heig;name_rodstat=LST$name;numb_rodstat=LST$numb
  juld_rodstat=LST$juld_fullts;temp_rodstat=LST$temp_fullts;relh_rodstat=LST$relh_fullts
  pabs_rodstat=LST$pabs_fullts;rrat_rodstat=LST$rrat_fullts;widd_rodstat=LST$widd_fullts
  wiff_rodstat=LST$wiff_fullts;qspe_rodstat=LST$qspe_fullts;glob_rodstat=LST$glob_fullts
  #
  # Merge the routine observation data with the cnrm data
  #
  lats_cnrmstat=c(lats_cnrmstat,lats_rodstat)
  lons_cnrmstat=c(lons_cnrmstat,lons_rodstat)
  heig_cnrmstat=c(heig_cnrmstat,heig_rodstat)
  name_cnrmstat=c(name_cnrmstat,name_rodstat)
  numb_cnrmstat=c(numb_cnrmstat,numb_rodstat)
  #
  # Check for consistant time series
  #
  if (length(juld_cnrmstat)!=length(juld_rodstat)) {
    print(length(juld_cnrmstat))
    print(length(juld_rodstat))
    stop("Inconsistant length of time series")
  }
  #
  diff=juld_cnrmstat-juld_rodstat
  if (max(abs(diff))>0) stop("Difference in time series")
  #
  temp_cnrmstat=rbind(temp_cnrmstat,temp_rodstat)
  relh_cnrmstat=rbind(relh_cnrmstat,relh_rodstat)
  qspe_cnrmstat=rbind(qspe_cnrmstat,qspe_rodstat)
  pabs_cnrmstat=rbind(pabs_cnrmstat,pabs_rodstat)
  rrat_cnrmstat=rbind(rrat_cnrmstat,rrat_rodstat)
  widd_cnrmstat=rbind(widd_cnrmstat,widd_rodstat)
  wiff_cnrmstat=rbind(wiff_cnrmstat,wiff_rodstat)
  glob_cnrmstat=rbind(glob_cnrmstat,glob_rodstat)
  lwdw_cnrmstat=rbind(lwdw_cnrmstat,array(NA,length(glob_rodstat)))
  #
  jdbeg_ms=NA
  jdend_ms=NA
  #
} else if (flag_dijonobs>0.5) {
  #
  print("     ")
  print("Model results will be evaluated against the Dijon surface stations")
  print("     ")
  #
  # The station height is 3 m above ground, the values are
  # instantanouus once per hour until 2016 and then based on 2-3
  # values per hour
  #
  # SURF network: Only temperature and relative humidity
  #
  filesave="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/SURF_STATIONS/RDATA/SAVE_DIJON_SURFSTATIONS"
  if (!file.exists(filesave)) {
    seload_dijsurf_stations("/home/schoetterr/Bureau/SURFEX/DIJON_OBS/",filesave)
  }
  #
  load(file=filesave)
  lats_surfstat=LST$lats;lons_surfstat=LST$lons;name_surfstat=LST$name;numb_surfstat=LST$numb;heig_surfstat=LST$heig
  juld_surfstat=LST$juld_fullts;temp_surfstat=LST$temp_fullts
  relh_surfstat=LST$relh_fullts;qspe_surfstat=LST$qspe_fullts
  surf_zs=3.0
  #
  # For the SURF network, observations are shifted 30 min backward,
  # since model output is averaged over the last 30 min.
  #
  juld_surfstat=juld_surfstat + (0.6/24)
  #
  # Routine observation station at Longvic airport
  #
  filesave="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/ROD_Longvic/SAVE_ROD_Longvic"
  if (!file.exists(filesave)) {
    juldayrodbeg=myjulday(06,01,2014)
    juldayrodend=myjulday(12,31,2016)
    seload_rod_station("/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/ROD_Longvic/",filesave,juldayrodbeg,juldayrodend,"Longvic",0)
  }
  #
  load(file=filesave)
  lats_rodstat=LST$lats;lons_rodstat=LST$lons;heig_rodstat=LST$heig;name_rodstat=LST$name;numb_rodstat=LST$numb
  juld_rodstat=LST$juld_fullts;temp_rodstat=LST$temp_fullts;relh_rodstat=LST$relh_fullts
  pabs_rodstat=LST$pabs_fullts;rrat_rodstat=LST$rrat_fullts;widd_rodstat=LST$widd_fullts
  wiff_rodstat=LST$wiff_fullts;qspe_rodstat=LST$qspe_fullts;glob_rodstat=LST$glob_fullts
  #
  # Airport data are written on /cnrm fields, since no other observations are available
  #
  cnrm_zs=2.0
  #
  juld_cnrmstat=juld_rodstat # -(1.0/24)
  #
  lats_cnrmstat=lats_rodstat
  lons_cnrmstat=lons_rodstat
  heig_cnrmstat=heig_rodstat
  name_cnrmstat=name_rodstat
  numb_cnrmstat=numb_rodstat
  #
  temp_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  relh_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  qspe_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  pabs_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  rrat_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  widd_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  wiff_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  glob_cnrmstat=array(NA,c(1,length(temp_rodstat)))
  #
  temp_cnrmstat[1,]=temp_rodstat
  relh_cnrmstat[1,]=relh_rodstat
  qspe_cnrmstat[1,]=qspe_rodstat
  pabs_cnrmstat[1,]=pabs_rodstat
  rrat_cnrmstat[1,]=rrat_rodstat
  widd_cnrmstat[1,]=widd_rodstat
  wiff_cnrmstat[1,]=wiff_rodstat
  glob_cnrmstat[1,]=glob_rodstat
  lwdw_cnrmstat=array(NA,c(1,length(glob_rodstat)))
  #
  # At the moment no weather types: set to NA
  #
  jdbeg_ms=NA
  jdend_ms=NA
  #
} else if (flag_hkongobs>0.5) {
  #
  print("                      ")
  print("Model results will be evaluated against the Hong Kong Obervatory stations")
  print("                      ")
  #
  filesave="../../../PROJETS/PROCORE/MetStations/RSAVE_HKO_Stations"
  if (!file.exists(filesave)) {
    seload_hko_stations("../../../PROJETS/PROCORE/MetStations/Raw_upto_20181231/",filesave)
  }
  #
  load(file=filesave)
  numb_cnrmstat=LST$numb_hko;lats_cnrmstat=LST$lats_hko;lons_cnrmstat=LST$lons_hko;heig_cnrmstat=LST$heig_hko
  hiff_cnrmstat=LST$hiff_hko;hite_cnrmstat=LST$hite_hko;town_cnrmstat=LST$town_hko;natu_cnrmstat=LST$natu_hko
  watr_cnrmstat=LST$watr_hko;seaf_cnrmstat=LST$seaf_hko;roof_cnrmstat=LST$roof_hko;juld_cnrmstat=LST$juld_fullts
  widd_cnrmstat=LST$widd_fullts;wiff_cnrmstat=LST$wiff_fullts;temp_cnrmstat=LST$temp_fullts
  relh_cnrmstat=LST$relh_fullts;qspe_cnrmstat=LST$qspe_fullts;pabs_cnrmstat=LST$mslp_fullts
  rf_cnrmstat=LST$rf_fullts;glob_cnrmstat=LST$radg_fullts;rrat_cnrmstat=LST$rrat_fullts
  name_cnrmstat=LST$name_hko
  #
  # Conversion between UTC and Hong Kong time
  #
  juld_cnrmstat = juld_cnrmstat - 8.0/24.0
  #
  # No SURF stations
  #
  lats_surfstat=NA
  lons_surfstat=NA
  name_surfstat=NA
  numb_surfstat=NA
  heig_surfstat=NA
  #
  # At the moment no weather types: set to NA
  #
  jdbeg_ms=NA
  jdend_ms=NA
  #
} else {
  #
  print("                                               ")
  print("No evaluation against station data will be made")
  print("                                               ")
  #
  lats_surfstat=NA
  lons_surfstat=NA
  name_surfstat=NA
  numb_surfstat=NA
  heig_surfstat=NA
  #
  lats_cnrmstat=NA
  lons_cnrmstat=NA
  name_cnrmstat=NA
  numb_cnrmstat=NA
  heig_cnrmstat=NA
  #
  print("                                               ")
  print("No weather type dependend analysis will be made")
  print("                                               ")
  #
  jdbeg_ms=NA
  jdend_ms=NA
  #
}
#
# Create a plot with BLDFRA (as typical PGDPLOT), but with station network
#
# varlist_stat=c("ZS","D_BLDFRA")
# titlist_stat=c("Elevation [m]","Plane area building density [m²/m²]")
#
# for (nest in nest_vec) for (vari in 1:length(varlist_stat)) {
#
#   plotdir=paste(pmaps,name,"/REFER/PREP_PGD_NEST_",nest,sep="")
#   filein =paste(pdsup,name,"/REFER/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_",varlist_stat[vari],".nc4",sep="")
#   filepl =paste(pmaps,name,"/REFER/PREP_PGD_NEST_",nest,"/",varlist_stat[vari],"_STATIONS.pdf",sep="")
#
#   if (file.exists(filein)) {
#
#     LST=myno_map(filein)
#     fieldplot=LST$data_map
#
# Multiply with TOWN fraction
#
#     if ( (varlist_stat[vari]=="D_BLDFRA")||(varlist_stat[vari]=="D_WALL_O_H")||(varlist_stat[vari]=="D_NB_POP") ) {
#       #
#       LST=myno_map(paste(pdsup,name,"/REFER/PGD/PREP_PGD_NEST_",nest,"/PGD_Nest_",nest,"_FRAC_TOWN.nc4",sep=""))
#       town=LST$data_map
#
#      if (length(town)!=length(fieldplot)) stop("Inconsistant length of entries")
#       fieldplot=fieldplot*town
#
#     }
#
#     seplotonline(LONS[1:lenmask[count_nest],count_nest,1],LATS[1:lenmask[count_nest],count_nest,1],fieldplot,filepl,
#                  varname=varlist_stat[vari],nametitle=titlist_stat[vari],flaginc=0,fzoom=2.5,lplotstation=1,
#                 lats_stat_1=lats_surfstat,lons_stat_1=lons_surfstat,numb_stat_1=numb_surfstat,
#                  lats_stat_2=lats_cnrmstat,lons_stat_2=lons_cnrmstat,numb_stat_2=numb_cnrmstat)
#
#  }
#  #
# }
#
# ####################################################
# Prepare Meso-NH output
# ####################################################
#
# Switch of variables to be evaluated for Hong Kong
#
if (grepl("HK",name)) {
  print("  ")
  print("Change names of variables to be evaluated for Hong Kong")
  print("  ")
  varieval_capitoul_cnrmstat = varieval_hkostat
  nameeval_capitoul_cnrmstat = nameeval_hkostat
  uniteval_capitoul_cnrmstat = uniteval_hkostat
}
#
if (lloop>0.5) {
  nrunday_loop=nrunday
} else {
  nrunday_loop=1
}
#
if (nrunday>0.5) {
#
pathbase=paste(pdsup,name,"/",sep="")
system(paste("mkdir -p",pathbase))
#
pathbaseurbinc=paste(pdsup,name,"/URBINC/",sep="")
system(paste("mkdir -p",pathbaseurbinc))
#
juldaybeg=myjulday(montini,dayyini,yearini)
#
prsv_first=array(NA,c(length(scen),length(nest_vec)))
julobs=c()
#
for (day in 1:nrunday_loop) {
  #
  # Loop over scenarios and nests
  #
  count_nest=0
  #
  for (nest in nest_vec) {
  #
  count_nest=count_nest+1
  #
  for (sce in 1:length(scen)) {
  #
  # Choose the list of variables to download depending on the scenario
  #
  if (scen[sce]=="REFER") {
    varlistonline=varlistonline_REFER
  } else if (scen[sce]=="NOURB") {
    varlistonline=varlistonline_NOURB
  } else {
    print(scen[sce])
    stop("No rule implemented for this scenario")
  }
  #
  # The output path
  #
  pout=paste(pdsup,name,"/",scen[sce],"/ONLINE",sep="")
  system(paste("mkdir -p",pout))
  #
  # #################################################################
  # Download the results from supercomputer
  # #################################################################
  #
  juldayact=juldaybeg+(day-1)
  LST=mycaldat(juldayact)
  montact=LST$m;dayyact=LST$d;yearact=LST$y
  #
  namemont=sprintf("%02d",montact)
  namedayy=sprintf("%02d",dayyact)
  nameday=paste(yearact,namemont,namedayy,sep="")
  #
  print(paste("Preparing results for ",scen[sce],"-",nameday, " Nest ",nest,sep=""))
  #
  if ((sce==1)&&(nest==nest_vec[1])) {
    if ((day>=ndaybeg_ts)&&(day<=ndayend_ts)) {
      julobs=c(julobs,juldayact)
    }
  }
  #
  pout=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/",sep="")
  pathurbinc=paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/",sep="")
  #
  if (lnew==1) {
    #
    system(paste("rm -Rf ",pout,sep=""))
    system(paste("mkdir ",pout,sep=""))
    #
    system(paste("rm -Rf ",pathurbinc))
    system(paste("mkdir ",pathurbinc,sep=""))
    #
  } else {
    system(paste("mkdir -p",pout))
    system(paste("mkdir -p",pathurbinc))
  }
  #
  prsv=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/RDATA/",sep="")
  #
  if (day==1) prsv_first[sce,count_nest]=prsv
  #
  if (lnew==1) {
    system(paste("rm -Rf",prsv))
    system(paste("mkdir",prsv))
  } else {
    system(paste("mkdir -p",prsv))
  }
  #
  # The paths for the simulations
  #
  pathin_supc=paste(pscra,padd,name,"/",scen[sce],"/RUNDAY_",nameday,"/DIAG/",sep="")
  pathin_cnrm=paste(pdsup,name,"/",scen[sce],"/ONLINE/RUNDAY_",nameday,"/DIAG/",sep="")
  #
  # Create cnrm folders if the dont already exist
  #
  system(paste("mkdir -p ",pdsup,name,"/",scen[sce],"/ONLINE/RUNDAY_",nameday,"/",sep=""))
  system(paste("mkdir -p ",pdsup,name,"/",scen[sce],"/ONLINE/RUNDAY_",nameday,"/DIAG/",sep=""))
  #
  # Only at first day: treatment of the canopy geometry
  #
  if (day==1) {
    #
    # Assign the canyon geometry variable list depending on scenario
    #
    if      (scen[sce]=="REFER") {
      varlist_geocanopy = varlist_geocanopy_REFER
    } else if (scen[sce]=="NOURB") {
      varlist_geocanopy = varlist_geocanopy_NOURB
    } else {
      print(scen[sce])
      stop("No rule implemented for this scenario")
    }
    #
    # Read the canopy geometry files into R (only one time step)
    #
    for (vari in 1:length(varlist_geocanopy)) {
      if (!file.exists(paste(prsv,varlist_geocanopy[vari],sep=""))) {
	pathin=paste(pscra,padd,name,"/",scen[sce],"/RUNDAY_",nameday,"/DIAG/*.",nest,".*",varlist_geocanopy[vari],"*",sep="")
        system(paste("scp schoetterr@",supcmp,":",pathin," ",pathin_cnrm," >> OUTBIN 2>> OUTBIN",sep=""))
        filetest=paste(pathin_cnrm,"/",segallnm[count_nest],"001",clvl0,varlist_geocanopy[vari],".nc4",sep="")
        if (file.exists(filetest)) {
    	   LST=myno_map(filetest)
           filesave=paste(prsv,varlist_geocanopy[vari],sep="")
    	   save(LST,file=filesave)
    	}
	# system(paste("rm -f ",pathin_cnrm,"/*",varlist_geocanopy[vari],"*",sep=""),wait=FALSE)
      }
    }
    #
    # Determine the canopy level closest to the observations
    #
    if (!is.na(lats_surfstat[1])) {
      #
      if (!file.exists(paste(prsv_first[sce,count_nest],"STATION_LEVELS",sep=""))) {
        #
        for (lvl in 1:6) {
          #
          filetest=paste(prsv,"TEB_CAN_Z0",lvl,sep="")
          if (file.exists(filetest)) {
            load(file=paste(prsv,"TEB_CAN_Z0",lvl,sep=""))
            teb_lvl=LST$data_map
          } else {
            teb_lvl=NA
          }
          #
          filetest=paste(prsv,"ISBA_CAN_Z0",lvl,sep="")
          if (file.exists(filetest)) {
            load(file=paste(prsv,"ISBA_CAN_Z0",lvl,sep=""))
            isb_lvl=LST$data_map
          } else {
            isb_lvl=NA
          }
          #
          if (lvl==1) {
            teb_can_levels=array(NA,c(length(teb_lvl),6))
            isb_can_levels=array(NA,c(length(isb_lvl),6))
          }
          #
          teb_can_levels[,lvl]=teb_lvl
          isb_can_levels[,lvl]=isb_lvl
          #
        }
        #
        # Determine the canopy level closest to the observations
        #
        temp_lvl_teb=array(NA,length(numb_surfstat))
        temp_lvl_isb=array(NA,length(numb_surfstat))
        wind_lvl_teb=array(NA,length(numb_surfstat))
        wind_lvl_isb=array(NA,length(numb_surfstat))
        #
	print("                            ")
        print("Before attribution of levels")
	print(dim(teb_can_levels))
	print(summary(c(teb_can_levels)))
        print("                            ")
	print(dim(isb_can_levels))
	print(summary(c(isb_can_levels)))
	#
	print(length(temp_lvl_teb))
        print(length(wind_lvl_teb))
	print(length(temp_lvl_isb))
	print(length(wind_lvl_isb))
        #
        for (stat in 1:length(numb_surfstat)) {
	  #
          print(" ")
	  print("Find canopy levels of station")
	  print(name_surfstat[stat])
	  print(paste("Height of wind anemometer :",hiff_surfstat[stat]))
	  print(paste("Height of temperature anemometer :",hite_surfstat[stat]))
	  print(paste("Flag for rooftop station :",roof_surfstat[stat]))
          #
          y_stat=array(NA,c(1,2))
          y_stat[1,1] = lons_surfstat[stat]
          y_stat[1,2] = lats_surfstat[stat]
          #
          x_model=array(NA,c(length(LONS[1:lenmask[count_nest],count_nest,sce]),2))
          x_model[,1] = LONS[1:lenmask[count_nest],count_nest,sce]
          x_model[,2] = LATS[1:lenmask[count_nest],count_nest,sce]
          #
          dist_stat = distm(x_model,y_stat,fun=distHaversine)
          amin=min(dist_stat)
          print(paste("The minimum distance to station",name_surfstat[stat],"is",format(amin/1000.0,digits=2),"km"))
          print(paste("Longitude:",lons_surfstat[stat]))
          print(paste("Latitude :",lats_surfstat[stat]))
          #
          ind_min=which(dist_stat==amin)
          #
          aux_town = FRAC_TOWN   [1:lenmask[count_nest],count_nest,sce]
          aux_natr = FRAC_NATURE [1:lenmask[count_nest],count_nest,sce]
          aux_seaf = FRAC_SEA    [1:lenmask[count_nest],count_nest,sce]
          aux_watr = FRAC_WATER  [1:lenmask[count_nest],count_nest,sce]	  
          #
          print(paste("TOWN",aux_town[ind_min]))
          print(paste("NATU",aux_natr[ind_min]))
          print(paste("SEAF",aux_seaf[ind_min]))
          print(paste("WATR",aux_watr[ind_min]))
          #
	  # In the case of rooftop station, the highest canopy level is selected
	  #
          DIM_LVL_TEB=dim(teb_can_levels)
          DIM_LVL_ISB=dim(isb_can_levels)	  
          #
          if ( roof_surfstat[stat]>0.5) {
	    #
            wind_lvl_teb[stat]=DIM_LVL_TEB[2]
            temp_lvl_teb[stat]=DIM_LVL_TEB[2]
            wind_lvl_isb[stat]=DIM_LVL_ISB[2]
            temp_lvl_isb[stat]=DIM_LVL_ISB[2]
	    #
          } else {
            #
	    # Otherwise the closest level to the station height is taken
	    #
            diff_teb_wind=abs(teb_can_levels[ind_min,]-hiff_surfstat[stat])
            diff_teb_temp=abs(teb_can_levels[ind_min,]-hite_surfstat[stat])
	    diff_isb_wind=abs(isb_can_levels[ind_min,]-hiff_surfstat[stat])
            diff_isb_temp=abs(isb_can_levels[ind_min,]-hite_surfstat[stat])
            #
            ind=which(diff_teb_wind==min(diff_teb_wind))
            wind_lvl_teb[stat]=ind[1]
            #
            ind=which(diff_teb_temp==min(diff_teb_temp))
            temp_lvl_teb[stat]=ind[1]
            #
            ind=which(diff_isb_wind==min(diff_isb_wind))
            wind_lvl_isb[stat]=ind[1]
            #
            ind=which(diff_isb_temp==min(diff_isb_temp))
            temp_lvl_isb[stat]=ind[1]
            #
	  }
          #
       	  print("The selected levels")
	  print(paste("Wind-TEB:",wind_lvl_teb[stat]))
 	  print(paste("Temp-TEB:",temp_lvl_teb[stat]))
	  print(paste("Wind-ISB:",wind_lvl_isb[stat]))
 	  print(paste("Temp-ISB:",temp_lvl_isb[stat]))
          #
       	  print("The selected level heights")
	  print(paste("Wind-TEB:",teb_can_levels[ind_min,wind_lvl_teb[stat]]))
 	  print(paste("Temp-TEB:",teb_can_levels[ind_min,temp_lvl_teb[stat]]))
	  print(paste("Wind-ISB:",isb_can_levels[ind_min,wind_lvl_isb[stat]]))
 	  print(paste("Temp-ISB:",isb_can_levels[ind_min,temp_lvl_isb[stat]]))
          #
        }
        #
        LST=list(wind_lvl_teb=wind_lvl_teb,temp_lvl_teb=temp_lvl_teb,
	         wind_lvl_isb=wind_lvl_isb,temp_lvl_isb=temp_lvl_isb)
        #
        save(LST,file=paste(prsv_first[sce,count_nest],"STATION_LEVELS",sep=""))
        #
      }
      #
    }
  }
  #
  # Load the station levels
  #
  if (!is.na(lats_surfstat[1])) {
    #
    load(file=paste(prsv_first[sce,count_nest],"STATION_LEVELS",sep=""))
    wind_lvl_teb=LST$wind_lvl_teb;temp_lvl_teb=LST$temp_lvl_teb
    wind_lvl_isb=LST$wind_lvl_isb;temp_lvl_isb=LST$temp_lvl_isb
    #
    # The required variables from the canopy are
    # merged with the varlistonline vector
    #
    if (scen[sce]!="NOURB") {
      for (tt in unique(c(temp_lvl_teb[which(!is.na(temp_lvl_teb))]))) {
        varlistonline=c(varlistonline,paste("TEB_CAN_TM0",tt,sep=""))
        varlistonline=c(varlistonline,paste("TEB_CAN_QM0",tt,sep=""))
        varlistonline=c(varlistonline,paste("TEB_CAN_RM0",tt,sep=""))
        varlistonline=c(varlistonline,paste("TEB_CAN_PM0",tt,sep=""))
      }
    }
    #
    for (tt in unique(c(temp_lvl_isb[which(!is.na(temp_lvl_isb))]))) {
      varlistonline=c(varlistonline,paste("ISB_CAN_TM0",tt,sep=""))
      varlistonline=c(varlistonline,paste("ISB_CAN_QM0",tt,sep=""))
      varlistonline=c(varlistonline,paste("ISB_CAN_RM0",tt,sep=""))
      varlistonline=c(varlistonline,paste("ISB_CAN_PM0",tt,sep=""))
    }
    #
  }
  #
  # The wind on canopy levels is treated in any case
  #
  for (tt in 1:6) {
    if (scen[sce]!="NOURB") varlistonline=c(varlistonline,paste("TEB_CAN_UM0",tt,sep=""))
    varlistonline=c(varlistonline,paste("ISB_CAN_UM0",tt,sep=""))
  }
  #
  # #########################################################################
  # Read the data into R 
  # ##########################################################################
  #
  # Read as many levels than heights given in zlvl
  # for TEMPMME,RHME,QSPECME,PABSMME,WIDDME,WIFFME,UMME,VMME
  #
  if (length(zlvl)>0) {
    #
    varlistonline = varlistonline[!varlistonline %in% "TEMPMME"]
    varlistonline = varlistonline[!varlistonline %in% "RHME"]
    varlistonline = varlistonline[!varlistonline %in% "QSPECME"]
    varlistonline = varlistonline[!varlistonline %in% "PABSMME"]
    varlistonline = varlistonline[!varlistonline %in% "WIFFME"]
    varlistonline = varlistonline[!varlistonline %in% "WIDDME"]
    varlistonline = varlistonline[!varlistonline %in% "UMME"]
    varlistonline = varlistonline[!varlistonline %in% "VMME"]
    #
    for (lvl in 1:length(zlvl)) {
      #
      clvlnn=paste("_LVL_",lvl+1,"_SPLIT_",sep="")
      #
      varlistonline=c(varlistonline,paste(clvlnn,"TEMPMME",sep=""))
      varlistonline=c(varlistonline,paste(clvlnn,"RHME"   ,sep=""))    
      varlistonline=c(varlistonline,paste(clvlnn,"QSPECME",sep=""))
      varlistonline=c(varlistonline,paste(clvlnn,"PABSMME",sep=""))
      varlistonline=c(varlistonline,paste(clvlnn,"WIFFME" ,sep=""))
      varlistonline=c(varlistonline,paste(clvlnn,"WIDDME" ,sep=""))
      varlistonline=c(varlistonline,paste(clvlnn,"UMME" ,sep=""))
      varlistonline=c(varlistonline,paste(clvlnn,"VMME" ,sep=""))
      #
    }
    #
  }
  #
  for (vari in 1:length(varlistonline)) {
    if (!file.exists(paste(prsv,varlistonline[vari],sep=""))) {
      #
      if ( (varlistonline[vari]=="DIRFLASWD_SUM") || (varlistonline[vari]=="SCAFLASWD_SUM")||
           (varlistonline[vari]=="TOTFLASWD_SUM")) {
        clvl=clvl2
	ncvari=1
      } else if ( grepl("UMME",varlistonline[vari])    || grepl("VMME",varlistonline[vari])   ||
                  grepl("TEMPMME",varlistonline[vari]) || grepl("QSPECME",varlistonline[vari])||
                  grepl("RHME",varlistonline[vari])    || grepl("WIDDME",varlistonline[vari]) ||
                  grepl("WIFFME",varlistonline[vari])  || grepl("PABSMME",varlistonline[vari]) ) {
        clvl=""
        ncvari=1
      } else {
        clvl=clvl0
        ncvari=0
      }
      #
      if (  grepl("TEMPMME",varlistonline[vari])|| grepl("TWALL",varlistonline[vari])||
            grepl("TROOF",varlistonline[vari])  || grepl("TROAD",varlistonline[vari])||
	    grepl("TRADM",varlistonline[vari])  || grepl("T2M",varlistonline[vari])||
	    grepl("CAN_TM",varlistonline[vari]) || grepl("TIBLD",varlistonline[vari]) ) {
        addc=-XTT
	mulc=1.0
      } else if (grepl("HU2M",varlistonline[vari])) {
        addc=0.0
	mulc=100.0
      } else if (grepl("Q2M",varlistonline[vari])||grepl("QSPEC",varlistonline[vari])) {
        addc=0.0
	mulc=1000.0
      } else {
        addc=0.0
	mulc=1.0
      }
      #
      # Download
      #
      pathin=paste(pscra,padd,name,"/",scen[sce],"/RUNDAY_",nameday,"/DIAG/*.",nest,".*",clvl,"*",varlistonline[vari],"*",sep="")
      system(paste("scp schoetterr@",supcmp,":",pathin," ",pathin_cnrm," >> OUTBIN 2>> OUTBIN",sep=""))
      #
      filetest=paste(pathin_cnrm,"/",segallnm[count_nest],"001",clvl,varlistonline[vari],".nc4",sep="")
      #
      if (file.exists(filetest)) {
        #
	for (ttt in 1:24) {
	  #
	  filein=paste(pathin_cnrm,"/",segallnm[count_nest],sprintf("%03d",ttt),clvl,varlistonline[vari],".nc4",sep="")
          LST=myno_map(filein,ncvari)
	  datahour=LST$data_map
	  #
          if (ttt==1) vals_day=array(NA,c(length(datahour),24))
          vals_day[,ttt]=mulc*(datahour+addc)
	  #
	}
        #
        filesave=paste(prsv,varlistonline[vari],sep="")
        LST=list(vals_day=vals_day)
        save(LST,file=filesave)
	#
	# system(paste("rm -f ",pathin_cnrm,"/*",varlistonline[vari],"*",sep=""),wait=FALSE)
	#
      } else {
        # print(" ")
	# print("CAUTION: file not found")
	# print(filetest)
      }
      #
    }
  }
  #
  # ##############################################################
  # Calculations related to UTCI and TRAD
  # ##############################################################
  #
  # Calculate the weighted average TEB UTCI and TRAD in the street canyon 
  #
  if (max(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
    #
    if ((!file.exists(paste(prsv,"UTCI_CANYON",sep="")))||(!file.exists(paste(prsv,"TRAD_CANYON",sep="")))) {
      #
      load(file=paste(prsv,"UTCIMSU_TEB",sep=""));utci_sun=LST$vals_day
      load(file=paste(prsv,"UTCIMSH_TEB",sep=""));utci_sha=LST$vals_day
      load(file=paste(prsv,"TRADMSU_TEB",sep=""));trad_sun=LST$vals_day
      load(file=paste(prsv,"TRADMSH_TEB",sep=""));trad_sha=LST$vals_day
      load(file=paste(prsv,"ROAD_SHADE" ,sep=""));fsha=LST$vals_day
      #
      utci_canyon=fsha*utci_sha+(1.0-fsha)*utci_sun
      trad_canyon=fsha*trad_sha+(1.0-fsha)*trad_sun
      #
      LST=list(vals_day=utci_canyon)
      save(LST,file=paste(prsv,"UTCI_CANYON",sep=""))
      #
      LST=list(vals_day=trad_canyon)
      save(LST,file=paste(prsv,"TRAD_CANYON",sep=""))
      #
    }
  }
  #
  # Calculate the weighted average ISBA UTCI and TRAD 
  # over shaded and sunlit areas 
  #
  if (max(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
    #
    if ((!file.exists(paste(prsv,"UTCI_NATURE",sep="")))||(!file.exists(paste(prsv,"TRAD_NATURE",sep="")))) {
      #
      if (file.exists(paste(prsv,"UTCIMSU_ISBA",sep=""))) {
        load(file=paste(prsv,"UTCIMSU_ISBA",sep=""))
        utci_sun=LST$vals_day
      } else {
        utci_sun=array(NA,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"UTCIMSH_ISBA",sep=""))) {
        load(file=paste(prsv,"UTCIMSH_ISBA",sep=""))
        utci_sha=LST$vals_day
      } else {
        utci_sha=array(NA,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"TRADMSU_ISBA",sep=""))) {
        load(file=paste(prsv,"TRADMSU_ISBA",sep=""))
        trad_sun=LST$vals_day
      } else {
        trad_sun=array(NA,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"TRADMSH_ISBA",sep=""))) {
        load(file=paste(prsv,"TRADMSH_ISBA",sep=""))
        trad_sha=LST$vals_day
      } else {
        trad_sha=array(NA,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      utci_nature=FRAC_FOREST[1:lenmask[count_nest],count_nest,sce]*utci_sha+(1.0-FRAC_FOREST[1:lenmask[count_nest],count_nest,sce])*utci_sun
      trad_nature=FRAC_FOREST[1:lenmask[count_nest],count_nest,sce]*trad_sha+(1.0-FRAC_FOREST[1:lenmask[count_nest],count_nest,sce])*trad_sun
      #
      LST=list(vals_day=utci_nature)
      save(LST,file=paste(prsv,"UTCI_NATURE",sep=""))
      #
      LST=list(vals_day=trad_nature)
      save(LST,file=paste(prsv,"TRAD_NATURE",sep=""))
      #
    }
  }
  #
  # Calculate the aggregated UTCI over urban and rural areas
  #
  if ((!file.exists(paste(prsv,"UTCI_AGG",sep="")))||(!file.exists(paste(prsv,"TRAD_AGG",sep="")))) {
    #
    if (max(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
      #
      load(file=paste(prsv,"UTCI_CANYON",sep=""));utci_teb=LST$vals_day
      load(file=paste(prsv,"TRAD_CANYON",sep=""));trad_teb=LST$vals_day
      #
      utci_teb[which(is.na(utci_teb))]=0.0
      trad_teb[which(is.na(trad_teb))]=0.0
      #
    } else {
      utci_teb=array(0.0,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      trad_teb=array(0.0,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
    }
    #
    if (max(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
      #
      load(file=paste(prsv,"UTCI_NATURE",sep=""));utci_isba=LST$vals_day
      load(file=paste(prsv,"TRAD_NATURE",sep=""));trad_isba=LST$vals_day
      #
      utci_isba[which(is.na(utci_isba))]=0.0
      trad_isba[which(is.na(trad_isba))]=0.0
      #
    } else {
      utci_isba=array(0.0,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      trad_isba=array(0.0,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
    }
    #
    utci_agg_day=array(NA,dim(utci_isba))
    trad_agg_day=array(NA,dim(trad_isba))
    #
    for (tt in 1:length(nout_time)) {
      utci_agg_day[,tt] = (FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*utci_isba[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*utci_teb[,tt])/
                          (FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce])
      trad_agg_day[,tt] = (FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*trad_isba[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*trad_teb[,tt])/
                          (FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce])
    }
    #
    LST=list(vals_day=utci_agg_day)
    save(LST,file=paste(prsv,"UTCI_AGG",sep=""))
    #
    LST=list(vals_day=trad_agg_day)
    save(LST,file=paste(prsv,"TRAD_AGG",sep=""))
    #
  }
  #
  # ##############################################################
  # Aggregate fluxes and 2m-temperature over the 4 SURFEX tiles
  # ##############################################################
  #
  if (!file.exists(paste(prsv,"HAGG"    ,sep=""))||!file.exists(paste(prsv,"LAGG",sep=""))||
      !file.exists(paste(prsv,"RAGG"    ,sep=""))||!file.exists(paste(prsv,"TREAL_2M",sep=""))||
      !file.exists(paste(prsv,"RELHU_2M",sep=""))||!file.exists(paste(prsv,"QSPEC_2M",sep=""))) {
    #
    if (max(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
      #
      load(file=paste(prsv,"H_TEB",sep=""))
      hteb=LST$vals_day;hteb[which(is.na(hteb))]=-9999.9
      #
      load(file=paste(prsv,"LE_TEB",sep=""))
      lteb=LST$vals_day;lteb[which(is.na(lteb))]=-9999.9
      #
      load(file=paste(prsv,"RN_TEB",sep=""))
      rteb=LST$vals_day;rteb[which(is.na(rteb))]=-9999.9
      #
      load(file=paste(prsv,"T2MMEA_TEB",sep=""))
      t2m_teb=LST$vals_day;t2m_teb[which(is.na(t2m_teb))]=-9999.9
      #
      load(file=paste(prsv,"Q2MMEA_TEB",sep=""))
      q2m_teb=LST$vals_day;q2m_teb[which(is.na(q2m_teb))]=-9999.9
      #
      load(file=paste(prsv,"HU2MMEA_TEB",sep=""))
      r2m_teb=LST$vals_day;r2m_teb[which(is.na(r2m_teb))]=-9999.9
      #
    } else {
      hteb    = array(-9999.9,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      lteb    = array(-9999.9,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      rteb    = array(-9999.9,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      t2m_teb = array(-9999.9,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      q2m_teb = array(-9999.9,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      r2m_teb = array(-9999.9,c(length(FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
    }
    #
    if (max(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
      #
      load(file=paste(prsv,"H_ISBA",sep=""))
      hisb=LST$vals_day;hisb[which(is.na(hisb))]=-9999.9
      #
      load(file=paste(prsv,"LE_ISBA",sep=""))
      lisb=LST$vals_day;lisb[which(is.na(lisb))]=-9999.9
      #
      load(file=paste(prsv,"RN_ISBA",sep=""))
      risb=LST$vals_day;risb[which(is.na(risb))]=-9999.9
      #
      load(file=paste(prsv,"T2MMEA_ISBA",sep=""))
      t2m_isb=LST$vals_day;t2m_isb[which(is.na(t2m_isb))]=-9999.9
      #
      load(file=paste(prsv,"Q2MMEA_ISBA",sep=""))
      q2m_isb=LST$vals_day;q2m_isb[which(is.na(q2m_isb))]=-9999.9
      #
      load(file=paste(prsv,"HU2MMEA_ISBA",sep=""))
      r2m_isb=LST$vals_day;r2m_isb[which(is.na(r2m_isb))]=-9999.9
      #
    } else {
      hisb   =array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      lisb   =array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      risb   =array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      t2m_isb=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      q2m_isb=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      r2m_isb=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
    }
    #
    if (max(FRAC_WATER[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
      #
      load(file=paste(prsv,"H_WAT",sep=""))
      hwat=LST$vals_day;hwat[which(is.na(hwat))]=-9999.9
      #
      load(file=paste(prsv,"LE_WAT",sep=""))
      lwat=LST$vals_day;lwat[which(is.na(lwat))]=-9999.9
      #
      load(file=paste(prsv,"RN_WAT",sep=""))
      rwat=LST$vals_day;rwat[which(is.na(rwat))]=-9999.9
      #
      if (file.exists(paste(prsv,"T2MMEA_WAT",sep=""))) {
        #
        load(file=paste(prsv,"T2MMEA_WAT",sep=""))
        t2m_wat=LST$vals_day;t2m_wat[which(is.na(t2m_wat))]=-9999.9
        #
        load(file=paste(prsv,"Q2MMEA_WAT",sep=""))
        q2m_wat=LST$vals_day;q2m_wat[which(is.na(q2m_wat))]=-9999.9
        #
        load(file=paste(prsv,"HU2MMEA_WAT",sep=""))
        r2m_wat=LST$vals_day;r2m_wat[which(is.na(r2m_wat))]=-9999.9
      } else {
        print("CAUTION: No water temperature found, set to default")
        t2m_wat=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
        q2m_wat=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
        r2m_wat=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
    } else {
      hwat   =array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      lwat   =array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      rwat   =array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      t2m_wat=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      q2m_wat=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      r2m_wat=array(-9999.9,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
    }
    #
    if (max(FRAC_SEA[1:lenmask[count_nest],count_nest,sce])>1.0e-6) {
      #
      if (file.exists(paste(prsv,"H_SEA",sep=""))) {
        load(file=paste(prsv,"H_SEA",sep=""))
        hsea=LST$vals_day;hsea[which(is.na(hsea))]=-9999.9
      } else {
        hsea   =array(0.0  ,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"LE_SEA",sep=""))) {
        load(file=paste(prsv,"LE_SEA",sep=""))
        lsea=LST$vals_day;lsea[which(is.na(lsea))]=-9999.9
      } else {
        lsea   =array(0.0  ,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"RN_SEA",sep=""))) {
        load(file=paste(prsv,"RN_SEA",sep=""))
        rsea=LST$vals_day;rsea[which(is.na(rsea))]=-9999.9
      } else {
        rsea   =array(0.0,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"T2MMEA_SEA",sep=""))) {
        load(file=paste(prsv,"T2MMEA_SEA",sep=""))
        t2m_sea=LST$vals_day;t2m_sea[which(is.na(t2m_sea))]=-9999.9
      } else {
        t2m_sea=array(0.0,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"Q2MMEA_SEA",sep=""))) {
        load(file=paste(prsv,"Q2MMEA_SEA",sep=""))
        q2m_sea=LST$vals_day;q2m_sea[which(is.na(q2m_sea))]=-9999.9
      } else {
        q2m_sea   =array(0.0  ,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
      if (file.exists(paste(prsv,"HU2MMEA_SEA",sep=""))) {
        load(file=paste(prsv,"HU2MMEA_SEA",sep=""))
        r2m_sea=LST$vals_day;r2m_sea[which(is.na(r2m_sea))]=-9999.9
      } else {
        r2m_sea   =array(0.0  ,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      }
      #
    } else {
      hsea   =array(-9999.9,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      lsea   =array(-9999.9,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      rsea   =array(-9999.9,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      t2m_sea=array(-9999.9,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      q2m_sea=array(-9999.9,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      r2m_sea=array(-9999.9,c(length(FRAC_SEA[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
    }
    #
    # Aggregation of fluxes
    #
    hagg_day=array(NA,dim(hteb));lagg_day=array(NA,dim(lteb));ragg_day=array(NA,dim(rteb))
    #
    for (tt in 1:length(nout_time)) {
      #
      hagg_day[,tt]=FRAC_WATER [1:lenmask[count_nest],count_nest,sce]*hwat[,tt]+FRAC_SEA [1:lenmask[count_nest],count_nest,sce]*hsea[,tt]+
                    FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*hisb[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*hteb[,tt]
      #
      lagg_day[,tt]=FRAC_WATER [1:lenmask[count_nest],count_nest,sce]*lwat[,tt]+FRAC_SEA [1:lenmask[count_nest],count_nest,sce]*lsea[,tt]+
                    FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*lisb[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*lteb[,tt]
      #
      ragg_day[,tt]=FRAC_WATER [1:lenmask[count_nest],count_nest,sce]*rwat[,tt]+FRAC_SEA [1:lenmask[count_nest],count_nest,sce]*rsea[,tt]+
                    FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*risb[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*rteb[,tt]
      #
    }
    #
    # Aggregation of air temperature, relative and specific humidity over tiles
    #
    tr2m_day=array(NA,dim(t2m_teb));rh2m_day=array(NA,dim(t2m_teb));qs2m_day=array(NA,dim(t2m_teb))
    #
    for (tt in 1:length(nout_time)) {
      #
      tr2m_day[,tt] = FRAC_WATER [1:lenmask[count_nest],count_nest,sce]*t2m_wat[,tt]+FRAC_SEA [1:lenmask[count_nest],count_nest,sce]*t2m_sea[,tt]+
                      FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*t2m_isb[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*t2m_teb[,tt]
      #
      rh2m_day[,tt] = FRAC_WATER [1:lenmask[count_nest],count_nest,sce]*r2m_wat[,tt]+FRAC_SEA [1:lenmask[count_nest],count_nest,sce]*r2m_sea[,tt]+
                      FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*r2m_isb[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*r2m_teb[,tt]
      #
      qs2m_day[,tt] = FRAC_WATER [1:lenmask[count_nest],count_nest,sce]*q2m_wat[,tt]+FRAC_SEA [1:lenmask[count_nest],count_nest,sce]*q2m_sea[,tt]+
                      FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]*q2m_isb[,tt]+FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*q2m_teb[,tt]
      #
    }
    #
    LST=list(vals_day=hagg_day);save(LST,file=paste(prsv,"HAGG",sep=""))
    LST=list(vals_day=lagg_day);save(LST,file=paste(prsv,"LAGG",sep=""))
    LST=list(vals_day=ragg_day);save(LST,file=paste(prsv,"RAGG",sep=""))
    LST=list(vals_day=tr2m_day);save(LST,file=paste(prsv,"TREAL_2M",sep=""))
    LST=list(vals_day=rh2m_day);save(LST,file=paste(prsv,"RELHU_2M",sep=""))
    LST=list(vals_day=qs2m_day);save(LST,file=paste(prsv,"QSPEC_2M",sep=""))
    #
  }
  #
  # #######################################################################
  # Calculation of relative humidity and specific humidity on canopy levels
  # #######################################################################
  #
  vec_tile=c("TEB","ISB")
  #
  for (lvl in 1:6) for (tile in 1:length(vec_tile)) {
    #
    filetest=paste(prsv,vec_tile[tile],"_CAN_TM0",lvl,sep="")
    #
    if ( ((!file.exists(paste(prsv,vec_tile[tile],"_CAN_TREAL_",lvl,sep="")))||
          (!file.exists(paste(prsv,vec_tile[tile],"_CAN_RELHU_",lvl,sep="")))||
          (!file.exists(paste(prsv,vec_tile[tile],"_CAN_QSPEC_",lvl,sep=""))) )&
          file.exists(filetest)) {
      #
      load(file=paste(prsv,vec_tile[tile],"_CAN_TM0",lvl,sep=""));treal=LST$vals_day
      load(file=paste(prsv,vec_tile[tile],"_CAN_QM0",lvl,sep=""));qspe=LST$vals_day
      load(file=paste(prsv,vec_tile[tile],"_CAN_RM0",lvl,sep=""));relh=100.0*LST$vals_day
      load(file=paste(prsv,vec_tile[tile],"_CAN_PM0",lvl,sep=""));pabs=LST$vals_day
      #
      # Calculate the air density (dry air)
      # 
      rhoair = pabs/(XRD*(treal+XTT))
      #
      # Convert absolute humidity to specific humidity [g/kg]
      #
      qspe=1000.0*qspe/rhoair
      #
      # Check relative humidity
      #
      relh[which(relh>100.0)]=100.0
      relh[which(relh<  0.0)]=0.0     
      #
      amin=min(relh,na.rm=TRUE)
      amax=max(relh,na.rm=TRUE)
      if (amin<0.0)   {print(amin);stop("Unplausible value for canyon relative humidity")}
      if (amax>100.0) {print(amax);stop("Unplausible value for canyon relative humidity")}
      #
      # Save the results
      #
      LST=list(vals_day=treal)
      save(LST,file=paste(prsv,vec_tile[tile],"_CAN_TREAL_",lvl,sep=""))
      #
      LST=list(vals_day=relh)
      save(LST,file=paste(prsv,vec_tile[tile],"_CAN_RELHU_",lvl,sep=""))
      #
      LST=list(vals_day=qspe)
      save(LST,file=paste(prsv,vec_tile[tile],"_CAN_QSPEC_",lvl,sep=""))
      #
    }
  }
  #
  # ###################################################
  # Calculation of wind speed and wind direction
  # - Lowest level of Meso-NH (10 m above the urban canopy layer)
  # - Values at 2 m of ISBA and TEB canopy
  # - Velocity ratio for 2 m wind speed
  # ###################################################
  #
  # Only if SBL activated
  #
  if ( file.exists(paste(prsv,"TEB_CAN_Z02",sep=""))||file.exists(paste(prsv,"ISBA_CAN_Z02",sep=""))) {
  #
  if (!file.exists(paste(prsv,"VRAT_TEB_2m" ,sep="")) &
      !file.exists(paste(prsv,"VRAT_ISBA_2m",sep=""))) {
    #
    # Test height of canopy levels (first day)
    #
    if (day==1) {
      #
      if (scen[sce]!="NOURB") {
        #
        load(file=paste(prsv,"TEB_CAN_Z02",sep=""))
        zz02=LST$data_map
	#
        if (length(zz02)<10) stop("Error in canyon grid dimension")
        #
        if (max(abs(zz02-2.0),na.rm=TRUE)>1.0e-4) {
          print(max(zz02));print(min(zz02))
          print("CAUTION: TEB canopy level 2 not at 2 m")
        }
	#
      }
      #
      if (file.exists(paste(prsv,"ISBA_CAN_Z02",sep=""))) {
        #
        load(file=paste(prsv,"ISBA_CAN_Z02",sep=""))
        zz02=LST$data_map
        if (length(zz02)<10) stop("Error in canyon grid dimension")
        #
        if (max(abs(zz02-2.0),na.rm=TRUE)>1.0e-4) {
          print(max(zz02));print(min(zz02))
          print("CAUTION ISBA canopy level 2 not at 2 m")
        }
      #
      }
      #
    }
    #
    # Load canopy wind speed of TEB and ISBA
    #
    if (scen[sce]!="NOURB") {
      load(file=paste(prsv,"TEB_CAN_UM02",sep=""))
      ff_teb=LST$vals_day
    }
    #
    if (file.exists(paste(prsv,"ISB_CAN_UM02",sep=""))) {
      load(file=paste(prsv,"ISB_CAN_UM02",sep=""))
      ff_isb=LST$vals_day
    } else {
      ff_isb=array(NA,c(length(FRAC_NATURE[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
    }
    #
    # Load wind speed of lowest Meso-NH level
    #
    if (file.exists(paste(prsv,"WIFFME",sep=""))) {
      load(file=paste(prsv,"WIFFME",sep=""));wiff_day=LST$vals_day
    } else {
      print("CAUTION: No 10 m wind found, set to default")
      wiff_day = 99999
    }
    #
    # Calculate velocity ratio for TEB and ISBA
    #
    if (scen[sce]!="NOURB") vrat_teb = ff_teb/wiff_day
    vrat_isb = ff_isb/wiff_day
    #
    # Save data
    #
    if (scen[sce]!="NOURB") {
      #
      LST=list(vals_day=ff_teb)
      save(LST,file=paste(prsv,"FF_TEB_2m",sep=""))
      #
      LST=list(vals_day=vrat_teb)
      save(LST,file=paste(prsv,"VRAT_TEB_2m",sep=""))
      #
    }
    #
    LST=list(vals_day=ff_isb)
    save(LST,file=paste(prsv,"FF_ISBA_2m",sep=""))
    #
    LST=list(vals_day=vrat_isb)
    save(LST,file=paste(prsv,"VRAT_ISBA_2m",sep=""))
    #
  }
  #
  }
  #
  # ##############################################
  # Calculate hourly cumulated precipitation
  # ##############################################
  #
  if (!file.exists(paste(prsv,"PREC_HOURCUM",sep=""))) {
    #
    # Reconstruct the output directory of the previous day 
    #
    if (day>1.5) {
      #
      juldayact_reco=juldaybeg+(day-2)
      LST=mycaldat(juldayact_reco)
      montact_reco=LST$m;dayyact_reco=LST$d;yearact_reco=LST$y
      #
      namemont_reco=sprintf("%02d",montact_reco)
      namedayy_reco=sprintf("%02d",dayyact_reco)
      nameday_reco=paste(yearact_reco,namemont_reco,namedayy_reco,sep="")
      #
      prsv_reco=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday_reco,"_Nest_",nest,"/RDATA/",sep="")
      #
    }
    #
    if (file.exists(paste(prsv,"ACPRT",sep=""))) {
      #
      load(file=paste(prsv,"ACPRT",sep=""))
      prec_current_day=LST$vals_day
      #
      if (day>1.5) {
        load(file=paste(prsv_reco,"ACPRT",sep=""))
        prec_previous_day=LST$vals_day
      }
      #
      # Recontruct the hourly precipitation sum
      #
      prec_day=array(NA,c(length(LONS[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
      #
      for (seg in 1:length(nout_time)) {
        #
        if (seg==1) {
          if (day==1) {
            prec_day[,seg] = prec_current_day[,seg]
          } else {
            prec_day[,seg] = prec_current_day[,seg]-prec_previous_day[,length(nout_time)]
          }
        } else {
          prec_day[,seg] = prec_current_day[,seg] - prec_current_day[,seg-1]
        }
        #
      }
      #
      if (min(prec_day)<(-1.0e-2)) {print(min(prec_day));print("CAUTION: Wrong hourly precipitation")}
      prec_day[which(prec_day<0)]=0.0
      #
      if (max(prec_day)>100.0) {print(max(prec_day));print("CAUTION: Wrong hourly precipitation")}
      #
      # Save the results for the day
      #
      LST=list(vals_day=prec_day)
      save(LST,file=paste(prsv,"PREC_HOURCUM",sep=""))
      #
    } else {
      print("                                       ")
      print("CAUTION: Precipitation is not available")
    }
  }
  #
  # ################################################################
  # Multiply TEB variables with TOWN fraction when necessary
  # ################################################################
  #
  for (mult in 1:length(listtomult)) {
    if (!file.exists(paste(prsv,listtomult[mult],"_MUL",sep=""))) {
      filetest=paste(prsv,listtomult[mult],sep="")
      if (file.exists(filetest)) {
        load(file=paste(prsv,listtomult[mult],sep=""))
        vals=LST$vals_day
        vals_day=array(NA,c(length(LONS[1:lenmask[count_nest],count_nest,sce]),length(nout_time)))
        for (tt in 1:length(nout_time)) {
          vals_day[,tt]=FRAC_TOWN[1:lenmask[count_nest],count_nest,sce]*vals[,tt]
        }
        LST=list(vals_day=vals_day)
        save(LST,file=paste(prsv,listtomult[mult],"_MUL",sep=""))
      }
    }
  }
  #
  # #########################################################################
  # Calculate the total anthropogenic heat flux
  # ##########################################################################
  #
  if ( !file.exists(paste(prsv,"QANT_TOTAL",sep=""))||
       !file.exists(paste(prsv,"QANT_BUILDINGS",sep="")) ) {
    #
    filetest=paste(prsv,"HVAC_HT_MUL",sep="")
    #
    if (file.exists(filetest)) {
      #
      load(file=paste(prsv,"HVAC_HT_MUL",sep=""));hvac_ht_day=LST$vals_day
      load(file=paste(prsv,"HVAC_CL_MUL",sep=""));hvac_cl_day=LST$vals_day
      load(file=paste(prsv,"QINACT_MUL",sep=""));qinact_day=LST$vals_day
      #
      if (file.exists(paste(prsv,"QINHOTW_MUL",sep=""))) {
        load(file=paste(prsv,"QINHOTW_MUL",sep=""));qinhot_day=LST$vals_day
      } else {
        print("CAUTION: No data for QINHOTW in this model version, set to 0")
        qinhot_day=array(0,dim(qinact_day))
      }
      #
      load(file=paste(prsv,"H_TRAACT_MUL",sep=""));htraff_day=LST$vals_day
      load(file=paste(prsv,"LE_TRAACT_MUL",sep=""));ltraff_day=LST$vals_day
      load(file=paste(prsv,"H_INDACT_MUL",sep=""));hindus_day=LST$vals_day
      load(file=paste(prsv,"LE_INDACT_MUL",sep=""));lindus_day=LST$vals_day
      #
      # The total anthropogenic heat release
      #
      vals_day=hvac_ht_day+hvac_cl_day+qinact_day+qinhot_day+htraff_day+ltraff_day+hindus_day+lindus_day
      LST=list(vals_day=vals_day)
      save(LST,file=paste(prsv,"QANT_TOTAL",sep=""))
      #   
      # The building-related anthropogenic heat release
      #
      vals_day=hvac_ht_day+hvac_cl_day+qinact_day+qinhot_day
      LST=list(vals_day=vals_day)
      save(LST,file=paste(prsv,"QANT_BUILDINGS",sep=""))
      #
    }
  }
  #
  # ############################################################
  # Sum the anthropogenic heat release over the CAPITOUL domain
  # ############################################################
  #
  if (flag_capitoul>0.5) {
    #
    lonmin_CAP=01.346;lonmax_CAP=01.534
    latmin_CAP=43.532;latmax_CAP=43.668
    #
    if (!file.exists(paste(prsv,"QANT_TOTAL_CAPITOUL",sep=""))||!file.exists(paste(prsv,"QANT_BUILDINGS_CAPITOUL",sep=""))) {
      #
      filetest=paste(prsv,"QANT_TOTAL",sep="")
      #
      if (file.exists(filetest)) {
        #
        load(file=paste(prsv,"QANT_TOTAL",sep=""));qanttotal_day=LST$vals_day
        load(file=paste(prsv,"QANT_BUILDINGS",sep=""));qantbuild_day=LST$vals_day
        #
        ind_CAP=which((LONS[1:lenmask[count_nest],count_nest,sce]>lonmin_CAP)&(LONS[1:lenmask[count_nest],count_nest,sce]<lonmax_CAP)&
	              (LATS[1:lenmask[count_nest],count_nest,sce]>latmin_CAP)&(LATS[1:lenmask[count_nest],count_nest,sce]<latmax_CAP))
        #
        qanttotal_sel=qanttotal_day[ind_CAP,,drop=FALSE]
        qantbuild_sel=qantbuild_day[ind_CAP,,drop=FALSE]
        #
        qanttotal_meanhour=colMeans(qanttotal_sel,dims=1,na.rm=TRUE)
        qantbuild_meanhour=colMeans(qantbuild_sel,dims=1,na.rm=TRUE)
        #
        LST=list(qanttotal_meanhour=qanttotal_meanhour)
        save(LST,file=paste(prsv,"QANT_TOTAL_CAPITOUL",sep=""))
        #
        LST=list(qantbuild_meanhour=qantbuild_meanhour)
        save(LST,file=paste(prsv,"QANT_BUILDINGS_CAPITOUL",sep=""))
        #
      }
    }
    #
  }
  #
  # ############################################################
  # Extract variables at stations
  # ############################################################
  #
  # SURF stations
  #
  if (!is.na(lats_surfstat[1])) {
    #
    filesurfstationindex=paste(prsv_first[sce,count_nest],"SURF_STAT_INDEX",sep="")
    #
    if (!file.exists(paste(prsv,"SURF_STATION_EXTRACT",sep=""))) {   
      #
      if ((day==1)&&(!file.exists(filesurfstationindex))) {
        #
        print("                             ")
        print("Extraction of station indices")
        print("                             ")
        #
        load(file=paste(prsv,varlistonline[1],sep=""));vals_day=LST$vals_day
        DIM=dim(vals_day)
        #
        surfstat_index=array(NA,c(length(lats_surfstat),100))
        #
        for (stat in 1:length(lats_surfstat)) {
          #
          y_stat=array(NA,c(1,2))
          y_stat[1,1] = lons_surfstat[stat]
          y_stat[1,2] = lats_surfstat[stat]
          #
          x_model=array(NA,c(DIM[1],2))
          x_model[,1] = LONS[1:lenmask[count_nest],count_nest,sce]
          x_model[,2] = LATS[1:lenmask[count_nest],count_nest,sce]
          #
          dist_stat = distm(x_model,y_stat,fun=distHaversine)
          amin=min(dist_stat)
          print(paste("The minimum distance to station",name_surfstat[stat],"is",format(amin/1000.0,digits=2),"km"))
          print(paste("Longitude:",lons_surfstat[stat]))
          print(paste("Latitude :",lats_surfstat[stat]))
          #
          # The station is only considered when the
          # minimum distance is below 0.5*sqrt(2)*dx
	  # All model grid points within 2*dx distance are considered
          #
          if (amin<(0.5*xdx[count_nest]*sqrt(2))) {
            indmin=which(dist_stat<=(xdx[count_nest]))
            print(paste("A total of ",length(indmin),"grid points within",format(xdx[count_nest]/1000.0,digits=2),"km"))
	    print("Model elevation around station ")
	    print(HEIG[indmin,count_nest,sce])
            print("                               ")
            surfstat_index[stat,1:length(indmin)]=indmin
          } else {
            print("                                          ")
            print("This station is excluded from the analysis")
            print("                                          ")
            surfstat_index[stat,]=NA
          }
          #
        }
        #
        LST=list(surfstat_index=surfstat_index)
        save(LST,file=filesurfstationindex)
        #
      }
      else {
        load(file=filesurfstationindex)
        surfstat_index=LST$surfstat_index
      }
      #
      for (vareval in 1:length(varieval_capitoul_surfstat)) {
        #
	print(" ")
	print(paste("Treating",varieval_capitoul_surfstat[vareval]))
	print(" ")
        #
        # Load model output at all canyon levels
        #
        # Get field dimensions
        #
        load(file=paste(prsv,varlistonline[1],sep=""))
        vals_day=LST$vals_day
        DIM=dim(vals_day)
        #
        teb_can_vals=array(NA,c(DIM,6))
        isb_can_vals=array(NA,c(DIM,6))
        #
        for (lvl in 1:6) for (tile in 1:length(vec_tile)) {
          #
          filecan=paste(prsv,vec_tile[tile],"_CAN_",varieval_capitoul_surfstat[vareval],"_",lvl,sep="")
          #
          if (file.exists(filecan)) {
            load(file=filecan);vals_day=LST$vals_day
            if (tile==1) teb_can_vals[,,lvl]=vals_day
            if (tile==2) isb_can_vals[,,lvl]=vals_day
          }
          #
        }
        #
        # Extract the canyon output at the stations
        #
        DIMCAN=dim(teb_can_vals)
        vals_tebcan=array(NA,c(length(lats_surfstat),DIMCAN[2],DIMCAN[3]))
        vals_isbcan=array(NA,c(length(lats_surfstat),DIMCAN[2],DIMCAN[3]))
        frac_town_surf=array(NA,length(lats_surfstat))
        frac_natu_surf=array(NA,length(lats_surfstat))
        frac_watr_surf=array(NA,length(lats_surfstat))
        frac_seaf_surf=array(NA,length(lats_surfstat))
        #
        for (stat in 1:length(lats_surfstat)) {
	  #
	  ind_stat=na.omit(surfstat_index[stat,])
          #
          if (length(ind_stat>0)) {
            #
            frac_town_surf[stat]=mean(FRAC_TOWN  [ind_stat,count_nest,sce])
            frac_natu_surf[stat]=mean(FRAC_NATURE[ind_stat,count_nest,sce])
            frac_watr_surf[stat]=mean(FRAC_WATER [ind_stat,count_nest,sce])
            frac_seaf_surf[stat]=mean(FRAC_SEA   [ind_stat,count_nest,sce])	  
            #
            ctl=frac_town_surf[stat]+frac_natu_surf[stat]+frac_watr_surf[stat]+frac_seaf_surf[stat]
            if (abs(ctl-1.0)>1.0e-4) stop("Wrong sum of fractions at stations")
            #
	    for (nn in 1:DIMCAN[2]) for (mm in 1:DIMCAN[3]) {
              vals_tebcan[stat,nn,mm]=mean(teb_can_vals[ind_stat,nn,mm],na.rm=TRUE)
              vals_isbcan[stat,nn,mm]=mean(isb_can_vals[ind_stat,nn,mm],na.rm=TRUE)
	    }
            #
	  }
	  #
        }
        #
        if (vareval==1) vals_surfstat=array(NA,c(length(varieval_capitoul_surfstat),length(lats_surfstat),length(nout_time)))
        #
        # Find the correct canyon level for the stations
        # The level is assigned according to the station type
        #
        DIM=dim(vals_tebcan)
        #
        tebcan_lvl_vals=array(NA,c(DIM[1],DIM[2]))
        isbcan_lvl_vals=array(NA,c(DIM[1],DIM[2]))
        #
        load(file=paste(prsv_first[sce,count_nest],"STATION_LEVELS",sep=""))
        wind_lvl_teb=LST$wind_lvl_teb;temp_lvl_teb=LST$temp_lvl_teb
        wind_lvl_isb=LST$wind_lvl_isb;temp_lvl_isb=LST$temp_lvl_isb
        #
        aux_town = FRAC_TOWN   [1:lenmask[count_nest],count_nest,sce]
        aux_natr = FRAC_NATURE [1:lenmask[count_nest],count_nest,sce]
        aux_seaf = FRAC_SEA    [1:lenmask[count_nest],count_nest,sce]
        aux_watr = FRAC_WATER  [1:lenmask[count_nest],count_nest,sce]
        #
        for (stat in 1:DIM[1]) {
          #
          print(paste("Treating",name_surfstat[stat]))
          #
          if (length(ind_stat>0)) {
	    #
            for (nn in 1:DIM[2]) {
              tebcan_lvl_vals[stat,nn]=vals_tebcan[stat,nn,temp_lvl_teb[stat]]
              isbcan_lvl_vals[stat,nn]=vals_isbcan[stat,nn,temp_lvl_isb[stat]]
            }
            #
            aux_town_stat = mean(aux_town[ind_stat])
            aux_natu_stat = mean(aux_natr[ind_stat])
            aux_seaf_stat = mean(aux_seaf[ind_stat])
            aux_watr_stat = mean(aux_watr[ind_stat])
	    #
            # Get the values at 2 m on sea and water
	    #
            if (varieval_capitoul_surfstat[vareval]=="TREAL") {
              varselect="T2MMEA_"
	    } else if (varieval_capitoul_surfstat[vareval]=="RELHU") {
	      varselect="HU2MMEA_"
	    } else if (varieval_capitoul_surfstat[vareval]=="QSPEC") {
	      varselect="Q2MMEA_"
	    } else {
              stop("No rule implemented for this variable")
            }
            #
            filecan=paste(prsv,varselect,"SEA",sep="")
            load(file=filecan);vals2m_sea=LST$vals_day
	    #
            filecan=paste(prsv,varselect,"WAT",sep="")
            load(file=filecan);vals2m_wat=LST$vals_day
            #
            DIM_day=dim(vals2m_wat)
            sea_lvl2_vals=array(0,DIM_day[2])
  	    wat_lvl2_vals=array(0,DIM_day[2])
	    for (hour in 1:DIM_day[2]) {
              sea_lvl2_vals[hour]=mean(vals2m_sea[ind_stat,hour],na.rm=TRUE)
              wat_lvl2_vals[hour]=mean(vals2m_wat[ind_stat,hour],na.rm=TRUE)
	    }
            #
            # Interpolation between SURFEX tiles depending on station characteristics
            #
            aux = array(0,c(DIM[2]))
	    sum = 0
	    #
	    if (aux_town_stat>0) {
              aux = aux + town_surfstat[stat]*tebcan_lvl_vals[stat,]
	      sum = sum + town_surfstat[stat]
            }
            #
            if (aux_natu_stat>0) {
              aux = aux + natu_surfstat[stat]*isbcan_lvl_vals[stat,]
	      sum = sum + natu_surfstat[stat]
            }
            #
            if (aux_seaf_stat>0) {
              aux = aux + seaf_surfstat[stat]*sea_lvl2_vals
	      sum = sum + seaf_surfstat[stat]
            }
            #
            if (aux_watr_stat>0) {
              aux = aux + watr_surfstat[stat]*wat_lvl2_vals
	      sum = sum + watr_surfstat[stat]
            }
            #
	    print(aux_town_stat)
	    print(aux_natu_stat)
	    print(aux_seaf_stat)
	    print(aux_watr_stat)
	    print(town_surfstat[stat])
	    print(natu_surfstat[stat])
	    print(seaf_surfstat[stat])
	    print(watr_surfstat[stat])
	    print(sum)
	    #
            if (sum<1.0e-4) stop("No corresponding surface fractions found")
            #
            vals_surfstat[vareval,stat,]=aux/sum
            #
	  }
        }
        #
      }
      #
      # Save model data required for evaluation against station data
      #
      LST=list(vals_surfstat=vals_surfstat)
      save(LST,file=paste(prsv,"SURF_STATION_EXTRACT",sep=""))
      #
    }
    #
  }
  #
  # The CNRM network only
  #
  if (!is.na(lats_cnrmstat[1])) {
    #
    filecnrmstationindex=paste(prsv_first[sce,count_nest],"CNRM_STAT_INDEX",sep="")
    #
    if (!file.exists(paste(prsv,"CNRM_STATION_EXTRACT",sep=""))) {   
      #
      if (day==1) {
        #
        if (!file.exists(filecnrmstationindex)) {
          #
          print("                                          ")
          print("Extraction of station indices at first day")
          print("                                          ")
          #
          # Load model output for one variable
          #
          load(file=paste(prsv,varlistonline[1],sep=""));vals_day=LST$vals_day
          DIM=dim(vals_day)
          #
          cnrmstat_index=array(NA,c(length(lats_cnrmstat),100))
          #
          for (stat in 1:length(lats_cnrmstat)) {
            #
            y_stat=array(NA,c(1,2))
            y_stat[1,1] = lons_cnrmstat[stat]
            y_stat[1,2] = lats_cnrmstat[stat]
            #
            x_model=array(NA,c(DIM[1],2))
            x_model[,1] = LONS[1:lenmask[count_nest],count_nest,sce]
            x_model[,2] = LATS[1:lenmask[count_nest],count_nest,sce]
            #
            dist_stat=distm(x_model,y_stat,fun=distHaversine)
            amin=min(dist_stat)
            #
            print(paste("The minimum distance to station",name_cnrmstat[stat],"is",format(amin/1000.0,digits=2),"km"))
            #
            # The station is only considered when the
            # minimum distance is below 0.5*sqrt(2)*dx
            # All model grid points within 2*dx distance are considered
            #
            if (amin<(0.5*xdx[count_nest]*sqrt(2))) {
	      indmin=which(dist_stat==amin)
	      indmin=indmin[1]
              # indmin=which(dist_stat<=(xdx[count_nest]))
              print(paste("A total of ",length(indmin),"grid points within",format(xdx[count_nest]/1000.0,digits=2),"km"))
              cnrmstat_index[stat,1:length(indmin)]=indmin
            } else {
              print("                                          ")
              print("This station is excluded from the analysis")
              print("                                          ")
              cnrmstat_index[stat,]=NA
            }
          #
          }
          #
          LST=list(cnrmstat_index=cnrmstat_index)
          save(LST,file=filecnrmstationindex)
          #
        }
      }
      #
      load(file=filecnrmstationindex)
      cnrmstat_index=LST$cnrmstat_index
      #
      # Load model output for the variable to be evaluated
      # Contrary to the SURF stations, the values are not taken from the canopy
      # However, the model level for evaluation is selected for each station
      # depending on the sensor position.
      #
      pathprofile=paste0(pprofile,name,"/")
      system(paste("mkdir -p",pathprofile))
      #
      pathprofile=paste0(pprofile,name,"/",scen[sce])
      system(paste("mkdir -p",pathprofile))
      #
      pathprofile=paste0(pprofile,name,"/",scen[sce],"/",nameday,"/")
      system(paste("mkdir -p",pathprofile))
      #
      pathprofile=paste0(pprofile,name,"/",scen[sce],"/",nameday,"/Nest_",nest,"/")
      system(paste("mkdir -p",pathprofile))
      #
      print(pathprofile)
      #
      for (vareval in 1:length(varieval_capitoul_cnrmstat)) {
        #
        # The relevant sensor height is selected as a function of the
        # meteorological variable
	#
	if ( (varieval_capitoul_cnrmstat[vareval]=="TEMPMME")       || (varieval_capitoul_cnrmstat[vareval]=="RHME") ||
             (varieval_capitoul_cnrmstat[vareval]=="QSPECME")       || (varieval_capitoul_cnrmstat[vareval]=="PABSME") ||
             (varieval_capitoul_cnrmstat[vareval]=="TOTFLASWD_SUM") || (varieval_capitoul_cnrmstat[vareval]=="PREC_HOURCUM") ) {
          z_sensor=hite_cnrmstat
        } else if ((varieval_capitoul_cnrmstat[vareval]=="WIDDME") || (varieval_capitoul_cnrmstat[vareval]=="WIFFME")) {
          z_sensor=hiff_cnrmstat
        } else {
          stop("No rule implemented for this variable")
        }
        #
	for (stat in 1:length(name_cnrmstat)) {
          #
          z_sensor_stat=z_sensor[stat]
	  #
          # Find the model level closest to the sensor height
          #
          adiff=abs(zlvl-z_sensor_stat)
	  ind_min=which(adiff==min(adiff))
	  ind_min=ind_min[1]
	  #
	  zlvl_sel=zlvl[ind_min]
	  adiff = abs(zlvl_sel-z_sensor_stat)
	  if (adiff>15) {
            print("Station height too far away from model level")
            print(paste("Station height",z_sensor_stat))
            print(paste("Model level",zlvl_sel))
	  }
	  #
          # The file to load
          #
	  if ( (varieval_capitoul_cnrmstat[vareval]=="TOTFLASWD_SUM") ||
               (varieval_capitoul_cnrmstat[vareval]=="PREC_HOURCUM") ) {
            fileload = paste(prsv,varieval_capitoul_cnrmstat[vareval],sep="")
          } else {
	    fileload = paste(prsv,"_LVL_",ind_min+1,"_SPLIT_",varieval_capitoul_cnrmstat[vareval],sep="")
          }
	  #
          load(file=fileload)
          vals_day=LST$vals_day
	  #
          vals_model=array(NA,length(nout_time))
          ind_stat=na.omit(cnrmstat_index[stat,])
          for (seg in 1:length(nout_time)) vals_model[seg]=mean(vals_day[ind_stat,seg],na.rm=TRUE)
          #
          if ((vareval==1)&&(stat==1)) {
            vals_cnrmstat=array(NA,c(length(varieval_capitoul_cnrmstat),length(name_cnrmstat),length(nout_time)))
	    prof_cnrmstat=array(NA,c(length(varieval_capitoul_cnrmstat),length(zlvl),length(name_cnrmstat),length(nout_time)))
          }
          #
          vals_cnrmstat[vareval,stat,]=vals_model
          #
	  # Load the vertical profiles
	  #
          if ( (varieval_capitoul_cnrmstat[vareval]!="TOTFLASWD_SUM") &
               (varieval_capitoul_cnrmstat[vareval]!="PREC_HOURCUM") ) {
            #
            for (lvl in 1:length(zlvl)) {
              #
              fileload = paste(prsv,"_LVL_",lvl+1,"_SPLIT_",varieval_capitoul_cnrmstat[vareval],sep="")
              load(file=fileload)
              vals_day=LST$vals_day
              #
              vals_model=array(NA,length(nout_time))
              ind_stat=na.omit(cnrmstat_index[stat,])
              for (seg in 1:length(nout_time)) vals_model[seg]=mean(vals_day[ind_stat,seg],na.rm=TRUE)
              #
	      prof_cnrmstat[vareval,lvl,stat,]=vals_model
              #
            }
	    #
            # Plot the vertical profile at 14:00 and 20:00 HK-TIME (UTC 4 and 12)
            #
	    for (hour in c(4,12)) {
              #
              nameleg = c("Meso-NH")
              colors  = c("black")
              ltys    = c(1)
              #
              vertprof(prof_cnrmstat[vareval,,stat,hour],zlvl,paste0("PROFILE_HOUR_",hour,"_UTC_Station_",numb_cnrmstat[stat]),
                       uniteval_capitoul_cnrmstat[vareval],nameeval_capitoul_cnrmstat[vareval],
                       varieval_capitoul_cnrmstat[vareval],nameleg,colors,ltys,path=pathprofile)
              #
            }
	  }
	  #
        }
      }
      #
      # Save model data required for evaluation against station data
      #
      LST=list(vals_cnrmstat=vals_cnrmstat,prof_cnrmstat=prof_cnrmstat)
      save(LST,file=paste(prsv,"CNRM_STATION_EXTRACT",sep=""))
      #
    }
    #
  }
  #
  # ############################################################
  # Calculation of evaluation measures for each day
  # where observations are available
  # - BIAS
  # - RMSE
  # ############################################################
  #
  # Temperature and relative humidity at merged SURF+CNRM stations
  #
  if (!is.na(lats_surfstat[1])) {
    #
    if (!file.exists(paste(prsv,"SURF_EVAL_MEASURES",sep=""))) {
      #
      load(file=paste(prsv,"SURF_STATION_EXTRACT",sep=""))
      vals_surfstat=LST$vals_surfstat
      #
      # The Julian dates of the model results (hourly values)
      #
      frac_day=nout_time/24
      julday_hour=juldayact+frac_day
      #
      # Find the time indices in the station data
      # The observations are only taken if they are closer than 15
      # minutes to the model results
      #
      ind_time=array(NA,length(frac_day))
      for (nhour in 1:length(frac_day)) {
        diff_time=abs(juld_surfstat-julday_hour[nhour])
        mind_time=min(diff_time)
        index_time=which(mind_time==diff_time)
        if (mind_time<(31.0/1440.0)) ind_time[nhour]=index_time[1]
      }
      #
      # Extract the observations at the time indices
      #
      obs_data_day=array(NA,c(length(varieval_capitoul_surfstat),length(lats_surfstat),length(nout_time)))
      for (nhour in 1:length(nout_time)) {
        if (!is.na(ind_time[nhour])) {
          obs_data_day[1,,nhour]=temp_surfstat[,ind_time[nhour]]
          obs_data_day[2,,nhour]=relh_surfstat[,ind_time[nhour]]
          obs_data_day[3,,nhour]=qspe_surfstat[,ind_time[nhour]]
        }
      }
      #
      # The evaluation measures are calculated if more than
      # 2/3 of the daily values are available for the specific variables
      #
      bias_day=array(NA,c(length(varieval_capitoul_surfstat),length(lats_surfstat)))
      rmse_day=array(NA,c(length(varieval_capitoul_surfstat),length(lats_surfstat)))
      #
      for (metvar in 1:length(varieval_capitoul_surfstat)) for (stat in 1:length(lats_surfstat)) {
        #
	# Exclude values with NA in obs
	#
	vals_surfstat[metvar,stat,which(is.na(obs_data_day [metvar,stat,]))] = NA
        #
        obs_vec = obs_data_day [metvar,stat,]
        sim_vec = vals_surfstat[metvar,stat,]
        #
        if (length(which(!is.na(obs_vec)))>(0.66*length(nout_time))) {
          #
          err_vec=sim_vec-obs_vec
          bias_day[metvar,stat]=mean(err_vec,na.rm=TRUE)
          rmse_day[metvar,stat]=sqrt(mean(err_vec*err_vec,na.rm=TRUE))
          #
        }
      }
      #
      # Save the evaluation measures for the given day
      #
      LST=list(obs_data_day=obs_data_day,vals_surfstat=vals_surfstat,
               bias_day=bias_day,rmse_day=rmse_day,
	       lats_surfstat=lats_surfstat,lons_surfstat=lons_surfstat,
	       heig_surfstat=heig_surfstat,name_surfstat=name_surfstat,
	       numb_surfstat=numb_surfstat)
      #
      save(LST,file=paste(prsv,"SURF_EVAL_MEASURES",sep=""))
      #
    }
  #
  }
  #
  # Additional variables at CNRM stations
  #
  if (!is.na(lats_cnrmstat[1])) {
    #
    if (!file.exists(paste(prsv,"CNRM_EVAL_MEASURES",sep=""))) {  
      #
      load(file=paste(prsv,"CNRM_STATION_EXTRACT",sep=""))
      vals_cnrmstat=LST$vals_cnrmstat
      #
      # The Julian dates of the model results (hourly values)
      #
      frac_day=nout_time/24
      julday_hour=juldayact+frac_day
      #
      # Find the time indices in the station data
      # The observations are only taken if they are closer
      # than 31 minutes to the model output
      #
      ind_time=array(NA,length(nout_time))
      for (nhour in 1:length(nout_time)) {
        diff_time=abs(juld_cnrmstat-julday_hour[nhour])
        mind_time=min(diff_time)
        index_time=which(mind_time==diff_time)
        if (mind_time<(60.0/1440.0)) ind_time[nhour]=index_time[1]
      }
      #
      # Extract the observations at the time indices
      #
      obs_data_day=array(NA,c(length(varieval_capitoul_cnrmstat),length(lats_cnrmstat),length(nout_time)))
      #
      for (nhour in 1:length(nout_time)) {
        if (!is.na(ind_time[nhour])) {
          for (cnrmvari in 1:length(varieval_capitoul_cnrmstat)) {
            #
            if      (varieval_capitoul_cnrmstat[cnrmvari]=="TEMPMME")      obs_data_day[cnrmvari,,nhour]=temp_cnrmstat[,ind_time[nhour]]
            else if (varieval_capitoul_cnrmstat[cnrmvari]=="RHME")         obs_data_day[cnrmvari,,nhour]=relh_cnrmstat[,ind_time[nhour]]
            else if (varieval_capitoul_cnrmstat[cnrmvari]=="QSPECME")      obs_data_day[cnrmvari,,nhour]=qspe_cnrmstat[,ind_time[nhour]]
            else if (varieval_capitoul_cnrmstat[cnrmvari]=="PREC_HOURCUM")  obs_data_day[cnrmvari,,nhour]=rrat_cnrmstat[,ind_time[nhour]]
            else if (varieval_capitoul_cnrmstat[cnrmvari]=="TOTFLASWD_SUM") obs_data_day[cnrmvari,,nhour]=glob_cnrmstat[,ind_time[nhour]]
            else if (varieval_capitoul_cnrmstat[cnrmvari]=="WIFFME")        obs_data_day[cnrmvari,,nhour]=wiff_cnrmstat[,ind_time[nhour]]
            else if (varieval_capitoul_cnrmstat[cnrmvari]=="WIDDME")        obs_data_day[cnrmvari,,nhour]=widd_cnrmstat[,ind_time[nhour]]
            else if (varieval_capitoul_cnrmstat[cnrmvari]=="FLALWD")        obs_data_day[cnrmvari,,nhour]=lwdw_cnrmstat[,ind_time[nhour]]
            else {
              print(varieval_capitoul_cnrmstat[cnrmvari])
	      stop("This variable in not known in CNRM SURSTAT")
	    }
          }
        }
      }
      #
      # The evaluation measures are calculated if more than
      # 2/3 of the output time steps are available for the specific variables
      #
      bias_day=array(NA,c(length(varieval_capitoul_cnrmstat),length(lats_cnrmstat)))
      rmse_day=array(NA,c(length(varieval_capitoul_cnrmstat),length(lats_cnrmstat)))
      #
      for (metvar in 1:length(varieval_capitoul_cnrmstat)) for (stat in 1:length(lats_cnrmstat)) {
        #
	# Exclude values with NA in obs
	#
	vals_cnrmstat[metvar,stat,which(is.na(obs_data_day [metvar,stat,]))] = NA
        #
        obs_vec = obs_data_day[metvar,stat,]
        sim_vec = vals_cnrmstat[metvar,stat,]
        #
        if (length(which(!is.na(obs_vec)))>(0.66*length(nout_time))) {
          err_vec=sim_vec-obs_vec
          bias_day[metvar,stat]=mean(err_vec,na.rm=TRUE)
          rmse_day[metvar,stat]=sqrt(mean(err_vec*err_vec,na.rm=TRUE))
        }
      }
      #
      # Save the evaluation measures for the given day
      #
      LST=list(obs_data_day=obs_data_day,vals_cnrmstat=vals_cnrmstat,
               bias_day=bias_day,rmse_day=rmse_day,lats_cnrmstat=lats_cnrmstat,
	       lons_cnrmstat=lons_cnrmstat,heig_cnrmstat=heig_cnrmstat,
               name_cnrmstat=name_cnrmstat,numb_cnrmstat=numb_cnrmstat)
      #
      save(LST,file=paste(prsv,"CNRM_EVAL_MEASURES",sep=""))
      #
    }
  #
  # Calculate daily averages of CNRM_EVAL_MEASURES
  #
  filecnrmdaily=paste(prsv,"CNRM_DAILY",sep="")
  #
  if (!file.exists(filecnrmdaily)) {
    #
    load(file=paste(prsv,"CNRM_EVAL_MEASURES",sep=""))
    obs_data_day=LST$obs_data_day;sim_data_day=LST$vals_cnrmstat
    #
    obs_data_daymean=array(NA,c(length(varieval_capitoul_cnrmstat),length(lats_cnrmstat)))
    sim_data_daymean=array(NA,c(length(varieval_capitoul_cnrmstat),length(lats_cnrmstat)))
    #
    for (metvar in 1:length(varieval_capitoul_cnrmstat)) for (stat in 1:length(lats_cnrmstat)) {
      #
      # Exclude values with NA in obs and with NA in simulation (out of domain)
      #
      sim_data_day[metvar,stat,which(is.na(obs_data_day [metvar,stat,]))] = NA
      obs_data_day[metvar,stat,which(is.na(sim_data_day [metvar,stat,]))] = NA
      #
      # Calculate daily average per station
      #
      obs_vec = obs_data_day[metvar,stat,]
      sim_vec = sim_data_day[metvar,stat,]
      #
      if (length(which(!is.na(obs_vec)))>=18) {
        #
        if (varieval_capitoul_cnrmstat[metvar]=="PREC_HOURCUM") {
          obs_mean=sum(obs_vec,na.rm=TRUE)
          sim_mean=sum(sim_vec,na.rm=TRUE)
        } else if (varieval_capitoul_cnrmstat[metvar]=="WIDDME") {
          obs_mean=mean(circular(na.omit(obs_vec),units="degrees"))
      	  sim_mean=mean(circular(na.omit(sim_vec),units="degrees"))
        } else {
          obs_mean=mean(obs_vec,na.rm=TRUE)
     	  sim_mean=mean(sim_vec,na.rm=TRUE)
        }
	#
        obs_data_daymean[metvar,stat]=obs_mean
        sim_data_daymean[metvar,stat]=sim_mean
	#
      }
      #
    }
    #
    # Calculate domain averages
    #
    obs_data_daydommean=array(NA,length(varieval_capitoul_cnrmstat))
    sim_data_daydommean=array(NA,length(varieval_capitoul_cnrmstat))
    #
    for (metvar in 1:length(varieval_capitoul_cnrmstat)) {
      if (varieval_capitoul_cnrmstat[metvar]=="WIDDME") {
        obs_data_daydommean[metvar]=mean(circular(na.omit(obs_data_daymean[metvar,]),units="degrees"))
        sim_data_daydommean[metvar]=mean(circular(na.omit(sim_data_daymean[metvar,]),units="degrees"))
      } else {
        obs_data_daydommean[metvar]=mean(obs_data_daymean[metvar,],na.rm=TRUE)
        sim_data_daydommean[metvar]=mean(sim_data_daymean[metvar,],na.rm=TRUE)
      }
    }
    #
    LST=list(obs_data_daydommean=obs_data_daydommean,sim_data_daydommean=sim_data_daydommean,varieval_capitoul_cnrmstat=varieval_capitoul_cnrmstat)
    save(LST,file=filecnrmdaily)
    #
  }
  #
  }
  #
  # #############################################################
  # Calculate averages over selected domain for time series plots
  # #############################################################
  #
  for (vari in 1:length(varplot)) {
    filesave_meandom=paste(prsv,varplot[vari],"_MEANDOM",sep="")
    if (!file.exists(filesave_meandom)) {
      if (file.exists(paste(prsv,varplot[vari],sep=""))) {
        load(file=paste(prsv,varplot[vari],sep=""))
        vals_day=LST$vals_day
        ind_DOM=which((LONS[1:lenmask[count_nest],count_nest,sce]>lonmin_ts)&(LONS[1:lenmask[count_nest],count_nest,sce]<lonmax_ts)&
	              (LATS[1:lenmask[count_nest],count_nest,sce]>latmin_ts)&(LATS[1:lenmask[count_nest],count_nest,sce]<latmax_ts))
        vals_day_sel =vals_day[ind_DOM,,drop=FALSE]
        vals_meanhour=colMeans(vals_day_sel,dims=1,na.rm=TRUE)
        LST=list(vals_meanhour=vals_meanhour)
        save(LST,file=filesave_meandom)
      }
    }
  }
  #
  # End loop over scenarios
  #
  }
  #
  # ##########################################################
  # Calculate the urban increment for a list of variables
  # ##########################################################
  #
  print(paste("Preparing results for URBINC-",nameday,", Nest ",nest,sep=""))
  #
  if (length(scen)>1) {
    for (vari in 1:length(varlist_urbinc)) {
      #
      # Spatial distribution of urban increment
      #
      if (!file.exists(paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/",varlist_urbinc[vari],sep=""))) {
        #
        load(file=paste(pdsup,name,"/REFER/ONLINE/",nameday,"_Nest_",nest,"/RDATA/",varlist_urbinc[vari],sep=""))
        vals_day_refer=LST$vals_day
        #
        load(file=paste(pdsup,name,"/NOURB/ONLINE/",nameday,"_Nest_",nest,"/RDATA/",varlist_urbinc[vari],sep=""))
        vals_day_nourb=LST$vals_day
        #
        vals_urbinc=vals_day_refer-vals_day_nourb
        #
        LST=list(vals_day=vals_urbinc)
        save(LST,file=paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/",varlist_urbinc[vari],sep=""))
        #
      }
      #
      # Averages over selected domain 
      #
      filesave_meandom=paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/",varlist_urbinc[vari],"_MEANDOM",sep="")
      if (!file.exists(filesave_meandom)) {
        #
        load(file=paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/",varlist_urbinc[vari],sep=""))
        vals_day=LST$vals_day
        #
        ind_DOM=which((LONS[1:lenmask[count_nest],count_nest,sce]>lonmin_ts)&(LONS[1:lenmask[count_nest],count_nest,sce]<lonmax_ts)&
	              (LATS[1:lenmask[count_nest],count_nest,sce]>latmin_ts)&(LATS[1:lenmask[count_nest],count_nest,sce]<latmax_ts))
        #
        vals_day_sel =vals_day[ind_DOM,,drop=FALSE]
        vals_meanhour=colMeans(vals_day_sel,dims=1,na.rm=TRUE)
        #
        LST=list(vals_meanhour=vals_meanhour)
        save(LST,file=filesave_meandom)
        #
      }
      #
    }
  }
  #
  # End loop over nests
  #
  }
  #
  # End loop over days
  #
}
#
# ############################################################
# Aggregate data over selected time periods
# ############################################################
#
print("                                         ")
print("Aggregation over selected time intervals ")
print("                                         ")
#
# Select the model level for the wind vectors to plot
#
lvl_wivec=c()
#
for (agl in hagl_wivec) {
  #
  adiff=abs(zlvl-agl)
  amin=min(adiff)
  if (amin>10) {print(amin);stop("Too large difference between model level and requested wind level")}
  indmin=which(adiff==amin)
  lvl_wivec=c(lvl_wivec,indmin[1])
  #
}
#
# Loop over scenarios
#
for (nest in nest_vec) {
  #
  for (sce in 1:(length(scen)+1)) {
    #
    if (sce<=length(scen)) {
      pagg=paste(pdsup,name,"/",scen[sce],"/ONLINE/AGG_RDATA_Nest_",nest,"/",sep="")
      varlistagg=varplot
      perio_vec_act=perio_vec
      hourbeg_act=hourbeg
      hourend_act=hourend
    } else {
      pagg=paste(pdsup,name,"/URBINC/AGG_RDATA_Nest_",nest,"/",sep="")
      varlistagg=varlist_urbinc
      perio_vec_act=perio_vec
      hourbeg_act=hourbeg
      hourend_act=hourend
    }
    #
    # Add the wind components at model levels
    #
    for (wivec_lvl in lvl_wivec) {
      varlistagg=c(varlistagg,paste0("_LVL_",wivec_lvl,"_SPLIT_UMME"))
      varlistagg=c(varlistagg,paste0("_LVL_",wivec_lvl,"_SPLIT_VMME"))      
    }
    #
    if (lnew>0.5) {
      system(paste("rm -Rf",pagg))
      system(paste("mkdir" ,pagg))
    } else {
      system(paste("mkdir -p",pagg))
    }
    #
    for (vari in 1:length(varlistagg)) for (seas in 1:length(seasvec)) for (period in 1:length(perio_vec_act)) {
      #
      fileout=paste(pagg,varlistagg[vari],"_",seasvec[seas],"_",perio_vec_act[period],sep="")
      #
      # Use first day of season to test whether file exists
      #
      LST=mycaldat(julday_seasons[seas,1])
      montact=LST$m;dayyact=LST$d;yearact=LST$y
      #
      namemont=sprintf("%02d",montact)
      namedayy=sprintf("%02d",dayyact)
      nameday=paste(yearact,namemont,namedayy,sep="")
      #
      if (sce<=length(scen)) {
        fileinp=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/RDATA/",varlistagg[vari],sep="")
      } else {
        fileinp=paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/",varlistagg[vari],sep="")
      }
      #
      if ((!file.exists(fileout))&&(file.exists(fileinp))) {
        #
        if (sce<=length(scen)) {
          print(paste("Aggregation for ",scen[sce]," Nest ",nest," ",varlistagg[vari],"_",seasvec[seas],"_",perio_vec_act[period],sep=""))
        } else {
          print(paste("Aggregation for URBINC, Nest ",nest," ",varlistagg[vari],"_",seasvec[seas],"_",perio_vec_act[period],sep=""))
        }
        #
        for (dayind in na.omit(julday_seasons[seas,])) {
          #
          LST=mycaldat(dayind);montact=LST$m;dayyact=LST$d;yearact=LST$y
          #
          namemont=sprintf("%02d",montact)
          namedayy=sprintf("%02d",dayyact)
          nameday=paste(yearact,namemont,namedayy,sep="")
          #
          if (sce<=length(scen)) {
            load(file=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/RDATA/",varlistagg[vari],sep=""))
          } else {
            load(file=paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/",varlistagg[vari],sep=""))
          }
          #
          vals_day=LST$vals_day
          #
          if (dayind==julday_seasons[seas,1]) {
            DIM_LOAD=dim(vals_day)
            cursum=array(0.0,DIM_LOAD[1])
            count =0
          }
          #
          seldat=vals_day[,hourbeg_act[period]:hourend_act[period],drop=FALSE]
          #
          # For precipitation, the sum over the selected 
          # time period is taken, otherwise the mean is calculated
          #
          if (varlistagg[vari]=="PREC_HOURCUM") {
            meanper=rowSums(seldat,dims=1)
          } else {
          meanper=rowMeans(seldat,dims=1)
          }
          #
          cursum=cursum+meanper
          count =count +1
          #
        }
        #
        # For precipitation, the sum over the selected 
        # time period is taken, otherwise the mean is calculated
        #
        if (varlistagg[vari]=="PREC_HOURCUM") {
          fieldplot=cursum
        } else {
          fieldplot=cursum/count
        }
        #
        LST=list(fieldplot=fieldplot)
        save(LST,file=fileout)
        #
      }
    #
    }
  }
}
#
}
#
# ############################################################
# Calculate the average daily cycle of the station data
# over the selected time periods
# ############################################################
#
print("                                        ")
print("Calculation of daily cycle over seasons ")
print("                                        ")
#
count_nest=0
#
for (nest in nest_vec) {
#
count_nest=count_nest+1
#
for (sce in 1:length(scen)) {
#
pagg=paste(pdsup,name,"/",scen[sce],"/ONLINE/AGG_RDATA_Nest_",nest,"/",sep="")
#
if (!is.na(numb_surfstat[1])) {
#
# Merged SURF and CNRM stations (temperature and relative humidity)
#
sim_dcyc  = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)))
obs_dcyc  = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)))
bias_dcyc = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)))
rmse_dcyc = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)))
#
sim_surfstat_vals = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)*nmaxday))
obs_surfstat_vals = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)*nmaxday))
#
for (seas in 1:length(seasvec)) {
  #
  indbeg=1
  #
  for (dayind in na.omit(julday_seasons[seas,])) {
    #
    LST=mycaldat(dayind);montact=LST$m;dayyact=LST$d;yearact=LST$y
    #
    namemont=sprintf("%02d",montact)
    namedayy=sprintf("%02d",dayyact)
    nameday=paste(yearact,namemont,namedayy,sep="")
    #
    load(file=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/RDATA/SURF_EVAL_MEASURES",sep=""))
    obs_data_day=LST$obs_data_day;sim_data_day=LST$vals_surfstat
    #
    if (dayind==julday_seasons[seas,1]) {
      count      =array(0  ,dim(obs_data_day))
      count_theo =array(0  ,dim(obs_data_day))
      cur_obs_sum=array(0.0,dim(obs_data_day))
      cur_sim_sum=array(0.0,dim(obs_data_day))
      cur_bia_sum=array(0.0,dim(obs_data_day))
      cur_rms_sum=array(0.0,dim(obs_data_day))
    }
    #
    # Add new day (only if there are no missing observations)
    #
    for (vareval in 1:length(varieval_capitoul_surfstat)) for (stat in 1:length(numb_surfstat)) {
      #
      sim_surfstat_vals[seas,vareval,stat,indbeg:(indbeg-1+length(sim_data_day[vareval,stat,]))] = sim_data_day[vareval,stat,]
      obs_surfstat_vals[seas,vareval,stat,indbeg:(indbeg-1+length(obs_data_day[vareval,stat,]))] = obs_data_day[vareval,stat,]
      #
      for (nhour in 1:length(nout_time)) {
        #
        count_theo[vareval,stat,nhour]=count_theo[vareval,stat,nhour] + 1
        #
        if (!is.na(obs_data_day[vareval,stat,nhour])) {
          bias=sim_data_day[vareval,stat,nhour]-obs_data_day[vareval,stat,nhour]
          cur_obs_sum[vareval,stat,nhour]=cur_obs_sum[vareval,stat,nhour]+obs_data_day[vareval,stat,nhour]
          cur_sim_sum[vareval,stat,nhour]=cur_sim_sum[vareval,stat,nhour]+sim_data_day[vareval,stat,nhour]
          cur_bia_sum[vareval,stat,nhour]=cur_bia_sum[vareval,stat,nhour]+bias
          cur_rms_sum[vareval,stat,nhour]=cur_rms_sum[vareval,stat,nhour]+bias*bias
          count[vareval,stat,nhour]=count[vareval,stat,nhour] + 1
        }
      }
      #
    }
    #
    indbeg=indbeg+length(nout_time)
    #
  }
  #
  # Calculate the average over the given period
  # Only if obervation data are available for more then 80% of the hours
  #
  for (vareval in 1:length(varieval_capitoul_surfstat)) for (stat in 1:length(numb_surfstat)) for (nhour in 1:length(nout_time)) {
    if (count[vareval,stat,nhour]>(0.8*count_theo[vareval,stat,nhour])) {
      sim_dcyc  [seas,vareval,stat,nhour] = cur_sim_sum[vareval,stat,nhour]/count[vareval,stat,nhour]
      obs_dcyc  [seas,vareval,stat,nhour] = cur_obs_sum[vareval,stat,nhour]/count[vareval,stat,nhour]
      bias_dcyc [seas,vareval,stat,nhour] = cur_bia_sum[vareval,stat,nhour]/count[vareval,stat,nhour]
      rmse_dcyc [seas,vareval,stat,nhour] = sqrt(cur_rms_sum[vareval,stat,nhour]/count[vareval,stat,nhour])
    }
  }
  #
}
#
# Save results (one file for all seasons)
#
LST=list(sim_dcyc=sim_dcyc,obs_dcyc=obs_dcyc,bias_dcyc=bias_dcyc,rmse_dcyc=rmse_dcyc,
         sim_surfstat_vals=sim_surfstat_vals,obs_surfstat_vals=obs_surfstat_vals)
save(LST,file=paste(pagg,"SURF_DYCYC_AGG_SEAS",sep=""))
#
}
#
# CNRM stations (additional variables)
#
if (!is.na(lats_cnrmstat[1])) {
  #
  sim_dcyc =array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)))
  obs_dcyc =array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)))
  bias_dcyc=array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)))
  rmse_dcyc=array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)))
  #
  sim_cnrmstat_vals = array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)*nmaxday))
  obs_cnrmstat_vals = array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)*nmaxday))
  #
  sim_wiff=array(NA,c(length(seasvec),length(numb_cnrmstat),length(nout_time)*nmaxday))
  obs_wiff=array(NA,c(length(seasvec),length(numb_cnrmstat),length(nout_time)*nmaxday))
  sim_widd=array(NA,c(length(seasvec),length(numb_cnrmstat),length(nout_time)*nmaxday))
  obs_widd=array(NA,c(length(seasvec),length(numb_cnrmstat),length(nout_time)*nmaxday))
  #
  for (seas in 1:length(seasvec)) {
    #
    indbeg=1
    #
    for (dayind in na.omit(julday_seasons[seas,])) {
      #
      LST=mycaldat(dayind);montact=LST$m;dayyact=LST$d;yearact=LST$y
      #
      namemont=sprintf("%02d",montact)
      namedayy=sprintf("%02d",dayyact)
      nameday=paste(yearact,namemont,namedayy,sep="")
      #
      load(file=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/RDATA/CNRM_EVAL_MEASURES",sep=""))
      obs_data_day=LST$obs_data_day;sim_data_day=LST$vals_cnrmstat
      #
      if (dayind==julday_seasons[seas,1]) {
        count      =array(0  ,dim(obs_data_day))
        count_theo =array(0  ,dim(obs_data_day))
        cur_obs_sum=array(0.0,dim(obs_data_day))
        cur_sim_sum=array(0.0,dim(obs_data_day))
        cur_bia_sum=array(0.0,dim(obs_data_day))
        cur_rms_sum=array(0.0,dim(obs_data_day))
      }
      #
      # Add new day (only if there are no missing observations)
      #
      for (vareval in 1:length(varieval_capitoul_cnrmstat)) for (stat in 1:length(numb_cnrmstat)) {
        #
        sim_cnrmstat_vals[seas,vareval,stat,indbeg:(indbeg-1+length(sim_data_day[vareval,stat,]))] = sim_data_day[vareval,stat,]
        obs_cnrmstat_vals[seas,vareval,stat,indbeg:(indbeg-1+length(obs_data_day[vareval,stat,]))] = obs_data_day[vareval,stat,]
        #
        for (nhour in 1:length(nout_time)) {
          #
          count_theo[vareval,stat,nhour]=count_theo[vareval,stat,nhour] + 1
          if (!is.na(obs_data_day[vareval,stat,nhour])) {
            bias=sim_data_day[vareval,stat,nhour]-obs_data_day[vareval,stat,nhour]
            cur_obs_sum[vareval,stat,nhour]=cur_obs_sum[vareval,stat,nhour]+obs_data_day[vareval,stat,nhour]
            cur_sim_sum[vareval,stat,nhour]=cur_sim_sum[vareval,stat,nhour]+sim_data_day[vareval,stat,nhour]
            cur_bia_sum[vareval,stat,nhour]=cur_bia_sum[vareval,stat,nhour]+bias
            cur_rms_sum[vareval,stat,nhour]=cur_rms_sum[vareval,stat,nhour]+bias*bias
            count[vareval,stat,nhour]=count[vareval,stat,nhour] + 1
          }
          #
        }
        #
        # Extract the hourly wind speed and wind direction for the selected season
        #
        if (varieval_capitoul_cnrmstat[vareval]=="WIFFME") {
          sim_wiff[seas,stat,indbeg:(indbeg-1+length(sim_data_day[vareval,stat,]))]=sim_data_day[vareval,stat,]
          obs_wiff[seas,stat,indbeg:(indbeg-1+length(obs_data_day[vareval,stat,]))]=obs_data_day[vareval,stat,]
        }
        #
        if (varieval_capitoul_cnrmstat[vareval]=="WIDDME") {
          sim_widd[seas,stat,indbeg:(indbeg-1+length(sim_data_day[vareval,stat,]))]=sim_data_day[vareval,stat,]
          obs_widd[seas,stat,indbeg:(indbeg-1+length(obs_data_day[vareval,stat,]))]=obs_data_day[vareval,stat,]
        }
        #
      }
      #
      indbeg=indbeg+length(nout_time)
      #
    }
    #
    # Calculate the average over the given period
    # Only if obervation data are available for more then 80% of the hours
    #
    for (vareval in 1:length(varieval_capitoul_cnrmstat)) for (stat in 1:length(numb_cnrmstat)) for (nhour in 1:length(nout_time)) {
      #
      if (count[vareval,stat,nhour]>(0.8*count_theo[vareval,stat,nhour])) {
        #
        # For precipition, the sum over the selected period is taken.
        # The BIAS and RMSE are not defined
        #
        if (varieval_capitoul_cnrmstat[vareval]=="PREC_HOURCUM") {
          sim_dcyc  [seas,vareval,stat,nhour] = cur_sim_sum[vareval,stat,nhour]
          obs_dcyc  [seas,vareval,stat,nhour] = cur_obs_sum[vareval,stat,nhour]
          bias_dcyc [seas,vareval,stat,nhour] = NA
          rmse_dcyc [seas,vareval,stat,nhour] = NA
        } else {
          sim_dcyc  [seas,vareval,stat,nhour] = cur_sim_sum[vareval,stat,nhour]/count[vareval,stat,nhour]
          obs_dcyc  [seas,vareval,stat,nhour] = cur_obs_sum[vareval,stat,nhour]/count[vareval,stat,nhour]
          bias_dcyc [seas,vareval,stat,nhour] = cur_bia_sum[vareval,stat,nhour]/count[vareval,stat,nhour]
          rmse_dcyc [seas,vareval,stat,nhour] = sqrt(cur_rms_sum[vareval,stat,nhour]/count[vareval,stat,nhour])
        }
        #
      }
    }
    #
  }
  #
  # Save results (one file for all seasons)
  #
  LST=list(sim_dcyc=sim_dcyc,obs_dcyc=obs_dcyc,bias_dcyc=bias_dcyc,rmse_dcyc=rmse_dcyc,
           sim_wiff=sim_wiff,obs_wiff=obs_wiff,sim_widd=sim_widd,obs_widd=obs_widd,
           sim_cnrmstat_vals=sim_cnrmstat_vals,obs_cnrmstat_vals=obs_cnrmstat_vals)
  #
  save(LST,file=paste(pagg,"CNRM_DYCYC_AGG_SEAS",sep=""))
  #
}
#
# ######################################################
# Plot the daily cycles
# One plot per station, variable and season
# ######################################################
#
print("                                                       ")
print(paste("Plot the daily cycles for ",scen[sce],"Nest",nest))
print("                                                       ")
#
# Create plot output directory for daily cycles
#
system(paste("mkdir -p ",pplot,"DYCYC_Online/",sep=""))
system(paste("mkdir -p ",pplot,"DYCYC_Online/",name,"/",sep=""))
pdaycycle=paste(pplot,"DYCYC_Online/",name,"/",scen[sce],"_Nest_",nest,"/",sep="")
system(paste("mkdir -p",pdaycycle))
#
# Create plot output directory for PDF plots
#
system(paste("mkdir -p ",pplot,"PDFs_Online/",sep=""))
system(paste("mkdir -p ",pplot,"PDFs_Online/",name,"/",sep=""))
ppdfplots=paste(pplot,"PDFs_Online/",name,"/",scen[sce],"_Nest_",nest,"/",sep="")
system(paste("mkdir -p",ppdfplots))
#
# Create plot output directory for wind roses
#
system(paste("mkdir -p ",pplot,"WINDROSES_Online/",sep=""))
system(paste("mkdir -p ",pplot,"WINDROSES_Online/",name,"/",sep=""))
pwindrose=paste(pplot,"WINDROSES_Online/",name,"/",scen[sce],"_Nest_",nest,"/",sep="")
system(paste("mkdir -p",pwindrose))
#
nameleg    = c("Meso-NH","Station observation")
colors     = c("red","black")
ltys       = c(1,1)
data       = array(NA,c(1,1,length(nout_time),2))
incdateout = 1/length(nout_time)
#
# Merged SURF and CNRM stations
#
if (!is.na(lats_surfstat[1])) {
#
load(file=paste(pagg,"SURF_DYCYC_AGG_SEAS",sep=""))
sim_dcyc=LST$sim_dcyc;obs_dcyc=LST$obs_dcyc;bias_dcyc=LST$bias_dcyc;rmse_dcyc=LST$rmse_dcyc
sim_surfstat_vals=LST$sim_surfstat_vals;obs_surfstat_vals=LST$obs_surfstat_vals
#
sim_seasmean_surfstat = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(perio_vec)))
obs_seasmean_surfstat = array(NA,c(length(seasvec),length(varieval_capitoul_surfstat),length(numb_surfstat),length(perio_vec)))
#
for (vareval in 1:length(varieval_capitoul_surfstat)) for (stat in 1:length(numb_surfstat)) for (seas in 1:length(seasvec)) {
  #
  data[1,1,,1]=sim_dcyc[seas,vareval,stat,]
  data[1,1,,2]=obs_dcyc[seas,vareval,stat,]
  #
  # Store the seasonal mean at each station
  #
  for (period in 1:length(perio_vec)) {
    sim_seasmean_surfstat[seas,vareval,stat,period]=mean(sim_dcyc[seas,vareval,stat,hourbeg[period]:hourend[period]])
    obs_seasmean_surfstat[seas,vareval,stat,period]=mean(obs_dcyc[seas,vareval,stat,hourbeg[period]:hourend[period]])
  }
  #
  # Plot the avergage diurnal cycle
  #
  plotdaycy(data,incdateout,paste("EVAL_Station_",numb_surfstat[stat],sep=""),
            uniteval_capitoul_surfstat[vareval],nameeval_capitoul_surfstat[vareval],seasvec[seas],
            varieval_capitoul_surfstat[vareval],nameleg,colors,ltys,
            titlename=name_surfstat[stat],bias=mean(bias_dcyc[seas,vareval,stat,]),
            mae=mean(abs(bias_dcyc[seas,vareval,stat,])),
            unieval=unitbias_capitoul_surfstat[vareval],path=pdaycycle)
  #
  # Plot pdf
  #
  # obs_vec=obs_surfstat_vals[seas,vareval,stat,]
  # sim_vec=sim_surfstat_vals[seas,vareval,stat,]
  #
  # obs_vec[which(is.na(sim_vec))]=NA
  # sim_vec[which(is.na(obs_vec))]=NA
  #
  # mypdfplot(obs_vec,sim_vec,
  #           paste("PDF_HOURLY_Station_",numb_surfstat[stat],"_",varieval_capitoul_surfstat[vareval],"_",seasvec[seas],sep=""),
  #           titlename=name_surfstat[stat],varname=nameeval_capitoul_surfstat[vareval],
  #           uniname=uniteval_capitoul_surfstat[vareval],path=ppdfplots)
  #
}
}
#
# CNRM stations
#
if (!is.na(lats_cnrmstat[1])) {
#
load(file=paste(pagg,"CNRM_DYCYC_AGG_SEAS",sep=""))
sim_dcyc=LST$sim_dcyc;obs_dcyc=LST$obs_dcyc;bias_dcyc=LST$bias_dcyc;rmse_dcyc=LST$rmse_dcyc
sim_wiff=LST$sim_wiff;obs_wiff=LST$obs_wiff;sim_widd=LST$sim_widd;obs_widd=LST$obs_widd
sim_cnrmstat_vals=LST$sim_cnrmstat_vals;obs_cnrmstat_vals=LST$obs_cnrmstat_vals
#
sim_seasmean_cnrmstat = array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(perio_vec)))
obs_seasmean_cnrmstat = array(NA,c(length(seasvec),length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(perio_vec)))
#
for (stat in 1:length(numb_cnrmstat)) for (seas in 1:length(seasvec)) {
  #
  # Diurnal cycle for all variables
  #
  for (vareval in 1:length(varieval_capitoul_cnrmstat)) {
    #
    data[1,1,,1]=sim_dcyc[seas,vareval,stat,]
    data[1,1,,2]=obs_dcyc[seas,vareval,stat,]
    #
    # For precipitation, the sum is taken
    #
    if (varieval_capitoul_cnrmstat[vareval]=="PREC_HOURCUM") {
      for (period in 1:length(perio_vec)) {
        sim_seasmean_cnrmstat[seas,vareval,stat,period]=sum(sim_dcyc[seas,vareval,stat,hourbeg[period]:hourend[period]])
        obs_seasmean_cnrmstat[seas,vareval,stat,period]=sum(obs_dcyc[seas,vareval,stat,hourbeg[period]:hourend[period]])
      }
    } else {
      for (period in 1:length(perio_vec)) {
        sim_seasmean_cnrmstat[seas,vareval,stat,period]=mean(sim_dcyc[seas,vareval,stat,hourbeg[period]:hourend[period]])
        obs_seasmean_cnrmstat[seas,vareval,stat,period]=mean(obs_dcyc[seas,vareval,stat,hourbeg[period]:hourend[period]])
      }
    }
    #
    plotdaycy(data,incdateout,paste("EVAL_Station_",numb_cnrmstat[stat],sep=""),
              uniteval_capitoul_cnrmstat[vareval],nameeval_capitoul_cnrmstat[vareval],seasvec[seas],
              varieval_capitoul_cnrmstat[vareval],nameleg,colors,ltys,
              titlename=name_cnrmstat[stat],bias=mean(bias_dcyc[seas,vareval,stat,]),
              mae=mean(abs(bias_dcyc[seas,vareval,stat,])),
              unieval=uniteval_capitoul_cnrmstat[vareval],path=pdaycycle)
    #
  }
  #
  # Wind roses
  #
  # Extract hourly values for season and station
  #
  sim_wiff_ext = sim_wiff[seas,stat,]
  obs_wiff_ext = obs_wiff[seas,stat,]
  sim_widd_ext = sim_widd[seas,stat,]
  obs_widd_ext = obs_widd[seas,stat,]
  #
  # Assure mutual NA
  # - FF and DD
  # - obs and sim
  #
  obs_wiff_ext[which(is.na(obs_widd_ext))]=NA
  obs_widd_ext[which(is.na(obs_wiff_ext))]=NA
  sim_wiff_ext[which(is.na(sim_widd_ext))]=NA
  sim_widd_ext[which(is.na(sim_wiff_ext))]=NA
  #
  obs_wiff_ext[which(is.na(sim_wiff_ext))]=NA
  obs_widd_ext[which(is.na(sim_widd_ext))]=NA
  sim_wiff_ext[which(is.na(obs_wiff_ext))]=NA
  sim_widd_ext[which(is.na(obs_widd_ext))]=NA
  #
  # Remove NA
  #
  obs_wiff_ext=na.omit(obs_wiff_ext)
  obs_widd_ext=na.omit(obs_widd_ext)
  sim_wiff_ext=na.omit(sim_wiff_ext)
  sim_widd_ext=na.omit(sim_widd_ext)
  #
  obs_wiff_ext=na.omit(obs_wiff_ext)
  obs_widd_ext=na.omit(obs_widd_ext)
  sim_wiff_ext=na.omit(sim_wiff_ext)
  sim_widd_ext=na.omit(sim_widd_ext)
  #
  # Check for consistent lengths
  #
  if (length(obs_wiff_ext)!=length(sim_wiff_ext)) stop("Inconsistent lengths of wind variables")
  if (length(obs_widd_ext)!=length(sim_widd_ext)) stop("Inconsistent lengths of wind variables")
  if (length(obs_wiff_ext)!=length(obs_widd_ext)) stop("Inconsistent lengths of wind variables")
  if (length(sim_wiff_ext)!=length(sim_widd_ext)) stop("Inconsistent lengths of wind variables")
  #
  # Plot wind rose for model and station data
  #
  if ((length(obs_wiff_ext)>100)&&(length(sim_wiff_ext)>100)) {
    #
    nameplot=paste(pwindrose,"Station_",numb_cnrmstat[stat],"_",seasvec[seas],"_OBS.pdf",sep="")
    windrose(spd=obs_wiff_ext,dir=obs_widd_ext,nameplot=nameplot)
    #
    nameplot=paste(pwindrose,"Station_",numb_cnrmstat[stat],"_",seasvec[seas],"_SIM.pdf",sep="")
    windrose(spd=sim_wiff_ext,dir=sim_widd_ext,nameplot=nameplot)
    #
  }
  #
}
}
#
# Plot the seasonal means at the stations against the building fraction
#
nameleg = c("Stations","Meso-NH")
colors  = c("black","red")
ltys    = c(1,1)
#
# SURF Stations
#
if (!is.na(lats_surfstat[1])) {
#
# Create directory for stationeval
#
pbase=paste(pstat,name,sep="")
system(paste("mkdir -p ",pbase))
#
pbase=paste(pstat,name,"/",scen[sce],"_Nest_",nest,sep="")
system(paste("mkdir -p ",pbase))
#
load(file=filesurfstationindex)
surfstat_index=LST$surfstat_index
#
if (exists("BLDFRA")) {
  #
  bldf_model=array(NA,length(lats_surfstat))
  heig_model=array(NA,length(lats_surfstat))
  waoh_model=array(NA,length(lats_surfstat))
  #
  print("Station, bld, heig, wall_o_hor")
  #
  for (stat in 1:length(lats_surfstat)) {
    #
    ind_stat=na.omit(surfstat_index[stat,])
    #
    bldf_model[stat]=mean(FRAC_TOWN[ind_stat,count_nest,sce]*BLDFRA[ind_stat,count_nest,sce])
    heig_model[stat]=mean(BLDHEI[ind_stat,count_nest,sce])
    waoh_model[stat]=mean(FRAC_TOWN[ind_stat,count_nest,sce]*WALLOH[ind_stat,count_nest,sce])
    #
    print(paste(numb_surfstat[stat],bldf_model[stat],heig_model[stat],waoh_model[stat]))
    #
  }
  #
  # Write information on building density and wall surface density into file
  #
  LST=list(numb_surfstat=numb_surfstat,bldf_model=bldf_model,heig_model=heig_model,waoh_model=waoh_model)
  #
  pbase=paste(pevalm,name,sep="")
  system(paste("mkdir -p ",pbase))
  #
  pbase=paste(pevalm,name,"/",scen[sce],sep="")
  system(paste("mkdir -p ",pbase))
  #
  pbase=paste(pevalm,name,"/",scen[sce],"/Nest_",nest,sep="")
  system(paste("mkdir -p ",pbase))
  #
  filesave=paste(pbase,"/PGD_FIELDS_SURF",sep="")
  save(LST,file=filesave)
  #
} else {
  bldf_model=array(0.0,length(lats_surfstat))
  heig_model=array(0.0,length(lats_surfstat))
  waoh_model=array(0.0,length(lats_surfstat))
}
#
}
#
# CNRM Stations
#
if (!is.na(lats_cnrmstat[1])) {
#
load(file=filecnrmstationindex)
cnrmstat_index=LST$cnrmstat_index
#
if (exists("BLDFRA")) {
  #
  bldf_model=array(NA,length(lats_cnrmstat))
  heig_model=array(NA,length(lats_cnrmstat))
  waoh_model=array(NA,length(lats_cnrmstat))
  #
  for (stat in 1:length(lats_cnrmstat)) {
    #
    ind_stat=na.omit(cnrmstat_index[stat,])
    #
    bldf_model[stat]=mean(FRAC_TOWN[ind_stat,count_nest,sce]*BLDFRA[ind_stat,count_nest,sce])
    heig_model[stat]=mean(BLDHEI[ind_stat,count_nest,sce])
    waoh_model[stat]=mean(FRAC_TOWN[ind_stat,count_nest,sce]*WALLOH[ind_stat,count_nest,sce])
    #
    print(paste(numb_cnrmstat[stat],bldf_model[stat],heig_model[stat],waoh_model[stat]))
    #
  }
  #
  # Write information on building density and wall surface density into file
  #
  LST=list(numb_cnrmstat=numb_cnrmstat,bldf_model=bldf_model,heig_model=heig_model,waoh_model=waoh_model)
  #
  pbase=paste(pevalm,name,sep="")
  system(paste("mkdir -p ",pbase))
  #
  pbase=paste(pevalm,name,"/",scen[sce],sep="")
  system(paste("mkdir -p ",pbase))
  #
  pbase=paste(pevalm,name,"/",scen[sce],"/Nest_",nest,sep="")
  system(paste("mkdir -p ",pbase))
  #
  filesave=paste(pbase,"/PGD_FIELDS_CNRM",sep="")
  save(LST,file=filesave)
  #
} else {
  bldf_model=array(0.0,length(cnrmstat_index))
  heig_model=array(0.0,length(cnrmstat_index))
  waoh_model=array(0.0,length(cnrmstat_index))
}
#
}
#
# END of part that is only executed when MESO-NH output is available
#
}
}
#
# ############################################################
# Plot time series
# ############################################################
#
if ((ndayend_ts-ndaybeg_ts)>0) {
  #
  print("  ")
  print("Plot the time series")
  print("  ")
  #
  # Create directory for time series of daily values
  #
  system(paste("mkdir -p ",pplot,"TSDAY_Online/",sep=""))
  system(paste("mkdir -p ",pplot,"TSDAY_Online/",name,"/",sep=""))
  #
  # Plot time series of model output and observation data
  #
  for (sce in 1:length(scen)) for (nest in nest_vec) {
    #
    # Create directories for scenarios
    #
    ptsday=paste(pplot,"TSDAY_Online/",name,"/",scen[sce],"_Nest_",nest,"/",sep="")
    system(paste("mkdir -p",ptsday))
    #
    ptshour=paste(pplot,"TSHOUR_Online/",name,"/",scen[sce],"_Nest_",nest,"/",sep="")
    system(paste("mkdir -p",ptshour))
    #
    ppdfplots=paste(pplot,"PDFs_Online/",name,"/",scen[sce],"_Nest_",nest,"/",sep="")
    system(paste("mkdir -p",ppdfplots))
    #
    juldayseries=array(NA,ndayend_ts-ndaybeg_ts+1)
    qanttotal   =array(NA,ndayend_ts-ndaybeg_ts+1)
    qantbuild   =array(NA,ndayend_ts-ndaybeg_ts+1)
    #
    bias_day_surfts = array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat),ndayend_ts-ndaybeg_ts+1))
    rmse_day_surfts = array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat),ndayend_ts-ndaybeg_ts+1))
    obsd_day_surfts = array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat),ndayend_ts-ndaybeg_ts+1))
    simd_day_surfts = array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat),ndayend_ts-ndaybeg_ts+1))
    #
    bias_day_cnrmts = array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),ndayend_ts-ndaybeg_ts+1))
    rmse_day_cnrmts = array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),ndayend_ts-ndaybeg_ts+1))
    obsd_day_cnrmts = array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),ndayend_ts-ndaybeg_ts+1))
    simd_day_cnrmts = array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),ndayend_ts-ndaybeg_ts+1))
    #
    # Vectors with hourly model and simulation results for evaluation of hourly values
    #
    obs_hourly_surfts = array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)*(ndayend_ts-ndaybeg_ts+1)))
    sim_hourly_surfts = array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat),length(nout_time)*(ndayend_ts-ndaybeg_ts+1)))
    obs_hourly_cnrmts = array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)*(ndayend_ts-ndaybeg_ts+1)))
    sim_hourly_cnrmts = array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat),length(nout_time)*(ndayend_ts-ndaybeg_ts+1)))
    julhour           = array(NA,c(length(nout_time)*(ndayend_ts-ndaybeg_ts+1)))
    #
    count        = 0
    count_hourly = 1
    #
    for (day in ndaybeg_ts:ndayend_ts) {
      #
      count=count+1
      #
      juldayact=juldaybeg+(day-1)
      LST=mycaldat(juldayact)
      montact=LST$m;dayyact=LST$d;yearact=LST$y
      #
      juldayseries[count]=juldayact
      #
      julhour[(1+24*(count-1)):(count*24)]=seq(juldayseries[count],juldayseries[count]+1.0-1.0/24.0,1.0/24.0)
      #
      namemont=sprintf("%02d",montact)
      namedayy=sprintf("%02d",dayyact)
      nameday=paste(yearact,namemont,namedayy,sep="")
      #
      prsv=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/RDATA/",sep="")
      #
      # Time series of anthropogenic heat release over CAPITOUL domain
      #
      if ((scen[sce]=="REFER")&&(flag_capitoul>0.5)) {
        #
        load(file=paste(prsv,"QANT_TOTAL_CAPITOUL",sep=""))
        qanttotal[count]=mean(LST$qanttotal_meanhour)
        #
        load(file=paste(prsv,"QANT_BUILDINGS_CAPITOUL",sep=""))
        qantbuild[count]=mean(LST$qantbuild_meanhour)
        #
      }
      #
      # Time series of daily average values
      # and evaluation measures based on the hourly data
      #
      # SURF stations
      #
      if (!is.na(lats_surfstat[1])) {
        #
        load(file=paste(prsv,"SURF_EVAL_MEASURES",sep=""))
        obs_data_day=LST$obs_data_day;sim_data_day=LST$vals_surfstat
        #
        bias_day_surfts[,,count]=LST$bias_day
        rmse_day_surfts[,,count]=LST$rmse_day
        #
        for (metvar in 1:length(varieval_capitoul_surfstat)) for (stat in 1:length(numb_surfstat)) {
          ind_not_na=which(!is.na(obs_data_day[metvar,stat,]))
          if (length(ind_not_na)>18) {
            obsd_day_surfts[metvar,stat,count] = mean(obs_data_day[metvar,stat,ind_not_na])
            simd_day_surfts[metvar,stat,count] = mean(sim_data_day[metvar,stat,ind_not_na])
          }
          obs_hourly_surfts[metvar,stat,count_hourly:(count_hourly+length(nout_time)-1)] = obs_data_day[metvar,stat,]
          sim_hourly_surfts[metvar,stat,count_hourly:(count_hourly+length(nout_time)-1)] = sim_data_day[metvar,stat,]
        }
        #
      }
      #
      # CNRM stations
      #
      if (!is.na(lats_cnrmstat[1])) {
        #
        load(file=paste(prsv,"CNRM_EVAL_MEASURES",sep=""))
        obs_data_day=LST$obs_data_day
        sim_data_day=LST$vals_cnrmstat
        #
        for (metvar in 1:length(varieval_capitoul_cnrmstat)) {
          #
          if (varieval_capitoul_cnrmstat[metvar]=="PREC_HOURCUM") {
            bias_day_cnrmts[,,count]=NA
            rmse_day_cnrmts[,,count]=NA
          } else {
            bias_day_cnrmts[,,count]=LST$bias_day
            rmse_day_cnrmts[,,count]=LST$rmse_day
          }
          #
          for (stat in 1:length(numb_cnrmstat)) {
            #
            ind_not_na=which(!is.na(obs_data_day[metvar,stat,]))
            if (length(ind_not_na)>18) {
              if (varieval_capitoul_cnrmstat[metvar]=="PREC_HOURCUM") {
                obsd_day_cnrmts[metvar,stat,count] = sum(obs_data_day[metvar,stat,ind_not_na])
                simd_day_cnrmts[metvar,stat,count] = sum(sim_data_day[metvar,stat,ind_not_na])
              } else {
                obsd_day_cnrmts[metvar,stat,count] = mean(obs_data_day[metvar,stat,ind_not_na])
                simd_day_cnrmts[metvar,stat,count] = mean(sim_data_day[metvar,stat,ind_not_na])
              }
            }
            obs_hourly_cnrmts[metvar,stat,count_hourly:(count_hourly+length(nout_time)-1)] = obs_data_day[metvar,stat,]
            sim_hourly_cnrmts[metvar,stat,count_hourly:(count_hourly+length(nout_time)-1)] = sim_data_day[metvar,stat,]
          }
        }
      #
      }
      #
      count_hourly=count_hourly+length(nout_time)
      #
    }
    #
    # Plot time series of hourly values (CNRM stations)
    #
    if (!is.na(lats_cnrmstat[1])) {
      #
      for (metvar in 1:length(varieval_capitoul_cnrmstat)) for (stat in 1:length(numb_cnrmstat)) {
        #
        nameleg = c("Stations","Meso-NH")
        colors  = c("black","red")
        ltys    = c(1,1)
        #
	data=array(NA,c(1,length(sim_hourly_cnrmts[metvar,stat,]),2))
	#
        data[1,,1]=obs_hourly_cnrmts[metvar,stat,]
        data[1,,2]=sim_hourly_cnrmts[metvar,stat,]
        #
        err  = data[1,,2]-data[1,,1]
	bias = mean(err,na.rm=TRUE)
        mae  = mean(abs(err),na.rm=TRUE)
        rmse = sqrt(mean(err*err,na.rm=TRUE))
        #
        plottshour(data,julhour,paste0("EVAL_Station_",numb_cnrmstat[stat]),
                   uniteval_capitoul_cnrmstat[metvar],nameeval_capitoul_cnrmstat[metvar],
                   varieval_capitoul_cnrmstat[metvar],nameleg,colors,ltys,bias=bias,
	           rmse=rmse,titlename=name_cnrmstat[stat],path=ptshour)
        #
      }
      #
    }
    #
    # Load observation based data of anthropogenic heat
    # Only full year of CAPITOUL period
    #
    if ((nrunday>=365)&&(flag_capitoul>0.5)) {
      #
      load(file="/home/schoetterr/Bureau/SURFEX/CAPITOUL_OBS/qf/Processed_data/City/RSAVE/ALL_2D_CAPITOUL")
      #
      julobs=LST$datejults;tra_daya_obs=LST$traconday;gas_daya_obs=LST$gasconday
      ele_daya_obs=LST$eleconday;otr_daya_obs=LST$otrconday;tot_daya_obs=LST$totconday
      gas_dcyc_obs=LST$gas_dcyc;ele_dcyc_obs=LST$ele_dcyc;tra_dcyc_obs=LST$tra_dcyc
      heat_dcyc_obs=LST$heat_dcyc;qinh_dcyc_obs=LST$qinh_dcyc
      gascondayseas=LST$gascondayseas;elecondayseas=LST$elecondayseas
      #
      # Conversion factor GWd*d -> W/m² (15x15 km² domain)
      #
      cfl=1.0e-9*(15000*15000)
      #
      # Create plot with time series of anthropogenic heat release
      #
      nameleg=c("Simulated, non traffic","Inventory, non traffic")
      colors =c("red","black")
      ltys   =c(1,1)
      data   =array(NA,c(1,365,2))
      data[1,,1]=cfl*qantbuild
      data[1,,2]=(tot_daya_obs-tra_daya_obs)
      #
      err  = data[1,,2]-data[1,,1]
      mae  = mean(abs(err))
      rmse = sqrt(mean(err*err))
      #
      plottsdav(data,julobs,"EVAL_QA_NONTRAFFIC",uni="Gwd","Energy release",
                name,nameleg,colors,ltys,path=ptsday,mae=mae,rmse=rmse)
      #  
    }
    #
    # Save extracted time series for further plotting
    #
    LST=list(juldayseries=juldayseries,numb_surfstat=numb_surfstat,numb_cnrmstat=numb_cnrmstat,
             obsd_day_surfts=obsd_day_surfts,
             simd_day_surfts=simd_day_surfts,obsd_day_cnrmts=obsd_day_cnrmts,
             simd_day_cnrmts=simd_day_cnrmts,obs_hourly_surfts=obs_hourly_surfts,
             sim_hourly_surfts=sim_hourly_surfts,obs_hourly_cnrmts=obs_hourly_cnrmts,
             sim_hourly_cnrmts=sim_hourly_cnrmts)
    #
    save(LST,file=paste(pdsup,name,"/",scen[sce],"/ONLINE/SAVE_TS_Nest_",nest,sep=""))
    #
    # Calculation of standard evaluation measures for hourly values 
    # These are: BIAS, RMSE, HITR and SKVAR
    # The desired accuracy is chosen according to Cox et al. (1998)
    # to be ±2 K for air temperature, ±1 g/kg for specific humidity,
    # ±1 m/s for wind speed, ±30° for wind direction and ±1.7 hPa for pressure
    #
    # SURF stations
    #
    if (!is.na(lats_surfstat[1])) {
      #
      bias_hourly_surf=array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat)))
      rmse_hourly_surf=array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat)))
      hitr_hourly_surf=array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat)))
      skva_hourly_surf=array(NA,c(length(varieval_capitoul_surfstat),length(numb_surfstat)))
      #
      for (metvar in 1:length(varieval_capitoul_surfstat)) {
        if (varieval_capitoul_surfstat[metvar]=="TREAL") {
          des_acc=2.0
        } else if (varieval_capitoul_surfstat[metvar]=="RELHU") {
          des_acc=5.0
        } else if (varieval_capitoul_surfstat[metvar]=="QSPEC") {
          des_acc=1.0
        } else {
          print(varieval_capitoul_surfstat[metvar])
          stop("No rule for this SURSTAT variable")
        }
        for (stat in 1:length(numb_surfstat)) {
          LST=secalceval(obs_hourly_surfts[metvar,stat,],sim_hourly_surfts[metvar,stat,],des_acc)
          bias_hourly_surf[metvar,stat]=LST$bias
          rmse_hourly_surf[metvar,stat]=LST$rmse
          hitr_hourly_surf[metvar,stat]=LST$hitr
          skva_hourly_surf[metvar,stat]=LST$skva
        }
      }
    }
    #
    # CNRM stations
    #
    if (!is.na(lats_cnrmstat[1])) {
      bias_hourly_cnrm=array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat)))
      rmse_hourly_cnrm=array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat)))
      hitr_hourly_cnrm=array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat)))
      skva_hourly_cnrm=array(NA,c(length(varieval_capitoul_cnrmstat),length(numb_cnrmstat)))
      #
      for (metvar in 1:length(varieval_capitoul_cnrmstat)) {
        if (varieval_capitoul_cnrmstat[metvar]=="TEMPMME") {
          des_acc=2.0
        } else if (varieval_capitoul_cnrmstat[metvar]=="RHME") {
          des_acc=5.0
        } else if (varieval_capitoul_cnrmstat[metvar]=="QSPECME") {
          des_acc=1.0
        } else if (varieval_capitoul_cnrmstat[metvar]=="PREC_HOURCUM") {
          des_acc=1.0
        } else if (varieval_capitoul_cnrmstat[metvar]=="TOTFLASWD_SUM") {
          des_acc=50.0
        } else if (varieval_capitoul_cnrmstat[metvar]=="WIFFME") {
          des_acc=1.0
        } else if (varieval_capitoul_cnrmstat[metvar]=="WIDDME") {
          des_acc=30.0
        } else if (varieval_capitoul_cnrmstat[metvar]=="FLALWD") {
          des_acc=30.0	  
        } else {
          print(varieval_capitoul_cnrmstat[metvar])
          stop("No rule for this CNRMSTAT variable")
        }
        for (stat in 1:length(numb_cnrmstat)) {
          LST=secalceval(obs_hourly_cnrmts[metvar,stat,],sim_hourly_cnrmts[metvar,stat,],des_acc)
          bias_hourly_cnrm[metvar,stat]=LST$bias
          rmse_hourly_cnrm[metvar,stat]=LST$rmse
          hitr_hourly_cnrm[metvar,stat]=LST$hitr
          skva_hourly_cnrm[metvar,stat]=LST$skva
        }
      }
    }
    #
    # Write results for evaluation measures in file
    #
    # SURF stations
    #
    if (!is.na(lats_surfstat[1])) {
      #
      fileevaldat=paste(pstat,name,"/",scen[sce],"_Nest_",nest,"/",sep="")
      system(paste("mkdir -p",fileevaldat))
      #
      for (metvar in 1:length(varieval_capitoul_surfstat)) {
        #
        fileevaldat=paste(pstat,name,"/",scen[sce],"_Nest_",nest,"/EVAL_",varieval_capitoul_surfstat[metvar],".txt",sep="")
        system(paste("rm -f",fileevaldat))
        #
        write("                                      ",file=fileevaldat,append=TRUE,sep="")
        write("Station   Bias   RMSE   HITR   SKVAR  ",file=fileevaldat,append=TRUE,sep="")
        #
        for (stat in 1:length(numb_surfstat)) {
          write(paste(format(numb_surfstat[stat],digits=3,width=8),
                      format(bias_hourly_surf[metvar,stat],digits=3,width=8),
                      format(rmse_hourly_surf[metvar,stat],digits=3,width=8),
                      format(hitr_hourly_surf[metvar,stat],digits=3,width=8),
                      format(skva_hourly_surf[metvar,stat],digits=3,width=8),sep=""),
		      file=fileevaldat,append=TRUE)
        }
        #
      }
    }
    #
    if (!is.na(lats_cnrmstat[1])) {
      #
      fileevaldat=paste(pstat,name,"/",scen[sce],"_Nest_",nest,"/",sep="")
      system(paste("mkdir -p",fileevaldat))
      #
      for (metvar in 1:length(varieval_capitoul_cnrmstat)) {
        #
        fileevaldat=paste(pstat,name,"/",scen[sce],"_Nest_",nest,"/EVAL_",varieval_capitoul_cnrmstat[metvar],".txt",sep="")
        system(paste("rm -f",fileevaldat))
        #
        write("                                      ",file=fileevaldat,append=TRUE,sep="")
        write("Station   Bias   RMSE   HITR   SKVAR  ",file=fileevaldat,append=TRUE,sep="")
        #
        for (stat in 1:length(numb_cnrmstat)) {
          write(paste(format(numb_cnrmstat[stat],digits=3,width=8),
                      format(bias_hourly_cnrm[metvar,stat],digits=3,width=8),
                      format(rmse_hourly_cnrm[metvar,stat],digits=3,width=8),
                      format(hitr_hourly_cnrm[metvar,stat],digits=3,width=8),
                      format(skva_hourly_cnrm[metvar,stat],digits=3,width=8),sep=""),
		      file=fileevaldat,append=TRUE)
        }   
      }
    }
    #
    # Create plots with time series of simulated and observed values 
    # at the locations of the different stations 
    #
    nameleg=c("Meso-NH","Station observation")
    colors =c("red","black")
    ltys   =c(1,1)
    data   =array(NA,c(1,(ndayend_ts-ndaybeg_ts+1),2))
    #
    # SURF stations
    #
    if (!is.na(lats_surfstat[1])) {
      #
      for (metvar in 1:length(varieval_capitoul_surfstat)) for (stat in 1:length(numb_surfstat)) {
        #
        data[1,,1]=simd_day_surfts[metvar,stat,]
        data[1,,2]=obsd_day_surfts[metvar,stat,]
        #
        # Time series of daily averages
        #
	err  = data[1,,2]-data[1,,1]
        mae  = mean(abs(err),na.rm=TRUE)
        rmse = sqrt(mean(err*err,na.rm=TRUE))
        #
        plottsdav(data,juldayseries,paste("EVAL_Station_",numb_surfstat[stat],sep=""),
                  uniteval_capitoul_surfstat[metvar],nameeval_capitoul_surfstat[metvar],
                  varieval_capitoul_surfstat[metvar],nameleg,colors,ltys,
		  mae=mae,rmse=rmse,
                  titlename=name_surfstat[stat],path=ptsday)
       #
       # PDF Plot of daily averages
       #
       mypdfplot(obsd_day_surfts[metvar,stat,],simd_day_surfts[metvar,stat,],
                 paste("PDF_DAILY_Station_",numb_surfstat[stat],"_",varieval_capitoul_surfstat[metvar],sep=""),
                 titlename=name_surfstat[stat],varname=nameeval_capitoul_surfstat[metvar],
                 uniname=uniteval_capitoul_surfstat[metvar],path=ppdfplots)
       #
     }
   }
   #
   # CNRM stations
   #
   if (!is.na(lats_cnrmstat[1])) {
     #
     for (metvar in 1:length(varieval_capitoul_cnrmstat)) for (stat in 1:length(numb_cnrmstat)) {
       #
       data[1,,1]=simd_day_cnrmts[metvar,stat,]
       data[1,,2]=obsd_day_cnrmts[metvar,stat,]
       #
       err  = data[1,,2]-data[1,,1]
       mae  = mean(abs(err),na.rm=TRUE)
       rmse = sqrt(mean(err*err,na.rm=TRUE))
       #
       plottsdav(data,juldayseries,paste("EVAL_Station_",numb_cnrmstat[stat],sep=""),
                 uniteval_capitoul_cnrmstat[metvar],nameeval_capitoul_cnrmstat[metvar],
                 varieval_capitoul_cnrmstat[metvar],nameleg,colors,ltys,
		 mae=mae,rmse=rmse,
                 titlename=name_cnrmstat[stat],path=ptsday)
       #
       mypdfplot(obsd_day_cnrmts[metvar,stat,],simd_day_cnrmts[metvar,stat,],
                 paste("PDF_DAILY_Station_",numb_cnrmstat[stat],"_",varieval_capitoul_cnrmstat[metvar],sep=""),
                 titlename=name_cnrmstat[stat],varname=nameeval_capitoul_cnrmstat[metvar],
                 uniname=uniteval_capitoul_cnrmstat[metvar],path=ppdfplots)
       #
     }
   }
   #
  }
  #
  # Plot time series of model output averaged over a selected subdomain
  #
  lplotav=0.0
  #
  if (lplotav>0.5) {
  #
  for (sce in 1:(length(scen)+1)) for (nest in nest_vec) {
    #
    # Create directories for scenarios
    #
    if (sce <= length(scen)) {
      print(paste("Plotting average time series for ",scen[sce]))
      ptsday=paste(pplot,"TSDAY_Online/"  ,name,"/",scen[sce],"_Nest_",nest,"/",sep="")
      ppdfplots=paste(pplot,"PDFs_Online/",name,"/",scen[sce],"_Nest_",nest,"/",sep="")
      var_ts=varplot
    } else {
      print(paste("Plotting average time series for URBINC"))
      ptsday   =paste(pplot,"TSDAY_Online/",name,"/URBINC_Nest_",nest,"/",sep="")
      ppdfplots=paste(pplot,"PDFs_Online/" ,name,"/URBINC_Nest_",nest,"/",sep="")
      var_ts=varlist_urbinc
    }
    #
    system(paste("mkdir -p",ptsday))
    system(paste("mkdir -p",ppdfplots))
    #
    # Load data
    #
    vals_day_domts = array(NA,c(length(var_ts),ndayend_ts-ndaybeg_ts+1))
    juldayseries   = array(NA,ndayend_ts-ndaybeg_ts+1)
    #
    count = 0
    for (day in ndaybeg_ts:ndayend_ts) {
      #
      count=count+1
      #
      juldayact=juldaybeg+(day-1)
      LST=mycaldat(juldayact)
      montact=LST$m;dayyact=LST$d;yearact=LST$y
      juldayseries[count]=juldayact
      #
      namemont=sprintf("%02d",montact)
      namedayy=sprintf("%02d",dayyact)
      nameday=paste(yearact,namemont,namedayy,sep="")
      #
      if (sce <= length(scen)) {
        prsv=paste(pdsup,name,"/",scen[sce],"/ONLINE/",nameday,"_Nest_",nest,"/RDATA/",sep="")
      } else {
        prsv=paste(pdsup,name,"/URBINC/",nameday,"_Nest_",nest,"/RDATA/",sep="")
      }
      #
      for (vari in 1:length(var_ts)) {
        if (file.exists(paste(prsv,var_ts[vari],"_MEANDOM",sep=""))) {
          load(file=paste(prsv,var_ts[vari],"_MEANDOM",sep=""))
          vals_day_domts[vari,count]=mean(LST$vals_meanhour)
        }
      }
      #
    }
    #
    # Create plots with time series of domain averaged values
    #
    nameleg=c("Meso-NH")
    colors =c("black")
    ltys   =c(1)
    data   =array(NA,c(1,(ndayend_ts-ndaybeg_ts+1),1))
    #
    for (vari in 1:length(var_ts)) {
      #
      data[1,,1]=vals_day_domts[vari,]
      #
      plottsdav(data,juldayseries,paste("TS_",var_ts[vari],sep=""),
                "","","",nameleg,colors,ltys,titlename="",path=ptsday)
     #
    }
    #
  }
  #
  } else {
    print("                                           ")
    print("No average time series plotted to save time")
    print("                                           ")
  }
  # 
}
#
# ##################################################################
# Evaluation of the surface energy budget 
# in a 500 m radius around the CAPITOUL mast
# Only finest nest, reference scenario
# ##################################################################
#
if ((nrunday>360)&(flag_capitoul>0.5)) {
  #
  print("                                                 ")
  print("Evaluation of fluxes at CAPITOUL mast, only REFER")
  print("                                                 ")
  #
  pagg=paste(pdsup,name,"/REFER/ONLINE/AGG_RDATA_Nest_",nest_vec[length(nest_vec)],"/",sep="")
  #
  if (!file.exists(paste(pagg,"SIM_MAST",sep=""))) {
    #
    # Load model output for one variable to get dimensions
    #
    LST=mycaldat(juldaybeg)
    montact=LST$m;dayyact=LST$d;yearact=LST$y
    #
    namemont=sprintf("%02d",montact)
    namedayy=sprintf("%02d",dayyact)
    nameday=paste(yearact,namemont,namedayy,sep="")
    #
    prsv=paste(pdsup,name,"/REFER/ONLINE/",nameday,"_Nest_",nest_vec[length(nest_vec)],"/RDATA/",sep="")
    #
    load(file=paste(prsv,varieval_capitoul_cnrmstat[1],sep=""))
    vals_day=LST$vals_day
    DIM=dim(vals_day)
    #
    # Calculate distance to mast
    #
    y_stat=array(NA,c(1,2))
    y_stat[1,1] = 01.4454
    y_stat[1,2] = 43.6035
    #
    x_model=array(NA,c(DIM[1],2))
    x_model[,1] = LONS[1:lenmask[length(nest_vec)],length(nest_vec),1]
    x_model[,2] = LATS[1:lenmask[length(nest_vec)],length(nest_vec),1]
    #
    distmast = distm(x_model,y_stat,fun=distHaversine)
    amin=min(distmast)
    print(paste("The minimum distance to CAPITOUL mast is",format(amin/1000.0,digits=2),"km"))
    #
    # Find points in less than 500 m distance
    #
    indmast=which(distmast<500.0)
    if (length(indmast)<0.5) stop("No grid point in 500 m distance of mast found")
    #
    print(" ")
    print(paste("A total of ",length(indmast),"are within 500 m of mast"))
    #
    # Aggregation over seasons for HAGG, LAGG and RAGG at the location of the mast
    #
    hagg_dcyc = array(NA,c(length(seasvec),length(nout_time)))
    lagg_dcyc = array(NA,c(length(seasvec),length(nout_time)))
    ragg_dcyc = array(NA,c(length(seasvec),length(nout_time)))
    #
    for (seas in 1:length(seasvec)) {
      #
      count=0
      #
      for (dayind in na.omit(julday_seasons[seas,])) {
        #
        LST=mycaldat(dayind);montact=LST$m;dayyact=LST$d;yearact=LST$y
        #
        namemont=sprintf("%02d",montact)
        namedayy=sprintf("%02d",dayyact)
        nameday=paste(yearact,namemont,namedayy,sep="")
        #
        prsv=paste(pdsup,name,"/REFER/ONLINE/",nameday,"_Nest_",nest_vec[length(nest_vec)],"/RDATA/",sep="")
        #
        load(file=paste(prsv,"HAGG",sep=""))
        hagg=LST$vals_day;hagg=hagg[indmast,,drop=FALSE]
        #
        load(file=paste(prsv,"LAGG",sep=""))
        lagg=LST$vals_day;lagg=lagg[indmast,,drop=FALSE]
        #
        load(file=paste(prsv,"RAGG",sep=""))
        ragg=LST$vals_day;ragg=ragg[indmast,,drop=FALSE]
        #
        if (dayind==julday_seasons[seas,1]) {
          count=0
          hagg_sum=array(0.0,length(nout_time))
          lagg_sum=array(0.0,length(nout_time))
          ragg_sum=array(0.0,length(nout_time))
        }
        #
        # Average over selected grid points
        #
        hagg=colMeans(hagg,dims=1,na.rm=TRUE)
        lagg=colMeans(lagg,dims=1,na.rm=TRUE)
        ragg=colMeans(ragg,dims=1,na.rm=TRUE)
        #
        hagg_sum=hagg_sum+hagg
        lagg_sum=lagg_sum+lagg
        ragg_sum=ragg_sum+ragg
        #
        count=count + 1
        #
      }
      #
      # Calculate the average over the given period
      #
      hagg_dcyc[seas,] = hagg_sum/count
      lagg_dcyc[seas,] = lagg_sum/count
      ragg_dcyc[seas,] = ragg_sum/count
      #
    }
    #
    # Calculate of the time series of daily averages at mast
    #
    hagg_tsdy = array(NA,c(length(seasvec),nrunday))
    lagg_tsdy = array(NA,c(length(seasvec),nrunday))
    ragg_tsdy = array(NA,c(length(seasvec),nrunday))
    #
    for (day in 1:nrunday) {
      #
      juldayact=juldaybeg+(day-1)
      LST=mycaldat(juldayact)
      montact=LST$m;dayyact=LST$d;yearact=LST$y
      #
      namemont=sprintf("%02d",montact)
      namedayy=sprintf("%02d",dayyact)
      nameday=paste(yearact,namemont,namedayy,sep="")
      #
      prsv=paste(pdsup,name,"/REFER/ONLINE/",nameday,"_Nest_",nest_vec[length(nest_vec)],"/RDATA/",sep="")
      #
      load(file=paste(prsv,"HAGG",sep=""))
      hagg=LST$vals_day;hagg_tsdy[day]=mean(hagg[indmast,])
      #
      load(file=paste(prsv,"LAGG",sep=""))
      lagg=LST$vals_day;lagg_tsdy[day]=mean(lagg[indmast,])
      #
      load(file=paste(prsv,"RAGG",sep=""))
      ragg=LST$vals_day;ragg_tsdy[day]=mean(ragg[indmast,])
      #
    }
    #
    # Save the simulated mast data
    #
    LST=list(hagg_tsdy=hagg_tsdy,lagg_tsdy=lagg_tsdy,ragg_tsdy=ragg_tsdy,
             hagg_dcyc=hagg_dcyc,lagg_dcyc=lagg_dcyc,ragg_dcyc=ragg_dcyc)
    # 
    save(LST,file=paste(pagg,"SIM_MAST",sep=""))
    #
  }
  #
  # Load aggregated mast fluxes 
  # TODO: preparation as a function of settings in this script!
  #       and shift to the beginning of this script
  #
  load(file=paste(pagg,"SIM_MAST",sep=""))
  hagg_tsdy=LST$hagg_tsdy;lagg_tsdy=LST$lagg_tsdy;ragg_tsdy=LST$ragg_tsdy
  hagg_dcyc=LST$hagg_dcyc;lagg_dcyc=LST$lagg_dcyc;ragg_dcyc=LST$ragg_dcyc
  #
  # Load the observed mast data
  # CAUTION: they are DJF,MAM,JJA,SON
  #
  fobs=paste(pdata,"MASTDAT",sep="")
  if (!file.exists(fobs)) seprepmast(fobs)
  #
  load(file=fobs)
  hagg_dcyc_obs=LST$sendcyc;lagg_dcyc_obs=LST$latdcyc;ragg_dcyc_obs=LST$raddcyc
  #
  hagg_dcyc_obs_new=array(NA,dim(hagg_dcyc_obs))
  lagg_dcyc_obs_new=array(NA,dim(lagg_dcyc_obs))
  ragg_dcyc_obs_new=array(NA,dim(ragg_dcyc_obs))
  #
  for (nn in 1:3) {
    hagg_dcyc_obs_new[nn,]=hagg_dcyc_obs[nn+1,]
    lagg_dcyc_obs_new[nn,]=lagg_dcyc_obs[nn+1,]
    ragg_dcyc_obs_new[nn,]=ragg_dcyc_obs[nn+1,]
  }
  #
  hagg_dcyc_obs_new[4,]=hagg_dcyc_obs[1,]
  lagg_dcyc_obs_new[4,]=lagg_dcyc_obs[1,]
  ragg_dcyc_obs_new[4,]=ragg_dcyc_obs[1,]
  #
  hagg_dcyc_obs=hagg_dcyc_obs_new
  lagg_dcyc_obs=lagg_dcyc_obs_new
  ragg_dcyc_obs=ragg_dcyc_obs_new
  #
  # The time stepping of the mast data needs
  # to be adjusted if not half-hourly data
  #
  DIM1=dim(hagg_dcyc)
  DIM2=dim(hagg_dcyc_obs)
  #
  if (DIM1[2]==DIM2[2]) {
    # No action required
  }
  else if (DIM1[2]==(DIM2[2]/2)) {
    ind=seq(2,DIM2[2],2)
    hagg_dcyc_obs=hagg_dcyc_obs[,ind]
    lagg_dcyc_obs=lagg_dcyc_obs[,ind]
    ragg_dcyc_obs=ragg_dcyc_obs[,ind]
  }
  else {
    stop("Inconsistant length of simulated and observed data")
  }
  #
  # Shift the mast by 1 h (local winter time -> UTC)
  #
  hagg_dcyc_obs_new=array(NA,dim(hagg_dcyc_obs))
  lagg_dcyc_obs_new=array(NA,dim(lagg_dcyc_obs))
  ragg_dcyc_obs_new=array(NA,dim(ragg_dcyc_obs))
  #
  hagg_dcyc_obs_new[,1:23]=hagg_dcyc_obs[,2:24]
  lagg_dcyc_obs_new[,1:23]=lagg_dcyc_obs[,2:24]
  ragg_dcyc_obs_new[,1:23]=ragg_dcyc_obs[,2:24]
  #
  hagg_dcyc_obs_new[,24]=hagg_dcyc_obs[,1]
  lagg_dcyc_obs_new[,24]=lagg_dcyc_obs[,1]
  ragg_dcyc_obs_new[,24]=ragg_dcyc_obs[,1]
  #
  hagg_dcyc_obs=hagg_dcyc_obs_new
  lagg_dcyc_obs=lagg_dcyc_obs_new
  ragg_dcyc_obs=ragg_dcyc_obs_new
  #
  # Create plot
  #
  # Define x-axis
  #
  incdateout=1/length(nout_time)
  xlim=c(0,24)
  xaxisout=24*seq(0,(1-incdateout),incdateout)
  xaxislab=seq(0,24,2)
  #
  cexall=1.0;pchall=1.0;lwdall=1.0
  #
  # At the moment only the meteorological seasons are plotted,
  # not the weather types, since the mast data are not available for
  # the weather types
  #
  for (seas in 1:4) for (var in 1:3) {
    #
    # Settings depending on variable
    #
    if (var==1) {
      nam="SENFLX"
      lab="Sensible heat flux [W/m²]"
      sim=hagg_dcyc
      obs=hagg_dcyc_obs     
      ymin=-100;ymax=+350;yinc=+50
      if (seas==4) {ymin=-80;ymax=+200;yinc=+20}
    }
    else if (var==2) {
      nam="LATFLX"
      lab="Latent heat flux [W/m²]"
      sim=lagg_dcyc
      obs=lagg_dcyc_obs
      ymin=-100;ymax=+350;yinc=+50
      if (seas==4) {ymin=-80;ymax=+200;yinc=+20}
    }
    else if (var==3) {
      nam="RNET"
      lab="Net radiative flux [W/m²]"
      sim=ragg_dcyc
      obs=ragg_dcyc_obs
      ymin=-100;ymax=+550;yinc=+50
      if (seas==4) {ymin=-80;ymax=+200;yinc=+20}
    }
    else {
      stop("This case is not implemented")
    }
    #
    yaxis=seq(ymin,ymax,yinc)
    ylim=c(ymin,ymax)
    #
    pdf(paste(ppcyc,"EVALMAST_",name,"_",nam,"_",seasvec[seas],".pdf",sep=""))
    par(mar=c(5,5,1,5))
    #
    plot0(axes=FALSE,ylim=ylim,xlim=xlim,xlab=NA,ylab=NA,frame.plot=T)
    #
    lines(x=xaxisout,y=obs[seas,],pch=pchall,cex=cexall,lwd=lwdall,col="black")
    lines(x=xaxisout,y=sim[seas,],pch=pchall,cex=cexall,lwd=lwdall,col="red")
    #
    axis(1,at=xaxislab,lab=xaxislab,las=1,cex.axis=1.3)
    axis(2,at=yaxis,lab=yaxis,las=2,cex.axis=1.3)
    #
    mtext(lab,side=2,line=3,cex=1.5,col="black")
    # 
    legendvec=c("Mast","Simulation")
    #
    legend("top",legendvec,lty=1,col=c("black","red"),lwd=2,cex=1.2)
    #
    garbage <- dev.off()
    #
  }
  #
}
#
# ############################################################
# Quiver plot for wind over selected time intervals
# ############################################################
#
print("                     ")
print("Plotting wind vector ")
print("                     ")
#
if (nrunday>0.5) {
#
# Plot maps for each scenario and urban increment
#
count_nest=0
#
for (nest in nest_vec) {
  #
  count_nest=count_nest+1
  #
  for (sce in 1:(length(scen)+1)) {
    #
    if (sce<=length(scen)) {
      pagg=paste(pdsup,name,"/",scen[sce],"/ONLINE/AGG_RDATA_Nest_",nest,"/",sep="")
      pplot=paste(pmaps,name,"/",scen[sce],"/Online/",sep="")
      flaginc=0
    } else {
      pagg=paste(pdsup,name,"/URBINC/AGG_RDATA_Nest_",nest,"/",sep="")
      pplot=paste(pmaps,name,"/URBINC_Nest_",nest,"/",sep="")
      flaginc=1
    }
    #
    system(paste("mkdir -p",pplot))
    #
    for (period in 1:length(perio_vec)) for (seas in 1:length(seasvec)) for (lvl in 1:length(lvl_wivec)) {
      #
      filein_1 = paste(pagg,paste0("_LVL_",lvl_wivec[lvl],"_SPLIT_UMME_"),seasvec[seas],"_",perio_vec[period],sep="")
      filein_2 = paste(pagg,paste0("_LVL_",lvl_wivec[lvl],"_SPLIT_VMME_"),seasvec[seas],"_",perio_vec[period],sep="")
      #
      if (file.exists(filein_1)&&file.exists(filein_2)) {
        #
        load(file=filein_1);umme_all=LST$fieldplot
        load(file=filein_2);vmme_all=LST$fieldplot
        #
	# Entire grid
	#
        nameplot=paste(pplot,"WIVEC_Nest",nest,"_","AGL_",hagl_wivec[lvl],"_",seasvec[seas],"_",perio_vec[period],"_FULL.pdf",sep="")
	#
	lons_all=LONS[1:lenmask[count_nest],count_nest,1]
	lats_all=LATS[1:lenmask[count_nest],count_nest,1]
	#
	lons_sel=lons_all
	lats_sel=lats_all
	umme_sel=umme_all
	vmme_sel=vmme_all
	#
        myquiver(x=lons_sel,y=lats_sel,u=umme_sel,v=vmme_sel,nameplot=nameplot,flaginc=flaginc)
        #
	# Subdomains
        #
	if (length(name_zoom)>0) {
	  #
          for (sub in 1:length(name_zoom)) {
            #
            nameplot=paste(pplot,"WIVEC_Nest",nest,"_","AGL_",hagl_wivec[lvl],"_",seasvec[seas],"_",perio_vec[period],"_",name_zoom[sub],".pdf",sep="")
            #
            ind_sub=which( (lats_all>=latsmin_zoom[sub])&(lats_all<=latsmax_zoom[sub])&
                           (lons_all>=lonsmin_zoom[sub])&(lons_all<=lonsmax_zoom[sub]) )
            #
            lons_sel=lons_all[ind_sub]
	    lats_sel=lats_all[ind_sub]
	    umme_sel=umme_all[ind_sub]
	    vmme_sel=vmme_all[ind_sub]
	    #
            myquiver(x=lons_sel,y=lats_sel,u=umme_sel,v=vmme_sel,nameplot=nameplot,flaginc=flaginc)
            #
          }
	}
        #
      }
    }
    #
  }
#
}
}
#
# ############################################################
# Plot maps of averages over selected time intervals
# ############################################################
#
print("              ")
print("Plotting maps ")
print("              ")
#
if (nrunday>0.5) {
#
# Plot maps for each scenario and urban increment
#
count_nest=0
for (nest in nest_vec) {
  count_nest=count_nest+1
  for (sce in 1:(length(scen)+1)) {
    #
    if (sce<=length(scen)) {
      pagg=paste(pdsup,name,"/",scen[sce],"/ONLINE/AGG_RDATA_Nest_",nest,"/",sep="")
      pplot=paste(pmaps,name,"/",scen[sce],"/Online/",sep="")
      varlistplot=varplot
      titleplot=titlevec
      flaginc=0
    } else {
      pagg=paste(pdsup,name,"/URBINC/AGG_RDATA_Nest_",nest,"/",sep="")
      pplot=paste(pmaps,name,"/URBINC_Nest_",nest,"/",sep="")
      varlistplot=varlist_urbinc
      titleplot=titleurbincvec
      flaginc=1
    }
    #
    for (period in 1:length(perio_vec)) for (seas in 1:length(seasvec)) for (vari in 1:length(varlistplot)) {
      #
      filein=paste(pagg,varlistplot[vari],"_",seasvec[seas],"_",perio_vec[period],sep="")
      file_ut = paste(pagg,"UMME_",seasvec[seas],"_",perio_vec[period],sep="")
      file_vt = paste(pagg,"VMME_",seasvec[seas],"_",perio_vec[period],sep="")
      #
      if (file.exists(filein)) {
        #
	# Load primary variable to be plotted
	#
        load(file=paste(pagg,varlistplot[vari],"_",seasvec[seas],"_",perio_vec[period],sep=""))
        fieldplot=LST$fieldplot
        #
	# Load wind componenets
        #
	if (file.exists(file_ut)) {
          load(file=file_ut);umme=LST$fieldplot
          load(file=file_vt);vmme=LST$fieldplot
	} else {
          umme=NA
          vmme=NA
        }
        #
        if (varlistplot[vari]=="TREAL_2M") {
	  varisave="TREAL"
	} else {
	  varisave="DUMMY"
	}
        #
        nameplot=paste(pplot,varlistplot[vari],"_Nest_",nest,"_",seasvec[seas],"_",perio_vec[period],".pdf",sep="")
        #
        seplotonline(LONS[1:lenmask[count_nest],count_nest,1],LATS[1:lenmask[count_nest],count_nest,1],fieldplot,
	             nameplot,varlistplot[vari],nametitle=titleplot[vari],flaginc=flaginc,lplotstation=0,
                     namesim=paste(name,"/",scen[sce],"_Nest_",nest,sep=""),lquiver=0,ut=umme,vt=vmme,
		     seas=seasvec[seas],perio=perio_vec[period],variable=varisave)
        #
      }
    }
    #
  }
  #
}
}
#
print("All plots finished")
#
