! ########################
        SUBROUTINE URBAN_HYDRO_SOIL(O, PURBDRAIN,                           &
                                    PTSTEP, PPS, PWSOIL, PTG, PDZG,         &
                                    PWSAT, PWFC, PWWILT,                    &
                                    PCONDSAT, PMPOTSAT, PBCOEF,             &
                                    PWG, PWGI,                              &
                                    PRUNOFFSOIL, PDRAIN ,                   &
                                    PFWTD, PWTD, PTOPQS, PQSB               )
!###############################################
!!****  *URBAN_HYDRO_SOIL*  
!
!!    PURPOSE
!!    -------
!     
!     Calculates the evolution of the water variables for TEB (Road & Roof) i.e.,
!     deep-soil volumetric water content
!     Also determine the runoff and drainage into the soil.
!
!!**  METHOD
!!    ------
!
!     Direct calculation
!
!!    EXTERNAL
!!    --------
!
!     None
!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!!      
!!    REFERENCE
!!    ---------
!!
!!    Boone (2000)
!!    Boone et al. (2000)
!!      
!!    AUTHOR
!!    ------
!!
!!!!      JM Brun		  * IFSTTAR * 
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    02/04/2012   J-M Brun
!-------------------------------------------------------------------------------
!
!*       0.     DECLARATIONS
!               ------------
!
USE MODD_SURF_PAR,       ONLY : XUNDEF
USE MODD_CSTS,           ONLY : XTT, XRHOLW
USE MODD_ISBA_PAR,       ONLY : XWGMIN
USE MODD_ISBA_n,         ONLY : ISBA_K_t, ISBA_P_t, ISBA_PE_t 
USE MODD_ISBA_OPTIONS_n, ONLY : ISBA_OPTIONS_t
!
USE YOMHOOK   ,ONLY : LHOOK,   DR_HOOK
USE PARKIND1  ,ONLY : JPRB
!
USE MODI_HYDRO_SOILDIF  
!
USE MODE_THERMOS
!
IMPLICIT NONE
!
!----------------------------------------------------------------------------------
!*      0.1    Declarations of arguments
!-----------------------------------------------
!
TYPE(ISBA_OPTIONS_t), INTENT(INOUT) :: O
REAL,                 INTENT(IN)    :: PURBDRAIN    ! Fraction of deep drainage 
REAL,                 INTENT(IN)    :: PTSTEP       ! Time step (s)
REAL, DIMENSION(:)  , INTENT(IN)    :: PPS          ! Pressure at the surface (Pa)
REAL, DIMENSION(:)  , INTENT(IN)    :: PWSOIL       ! Water that infiltrates from the surface (kg/m2/s)
REAL, DIMENSION(:,:), INTENT(IN)    :: PTG          ! Soil temperature profile (K)
REAL, DIMENSION(:,:), INTENT(IN)    :: PDZG         ! Thickness of soil layers (m)
REAL, DIMENSION(:,:), INTENT(IN)    :: PWSAT        ! Water content at saturation (m3/m3)
REAL, DIMENSION(:,:), INTENT(IN)    :: PWFC         ! Field capacity              (m3/m3)
REAL, DIMENSION(:,:), INTENT(IN)    :: PWWILT       ! Wilting point               (m3/m3)
REAL, DIMENSION(:,:), INTENT(IN)    :: PCONDSAT     ! Hydraulic conductivity at sat (m/s)
REAL, DIMENSION(:,:), INTENT(IN)    :: PMPOTSAT     ! Matrix potential at sat     (m)
REAL, DIMENSION(:,:), INTENT(IN)    :: PBCOEF       ! b parameter for retention curve (-)
REAL, DIMENSION(:,:), INTENT(INOUT) :: PWG          ! Soil volumetric water content profile (m3/m3)
REAL, DIMENSION(:,:), INTENT(INOUT) :: PWGI         ! Soil volumetric water (ice) content profile (m3/m3)
REAL, DIMENSION(:)  , INTENT(OUT)   :: PRUNOFFSOIL  ! Lateral runoff  (kg/m2/s) 
REAL, DIMENSION(:)  , INTENT(OUT)   :: PDRAIN       ! Drainage (kg/m2/s)
!
REAL, DIMENSION(:)  , INTENT(IN)    :: PFWTD        !grid-cell fraction of water table to rise
REAL, DIMENSION(:)  , INTENT(IN)    :: PWTD         !water table depth (m)
REAL, DIMENSION(:,:), INTENT(IN)    :: PTOPQS       ! Topmodel subsurface flow by layer (m/s)
REAL, DIMENSION(:)  , INTENT(OUT)   :: PQSB         !Lateral subsurface flow [kg/m²/s]
!
!----------------------------------------------------------------------------------
!*      0.2    Declarations of local variables
!-----------------------------------------------
!
TYPE(ISBA_K_t),  POINTER :: HK                              ! ISBA structures for call hydro_soildif
TYPE(ISBA_P_t),  POINTER :: HP
TYPE(ISBA_PE_t), POINTER :: HPE
INTEGER                                        :: JWRK      ! loop control                                       
INTEGER                                        :: JJ,JL,IL  ! loop index
INTEGER                                        :: JNDT, JDT ! time splitting indicies
INTEGER                                        :: INL,IDEPTH
INTEGER                                        :: IMAX_LAYER
INTEGER                                        :: ILAYER_HORT
REAL,    DIMENSION(SIZE(PTG,1))                :: ZWDRAIN               ! minimum Wg for drainage (m3 m-3)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZFSAT                 ! Saturated fraction (WARNING !!!! inutilisé)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZLETR                 ! transpiration rate (W m-2)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZLEG                  ! bare-soil evaporation rate (W m-2)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZLEGI                 ! surface layer sublimation rate (W m-2)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZF2WGHT               ! root-zone transpiration weights (-)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZROOTFRAC             ! Cumulative root fraction (-)cumulative root fraction (-)
REAL,    PARAMETER                             :: ZRICHARDSDTMAX = 900. ! Maximum timescale for Richard's Eq. (s)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZPS,ZTI,ZQSAT,ZQSATI
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZWSAT                 ! Ice modified soil porosity (m3 m-3) 
REAL,    DIMENSION(SIZE(PTG,1))                :: ZEXCESS               ! Excess soil water which is used as a constraint
!
REAL(KIND=JPRB) :: ZHOOK_HANDLE
!
!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
!
IF (LHOOK) CALL DR_HOOK('URBAN_HYDRO_SOIL',0,ZHOOK_HANDLE)
!
!--------------------------------------------------------------------------
!
!*      0.     Allocation and Initialisation of ISBA structures
!              --------------
!
ALLOCATE(HK%XBCOEF   (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HK%XWSAT    (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HK%XMPOTSAT (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HK%XWFC     (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HK%XWTD     (SIZE(PWG,1)))
ALLOCATE(HK%XFWTD    (SIZE(PWG,1)))
!
ALLOCATE(HP%XTOPQS   (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HP%XDG      (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HP%XDZG     (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HP%XDZDIF   (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HP%XCONDSAT (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HP%NWG_LAYER(SIZE(PWG,1)))
!
ALLOCATE(HPE%XWG     (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HPE%XWGI    (SIZE(PWG,1),SIZE(PWG,2)))
ALLOCATE(HPE%XTG     (SIZE(PWG,1),SIZE(PWG,2)))
!
HK%XBCOEF  (:,:) = PBCOEF  (:,:)
HK%XWSAT   (:,:) = PWSAT   (:,:)
HK%XMPOTSAT(:,:) = PMPOTSAT(:,:)
HK%XWFC    (:,:) = PWFC    (:,:)
HK%XWTD    (:)   = PWTD    (:)
HK%XFWTD   (:)   = PFWTD   (:)
!
HP%XTOPQS  (:,:) = PTOPQS  (:,:)
HP%XDZG    (:,:) = PDZG    (:,:)
HP%XDG     (:,:) = 0.
HP%XDZDIF  (:,:) = 0.
HP%XCONDSAT(:,:) = PCONDSAT(:,:)
HP%NWG_LAYER(:)  = SIZE(PTG,2)
!
HPE%XWG    (:,:) = PWG     (:,:)
HPE%XWGI   (:,:) = PWGI    (:,:)
HPE%XTG    (:,:) = PTG     (:,:)
!
!*      1.     Initialisation
!              --------------
!
ZWSAT(:,:) = MAX(XWGMIN, PWSAT(:,:)-PWGI(:,:))  
ZEXCESS(:) = 0.
!
! Root fraction is zero under roads and buildings
ZROOTFRAC(:,:) = 0.
!
! Topmodel saturated fraction, initialisé à 0. dans GARDEN, inutilisé dans URBAN_HYDRO_SOILDIF
ZFSAT(:)       = 0.0 
ZWDRAIN(:)     = 0.
!
! No evaporation
ZLETR(:)         =  0.
ZLEG(:)          =  0.
ZLEGI(:)         =  0.
!
! Moisture layers
IMAX_LAYER   = SIZE(PTG,2)
ILAYER_HORT  = SIZE(PTG,2)
!
! Cumultative depth of soil layers
HP%XDG(:,1) = PDZG(:,1) 
DO JWRK=2,SIZE(PDZG,2) 
   HP%XDG(:,JWRK) = HP%XDG(:,JWRK-1) + PDZG(:,JWRK)  
END DO
!
! Distance between the midpoints of two consecutive layers (m)
DO JWRK=1,SIZE(PDZG,2)-1
   HP%XDZDIF(:,JWRK) = (PDZG(:,JWRK) + PDZG(:,JWRK+1))/2.  
END DO
HP%XDZDIF(:,SIZE(PDZG,2)) = PDZG(:,SIZE(PDZG,2))/2. 
!
! No transpiration
ZF2WGHT(:,:) = 0.
!
!
!*      2.     RUN-OFF (cf hydro.F90) 
!              ----------------------
!
INL = MAXVAL(HP%NWG_LAYER(:))
!
! Initialize some field
! ---------------------
!
ZPS(:,:) = XUNDEF
ZTI(:,:) = XUNDEF
DO JL=1,INL
   DO JJ=1,SIZE(PTG,1)
      IDEPTH=HP%NWG_LAYER(JJ)
      IF(JL<=IDEPTH)THEN
        ZPS(JJ,JL) = PPS(JJ)
        ZTI(JJ,JL) = MIN(XTT,PTG(JJ,JL))
      ENDIF
   ENDDO
ENDDO
!
! Compute specific humidity at saturation for the vapor conductivity
! ------------------------------------------------------------------
!
ZQSAT (:,:) = QSAT (PTG(:,:),ZPS(:,:),HP%NWG_LAYER(:),INL)
ZQSATI(:,:) = QSATI(ZTI(:,:),ZPS(:,:),HP%NWG_LAYER(:),INL)
!
!*      3.     Call hydro_soildif
!              ------------------
!
PRUNOFFSOIL(:) = 0.
! 
! PWSOIL is converted in m/s
!
CALL HYDRO_SOILDIF(O, HK, HP, HPE, PTSTEP,                   &
                   PWSOIL*1E+3, ZLETR, ZLEG, ZLEGI, ZF2WGHT, &
                   PPS, ZQSAT, ZQSATI,                       &
                   PDRAIN, PRUNOFFSOIL,                      &
                   IMAX_LAYER, PQSB                          )
!
!IF (PFRECHARGE.GT.0.) THEN
IF (PURBDRAIN.GT.0.) THEN
  DO JJ=0,SIZE(PTG,1)
!
    IDEPTH=HP%NWG_LAYER(JJ)
!
!   Drainage or baseflow out of bottom of model (slow time response) (kg m-2 s-1):
    PDRAIN(JJ)     = (1.-PURBDRAIN) * PDRAIN(JJ)
!   Update liquide water content (m3/m3)
    PWG(JJ,IDEPTH) = PWG(JJ,IDEPTH) + PURBDRAIN*PDRAIN(JJ)*PTSTEP/XRHOLW/PDZG(JJ,IDEPTH)
!
!   When the water content in the last layer exceeds the water content at saturation, 
!   then the water surplus is added to the layers above:   
    IF(PWG(JJ,IDEPTH)>ZWSAT(JJ,IDEPTH)) THEN
!
      IL=IDEPTH
!
!     Calculation of the water surplus in last layer:
      ZEXCESS(JJ)= MAX(0.0, PWG(JJ,IL) - ZWSAT(JJ,IL))
!     Water content that stays in last layer:
      PWG(JJ,IL) = MIN(PWG(JJ,IL),ZWSAT(JJ,IL))
! 
      DO WHILE (ZEXCESS(JJ) > 0.0 .AND. IL > 1) ! test
!        
!       Update of the water content in the layer above:
        PWG(JJ,IL-1) = PWG(JJ,IL-1) + ZEXCESS(JJ)*(PDZG(JJ,IL)/PDZG(JJ,IL-1))
!       Calculation of the water surplus in layer above:
        ZEXCESS(JJ)  = MAX(0.0, PWG(JJ,IL-1) - ZWSAT(JJ,IL-1))
!       Water content that stays in current layer:
        PWG(JJ,IL-1) = MIN(PWG(JJ,IL-1),ZWSAT(JJ,IL-1)) 
!       Index of layer is incremented to the top
        IL=IL-1
!
      ENDDO
!
      IF(IL==1 .AND. PWG(JJ,1)>ZWSAT(JJ,1)) THEN
!
!       Calculation of the water surplus in first layer:
        ZEXCESS(JJ)     = MAX(0.0, PWG(JJ,1) - ZWSAT(JJ,1))
!       Water content that stays in first layer:
        PWG(JJ,1)       = MIN(PWG(JJ,1),ZWSAT(JJ,1))
!       Water excess is added to surface runoff mm/s:
        PRUNOFFSOIL(JJ) = PRUNOFFSOIL(JJ) + (ZEXCESS(JJ) * (PDZG(JJ,1)/PTSTEP)) * XRHOLW 
!                 
      ENDIF
    ENDIF
  ENDDO
ENDIF
!
!-------------------------------------------------------------------------------
!
!
IF (LHOOK) CALL DR_HOOK('URBAN_HYDRO_SOIL',1,ZHOOK_HANDLE)
!-------------------------------------------------------------------------------
!
END SUBROUTINE URBAN_HYDRO_SOIL


