#include <define.h>
  PROGRAM CLM
! ======================================================================
! The Common Land Model was developed in cooperation with
!     Beijing Normal University                             (Dai)
!     Georgia Institute of Technology                       (Dickinson)
!     National Center for Atmospheric Research              (Bonan, Oleson)
!     University of Arizona                                 (Zeng)
!     University of Texas at Austin                         (Yang)
!     GSFC/NASA                                             (Houser, Bosilovich)
!     COLA                                                  (Dirmeyer, Schlosser)
!     Colorado State University at Fort Collins             (Denning, Baker)
!
! Reference:
!     [1] Dai et al., 2003: The Common Land Model (CLM).
!         Bull. of Amer. Meter. Soc., 84: 1013-1023
!     [2] Dai et al., 2004: A two-big-leaf model for canopy temperature,
!         photosynthesis and stomatal conductance. J. Climate, 17:2281-2299
!
!     Author: Yongjiu Dai, January 2004
! ======================================================================
      use precision
      implicit none
!----------------------------------------------------------------------
! Define the dimension of model array
!----------------------------------------------------------------------
      integer nl_soil_       ! number of soil layers
      integer maxsnl_        ! max number of snow layers
      integer nfcon_         ! number of time constant variables
      integer nftune_        ! number of clm tunable constants
      integer nfvar_         ! number of time varying variables
      integer nforc_         ! number of forcing variables
      integer nfldv_         ! number of output fluxes
      integer nflai_         ! number of leaf time varying variables
      integer maxpatch_      ! maximum number of patches in model grid
      integer nlandcateg_    ! number of land cover categories
      integer nsoilcateg_    ! number of soil texture categories
      parameter(nl_soil_    = 10)
      parameter(maxsnl_     = -5)
      parameter(nfcon_      = 9*nl_soil_+29)
      parameter(nftune_     = 14)
      parameter(nfvar_      = 5*(nl_soil_-maxsnl_)+52)
      parameter(nforc_      = 18)
      parameter(nfldv_      = 98)
      parameter(nflai_      = 4)
      parameter(maxpatch_   = 26)
      parameter(nlandcateg_ = 26)
      parameter(nsoilcateg_ = 17)
! ----------------local variables ---------------------------------
      integer, parameter :: nl_soil  = nl_soil_  ! number of soil layers
      integer, parameter :: maxsnl   = maxsnl_   ! maximum number of snow layers
      integer, parameter :: nftune   = nftune_   ! number of clm tunable constants
      integer, parameter :: nfcon    = nfcon_    ! number of time constant variables
      integer, parameter :: nforc    = nforc_    ! number of forcing variables
      integer, parameter :: nfvar    = nfvar_    ! number of time varying variables
      integer, parameter :: nfldv    = nfldv_    ! number of output variables
      integer, parameter :: nflai    = nflai_    ! number of leaf time varying variables
      integer, parameter :: maxpatch = maxpatch_ ! maximum number of patches in a grid
      real    :: deltim                          ! time step (senconds)
      integer :: mstep                           ! model step for simulation [-]
!
      integer :: lusrf                           ! logical unit number of surface data
      integer :: lulai                           ! logical unit number of LAI data
      integer :: lumet                           ! logical unit number of meteorological forcing
      integer :: lhistTimeConst                  ! logical unit number of restart time-invariant file
      integer :: lhistTimeVar                    ! logical unit number of restart time-varying file
      integer :: luout                           ! logical unit number of output
      integer :: lon_points                      ! number of longitude points on model grid
      integer :: lat_points                      ! number of latitude points on model grid
      integer :: numpatch                        ! total number of patches of grids
!
      character(LEN=256) :: site                 ! site name
      character(LEN=256) :: fsurdat              ! file name of surface data
      character(LEN=256) :: flaidat              ! file name of time-varying vegetation data
      character(LEN=256) :: fmetdat              ! file name meteorological data
      character(LEN=256) :: fhistTimeConst       ! file name of restart time-invariant file
      character(LEN=256) :: fhistTimeVar         ! file name of restart time-varying file
      character(LEN=256) :: foutdat              ! file name of output file
	  character(LEN=256) :: cdate
	  character(LEN=256) :: name
#if(!defined EcoDynamics)
      real, allocatable :: mflai(:,:,:,:)    ! time varying vegetation parameters
      real, allocatable :: flai(:,:)         ! time varying vegetation parameters
!	  integer, allocatable :: lu(:,:)
      real, allocatable :: lai(:,:,:)         ! time varying vegetation parameters
      real, allocatable :: fveg(:,:,:)         ! time varying vegetation parameters
      real, allocatable :: sai(:,:,:)         ! time varying vegetation parameters
      real, allocatable :: green(:,:,:)         ! time varying vegetation parameters
#endif
      real, allocatable :: forcxy(:,:,:)     ! atmospheric forcing
      real, allocatable :: fldxy(:,:,:)      ! output fluxes in 2-dimension
!
      integer   idate(3)                         ! calendar (year, julian day, seconds)
      integer,  allocatable :: numpatch_lat(:)   ! number of patches of grids at lon. strip
      integer,  allocatable :: ixy_patch(:)      ! patch longitude index
      integer,  allocatable :: jxy_patch(:)      ! patch latitude index
      integer,  allocatable :: mxy_patch(:)      ! patch subgrid index of lnd point
      real, allocatable :: wtxy_patch(:)     ! patch weight
      real, allocatable :: fcon(:,:)         ! time constant variables
      real, allocatable :: forc(:,:)         ! forcing variables
      real, allocatable :: fvar(:,:)         ! time varying variables
      real, allocatable :: fldv(:,:)         ! output fluxes
      real  ftune(nftune)                    ! clm tunable constants
!
      real, allocatable :: oro(:)            ! ocean(0)/seaice(2)/ flag
      real, allocatable :: a(:,:,:)          !
      real, allocatable :: fldxy_r(:,:,:)    ! output fluxes in 2-dimension
      integer, allocatable :: itypwat(:)         ! land water type
!
      logical :: lwrite                          ! true: write out frequency
      logical :: doalb                           ! true if time for surface albedo calculation
      logical :: dolai                           ! true if time for time-varying vegetation paramter
      logical :: dosst                           ! true if time for update sst/ice/snow
!
      integer :: istep                           ! looping step
      integer :: i,j,k,l,m                       ! looping indices
      integer :: nac                             ! number of accumulation
      integer :: idate_p(3)                      ! current model calendar
      namelist /clmexp/ site,                   &!1
                        flaidat,                &!2
                        fmetdat,                &!3
                        fhistTimeConst,         &!4
                        fhistTimeVar,           &!5
                        foutdat,                &!6
                        lhistTimeConst,         &!7
                        lhistTimeVar,           &!8
                        lulai,                  &!9
                        lumet,                  &!10
                        luout,                  &!11
                        lon_points,             &!12
                        lat_points,             &!13
                        numpatch,               &!14
                        deltim,                 &!15
                        mstep                    !16
! ======================================================================
! define the run and open files (for off-line use)
      read(5,clmexp) 
      allocate (numpatch_lat(lat_points))
      allocate (ixy_patch(numpatch))
      allocate (jxy_patch(numpatch))
      allocate (mxy_patch(numpatch))
      allocate (wtxy_patch(numpatch))
      allocate (fcon(numpatch,nfcon))
      allocate (forc(numpatch,nforc))
      allocate (fvar(numpatch,nfvar))
      allocate (fldv(numpatch,nfldv))
#if(!defined EcoDynamics)
      allocate (mflai(lon_points,lat_points,maxpatch,nflai))
      allocate (flai(numpatch,nflai))
!	  allocate (lu(lon_points,lat_points,maxpatch))
	  allocate (lai(lon_points,lat_points,maxpatch))
	  allocate (fveg(lon_points,lat_points,maxpatch))
	  allocate (sai(lon_points,lat_points,maxpatch))
	  allocate (green(lon_points,lat_points,maxpatch))
#endif
      allocate (forcxy(lon_points,lat_points,nforc))
      allocate (fldxy(lon_points,lat_points,nfldv))
      allocate (fldxy_r(lon_points,lat_points,nfldv))
      allocate (a(lon_points,lat_points,nfldv))
      allocate (oro(numpatch)) 
      allocate (itypwat(numpatch)) 
#if(!defined EcoDynamics)
      OPEN(unit=lulai,file=flaidat,form='formatted',&
                           status='old',action='read')
#endif
! Open for meteorological forcing data
      OPEN(unit=lumet,file=fmetdat,form='formatted',&
                           status='old',action='read')
! Open for model time invariant constant data
      OPEN(unit=lhistTimeConst,file=fhistTimeConst,form='unformatted',&
                           status='unknown',action='read')
! Open for model time varying data (model state variables)
      OPEN(unit=lhistTimeVar,file=fhistTimeVar,form='unformatted',&
                           status='unknown',action='read')
      CALL rstTimeConstRead (lat_points,lhistTimeConst,nfcon,nftune,&
                             numpatch,numpatch_lat,ixy_patch,jxy_patch,mxy_patch,wtxy_patch,&
                             fcon,ftune)
      CALL rstTimeVarRead (lhistTimeVar,nfvar,numpatch,idate,fvar)
! ======================================================================
! begin time stepping loop
! ======================================================================
      fldxy(:,:,:)=0.
      nac = 0

!	  open(12345,file='lulc_BJ.dat',action='read')
!         do j = 1, lat_points
!            do i = 1, lon_points
!			  read(12345,*) lu(i,j)
!			enddo
!	     enddo

      do istep = 1, mstep
         idate_p(:) = idate(:)
!           write(*,*) idate(3)
! Read in the atmospheric forcing
         CALL GETMET (lumet,lon_points,lat_points,nforc,forcxy)
! Mapping atmospheric fields to force clm: [lon_points]x[lat_points] grid
!     -> [numpatch] vector of subgrid points
         do k = 1, numpatch   !clm vector index
            i = ixy_patch(k)  !longitude index
            j = jxy_patch(k)  !latitude index
            forc(k,1:) = forcxy(i,j,1:) 
         end do
! doalb is true when the next time step is a radiation time step
         doalb = .true. 
         dolai = .true.
         dosst = .false.
         oro(:) = 1.
#if(!defined EcoDynamics)
         if(dolai)then
       ! read(lulai) 
     do j = 1,lat_points
         do i = 1, lon_points
		   do k = 1, maxpatch
            read(lulai,*) &
            lai  (i,j,k),sai(i,j,k),&
            green(i,j,k),fveg (i,j,k)
		   enddo
         end do
     enddo
!     close (lulai)

		 do j = 1, lat_points
            do i = 1, lon_points
			    do k=1,maxpatch
                     mflai(i,j,k,1)=lai(i,j,k)
                     mflai(i,j,k,2)=sai(i,j,k)
                     mflai(i,j,k,3)=green(i,j,k)
                     mflai(i,j,k,4)=fveg(i,j,k)
  				enddo
            enddo
         enddo

         do k = 1, numpatch
            i = ixy_patch(k)
            j = jxy_patch(k)
            m = mxy_patch(k)
            flai(k,:) = mflai(i,j,m,:) !leaf area index
         enddo
         endif
!		 close(12111)
#endif
! Calendar for NEXT time step
         CALL TICKTIME (deltim,idate)
! Call clm driver
         CALL CLMDRIVER (nl_soil,maxsnl,numpatch,idate,deltim,&
                         nftune,nfcon,nforc,nfvar,nfldv,&
                         ftune,fcon,forc,fvar,fldv,&
#if(!defined EcoDynamics)
                         flai,&
#endif
                         dolai,doalb,dosst,oro)
! Mapping subgrid patch [numpatch] vector of subgrid points to
!     -> [lon_points]x[lat_points] grid average
         itypwat(:) = nint(fcon(:,3))
         CALL vec2xy(lat_points,lon_points,numpatch,&
                     ixy_patch,jxy_patch,wtxy_patch,itypwat,nfcon,nforc,nfldv,&
                     fcon,forcxy,fldv,fldxy_r)
         nac = nac + 1
         fldxy(:,:,:) = fldxy(:,:,:) + fldxy_r(:,:,:)
! Logical idenfication for writing output and open file
         lwrite = .false.
         CALL lpwrite(idate_p,idate,lhistTimeVar,luout,foutdat,lwrite)
! Write out the model variables for restart run [histTimeVar] and the histroy file
         if(lwrite)then
            CALL rstTimeVarWrite (lhistTimeVar,nfvar,numpatch,idate,fvar)
            a(:,:,:) = fldxy(:,:,:) / float(nac)
            CALL flxwrite (luout,lon_points,lat_points,nfldv,a)
            fldxy(:,:,:) = 0.0
            lwrite = .false.
            nac = 0
         endif
  end do
!  10000 format(f14.5)
      write(6,*) 'CLM Execution Completed'
  END PROGRAM CLM
! ----------------------------------------------------------------------
! EOP
