# -*- coding: utf-8 -*-
import numpy as np
from netCDF4 import Dataset
import erosion
###########################################
#This script calculates global yearly average soil erosion rates
#Transient simulation = changing land use and climate (CC+LUC), changing climate only (CC),
#or changing land use only (LUC)
#Equilibrium simulation = average soil erosion rates for the period 1850-1860 based on land use
#of the year 1851 & the average rainfall erosivity of the period 1850-1860
#When using this model code please cite: "Naipal ,V., Ciais, P., Wang, Y., Lauerwald, R.,
#Guenet, B., Van Oost, K. (2018). Global soil organic carbon removal by water erosion under
#climate change and land use change during 1850–2005 AD. Biogeosciences, 15, 4459–4480,
#https://doi.org/10.5194/bg-15-4459-2018"
############################################

#settings
dir1=' ' #Enter here your work directory
rows=1680;kols=4322 #global 5 arcmin resolution grid
years=range(1851,2006) #transient period
landcover=("tree","crop","gras","bare",)
simulation="CC+LUC" #select simulation type from: equilibrium,CC+LUC,CC,LUC

#general input data
data = Dataset('%s/topo/S_scaled_5m.nc' % (dir1),'r') #S factor
S= data.variables['s'][:]
S[S<0.] = 0.
S[np.isnan(S)==True] = 0.
S=np.ravel(S)
data.close()

data = Dataset('%s/soil/K_volcanic_nogravel_5m.nc' % (dir1),'r')
K = data.variables['k'][:] # K factor without gravel correction but with volcanic soil correction
K[K<0.] = 0.
K[np.isnan(K)==True] = 0.
K=np.ravel(K)
data.close()

data = Dataset('%s/soil/gravel_topsoil_5m.nc' % (dir1),'r')
gravel = data.variables['GRAV'][:] # gravel content in %
gravel=gravel.astype(float)
gravel[gravel<0.] = 0.
gravel[np.isnan(gravel)==True] = 0.
gravel = np.ravel(gravel)
data.close()

if simulation =='equilibrium':
  ####################################
  # Equilibrium 
  ###################################
  data0 = Dataset('%s/PFTmap_WithoutNoBio/5m/PFTmap_LUHv2_BM3_HoughtonCountryForestarea_withoutNoBio_%04i_5m.nc' % (dir1,1851),'r') #crop fraction
  crop = data0.variables['maxvegetfrac'][0,11:13,:,:]
  crop=crop.reshape(2,rows,kols)
  crop[crop>1.] = np.nan
  crop[crop<0.] = np.nan
  crop=np.nansum(crop,axis=0)
  crop[np.isnan(crop)==True] = 0.
  crop=np.ravel(crop)
  data0.close()

  data2 = Dataset('%s/climate/R_factor/R/ISIMIP2b_R_1851-1860_mean_5m.nc' % (dir1),'r') # R factor
  R = data2.variables['r'][:] 
  R[R<0]=0.
  R[R>10e35]=0.
  R[np.isnan(R)==True] = 0.
  R=np.ravel(R)
  data2.close()    

  nc = Dataset('%s/C_factor/lowres/new_method/new/C_eq_5m.nc' % (dir1),'r') # C factor
  for lc in range(len(landcover)):    
    C = nc.variables['c_lc'][lc,:,:]
    C[C<0]=0.
    C[np.isnan(C)==True] = 0.
    C=np.ravel(C)   
      
    #calculate erosion for this timeperiod
    veget=landcover[lc]
    E=erosion.erosion.main(S,K,R,C,veget,crop,gravel,rows,kols)
    E = E.reshape(rows,kols)      
    #output
    output = Dataset('%s/erosion/E_%s_eq.nc' % (dir1,veget),'w')
    output.createDimension('lat',rows)
    output.createDimension('lon',kols)
    output.createVariable('E','d',('lat','lon',))
    output.variables['E'][:] = E
    output.close()    
  nc.close()    

if simulation =='CC+LUC':
  ####################################
  # CC + LUC
  ###################################
  for t in years:
    data = Dataset('%s/climate/R_factor/R/ISIMIP2b_R_%04i_5m.nc' % (dir1,t),'r') # R factor
    R = data.variables['r'][:] 
    R[R<0]=0.
    R[R>10e35]=0.
    R[np.isnan(R)==True] = 0.
    R=np.ravel(R)
    data.close()    

    data = Dataset('%s/landcover/PFTmap_WithoutNoBio/5m/PFTmap_LUHv2_BM3_HoughtonCountryForestarea_withoutNoBio_%04i_5m.nc' % (dir1,t),'r')  #crop fraction
    crop = data.variables['maxvegetfrac'][0,11:13,:,:]
    crop=crop.reshape(2,rows,kols)
    crop[crop>1.] = np.nan
    crop[crop<0.] = np.nan
    crop=np.nansum(crop,axis=0)
    crop[np.isnan(crop)==True] = 0.
    crop=np.ravel(crop)
      
    for lc in range(len(landcover)):
      data = Dataset('%s/C_factor/lowres/new_method/new/C_%04i_5m.nc' % (dir1,t),'r')
      C = data.variables['c_lc'][lc,:,:] # C factor
      C[C<0]=0.
      C[np.isnan(C)==True] = 0.
      C=np.ravel(C)   
      data.close()
    
      #calculate erosion for this timeperiod
      veget=landcover[lc]
      E=erosion.erosion.main(S,K,R,C,veget,crop,gravel,rows,kols)
      E = E.reshape(rows,kols)
      
      #output 
      output = Dataset('%s/erosion/E_%s_%04i.nc' % (dir1,veget,t),'w')
      output.createDimension('lat',rows)
      output.createDimension('lon',kols)
      output.createVariable('E','d',('lat','lon',))
      output.variables['E'][:] = E
      output.close()

if simulation =='CC':
  ####################################
  # CC only (LUC=set to 1851)
  ###################################
  data0 = Dataset('%s/landcover/PFTmap_WithoutNoBio/5m/PFTmap_LUHv2_BM3_HoughtonCountryForestarea_withoutNoBio_%04i_5m.nc' % (dir1,1851),'r')  #crop fraction
  crop = data0.variables['maxvegetfrac'][0,11:13,:,:]
  crop=crop.reshape(2,rows,kols)
  crop[crop>1.] = np.nan
  crop[crop<0.] = np.nan
  crop=np.nansum(crop,axis=0)
  crop[np.isnan(crop)==True] = 0.
  crop=np.ravel(crop)
  data0.close()

  for t in years:
    data2 = Dataset('%s/climate/R_factor/R/ISIMIP2b_R_%04i_5m.nc' % (dir1,t),'r') # R factor
    R = data2.variables['r'][:] 
    R[R<0]=0.
    R[R>10e35]=0.
    R[np.isnan(R)==True] = 0.
    R=np.ravel(R)
    data2.close()    
      
    for lc in range(len(landcover)):
      data1 = Dataset('%s/C_factor/lowres/new_method/new_CC/C_%04i_5m.nc' % (dir1,t),'r') # C factor
      C = data1.variables['c_lc'][lc,:,:]
      C[C<0]=0.
      C[np.isnan(C)==True] = 0.
      C=np.ravel(C)   
        
      #calculate erosion for this timeperiod
      veget=landcover[lc]
      E=erosion.erosion.main(S,K,R,C,veget,crop,gravel,rows,kols)
      E = E.reshape(rows,kols)      
      #output
      output = Dataset('%s/erosion_CC/E_%s_%04i.nc' % (dir1,veget,t),'w')
      output.createDimension('lat',rows)
      output.createDimension('lon',kols)
      output.createVariable('E','d',('lat','lon',))
      output.variables['E'][:] = E
      output.close()    
      data1.close()    

if simulation =='LUC':
  ####################################
  # LUC only (CC=set to 1851-1860)
  ###################################
  data0 = Dataset('%s/climate/R_factor/R/ISIMIP2b_R_1851-1860_mean_5m.nc' % (dir1),'r') # R factor
  R = data0.variables['r'][:] 
  R[R<0]=0.
  R[R>10e35]=0.
  R[np.isnan(R)==True] = 0.
  R=np.ravel(R)
  data0.close() 

  for t in years: 
    data1 = Dataset('%s/landcover/PFTmap_WithoutNoBio/5m/PFTmap_LUHv2_BM3_HoughtonCountryForestarea_withoutNoBio_%04i_5m.nc' % (dir1,t),'r') #crop fraction
    crop = data1.variables['maxvegetfrac'][1,11:13,:,:]
    crop=crop.reshape(2,rows,kols)
    crop[crop>1.] = np.nan
    crop[crop<0.] = np.nan
    crop=np.nansum(crop,axis=0)
    crop[np.isnan(crop)==True] = 0.
    crop=np.ravel(crop)
    data1.close()

    for lc in range(len(landcover)):    
      data1 = Dataset('%s/C_factor/lowres/new_method/new_LUC/C_%04i_5m.nc' % (dir1,t),'r') # C factor
      C = data2.variables['c_lc'][lc,:,:]
      C[C<0]=0.
      C[np.isnan(C)==True] = 0.
      C=np.ravel(C)   
        
      #calculate erosion for this timeperiod
      veget=landcover[lc]
      E=erosion.erosion.main(S,K,R,C,veget,crop,gravel,rows,kols)
      E = E.reshape(rows,kols)      
      #output
      output = Dataset('%s/erosion_LUC/E_%s_%04i.nc' % (dir1,veget,t),'w')
      output.create_dimension('lat',rows)
      output.create_dimension('lon',kols)
      output.create_variable('E','d',('lat','lon',))
      output.variables['E'][:] = E
      output.close()    
      data1.close()    	    

