clc; clear;

%% Physical constants

Omega = 7.29212e-5;                 % angular frequency (rad/sec)
R_earth = 6371220.0;                % mean radius (m)
g = 9.80616;                        % gravity acceleration (m/sec^2)

%% Solver parameters

latitude_d = [-30 30];              % location of zonal boundaries (deg)
longitude_d = [-180 180];           % location of meridional (periodical) boundaries (deg)

latitude = latitude_d * pi / 180;
longitude = longitude_d * pi / 180;

% channel depth (m)
H0 = 30;
% H0 = 0.5;                           

% duration of the simulation (days)
days = 200;                         % Rossby H0 = 30 m
% days = 20;                          % EIG H0 = 30 m
% days = 1300;                        % Rossby H0 = 0.5 m
% days = 60;                          % EIG H0 = 0.5 m

nl = 1;                             % linear or non-linear
ra = 0.00;                          % Robert-Asselin filter
dt = 600;                           % time step (sec)
N = 86400 * days / dt;              % number of iterations

% sampling spacing 
plot_rate = 3600/dt * 48;           % Rossby H0 = 30 m evry 48 hours
% plot_rate = 3600/dt * 4;            % EIG H0 = 30 m evry 4 hours
% plot_rate = 3600/dt * 312;          % Rossby H0 = 0.5 m evry 312 hours 
% plot_rate = 3600/dt * 12;           % EIG H0 = 0.5 m evry 12 hours

save_data = 1;                      % save data or not
gifname = 'test.gif';               % name of gif file
dataname = 'test.mat';              % name of data file

% Grid properties
n_x = 721;                          % number of cells in the zonal direction
n_y = 121;                          % number of cells in the meridional direction

% Zonal and meridional vectors
x = linspace(longitude(1),longitude(2),n_x);
y = linspace(latitude(1),latitude(2),n_y);

dx = x(end)-x(end-1);
dy = y(end)-y(end-1);

x(end+1) = x(end)+dx;
y(end+1) = y(end)+dy;

% Grid size
nx = n_x + 1;
ny = n_y + 1;

% Relevant matrix indices.
i = 2 : (nx - 1);
im = i - 1; 
ip = i + 1;
im(1) = nx - 1; ip(end) = 2;        % periodical boundaries in zonal direction

j = 2 : (ny - 1);
jm = j - 1;
jp = j + 1;

% Coriolis parameter 
fU = repmat(2 * Omega .* sin(y - 0.5 * dy), nx, 1); 
fV = repmat(2 * Omega .* sin(y), nx, 1);

% Other variables
cos_y = repmat(cos(y), nx, 1);
tan_y = repmat(tan(y), nx, 1);

d2x = 2 * dx;
d4x = 4 * dx;
dx2 = dx^2;
d2y = 2 * dy;
d4y = 4 * dy;
dy2 = dy^2;
tm = 1; tn = 2; tp = 3;             % Time indices
delt = dt;
U = zeros(nx, ny, 3);
V = zeros(nx, ny, 3);
H = zeros(nx, ny, 3);
grav = zeros(nx, ny);
vel = zeros(nx, ny);
adv = zeros(nx, ny);
xadv1 = zeros(nx, ny);
xadv2 = zeros(nx, ny);
xadv = zeros(nx, ny);
sph = zeros(nx, ny);

DH1 = zeros(nx, ny);
DH2 = zeros(nx, ny);


% For data logging
data_idx = 1;
data_logs = 1 + floor(N / plot_rate) + 1 * (mod(N, plot_rate) ~= 0);
if (save_data)
    dataf = matfile(dataname, 'writable', true);
    dataf.time = zeros(1, data_logs);
    dataf.lat = y(j)*180/pi;
    dataf.lon = x(i)*180/pi;
    dataf.H = zeros(nx - 2, ny - 2, data_logs);
    dataf.U = zeros(nx - 2, ny - 2, data_logs);
    dataf.V = zeros(nx - 2, ny - 2, data_logs);
end


%% Initial conditions 

amp = 1e-8;                     % wave-amplitude (m/sec)
k = 5;                          % spherical wave-number (dimensionless)
n = 1;                          % wave-mode (dimensionless)

% wave-type
wt = 'Rossby';
% wt = 'EIG';

[uu,vv,hh,~,~] = matsuno(y',x,0,H0,k,n,amp, wt);
H(:, :, tm) = squeeze(hh)' + H0;
U(:, :, tm) = squeeze(uu)' .* H(:, :, tm);
V(:, :, tm) = squeeze(vv)' .* H(:, :, tm);

% Periodic boundaries in zonal direction.
H(1, :, tm) = H(nx - 1, :, tm); H(nx, :, tm) = H(2, :, tm);
U(1, :, tm) = U(nx - 1, :, tm); U(nx, :, tm) = U(2, :, tm);
V(1, :, tm) = V(nx - 1, :, tm); V(nx, :, tm) = V(2, :, tm);

% Rigid boundaries in meridional direction.
V(:, 1, tm) = 0; V(:, ny - 1, tm) = 0;

% Setting current and past same to be the same for first iteration.
H(:, :, tn) = H(:, :, tm);
U(:, :, tn) = U(:, :, tm);
V(:, :, tn) = V(:, :, tm);

hold off;

%% Time stepping loop

for t = 0 : N
    
    % Plotting
    if (mod(t, plot_rate) == 0) || t == N
        [L,T] = meshgrid(x(i) ,y(j) );

        clf;

        pcolor(L * 180 / pi ,T * 180 / pi ,(V(i, j, tn)./H(i, j, tn))');

        colorbar, shading interp;
        colormap jet
        
        title(sprintf('%g days',t * dt / 3600 /24))
        
        hold on;
        xlabel('longitude [\circE]');
        ylabel('latitude [\circN]');
        axis([longitude_d(1) longitude_d(2) latitude_d(1) latitude_d(2) -1.5*amp 1.5*amp]);
        drawnow;
        
        if (save_data)
            % GIF creation
            frame = getframe(1);
            img = frame2im(frame);
            [imind, cm] = rgb2ind(img, 256);
            if (t == 0)
                imwrite(imind, cm, gifname, 'gif', 'Loopcount', inf, 'DelayTime', 0);
            else
                imwrite(imind, cm, gifname, 'gif', 'WriteMode', 'append', 'DelayTime', 0);
            end 
            
            % Data logging
            dataf.time(1, data_idx) = t * dt / 86400;
            dataf.H(:, :, data_idx) = H(i, j, tn);
            dataf.U(:, :, data_idx) = U(i, j, tn)./ H(i, j, tn);
            dataf.V(:, :, data_idx) = V(i, j, tn)./ H(i, j, tn);
            data_idx = data_idx + 1;
            
        end
    end
    
    % Advancing H.    
    H(i, j, tp) = H(i, j, tm) - delt * (1 ./ (R_earth * cos_y(i , j))) .*  ((U(i, j, tn) - U(im, j, tn)) / dx + (V(i, j, tn).* cos_y(i,j) - V(i, jm, tn).* cos_y(i,jm))/ dy);
    H(1, :, tp) = H(nx - 1, :, tp); H(nx, :, tp) = H(2, :, tp); % Periodic boundaries in zonal direction
                                                
    % Advancing U.
    grav(i, j) = (H(ip, j, tn) - H(i, j, tn)) .* (H(ip, j, tn) + H(i, j, tn)) / d2x; % Gravitation: 0.5 * d/dx(H^2)
    vel(i, j) = (V(i, j, tn) + V(i, jm, tn) + V(ip, j, tn) + V(ip, jm, tn)) / 4; % Average velocity
    adv(i, j) = ((U(ip, j, tn) + U(i, j, tn)).^2 ./ H(ip, j, tn) - (U(i, j, tn) + U(im, j, tn)).^2 ./ H(i, j, tn)) / d4x; % Advection: d/dx(U^2/H)
    % Cross advection
    DH1(i, j) = H(i, j, tn) + H(i, jp, tn) + H(ip, j, tn) + H(ip, jp, tn);
    DH2(i, j) = H(i, j, tn) + H(i, jm, tn) + H(ip, j, tn) + H(ip, jm, tn);
    xadv1(i, j) = (U(i, jp, tn) + U(i, j, tn)) .* (V(i, j, tn) + V(ip, j, tn)) ./ DH1(i, j);
    xadv1(:, ny - 1) = 0;
    xadv2(i, j) = (U(i, j, tn) + U(i, jm, tn)) .* (V(ip, jm, tn) + V(i, jm, tn)) ./ DH2(i, j);
    xadv2(:, 2) = 0;
    xadv(i, j) = (xadv1(i, j) - xadv2(i, j)) / dy; % d/dy(UV/H)
    sph(i, j) = -2 * U(i, j, tn) .* V(i, j, tn) ./ H(i, j, tn) .* tan_y(i , j); 

    % Finite difference equation
    U(i, j, tp) = U(i, j, tm) - delt *(g ./ (R_earth * cos_y(i,j)) .* grav(i, j) - fU(i, j) .* vel(i, j)) - nl * delt * 1/R_earth * (1 ./cos_y(i , j) .* adv(i, j) + xadv(i, j) + sph(i,j));
    U(1, :, tp) = U(nx - 1, :, tp); U(nx, :, tp) = U(2, :, tp); % Periodic boundaries in zonal direction
    
    % Advancing V.
    grav(i, j) = (H(i, jp, tn) - H(i, j, tn)) .* (H(i, jp, tn) + H(i, j, tn)) / d2y; % Gravitation [0.5*d/dy(H^2)]
    vel(i, j) = (U(i, j, tn) + U(im, j, tn) + U(i, jp, tn) + U(im, jp, tn)) / 4; % Average velocity
    adv(i, j) = ((V(i, jp, tn) + V(i, j, tn)).^2 ./ H(i, jp, tn) - (V(i, j, tn) + V(i, jm, tn)).^2 ./ H(i, j, tn)) / d4y; % Advection: d/dy(V^2/H)
    % Cross advection
    DH1(i, j) = H(i, j, tn) + H(i, jp, tn) + H(ip, j, tn) + H(ip, jp, tn);
    DH2(i, j) = H(i, j, tn) + H(im, j, tn) + H(i, jp, tn) + H(im, jp, tn);
    xadv1(i, j) = (U(i, jp, tn) + U(i, j, tn)) .* (V(i, j, tn) + V(ip, j, tn)) ./ DH1(i, j);
    xadv2(i, j) = (U(im, j, tn) + U(im, jp, tn)) .* (V(i, j, tn) + V(im, j, tn)) ./ DH2(i, j);
    xadv(i, j) = (xadv1(i, j) - xadv2(i, j)) / dx; % d/dx(UV/H)
    sph(i, j) = (U(i, j, tn) - V(i, j, tn)) .* (U(i, j, tn) + V(i, j, tn)) ./ H(i, j, tn) .* tan_y(i , j);

    % Finite difference equation
    V(i, j, tp) = V(i, j, tm) - delt *(g / R_earth * grav(i, j) + fV(i, j) .* vel(i, j)) - nl * delt * 1/R_earth * (adv(i, j) + 1 ./cos_y(i , j) .*  xadv(i, j) + sph(i,j));
    V(1, :, tp) = V(nx - 1, :, tp); V(nx, :, tp) = V(2, :, tp); % Periodic boundaries in zonal direction
    V(:, 1, tp) = 0; V(:, ny - 1, tp) = 0; % Rigid boundaries in meridional direction
    
    % Robert-Asselin Filter.
    if (t > 5)
        U(:, :, tn) = U(:, :, tn) + ra * (U(:, :, tp) - 2 * U(:, :, tn) + U(:, :, tm));
        V(:, :, tn) = V(:, :, tn) + ra * (V(:, :, tp) - 2 * V(:, :, tn) + V(:, :, tm));
        H(:, :, tn) = H(:, :, tn) + ra * (H(:, :, tp) - 2 * H(:, :, tn) + H(:, :, tm));
    end
    
    % Moving on in time.
    delt = 2 * dt;
    tm = mod(tm, 3) + 1;
    tn = mod(tn, 3) + 1;
    tp = mod(tp, 3) + 1;
end