function [u,v,h,Phi,omega]=matsuno(lat,lon,time,H0,k,n,amp,wave_type)
% MATSUNO Evaluates the analytic solutions for the proposed test case on
% arbitrary lat x lon grids.
%
% *This code is only valid for wave-numbers k>=1 and wave-modes n>=1.
% Special treatments are required for k=0 and n=-1,0/-.*
%
% Inputs:
%     lat        - *column* vector of desired latitudes (radians) 
%     lon        - *row* vector of desired longitudes (radians)
%     time       - 1D array of desired times (sec)
%     H0         - Layer's mean thickness (m)
%     k          - spherical wave-number (dimensionless)
%     n          - wave-mode (dimensionless)
%     amp        - wave-amplitude (m/sec)
%     wave_type  - string of wave-type: 'WIG', 'Rossby', or 'EIG'
%
% Outouts:
%     u          - 3D array (time,lat,lon): zonal velocity (m/sec)
%     v          - 3D array (time,lat,lon): meridional velocity (m/sec)
%     h          - 3D array (time,lat,lon): free-surface height anomaly (m)
%     Phi        - 3D array (time,lat,lon): geopotential height (m^2/sec^2)
%     omega      - wave-frequency (rad/sec)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % global parameters
    OMEGA = 7.29212e-5;                % angular frequency (rad/sec)
    G = 9.80616;                       % gravity acceleration (m/sec^2)
    A = 6371220.0;                     % mean radius (m)
    EPSILON = (2.0*OMEGA*A)^2/(G*H0);  % Lamb's parameter
    
    % get fields
    [u,v,h,Phi] = get_fields(lat,lon,time,k,n,amp,wave_type);
    
    % get frequencies
    omega = get_omega(k,n,wave_type);
    
    function omega=get_omega(k,n,wave_type)
    % Evaluates the wave-frequency.

        omegaj = zeros(1,3);
        delta0 = 3.0*(G*H0*(k/A)^2+2.0*OMEGA*(G*H0)^0.5/A*(2*n+1));
        delta4 = -54.0*OMEGA*G*H0*k/A^2;

        for j=1:3
            deltaj = (delta4^2 - 4.0*delta0^3 + 0.0*1i)^0.5;
            deltaj = (0.5 * (delta4 + deltaj))^(1.0/3.0);
            deltaj = deltaj * exp(2.0*pi*1i*j/3.0);
            omegaj(1,j) = real(-1.0/3.0*(deltaj + delta0/deltaj));
        end

        if strcmp(wave_type,'Rossby')
            omega = -min(abs(omegaj));
        elseif strcmp(wave_type,'WIG')
            omega = min(omegaj);
        elseif strcmp(wave_type,'EIG')
            omega = max(omegaj);
        end
    
    end
    
    function H_n=get_hermite_polynomial(x,n)
    % Evaluates the normalized Hermite polynomial of degree n using the
    % three-term recurrence relation.

        if n == 0
            H_n = (1/pi)^0.25*ones(size(x));
        elseif n == 1
            H_n = (4/pi)^0.25*x;
        elseif n >= 2
            H_n = (2.0/n)^0.5*x.*get_hermite_polynomial(x,n-1) - ...
                  ((n-1)/n)^0.5*get_hermite_polynomial(x,n-2);             
        end  
        
    end

    function psi_n=get_psi(lat,n,amp)
    % Evaluates the eigenfunction psi.

        % re-scale latitude
        y = EPSILON^0.25*lat;

        % Gaussian envelope
        ex = exp(-0.5*y.^2);

        psi_n = amp*ex.*get_hermite_polynomial(y,n);

    end

    function [u_hat,v_hat,p_hat]=get_amplitudes(lat,k,n,amp,wave_type)
    % Evaluates the latitude dependent amplitudes

        omega = get_omega(k,n,wave_type);

        psi_n = get_psi(lat,n,amp);
        psi_n_plus_1 = get_psi(lat,n+1,amp);
        psi_n_minus_1 = get_psi(lat,n-1,amp);

        v_hat = psi_n;
             
        u_hat = (- ((n+1)/2)^0.5*(omega/(G*H0)^0.5+k/A)*psi_n_plus_1 ...
                 - ((n)/2)^0.5*(omega/(G*H0)^0.5-k/A)*psi_n_minus_1);
             
        p_hat = (- ((n+1)/2)^0.5*(omega+(G*H0)^0.5*k/A)*psi_n_plus_1 ...
                 + ((n)/2)^0.5*(omega-(G*H0)^0.5*k/A)*psi_n_minus_1);

        % pre-factors
        u_hat = G*H0*EPSILON^0.25/(1i*A*(omega^2-G*H0*(k/A)^2))*u_hat;
        p_hat = G*H0*EPSILON^0.25/(1i*A*(omega^2-G*H0*(k/A)^2))*p_hat;
        
    end

    function [u,v,h,Phi]=get_fields(lat,lon,time,k,n,amp,wave_type)
    % Evaluates the fields.

        % number of grid points
        nj = length(lat);
        ni = length(lon);
        nt = length(time);

        % preallocate
        u = zeros(nt,nj,ni);
        v = zeros(nt,nj,ni);
        Phi = zeros(nt,nj,ni);

        % frequency
        omega = get_omega(k,n,wave_type);

        % latitude-dependent amplitudes
        [u_hat,v_hat,p_hat] = get_amplitudes(lat,k,n,amp,wave_type);

        % adding time and longitude dependence
        for t=1:nt
            u(t,:,:) = real(u_hat*exp(1i*(k*lon-omega*time(t))));
            v(t,:,:) = real(v_hat*exp(1i*(k*lon-omega*time(t))));
            Phi(t,:,:) = real(p_hat*exp(1i*(k*lon-omega*time(t))));

        end
        
        % transform to free-surface height anomaly
        h = Phi/G;

    end
        
end

    
