SUBROUTINE ec2ald_sst(infile,outfile)

 !
 ! Interpolation of SST and SIC with land-sea mask
 !
 ! Based on ec2ald by Ulf Andrae, 2005, SMHI
 ! Mariken Homleid, 2008, met.no
 !
 ! Jisk Attema, KNMI, 2012 Add RACMO as possible host model

 ! Modules

 USE glkind, ONLY: jpim,jprb
 USE typez,     ONLY : geometry,grib_api_key
 USE grib_api
 USE grib_api_list,  ONLY : glist,lgribprint,lgribinfo
 USE interpol,  ONLY : setup_interpol,setup_weights,interpolate,        &
                       int_print,missing_data
 USE namelist_control, ONLY : outgeo,readkey,order,climate_file,        &
                              printlev,outkey,rmisval
 USE cadre,     ONLY : use_saved_cadre
 USE konstants, ONLY : eps

 IMPLICIT NONE

 ! Input

 CHARACTER(LEN=*), INTENT(IN   ) :: infile
 CHARACTER(LEN=*), INTENT(INOUT) :: outfile
 LOGICAL :: lsmask = .TRUE.

 ! Local

 INTEGER(KIND=jpim) :: i,j,iret,     &
            faunit,                  &
            nreadkey,                &
            knrec,                   &
            ksst,kice,kts,           &
            mr,gid

 REAL(KIND=jprb)   :: wx,wy
 REAL(KIND=jprb)   :: FMIN, FMAX, FMEAN

 REAL(KIND=jprb),    ALLOCATABLE ::  allfld(:,:,:),                         &
                         fullfld(:,:,:),rdum(:,:)

 LOGICAL, ALLOCATABLE :: ltmp(:,:), mask1(:,:)

 LOGICAL :: lwrap = .FALSE.

 TYPE (grib_api_key)  :: climkey
 TYPE (geometry) :: ingeo,climgeo,filegeo,intpgeo

 ! ----------------------------------------------------------------

 ! Open and close the climate file to get the output geometry
 faunit = 23
 use_saved_cadre = .TRUE.
 CALL openfa(faunit,climate_file,climgeo,climkey,knrec)
 climgeo%nlev = -1
 IF (printlev > 0 ) THEN
    WRITE(6,*)'Climate geometry:'
    CALL print_geo(climgeo)
 ENDIF

 !
 ! Open infile in order to get geometry and fill file index
 !

 lgribprint = .FALSE.
 lgribinfo  = .FALSE.
 gid = -1

 CALL gropen(gid,infile)

 ! Copy geometric information
 ingeo   = glist(gid)%geo
 filegeo = glist(gid)%geo

 IF ( ingeo%gridtype == 'regular_ll' ) THEN

    !
    ! Check if this is an wrap around field
    !
    ! make it between [-180,180]
    !

    wx = ingeo%west - ingeo%dlon
    IF ( wx >  180._jprb ) wx = wx - 360._jprb
    IF ( wx < -180._jprb ) wx = wx + 360._jprb
    wy = ingeo%east
    IF ( wy >  180._jprb ) wy = wy - 360._jprb
    IF ( wy < -180._jprb ) wy = wy + 360._jprb

    IF ( ABS(wx-wy) < 1.e-3_jprb ) THEN

      WRITE(6,*)'This is a wrap around field'
      WRITE(6,*)'Old geometry'
      CALL print_geo(ingeo)

      !
      ! Add one column at each end
      !

      lwrap      = .TRUE.
      i = 4
      ingeo%nlon = ingeo%nlon + i
      ingeo%east = ingeo%east + (i/2)*ingeo%dlon
      ingeo%west = ingeo%west - (i/2)*ingeo%dlon

      !
      ! Make sure ingeo%west < ingeo%east
      !

      IF ( ingeo%west > ingeo%east ) THEN
        ingeo%west = ingeo%west - 360._jprb
        ingeo%east = ingeo%east - 360._jprb
      ENDIF

      WRITE(6,*)'New geometry'
      CALL print_geo(ingeo)
      WRITE(6,*)

    ENDIF

 ENDIF

 !
 ! Setup output geometry and namelist
 !

 CALL setup_ecmwf_sst(nreadkey,climgeo)

 IF (printlev > 0 ) THEN

    WRITE(6,*)'Input geometry:'
    CALL print_geo(ingeo)

    WRITE(6,*)'Output geometry:'
    CALL print_geo(outgeo)

 ENDIF

 !
 ! Allocate and initialize fields
 !
 ALLOCATE(allfld(outgeo%nlon,outgeo%nlat,nreadkey), &
          fullfld(ingeo%nlon, ingeo%nlat,nreadkey))

  allfld = 0._jprb
 fullfld = 0._jprb


 !
 ! Read ECMWF SST, SIC
 !

 WRITE(6,*)'Read SST, SIC and Ts '

 DO j=1,nreadkey


    ! Store position of fields
    SELECT CASE(readkey(j)%faname)

    CASE('SURFSEA.TEMPERA')
      ksst = j
    CASE('SURFSEA.ICECONC')
      kice = j
    CASE('SURFTEMPERATURE')
      kts = j
    CASE DEFAULT
      WRITE(6,*)'Dont know key:'
      CALL print_grib_api_key('Could not find:',readkey(j))
      CALL abort
    END SELECT

    !
    ! Read from file
    !

    CALL gread(gid,filegeo%nlon,filegeo%nlat,readkey(j),  &
               fullfld(1:filegeo%nlon,1:filegeo%nlat,j))

    IF ( readkey(j)%pos == -1 .OR. readkey(j)%pos == -99 ) THEN

      readkey(j)%tri = 1
      CALL gread(gid,filegeo%nlon,filegeo%nlat,readkey(j),  &
                 fullfld(1:filegeo%nlon,1:filegeo%nlat,j))

      IF ( readkey(j)%pos == -1 .OR. readkey(j)%pos == -99 ) THEN
        CALL print_grib_api_key('Could not find:',readkey(j))
        CALL abort
      ENDIF

    ENDIF

    IF ( lwrap ) CALL wrap_field(filegeo,ingeo,nreadkey,j,fullfld)

 ENDDO

 CALL grib_close_file(glist(gid)%unit,iret)

 !
 ! Setup interpolation
 !

 IF (printlev > 1 ) THEN
    WRITE(6,*)
    WRITE(6,*)'Prepare for interpolation'
    WRITE(6,*)
 ENDIF

 !
 ! Do not interpolate the extention zone
 !
 intpgeo = outgeo
 intpgeo%nlon = outgeo%nlon - outgeo%nlon_e
 intpgeo%nlat = outgeo%nlat - outgeo%nlat_e

 ! Setup the interpolation weights
 ALLOCATE(ltmp(ingeo%nlon,ingeo%nlat))
 CALL setup_interpol(ingeo,intpgeo)


 DO j=1,nreadkey

    IF (lsmask .AND. j /= kts) THEN
       WRITE(6,*) 'Define interpolation mask ltmp from sst>271. '
       ltmp = ( ABS(fullfld(1:ingeo%nlon,1:ingeo%nlat,ksst)-rmisval) > eps )
    ELSE
       ltmp = .TRUE.
    ENDIF

    int_print = printlev
    CALL setup_weights(order,intpgeo,ingeo,ltmp)

    FMIN = MINVAL(fullfld(1:ingeo%nlon,1:ingeo%nlat,j))
    FMAX = MAXVAL(fullfld(1:ingeo%nlon,1:ingeo%nlat,j))
    FMEAN = SUM(fullfld(1:ingeo%nlon,1:ingeo%nlat,j))/  &
          FLOAT(ingeo%nlon*ingeo%nlat)
    WRITE(6,*)'Before interpolate, j =', j
    WRITE(6,*)' FMIN =', FMIN, ' FMEAN =', FMEAN, ' FMAX =', FMAX

    IF (printlev > 1 ) CALL print_grib_api_key('Interpolate:',readkey(j))
    allfld(:,:,j) = missing_data
    CALL interpolate(order,                      &
                  ingeo%nlon, ingeo%nlat,fullfld(:,:,j), &
                  intpgeo%nlon,intpgeo%nlat,             &
                  allfld(1:intpgeo%nlon,1:intpgeo%nlat,j))

    FMIN = MINVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,j))
    FMAX = MAXVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,j))
    FMEAN = SUM(allfld(1:intpgeo%nlon,1:intpgeo%nlat,j))/  &
          FLOAT(intpgeo%nlon*intpgeo%nlat)
    WRITE(6,*)'After interpolate, j =', j
    WRITE(6,*)' FMIN =', FMIN, ' FMEAN =', FMEAN, ' FMAX =', FMAX

 ENDDO

 DEALLOCATE(ltmp,fullfld)

 ! v1: Replace SST values over sea ice with 271. (~ -2.16 deg. C)
 ! v2: Replace SST values over sea ice with surface temperature
 ! v3: Replace SST values < -1 with surface temperature
 ! v5: Replace SST values < 272.16 with surface temperature
 WHERE( ABS(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst)-missing_data)>1. .and. &
            allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst) < 272.16_jprb)
    allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst) = allfld(1:intpgeo%nlon,1:intpgeo%nlat,kts)
 END WHERE

 FMIN = MINVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst))
 FMAX = MAXVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst))
 FMEAN = SUM(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst))/  &
                    FLOAT(intpgeo%nlon*intpgeo%nlat)
 WRITE(6,*)'After repl. with Ts over ice, ksst =', ksst
 WRITE(6,*)' FMIN =', FMIN, ' FMEAN =', FMEAN, ' FMAX =', FMAX

 ! Fill SST field along coast
 WRITE(6,*) 'Call fill_missing to fill data gaps, mr<0 ==> fill all gaps'
 mr = -5
 WRITE(6,*) 'Max radius to search for valid data points: mr =', mr
 ! Mask 1 tells you where you have your missing data
 ! Mask 2 tells which data you can use when searching
 ALLOCATE(mask1(intpgeo%nlon,intpgeo%nlat))
 mask1 = ( ABS(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst)-missing_data)>1. )
 ALLOCATE(rdum(intpgeo%nlon,intpgeo%nlat))

 CALL fill_missing(intpgeo%nlon,intpgeo%nlat,mr,                             &
                   mask1,mask1,allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst),   &
                   rdum,.FALSE.,1.0_jprb,.FALSE.,0)

 FMIN = MINVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst))
 FMAX = MAXVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst))
 FMEAN = SUM(allfld(1:intpgeo%nlon,1:intpgeo%nlat,ksst))/  &
                    FLOAT(intpgeo%nlon*intpgeo%nlat)
 WRITE(6,*)'After fill_missing, ksst =', ksst
 WRITE(6,*)' FMIN =', FMIN, ' FMEAN =', FMEAN, ' FMAX =', FMAX

! Fill SIC field along coast
 WRITE(6,*) 'Call fill_missing to fill data gaps, mr<0 ==> fill all gaps'
 mr = -5
 WRITE(6,*) 'Max radius to search for valid data points: mr =', mr
 ! Mask 1 tells you where you have your missing data
 ! Mask 2 tells which data you can use when searching
 mask1 = ( ABS(allfld(1:intpgeo%nlon,1:intpgeo%nlat,kice)-missing_data)>1. )

 CALL fill_missing(intpgeo%nlon,intpgeo%nlat,mr,                             &
                   mask1,mask1,allfld(1:intpgeo%nlon,1:intpgeo%nlat,kice),   &
                   rdum,.FALSE.,1.0_jprb,.FALSE.,0)

 DEALLOCATE(mask1)
 DEALLOCATE(rdum)


 FMIN = MINVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,kice))
 FMAX = MAXVAL(allfld(1:intpgeo%nlon,1:intpgeo%nlat,kice))
 FMEAN = SUM(allfld(1:intpgeo%nlon,1:intpgeo%nlat,kice))/  &
                    FLOAT(intpgeo%nlon*intpgeo%nlat)
 WRITE(6,*)'After fill_missing, kice =', kice
 WRITE(6,*)' FMIN =', FMIN, ' FMEAN =', FMEAN, ' FMAX =', FMAX

 !
 ! Biperiodize the interpolated gridpoint data (because it is not done in output_2_fa ) ?
 ! BAD IDEA HERE: better set to missing!
 !

 DO j=1,nreadkey

    IF (readkey(j)%nnn == 1) CYCLE
    allfld(  (outgeo%nlon-outgeo%nlon_e+1):outgeo%nlon,:, j) = missing_data
    allfld( :, (outgeo%nlat-outgeo%nlat_e+1):outgeo%nlat, j) = missing_data

    IF (printlev > 1 ) CALL print_grib_api_key('Fill extension zone:',readkey(j))

 ENDDO

 !
 ! Output
 !

 IF (LEN(TRIM(outfile)) == 0 ) outfile = TRIM(infile)//'.ald'

 IF (ASSOCIATED(glist(gid)%ahalf)) DEALLOCATE(glist(gid)%ahalf,glist(gid)%bhalf)
 IF (ASSOCIATED(glist(gid)%afull)) DEALLOCATE(glist(gid)%afull,glist(gid)%bfull)

 ALLOCATE(glist(gid)%ahalf(outgeo%nlev+1),glist(gid)%bhalf(outgeo%nlev+1))
 glist(gid)%ahalf = (/0.0_jprb,0.0_jprb/)
 glist(gid)%bhalf = (/0.5_jprb,1.0_jprb/)

 CALL  output_2_fa(outkey,        &
                   nreadkey,      &
                   outfile,       &
                   .FALSE.,       &
                   allfld,        &
                   outgeo%nlon,outgeo%nlat,gid)

 DEALLOCATE(allfld)

 RETURN

END SUBROUTINE ec2ald_sst
