module theoretic_snow
! Simplest snow model, based on assumption of constancy of snow thickness and
! properties during model run.
!
! AUTHOR
! Y. Batrak
!
! MODIFICATIONS
! Original  04/2014

  use abstract_snow,  only: tAbstract_snow

  use YOMHOOK,        only: LHOOK, DR_HOOK
  use PARKIND1,       only: JPRB
  implicit none
  private

    integer, save :: num_snow_layers
    real,    save :: snow_height

    type, public, extends( tAbstract_snow ) :: tTheoretic_snow
        integer ::      &
            num_layers
        real, dimension( :, : ), pointer ::    &
            T,          &
            z,          &
            dz,         &
            z_diff
        real, dimension( : ),   pointer :: &
            F_bottom,   &   !< Heat flux from the snow to snow-ice interface.
            albedo          !< Snow integral albedo.
      contains
        procedure :: construct        => construct_snow
        procedure :: run              => run_snow
        procedure :: surf_temperature => surf_snow

        procedure :: exists
        procedure :: heat_flux_from_snow

        procedure :: prep             => snow_prep

        procedure :: surf_albedo

        procedure :: get_model_fields
    end type

  contains
    !===========================================================================
    !TODO: perform snow PREP
    !===========================================================================
    subroutine snow_prep( m, hPROGRAM, hATMFILE, hATMFILETYPE, hPGDFILE, hPGDFILETYPE )
      use MODD_SURF_PAR,  only: XUNDEF
      implicit none
        class( tTheoretic_snow )           :: m
        character( LEN = * ), intent( in ) :: hPROGRAM
        character( LEN = * ), intent( in ) :: hATMFILE
        character( LEN = * ), intent( in ) :: hATMFILETYPE
        character( LEN = * ), intent( in ) :: hPGDFILE
        character( LEN = * ), intent( in ) :: hPGDFILETYPE

        integer       :: i
        real, pointer :: z_ice_surf_temperature(:)

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:PREP', 0, ZHOOK_HANDLE )

        write(*,*) 'Looking for ice temperature...'
        nullify( z_ice_surf_temperature )
        do i = 1, size( m%mf )
            if( m%mf(i)%c_name == 'TICE' ) then
                write(*,*) 'Found'
                if( associated(m%mf(i)%p1) ) z_ice_surf_temperature => m%mf(i)%p1
                if( associated(m%mf(i)%p2) ) z_ice_surf_temperature => m%mf(i)%p2(:,1)
            end if
        end do
        if( .NOT. associated( z_ice_surf_temperature ) ) stop 'NOT found.'

        !-----------------------------------------------------------------------
        !Fill each snow layer by the ice surface temperature
        do i = 1, m%num_layers
            where(m%T(:,i) == XUNDEF) m%T(:,i) = z_ice_surf_temperature
        end do
        !-----------------------------------------------------------------------

        where( z_ice_surf_temperature /= XUNDEF ) m%albedo = 0.85

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:PREP', 1, ZHOOK_HANDLE )
    end subroutine
    !===========================================================================
    ! Allocate snow variables and setup model grid.
    !===========================================================================
    subroutine construct_snow( this, num_points )
      use MODD_SURF_PAR,  only: XUNDEF
      use modn_simpl_ice, only: NICE_SNOW_NLAYERS,  &
                                XICE_SNOW_HEIGHT
      use MODI_SET_GRID
      use MODI_ABOR1_SFX
      implicit none
        class( tTheoretic_snow ) :: this
        integer, intent( in )    :: num_points

        integer :: i, n
        real    :: z_skin

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:CONSTRUCT', 0, ZHOOK_HANDLE )
        !
        !   Specification of ice layer depths
        if ( NICE_SNOW_NLAYERS < 3 ) then
            call abor1_sfx('Less than 3 snow layers.')
        end if

        num_snow_layers = NICE_SNOW_NLAYERS
        snow_height     = XICE_SNOW_HEIGHT

        this%num_points = num_points
        this%num_layers = num_snow_layers

        this%locked     = .FALSE.

        allocate( &
                  this%z     ( this%num_points, this%num_layers ),  &
                  this%dz    ( this%num_points, this%num_layers ),  &
                  this%z_diff( this%num_points, this%num_layers ),  &
                  this%T     ( this%num_points, this%num_layers )   &
                )
        allocate(  this%F_bottom( num_points ),    &
                   this%albedo  ( num_points ) )

        this%T         = XUNDEF
        this%albedo    = XUNDEF
        this%F_bottom  = XUNDEF

        this%z         = XUNDEF

        if( num_points > 0 ) then
            if( snow_height > 0.2 ) then
                z_skin = 0.05
            else
                z_skin = snow_height*0.05/0.2
            end if
            call set_grid( z_skin, snow_height, this%z, this%dz, this%z_diff )
        end if

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:CONSTRUCT', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Perform model run.
    !===========================================================================
    subroutine run_snow( snow, T_ice, forc, diag )
      use MODD_CSTS,     only: XTTSI,   &
                               XCPD,    &
                               XSTEFAN, &
                               XLSTT,   &
                               XOMEGA,  &
                               XDAY

      use MODD_SURF_PAR, only: XUNDEF

      use MODD_SURF_ATM, only: LDRAG_COEF_ARP,  &
                               LRRGUST_ARP,     &
                               XRRSCALE,        &
                               XRRGAMMA,        &
                               XUTILGUST,       &
                               XRIMAX

      use MODD_SNOW_PAR, only: XZ0SN,   &
                               XZ0HSN,  &
                               XWCRN

      use MODE_SNOW3L,   only: SNOW3LSCAP
      use MODE_THERMOS,  only: qsati,   &
                               qsat,    &
                               dqsati,  &
                               dqsat
      use MODE_ICE_ALBEDO

      use MODI_LIN_SPACE
      use MODI_SOIL_HEATDIF
      use MODI_SURFACE_RI
      use MODI_SURFACE_AERO_COND
      use MODI_SURFACE_CD
      use MODI_WIND_THRESHOLD

      use MODN_SIMPL_ICE

      use abstract_model, only: tForcing, tIce_diag
      implicit none
        class( tTheoretic_snow )               :: snow
        real,                 intent( in     ) :: T_ice( : )
        type ( tForcing    ), intent( in     ) :: forc
        type ( tIce_diag   ), intent( in out ) :: diag

        real, dimension( size( forc%Ta ) ) :: &
            zDirCOSzw,  &
            z_Ri,        &
            z_Ch,        &
            z_Cd,        &
            z_Cdn,       &
            z_SFTh,      &
            z_SFTq,      &
            z_Ustar,     &
            z_RESa,      &

            z_Qsat,      &
            z_dQsat,     &

            z_Z0HICE,    &
            z_Z0ICE,     &

            zVmod,      &
            zAc,        &
            zRa,        &
            zUstar2,    &

            ZCT,        &
            ZTTSI,      &
            zSnowEmiss, &
            zterm2,     &
            zterm1,     &
            zExt,       &
            zGrainSize, &

            zBalance,   &
            zLinTerm,   &

            z_alpha, z_beta, z_gamma_T, z_gamma_q,  &
            ppet_a_sfc, ppet_b_sfc,                 &
            ppeq_a_sfc, ppeq_b_sfc,                 &

            z_tmp,      &

            ZRRCOR,     &
            ZFP
        real, dimension( size( forc%Ta ), snow%num_layers ) :: &
            zsnowcond,   &
            zSnowHcap,   &
            z_dq_dz,     &
            z_q_z


        integer :: i, m
        real    :: rhoSnow

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:RUN', 0, ZHOOK_HANDLE )

        m = size( forc%Ta )

        zDirCOSzw = 1.
        !
        z_RI   (:) = XUNDEF
        z_CH   (:) = XUNDEF
        z_CD   (:) = XUNDEF
        z_CDN  (:) = XUNDEF
        !
        z_SFTH (:)=XUNDEF
        z_SFTQ (:)=XUNDEF
        z_USTAR(:)=XUNDEF
        z_RESA (:)=XUNDEF

        !-----------------------------------------------------------------------
        ! Initialization of snow temperature in points with new snow
        do i = 1, m
            if( snow%T( i, 1 ) == XUNDEF ) then
                call lin_space( T_ice( i ) - 1., T_ice( i ), snow%z( i, : ), snow%T( i, : ) )
                snow%albedo = 0.85
            else
            end if
        end do
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Snow albedo evolution
        if( .NOT.snow%locked ) then
            where( snow%T(:m,1) < 273.16 )
                snow%albedo(:m) = max( .5, snow%albedo(:m) - 0.008*forc%Tstep/xDay )
            elsewhere
                ! Some kind of wet snow's albedo degradation
                snow%albedo(:m) = (snow%albedo(:m) - .5)*exp( -0.24*forc%Tstep/xDay ) + .5
            end where

            ! correction of snow albedo in case of snowfall
            zExt = 1.E2/max( 50., 109. + 6.*( forc%Ta - 273.16 ) + 26.*sqrt(forc%V) )
            zExt = min(1.,forc%Tstep*forc%Prate_s*zExt)
            snow%albedo(:m) = 0.85*zExt + ( 1. - zExt )*snow%albedo(:m)

        end if
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Saturated specified humidity near the snow surface
        z_Qsat  = QSATI ( snow%T(:m,1), forc%Psurf        )
        z_dQsat = DQSATI( snow%T(:m,1), forc%Psurf, z_Qsat )
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Calculate the drag coefficient for momentum (z_CD)

        ! Richardson number
        CALL SURFACE_RI(snow%T(:m,1), z_Qsat, forc%EXNs, forc%EXNa, forc%Ta, forc%Qa, forc%Zref, forc%Uref, zDirCOSzw, forc%V, z_Ri)
        z_RI(:) = MIN(z_RI(:),XRIMAX)

        ! Z0 for  snow
        z_Z0HICE(:) = XZ0HSN
        z_Z0ICE (:) = XZ0SN

        ! Drag coefficient
        zVmod = WIND_THRESHOLD( forc%V, forc%Uref )
        IF ( LDRAG_COEF_ARP ) THEN
            CALL SURFACE_CDCH_1DARP(forc%ZREF, z_Z0ICE, z_Z0HICE , ZVMOD, forc%Ta, snow%T(:m,1), &
                                    forc%QA, z_QSAT, z_CD, z_CDN, z_CH                 )
            ZRA(:) = 1. / ( z_CH(:) * ZVMOD(:) )
        ELSE
            CALL SURFACE_CD( z_Ri, forc%Zref, forc%Uref, z_Z0ICE, z_Z0HICE , z_Cd, z_Cdn)
            ! Drag coefficient for heat and aerodynamical resistance
            CALL SURFACE_AERO_COND( z_RI, forc%ZREF, forc%UREF, ZVMOD, z_Z0ICE, z_Z0HICE , ZAC, ZRA, z_CH )
        END IF
        ZUSTAR2 = z_CD*ZVMOD**2
        z_RESA(:) = ZRA(:)

        IF ( LRRGUST_ARP ) THEN
            ZFP(:)=MAX(0.0,forc%pRate_r(:)+forc%pRate_s(:))
            ZRRCOR(:)=SQRT(1.0+((((ZFP(:)/(ZFP(:)+XRRSCALE))**XRRGAMMA)*XUTILGUST)**2) &
              /(z_CD(:)*ZVMOD(:)**2))

            z_CD  = z_CD *ZRRCOR
            z_CH  = z_CH *ZRRCOR
            z_CDN = z_CDN*ZRRCOR
        ENDIF
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Snow temperature evolution

        !TODO: rhoSnow should be array
        rhoSnow = 300.
        ! Initializations

        ! Snow thermal conductivity [W/(K m)]
        do i = 1, m
          zSnowCond( i, : ) = &
            0.02 + 2.5E-6*rhoSnow**2 +&
            ( -0.06023 - 2.5425/( snow%T( i, : ) - 289.99 ) )*(1.E5/forc%Psurf( i ) )
        end do

        ! Snow volumetric heat capacity [J/(m3 K)]
        zSnowHcap(:,:) = SNOW3LSCAP( rhoSnow )

        ! Thermal resistance of snow, McNider et al. 1994
        zct = 1./sqrt( zSnowHcap( :, 1 )*zSnowCond( :, 1 )/xOmega*0.5 )
        zct = 1./( zSnowHcap( :, 1 )*snow%z(:m,1) )

        ! Deep temperature for lower boundary condition
        ZTTSI(:) = T_ice

        ! Snow emissivity of longwave radiation
        zSnowEmiss  = .99


        zGrainSize = 1.6E-4 + 1.1E-13*rhoSnow**4
        ! Shortwave radiation extinction coefficient, Bohern and Barkstrom 1974, [1/m]
        zExt       = 3.8E-3*rhoSnow/sqrt( zGrainSize )

        do i = 1, m
            z_q_z  (i,:) = ( 1. - snow%albedo(i) )*forc%SW(i)*exp(-zExt(i)*snow%z(i,:))
            z_dq_dz(i,:) = z_q_z  (i,:)*zExt(i)
        end do

        !print*, z_dq_dz(1,:)
        !print*, z_q_z  (1,:)
        !print*, ( 1. - snow%albedo(1) )*forc%SW(1)
        !read(*,*)

        !coefficients for implicit coupling
        z_tmp = (1. - forc%ppet_a_coef*forc%rhoa*z_ch*zvmod)

        ppet_a_sfc = -forc%ppet_a_coef*forc%rhoa*z_ch*zvmod*(forc%EXNa/forc%EXNs)/z_tmp
        ppet_b_sfc = forc%ppet_b_coef*forc%EXNa/z_tmp

        z_tmp = 1. - forc%ppeq_a_coef*forc%rhoa*z_ch*zvmod

        ppeq_a_sfc = - forc%ppeq_a_coef*forc%rhoa*z_ch*zvmod*z_dqsat/z_tmp
        ppeq_b_sfc = (forc%ppeq_b_coef - forc%ppeq_a_coef*forc%rhoa*z_ch*zvmod*( z_qsat - z_dqsat*snow%T(:m,1) ) )/z_tmp

        !surface energy balance

        z_tmp = 0.
        where( ppet_a_sfc*snow%T(:m,1) + ppet_b_sfc > 273.16 ) z_tmp = 1.

        z_alpha = 1./(forc%tStep*zct)                                           &
                  + 4.*zSnowEmiss*xStefan*snow%T(:m,1)**3                       &
                  + forc%RHOa*xcpd*z_ch*zvmod/forc%EXNs                         &
                  + xlstt*forc%RHOa*z_ch*zvmod*z_dqsat                          &
                  + zSnowCond(:,1)/snow%z_diff(:m,1)                            &
                  - z_tmp*forc%Prate_r*4187.*ppet_a_sfc

        z_beta  = 1./(forc%tStep*zct)*snow%T(:m,1)                              &
                  + ( 1. - snow%albedo(:m) )*(forc%SW - z_q_z(:,1))             &
                  + forc%LW                                                     &
                  + 3.*zSnowEmiss*xStefan*snow%T(:m,1)**4                       &
                  - xlstt*forc%RHOa*z_ch*zvmod*( z_qsat - z_dqsat*snow%T(:m,1) )&
                  + z_tmp*forc%Prate_r*4187.*(ppet_b_sfc - 273.16)

        z_gamma_T = forc%RHOa*xcpd*z_ch*zvmod/forc%EXNa
        z_gamma_q = forc%RHOa*xlstt*z_ch*zvmod

        z_tmp = z_alpha - z_gamma_T*ppet_a_sfc - z_gamma_q*ppeq_a_sfc

        zterm1 = (z_beta + z_gamma_T*ppet_b_sfc + z_gamma_q*ppeq_b_sfc)/z_tmp
        zterm2 = zSnowCond(:,1)/snow%z_diff(:m,1)/z_tmp
        ! Determine the soil temperatures:
        CALL SOIL_HEATDIF(forc%TSTEP,snow%Z(:m,:),snow%DZ(:m,:),snow%Z_DIFF(:m,:),zsnowcond,      &
                             zSnowHcap,ZCT,ZTERM1,ZTERM2,ZTTSI,snow%T(:m,:), z_dq_dz )

        where( snow%T(:m,:) > 273.15 )
            snow%T(:m,:) = 273.15
        end where
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! The fluxes

        z_SFTH (:) = XCPD * forc%RHOa(:) * z_CH(:) * ZVMOD(:) * ( (1. -      ppet_a_sfc)*snow%T(:m,1) - ppet_b_sfc )
        z_SFTQ (:) =        forc%RHOa(:) * z_CH(:) * ZVMOD(:) * ( z_QSAT(:) - ppeq_a_sfc *snow%T(:m,1) - ppeq_b_sfc )
        z_USTAR(:) = SQRT(ZUSTAR2(:))

        diag%PZ0ICE (:m) = z_Z0ice
        diag%PZ0HICE(:m) = z_Z0Hice
        !  surface fluxes : latent heat, sensible heat, friction fluxes
        diag%PSFTH  (:m) = z_SFTH
        diag%PSFTQ  (:m) = z_SFTQ
        diag%PUSTAR (:m) = z_USTAR

        diag%PQSAT  (:m) = z_QSAT
        diag%PCD    (:m) = z_CD
        diag%PCDN   (:m) = z_CDN
        diag%PCH    (:m) = z_CH
        diag%PRI    (:m) = z_RI
        diag%PRESA  (:m) = z_RESA

        !-----------------------------------------------------------------------
        ! Heat flux from snow to ice.

        ! Ice albedo hard coded to 0.5, but this shouldn't play any significant role.
        snow%F_bottom(:m) = &
            ( 1. - snow%albedo(:m) )*( 1. - 0.5 )*forc%sw*exp( -zExt*snow_height ) &
            - zsnowcond( :, snow%num_layers ) * (T_ice - snow%T( :m, snow%num_layers )) / snow%z_diff( :m, snow%num_layers )
        !-----------------------------------------------------------------------

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:RUN', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Get snow surface temperature.
    !===========================================================================
    subroutine surf_snow( snow, temp )
      implicit none
        class( tTheoretic_snow ) :: snow
        real, intent( out )      :: temp( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:SURF_T', 0, ZHOOK_HANDLE )

        temp = snow%T( :, 1 )

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:SURF_T', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine surf_albedo( snow, alb )
      implicit none
        class( tTheoretic_snow ) :: snow
        real, intent( out )      :: alb( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:SURF_ALB', 0, ZHOOK_HANDLE )

        alb = snow%albedo

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:SURF_ALB', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Check for snow existence. For this simple model we always have snow.
    !===========================================================================
    function exists( snow ) result( res )
      implicit none
        class( tTheoretic_snow ) :: snow
        logical                  :: res( snow%num_points )

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:EXISTS', 0, ZHOOK_HANDLE )

        res = .TRUE.

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:EXISTS', 1, ZHOOK_HANDLE )
    end function

    !===========================================================================
    ! Return heat flux from snow to ice.
    !===========================================================================
    subroutine heat_flux_from_snow( snow, flux )
      implicit none
        class( tTheoretic_snow ) :: snow
        real, intent( out ) :: flux( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:HEAT_FLUX_FROM_SNOW', 0, ZHOOK_HANDLE )

        flux = snow%F_bottom(:size(flux))

        if( LHOOK ) call DR_HOOK( 'THEORETIC_SNOW:HEAT_FLUX_FROM_SNOW', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine get_model_fields( m, mf )
        use abstract_model, only: model_field
        implicit none
        class( tTheoretic_snow ),         intent( in  ) :: m
        type( model_field ), allocatable, intent( out ) :: mf(:)

        allocate( mf(3) )

        mf    = [model_field(                      &
                   'TSNOW_ICE',                    &
                   'Snow temperature',             &
                   'K',                            &
                   [m%num_points, m%num_layers, 0],&
                   P2 = m%T                        &
                            ),                     &
                 model_field(                      &
                   'ASNOW_ICE',                    &
                   'Snow albedo',                  &
                   'dimensionless',                &
                   [m%num_points, 0, 0],           &
                   P1 = m%albedo                   &
                            ),                     &
                 !model_field( P2 = m%z,        L_INTERNAL = .TRUE. ), &
                 !model_field( P2 = m%dz,       L_INTERNAL = .TRUE. ), &
                 !model_field( P2 = m%z_diff,   L_INTERNAL = .TRUE. ), &
                 model_field( N_CONFIG=[0,0,0], P1 = m%F_bottom, L_INTERNAL = .TRUE. )  &
                ]
    end subroutine
end module
