module simple_ice
! This module implements simple ice scheme, according to interface defined in
! abstract_ice module. This simple scheme based on assumption of constant ice
! thickness and properties. Prognostic variable --- ice temperature.
!
! AUTHOR
! Y. Batrak
!
! MODIFICATIONS
! Original  04/2014
  use abstract_ice, only: tAbstract_ice

  use YOMHOOK,    only: LHOOK, DR_HOOK
  use PARKIND1,   only: JPRB
  implicit none

  private

    type, public, extends( tAbstract_ice ) :: tSimple_ice
        integer ::       &
            num_layers
        real, dimension( :, : ), pointer :: &
            z,     &               !< Depth of lower boundary for each vertical layer.
            dz,    &               !< Thickness of each layer.
            z_diff,&               !< Distance between two layers.
            T => null ()           !< Mean temperature of ice layers.
         real, private, pointer          :: &
            heat_cond(:) => null() !< Pointer for providing information about ice heat
                                   !! conductivity to snow model. (In the case of using '3-L' snow scheme.)
      contains
        procedure :: construct        => construct_ice !< Allocate model variables, initialize model grid.
        procedure :: run              => run_ice
        procedure :: surf_temperature => surf_ice

        procedure :: prep             => ice_prep

        procedure :: surf_alb

        procedure, pass :: release
        procedure, pass :: get_model_fields
    end type

  contains
    !===========================================================================
    ! Calculate initial fields for SIMPLE_ICE scheme. Currently only uniform
    ! prescribed profile supported.
    !===========================================================================
    subroutine ice_prep( m, hPROGRAM, hATMFILE, hATMFILETYPE, hPGDFILE, hPGDFILETYPE )


      use MODD_TYPE_DATE_SURF
      use MODD_CSTS,           only: XTTSI, XTTS, XTT, XPI
      use MODD_SURF_PAR,       only: XUNDEF
      use MODD_SEAFLUX_n,      only: XSST, XSIC, LHANDLE_SIC, LSIC_FROM_FILE, &
                                     LMISSING_ICE_VARIABLES

      use MODD_PREP,           only: CINGRID_TYPE, CINTERP_TYPE

      use MODD_GRID_GAUSS,     only: NINLA, NINLO
      use MODD_GRID_AROME,     only: NX,    NY

      use MODD_SEAFLUX_GRID_n, only: CGRID_SEA      => CGRID, &
                                     XGRID_PAR_SEA  => XGRID_PAR, &
                                     XLAT_SEA       => XLAT, &
                                     XLON_SEA       => XLON
      use MODN_PREP_SEAFLUX,   only: CFILE_SIC, CTYPE_SIC

      use MODE_POS_SURF

      use MODI_ABOR1_SFX

      use MODI_GET_LUOUT
      use MODI_OPEN_NAMELIST
      use MODI_CLOSE_NAMELIST
      use MODI_LIN_SPACE
      use MODI_PREP_GRIB_GRID
      use MODI_HOR_INTERPOL
      use MODI_PREP_OUTPUT_GRID
      use MODI_CLEAN_PREP_OUTPUT_GRID

      use MODI_SIGNED_DISTANCE_FIELD

      use NN_EXTRAPOLATE
      implicit none
        class( tSimple_ice )               :: m
        character( LEN = * ), intent( in ) :: hPROGRAM
        character( LEN = * ), intent( in ) :: hATMFILE
        character( LEN = * ), intent( in ) :: hATMFILETYPE
        character( LEN = * ), intent( in ) :: hPGDFILE
        character( LEN = * ), intent( in ) :: hPGDFILETYPE

        integer :: iNam, iLUOUT
        logical :: gFound

        real, allocatable   :: tmp(:)
        integer             :: num_vals, n

        integer             :: i, ibeg, iend

        real, allocatable   :: XICE_TUNIF( : ),           &
                               XSIC_DRIVEN_MIN_SIC,       &
                               XSIC_DRIVEN_MAX_SIC,       &
                               XSIC_DRIVEN_MIN_THICKNESS
        logical             :: LINIT_FROM_SST,            &
                               LFAKE_PREP,                &
                               LPREP_ONLY_NEW_ICE,        &
                               LEXTRAPOLATE_FROM_FORECAST,&
                               LSIC_DRIVEN_THICKNESS

        character(LEN = 255):: CFORECAST_GRIB

        real                :: z_threshold,               &
                               z_surf_temp

        character( LEN = 6 ):: YINMODEL ! model from which GRIB file originates
        type( DATE_TIME )   :: TZTIME_GRIB    ! current date and time

        real, pointer       :: Z_FC_SIC_RAW (:),          &
                               Z_FC_TICE_RAW(:),          &
                               Z_LONS_RAW   (:),          &
                               Z_LATS_RAW   (:),          &
                               Z_SIC_RAW    (:)

        real, pointer       :: Z_FC_SIC     (:,:),        &
                               Z_FC_TICE    (:,:),        &
                               Z_LONS       (:,:),        &
                               Z_LATS       (:,:),        &

                               Z_UN         (:,:),        &
                               Z_UNX        (:,:),        &
                               Z_UNY        (:,:),        &
                               Z_DX         (:,:),        &
                               Z_DY         (:,:)


        real, allocatable   :: Z_SDF              (:,:),  &
                               Z_INTERPOLATED_TICE(:,:),  &
                               Z_INTERPOLATED_SIC (:,:),  &
                               Z_INTERPOLATED_SDF (:,:),  &
                               Z_INTERPOLATED_LONS(:,:),  &
                               Z_INTERPOLATED_LATS(:,:)

        real                :: z_mean

        integer :: IL, JL


        real( KIND = JPRB ) :: ZHOOK_HANDLE

        namelist /NAM_PREP_SIMPLE_ICE/  &
            XICE_TUNIF,                 &
            LINIT_FROM_SST,             &
            LFAKE_PREP,                 &
            LPREP_ONLY_NEW_ICE,         &
            LEXTRAPOLATE_FROM_FORECAST, &
            CFORECAST_GRIB

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:PREP', 0, ZHOOK_HANDLE )
        n = 0

        nullify(Z_FC_SIC_RAW , &
                Z_FC_TICE_RAW, &
                Z_LONS_RAW   , &
                Z_LATS_RAW   , &
                Z_SIC_RAW    , &
                Z_FC_SIC     , &
                Z_FC_TICE    , &
                Z_LONS       , &
                Z_LATS       , &
                Z_UN         , &
                Z_UNX        , &
                Z_UNY        , &
                Z_DX         , &
                Z_DY           )

        !-----------------------------------------------------------------------
        ! Namelist standard values
        LINIT_FROM_SST             = .FALSE.
        LFAKE_PREP                 = .FALSE.
        LPREP_ONLY_NEW_ICE         = .FALSE.
        LEXTRAPOLATE_FROM_FORECAST = .FALSE.
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Read namelist with prep options for SIMPLE_ICE scheme
        call open_namelist( hPROGRAM, iNam  )
        CALL get_luout    ( hPROGRAM, iLUOUT)

        call posnam( iNam, 'NAM_PREP_SIMPLE_ICE', gFound, iLUOUT )
        if( gFound ) read( UNIT = iNam, NML = NAM_PREP_SIMPLE_ICE )
        call close_namelist( hPROGRAM, iNam )
        !-----------------------------------------------------------------------

        if( LHANDLE_SIC ) then
            if(.NOT. LSIC_FROM_FILE) then
                !During PREP step XSIC not allocated yet.
                !In ASSIM step XSIC already allocated.
                if(.NOT. associated(XSIC)) allocate( XSIC(size(XSST)) )
                where( XSST(:) > XTT - 1 )
                    XSIC = 0.
                elsewhere
                    XSIC = (XTT - 1.0 - XSST)/0.6
                end where
                where( XSIC > 1. ) XSIC = 1.
            end if
        end if

        !-----------------------------------------------------------------------
        ! Prepare ice and snow models without setting it's state
        if( .NOT. associated(m%T) ) then
            call m%construct( size( XSST ) )
            call m%get_model_fields( m%MF )
        end if
        !-----------------------------------------------------------------------
        allocate( XICE_TUNIF( m%num_layers ), tmp( m%num_layers + 2 ) )

        write(*,'(A)') repeat('-', len(hPROGRAM)     + &
                                   len(hATMFILE)     + &
                                   len(hATMFILETYPE) + &
                                   len(hPGDFILE)     + &
                                   len(hPGDFILETYPE) + 5 )
        write(*,'(5(A,"|"))')  hPROGRAM, hATMFILE, hATMFILETYPE, hPGDFILE, hPGDFILETYPE

        XICE_TUNIF = XUNDEF

        if( LFAKE_PREP ) then
            write(*,*) 'FAKE PREP: do nothing...'
            if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:PREP', 1, ZHOOK_HANDLE )
            return
        end if

        if( LINIT_FROM_SST .AND. LEXTRAPOLATE_FROM_FORECAST ) then
            write(*,*) 'Incompatible options: LINIT_FROM_SST and LEXTRAPOLATE_FROM_FORECAST...'
            write(*,*) 'Switching to LEXTRAPOLATE_FROM_FORECAST...'
            LEXTRAPOLATE_FROM_FORECAST = .TRUE.
            LINIT_FROM_SST             = .FALSE.
            write(*,*) 'Done'
        end if

        if( LEXTRAPOLATE_FROM_FORECAST.AND.(.NOT.LHANDLE_SIC.OR..NOT.LSIC_FROM_FILE) ) then
            call ABOR1_SFX( 'For use of LEXTRAPOLATE_FROM_FORECAST, option LSIC_FROM_FILE should be enabled' )
        end if

        !-----------------------------------------------------------------------
        ! Perfrom ice fileds preparations
        fill_by_uniform: if( .NOT. all( XICE_TUNIF == XUNDEF) ) then
            write( *, * ) 'Filling ice by the uniform prescribed profile...'
            num_vals = count( XICE_TUNIF /= XUNDEF )

            if( num_vals == 1 ) then
                !No ice profile, only mean temperature.
                m%T = XICE_TUNIF(1)
            else
                call lin_space( 1., real(num_vals), [( real(n), n = 1, m%num_layers )], m%z( 1, : ) )
                ! Linear interpolation from given profile to model grid.
                do i = 1, num_vals - 1
                    ibeg = count( m%z( 1, : ) < i     )
                    iend = count( m%z( 1, : ) < i + 1 )

                    if( iend == m%num_layers - 1 ) iend = m%num_layers

                    call lin_space( XICE_TUNIF(i), XICE_TUNIF(i+1), [real(i),m%z(1,ibeg+1:iend),i+1.], tmp(:2 + iend - ibeg) )
                    do n = 1, m%num_points
                        m%T(n, ibeg+1:iend) = tmp(2:1 + iend - ibeg)
                    end do
                end do
            end if
            write(*,*) 'Done'

            n = m%num_points
        else fill_by_uniform
            !Try other possiblities to prepare ice initial conditions.
            init_from_sst: if( LINIT_FROM_SST ) then
                write( *, * ) 'Clearing melted ice...'
                if( LHANDLE_SIC ) then

                    call m%rem( XSIC == 0. )
                    z_surf_temp = real( count( XSST < XTTSI ) )
                    if( z_surf_temp > 0. ) then
                        z_surf_temp = sum( XSST, MASK = XSST < XTTSI )/z_surf_temp
                        write(*,*) 'Mean ice surface temperature: ', z_surf_temp
                    else
                        z_surf_temp = XTTSI
                        write(*,*) 'Grid cells fully covered by ice not found.'
                    end if
                else
                    call m%rem(XSST > XTTS)
                end if
                write( *, * ) 'Filling ice by the SST data...'
                n = 0
                do i = 1, m%num_points
                    if( .NOT. LHANDLE_SIC ) then
                        z_threshold = XTTS
                        z_surf_temp = XSST(i)
                    end if
                    if ( LHANDLE_SIC ) then
                        if( XSIC(i) > 0. .AND. ( .NOT. LPREP_ONLY_NEW_ICE .OR. m%T(i,1) == XUNDEF ) ) then
                            call lin_space(                                     &
                                XSST(i),                                        &
                                XTTSI,                                          &
                                [m%z(i,:) - .5*m%dz(i,:), m%z(i,m%num_layers)], &
                                tmp(:m%num_layers + 1)                          )
                            m%T(i,:) = tmp(:m%num_layers)
                            n = n + 1
                        end if
                    else
                        if( XSST(i) < z_threshold .AND. ( .NOT. LPREP_ONLY_NEW_ICE .OR. m%T(i,1) == XUNDEF ) ) then
                            call lin_space(                                     &
                                z_surf_temp,                                    &
                                XTTSI,                                          &
                                [m%z(i,:) - .5*m%dz(i,:), m%z(i,m%num_layers)], &
                                tmp(:m%num_layers + 1)                          )
                            m%T(i,:) = tmp(:m%num_layers)
                            n = n + 1
                        end if
                    end if
                end do
            end if init_from_sst

            extrapolate_from_forecast: if( LEXTRAPOLATE_FROM_FORECAST .AND. .NOT. LMISSING_ICE_VARIABLES) then
                !Read ice temperature and SIC from previous forecast
                write(*,*) 'Extrapolating ice temperatures from old ice border to new ice...'
                call prep_grib_grid( CFORECAST_GRIB, ILUOUT, YINMODEL,CINGRID_TYPE,TZTIME_GRIB)
                write(*,*) 'Reading SIC from ' //trim( CFORECAST_GRIB )//'...'
                call read_from_grib( CFORECAST_GRIB, ILUOUT, YINMODEL, 720, Z_FC_SIC_RAW, Z_LATS_RAW, Z_LONS_RAW )
                write(*,*) 'Reading TICE from '//trim( CFORECAST_GRIB )//'...'
                call read_from_grib( CFORECAST_GRIB, ILUOUT, YINMODEL, 731, Z_FC_TICE_RAW )

                where( Z_FC_SIC_RAW > 1 .OR. Z_FC_SIC_RAW < 0. ) Z_FC_SIC_RAW = 0.

                print*,nx,ny, count(Z_FC_SIC_RAW > 0.)

                have_forecasted_ice: if( count(Z_FC_SIC_RAW > 0.) == 0 ) then

                    write(*,*) 'Previous forecast has no sea ice...'
                    write(*,*) 'Setting temperature of new ice to freezing point...'

                    n = 0
                    do i = 1, m%num_points
                        if( XSIC(i) > 0. .AND. ( .NOT. LPREP_ONLY_NEW_ICE .OR. m%T(i,1) == XUNDEF ) ) then

                            m%T(i,:) = XTTSI
                            n = n + 1
                        end if
                    end do
                else have_forecasted_ice
                    Z_FC_SIC ( 1:NX, 1:NY ) => Z_FC_SIC_RAW (:)
                    Z_FC_TICE( 1:NX, 1:NY ) => Z_FC_TICE_RAW(:)
                    Z_LATS   ( 1:NX, 1:NY ) => Z_LATS_RAW   (:)
                    Z_LONS   ( 1:NX, 1:NY ) => Z_LONS_RAW   (:)

                    Z_LONS = Z_LONS*XPI/180.
                    Z_LATS = Z_LATS*XPI/180.

                    allocate( Z_SDF( NX, NY ) )

                    call SIGNED_DISTANCE_FIELD( Z_FC_SIC > 0, Z_SDF )

                    do n = 1, 5
                        do JL = 2, ny - 1
                            do IL = 2, nx - 1
                                Z_SDF(IL,JL) =(Z_SDF(IL  ,JL  ) + &
                                               Z_SDF(IL+1,JL  ) + &
                                               Z_SDF(IL-1,JL  ) + &
                                               Z_SDF(IL  ,JL+1) + &
                                               Z_SDF(IL  ,JL-1) + &
                                               Z_SDF(IL+1,JL+1) + &
                                               Z_SDF(IL-1,JL-1) + &
                                               Z_SDF(IL+1,JL-1) + &
                                               Z_SDF(IL-1,JL+1) )/9.
                            end do
                        end do
                    end do

                    where( Z_FC_TICE > 300 .OR. Z_FC_TICE < 150 ) Z_FC_TICE = XTTSI
                    call nearest_neighbour_extrapolate( Z_FC_TICE_RAW, Z_LONS_RAW, Z_LATS_RAW, O_MASK = Z_FC_SIC_RAW > 0  )
                    !---------------------------------------------------------------
                    ! Interpolate to current model domain (or subdomain for parallel run)
                    allocate( Z_INTERPOLATED_TICE(m%num_points,1), &
                              Z_INTERPOLATED_SIC (m%num_points,1), &
                              Z_INTERPOLATED_SDF (m%num_points,1) )

                    call PREP_OUTPUT_GRID(ILUOUT,CGRID_SEA,XGRID_PAR_SEA,XLAT_SEA,XLON_SEA)

                    Z_FC_TICE( 1:NX*NY, 1:1 ) => Z_FC_TICE_RAW(:)
                    Z_FC_SIC ( 1:NX*NY, 1:1 ) => Z_FC_SIC_RAW (:)


                    call HOR_INTERPOL( ILUOUT, Z_FC_TICE, Z_INTERPOLATED_TICE )
                    call HOR_INTERPOL( ILUOUT, Z_FC_SIC,  Z_INTERPOLATED_SIC  )

                    call HOR_INTERPOL( ILUOUT, reshape( Z_SDF, [NX*NY,1] ),  Z_INTERPOLATED_SDF  )
                    where( .NOT. Z_INTERPOLATED_SIC(:,1) > 0 ) &
                        Z_INTERPOLATED_TICE(:,1) = Z_INTERPOLATED_TICE(:,1)*exp(.1*(XSIC - 1.)*abs(Z_INTERPOLATED_SDF(:,1))) + &
                                                   XTTSI*(1. -              exp(.1*(XSIC - 1.)*abs(Z_INTERPOLATED_SDF(:,1))))

                    call CLEAN_PREP_OUTPUT_GRID
                    !---------------------------------------------------------------

                    !---------------------------------------------------------------
                    ! Calculate temperature of new ice
                    n = 0
                    do i = 1, m%num_points
                        if( XSIC(i) > 0. .AND. ( .NOT. LPREP_ONLY_NEW_ICE .OR. m%T(i,1) == XUNDEF ) ) then
                            call lin_space(                                     &
                                Z_INTERPOLATED_TICE(i,1),                       &
                                XTTSI,                                          &
                                [m%z(i,:) - .5*m%dz(i,:), m%z(i,m%num_layers)], &
                                tmp(:m%num_layers + 1)                          )
                            m%T(i,:) = tmp(:m%num_layers)
                            n = n + 1
                        end if
                    end do
                    !---------------------------------------------------------------

                    deallocate( Z_SDF, Z_INTERPOLATED_TICE, &
                                       Z_INTERPOLATED_SIC,  &
                                       Z_INTERPOLATED_SDF   )

                end if have_forecasted_ice
            end if extrapolate_from_forecast

            write(*,*) 'Done for',n, 'point(s) of', m%num_points
        end if fill_by_uniform
        !-----------------------------------------------------------------------

        deallocate( XICE_TUNIF, tmp )

        if ( associated(m%snow) ) then
            allocate( m%snow%mf( size(m%mf) ) )
            m%snow%mf = m%mf

            call m%snow%PREP( hPROGRAM, hATMFILE, hATMFILETYPE, hPGDFILE, hPGDFILETYPE )
        end if

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:PREP', 1, ZHOOK_HANDLE )
      contains
        subroutine read_from_grib( HGRIB, KLUOUT, HINMODEL, NLEV, PFLD, PLATS, PLONS )
          use MODD_SURF_PAR,   only: XUNDEF
          use MODD_PREP,       only: CINGRID_TYPE
          use MODI_ABOR1_SFX

          use GRIB_API,        only: kindOfInt

          use MODE_READ_GRIB,  only: READ_GRIB

          implicit none
!
            character( LEN = * ), intent(in)  :: HGRIB     ! Grib file name
            integer,              intent(in)  :: KLUOUT    ! logical unit of output listing
            character( LEN = 6 ), intent(in)  :: HINMODEL  ! Grib originating model
            integer,              intent(in)  :: NLEV
            real, dimension(:),   pointer     :: PFLD
            real, dimension(:),   pointer, optional :: PLATS,     &
                                                       PLONS      !

            integer                           :: IL, JL, NUNDEF
            integer(KIND=kindOfInt)           :: IRET

            integer :: ILTYPE, ILEV1

            character( LEN = 4 ) :: CLEV

            real(KIND=jprb) :: ZHOOK_HANDLE

            !-------------------------------------------------------------------
            if (LHOOK) call DR_HOOK('SIMPLE_ICE:PREP:READ_FROM_GRIB',0,ZHOOK_HANDLE)
            write(CLEV,'(I3)') NLEV

            if (HINMODEL=='ALADIN') then
                ILTYPE = 105
                ILEV1  = NLEV
                if(present( PLATS ).AND.present( PLONS )) then
!                    call READ_GRIB(HGRIB,KLUOUT,11,IRET,PFLD, KLTYPE = ILTYPE, KLEV1=ILEV1, PLATS = PLATS, PLONS = PLONS)
                    call READ_GRIB(HGRIB,KLUOUT,130,IRET,PFLD, KLTYPE = ILTYPE, KLEV1=ILEV1, PLATS = PLATS, PLONS = PLONS)
                else
!                    call READ_GRIB(HGRIB,KLUOUT,11,IRET,PFLD, KLTYPE = ILTYPE, KLEV1=ILEV1                              )
                    call READ_GRIB(HGRIB,KLUOUT,130,IRET,PFLD, KLTYPE = ILTYPE, KLEV1=ILEV1                              )
                end if
                if (IRET /= 0) call ABOR1_SFX('SIMPLE_ICE:PREP:READ_FROM_GRIB: ' // CLEV //'FIELD MISSING ')
            else
              call ABOR1_SFX('SIMPLE_ICE:PREP:READ_FROM_GRIB:OPTION NOT SUPPORTED '//HINMODEL)
            end if
            !
            if (LHOOK) call DR_HOOK('SIMPLE_ICE:PREP:READ_FROM_GRIB',1,ZHOOK_HANDLE)
        end subroutine
    end subroutine

    !===========================================================================
    ! Allocate model variables, setup model grid.
    !===========================================================================
    subroutine construct_ice( this, num_points )
      use e3L_snow, only: tE3L_snow, &
                          lift
      use theoretic_snow !, only: tTheoretic_snow

      use MODD_SURF_PAR,  only: XUNDEF
      use MODN_SEAFLUX_n, only: nIce_layer
      use MODN_SIMPL_ICE, only: XICE_THICKNESS, &
                                LICE_HAS_SNOW,  &
                                CICE_SNOW
      use MODI_ABOR1_SFX
      use MODI_SET_GRID
      use MODI_LIN_SPACE
      use MODI_READ_SURF
      implicit none
        class( tSimple_ice )               :: this       !< Ice model
        integer, intent( in )              :: num_points !< Number of grid cells

        integer               :: n, i
        real                  :: z_skin

        real                  :: z_ice_thickness( num_points )

        type( tE3L_snow ), pointer :: s

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:CONSTRUCT', 0, ZHOOK_HANDLE )

        !call this%release()

        this%num_points = num_points
        this%num_layers = nIce_layer

        allocate( this%z     ( num_points, nIce_layer ),  &
                  this%dz    ( num_points, nIce_layer ),  &
                  this%z_diff( num_points, nIce_layer ),  &
                  this%T     ( num_points, nIce_layer )   )

        !-----------------------------------------------------------------------
        ! Snow scheme initialization
        nullify( this%snow )
        if( LICE_HAS_SNOW ) then
            select case( CICE_SNOW )
              case ('3-L') ! Coupling with ISBA explicit snow scheme
                allocate( tE3L_snow :: this%snow )
                allocate( this%heat_cond(this%num_points) )

                s => lift( this%snow ) ! Lift snow from generic to specific level

                s%soil_dz        => this%z(:,1)
                s%soil_heat_cond => this%heat_cond

                !call this%snow%construct( this%num_points )
              case ('S-D') ! Default snow
                !call this%bind_snow()
                allocate( tTheoretic_snow :: this%snow )
              case default
                call abor1_sfx( 'Not implemented: ' // CICE_SNOW )
            end select
            call this%snow%construct( num_points )
        end if
        !-----------------------------------------------------------------------

        if( num_points == 0 ) then
            this%is_empty = .TRUE.
            if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:CONSTRUCT', 1, ZHOOK_HANDLE )
            return
        end if

        this%T = XUNDEF

        !-----------------------------------------------------------------------
        !   Specification of ice layer depths
        if ( nIce_layer < 3 ) then
            write(*,*)      'NICE_LAYER < 3'
            call abor1_sfx( 'NICE_LAYER < 3' )
        end if

        call regrid( this )

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:CONSTRUCT', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Change ice thicknesses and reconstruct ice thermal profile.
    ! Ice thickness was changed by adjusting the lowest thick layer of the icepack.
    !===========================================================================
    subroutine regrid( ice )
      use MODN_SIMPL_ICE, only: &
        XICE_THICKNESS,         &
        LSIC_DRIVEN_THICKNESS,  &
        XSIC_DRIVEN_MIN_SIC,    &
        XSIC_DRIVEN_MAX_SIC,    &
        XSIC_DRIVEN_MAX_THICKNESS
      use MODD_SEAFLUX_n, only: &
        XSIC,                   &
        LHANDLE_SIC
      use MODI_LIN_SPACE
      use MODI_SET_GRID
      implicit none
        class( tSimple_ice ) :: ice           !< Ice model.

        real    :: z_new_thk( ice%num_points )
        integer :: ji, jn, i_num_ice_layers
        real    :: z_skin

        real, dimension( ice%num_points, ice%num_layers ) :: &
            z_new_z,        &
            z_new_dz,       &
            z_new_diff

        z_new_thk = XICE_THICKNESS

        if (.NOT. LHANDLE_SIC) then
            LSIC_DRIVEN_THICKNESS = .FALSE.
                write(*,*) 'LHANDLE_SIC should be set to .TRUE. for using LSIC_DRIVEN_THICKNESS'
                write(*,*) 'LSIC_DRIVEN_THICKNESS switched OFF'
        end if

        if ( LSIC_DRIVEN_THICKNESS ) then
            where    ( XSIC(:) > XSIC_DRIVEN_MAX_SIC )
                z_new_thk = XSIC_DRIVEN_MAX_THICKNESS
            elsewhere( XSIC(:) < XSIC_DRIVEN_MIN_SIC )
                z_new_thk = XICE_THICKNESS
            elsewhere
                z_new_thk = XICE_THICKNESS + (XSIC_DRIVEN_MAX_THICKNESS - XICE_THICKNESS)/ &
                    (XSIC_DRIVEN_MAX_SIC - XSIC_DRIVEN_MIN_SIC)*(XSIC - XSIC_DRIVEN_MIN_SIC)
            end where
        end if

        i_num_ice_layers = ice%num_layers
        do ji = 1, ice%num_Points
            if( z_new_thk(ji) > 0.2 ) then
                z_skin = 0.05
            else
                z_skin = z_new_thk(ji)*0.05/0.2
            end if
            z_skin = min( z_skin, (z_new_thk(ji) - z_skin)/( i_num_ice_layers - 1.0 ) )
            ! Linear distribution of N-1 first layer thicknesses
            call lin_space( z_skin, XICE_THICKNESS,                  &
                            [( real(jn), jn = 1, i_num_ice_layers )],&
                            z_new_z( ji, : )                         &
                          )
            ! Adjusting thick bottom layer
            z_new_z(ji, i_num_ice_layers) = z_new_thk(ji)
        end do

        call set_grid( z_skin, XICE_THICKNESS, z_new_z(:,:), z_new_dz(:,:), z_new_diff(:,:) )

        ice%z     (:,:) = z_new_z   (:,:)
        ice%dz    (:,:) = z_new_dz  (:,:)
        ice%z_diff(:,:) = z_new_diff(:,:)
    end subroutine

    !===========================================================================
    ! Perform model run.
    !===========================================================================
    subroutine run_ice( ice, forc, diag )
      use MODD_CSTS,     only: XTTSI,   &
                               XCPD,    &
                               XSTEFAN, &
                               XLSTT,   &
                               XOMEGA, XPI, XDAY

      use MODD_SURF_PAR, only: XUNDEF

      use MODD_SURF_ATM, only: LDRAG_COEF_ARP,  &
                               LRRGUST_ARP,     &
                               XRRSCALE,        &
                               XRRGAMMA,        &
                               XUTILGUST,       &
                               XRIMAX

      use MODD_SNOW_PAR, only: XZ0SN,   &
                               XZ0HSN

      use MODE_THERMOS,  only: qsati,    &
                               dqsati
      use MODE_ICE_ALBEDO

      use MODI_LIN_SPACE
      use MODI_SOIL_HEATDIF
      use MODI_SURFACE_RI
      use MODI_SURFACE_CDCH_1DARP
      use MODI_SURFACE_AERO_COND
      use MODI_SURFACE_CD
      use MODI_WIND_THRESHOLD

      use MODN_SIMPL_ICE

      use abstract_model, only: tForcing, tIce_diag
      implicit none
        class( tSimple_ice )                   :: ice
        type ( tForcing    ), intent( in     ) :: forc
        type ( tIce_diag   ), intent( in out ) :: diag

        real, dimension( size( forc%Ta ) ) :: &
            zDirCOSzw,  &
            z_Ri,        &
            z_Ch,        &
            z_Cd,        &
            z_Cdn,       &
            z_SFTh,      &
            z_SFTq,      &
            z_Ustar,     &
            z_RESa,      &

            z_Qsat,      &
            z_dQsat,     &

            z_Z0HICE,    &
            z_Z0ICE,     &

            z_Vmod,      &
            zAc,        &
            zRa,        &
            zUstar2,    &

            ZCT,        &
            ZTTSI,      &
            zIce_albedo,&
            zIce_emiss, &
            zterm2,     &
            zterm1,     &

            zBalance,   &
            zLinTerm,   &

            zSnow_flux, &
            zSnow_flux_p,&
            zSnow_flux_m,&

            ZRRCOR,     &
            ZFP,        &

            z_alpha, z_beta, z_gamma_T, z_gamma_q,  &
            z_pet_a_sfc, z_pet_b_sfc,                 &
            z_peq_a_sfc, z_peq_b_sfc,                 &

            z_tmp
        real, dimension( size( forc%Ta ), ice%num_layers ) :: &
            ZICECOND,   &
            ZICEHCAP,   &
            z_dq_dz,     &
            z_q_z
        real, dimension( ice%num_layers ) :: &
            z_ext, z_i0

        integer :: zInitMask( size( forc%Ta ) )
        type( tForcing ) :: initForc

        real    :: maxErr

        integer :: i, n, m, numInitPoints
        logical :: needInit, has_snow, has_snow_points( ice%num_points )

        real( KIND = JPRB ) :: ZHOOK_HANDLE


        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:RUN', 0, ZHOOK_HANDLE )

        has_snow = associated( ice%snow )
        !if( has_snow ) has_snow = ice%snow%exists()

        m = size( forc%Ta )
        numInitPoints = 0

        zDirCOSzw = 1.
        !
        z_RI   (:) = XUNDEF
        z_CH   (:) = XUNDEF
        z_CD   (:) = XUNDEF
        z_CDN  (:) = XUNDEF
        !
        z_SFTH (:) = XUNDEF
        z_SFTQ (:) = XUNDEF
        z_USTAR(:) = XUNDEF
        z_RESA (:) = XUNDEF

        !if( associated( ice%heat_cond ) ) ice%heat_cond = XICE_HEAT_CONDUCTIVITY

        !-----------------------------------------------------------------------
        ! Initialization of sea ice temperature in points with new ice
        if(any( ice%T( :m, 1 ) == XUNDEF )) call regrid( ice )
        do i = 1, m
            if( ice%T( i, 1 ) == XUNDEF ) then
                call lin_space( min(xttsi, forc%Ta(i)), xttsi, ice%z( i, : ), ice%T( i, : ) )

                numInitPoints = numInitPoints + 1
            end if
        end do
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Saturated specified humidity near the ice surface
        z_Qsat  = QSATI ( ice%T(:m,1), forc%Psurf        )
        z_dQsat = DQSATI( ice%T(:m,1), forc%Psurf, z_Qsat )
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Calculate the drag coefficient for momentum (z_CD)

        !   Richardson number
        CALL SURFACE_RI( ice%T(:m,1), z_Qsat, forc%EXNs, forc%EXNa, forc%Ta, forc%Qa, forc%Zref, forc%Uref, zDirCOSzw, forc%V, z_Ri)
        z_RI(:) = MIN(z_RI(:),XRIMAX)

        !   Z0 for  sea ice
        z_Z0HICE(:) = XZ0HSN
        z_Z0ICE (:) = XZ0SN

        !   Drag coefficient
        z_Vmod = WIND_THRESHOLD( forc%V, forc%Uref )
        IF ( LDRAG_COEF_ARP ) THEN
            CALL SURFACE_CDCH_1DARP(forc%ZREF, z_Z0ICE, z_Z0HICE , Z_VMOD, forc%Ta, ice%T(:m,1), &
                                    forc%QA, z_QSAT, z_CD, z_CDN, z_CH                 )
            ZRA(:) = 1. / ( z_CH(:) * Z_VMOD(:) )
        ELSE
            CALL SURFACE_CD( z_Ri, forc%Zref, forc%Uref, z_Z0ICE, z_Z0HICE , z_Cd, z_Cdn)

            !   Drag coefficient for heat and aerodynamical resistance
            CALL SURFACE_AERO_COND( z_RI, forc%ZREF, forc%UREF, Z_VMOD, z_Z0ICE, z_Z0HICE , ZAC, ZRA, z_CH )
        END IF

        ZUSTAR2 = z_CD*Z_VMOD**2
        z_RESA(:) = ZRA(:)

        IF ( LRRGUST_ARP ) THEN
            ZFP(:)=MAX(0.0,forc%pRate_r(:)+forc%pRate_s(:))
            ZRRCOR(:)=SQRT(1.0+((((ZFP(:)/(ZFP(:)+XRRSCALE))**XRRGAMMA)*XUTILGUST)**2) &
              /(z_CD(:)*Z_VMOD(:)**2))

            z_CD  = z_CD *ZRRCOR
            z_CH  = z_CH *ZRRCOR
            z_CDN = z_CDN*ZRRCOR
        ENDIF

        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Ice temperature evolution
        !Parametrization
        call thermal_properties( ice%T(:m,:), 3., zIceHcap(:,:), zIceCond(:,:) )

        !Predefined values
        !   Sea ice thermal conductivity [W/(K m)]
        if( abs(XICE_HEAT_CAPACITY     - XUNDEF) > 1.E-6 ) ZICEHCAP(:,:) = XICE_HEAT_CAPACITY
        !   Sea ice volumetric heat capacity [J/(m3 K)]
        if( abs(XICE_HEAT_CONDUCTIVITY - XUNDEF) > 1.E-6 ) ZICECOND(:,:) = XICE_HEAT_CONDUCTIVITY


        if( associated( ice%heat_cond ) ) ice%heat_cond(:m) = ZICECOND(:,1)


        !   Thermal inertia of sea ice
        if( abs(XICE_THERMAL_RESISTANCE - XUNDEF) < 1.E-6 ) then
            !zct = 1./sqrt( zIceHcap( :, 1 )*zIceCond( :, 1 )/xOmega*0.5 )
            zct = 1./(zIceHcap( :, 1 )*ice%z(:m,1))
        else
            zct = XICE_THERMAL_RESISTANCE
        end if

        !   Deep temperature for lower boundary condition
        ZTTSI(:) = XTTSI

        !   Parametrization of ice albedo -- one of hightsi's albedo
        !   parametrization pack.
        zIce_albedo = ice_albedo( ice%T(:m,1), NICE_ALBEDO )

        !   Ice emissivity of longwave radiation
        zIce_emiss  = .99

        do i = 1, m
            where( ice%z(i,:) <= 0.1 )
                z_ext = 17.0
                z_i0  = 1.0
            elsewhere
                z_ext = 1.5
                z_i0  = 0.18
            end where
            z_q_z  (i,:) = ( 1. - zIce_albedo(i) )*forc%SW(i)*z_i0(:)*exp(-z_ext(:)*ice%z(i,:))
        end do

        z_dq_dz(:m, 1) = ( 1. - zice_albedo(:) )*forc%sw(:) - z_q_z(:m, 1)
        do i = 2, ice%num_layers
            z_dq_dz(:m, i) = z_q_z(:m, i - 1) - z_q_z(:m, i)
        end do


        if( has_snow ) then

            has_snow_points = ice%snow%exists()

            call ice%snow%run( ice%T(:m,1), forc, diag    )
            call ice%snow%heat_flux_from_snow( zSnow_flux )
        else
            has_snow_points = .FALSE.
        end if
        !coefficients for implicit coupling
        z_tmp = (1. - forc%ppet_a_coef*forc%rhoa*z_ch*z_vmod)

        z_pet_a_sfc = -forc%ppet_a_coef*forc%rhoa*z_ch*z_vmod*(forc%EXNa/forc%EXNs)/z_tmp
        z_pet_b_sfc = forc%ppet_b_coef*forc%EXNa/z_tmp

        z_tmp = 1. - forc%ppeq_a_coef*forc%rhoa*z_ch*z_vmod

        z_peq_a_sfc = - forc%ppeq_a_coef*forc%rhoa*z_ch*z_vmod*z_dqsat/z_tmp
        z_peq_b_sfc = (forc%ppeq_b_coef - forc%ppeq_a_coef*forc%rhoa*z_ch*z_vmod*( z_qsat - z_dqsat*ice%T(:m,1) ) )/z_tmp

        where( has_snow_points(:m) )

            zterm2 = ice%z_diff( :m, 2 )/( zCt*zIceCond(:,1)*forc%tStep )

            zterm1 = ( ice%T(:m,1) + forc%tStep*zCt*zSnow_flux )/( 1. + 1./zterm2 )
            zterm2 = 1./( 1. + zterm2 )

        elsewhere
            !surface energy balance
            z_alpha =                                                           &
                        1./(forc%tStep*zct)                                     &
                      + 4.*zIce_emiss*xStefan*ice%T(:m,1)**3                    &
                      + forc%RHOa*xcpd*z_ch*z_vmod/forc%EXNs                    &
                      + xlstt*forc%RHOa*z_ch*z_vmod*z_dqsat                     &
                      + zIceCond(:,1)/ice%z_diff(:m,1)

            z_beta  =                                                           &
                        1./(forc%tStep*zct)*ice%T(:m,1)                         &
                      + ( 1. - zIce_albedo )*(forc%SW - z_q_z(:,1))             &
                      + forc%LW                                                 &
                      + 3.*zIce_emiss*xStefan*ice%T(:m,1)**4                    &
                      - xlstt*forc%RHOa*z_ch*z_vmod*( z_qsat - z_dqsat*ice%T(:m,1) )

            z_gamma_T = forc%RHOa*xcpd*z_ch*z_vmod/forc%EXNa
            z_gamma_q = forc%RHOa*xlstt*z_ch*z_vmod

            z_tmp = z_alpha - z_gamma_T*z_pet_a_sfc - z_gamma_q*z_peq_a_sfc

            zterm1 = (z_beta + z_gamma_T*z_pet_b_sfc + z_gamma_q*z_peq_b_sfc)/z_tmp
            zterm2 = zIceCond(:,1)/ice%z_diff(:m,1)/z_tmp

        end where

        if( has_snow ) then
            CALL SOIL_HEATDIF(forc%TSTEP,ice%Z(:m,:),ice%DZ(:m,:),ice%Z_DIFF(:m,:),ZICECOND,      &
                                 ZICEHCAP,ZCT,ZTERM1,ZTERM2,ZTTSI,ice%T(:m,:) )
        else
            call SOIL_HEATDIF(forc%TSTEP,ice%Z(:m,:),ice%DZ(:m,:),ice%Z_DIFF(:m,:),ZICECOND,      &
                                 ZICEHCAP,ZCT,ZTERM1,ZTERM2,ZTTSI,ice%T(:m,:), z_dq_dz )
        end if

        where(ice%T(:m,:) > XTTSI)
            ice%T(:m,:) = XTTSI
        end where
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! The fluxes
        z_SFTH (:) = XCPD * forc%RHOa(:) * z_CH(:) * Z_VMOD(:) * ( (1. -       z_pet_a_sfc)*ice%T(:m,1) - z_pet_b_sfc )
        z_SFTQ (:) =        forc%RHOa(:) * z_CH(:) * Z_VMOD(:) * ( z_QSAT(:) - z_peq_a_sfc *ice%T(:m,1) - z_peq_b_sfc )
        z_USTAR(:) = SQRT(ZUSTAR2(:))
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Diagnostics
        where( .NOT. has_snow_points(:m) )
            diag%PZ0ICE (:m) = z_Z0ice
            diag%PZ0HICE(:m) = z_Z0Hice
            diag%PSFTH  (:m) = z_SFTH
            diag%PSFTQ  (:m) = z_SFTQ
            diag%PUSTAR (:m) = z_USTAR

            diag%PQSAT  (:m) = z_QSAT
            diag%PCD    (:m) = z_CD
            diag%PCDN   (:m) = z_CDN
            diag%PCH    (:m) = z_CH
            diag%PRI    (:m) = z_RI
            diag%PRESA  (:m) = z_RESA
        end where
        !-----------------------------------------------------------------------

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:RUN', 1, ZHOOK_HANDLE )

      contains
        elemental subroutine thermal_properties( p_t, p_s, p_c, p_k )
          use MODD_CSTS, only: XTT
          implicit none
            real, intent( in  ) :: p_t, p_s ! Ice temperature [K] and salinity [ppt]
            real, intent( out ) :: p_c, p_k ! Ice heat capacity [J/(m3 K)] and heat thermal conductivity [W/(K m)]

            real, parameter ::      &
                pp_ci = 1.883E6,    & ! Volumetric heat capacity of pure ice
                pp_l  = 3.014E8,    & ! Volumetric heat of fusion of pure ice

                pp_ka = 0.03,       & ! Air heat conductivity
                pp_va = 0.025         ! Fractional value of air in sea ice

            real ::      &
                z_theta, &  ! p_t in degC
                z_tfs,   &  ! Freezing temperature of sea ice with bulk salinity p_s

                z_ki,    &  ! Pure ice heat condictivity
                z_kb,    &  ! Brine heat conductivity
                z_kbi       ! Heat conductivity of bubbly ice


            z_theta = p_t - XTT
            z_tfs   = -5.33E-7*p_s**3 - 9.37E-6*p_s**2 - 0.0592*p_s + 273.15


            p_c     = pp_ci - (z_tfs - XTT)/z_theta**2*pp_l


            z_kb    = 0.4184*( 1.25 + 0.030  *z_theta + 0.00014*z_theta**2 )
            z_ki    = 1.16  *( 1.91 - 8.66E-3*z_theta + 2.97E-5*z_theta**2 )

            z_kbi   = ( 2.0*z_ki + pp_ka - 2.0*pp_va*(z_ki-pp_ka) )/   &
                      ( 2.0*z_ki + pp_ka + 2.0*pp_va*(z_ki-pp_ka) )*z_ki

            p_k     = z_kbi - (z_kbi - z_kb)*(z_tfs - XTT)/z_theta

            if (p_k < 1.73) p_k = 1.73 !hightsi-like guard

        end subroutine thermal_properties
    end subroutine

    !===========================================================================
    ! Extract ice surface temperature
    !===========================================================================
    subroutine surf_ice( ice, temp )
      implicit none
        class( tSimple_ice ) :: ice
        real, intent( out )  :: temp( : )

        logical              :: has_snow, snow_points( ice%num_points )
        real                 :: snow_temp( ice%num_points )

        real( KIND = JPRB )  :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:SURF_T', 0, ZHOOK_HANDLE )

        has_snow = associated( ice%snow )
        if( has_snow ) then
            snow_points = ice%snow%exists()
            call ice%snow%surf_temperature( snow_temp )
        else
            snow_points = .FALSE.
        end if

        where( snow_points )
            temp = snow_temp
        elsewhere
            temp = ice%T( :, 1 )
        end where

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:SURF_T', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Deallocate model variables.
    !===========================================================================
    subroutine release( m )
      implicit none
        class( tSimple_ice ) :: m

        if(associated( m%z       )) deallocate( m%z       )
        if(associated( m%dz      )) deallocate( m%dz      )
        if(associated( m%z_diff  )) deallocate( m%z_diff  )
        if(associated( m%T       )) deallocate( m%T       )

        if(associated( m%snow    )) deallocate( m%snow    )

        nullify    ( m%heat_cond )
    end subroutine

    !===========================================================================
    ! Extarct surface albedo. If snow layer exists upon the ice return snow
    ! labedo, otherwise -- return ice albedo.
    !===========================================================================
    subroutine surf_alb( ice, alb )
      use MODE_ICE_ALBEDO
      use MODD_SURF_PAR,  only: XUNDEF
      use MODN_SIMPL_ICE, only: NICE_ALBEDO
      implicit none
        class( tSimple_ice ) :: ice
        real, intent( out )  :: alb( : )

        logical              :: has_snow, snow_points( ice%num_points )
        real                 :: snow_alb( ice%num_points )

        real( KIND = JPRB )  :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:SURF_ALB', 0, ZHOOK_HANDLE )

        has_snow = associated( ice%snow )
        if( has_snow ) then
            snow_points = ice%snow%exists()
            call ice%snow%surf_albedo( snow_alb )
        else
            snow_points = .FALSE.
        end if

        alb = XUNDEF
        where( snow_points )
            alb = snow_alb
        elsewhere
            alb = ice_albedo( ice%T(:,1), NICE_ALBEDO )
            where( ice%T(:,1) > 300 ) alb = 0.
        end where


        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:SURF_ALB', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Attach model fields to self-describing records for performing operations,
    ! affecting all fileds at once (such as IO or masking)
    !===========================================================================
    subroutine get_model_fields( m, mf )
        use abstract_model, only: model_field
        implicit none
        class( tSimple_ice ),             intent( in  ) :: m
        type( model_field ), allocatable, intent( out ) :: mf(:)

        type( model_field ), allocatable :: snow_fields(:)
        real( KIND = JPRB )  :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:GET_MODEL_FIELDS', 0, ZHOOK_HANDLE )

        if(associated( m%snow )) then
            call m%snow%get_model_fields( snow_fields )
        else
            allocate( snow_fields(0) )
        end if
        allocate( mf(4 + size(snow_fields)) )

        mf(:) = [model_field(                      &
                   'TICE',                         &
                   'Ice temperature',              &
                   'K',                            &
                   [m%num_points, m%num_layers, 0],&
                   P2 = m%T                        &
                            ),                     &
                   model_field( N_CONFIG=[0,0,0], P2 = m%z,      L_INTERNAL = .TRUE. ), &
                   model_field( N_CONFIG=[0,0,0], P2 = m%dz,     L_INTERNAL = .TRUE. ), &
                   model_field( N_CONFIG=[0,0,0], P2 = m%z_diff, L_INTERNAL = .TRUE. ), &
                   snow_fields]

        deallocate( snow_fields )
        if( LHOOK ) call DR_HOOK( 'SIMPLE_ICE:GET_MODEL_FIELDS', 1, ZHOOK_HANDLE )
    end subroutine
end module
