module E3L_snow
  use abstract_snow,  only: tAbstract_snow

  use YOMHOOK,        only: LHOOK, DR_HOOK
  use PARKIND1,       only: JPRB
  implicit none
  private

      integer, parameter :: num_layers = 3 !< Number of layers in ISBA 3-L snow scheme fixed

      type, public, extends( tAbstract_snow ) :: tE3L_snow
            real, pointer, dimension( :    ) :: &
                albedo,     & !< Snow albedo

                thrufal,    & !< Rate that liquid water leaves snow pack:
                grnd_flux,  & !< Soil/snow interface heat flux
                evap_cor      !< Evaporation/sublimation correction term
            real, pointer, dimension( :, : ) :: &
                heat,       & !< Snow layers heat content
                rho,        & !< Snow layers averaged density
                swe,        & !< Snow layers liquid Water Equivalent

                liq_water,  & !< Snow layes liquid water content
                T,          & !< Snow layers temperature
                dz            !< Snow layers thickness

             real, pointer, dimension( : ) :: &
                soil_heat_cond, &
                soil_dz
        contains
            procedure :: construct           => init_e3L
            procedure :: run                 => run_e3L
            !procedure :: rem                 => rem_e3L
            procedure :: surf_temperature    => surf_temperature_e3L
            procedure :: exists              => exists_e3L
            procedure :: heat_flux_from_snow => heat_flux_from_snow_e3L

            procedure :: prep                => prep_e3L

            procedure :: surf_albedo

            procedure, private, pass :: internal_run
            procedure :: get_model_fields
      end type

      public :: lift
  contains



    function lift( m ) result( r )
      implicit none
        class( tAbstract_snow ), pointer :: m
        type( tE3L_snow ),       pointer :: r

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:LIFT', 0, ZHOOK_HANDLE )

        select type( m )
          type is( tE3L_snow )
            r => m
        end select

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:LIFT', 1, ZHOOK_HANDLE )
    end function
    !===========================================================================
    ! Snow PREP taken from ISBA PREP procedure
    !===========================================================================
    subroutine prep_e3L( m, hprogram, hatmfile, hatmfiletype, hpgdfile, hpgdfiletype )
      use MODD_TYPE_SNOW
      use MODD_PREP_ISBA, only : &
        lsnow_ideal, &
        xwsnow, xtsnow, xrsnow, xasnow, &
        xsg1snow, xsg2snow, xhistsnow, xagesnow
      use MODD_SNOW_PAR, only : xz0sn, xemissn
      use MODD_SEAFLUX_n, only: xsst, ttime
      use MODD_PREP, only: linterp

      use MODI_READ_PREP_ISBA_SNOW
      use MODI_PREP_HOR_SNOW_FIELDS
      use MODI_INIT_SNOW_LW
      use MODN_SIMPL_ICE, only: LINIT_3L_SNOW_ISBA_UNIF
      implicit none
        class( tE3L_snow )                 :: m
        character( LEN = * ), intent( in ) :: hprogram
        character( LEN = * ), intent( in ) :: hatmfile
        character( LEN = * ), intent( in ) :: hatmfiletype
        character( LEN = * ), intent( in ) :: hpgdfile
        character( LEN = * ), intent( in ) :: hpgdfiletype

        character(LEN=6) :: y_filetype ! type of input file
        character(LEN=28) :: y_file     ! name of file
        character(LEN=3) :: dummy_snow
        integer :: dummy_layers
        type(surf_snow) :: tsnow
        real, dimension(:), pointer :: &
            z_wsnow, z_rsnow, z_tsnow, z_sg1snow, &
            z_sg2snow, z_histsnow, z_agesnow
        real :: z_asnow
        integer :: iluout, npatch, num_points
        logical :: gunif

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:PREP', 0, ZHOOK_HANDLE )

        if(.NOT.LINIT_3L_SNOW_ISBA_UNIF) return

        npatch = 1
        num_points = size(xsst)

        tsnow%scheme = '3-L'
        tsnow%nlayer = 3

        y_file = ''
        y_filetype = ''

        nullify(z_wsnow, z_rsnow, z_tsnow, z_sg1snow, &
                z_sg2snow, z_histsnow, z_agesnow)
        allocate(linterp(num_points))

        call inject_snow_variables()

        call get_luout(hprogram, iluout)

        gunif = .TRUE.

        call read_prep_isba_snow(hprogram,dummy_snow,dummy_layers,y_file,y_filetype)
        if (len_trim(y_file)>0 .and. len_trim(y_filetype)>0) gunif = .FALSE.

        z_wsnow      => xwsnow
        z_rsnow      => xrsnow
        z_tsnow      => xtsnow
        z_sg1snow    => xsg1snow
        z_sg2snow    => xsg2snow
        z_histsnow   => xhistsnow
        z_agesnow    => xagesnow

        z_asnow      = xasnow

        print*, z_wsnow   ,z_rsnow   ,z_tsnow   ,z_sg1snow ,&
        z_sg2snow ,z_histsnow,z_agesnow

        if(.NOT.gunif) call abor1_sfx('Snow-on-ice INJECTION-PREP through external datasets is not supported')

        call prep_hor_snow_fields(hprogram, 'SN_VEG ',                 &
                                  y_file, y_filetype,                  &
                                  hpgdfile, hpgdfiletype,              &
                                  iluout, gunif, npatch,               &
                                  num_points, tsnow, ttime,            &
                                  z_wsnow, z_rsnow, z_tsnow, z_asnow,  &
                                  lsnow_ideal, z_sg1snow,              &
                                  z_sg2snow, z_histsnow, z_agesnow     )
        call init_snow_lw(xemissn,tsnow)

        deallocate(z_wsnow, z_rsnow, z_tsnow, z_sg1snow, &
                   z_sg2snow, z_histsnow, z_agesnow)

        call inject_snow_variables(O_RESTORE=.TRUE.)
        deallocate(linterp)

        if(.NOT. associated(m%T)) call m%construct(num_points)

        m%albedo   (:)   = tsnow%alb(:,1)
        m%thrufal  (:)   = 0.
        m%grnd_flux(:)   = 0.
        m%evap_cor (:)   = 0.
print*, tsnow%alb, z_asnow
print*, size(tsnow%heat),size(tsnow%rho),size(tsnow%wsnow),size(tsnow%temp)

        m%heat     (:,:) = tsnow%heat(:,:,1)
        m%rho      (:,:) = tsnow%rho(:,:,1)
        m%swe      (:,:) = tsnow%wsnow(:,:,1)
        m%liq_water(:,:) = 0.
        m%T        (:,:) = tsnow%temp(:,:,1)
        m%dz(:,:) = m%swe(:,:)/m%rho(:,:)

        deallocate(tsnow%wsnow, tsnow%heat, tsnow%T, tsnow%temp, tsnow%rho, &
            tsnow%alb, tsnow%emis, tsnow%ts, tsnow%gran1, tsnow%gran2, &
            tsnow%hist, tsnow%age)

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:PREP', 1, ZHOOK_HANDLE )
      contains
        subroutine inject_snow_variables(o_restore)
          use modd_prep_isba, only : &
            cfile_snow, ctype_snow, lsnow_ideal, &
            xwsnow, xtsnow, &
            xrsnow, xasnow, &
            xsg1snow, xsg2snow, &
            xhistsnow, xagesnow
          use modd_prep_snow, only : lsnow_frac_tot
          implicit none
            logical, optional, intent(in) :: o_restore

            real, dimension(:), pointer, save :: &
                zs_wsnow, zs_rsnow, zs_tsnow, zs_sg1snow, &
                zs_sg2snow, zs_histsnow, zs_agesnow
            real, save :: z_asnow
            character(LEN=28), save :: y_file_snow
            character(LEN=6), save :: y_type_snow
            logical, save :: g_snow_ideal, g_snow_frac_tot

            if(.NOT.present(o_restore)) then
                nullify(zs_wsnow, zs_rsnow, zs_tsnow, zs_sg1snow, &
                        zs_sg2snow, zs_histsnow, zs_agesnow)
                allocate(zs_wsnow   (size(xwsnow   )), &
                         zs_rsnow   (size(xrsnow   )), &
                         zs_tsnow   (size(xtsnow   )), &
                         zs_sg1snow (size(xsg1snow )), &
                         zs_sg2snow (size(xsg2snow )), &
                         zs_histsnow(size(xhistsnow)), &
                         zs_agesnow (size(xagesnow ))  )
                zs_wsnow    = xwsnow
                zs_rsnow    = xrsnow
                zs_tsnow    = xtsnow
                zs_sg1snow  = xsg1snow
                zs_sg2snow  = xsg2snow
                zs_histsnow = xhistsnow
                zs_agesnow  = xagesnow

                !call move_alloc( FROM = xwsnow   , TO = zs_wsnow   )
                !call move_alloc( FROM = xrsnow   , TO = zs_rsnow   )
                !call move_alloc( FROM = xtsnow   , TO = zs_tsnow   )
                !call move_alloc( FROM = xsg1snow , TO = zs_sg1snow )
                !call move_alloc( FROM = xsg2snow , TO = zs_sg2snow )
                !call move_alloc( FROM = xhistsnow, TO = zs_histsnow)
                !call move_alloc( FROM = xagesnow , TO = zs_agesnow )

                z_asnow    =  xasnow

                y_file_snow = cfile_snow
                y_type_snow = ctype_snow
                g_snow_ideal= lsnow_ideal

                g_snow_frac_tot = lsnow_frac_tot
            else
                allocate(xwsnow   (size(zs_wsnow   )), &
                         xrsnow   (size(zs_rsnow   )), &
                         xtsnow   (size(zs_tsnow   )), &
                         xsg1snow (size(zs_sg1snow )), &
                         xsg2snow (size(zs_sg2snow )), &
                         xhistsnow(size(zs_histsnow)), &
                         xagesnow (size(zs_agesnow ))  )
                xwsnow    = zs_wsnow
                xrsnow    = zs_rsnow
                xtsnow    = zs_tsnow
                xsg1snow  = zs_sg1snow
                xsg2snow  = zs_sg2snow
                xhistsnow = zs_histsnow
                xagesnow  = zs_agesnow

                deallocate(zs_wsnow,zs_rsnow,zs_tsnow,zs_sg1snow, &
                           zs_sg2snow,zs_histsnow,zs_agesnow)

                !call move_alloc( TO = xwsnow   , FROM = zs_wsnow   )
                !call move_alloc( TO = xrsnow   , FROM = zs_rsnow   )
                !call move_alloc( TO = xtsnow   , FROM = zs_tsnow   )
                !call move_alloc( TO = xsg1snow , FROM = zs_sg1snow )
                !call move_alloc( TO = xsg2snow , FROM = zs_sg2snow )
                !call move_alloc( TO = xhistsnow, FROM = zs_histsnow)
                !call move_alloc( TO = xagesnow , FROM = zs_agesnow )

                xasnow    =  z_asnow

                cfile_snow = y_file_snow
                ctype_snow = y_type_snow
                lsnow_ideal= g_snow_ideal

                lsnow_frac_tot = g_snow_frac_tot
            end if
        end subroutine
    end subroutine
    !===========================================================================
    ! Allocate snow variables and set them to no-snow state.
    !===========================================================================
    subroutine init_e3L( this, num_points )
      use MODD_SURF_PAR,  only: XUNDEF
      use MODD_SNOW_PAR,  only: XRHOSMIN_ES
      implicit none
        class( tE3L_snow )     :: this
        integer, intent( in )  :: num_points

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:INIT', 0, ZHOOK_HANDLE )

        this%num_points = num_points
        this%locked = .FALSE.

        allocate(   &
            this%albedo     ( num_points ),&
            this%thrufal    ( num_points ),&
            this%grnd_flux  ( num_points ),&
            this%evap_cor   ( num_points ),&
            this%heat       ( num_points, num_layers ),&
            this%rho        ( num_points, num_layers ),&
            this%swe        ( num_points, num_layers ),&
            this%liq_water  ( num_points, num_layers ),&
            this%T          ( num_points, num_layers ),&
            this%dz         ( num_points, num_layers ) &
            )

        this%albedo    = XUNDEF
        this%heat      = 0.0
        this%rho       = XRHOSMIN_ES
        this%swe       = 0.0

        this%liq_water = 0.

        this%grnd_flux = 0.
        this%thrufal   = 0.
        this%evap_cor  = 0.

        this%T         = XUNDEF
        this%dz        = 0.


        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:INIT', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    !> Run, wrapper for calling explicit snow scheme.
    !===========================================================================
    subroutine run_e3L( snow, T_ice, forc, diag )
      use abstract_model,      only: tForcing, tIce_diag

      use MODD_CSTS,           only: XTT, XLMTT
      use MODD_SNOW_PAR,       only: XRHOSMAX_ES, XSNOWDMIN, XRHOSMIN_ES
      use MODD_SURF_PAR,       only: XUNDEF

      use MODI_SNOW3L
      implicit none
        class( tE3L_snow )                       :: snow!< model
        real,                   intent( in     ) :: T_ice( : )
        type ( tForcing      ), intent( in     ) :: forc
        type ( tIce_diag     ), intent( in out ) :: diag

        real, dimension( size(forc%Ta) ) :: &

            z_snow,        & ! Total snow depth
            z_snow_swe,    & ! Totoal SWE of snowpack
            z_snowfall,    & ! Maximum possible snowfall depth

            z_ablat_delta, &
            z_meltflux
        integer :: iPoint
        real    :: abl_factor

        integer ::                      &
            num_snow_points,            &
            snow_mask( size(forc%Ta) )
        integer :: m

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:RUN', 0, ZHOOK_HANDLE )

        m = size(forc%Ta)

        !-----------------------------------------------------------------------
        ! Calculate points with snow

        z_snow    (:) = 0.
        z_snow_SWE(:) = 0.

        z_snowfall    = forc%Prate_s*forc%tStep/XRHOSMAX_ES ! maximum possible snowfall depth

        ! Calculate preliminary snow depth (m)
        do iPoint = 1, m
            z_snow    ( iPoint ) = sum( snow%SWE(iPoint,:)/snow%RHO(iPoint,:) )
            z_snow_SWE( iPoint ) = sum( snow%SWE(iPoint,:)                    )
        end do

        ! Packing: Only call snow model when there is snow on the surface
        !          exceeding a minimum threshold OR if the equivalent
        !          snow depth falling during the current time step exceeds
        !          this limit.
        !
        ! counts the number of points where the computations will be made

        num_snow_points = 0
        snow_mask(:)    = 0

        do iPoint = 1, m
            if ( z_snow(iPoint) >= XSNOWDMIN .OR. z_snowfall(iPoint) >= XSNOWDMIN ) then
                num_snow_points = num_snow_points + 1
                snow_mask( num_snow_points ) = iPoint
            end if
        end do

        if( num_snow_points > 0 .AND..NOT. snow%locked ) call snow%internal_run( T_ice, forc, diag, snow_mask, num_snow_points )

        !---------------------------------------------------------------------------
        ! Remove trace amounts of snow and reinitialize snow prognostic variables
        ! if snow cover is ablated:
        z_ablat_delta(:) = 0.
        z_snow       (:) = 0.
        do iPoint = 1, m
            z_snow    ( iPoint ) = sum( snow%SWE(iPoint,:)/snow%RHO(iPoint,:) )
        end do

        where( z_snow(:) < XSNOWDMIN*1.1 )
            !PTHRUFAL(:)         = ZSNOWSWE_1D(:)/PTSTEP + PSR(:) ! kg m-2 s-1   Conserve mass
            !PSMELTFLUX(:)       = -PTHRUFAL(:)*XLMTT             ! W m-2        Conserve Energy
            !PLEL3L(:)           = 0.0
            !PLES3L(:)           = 0.0
            !PEVAP(:)            = 0.0
            !ZRRSNOW(:)          = 0.0
            z_ablat_delta (:) = 1.0
            snow%albedo   (:) = XUNDEF

            ! additional heat flux form vanished thin snow
            snow%grnd_flux(:m) = snow%grnd_flux(:m) - (z_snow_SWE(:m)/forc%tStep + forc%Prate_s)*XLMTT
        end where

        do iPoint = 1, m
            abl_factor = 1.0 - Z_ablat_delta(iPoint)
            snow%SWE      (iPoint,:) = abl_factor * snow%SWE      (iPoint,:)
            snow%heat     (iPoint,:) = abl_factor * snow%heat     (iPoint,:)
            snow%rho      (iPoint,:) = abl_factor * snow%rho      (iPoint,:) + ( 1. - abl_factor ) * XRHOSMIN_ES
            snow%T        (iPoint,:) = abl_factor * snow%T        (iPoint,:) + ( 1. - abl_factor ) * XUNDEF !XTT
            snow%liq_water(iPoint,:) = abl_factor * snow%liq_water(iPoint,:) + ( 1. - abl_factor ) * 0. !XUNDEF
            snow%dz       (iPoint,:) = abl_factor * snow%dz       (iPoint,:) + ( 1. - abl_factor ) * 0. !XUNDEF
        end do
        !-----------------------------------------------------------------------
        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:RUN', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    !> Actual scheme call
    !===========================================================================
    subroutine internal_run( snow, T_ice, forc, diag, snow_mask, num_snow_points )
      use abstract_model,      only: tForcing,      &
                                     tIce_diag
      use MODD_CSTS,           only: XTT
      use MODD_SNOW_PAR,       only: XRHOSMAX_ES,   &
                                     XSNOWDMIN,     &
                                     XRHOSMIN_ES,   &
                                     XZ0SN,         &
                                     XZ0HSN
      use MODD_SURF_ATM,       only: XRIMAX
      use MODD_SURF_PAR,       only: XUNDEF
      use MODD_TYPE_DATE_SURF, only: DATE_TIME

      use MODE_THERMOS,        only: QSATI

      !use MODI_SNOW3L
      use MODI_SURFACE_AERO_COND
      use MODI_SURFACE_CD
      use MODI_SURFACE_RI
      use MODI_WIND_THRESHOLD

      use MODN_SIMPL_ICE,      only: XICE_THICKNESS,&
                                     XICE_DENSITY

      use SNOW_73
      implicit none
        class( tE3L_snow )                       :: snow
        real,                   intent( in     ) :: T_ice( : )
        type ( tForcing      ), intent( in     ) :: forc
        type ( tIce_diag     ), intent( in out ) :: diag
        integer,                intent( in     ) :: snow_mask( : )
        integer,                intent( in     ) :: num_snow_points

        type( DATE_TIME )                        :: tptime    ! Replacement for unused TPTIME dummy argument of SNOW3L() routine

        real, dimension( num_snow_points, num_layers ) :: &
            ZP_SNOWSWE,     & ! Masked snow water equavalent
            ZP_SNOWDZ,      & ! Masked snow layer thickness
            ZP_SNOWRHO,     & ! Masked snow density
            ZP_SNOWHEAT,    & ! Masked snow heat
            ZP_SNOWTEMP,    & ! Masked snow temperature
            ZP_SNOWLIQ,     & ! Masked snow liquid water
            ZP_SNOWGRAN1,   & ! UNUSED in 3L scheme, passed to snow routine for API compatibility between 3L and CROCUS
            ZP_SNOWGRAN2,   & ! UNUSED in 3L scheme, passed to snow routine for API compatibility between 3L and CROCUS
            ZP_SNOWHIST,    & ! UNUSED in 3L scheme, passed to snow routine for API compatibility between 3L and CROCUS
            ZP_SNOWAGE        ! UNUSED in 3L scheme, passed to snow routine for API compatibility between 3L and CROCUS
        real, dimension( num_snow_points )             ::  &
            ZP_SNOWALB,     &
            ZP_PS,          & ! Surface pressure
            ZP_SRSNOW,      &
            ZP_RRSNOW,      &
            ZP_PSN3L,       &
            ZP_TA,          & ! Air temperature
            ZP_TG,          & ! Grouind temperature (ice surface temperature)
            ZP_SW_RAD,      & ! Short wave radiation flux
            ZP_QA,          & ! Air humidity
            ZP_VMOD,        & ! Wind speed
            ZP_LW_RAD,      & ! Thermal radiation flux
            ZP_RHOA,        & ! Air density
            ZP_UREF,        & ! Reference height for the wind speed
            ZP_EXNS,        & ! Exner function at the surface
            ZP_EXNA,        & ! Exner function at the lowest model level
            ZP_DIRCOSZW,    &
            ZP_ZREF,        & ! Reference height for temperature and humidity
            ZP_Z0NAT,       &
            ZP_Z0HNAT,      &
            ZP_Z0EFF,       &
            ZP_ALB,         & ! Ground albedo
            ZP_SOILCOND,    & ! Ground thermal conductivity
            ZP_D_G,         & ! Thickness of the topmost ground layer
            ZP_THRUFAL,     &
            ZP_GRNDFLUX,    &
            ZP_EVAPCOR,     &
            ZP_RNSNOW,      &
            ZP_HSNOW,       &
            ZP_GFLUXSNOW,   &
            ZP_HPSNOW,      &
            ZP_LES3L,       &
            ZP_LEL3L,       &
            ZP_EVAP,        &
            ZP_EMISNOW,     &
            ZP_CDSNOW,      &
            ZP_USTARSNOW,   &
            ZP_CHSNOW,      &
            ZP_SNOWHMASS,   &
            ZP_VEGTYPE,     & ! Vegetation type, set to indicate no permanent snow
            ZP_PEW_A_COEF,  & ! Implicit coupling coefficient for wind
            ZP_PEW_B_COEF,  & ! Implicit coupling coefficient for wind
            ZP_PET_A_COEF,  & ! Implicit coupling coefficient for temperature
            ZP_PET_B_COEF,  & ! Implicit coupling coefficient for temperature
            ZP_PEQ_A_COEF,  & ! Implicit coupling coefficient for humidity
            ZP_PEQ_B_COEF,  & ! Implicit coupling coefficient for humidity
            ZP_ZENITH,      & ! UNUSED in 3L snow
            ZP_LAT,ZP_LON,  & ! UNUSED in 3L snow

            ZP_QSAT, ZP_RI, ZP_CD, ZP_CDN, ZP_RESA, ZP_CH, ZP_AC

        logical :: OGLACIER
        integer :: JWRK, JJ, JI

        integer,save :: i = 0

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:INTERNAL_RUN', 0, ZHOOK_HANDLE )

        OGLACIER = .FALSE.
        !-----------------------------------------------------------------------
        ! pack the variables
        do JJ = 1, num_snow_points
            JI = snow_mask(JJ)
            ZP_SNOWSWE (JJ,:) = snow%SWE           (JI,:)
            ZP_SNOWRHO (JJ,:) = snow%RHO           (JI,:)
            ZP_SNOWHEAT(JJ,:) = snow%HEAT          (JI,:)
            ZP_SNOWTEMP(JJ,:) = snow%T             (JI,:)
            ZP_SNOWLIQ (JJ,:) = snow%LIQ_WATER     (JI,:)
            ZP_SNOWDZ  (JJ,:) = snow%DZ            (JI,:)

            ZP_SNOWALB (JJ)   = snow%albedo        (JI)
            ZP_PS      (JJ)   = forc%Psurf         (JI)
            ZP_SRSNOW  (JJ)   = forc%Prate_s       (JI)
            ZP_RRSNOW  (JJ)   = forc%Prate_r       (JI)
            ZP_PSN3L   (JJ)   = 1.
            ZP_TA      (JJ)   = forc%TA            (JI)
            ZP_TG      (JJ)   = T_ice              (JI)
            ZP_SW_RAD  (JJ)   = forc%SW            (JI)
            ZP_QA      (JJ)   = forc%QA            (JI)
            ZP_VMOD    (JJ)   = forc%V             (JI)
            ZP_LW_RAD  (JJ)   = forc%LW            (JI)
            ZP_RHOA    (JJ)   = forc%RHOA          (JI)
            ZP_UREF    (JJ)   = forc%UREF          (JI)
            ZP_EXNS    (JJ)   = forc%EXNS          (JI)
            ZP_EXNA    (JJ)   = forc%EXNA          (JI)
            ZP_DIRCOSZW(JJ)   = 1.
            ZP_ZREF    (JJ)   = forc%ZREF          (JI)
            ZP_Z0NAT   (JJ)   = XZ0SN
            ZP_Z0HNAT  (JJ)   = XZ0HSN
            ZP_Z0EFF   (JJ)   = XZ0SN
            ZP_ALB     (JJ)   = 0.5
            ZP_SOILCOND(JJ)   = snow%soil_heat_cond(JI)
            ZP_D_G     (JJ)   = snow%soil_dz       (JI)

            ZP_PEW_A_COEF(JJ) = forc%PPEW_A_COEF   (JI)
            ZP_PET_A_COEF(JJ) = forc%PPET_A_COEF   (JI)
            ZP_PEQ_A_COEF(JJ) = forc%PPEQ_A_COEF   (JI)

            ZP_PEW_B_COEF(JJ) = forc%PPEW_B_COEF   (JI)
            ZP_PET_B_COEF(JJ) = forc%PPET_B_COEF   (JI)
            ZP_PEQ_B_COEF(JJ) = forc%PPEQ_B_COEF   (JI)
        end do
        !-----------------------------------------------------------------------
        ! Replacement for unused arguments
        ZP_LAT       = XUNDEF
        ZP_LON       = XUNDEF
        ZP_ZENITH    = XUNDEF

        ZP_SNOWGRAN1 = XUNDEF
        ZP_SNOWGRAN2 = XUNDEF
        ZP_SNOWHIST  = XUNDEF
        ZP_SNOWAGE   = XUNDEF
        !-----------------------------------------------------------------------

        ZP_VEGTYPE = 0. ! No permanent snow

        where( sum( ZP_SNOWRHO(:,:)*ZP_SNOWDZ(:,:), DIM=2 ) >= XICE_THICKNESS*(1024.458 - XICE_DENSITY) )
        !    ZP_SRSNOW = 0.
        end where
        !print*, sum( ZP_SNOWDZ(:,:), DIM=2 )

        where (                                                        &
            ZP_SNOWALB == XUNDEF .OR.                                  &
            (ZP_SNOWALB < .75 .AND. sum(ZP_SNOWDZ(:,:), DIM=2) > 0.1 .AND. ZP_TA < 270.) &
        ) ZP_SNOWALB = .75

        !ZP_SNOWALB = .85
        !if( any( ZP_SNOWALB == XUNDEF ) ) print*,'----', ZP_SNOWDZ
        !-----------------------------------------------------------------------
        ! conversion of snow heat from J/m3 into J/m2
        where(ZP_SNOWSWE(:,:)>0.) &
            ZP_SNOWHEAT(:,:) = ZP_SNOWHEAT(:,:) / ZP_SNOWRHO (:,:) * ZP_SNOWSWE (:,:)

        !-----------------------------------------------------------------------
        call SNOW3L('RIL', TPTIME, OGLACIER, 'OLD', ZP_PEW_A_COEF, ZP_PEW_B_COEF,   &
                 ZP_PET_A_COEF, ZP_PEQ_A_COEF,ZP_PET_B_COEF, ZP_PEQ_B_COEF,   &
                 ZP_SNOWSWE, ZP_SNOWRHO, ZP_SNOWHEAT, ZP_SNOWALB,             &
                 ZP_SNOWGRAN1, ZP_SNOWGRAN2, ZP_SNOWHIST, ZP_SNOWAGE, forc%TSTEP, &
                 ZP_PS, ZP_SRSNOW, ZP_RRSNOW, ZP_PSN3L, ZP_TA,ZP_TG,          &
                 ZP_SW_RAD, ZP_QA, ZP_VMOD, ZP_LW_RAD, ZP_RHOA, ZP_UREF,      &
                 ZP_EXNS, ZP_EXNA, ZP_DIRCOSZW, ZP_ZREF, ZP_Z0NAT, ZP_Z0EFF,  &
                 ZP_Z0HNAT, ZP_ALB, ZP_SOILCOND, ZP_D_G, ZP_SNOWLIQ,          &
                 ZP_SNOWTEMP, ZP_SNOWDZ, ZP_THRUFAL, ZP_GRNDFLUX ,ZP_EVAPCOR, &
                 ZP_RNSNOW, ZP_HSNOW, ZP_GFLUXSNOW, ZP_HPSNOW, ZP_LES3L,      &
                 ZP_LEL3L, ZP_EVAP, ZP_RI, ZP_EMISNOW, ZP_CDSNOW, ZP_USTARSNOW,      &
                 ZP_CHSNOW, ZP_SNOWHMASS, ZP_VEGTYPE, ZP_ZENITH, ZP_LAT, ZP_LON)

        !-----------------------------------------------------------------------
        ! conversion of snow heat from J/m2 into J/m3
        WHERE(ZP_SNOWSWE(:,:)>0.) &
            ZP_SNOWHEAT(:,:) = ZP_SNOWHEAT(:,:)* ZP_SNOWRHO (:,:)  / ZP_SNOWSWE (:,:)
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Saturated specified humidity near the ice surface
        ZP_QSAT  = QSATI ( ZP_SNOWTEMP(:,1), ZP_PS )
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! Calculate the drag coefficient for momentum (PCD)

        !   Richardson number
        !CALL SURFACE_RI( ZP_SNOWTEMP(:,1), ZP_QSAT, ZP_EXNS, ZP_EXNA, ZP_TA, ZP_QA, ZP_ZREF, ZP_UREF, ZP_DIRCOSZW, ZP_VMOD, ZP_RI)
        !ZP_RI(:) = MIN( ZP_RI(:), XRIMAX )

        !   Drag coefficient
        ZP_VMOD = WIND_THRESHOLD( ZP_VMOD, ZP_UREF )
        CALL SURFACE_CD( ZP_RI, ZP_ZREF, ZP_UREF, ZP_Z0NAT, ZP_Z0HNAT , ZP_CD, ZP_CDN)

        !   Drag coefficient for heat and aerodynamical resistance
        CALL SURFACE_AERO_COND( ZP_RI, ZP_ZREF, ZP_UREF, ZP_VMOD, ZP_Z0NAT, ZP_Z0HNAT , ZP_AC, ZP_RESA, ZP_CH )
        !-----------------------------------------------------------------------

        !-----------------------------------------------------------------------
        ! unpack variables

        if( .NOT. snow%locked ) then
            do JJ = 1, num_snow_points
                JI = snow_mask(JJ)
                SNOW%SWE      (JI,:) = ZP_SNOWSWE  (JJ,:)
                SNOW%RHO      (JI,:) = ZP_SNOWRHO  (JJ,:)
                SNOW%HEAT     (JI,:) = ZP_SNOWHEAT (JJ,:)
                SNOW%T        (JI,:) = ZP_SNOWTEMP (JJ,:)
                SNOW%LIQ_WATER(JI,:) = ZP_SNOWLIQ  (JJ,:)
                SNOW%DZ       (JI,:) = ZP_SNOWDZ   (JJ,:)
            end do

            DO JJ = 1, num_snow_points
                JI = snow_mask(JJ)
                SNOW%ALBEDO    (JI)   = ZP_SNOWALB  (JJ)
                SNOW%THRUFAL   (JI)   = ZP_THRUFAL  (JJ)
                SNOW%GRND_FLUX (JI)   = ZP_GRNDFLUX (JJ)
                SNOW%EVAP_COR  (JI)   = ZP_EVAPCOR  (JJ)

                diag%PSFTH     (JI)   = ZP_HSNOW    (JJ) ! Sensible heat flux
                diag%PSFTQ     (JI)   = ZP_EVAP     (JJ) ! Water vapour flux

                diag%PUSTAR    (JI)   = ZP_USTARSNOW(JJ)
                diag%PCD       (JI)   = ZP_CDSNOW   (JJ)
                diag%PCH       (JI)   = ZP_CHSNOW   (JJ)

                diag%PCDN      (JI)   = ZP_CDN      (JJ)
                diag%PQSAT     (JI)   = ZP_QSAT     (JJ)
                diag%PRESA     (JI)   = ZP_RESA     (JJ)
                diag%PRI       (JI)   = ZP_RI       (JJ)

                !------
                !SNOW%GRND_FLUX (JI)   = ZP_GFLUXSNOW(JJ)
                !------
                !PRNSNOW   (JI)   = ZP_RNSNOW   (JJ) ! Net radiatiove flux from snow
                !PHPSNOW   (JI)   = ZP_HPSNOW   (JJ) ! Heat release from rainfall
                !PEMISNOW  (JI)   = ZP_EMISNOW  (JJ) ! Snow emissivity
                !PSNOWHMASS(JI)   = ZP_SNOWHMASS(JJ) ! Heat due mass redistribution
            end do

            !print*, ZP_GRNDFLUX
            !if(i>240) write( 20, * ) i, ZP_SNOWTEMP, 273, ZP_TG, ZP_SNOWDZ, ZP_GRNDFLUX(1)

            i = i + 1
            !if (i == 723) stop

        else

            SNOW%GRND_FLUX (snow_mask(:))  = ZP_GRNDFLUX(:)

        end if

        diag%PZ0ICE  = XZ0SN
        diag%PZ0HICE = XZ0HSN
        !-----------------------------------------------------------------------

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:INTERNAL_RUN', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    !
    !===========================================================================
    subroutine surf_temperature_e3L( snow, temp )
      implicit none
        class( tE3L_snow )      :: snow
        real, intent( out )     :: temp( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE
        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:SURF_TEMPERATURE', 0, ZHOOK_HANDLE )

        temp = snow%T( :, 1 )

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:SURF_TEMPERATURE', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    !
    !===========================================================================
    subroutine surf_albedo( snow, alb )
      implicit none
        class( tE3L_snow )      :: snow
        real, intent( out )     :: alb( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE
        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:ALB', 0, ZHOOK_HANDLE )

        alb = snow%albedo

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:ALB', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Checks snow presence
    !===========================================================================
    function exists_e3L( snow ) result( res )
      implicit none
        class( tE3L_snow )      :: snow
        logical                 :: res( snow%num_points )

        real( KIND = JPRB ) :: ZHOOK_HANDLE
        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:EXISTS', 0, ZHOOK_HANDLE )

        res = any( snow%SWE(:,:) > 0., 2 )

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:EXISTS', 1, ZHOOK_HANDLE )
    end function

    !===========================================================================
    ! Returns heat flux from the snow pack to underlaying ice sheet
    !===========================================================================
    subroutine heat_flux_from_snow_e3L( snow, flux )
      implicit none
        class( tE3L_snow )      :: snow
        real,     intent( out ) :: flux( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE
        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:HEAT_FLUX_FROM_SNOW', 0, ZHOOK_HANDLE )

        flux = snow%grnd_flux(:size(flux))

        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:HEAT_FLUX_FROM_SNOW', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Constructs descriptor array
    !===========================================================================
    subroutine get_model_fields( m, mf )
        use abstract_model, only: model_field
        use MODD_SNOW_PAR,  only: XRHOSMIN_ES
        implicit none
        class( tE3L_snow ),               intent( in  ) :: m
        type( model_field ), allocatable, intent( out ) :: mf(:)

        real( KIND = JPRB ) :: ZHOOK_HANDLE
        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:GET_MODEL_FIELDS', 0, ZHOOK_HANDLE )

        allocate( mf(10) )

        mf    = [model_field(                      &
                   'WSNOW_ICE',                    &
                   'Snow water equivalent',        &
                   'Kg/m2',                        &
                   [m%num_points, num_layers, 0],  &
                   P2 = m%swe,                     &
                   X_DEFAULT = 0.                  &
                            ),                     &
                 model_field(                      &
                   'RSNOW_ICE',                    &
                   'Snow density',                 &
                   'Kg/m3',                        &
                   [m%num_points, num_layers, 0],  &
                   P2 = m%rho,                     &
                   X_DEFAULT = XRHOSMIN_ES         &
                            ),                     &
                 model_field(                      &
                   'HSNOW_ICE',                    &
                   'Snow heat content',            &
                   'Kg/m3',                        &
                   [m%num_points, num_layers, 0],  &
                   P2 = m%heat,                    &
                   X_DEFAULT = 0.                  &
                            ),                     &
                 model_field(                      &
                   'ASNOW_ICE',                    &
                   'Snow albedo',                  &
                   'dimensionless',                &
                   [m%num_points, 0, 0],           &
                   P1 = m%albedo                   &
                            ),                     &
                 model_field(                      &
                   'TSNOW_ICE',                    &
                   'Snow temperature',             &
                   'K',                            &
                   [m%num_points, num_layers, 0],  &
                   .TRUE.,                         &
                   P2 = m%T                        &
                            ),                     &
                 model_field(                      &
                   'DSNOW_ICE',                    &
                   'Snow thickness',               &
                   'm',                            &
                   [m%num_points, num_layers, 0],  &
                   .TRUE.,                         &
                   P2 = m%dz,                      &
                   X_DEFAULT = 0.                  &
                            ),                     &
                 model_field(                      &
                   'LWSNOW_ICE',                   &
                   'Snow liquid water',            &
                   'm',                            &
                   [m%num_points, num_layers, 0],  &
                   .TRUE.,                         &
                   P2 = m%liq_water,               &
                   X_DEFAULT = 0.                  &
                            ),                     &

                 model_field( N_CONFIG=[0,0,0], P1 = m%thrufal,   L_INTERNAL = .TRUE., X_DEFAULT = 0. ), &
                 model_field( N_CONFIG=[0,0,0], P1 = m%grnd_flux, L_INTERNAL = .TRUE., X_DEFAULT = 0. ), &
                 model_field( N_CONFIG=[0,0,0], P1 = m%evap_cor,  L_INTERNAL = .TRUE., X_DEFAULT = 0. )  &

                ]
        if( LHOOK ) call DR_HOOK( 'E3L_SNOW:GET_MODEL_FIELDS', 1, ZHOOK_HANDLE )
    end subroutine
end module
