module default_ice
!Wrapper for default SURFEX iceflux scheme
!
! AUTHOR
! Y. Batrak
!
! MODIFICATIONS
! Original 04/2014
  use abstract_ice

  use YOMHOOK,    only: LHOOK, DR_HOOK
  use PARKIND1,   only: JPRB
  implicit none

    type, public, extends( tAbstract_ice ) :: tDefault_ice
        real, pointer     :: SST(:) !< Pointer to sea surface temperature from seaflux.
        real, allocatable :: T  (:) !< Masked SST for fluxes calculations.
      contains
        procedure :: construct        => default_construct
        procedure :: run              => default_run
        procedure :: surf_temperature => default_surf
        procedure :: surf_alb         => default_alb
        procedure :: mask             => default_mask

        !Nothing to prepare
        procedure :: prep             => dummy_prep
    end type
  contains
    subroutine default_construct(  this, num_points )
      use MODD_SEAFLUX_n, only: XSST, XTICE
      implicit none
        class( tDefault_ice )              :: this
        integer, intent( in )              :: num_points

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:INIT', 0, ZHOOK_HANDLE )

        if( num_points /= size( XSST ) ) stop "Something strange... Let's crash."

        if(num_points == 0) this%is_empty = .TRUE.

        this%SST => XSST
        this%num_points = num_points

        allocate( this%T( num_points ) )
        nullify ( this%snow )

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:INIT', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine default_run( ice, forc, diag )
      use abstract_model,   only: tForcing, tIce_diag
      use MODI_ICE_SEA_FLUX
      implicit none
        class( tDefault_ice )                   :: ice
        type ( tForcing     ), intent( in     ) :: forc
        type ( tIce_diag    ), intent( in out ) :: diag

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:RUN', 0, ZHOOK_HANDLE )

        call ICE_SEA_FLUX(                  &
            diag%PZ0ICE ( :             ),  &
            forc%Ta     ( :             ),  &
            forc%EXNa   ( :             ),  &
            forc%RHOa   ( :             ),  &
            ice%T       ( :size(forc%Ta)),  &
            forc%EXNs   ( :             ),  &
            forc%Qa     ( :             ),  &
            forc%Prate_r( :             ),  &
            forc%Prate_s( :             ),  &
            forc%V      ( :             ),  &
            forc%zRef   ( :             ),  &
            forc%uRef   ( :             ),  &
            forc%Psurf  ( :             ),  &
            diag%PQSAT  ( :             ),  &
            diag%PSFTH  ( :             ),  &
            diag%PSFTQ  ( :             ),  &
            diag%PUSTAR ( :             ),  &
            diag%PCD    ( :             ),  &
            diag%PCDN   ( :             ),  &
            diag%PCH    ( :             ),  &
            diag%PRI    ( :             ),  &
            diag%PRESA  ( :             ),  &
            diag%PZ0HICE( :             )   )

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:RUN', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine default_surf( ice, temp )
      implicit none
        class( tDefault_ice ) :: ice
        real, intent( out )   :: temp( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:SURF_T', 0, ZHOOK_HANDLE )

        temp = ice%SST

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:SURF_T', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine default_alb( ice, alb )
      use MODD_WATER_PAR, only: XALBWATICE
      implicit none
        class( tDefault_ice ) :: ice
        real, intent( out )   :: alb( : )

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:SURF_ALB', 0, ZHOOK_HANDLE )

        alb = XALBWATICE

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:SURF_ALB', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine default_mask( m, mask )
      implicit none
        class( tDefault_ice ) :: m
        integer, intent( in ) :: mask( : )

        integer :: i

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:MASK', 0, ZHOOK_HANDLE )

        do i = 1, size( mask )
            m%T( i ) = m%SST( mask(i) )
        end do

        if( LHOOK ) call DR_HOOK( 'DEFAULT_ICE:MASK', 1, ZHOOK_HANDLE )
    end subroutine

    !===========================================================================
    ! Dummy procedures for Default ice scheme.
    !===========================================================================
    subroutine dummy_prep( m, hPROGRAM, hATMFILE, hATMFILETYPE, hPGDFILE, hPGDFILETYPE )
      implicit none
        class( tDefault_ice )              :: m
        character( LEN = * ), intent( in ) :: hPROGRAM
        character( LEN = * ), intent( in ) :: hATMFILE
        character( LEN = * ), intent( in ) :: hATMFILETYPE
        character( LEN = * ), intent( in ) :: hPGDFILE
        character( LEN = * ), intent( in ) :: hPGDFILETYPE
    end subroutine
end module
