!     #######################################################################
!>  *COUPLING_SEA_ICE* - Driver of the ICE_FLUX scheme
!!
!!    PURPOSE
!!    -------
!!
!!    METHOD
!!    ------
!!
!!    REFERENCE
!!    ---------
!!
!!
!!    AUTHOR
!!    ------
!!     B. DECHARME
!!
!!    MODIFICATIONS
!!    -------------
!!     Original    02/2010
!!      Y. Batrak  06/2014 generic ice call
!!
!!---------------------------------------------------------------------
SUBROUTINE COUPLING_SEA_ICE(PTSTEP, KI, PSIC, PTA, PEXNA, PRHOA, PTICE, PEXNS,       &
                                PQA, PRAIN, PSNOW, PWIND, PZREF, PUREF,          &
                                PPS, PTWAT, PTT, PSFTH, PSFTQ, KICE_LAYER, PSW, PLW,    &
                                PQSAT, PZ0, PUSTAR, PCD, PCDN, PCH, PRI, PRESA, PZ0H,   &
                                PPEW_A_COEF ,PPEW_B_COEF ,PPET_A_COEF ,PPEQ_A_COEF ,PPET_B_COEF ,PPEQ_B_COEF )
USE MODD_SURF_PAR,      ONLY : XUNDEF

USE MODD_SEAFLUX_n,     ONLY:   &
    LHANDLE_SIC,                &
    CSEA_ICE,                   &
    XICE
!
!
USE YOMHOOK   ,ONLY : LHOOK,   DR_HOOK
USE PARKIND1  ,ONLY : JPRB
!
IMPLICIT NONE
!
!*      0.1    declarations of arguments
!
INTEGER,             INTENT(IN)  :: KI         !< number of points
INTEGER,             INTENT(IN)  :: KICE_LAYER !< number of ice layers
REAL, DIMENSION(KI), INTENT(IN)  :: PSIC       !< Sea ice concentration
!
REAL,                INTENT(IN)  :: PTSTEP    !< atmospheric time-step                 (s)
REAL, DIMENSION(KI), INTENT(IN)  :: PTA       !< air temperature forcing               (K)
REAL, DIMENSION(KI), INTENT(IN)  :: PEXNA     !< Exner function at atm. level
REAL, DIMENSION(KI), INTENT(IN)  :: PRHOA     !< air density                           (kg/m3)
REAL, DIMENSION(KI,KICE_LAYER), INTENT(INOUT)  :: PTICE  !< Ice Surface Temperature
REAL, DIMENSION(KI), INTENT(IN)  :: PEXNS     !< Exner function at sea surface
REAL, DIMENSION(KI), INTENT(IN)  :: PQA       !< air humidity forcing                  (kg/m3)
REAL, DIMENSION(KI), INTENT(IN)  :: PRAIN     !< liquid precipitation                  (kg/m2/s)
REAL, DIMENSION(KI), INTENT(IN)  :: PSNOW     !< snow precipitation                    (kg/m2/s)
REAL, DIMENSION(KI), INTENT(IN)  :: PWIND     !< module of wind at atm. wind level
REAL, DIMENSION(KI), INTENT(IN)  :: PZREF     !< atm. level for temp. and humidity
REAL, DIMENSION(KI), INTENT(IN)  :: PUREF     !< atm. level for wind
REAL, DIMENSION(KI), INTENT(IN)  :: PPS       !< pressure at atmospheric model surface (Pa)
REAL, DIMENSION(KI), INTENT(IN)  :: PTWAT     !< Water Surface Temperature
REAL,                INTENT(IN)  :: PTT       !< temperature of freezing point
REAL, DIMENSION(KI), INTENT(IN)  :: PSW       !< Downward showrtwave radiation
REAL, DIMENSION(KI), INTENT(IN)  :: PLW       !< Downward longwave radiation
!
REAL, DIMENSION(KI), INTENT(OUT) :: PSFTH,          & !< flux of heat                          (W/m2)
                                    PSFTQ,          & !< flux of water vapor                   (kg/m2/s)
                                    PUSTAR,         & !< friction velocity (m/s)
! diagnostics
                                    PQSAT,          & !< humidity at saturation
                                    PCD,            & !< heat drag coefficient
                                    PCDN,           & !< momentum drag coefficient
                                    PCH,            & !< neutral momentum drag coefficient
                                    PRI,            & !< Richardson number
                                    PRESA,          & !< aerodynamical resistance
                                    PZ0H,           & !< heat roughness length
                                    PZ0

REAL, DIMENSION(KI), INTENT(IN)  :: PPEW_A_COEF,    & ! implicit coefficients
                                    PPEW_B_COEF,    & ! needed if HCOUPLING='I'
                                    PPET_A_COEF,    &
                                    PPEQ_A_COEF,    &
                                    PPET_B_COEF,    &
                                    PPEQ_B_COEF
!
!*      0.2    declarations of local variables
!
INTEGER, DIMENSION(KI)           :: IMASK
INTEGER                          :: JJ, ISIZE
!
REAL(KIND=JPRB) :: ZHOOK_HANDLE
!
!-------------------------------------------------------------------------------------
! Preliminaries:
!-------------------------------------------------------------------------------------
!
IF (LHOOK) CALL DR_HOOK('COUPLING_SEA_ICE',0,ZHOOK_HANDLE)
PSFTH(:)=0.0
PSFTQ(:)=0.0
!
IMASK(:)=0
ISIZE   =0
IF( LHANDLE_SIC ) THEN
    DO JJ = 1, KI
       IF( PSIC(JJ)>0. ) THEN
         ISIZE=ISIZE+1
         IMASK(ISIZE)=JJ
       END IF
    END DO
ELSE
    DO JJ = 1, KI
       !TODO: Why PTT+10 ????
       IF(PTWAT(JJ)<PTT+10.)THEN
         ISIZE=ISIZE+1
         IMASK(ISIZE)=JJ
       END IF
    END DO
END IF

PSFTH  = XUNDEF
PSFTQ  = XUNDEF
PUSTAR = XUNDEF
PQSAT  = XUNDEF
PCD    = XUNDEF
PCDN   = XUNDEF
PCH    = XUNDEF
PRI    = XUNDEF
PRESA  = XUNDEF
PZ0H   = XUNDEF
PZ0    = XUNDEF
!
IF(ISIZE==0)THEN
  IF (LHOOK) CALL DR_HOOK('COUPLING_ICE_SEA_ICE',1,ZHOOK_HANDLE)
  RETURN
ENDIF
!
CALL TREAT_ICE(ISIZE,IMASK(:ISIZE),KICE_LAYER)
!
!=======================================================================================
!
IF (LHOOK) CALL DR_HOOK('COUPLING_SEA_ICE',1,ZHOOK_HANDLE)
CONTAINS
!
!=======================================================================================
SUBROUTINE TREAT_ICE(KSIZE,KMASK,KICE_LAYER)
USE ABSTRACT_MODEL, ONLY:   &
    TFORCING,               &
    TICE_DIAG
!USE MODD_SEAFLUX_n !, ONLY:   &
    !XICE
IMPLICIT NONE

INTEGER, INTENT(IN) :: KSIZE
INTEGER, INTENT(IN) :: KICE_LAYER
INTEGER, INTENT(IN), DIMENSION(:) :: KMASK

REAL                    :: ZTSTEP     ! atmospheric time-step                 (s)
REAL, DIMENSION(KSIZE), TARGET  ::  &
    ZTA,                            & ! air temperature forcing               (K)
    ZEXNA,                          & ! Exner function at atm. level
    ZRHOA,                          & ! air density                           (kg/m3)
    ZEXNS,                          & ! Exner function at sea surface
    ZQA,                            & ! air humidity forcing                  (kg/m3)
    ZRR,                            & ! liquid precipitation                  (kg/m2/s)
    ZRS,                            & ! snow precipitation                    (kg/m2/s)
    ZWIND,                          & ! module of wind at atm. wind level
    ZZREF,                          & ! atm. level for temp. and humidity
    ZUREF,                          & ! atm. level for wind
    ZPS,                            & ! pressure at atmospheric model surface (Pa)
    ZSFTH,                          & ! flux of heat                          (W/m2)
    ZSFTQ,                          & ! flux of water vapor                   (kg/m2/s)

    ZZ0,                            & ! roughness length over the sea ice
    ZQSAT,                          & ! humidity at saturation
    ZUSTAR,                         & ! friction velocity (m/s)
    ZCD,                            & ! Drag coefficient
    ZCDN,                           & ! Neutral Drag coefficient
    ZCH,                            & ! Heat transfer coefficient
    ZRI,                            & ! Richardson number
    ZRESA,                          & ! aerodynamical resistance
    ZZ0H,                           & ! heat roughness length over ice

    ZSW,                            &
    ZLW,                            &
    ZPEW_A_COEF,                    &
    ZPEW_B_COEF,                    &
    ZPET_A_COEF,                    &
    ZPET_B_COEF,                    &
    ZPEQ_A_COEF,                    &
    ZPEQ_B_COEF
TYPE( TFORCING  )       :: FORC
TYPE( TICE_DIAG )       :: DIAG
REAL(KIND=JPRB)         :: ZHOOK_HANDLE
!
IF (LHOOK) CALL DR_HOOK('COUPLING_SEA_ICE:TREAT_ICE',0,ZHOOK_HANDLE)
!
!-------------------------------------------------------------------------------------
!
ZTA        (:) = PTA        ( KMASK(:KSIZE) )
ZQA        (:) = PQA        ( KMASK(:KSIZE) )
ZRHOA      (:) = PRHOA      ( KMASK(:KSIZE) )
ZRR        (:) = PRAIN      ( KMASK(:KSIZE) )
ZRS        (:) = PSNOW      ( KMASK(:KSIZE) )
ZWIND      (:) = PWIND      ( KMASK(:KSIZE) )
ZPS        (:) = PPS        ( KMASK(:KSIZE) )
ZRR        (:) = PRAIN      ( KMASK(:KSIZE) )
ZRS        (:) = PSNOW      ( KMASK(:KSIZE) )
ZZREF      (:) = PZREF      ( KMASK(:KSIZE) )
ZUREF      (:) = PUREF      ( KMASK(:KSIZE) )
ZEXNA      (:) = PEXNA      ( KMASK(:KSIZE) )
ZEXNS      (:) = PEXNS      ( KMASK(:KSIZE) )
ZSW        (:) = PSW        ( KMASK(:KSIZE) )
ZLW        (:) = PLW        ( KMASK(:KSIZE) )
ZPEW_A_COEF(:) = PPEW_A_COEF( KMASK(:KSIZE) )
ZPEW_B_COEF(:) = PPEW_B_COEF( KMASK(:KSIZE) )
ZPET_A_COEF(:) = PPET_A_COEF( KMASK(:KSIZE) )
ZPET_B_COEF(:) = PPET_B_COEF( KMASK(:KSIZE) )
ZPEQ_A_COEF(:) = PPEQ_A_COEF( KMASK(:KSIZE) )
ZPEQ_B_COEF(:) = PPEQ_B_COEF( KMASK(:KSIZE) )

ZTSTEP    = PTSTEP
!-------------------------------------------------------------------------------
! Local variables
ZZ0   (:) = XUNDEF
ZQSAT (:) = XUNDEF
ZUSTAR(:) = XUNDEF
ZCD   (:) = XUNDEF
ZCDN  (:) = XUNDEF
ZCH   (:) = XUNDEF
ZRI   (:) = XUNDEF
ZRESA (:) = XUNDEF
ZZ0H  (:) = XUNDEF
ZSFTH (:) = XUNDEF
ZSFTQ (:) = XUNDEF
!-------------------------------------------------------------------------------

!-------------------------------------------------------------------------------
! Generic ICE preparation and run.
forc%tStep = zTstep

forc%zRef        => ZZREF
forc%uRef        => ZUREF

forc%Ta          => ZTA
forc%Qa          => ZQA
forc%RHOa        => ZRHOA
forc%Psurf       => ZPS
forc%V           => ZWIND

forc%SW          => ZSW
forc%LW          => ZLW

forc%Prate_r     => ZRR
forc%Prate_s     => ZRS

forc%EXNs        => ZEXNS
forc%EXNa        => ZEXNA

forc%ppew_a_coef => ZPEW_A_COEF
forc%ppew_b_coef => ZPEW_B_COEF
forc%ppet_a_coef => ZPET_A_COEF
forc%ppet_b_coef => ZPET_B_COEF
forc%ppeq_a_coef => ZPEQ_A_COEF
forc%ppeq_b_coef => ZPEQ_B_COEF

diag%PZ0ICE      => ZZ0
diag%PSFTH       => ZSFTH
diag%PSFTQ       => ZSFTQ
diag%PUSTAR      => ZUSTAR

diag%PQSAT       => ZQSAT
diag%PCD         => ZCD
diag%PCDN        => ZCDN
diag%PCH         => ZCH
diag%PRI         => ZRI
diag%PRESA       => ZRESA
diag%PZ0HICE     => ZZ0H

CALL XICE%MASK  ( KMASK      )
CALL XICE%RUN   ( FORC, DIAG )
CALL XICE%UNMASK( KMASK      )
!-------------------------------------------------------------------------------

!-------------------------------------------------------------------------------
! Outputs:

PSFTH   ( KMASK(:KSIZE) ) = ZSFTH (:)
PSFTQ   ( KMASK(:KSIZE) ) = ZSFTQ (:)

PUSTAR  ( KMASK(:KSIZE) ) = ZUSTAR(:)
PQSAT   ( KMASK(:KSIZE) ) = ZQSAT (:)
PCD     ( KMASK(:KSIZE) ) = ZCD   (:)
PCDN    ( KMASK(:KSIZE) ) = ZCDN  (:)
PCH     ( KMASK(:KSIZE) ) = ZCH   (:)
PRI     ( KMASK(:KSIZE) ) = ZRI   (:)
PRESA   ( KMASK(:KSIZE) ) = ZRESA (:)
PZ0H    ( KMASK(:KSIZE) ) = ZZ0H  (:)
PZ0     ( KMASK(:KSIZE) ) = ZZ0   (:)

CALL XICE%SURF_TEMPERATURE( PTICE(:,1) )
!-------------------------------------------------------------------------------
IF (LHOOK) CALL DR_HOOK('COUPLING_SEA_ICE:TREAT_ICE',1,ZHOOK_HANDLE)
END SUBROUTINE TREAT_ICE
END SUBROUTINE COUPLING_SEA_ICE
