module abstract_model
! Module ABSTRACT_MODEL contains definitions of types, used for performing model
! runs. Also, this module contains abstract type, which defines base interface to
! some surface prognostic or diagnostic scheme.
!
! AUTHOR
! Y. Batrak
!
! MODIFICATIONS
! Original  04/2014
  use MODD_SURF_PAR, only: XUNDEF
  use YOMHOOK,       only: LHOOK, DR_HOOK
  use PARKIND1,      only: JPRB
  implicit none
    private

    integer, parameter :: mf_grid = 1,  &
                          mf_layer= 2

    type, public :: model_field
        character( LEN = 64 ) :: c_name        = ''             !< Name of model field for IO operations.
        character( LEN = 64 ) :: c_comment     = ''             !< Comment for model field record.
        character( LEN = 16 ) :: c_units       = ''             !< Model field untis description.

        integer               :: n_config( 3 ) =  [0,0,0]       !< Model grid configuration.
        logical               :: l_diag        =  .FALSE.       !< Diagnostic fields may be written in separate files.
        logical               :: l_internal    =  .FALSE.       !< Internal model fields not used in IO operations.

        real, pointer         :: p1(:)         => null(),   &   !< Pointer to model grid.
                                 p2(:,:)       => null()        !< Pointer to multilayer model grid.
        real                  :: x_default     =  XUNDEF        !< Defaul dummy value for described model variable.
    end type

    type, public :: tForcing
        real :: tStep               !< Forcing time step
        real, pointer, dimension( : ) :: &
                zRef,           &   !< Reference height for temperature and humidity.
                uRef,           &   !< Reference height for wind speed.

                Ta,             &   !< Air temperature.
                Qa,             &   !< Air specific humidity.
                RHOa,           &   !< Air density.
                Psurf,          &   !< Surface pressure.
                V,              &   !< Wind speed.

                SW,             &   !< Shortwave radiation flux at the surface.
                LW,             &   !< Longwave radiation flux at the surface.

                Prate_r,        &   !< Rainfall rate.
                Prate_s,        &   !< Snowfall rate.

                EXNs,           &   !< Exner function at the surface.
                EXNa,           &   !< Exner function at the atmospheric forcing level.
                PPEW_A_COEF,    &   !< Implicit coupling A-coefficient for wind.
                PPEW_B_COEF,    &   !< Implicit coupling B-coefficient for wind.
                PPET_A_COEF,    &   !< Implicit coupling A-coefficient for temperature.
                PPEQ_A_COEF,    &   !< Implicit coupling A-coefficient for humidity.
                PPET_B_COEF,    &   !< Implicit coupling B-coefficient for temparature.
                PPEQ_B_COEF         !< Implicit coupling B-coefficient for humidity.
    end type

    type, public :: tIce_diag
        real, pointer, dimension( : ) :: &
            PZ0ICE,     &   !< Roughness length over the surface, [m]

            !  surface fluxes : latent heat, sensible heat, friction fluxes
            PSFTH,      &   !< Sensible heat flux  (W/m2)
            PSFTQ,      &   !< Water flux (kg/m2/s)
            PUSTAR,     &   !< Friction velocity (m/s)

            PQSAT,      &   !< Humidity at saturation
            PCD,        &   !< Momentum drag coefficient
            PCDN,       &   !< Neutral momentum drag coefficient
            PCH,        &   !< Heat drag coefficient
            PRI,        &   !< Richardson number
            PRESA,      &   !< Aerodynamical resistance
            PZ0HICE         !< Roughness length for heat fluxes over the surface
    end type

    type, public, abstract :: tAbstract_model
        integer :: num_points                       !< Total number of model cells.
        type( model_field ), allocatable :: mf(:)   !< Array of model fields descriptors.
        logical :: is_empty = .FALSE.
      contains
        procedure(iface_prep),      deferred, pass( m ) :: prep

        procedure(iface_construct), deferred, pass      :: construct        !< Prepare model for work, WITHOUT setting it's initial state.

        procedure,                            pass( m ) :: mask => amask    !< Configure model for working only with specified gridcells.
        procedure,                            pass( m ) :: unmask           !< Return from specific gridcells to whole grid.

        procedure,                            pass( m ) :: get_model_fields !< Return array of model field descriptors.

        procedure,                            pass      :: rem              !< Switch model fields to default values.

        procedure, nopass :: init_io !< Prepare NetCDF files for writing model output, used in OFFLINE mode.
        procedure, nopass :: io      !< Perform IO operation over model fields.
    end type

    abstract interface
        subroutine iface_construct( this, num_points )
          import :: tAbstract_model
            class( tAbstract_model ) :: this
            integer, intent( in )    :: num_points
        end subroutine

        subroutine iface_prep( m, hPROGRAM, hATMFILE, hATMFILETYPE, hPGDFILE, hPGDFILETYPE )
          import :: tAbstract_model
          implicit none
            class( tAbstract_model )           :: m             !< Model.
            character( LEN = * ), intent( in ) :: hPROGRAM      !< Program calling surf. schemes.
            character( LEN = * ), intent( in ) :: hATMFILE      !< Name of the Atmospheric file.
            character( LEN = * ), intent( in ) :: hATMFILETYPE  !< Type of the Atmospheric file.
            character( LEN = * ), intent( in ) :: hPGDFILE      !< Name of the Atmospheric file.
            character( LEN = * ), intent( in ) :: hPGDFILETYPE  !< Type of the Atmospheric file.
        end subroutine
    end interface

  contains
    subroutine init_io( mf, file_id, dimens, is_diag )
      use MODI_DEF_VAR_NETCDF
      implicit none
        type( model_field ),      intent( in ) :: mf(:)     !< Model fields.
        integer,                  intent( in ) :: file_id   !< Output NetCDF file descriptor.
        integer,                  intent( in ) :: dimens(:) !< Dimension vector.
        logical, optional,        intent( in ) :: is_diag

        character(LEN=100), dimension( 1 ) :: YATT_TITLE, YATT
        character(LEN=3)                   :: YPAS, YLVL

        integer :: i, j_layer, num_fields, num_layers
        logical :: l_diag

        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:INIT_IO', 0, ZHOOK_HANDLE )

        l_diag = .FALSE.
        if( present( is_diag ) ) l_diag = is_diag

        num_fields = size( mf )

        do i = 1, num_fields
            if( (l_diag .NEQV. mf(i)%l_diag) .OR. mf(i)%l_internal ) cycle
            yatt_title(1) = 'units'
            yatt      (1) = mf(i)%c_units

            if( mf(i)%n_config( mf_layer ) > 0 ) then
                num_layers = mf(i)%n_config( mf_layer )
                do j_layer = 1, num_layers
                    write( YPAS,'("_",I2.2)' ) j_layer ; YLVL = ADJUSTL(YPAS(:LEN_TRIM(YPAS)))
                    call DEF_VAR_NETCDF(            &
                        FILE_ID,                    &
                        trim(mf(i)%c_name)   //YLVL,&
                        trim(mf(i)%c_comment)//YLVL,&
                        DIMENS,                     &
                        YATT_TITLE,                 &
                        YATT )
                end do
            else
                call DEF_VAR_NETCDF(        &
                    FILE_ID,                &
                    trim(mf(i)%c_name),     &
                    trim(mf(i)%c_comment),  &
                    DIMENS,                 &
                    YATT_TITLE,             &
                    YATT )
            end if

            write( *, * ) 'Registered: ' // trim(mf(i)%c_name)
        end do
        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:INIT_IO', 1, ZHOOK_HANDLE )
    end subroutine

    ! Perform input/output operation over model fields
    subroutine io( mf, hProgram, is_diag, is_read )
      use MODI_WRITE_SURF
      use MODI_READ_SURF
      implicit none
        type( model_field ),  intent( in out ) :: mf(:)
        character( LEN = 6 ), intent( in ) :: hProgram
        logical, optional,    intent( in ) :: is_diag, is_read

        integer                :: JLAYER
        integer                :: IRESP          ! IRESP  : return-code if a problem appears
        character( LEN = 16  ) :: YRECFM         ! Name of the article to be read
        character( LEN = 4   ) :: YLVL
        character( LEN = 100 ) :: YCOMMENT       ! Comment string
        character( LEN = 25  ) :: YFORM          ! Writing format

        integer :: i, j_layer, num_fields, num_layers
        logical :: l_diag, l_read
        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:IO', 0, ZHOOK_HANDLE )


        l_diag = .FALSE.
        l_read = .FALSE.
        if( present( is_diag ) ) l_diag = is_diag
        if( present( is_read ) ) l_read = is_read

        num_fields = size(mf)

        do i = 1, num_fields
            if( (l_diag .NEQV. mf(i)%l_diag) .OR. mf(i)%l_internal ) cycle
            if( l_read .AND. mf(i)%l_diag ) cycle

            YRECFM = trim(mf(i)%c_name)
            YFORM  = '(A,I1.1,A4)'

            if( mf(i)%n_config( mf_layer ) > 0 ) then
                num_layers = mf(i)%n_config( mf_layer )
                do j_layer = 1, num_layers
                    write( YLVL, '("_",I2.2)' ) j_layer
                    YRECFM = trim(mf(i)%c_name) // adjustl(YLVL(:len_trim(YLVL)))
                    if (j_layer >= 10) &
                        YFORM = '(A,I2.2,A4)'
                    write(YCOMMENT,YFORM) 'X_Y_' // trim(mf(i)%c_name), j_layer, ' (K)'
                    if( l_read ) then
                        call READ_SURF ( HPROGRAM, YRECFM, mf(i)%P2(:,j_layer), IRESP )
                    else
                        call WRITE_SURF( HPROGRAM, YRECFM, mf(i)%P2(:,j_layer), IRESP, HCOMMENT = YCOMMENT )
                    end if
                    YCOMMENT = ''
                end do
            else
                write(YCOMMENT,YFORM) 'X_Y_' // trim(mf(i)%c_name)
                if( l_read ) then
                    call READ_SURF ( HPROGRAM, YRECFM, mf(i)%P1(:), IRESP )
                else
                    call WRITE_SURF( HPROGRAM, YRECFM, mf(i)%P1(:), IRESP, HCOMMENT = YCOMMENT )
                end if
            end if
        end do

        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:IO', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine get_model_fields( m, mf )
        implicit none
        class( tAbstract_model ), intent( in )          :: m
        type( model_field ), allocatable, intent( out ) :: mf(:)

        allocate( mf(0) )
    end subroutine

    subroutine amask( m, mask )
      implicit none
        class( tAbstract_model ) :: m         !< Model.
        integer, intent( in )    :: mask( : ) !< Mask for selecting working points.

        integer :: i,j
        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if(m%is_empty) return

        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:MASK', 0, ZHOOK_HANDLE )

        do i = 1, size( m%mf )
            if( size(mask) == 1 ) then
                if( mask(1) == 1 ) cycle
                if(associated(m%mf(i)%p1)) m%mf(i)%p1( [1,mask(1)]    ) = m%mf(i)%p1( [mask(1),1]    )
                if(associated(m%mf(i)%p2)) m%mf(i)%p2( [1,mask(1)], : ) = m%mf(i)%p2( [mask(1),1], : )
            else
                if(associated(m%mf(i)%p1)) m%mf(i)%p1( :size(mask)    ) = m%mf(i)%p1( mask    )
                if(associated(m%mf(i)%p2)) m%mf(i)%p2( :size(mask), : ) = m%mf(i)%p2( mask, : )
            end if
        end do
        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:MASK', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine unmask( m, mask )
      implicit none
        class( tAbstract_model ) :: m         !< Model.
        integer, intent( in )    :: mask( : ) !< Mask for selecting working points.

        integer :: i,j,k
        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if(m%is_empty) return

        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:UNMASK', 0, ZHOOK_HANDLE )

        do i = 1, size( m%mf )
            if( size(mask) == 1 ) then
                if( mask(1) == 1 ) cycle
                if(associated(m%mf(i)%p1)) m%mf(i)%p1( [1,mask(1)]    ) = m%mf(i)%p1( [mask(1),1]    )
                if(associated(m%mf(i)%p2)) m%mf(i)%p2( [1,mask(1)], : ) = m%mf(i)%p2( [mask(1),1], : )
            else
                if(associated(m%mf(i)%p1)) then
                    m%mf(i)%p1( mask    ) = m%mf(i)%p1( :size(mask)    )
                    where([( .NOT.any( mask == j ), j = 1, size( m%mf(i)%p1 ) )]) m%mf(i)%p1 = m%mf(i)%x_default
                end if
                if(associated(m%mf(i)%p2)) then
                    m%mf(i)%p2( mask, : ) = m%mf(i)%p2( :size(mask), : )
                    ! invert mask and clear unmasked values
                    do k = 1, size( m%mf(i)%p2, 2 )
                        where([( .NOT.any( mask == j ), j = 1, size( m%mf(i)%p2, 1 ) )]) m%mf(i)%p2(:,k) = m%mf(i)%x_default
                    end do
                    !where( reshape([((.NOT.any(mask == j),j=1,size(m%mf(i)%p2,1)),i=1,size(m%mf(i)%p2,2))],shape(m%mf(i)%p2)) ) m%mf(i)%p2 = m%mf(i)%x_default
                    !where( reshape(spread([(.NOT.any(mask == j),j=1,size(m%mf(i)%p2,1))],1,size(m%mf(i)%p2,2)),shape(m%mf(i)%p2)) ) m%mf(i)%p2 = m%mf(i)%x_default
                end if
            end if
        end do
        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:UNMASK', 1, ZHOOK_HANDLE )
    end subroutine

    subroutine rem( m, mask )
      implicit none
        class( tAbstract_model ) :: m
        logical, intent( in )    :: mask( : )

        integer :: i,k
        real( KIND = JPRB ) :: ZHOOK_HANDLE

        if(m%is_empty) return

        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:REM', 0, ZHOOK_HANDLE )

        do i = 1, size( m%mf )
            if(associated(m%mf(i)%p1)) where(mask) m%mf(i)%p1 = m%mf(i)%x_default
            if(associated(m%mf(i)%p2)) then
                do k = 1, size( m%mf(i)%p2, 2 )
                    where(mask) m%mf(i)%p2(:,k) = m%mf(i)%x_default
                end do
            end if
        end do
        if( LHOOK ) call DR_HOOK( 'ABSTRACT_MODEL:REM', 1, ZHOOK_HANDLE )
    end subroutine
end module
