#!/usr/bin/env python
# -*- coding: utf-8 -*- Time-stamp: <2018-03-07 14:26:30 sander>

# multirun.py calls CAABA/MECCA for each entry in input file
# Rolf Sander, 2018

##############################################################################

import os, sys, shutil
caabadir = os.path.realpath(os.path.dirname(__file__)+'/..')
sys.path.append(os.path.abspath(caabadir+'/pycaaba'))
from netCDF4 import Dataset, MFDataset
import subprocess
import time
from glob import glob
from pyteetime import tee # from pycaaba
from rstools import HLINE, runcmd # from pycaaba

DONTEDIT = 'This file was created by ' + os.path.basename(__file__) + ', DO NOT EDIT!'

##############################################################################

class multirun(object):
    """ perform multple CAABA runs
    """

    @classmethod
    def get_MaxTime(cls, ncfile):
        ncid = Dataset(ncfile, 'r') # r = read only
        MaxTime = len(ncid.dimensions['time'])
        ncid.close()
        return MaxTime

    ##########################################################################

    @classmethod
    def create_nml(cls, temp, press, input_ncfile, skeleton):
        tmp_nml = 'tmp_caaba.nml'
        NMLFILE = open(tmp_nml,'w+', 1) # 1=line-buffered
        print >> NMLFILE, "! -*- f90 -*- " + DONTEDIT
        print >> NMLFILE, "&CAABA"
        print >> NMLFILE, "USE_MECCA = T"
        print >> NMLFILE, "USE_READJ = T"
        print >> NMLFILE, "temp = %s" % (temp)
        print >> NMLFILE, "press = %s" % (press)
        #print >> NMLFILE, "relhum = 0.81"
        #print >> NMLFILE, "l_ignore_relhum = T"
        print >> NMLFILE, "photrat_channel = 'readj'"
        print >> NMLFILE, "init_spec = '%s'" % (input_ncfile)
        print >> NMLFILE, "input_readj = '%s'" % (input_ncfile)
        if (skeleton):
            print >> NMLFILE, "l_skeleton = T"
            print >> NMLFILE, "runtime_str = '8 hours'"
        else:
            print >> NMLFILE, "l_steady_state_stop = T"
        print >> NMLFILE, "/"
        NMLFILE.close()
        os.system('ln -fs ' + tmp_nml + ' caaba.nml')

    ##########################################################################

    @classmethod
    def run_batch(cls, input_ncfile, outputdir, line4):
        # execute qsub on dresden here...
        return

    ##########################################################################

    @classmethod
    def makeruns(cls, fullncfile):
        corencfile = os.path.splitext(os.path.basename(fullncfile))[0] # no dir, no suffix
        if (not os.path.isfile(fullncfile)):
            sys.exit('ERROR: %s does not exist' % (fullncfile))
        if (not os.path.isdir(caabadir+'/output/multirun')):
            os.mkdir(caabadir+'/output/multirun')
        outputdir = caabadir+'/output/multirun/'+corencfile
        # if outputdir exists already, rename it, including its modification time:
        if (os.path.isdir(outputdir)):
            os.rename(outputdir, outputdir + '-' + time.strftime(
                '%Y-%m-%d-%H:%M:%S', time.localtime(os.path.getmtime(outputdir))))
        os.mkdir(outputdir)

        # check if multirun is performed for skeletal mechanism reduction:
        if ('skeleton' in corencfile):
            skeleton = True
        else:
            skeleton = False

        os.mkdir(outputdir+'/runs')
        ncid_in = Dataset(fullncfile, 'r')

        timedim = None
        unlimited = ''
        if ('time' in ncid_in.dimensions):
            # normally, data are listed along the 'time' axis:
            timedim = 'time'
        else:
            # otherwise, obtain name and length of the unlimited dimension:
            for dim in ncid_in.dimensions:
                if (ncid_in.dimensions[dim].isunlimited()):
                    timedim = dim
                    unlimited = 'unlimited '
        if (timedim):
            lines = len(ncid_in.dimensions[timedim])
            print '%s\nSTART OF MULTIRUN (%d runs from %s"%s" axis of %s.nc)\n%s\n' % (
                HLINE, lines, unlimited, timedim, corencfile, HLINE)
        else:
            sys.exit('ERROR: Input file %s has neither "time" axis nor an unlimited dimension'
                     % (corencfile))

        #---------------------------------------------------------------------

        for line in range(lines):
            # define line number with 4 digits, starting with 0001:
            line4 = '%4.4d' % (line+1)
            input_ncfile = 'input_'+corencfile+'_'+line4+'.nc'
            # extract one line from fullncfile to input_ncfile:
            ncid_out = Dataset(input_ncfile, 'w', format='NETCDF3_CLASSIC')
            ncid_out.createDimension('time') # no size -> unlimited dimension
            for var in ncid_in.variables:
                #print var, ncid_in.variables[var][line]
                species = ncid_out.createVariable(var,'d',('time')) # f=float=sp, d=dp
                species[:] = ncid_in.variables[var][line]
            ncid_out.close()
            temp  = '%12.10g' % (ncid_in.variables['temp'][line])  # temperature [K]
            press = '%12.10g' % (ncid_in.variables['press'][line]) # pressure [Pa]
            cls.create_nml(temp, press, input_ncfile, skeleton)
            print 'Running simulation %s with T = %s K, p = %s Pa' % (
                line4, temp, press),
            # remove old *.nc files:
            map(os.remove, glob('caaba_*.nc'))
            # run the CAABA/MECCA box model:
            runcmd('./caaba.exe', 'caaba.log', verbosity=0)
            # move/copy model output to another directory:
            destdir = outputdir+'/runs/'+line4 # destination directory
            os.mkdir(destdir)
            shutil.move(input_ncfile, destdir+'/input.nc')
            for wildcard in (['caaba.log', 'caaba.nml', 'mecca.nml', '*.nc', '*.dat']):
                for filename in glob(wildcard):
                    #print 'copying %s to %s' % (filename, destdir)
                    shutil.copy2(filename, outputdir+'/runs/'+line4)
            print ': finished (MaxTime=%d)' % (cls.get_MaxTime('caaba_mecca.nc'))

        #---------------------------------------------------------------------

        # cleanup:
        map(os.remove, glob('tmp_*'))
        os.system('ln -fs nml/caaba_simple.nml caaba.nml')

        return outputdir

    ##########################################################################

    @classmethod
    def postprocessing(cls, outputdir):
        zipfilename = os.path.basename(os.getcwd())+'.zip'
        runcmd('gmake zip', 'gmake_zip.log')
        shutil.move(zipfilename, outputdir)

        print '\nMerging the netcdf files:'
        # based on code in montecarlo.tcsh
        olddir = os.getcwd()
        os.chdir(outputdir)

        # for all runs, copy MaxTime info from caaba_mecca.nc to caaba_mecca_c_end.nc:
        for murun in sorted(glob('runs/*')):
            ncid_caaba_out = Dataset( # r+ = read and modify:
                murun+'/caaba_mecca_c_end.nc', 'r+', format='NETCDF3_CLASSIC')
            MaxTimevar = ncid_caaba_out.createVariable(
                'MaxTime','d',('time','lev','lat','lon')) # f=float=sp, d=dp
            MaxTimevar[:] = cls.get_MaxTime(murun+'/caaba_mecca.nc')
            ncid_caaba_out.close()

        fullfilenames = ['caaba_mecca_c_end.nc', 'caaba_mecca_k_end.nc', 'input.nc']
        for fullfilename in fullfilenames:
            ncfile = os.path.basename(fullfilename)
            print 'Working on %s' % (ncfile)
            print 'Multirun',
            for murun in sorted(glob('runs/*')):
                basemurun = os.path.basename(murun)
                print ' %s' % (basemurun),
                if (not fullfilename == 'input.nc'):
                    # put Multirun number into time:
                    ncid_out = Dataset(murun+'/'+ncfile, 'r+', format='NETCDF3_CLASSIC')
                    # r+ = read and modify
                    ncid_out.variables['time'][:] = basemurun
                    ncid_out.close()
            print ' done'
            # concatenate files along time:
            ncid_caaba_out = Dataset(ncfile, 'w', format='NETCDF3_CLASSIC')
            ncid_caaba_out.createDimension('time') # unlimited dimension (no size specified)
            ncid_caaba_in = MFDataset('runs/*/'+ncfile)
            for var in ncid_caaba_in.variables:
                if (var=='lon'): continue
                if (var=='lat'): continue
                if (var=='lev'): continue
                # print var, ncid_in.variables[var][:]
                species = ncid_caaba_out.createVariable(var,'d',('time')) # f=float=sp, d=dp
                species[:] = ncid_caaba_in.variables[var][:]
            ncid_caaba_in.close()
            ncid_caaba_out.close()
        os.chdir(olddir)

    ##########################################################################

    @classmethod
    def complete(cls, fullncfile):
        outputdir = cls.makeruns(fullncfile)
        cls.postprocessing(outputdir)
        print '\n%s\nEND OF MULTIRUN\n%s' % (HLINE, HLINE)
        return outputdir

##############################################################################

if __name__ == '__main__':

    # ensure that multirun.py is started from main caaba/ directory:
    if (os.getcwd() != caabadir):
        sys.exit('ERROR: multirun.py must be started from caaba ' +
                 'directory, preferably via xcaaba')

    logfilename = 'multirun.log'
    tee.stdout_start(logfilename, append=False) # stdout

    # command line parameter must be a netcdf input file:
    if len(sys.argv)>1:
        fullncfile = os.path.abspath(sys.argv[1]) # convert to fullfilename
    else:
        sys.exit('ERROR: start multirun.py via xcaaba')

    outputdir = multirun.complete(fullncfile)
    
    tee.stdout_stop()
    shutil.move(logfilename, outputdir)
