! ########################
        SUBROUTINE URBAN_HYDRO_SOIL(HKSAT, HHORT, HRUNOFF,                  &
                                    PTSTEP, PPS, PWSOIL, PTG, PDZG,         &
                                    PWSAT, PWFC, PWWILT,                    &
                                    PCONDSAT, PMPOTSAT, PBCOEF,             &
                                    PWG, PWGI,                              &
                                    PRUNOFF_SOIL, PDRAIN ,                  &
                                    PFWTD, PWTD, PTOPQS, PQSB               )
!###############################################
!!****  *URBAN_HYDRO_SOIL*  
!
!!    PURPOSE
!!    -------
!     
!     Calculates the evolution of the water variables for TEB (Road & Roof) i.e.,
!     deep-soil volumetric water content
!     Also determine the runoff and drainage into the soil.
!
!!**  METHOD
!!    ------
!
!     Direct calculation
!
!!    EXTERNAL
!!    --------
!
!     None
!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!!      
!!    REFERENCE
!!    ---------
!!
!!    Boone (2000)
!!    Boone et al. (2000)
!!      
!!    AUTHOR
!!    ------
!!
!!!!      JM Brun		  * IFSTTAR * 
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    02/04/2012   J-M Brun
!-------------------------------------------------------------------------------
!
!*       0.     DECLARATIONS
!               ------------
!
USE MODD_SURF_PAR,     ONLY : XUNDEF
USE MODD_CSTS,         ONLY : XTT
!
USE YOMHOOK   ,ONLY : LHOOK,   DR_HOOK
USE PARKIND1  ,ONLY : JPRB
!
USE MODI_HYDRO_SOILDIF  
!
USE MODE_THERMOS
!
IMPLICIT NONE
!
!----------------------------------------------------------------------------------
!*      0.1    Declarations of arguments
!-----------------------------------------------
!
CHARACTER(LEN=3),     INTENT(IN)    :: HKSAT        ! soil hydraulic profile option ('DEF','SGH')
CHARACTER(LEN=*),     INTENT(IN)    :: HHORT        ! Horton runoff
CHARACTER(LEN=*),     INTENT(IN)    :: HRUNOFF      ! surface runoff formulation
REAL,                 INTENT(IN)    :: PTSTEP       ! Time step (s)
REAL, DIMENSION(:)  , INTENT(IN)    :: PPS          ! Pressure at the surface (Pa)
REAL, DIMENSION(:)  , INTENT(IN)    :: PWSOIL       ! Water that infiltrates from the surface (kg/m2/s)
REAL, DIMENSION(:,:), INTENT(IN)    :: PTG          ! Soil temperature profile (K)
REAL, DIMENSION(:,:), INTENT(IN)    :: PDZG         ! Thickness of soil layers (m)
REAL, DIMENSION(:,:), INTENT(IN)    :: PWSAT        ! Water content at saturation (m3/m3)
REAL, DIMENSION(:,:), INTENT(IN)    :: PWFC         ! Field capacity              (m3/m3)
REAL, DIMENSION(:,:), INTENT(IN)    :: PWWILT       ! Wilting point               (m3/m3)
REAL, DIMENSION(:,:), INTENT(IN)    :: PCONDSAT     ! Hydraulic conductivity at sat (m/s)
REAL, DIMENSION(:,:), INTENT(IN)    :: PMPOTSAT     ! Matrix potential at sat     (m)
REAL, DIMENSION(:,:), INTENT(IN)    :: PBCOEF       ! b parameter for retention curve (-)
REAL, DIMENSION(:,:), INTENT(INOUT) :: PWG          ! Soil volumetric water content profile (m3/m3)
REAL, DIMENSION(:,:), INTENT(INOUT) :: PWGI         ! Soil volumetric water (ice) content profile (m3/m3)
REAL, DIMENSION(:)  , INTENT(OUT)   :: PRUNOFF_SOIL ! Runoff  (kg/m2/s) = PRUNOFF_TOWN !!!! attention modif
REAL, DIMENSION(:)  , INTENT(OUT)   :: PDRAIN       ! Drainage (kg/m2/s)
!
REAL, DIMENSION(:)  , INTENT(IN)    :: PFWTD        !grid-cell fraction of water table to rise
REAL, DIMENSION(:)  , INTENT(IN)    :: PWTD         !water table depth (m)
REAL, DIMENSION(:,:), INTENT(IN)    :: PTOPQS       ! Topmodel subsurface flow by layer (m/s)
REAL, DIMENSION(:)  , INTENT(OUT)   :: PQSB         !Lateral subsurface flow [kg/m²/s]
!
!----------------------------------------------------------------------------------
!*      0.2    Declarations of local variables
!-----------------------------------------------
!
INTEGER                                        :: JWRK      ! loop control                                       
INTEGER                                        :: JJ,JL     ! loop index
INTEGER                                        :: JNDT, JDT ! time splitting indicies
INTEGER                                        :: INL,IDEPTH
INTEGER                                        :: IMAX_LAYER
INTEGER                                        :: ILAYER_HORT
INTEGER, DIMENSION(SIZE(PTG,1))                :: IWG_LAYER
REAL,    DIMENSION(SIZE(PTG,1))                :: ZWDRAIN               ! minimum Wg for drainage (m3 m-3)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZFSAT                 ! Saturated fraction (WARNING !!!! inutilisé)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZDG                   ! Cumulative soil depth (m)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZDZDIF                ! Distance between the midpoints of two consecutive layers (m)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZLETR                 ! transpiration rate (W m-2)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZLEG                  ! bare-soil evaporation rate (W m-2)
REAL,    DIMENSION(SIZE(PTG,1))                :: ZLEGI                 ! surface layer sublimation rate (W m-2)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZF2WGHT               ! root-zone transpiration weights (-)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZROOTFRAC             ! Cumulative root fraction (-)cumulative root fraction (-)
REAL,    PARAMETER                             :: ZRICHARDSDTMAX = 900. ! Maximum timescale for Richard's Eq. (s)
REAL,    DIMENSION(SIZE(PTG,1),SIZE(PTG,2))    :: ZPS,ZTI,ZQSAT,ZQSATI
!
REAL(KIND=JPRB) :: ZHOOK_HANDLE
!
!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
!
IF (LHOOK) CALL DR_HOOK('URBAN_HYDRO_SOIL',0,ZHOOK_HANDLE)
!
!--------------------------------------------------------------------------
!
!*      1.     Initialisation
!              --------------
!
! Root fraction is zero under roads and buildings
ZROOTFRAC(:,:) = 0.
!
! Topmodel saturated fraction, initialisé à 0. dans GARDEN, inutilisé dans URBAN_HYDRO_SOILDIF
ZFSAT(:)       = 0.0 
!
ZWDRAIN(:)     = 0.
!
! No evaporation
ZLETR(:)         =  0.
ZLEG(:)          =  0.
ZLEGI(:)         =  0.
!
! Moisture layers
IWG_LAYER(:) = SIZE(PTG,2)
IMAX_LAYER   = SIZE(PTG,2)
ILAYER_HORT  = SIZE(PTG,2)
!
! Cumultative depth of soil layers
ZDG   (:,1) = PDZG(:,1) 
DO JWRK=2,SIZE(PDZG,2) 
   ZDG   (:,JWRK) = ZDG(:,JWRK-1) + PDZG(:,JWRK)  
END DO
!
! Distance between the midpoints of two consecutive layers (m)
DO JWRK=1,SIZE(PDZG,2)-1
   ZDZDIF(:,JWRK) = (PDZG(:,JWRK) + PDZG(:,JWRK+1))/2.  
END DO
ZDZDIF(:,SIZE(PDZG,2)) = PDZG(:,SIZE(PDZG,2))/2. 
!
! No transpiration
ZF2WGHT(:,:) = 0.
!
!*      2.     RUN-OFF (cf hydro.F90) 
!              ----------------------
!
INL = MAXVAL(IWG_LAYER(:))
!
! Initialize some field
! ---------------------
!
ZPS(:,:) = XUNDEF
ZTI(:,:) = XUNDEF
DO JL=1,INL
   DO JJ=1,SIZE(PTG,1)
      IDEPTH=IWG_LAYER(JJ)
      IF(JL<=IDEPTH)THEN
        ZPS(JJ,JL) = PPS(JJ)
        ZTI(JJ,JL) = MIN(XTT,PTG(JJ,JL))
      ENDIF
   ENDDO
ENDDO
!
! Compute specific humidity at saturation for the vapor conductivity
! ------------------------------------------------------------------
!
ZQSAT (:,:) = QSAT (PTG(:,:),ZPS(:,:),IWG_LAYER(:),INL)
ZQSATI(:,:) = QSATI(ZTI(:,:),ZPS(:,:),IWG_LAYER(:),INL)
!
!*      3.     Call hydro_soildif
!              ------------------
!
CALL HYDRO_SOILDIF(HRUNOFF, PTSTEP,                                    &
                   PBCOEF, PWSAT, PCONDSAT, PMPOTSAT, PWFC, ZDG, PDZG, & 
                   ZDZDIF, PWSOIL, ZLETR, ZLEG, ZLEGI, ZF2WGHT,        &
                   PWG, PWGI, PTG, PPS, ZQSAT, ZQSATI,                 &
                   PDRAIN, PRUNOFF_SOIL, IWG_LAYER,                    &
                   IMAX_LAYER, PTOPQS, PQSB, PFWTD, PWTD     )
!-------------------------------------------------------------------------------
!
!
IF (LHOOK) CALL DR_HOOK('URBAN_HYDRO_SOIL',1,ZHOOK_HANDLE)
!-------------------------------------------------------------------------------
!
END SUBROUTINE URBAN_HYDRO_SOIL


