module methane_parameters
  implicit none

  integer, parameter :: dp = selected_real_kind(15,300) !64bit double precision


  ! Set-up of the model run

  ! a) Peat layer set-up
  ! peat depth = nlayers * layer_thickness [m]
  ! If one wants to use an uneven grid (e.g. logarithmic layer thicknesses)
  ! it should currently be given in the main program.
  integer,  parameter :: nlayers = 10
  real(dp), parameter :: layer_thickness = 0.2_dp

  ! b) Define input
  ! n_data is number of rows, cols_data is columns
  integer, parameter :: n_data = 2556,cols_data = 5

  
  ! Other parameters

  real(dp), parameter :: min_dz = 0.01_dp
  ! minimum thickness for boxes [m], because small boxes require a smaller time step

  integer, parameter :: n_flux_parts = 20
  ! indices for the bookkeeping arrays where we collect the output fluxes etc.
! *** methane transport ***
  !  1 = methane diffusion
  !  2 = methane plant transport
  !  3 = methane ebullition into air
  !  4 = methane ebullition total (when WTD is below peat surface, ebullition is
  !      released in the lowest air layer inside peat)
! *** methane loss & production ***
  !  5 = production of methane in anoxic respiration (not a flux)
  !  6 = oxidation of methane (not a flux, just a sink)
! *** CO2 transport ***
  !  7 = CO2 diffusion
  !  8 = CO2 plant transport
  !  9 = CO2 ebullition into air (CO2 ebullition is assumed to occur with methane)
  ! 10 = CO2 ebullition total (when WTD is below peat surface, ebullition is
  !      released in the lowest air layer inside peat)
! *** CO2 production ***
  ! 11 = production of CO2 in anoxic respiration (not a flux out)
  ! 12 = production of CO2 in oxic respiration (also not a flux)
  ! 13 = production of CO2 in methane oxidation (not a flux)
! *** Oxygen transport ***
  ! 14 = oxygen diffusion
  ! 15 = oxygen plant transport
  ! 16 = oxygen ebullition into air (oxygen ebullition is assumed to occur with methane)
  ! 17 = oxygen ebullition total (when WTD is below peat surface, ebullition is
  !      released in the lowest air layer inside peat)
! *** Oxygen consumption ***
  ! 18 = oxygen consumed in inhibition of methane production
  ! 19 = oxygen consumed in oxic respiration
  ! 20 = oxygen consumed in methane oxidation


  integer, parameter :: mdiff   =  1, &
                        mplant  =  2, &
                        mebuair =  3, &
                        mebutot =  4, &
                        mprod   =  5, &
                        moxi    =  6, &
                        cdiff   =  7, &
                        cplant  =  8, &
                        cebuair =  9, &
                        cebutot = 10, &
                        cprod   = 11, &
                        cresp   = 12, &
                        coxi    = 13, &
                        odiff   = 14, &
                        oplant  = 15, &
                        oebuair = 16, &
                        oebutot = 17, &
                        oprod   = 18, &
                        oresp   = 19, &
                        ooxi    = 20


  ! Scaling to choose a safe time step for numerics. Anything up to 1 should be
  ! safe, but values close to 1 might give oscillations (but only dampening
  ! type) in the solution of diffusion. 0.5 should be very safe.
  real(dp), parameter :: time_scale = 0.5

  ! Gas constant
  real(dp), parameter :: Rgas = 8.3144621_dp ! [J mol-1 K-1]

  ! Reference temperature for respiration and oxidation
  real(dp), parameter :: T0 = 283._dp ! [K]

  ! Atmospheric pressure [Pa]
  real(dp), parameter :: atm = 101325._dp ! atm pressure [Pa]

  ! Ebullition
  ! half-life of supersaturated dissolved methane, CO2 and O2
  real(dp), parameter :: ebu_hl = 1800._dp ! 30 minutes, 1800 s
  ! super saturation requirement for future use, now set to 1
  real(dp), parameter :: ssr = 1.0_dp

  ! methane production, oxygen inhibition on methanogenesis
  real(dp), parameter :: ny = 400._dp ! [m3 mol-1], Tang et al. 2010

  ! Oxygen respiration parameters
  real(dp), parameter :: V_R0 = 8.7e-5_dp,&      ! [mol m-3(peat) s-1]
                         K_R  = 0.22_dp,&        ! [mol m-3(water)]
                         deltaE_R = 93000._dp    ! [J mol-1]

  ! Methane oxidation parameters
  real(dp), parameter :: V_oxid_max = 2.7e-4_dp,&    ! [mol m-3(peat) s-1]
                         delta_E_oxid = 49000._dp ,& ! [J mol-1]
                         K_CH4 = 0.44_dp,&           ! [mol m-3(water)]
                         K_O2 = 0.33_dp              ! [mol m-3(water)]

  ! molar mass of Carbon, g(C) mol(C)-1 
  real(dp), parameter  :: M_C = 12.011_dp

  ! mole fractions
  real(dp), parameter :: methane_mole_fraction = 1.8e-6_dp  ! [mol mol(dry air)-1]
  real(dp), parameter :: co2_mole_fraction     = 400.e-6_dp ! [mol mol(dry air)-1]
  real(dp), parameter :: oxygen_mole_fraction  = 0.2095_dp  ! [mol mol(dry air)-1]

  ! fraction of methane
  ! mol methane produced per mol carbon, mol(CH4) mol(C)-1
  real(dp), parameter  :: fm = 0.25_dp

  ! Root parameters
  ! lambda_root: root depth distribution decay parameter, [m]
  ! root_tortuosity 
  ! root_km: root-ending area per root (dry)biomass [m2/kg]
  real(dp), parameter :: lambda_root = 0.24_dp 
  real(dp), parameter :: root_tortuosity = 1.0_dp 
  real(dp), parameter :: root_km = 0.17_dp 

  ! Specific leaf area
  real(dp), parameter :: sla = 23.0_dp

  ! peat porosity
  real(dp), parameter :: por = 0.85_dp

  ! coefficients for scaling diffusion in free air/water to diffusion in 
  ! air-filled and water-filled peat
  real(dp), parameter :: peat_coeff_a = 0.37_dp
  real(dp), parameter :: peat_coeff_w = 0.90_dp

end module methane_parameters
