  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  ! This is the main program of HIMMELI, HelsinkI Model of MEthane buiLd-up !
  ! and emIssion. Please see the user manual and article Raivonen et al.,   !
  ! Geoscientific Model Development Discussions xxx, 2017, for information. ! 
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

program methane_main

  use methane_parameters, only: dp,n_flux_parts, &
    mdiff,mplant,mebuair,mebutot,mprod      ,moxi, &
    cdiff,cplant,cebuair,cebutot,cprod,cresp,coxi, &
    odiff,oplant,oebuair,oebutot,oprod,oresp,ooxi, &
    Rgas,atm,fm,methane_mole_fraction,co2_mole_fraction, &
    oxygen_mole_fraction,nlayers,layer_thickness,lambda_root, &
    root_km,sla,por,root_tortuosity,n_data,cols_data, &
    peat_coeff_a,peat_coeff_w
  use methane_geometry, only: geometry,new_geom,gas_move_wtd, &
    find_wtd_index
  use methane_output, only: R_output

  implicit none

 
  real(dp), dimension(cols_data,n_data) :: input_data
  integer,  dimension(n_data) :: day_data
  real(dp), dimension(n_data) :: temp_data,wtd_data,lai_data, &
                                 root_data,resp_data

  ! Soil temperature profile input
  real(dp), dimension(nlayers,n_data) :: ztemp_data

  ! output
  real(dp) :: gases_out(n_flux_parts),gases_out_day(n_flux_parts), &
              gases_out_tot(n_flux_parts),gases_out_rates(n_flux_parts)

  ! time loop control:
  integer  :: time_i,n_steps,day_first,day_last
  real(dp) :: time,dt

  ! miscellaneous 
  integer :: i

  ! geometry

  ! In principle, we set the model grid to have 'nlayers' background layers,
  ! but we need one extra layer to deal with the moving water table.
  integer :: nz      ! will always be nlayers + 1
  integer :: index2m ! index of 2m depth (or bottom level if peat
                     ! thickness is less than 2m)
  integer :: i_wtd   ! the layer containing the WTD
  
  ! naming convention: arrays of size = nz end with P (P = "profile"), also
  ! levelsP although its size is nz+1

  real(dp) :: wtd,wtd_eff,wtd_new
  real(dp), allocatable :: fixed_levelsP(:)
  type(geometry) :: geom

  ! fixed_levelsP: coordinates in [meter] of levels that separate the layers in
  ! the background grid. The moving level will then be inside one of these
  ! layers, at water table level. Or perhaps above fixed_levels(1), if water is
  ! above the peat surface.
  ! geom: see type contents in module methane_geometry. Contains:
  !       levelsP, dzP (arrays)
  !       surfaceQ, airQ, waterQ (logical)
  !       s1, s2, a1, a2, w1, w2 (integer)
  ! wtd: water table depth (WTD), positive is below surface, negative
  !      value means the water surface is above peat surface
  ! wtd_eff: effective wtd, returned by subroutine new_geom. If the input wtd
  !          is too close (parameter min_dz, usually 1 cm) to a fixed level,
  !          we don't accept the value, but round to the fixed level, thus
  !          avoiding to ever have a layer less than 1 cm thick.
  ! por: peat porosity, volume fraction of air or water in peat

  ! roots
  real(dp) :: root_mass
  real(dp), allocatable :: rootsP(:)

  ! rootsP: fraction of roots in each layer, sums to 1
  ! root_mass: root (dry)biomass per m2 of peatland [kg]
 

  ! temperature
  real(dp) :: T_surface,T_deep
  real(dp), allocatable :: tempP(:)
  ! tempP:      temperature (Kelvin) in layers


  ! methane, CO2 and oxygen
  ! xP: amount of x in each layer [mol], really [mol] not [mol m-3]
  real(dp), allocatable :: methaneP(:) ! [mol]
  real(dp), allocatable :: co2P(:) ! [mol]
  real(dp), allocatable :: oxygenP(:) ! [mol]

  ! x_Ca: concentration of x in ambient air [mol m-3]
  real(dp) :: methane_Ca,co2_Ca,oxygen_Ca

  ! temporary
  real(dp), allocatable :: methane_newP(:)
  real(dp), allocatable :: co2_newP(:)
  real(dp), allocatable :: oxygen_newP(:)
  real(dp), allocatable :: alphaP(:),CwaterP(:)

  ! anoxic_respi_rate:     rate of anoxic respiration [mol C s-1]
  real(dp) :: anoxic_respi_rate

! *** methane transport ***
  !  1 = methane diffusion
  !  2 = methane plant transport
  !  3 = methane ebullition into air
  !  4 = methane ebullition total (when WTD is below peat surface, ebullition is
  !      released in the lowest air layer inside peat)
! *** methane loss & production ***
  !  5 = production of methane in anaerobic respiration (not a flux)
  !  6 = oxidation of methane (not a flux, just a sink)
! *** CO2 transport ***
  !  7 = CO2 diffusion
  !  8 = CO2 plant transport
  !  9 = CO2 ebullition into air (CO2 ebullition is assumed to occur with methane)
  ! 10 = CO2 ebullition total (when WTD is below peat surface, ebullition is
  !      released in the lowest air layer inside peat)
! *** CO2 production ***
  ! 11 = production of CO2 in anaerobic respiration (not a flux out)
  ! 12 = production of CO2 in aerobic respiration (also not a flux)
  ! 13 = production of CO2 in methane oxidation (not a flux)
! *** Oxygen transport ***
  ! 14 = oxygen diffusion
  ! 15 = oxygen plant transport
  ! 16 = oxygen ebullition into air (oxygen ebullition is assumed to occur with methane)
  ! 17 = oxygen ebullition total (when WTD is below peat surface, ebullition is
  !      released in the lowest air layer inside peat)
! *** Oxygen consumption ***
  ! 18 = oxygen consumed in inhibition of methane production
  ! 19 = oxygen consumed in aerobic respiration
  ! 20 = oxygen consumed in methane oxidation


! mass control
  real(dp) :: methane_time1,methane_time2, &
              methane_wtd_flux,methane_wtd_flux_tot, &
              methane_produced,methane_escaped, &
! CO2
              co2_time1,co2_time2, &
              co2_wtd_flux,co2_wtd_flux_tot, &
              co2_produced,co2_escaped, &
! O2
              oxygen_time1,oxygen_time2, &
              oxygen_wtd_flux,oxygen_wtd_flux_tot, &
              o2_consumed,o2_input,o2_escaped

  !!! main program !!!


  nz = nlayers + 1 ! number of layers including the extra moving layer


  allocate(fixed_levelsP(nz),geom%levelsP(nz+1),geom%dzP(nz))
  allocate(methaneP(nz),methane_newP(nz))
  allocate(co2P(nz),co2_newP(nz))
  allocate(oxygenP(nz),oxygen_newP(nz))
  allocate(rootsP(nz))
  allocate(alphaP(nz),CwaterP(nz))
  allocate(tempP(nz))

  ! Initialize height levels. If using an uneven grid, it should be
  ! given manually, see below.
   do i = 1,nz
    fixed_levelsP(i) = (i-1) * layer_thickness
   end do

  ! An example of logarithmic layers in 2 m of peat.
  ! fixed_levelsP = (/ 0.0, 0.06, 0.19, 0.45, 0.97, 2.0 /)
  

  ! Maximum rooting depth is 2 m. If peat thickness exceeds 2 m, 
  ! index2m shows the location of the 2m depth (otherwise index2m=nz).
  ! NB!: You need to have one fixed level exactly
  ! at 2m when peat thickness > 2m.
  if (maxval(fixed_levelsP).le.2.0) then
   index2m = nz
  else
   index2m = minloc(abs(fixed_levelsP - 2.0), 1)
  endif
 
  ! read main input data
  open(11,file='input_data.txt',action='READ')
  read(11,*) input_data
  close(11)

  ! read soil temperature data
  open(12,file='ztemp_data.txt',action='read')
  read(12, *) ztemp_data
  close(12)

  day_data  = floor(input_data(1,:)) ! day number
  temp_data = input_data(2,:) ! temperature
  wtd_data  = input_data(3,:) ! wtd depth
  lai_data  = input_data(4,:) ! aerenchymatous plant LAI
  resp_data = input_data(5,:) ! rate of anoxic respiration [mol m-2 s-1]
  root_data = lai_data/sla    ! aerenchymatous plant root dry biomass [kg m-2]

  day_first = day_data(1)
  day_last  = day_data(size(day_data)) ! takes the last element

  ! initialize geometry
  wtd_new = wtd_data(1)
  call new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd_new)
  wtd = wtd_eff

  write(*,*) 'levels:'
  write(*,'(11(9F7.3))') geom%levelsP
  write(*,*) 'wtd:', wtd

  ! initialize root distribution
  rootsP(1:nz) = &
    calculate_root_fraction(lambda_root,nz,geom%levelsP,fixed_levelsP(1),&
      index2m)

  write(*,*) 'roots:'
  write(*,'(10(6F6.3))') rootsP

  
  ! First set all temperatures equal
  T_surface = temp_data(1) + 273.15
  T_deep    = temp_data(1) + 273.15
  tempP     = temp_data(1) + 273.15

  ! x_Ca: concentration of x in ambient air [mol m-3]
  ! n/V = p/(R*T)
  methane_Ca = methane_mole_fraction*atm/(Rgas*T_surface) ! [mol m-3]
  co2_Ca     = co2_mole_fraction*atm/(Rgas*T_surface) ! [mol m-3]
  oxygen_Ca  = oxygen_mole_fraction*atm/(Rgas*T_surface) ! [mol m-3]

  ! In the beginning, the concentration profiles to zero.
  methaneP = 0 
  co2P = 0
  oxygenP  = 0

  ! time loop preparation
  time = day_first * 24*3600
  dt = 24*3600 ! 1 day in seconds

  gases_out = 0 ; gases_out_day = 0 ; gases_out_tot = 0

  write(*,*) ; write(*,*) 'methane:' ; write(*,'(11(6F6.3))') methaneP

  methane_time1 = sum(methaneP)
  co2_time1 = sum(co2P)
  oxygen_time1 = sum(oxygenP)

  methane_wtd_flux_tot = 0
  co2_wtd_flux_tot     = 0
  oxygen_wtd_flux_tot  = 0
  methane_wtd_flux     = 0
  oxygen_wtd_flux      = 0
  co2_wtd_flux         = 0   

  gases_out_rates = gases_out / dt

  call R_output(time,dt,nz,geom,por,methaneP,co2P,oxygenP,gases_out_rates,&
   methane_wtd_flux, oxygen_wtd_flux, co2_wtd_flux)

  n_steps = day_last - day_first + 1

  do time_i = 1, n_steps ! main time loop ********************************

  
    time = time + dt
    wtd_new = wtd_data(time_i)
    call new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd_new)

    ! If WTD has moved, adjust gas (and root) content in gridboxes
    if (wtd /= wtd_eff) then

      rootsP(1:nz) = &
        calculate_root_fraction(lambda_root,nz,geom%levelsP,fixed_levelsP(1),&
         index2m)

      
      alphaP = methane_solubility(tempP,Rgas) ! [m(ch4)3 m(water)-3]
      call gas_move_wtd(methane_newP,methane_wtd_flux,nz,wtd,wtd_eff,methaneP, &
        CwaterP,alphaP,fixed_levelsP)
      methaneP = methane_newP

      alphaP = co2_solubility(tempP,Rgas) ! [m(co2)3 m(water)-3]
      call gas_move_wtd(co2_newP,co2_wtd_flux,nz,wtd,wtd_eff,co2P, &
        CwaterP,alphaP,fixed_levelsP)
      co2P = co2_newP

      alphaP = oxygen_solubility(tempP,Rgas) ! [m(o2)3 m(water)-3]
      call gas_move_wtd(oxygen_newP,oxygen_wtd_flux,nz,wtd,wtd_eff,oxygenP, &
        CwaterP,alphaP,fixed_levelsP)
      oxygenP = oxygen_newP

      wtd = wtd_eff

      ! Update the fluxes that are caused by changes in wtd.
      methane_wtd_flux_tot = methane_wtd_flux_tot + methane_wtd_flux
      co2_wtd_flux_tot = co2_wtd_flux_tot + co2_wtd_flux
      oxygen_wtd_flux_tot  = oxygen_wtd_flux_tot  + oxygen_wtd_flux

     else ! wtd fluxes are zero
     methane_wtd_flux = 0.0
     oxygen_wtd_flux  = 0.0
     co2_wtd_flux     = 0.0
    end if

    ! Update soil temperatures. Data from i_wtd layer is used both
    ! for tempP(i_wtd) and tempP(i_wtd+1).
    i_wtd = max(find_wtd_index(geom%levelsP,wtd_new),1)
    tempP(:i_wtd) = ztemp_data(:i_wtd,time_i) + 273.15
    tempP(i_wtd+1:) = ztemp_data(i_wtd:,time_i) + 273.15

    T_surface       = 273.15 + temp_data(time_i)

    anoxic_respi_rate  = resp_data(time_i)
    root_mass = root_data(time_i)

    call bigstep(methaneP,co2P,oxygenP,gases_out, &
      nz,geom,por,dt,tempP,methane_Ca,co2_Ca,oxygen_Ca, &
      anoxic_respi_rate,fm, &
      rootsP,root_km,root_mass)

    gases_out_rates = gases_out / dt

    call R_output(time,dt,nz,geom,por,methaneP,co2P,oxygenP,gases_out_rates, &
                  methane_wtd_flux, oxygen_wtd_flux, co2_wtd_flux)

    gases_out_tot = gases_out_tot + gases_out

  end do ! end of main time loop *********************************************


  methane_time2 = sum(methaneP)
  co2_time2 = sum(co2P)
  oxygen_time2 = sum(oxygenP)

  write(*,*) ; write(*,*) 'Final results'
  write(*,*) 'levelsP:' ; write(*,'(11(11F9.3))') geom%levelsP
  write(*,*)
  write(*,*) 'methane_Ca:', methane_Ca
  write(*,*) 'methane:' ; write(*,'(11(11F12.6))') methaneP
  write(*,*)
  write(*,*) 'co2_Ca:', co2_Ca
  write(*,*) 'CO2:' ; write(*,'(11(11F12.6))') co2P
  write(*,*)
  write(*,*) 'oxygen_Ca:', oxygen_Ca
  write(*,*) 'oxygen:' ; write(*,'(11(11F12.6))') oxygenP
  write(*,*)
  write(*,*) 'fraction of methane:', fm
  write(*,*)
  write(*,*) 'Methane results:'
  write(*,*) 'column total, t_start: ',methane_time1
  write(*,*) 'column total, t_end:   ',methane_time2
  methane_produced = gases_out_tot(mprod)
  methane_escaped = gases_out_tot(mdiff) + gases_out_tot(mplant) &
       + gases_out_tot(mebuair)
  write(*,*) 'produced: ',methane_produced
  write(*,*) '    produced (anoxic respiration):   ',gases_out_tot(mprod)
  write(*,*) 'consumed: ',gases_out_tot(moxi)
  write(*,*) '    oxidized:                      ',gases_out_tot(moxi)
  write(*,*) 'escaped:  ',methane_escaped
  write(*,*) '    escaped due to diffusion:      ',gases_out_tot(mdiff)
  write(*,*) '    escaped due to plant transport:',gases_out_tot(mplant)
  write(*,*) '    escaped due to ebullition:     ',gases_out_tot(mebuair)
  write(*,*) '    (total ebullition was:         ',gases_out_tot(mebutot),')'
  write(*,*) 'change due to moving WTD: ',methane_wtd_flux_tot
  write(*,*) 'balance:  ', -(methane_time2 - methane_time1) &
       + methane_produced + gases_out_tot(moxi) + methane_escaped &
       - methane_wtd_flux_tot, ' mol'
  write(*,*) '(Something like x.xE-014 is acceptable)'
  write(*,*)
  write(*,*) 'CO2 results:'
  write(*,*) 'column total, t_start: ',co2_time1
  write(*,*) 'column total, t_end:   ',co2_time2
  co2_produced = gases_out_tot(cprod) &
       + gases_out_tot(cresp) + gases_out_tot(coxi)
  co2_escaped = gases_out_tot(cdiff) + gases_out_tot(cplant) &
       + gases_out_tot(cebuair)
  write(*,*) 'produced: ',co2_produced
  write(*,*) '    produced (anoxic respiration):      ',gases_out_tot(cprod)
  write(*,*) '    produced (oxic respiration):        ',gases_out_tot(cresp)
  write(*,*) '    produced (CH4 oxidation):      ',gases_out_tot(coxi)
  write(*,*) 'escaped:  ',co2_escaped
  write(*,*) '    escaped due to diffusion:      ',gases_out_tot(cdiff)
  write(*,*) '    escaped due to plant transport:',gases_out_tot(cplant)
  write(*,*) '    escaped due to ebullition:     ',gases_out_tot(cebuair)
  write(*,*) '    (total ebullition was:         ',gases_out_tot(cebutot),')'
  write(*,*) 'change due to moving WTD: ',co2_wtd_flux_tot
  write(*,*) 'balance:  ', -(co2_time2 - co2_time1) &
       + co2_produced + co2_escaped &
       - co2_wtd_flux_tot, 'mol'
  write(*,*)
  write(*,*) 'Oxygen results:'
  write(*,*) 'column total, t_start: ',oxygen_time1
  write(*,*) 'column total, t_end:   ',oxygen_time2
  o2_consumed = gases_out_tot(oprod) + gases_out_tot(oresp) + gases_out_tot(ooxi)
  o2_input = gases_out_tot(odiff) + gases_out_tot(oplant)
  o2_escaped = gases_out_tot(oebuair)
  write(*,*) 'consumed:        ',o2_consumed
  write(*,*) '    consumed (inhibition of CH4 production): ',gases_out_tot(oprod)
  write(*,*) '    consumed (oxic respiration):      ',gases_out_tot(oresp)
  write(*,*) '    consumed (CH4 oxidation): ',gases_out_tot(ooxi)
  write(*,*) 'transport input: ',o2_input
  write(*,*) '    diffusion input:              ',gases_out_tot(odiff)
  write(*,*) '    plant transport input:        ',gases_out_tot(oplant)
  write(*,*) 'escaped:         ',o2_escaped
  write(*,*) '    escaped due to ebullition:    ',gases_out_tot(oebuair)
  write(*,*) '    (total ebullition was:        ',gases_out_tot(oebutot),')'
  write(*,*) 'change due to moving WTD: ',oxygen_wtd_flux_tot
  write(*,*) 'balance:         ', -(oxygen_time2 - oxygen_time1) &
       + o2_consumed + o2_input + o2_escaped &
       - oxygen_wtd_flux_tot, 'mol'
contains

  subroutine bigstep(methaneP,co2P,oxygenP,gases_out_bigstep, &
    nz,geom,por,big_dt,tempP,methane_Ca,co2_Ca,oxygen_Ca, &
    anoxic_respi_rate,fm,rootsP,root_km,root_mass)
    integer, intent(in)        :: nz
    real(dp), intent(inout)    :: methaneP(nz),co2P(nz),oxygenP(nz)
    real(dp), intent(out)      :: gases_out_bigstep(n_flux_parts)
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: por,big_dt,tempP(nz),methane_Ca,co2_Ca, &
                                  oxygen_Ca,anoxic_respi_rate, &
                                  fm,rootsP(nz),root_km,root_mass
    real(dp)                   :: met_d1(nz),met_d2(nz),met_d3(nz),met_d4(nz), &
                                  met_d(nz)
    real(dp)                   :: co2_d1(nz),co2_d2(nz),co2_d3(nz),co2_d4(nz), &
                                  co2_d(nz)
    real(dp)                   :: oxy_d1(nz),oxy_d2(nz),oxy_d3(nz), &
                                  oxy_d4(nz),oxy_d(nz)
    real(dp)                   :: f_out1(n_flux_parts), &
                                  f_out2(n_flux_parts), &
                                  f_out3(n_flux_parts), &
                                  f_out4(n_flux_parts)
    real(dp)                   :: met_DaP(nz),met_DwP(nz),met_alphaP(nz), &
                                  met_DcondP(nz),met_root_condP(nz)
    real(dp)                   :: co2_DaP(nz),co2_DwP(nz),co2_alphaP(nz), &
                                  co2_DcondP(nz),co2_root_condP(nz)
    real(dp)                   :: oxy_DaP(nz),oxy_DwP(nz),oxy_alphaP(nz), &
                                  oxy_DcondP(nz),oxy_root_condP(nz)
    real(dp)                   :: max_dt_met,max_dt_co2,max_dt_oxy
    real(dp)                   :: max_diffusion_dt,max_respi_dt, &
                                  max_oxid_dt,max_ebu_dt,max_dt,dt
    real(dp)                   :: gases_out(n_flux_parts)

    integer                    :: n_small_steps, i

    met_DaP    = methane_D_air(tempP,peat_coeff_a)
    met_DwP    = methane_D_water(tempP,peat_coeff_w)
    met_alphaP = methane_solubility(tempP,Rgas)
    met_DcondP = diffusion_conductances(nz,geom,met_DaP,met_DwP,met_alphaP,&
                                         peat_coeff_w)
    max_dt_met = find_max_diffusion_dt(nz,geom,met_DcondP)

    co2_DaP    = co2_D_air(tempP,peat_coeff_a)
    co2_DwP    = co2_D_water(tempP,peat_coeff_w)
    co2_alphaP = co2_solubility(tempP,Rgas)
    co2_DcondP = diffusion_conductances(nz,geom,co2_DaP,co2_DwP,co2_alphaP,&
                                         peat_coeff_w)
    max_dt_co2 = find_max_diffusion_dt(nz,geom,co2_DcondP)

    oxy_DaP    = oxygen_D_air(tempP,peat_coeff_a)
    oxy_DwP    = oxygen_D_water(tempP,peat_coeff_w)
    oxy_alphaP = oxygen_solubility(tempP,Rgas)
    oxy_DcondP = diffusion_conductances(nz,geom,oxy_DaP,oxy_DwP,oxy_alphaP,&
                                         peat_coeff_w)
    max_dt_oxy = find_max_diffusion_dt(nz,geom,oxy_DcondP)

    met_root_condP = root_conductances(nz,geom,met_DaP,rootsP,root_km,&
                                        root_mass,root_tortuosity)
    co2_root_condP = root_conductances(nz,geom,co2_DaP,rootsP,root_km,&
                                        root_mass,root_tortuosity)
    oxy_root_condP = root_conductances(nz,geom,oxy_DaP,rootsP,root_km,&
                                        root_mass,root_tortuosity)

    max_diffusion_dt = min(max_dt_met,max_dt_co2,max_dt_oxy)
    max_respi_dt = find_max_respiration_dt(nz,geom,oxygenP,tempP,oxy_alphaP,por)
    max_oxid_dt = find_max_oxidation_dt(methaneP,met_alphaP,oxygenP, &
                                        oxy_alphaP,tempP,nz,geom,por)
    max_ebu_dt = find_max_ebullition_dt()

    max_dt = min(max_diffusion_dt,max_respi_dt,max_oxid_dt,max_ebu_dt)

    n_small_steps = ceiling(big_dt / max_dt)
    dt = big_dt / n_small_steps

    gases_out_bigstep = 0

    do i = 1, n_small_steps

      ! Runge-Kutta-4
      call derivative(methaneP, &
        co2P, &
        oxygenP, &
        met_d1,co2_d1,oxy_d1,f_out1, &
        nz,geom,por,methane_Ca,co2_Ca,oxygen_Ca, &
        anoxic_respi_rate,fm,tempP, &
        met_DcondP,co2_DcondP,oxy_DcondP,met_alphaP,co2_alphaP,oxy_alphaP, &
        rootsP,met_root_condP,co2_root_condP,oxy_root_condP)

      call derivative(methaneP + 0.5*dt*met_d1, &
        co2P + 0.5*dt*co2_d1, &
        oxygenP + 0.5*dt*oxy_d1, &
        met_d2,co2_d2,oxy_d2,f_out2, &
        nz,geom,por,methane_Ca,co2_Ca,oxygen_Ca, &
        anoxic_respi_rate,fm,tempP, &
        met_DcondP,co2_DcondP,oxy_DcondP,met_alphaP,co2_alphaP,oxy_alphaP, &
        rootsP,met_root_condP,co2_root_condP,oxy_root_condP)

      call derivative(methaneP + 0.5*dt*met_d2, &
        co2P + 0.5*dt*co2_d2, &
        oxygenP + 0.5*dt*oxy_d2, &
        met_d3,co2_d3,oxy_d3,f_out3, &
        nz,geom,por,methane_Ca,co2_Ca,oxygen_Ca, &
        anoxic_respi_rate,fm,tempP, &
        met_DcondP,co2_DcondP,oxy_DcondP,met_alphaP,co2_alphaP,oxy_alphaP, &
        rootsP,met_root_condP,co2_root_condP,oxy_root_condP)

      call derivative(methaneP + dt*met_d3, &
        co2P + dt*co2_d3, &
        oxygenP + dt*oxy_d3, &
        met_d4,co2_d4,oxy_d4,f_out4, &
        nz,geom,por,methane_Ca,co2_Ca,oxygen_Ca, &
        anoxic_respi_rate,fm,tempP, &
        met_DcondP,co2_DcondP,oxy_DcondP,met_alphaP,co2_alphaP,oxy_alphaP, &
        rootsP,met_root_condP,co2_root_condP,oxy_root_condP)

      met_d = (met_d1 + 2*met_d2 + 2*met_d3 + met_d4)/6
      co2_d = (co2_d1 + 2*co2_d2 + 2*co2_d3 + co2_d4)/6
      oxy_d = (oxy_d1 + 2*oxy_d2 + 2*oxy_d3 + oxy_d4)/6
      gases_out = (f_out1 + 2*f_out2 + 2*f_out3 + f_out4)/6

      methaneP = methaneP + dt * met_d
      co2P = co2P + dt * co2_d
      oxygenP  = oxygenP  + dt * oxy_d
      gases_out_bigstep = gases_out_bigstep + dt * gases_out

    end do

  end subroutine bigstep

  ! time derivative of all state variables (arrays), also outputs fluxes
  subroutine derivative(methaneP,co2P,oxygenP, &
    methane_d,co2_d,oxygen_d,f_out, &
    nz,geom,por,methane_Ca,co2_Ca,oxygen_Ca, &
    anoxic_respi_rate,fm,tempP, &
    met_DcondP,co2_DcondP,oxy_DcondP,met_alphaP,co2_alphaP,oxy_alphaP, &
    rootsP,met_root_condP,co2_root_condP,oxy_root_condP)
    real(dp), intent(out)      :: methane_d(nz),co2_d(nz),oxygen_d(nz), &
                                  f_out(n_flux_parts)
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: methaneP(nz),co2P(nz),oxygenP(nz),&
                                  por,methane_Ca,co2_Ca,oxygen_Ca, &
                                  anoxic_respi_rate,fm,tempP(nz), &
                                  met_DcondP(nz),co2_DcondP(nz),oxy_DcondP(nz), &
                                  met_alphaP(nz),co2_alphaP(nz),oxy_alphaP(nz), &
                                  rootsP(nz),met_root_condP(nz),co2_root_condP(nz), &
                                  oxy_root_condP(nz)
    real(dp)                   :: met_diffusionP(nz),co2_diffusionP(nz), &
                                  oxy_diffusionP(nz), &
                                  met_flux,co2_flux,oxy_flux, &
                                  met_prodP(nz),co2_prodP(nz),oxy_prodP(nz), &
                                  met_anox_resp,co2_anox_resp,oxy_anox_resp, &
                                  met_transportP(nz),co2_transportP(nz),oxy_transportP(nz), &
                                  oxy_respiP(nz),oxy_respi_tot, &
                                  co2_respiP(nz),co2_respi_tot, &
                                  met_oxidP(nz),met_oxid_tot, &
                                  co2_oxidP(nz),co2_oxid_tot, &
                                  oxy_oxidP(nz),oxy_oxid_tot, &
                                  mebu_rateP(nz),mebu_tot,mebu_out, &
                                  cebu_rateP(nz),cebu_tot,cebu_out, &
                                  oebu_rateP(nz),oebu_tot,oebu_out

    f_out = 0

    ! Diffusion
    ! Methane diffusion
    call diffusion(met_diffusionP,met_flux,methaneP,nz,geom,por,methane_Ca, &
      met_DcondP,met_alphaP)
    f_out(mdiff) = met_flux
    ! CO2 diffusion
    call diffusion(co2_diffusionP,co2_flux,co2P,nz,geom,por,co2_Ca, &
      co2_DcondP,co2_alphaP)
    f_out(cdiff) = co2_flux    
    ! Oxygen diffusion
    call diffusion(oxy_diffusionP,oxy_flux,oxygenP,nz,geom,por,oxygen_Ca, &
      oxy_DcondP,oxy_alphaP)   
    f_out(odiff) = oxy_flux


    ! Methane and CO2 production in anoxic respiration 
    call methane_production(met_prodP, met_anox_resp, co2_prodP, co2_anox_resp, &
         oxy_prodP, oxy_anox_resp, nz, geom, rootsP, oxygenP, anoxic_respi_rate, &
         fm, index2m)
    f_out(mprod) = met_anox_resp
    f_out(cprod) = co2_anox_resp
    f_out(oprod) = oxy_anox_resp
 
    ! Root aerenchyma transport
    ! Methane transport
    call root_transport(met_transportP,met_flux,methaneP,nz,geom, &
      met_root_condP,met_alphaP,methane_Ca,por)
    f_out(mplant) = met_flux
    ! CO2 transport
    call root_transport(co2_transportP,co2_flux,co2P,nz,geom, &
      co2_root_condP,co2_alphaP,co2_Ca,por)
    f_out(cplant) = co2_flux
    ! Oxygen transport
    call root_transport(oxy_transportP,oxy_flux,oxygenP,nz,geom, &
      oxy_root_condP,oxy_alphaP,oxygen_Ca,por)
    f_out(oplant) = oxy_flux

    ! Oxygen consumed & CO2 produced in oxic peat respiration
    call oxygen_respiration(oxy_respiP,oxy_respi_tot,co2_respiP,co2_respi_tot, &
         nz,geom,oxygenP,tempP,oxy_alphaP,por)
    f_out(oresp) = oxy_respi_tot
    f_out(cresp) = co2_respi_tot

    ! Methane oxidation
    call methane_oxidation(met_oxidP,met_oxid_tot, &
         co2_oxidP,co2_oxid_tot,oxy_oxidP,oxy_oxid_tot, &
         methaneP,met_alphaP,oxygenP,oxy_alphaP,tempP,nz,geom,por)
    f_out(moxi) = met_oxid_tot
    f_out(coxi) = co2_oxid_tot
    f_out(ooxi) = oxy_oxid_tot


    ! Methane, CO2 & oxygen ebullition
    call ebullition_rate(mebu_rateP,mebu_tot,mebu_out,methaneP,met_alphaP, &
         cebu_rateP,cebu_tot,cebu_out,co2P,co2_alphaP, &
         oebu_rateP,oebu_tot,oebu_out,oxygenP,oxy_alphaP, &
         nz,geom,por)
    f_out(mebuair) = mebu_out
    f_out(mebutot) = mebu_tot
    f_out(cebuair) = cebu_out
    f_out(cebutot) = cebu_tot
    f_out(oebuair) = oebu_out
    f_out(oebutot) = oebu_tot

    methane_d = met_diffusionP + met_transportP + mebu_rateP &
                + met_prodP              + met_oxidP 
    co2_d     = co2_diffusionP + co2_transportP + cebu_rateP &
                + co2_prodP + co2_respiP + co2_oxidP
    oxygen_d  = oxy_diffusionP + oxy_transportP + oebu_rateP &
                + oxy_prodP + oxy_respiP + oxy_oxidP
  end subroutine derivative

  pure function find_max_ebullition_dt() result(max_dt)
    use methane_parameters, only: ebu_hl,time_scale
    real(dp) :: max_dt
    max_dt = time_scale * ebu_hl
  end function find_max_ebullition_dt

  pure subroutine ebullition_rate(mrateP,mebu_tot,mebu_out,methaneP,met_alphaP, &
       crateP,cebu_tot,cebu_out,co2P,co2_alphaP, &
       orateP,oebu_tot,oebu_out,oxygenP,oxy_alphaP, &
       nz,geom,por)

    use methane_parameters, only: ebu_hl,Rgas,atm,ssr

    real(dp), intent(out)      :: mrateP(nz),mebu_tot,mebu_out, &
         crateP(nz),cebu_tot,cebu_out,orateP(nz),oebu_tot,oebu_out         
    real(dp), intent(in)       :: methaneP(nz),met_alphaP(nz), &
         co2P(nz),co2_alphaP(nz),oxygenP(nz),oxy_alphaP(nz),por
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    real(dp)                   :: z_mid(nz)
    real(dp)                   :: ppi(4),dppi(4) ! 4 gases: 1=N2, 2=CH4, 3=CO2, 4=O2
    real(dp)                   :: wtd,z,pressure,fact
    real(dp)                   :: met_Cw,co2_Cw,oxy_Cw
    integer                    :: i,n

    real(dp), parameter        :: dist_atm = 10.3_dp ,& ! 1 atm water height [m]
                                  k = log(2.0_dp)/ebu_hl

    ! if levelsP(1) < wtd, 
    ! then bubbles will transport gas from gas-saturated water layers 
    ! to the deepest air layer. 
    !
    ! if wtd <= levelsP(1), 
    ! then the transport is directly to atmosphere.
    !
    ! rateP:   The rate at which ebullition removes gas at each
    !          layer, [mol s-1].
    ! ebu_tot: The total rate ebullition is going on, even though it may not be
    !          transporting gas to atmosphere, but to the deepest air layer.
    ! ebu_out: The rate of ebullition releasing gas to atmosphere.

    ! Bubble formation occurs if the sum of the partial pressures 
    ! of all gases (N2, CH4, CO2, O2) in the water exceeds the sum 
    ! of the hydrostatic and the atmospheric pressures. N2 in water 
    ! is assumed to be in equilibrium with the atmospheric content,
    ! i.e. p(N2) = 0.78*atmospheric pressure.
    ! The temperature profile is not considered for p(N2).


    mrateP = 0 ; mebu_tot = 0 ; mebu_out = 0
    crateP = 0 ; cebu_tot = 0 ; cebu_out = 0
    orateP = 0 ; oebu_tot = 0 ; oebu_out = 0

    ppi(1) = 0.78 * atm ! partial pressure of N2 in all layers

    z_mid = (geom%levelsP(1:nz) + geom%levelsP(2:nz+1))/2 ! midpoint depths

    if(geom%surfaceQ) then
      wtd = geom%levelsP(1)
      do i = geom%s1,geom%s2
        z = z_mid(i) - wtd
        pressure = atm * (1 + z/dist_atm)
        met_Cw = methaneP(i) / geom%dzP(i) ! concentration in water
        co2_Cw = co2P(i) / geom%dzP(i)
        oxy_Cw = oxygenP(i) / geom%dzP(i)
        ppi(2) = met_Cw * Rgas * tempP(i) / met_alphaP(i) ! CH4 partial pressure
        ppi(3) = co2_Cw * Rgas * tempP(i) / co2_alphaP(i) ! CO2
        ppi(4) = oxy_Cw * Rgas * tempP(i) / oxy_alphaP(i) ! O2
        if (sum(ppi) > ssr * pressure) then
           fact = (sum(ppi) - pressure) / sum(ppi)
           do n = 1,4
              dppi(n) = fact * ppi(n) ! change in each gas [Pa]
           end do
           mrateP(i) = -k * dppi(2) /(Rgas * tempP(i)) * geom%dzP(i)
           crateP(i) = -k * dppi(3) /(Rgas * tempP(i)) * geom%dzP(i)
           orateP(i) = -k * dppi(4) /(Rgas * tempP(i)) * geom%dzP(i)
        end if
      end do
      mebu_out = sum(mrateP) ; mebu_tot = mebu_out
      cebu_out = sum(crateP) ; cebu_tot = cebu_out
      oebu_out = sum(orateP) ; oebu_tot = oebu_out
    end if

    if(geom%waterQ) then
      wtd = geom%levelsP(geom%w1)
      do i = geom%w1,geom%w2
        z = z_mid(i) - wtd
        pressure = atm * (1 + z/dist_atm)
        met_Cw = methaneP(i) / (geom%dzP(i) * por)! concentration in water
        co2_Cw = co2P(i) / (geom%dzP(i) * por)
        oxy_Cw = oxygenP(i) / (geom%dzP(i) * por)
        ppi(2) = met_Cw * Rgas * tempP(i) / met_alphaP(i) ! CH4 partial pressure
        ppi(3) = co2_Cw * Rgas * tempP(i) / co2_alphaP(i) ! CO2
        ppi(4) = oxy_Cw * Rgas * tempP(i) / oxy_alphaP(i) ! O2
        if (sum(ppi) > ssr * pressure) then
           fact = (sum(ppi) - pressure) / sum(ppi)
           do n = 1,4
              dppi(n) = fact * ppi(n) ! change in each gas [Pa]
           end do
           mrateP(i) = -k * dppi(2) /(Rgas * tempP(i)) * geom%dzP(i) * por
           crateP(i) = -k * dppi(3) /(Rgas * tempP(i)) * geom%dzP(i) * por
           orateP(i) = -k * dppi(4) /(Rgas * tempP(i)) * geom%dzP(i) * por
        end if
      end do

      if(geom%airQ) then
        mebu_tot = sum(mrateP)
        mrateP(geom%a2) = - mebu_tot ! bubbles reseased to deepest air layer
        cebu_tot = sum(crateP)
        crateP(geom%a2) = - cebu_tot
        oebu_tot = sum(orateP)
        orateP(geom%a2) = - oebu_tot
        ! ebu_out is already 0, no need to set again
      else
        mebu_out = sum(mrateP) ; mebu_tot = mebu_out
        cebu_out = sum(crateP) ; cebu_tot = cebu_out
        oebu_out = sum(orateP) ; oebu_tot = oebu_out
      end if

    end if
  end subroutine ebullition_rate

  pure function find_max_oxidation_dt(methaneP,met_alphaP,oxygenP,oxy_alphaP, &
    tempP,nz,geom,por) result(max_dt)

    use methane_parameters, only: Rgas,T0,V_oxid_max,delta_E_oxid,K_CH4,K_O2, &
                                  time_scale

    real(dp), intent(in)       :: methaneP(nz),met_alphaP(nz),oxygenP(nz), &
                                  oxy_alphaP(nz),tempP(nz),por
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    integer                    :: i
    real(dp)                   :: met_oxidP(nz),Cw_met,Cw_oxy,V_oxid_0, &
                                  V_oxidP(nz),turnover_rate_met, &
                                  turnover_rate_oxy,turnover_rate,derivative, &
                                  new_dt
    real(dp)                   :: max_dt

    max_dt = 24*3600 ! One day. Just to set something, before calculating actual
                     ! values.
    V_oxid_0 = V_oxid_max
    V_oxidP = V_oxid_0 * exp(delta_E_oxid / Rgas * (1/T0 - 1/tempP))

    met_oxidP = 0
    if (geom%airQ) then
      do i = geom%a1, geom%a2
        Cw_met = met_alphaP(i) * methaneP(i) / (por * geom%dzP(i))
        Cw_oxy = oxy_alphaP(i) * oxygenP(i) / (por * geom%dzP(i))
        met_oxidP(i) = V_oxidP(i) * (Cw_met / (K_CH4 + Cw_met)) * &
                                    (Cw_oxy / (K_O2  + Cw_oxy)) * geom%dzP(i)
      end do
    end if

    if (geom%waterQ) then
      do i = geom%w1, geom%w2
        Cw_met = methaneP(i) / (por * geom%dzP(i))
        Cw_oxy = oxygenP(i) / (por * geom%dzP(i))
        met_oxidP(i) = V_oxidP(i) * (Cw_met / (K_CH4 + Cw_met)) * &
                                    (Cw_oxy / (K_O2  + Cw_oxy)) * geom%dzP(i)
      end do
    end if

    do i = 1,nz
      if (met_oxidP(i) /= 0) then
        derivative = met_oxidP(i)
        turnover_rate_met = derivative / methaneP(i)
        turnover_rate_oxy = 2 * derivative / oxygenP(i)
        turnover_rate = max(turnover_rate_met,turnover_rate_oxy)
        new_dt = time_scale * 1 / turnover_rate
        if (new_dt < max_dt) then
          max_dt = new_dt
        end if
      end if
    end do
  end function find_max_oxidation_dt


  pure subroutine methane_oxidation(met_oxidP,met_oxid_tot, &
       co2_oxidP,co2_oxid_tot,oxy_oxidP,oxy_oxid_tot, &
       methaneP,met_alphaP,oxygenP,oxy_alphaP,tempP,nz,geom,por)

    use methane_parameters, only: Rgas,T0,V_oxid_max,delta_E_oxid,K_CH4,K_O2

    real(dp), intent(out)      :: met_oxidP(nz),met_oxid_tot, &
                                  co2_oxidP(nz),co2_oxid_tot,oxy_oxidP(nz),oxy_oxid_tot
    real(dp), intent(in)       :: methaneP(nz),met_alphaP(nz), &
                                  oxygenP(nz),oxy_alphaP(nz),tempP(nz),por
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    integer                    :: i
    real(dp)                   :: Cw_met,Cw_oxy,V_oxid_0,V_oxidP(nz)

    ! Dual-substrate Michaelis-Menten kinetics, Arah and Stephen (1998) Eq. 8.
    ! Assume a water phase that is in equilibrium with the gas phase,
    ! even for peat-air layers. No oxidation in the surface water above peat.

    V_oxid_0 = V_oxid_max ! Eq. 13 without root distribution term
      ! Cannot use the root distribution term, because our epsilon_r(z) is
      ! non-zero at z=0, so the maximum of epsilon_r(z)/z would be infinity

    ! Eq. 10 in Arah & Stephen 1998
    V_oxidP = V_oxid_0 * exp(delta_E_oxid / Rgas * (1/T0 - 1/tempP))

    met_oxidP = 0
    co2_oxidP = 0
    oxy_oxidP = 0

    if (geom%airQ) then
      do i = geom%a1, geom%a2
        Cw_met = met_alphaP(i) * methaneP(i) / (por * geom%dzP(i))
        Cw_oxy = oxy_alphaP(i) * oxygenP(i) / (por * geom%dzP(i))
        met_oxidP(i) = -V_oxidP(i) * (Cw_met / (K_CH4 + Cw_met)) * &
                                    (Cw_oxy / (K_O2  + Cw_oxy)) * geom%dzP(i)
        co2_oxidP(i) = -met_oxidP(i)
        oxy_oxidP(i) = 2._dp * met_oxidP(i)
      end do
    end if

    if (geom%waterQ) then
      do i = geom%w1, geom%w2
        Cw_met = methaneP(i) / (por * geom%dzP(i))
        Cw_oxy = oxygenP(i) / (por * geom%dzP(i))
        met_oxidP(i) = -V_oxidP(i) * (Cw_met / (K_CH4 + Cw_met)) * &
                                    (Cw_oxy / (K_O2  + Cw_oxy)) * geom%dzP(i)
        co2_oxidP(i) = -met_oxidP(i)
        oxy_oxidP(i) = 2._dp * met_oxidP(i)
      end do
    end if

    met_oxid_tot = sum(met_oxidP)
    co2_oxid_tot = sum(co2_oxidP)
    oxy_oxid_tot = sum(oxy_oxidP)

  end subroutine methane_oxidation

  pure function find_max_respiration_dt(nz,geom,oxygenP,tempP,alphaP,por) &
    result(max_dt)

    use methane_parameters, only: deltaE_R,Rgas,V_R0,K_R,T0,time_scale

    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: oxygenP(nz),tempP(nz),alphaP(nz),por
    real(dp)                   :: max_dt
    real(dp)                   :: respiP(nz),derivative,turnover_rate,new_dt
    integer                    :: i
    real(dp)                   :: Cw,V_R(nz)

    ! If dC/dt = -k*C, then k is the decay constant, or turnover rate, and 1/k
    ! is turnover time. And a safe timestep for Runge-Kutta 4 is something less
    ! than the turnover time, depending how much precision you want. We choose
    ! 0.5. Here we calculate the rate of the chemical reaction used to model
    ! respiration, and then divide by oxygen content, so effectively getting
    ! just the k out of the k*C.


    max_dt = 24*3600 ! One day. Just to set something, before calculating actual
                     ! values.
    V_R(:) = V_R0 * exp(deltaE_R / Rgas * (1/T0 - 1/tempP(:)))
    respiP = 0
    if (geom%airQ) then
      do i = geom%a1, geom%a2
        Cw = oxygenP(i) / (geom%dzP(i) * por) * alphaP(i)
        respiP(i) = V_R(i) * Cw / (K_R + Cw) * geom%dzP(i)
      end do
    end if
    if (geom%waterQ) then
      do i = geom%w1, geom%w2
        Cw = oxygenP(i) / (geom%dzP(i) * por)
        respiP(i) = V_R(i) * Cw / (K_R + Cw) * geom%dzP(i)
      end do
    end if

    do i = 1,nz
      if (respiP(i) /= 0) then
        derivative = respiP(i)
        turnover_rate = derivative / oxygenP(i)
        new_dt = time_scale * 1 / turnover_rate
        if (new_dt < max_dt) then
          max_dt = new_dt
        end if
      end if
    end do
  end function find_max_respiration_dt

  pure subroutine oxygen_respiration(oxy_respiP,oxy_respi_tot,co2_respiP,co2_respi_tot, &
       nz,geom,oxygenP,tempP,alphaP,por)

    use methane_parameters, only: deltaE_R,Rgas,V_R0,K_R,T0

    real(dp), intent(out)      :: oxy_respiP(nz),oxy_respi_tot,co2_respiP(nz),co2_respi_tot
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: oxygenP(nz),tempP(nz),alphaP(nz),por
    integer                    :: i
    real(dp)                   :: Cw,V_R(nz)

    ! This subroutine implements Arah and Stephen (1998) Eq. 7
    ! Assume a water phase that is in equilibrium with the gas phase,
    ! even for peat-air layers.
    !
    ! respiP:    respiration rate in each gridbox [mol s-1]
    ! respi_tot: total respiration rate in each gridbox [mol s-1]
    ! Cw:        water phase oxygen concentration in this layer

    V_R(:) = V_R0 * exp(deltaE_R / Rgas * (1/T0 - 1/tempP(:))) ! A&S Eq. 10
    oxy_respiP = 0
    co2_respiP = 0
    if (geom%airQ) then
      do i = geom%a1, geom%a2
        Cw = oxygenP(i) / (geom%dzP(i) * por) * alphaP(i)
        oxy_respiP(i) = -V_R(i) * Cw / (K_R + Cw) * geom%dzP(i)
        co2_respiP(i) = - oxy_respiP(i) ! peat + O2 -> CO2 + H2O
      end do
    end if
    if (geom%waterQ) then
      do i = geom%w1, geom%w2
        Cw = oxygenP(i) / (geom%dzP(i) * por)
        oxy_respiP(i) = -V_R(i) * Cw / (K_R + Cw) * geom%dzP(i)
        co2_respiP(i) = - oxy_respiP(i) ! peat + O2 -> CO2 + H2O
      end do
    end if
    oxy_respi_tot = sum(oxy_respiP)
    co2_respi_tot = sum(co2_respiP)
  end subroutine oxygen_respiration

  pure subroutine root_transport(root_transportP,gas_flux,gasP,nz,geom, &
    conductanceP,alphaP,gas_Ca,por)
    real(dp), intent(out)      :: root_transportP(nz),gas_flux
    real(dp), intent(in)       :: gasP(nz),conductanceP(nz),alphaP(nz),gas_Ca, &
                                  por
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    integer                    :: i
    real(dp)                   :: cons ! gas gas-phase concentration in layer i
    ! gas_flux: flux from atmosphere

    ! If there is water above the peat, it does not contain roots, so no
    ! transport for surface water layer. (Alhough, we transport gas from roots
    ! to atmosphere, without accounting for the length of the plant "pipes"
    ! above peat surface. So assume plant stems are always above surface water,
    ! and that the aboveground part has no resistance.)

    ! Note: Root transport is computed also in the air-filled peat layers.

    i = 1
    if (geom%surfaceQ .or. geom%dzP(i) == 0) then
      root_transportP(i) = 0
    else
      cons = gasP(i) / (por * geom%dzP(i))
      root_transportP(i) = conductanceP(i) * (gas_Ca - cons)
      ! This will always be an air layer. geom%w1 cannot be 1.
    end if

    do i = 2,nz
      if (geom%dzP(i) == 0) then
        root_transportP(i) = 0
      else if (geom%waterQ .and. geom%w1 <= i) then
        cons = gasP(i) / (alphaP(i) * por * geom%dzP(i))
        root_transportP(i) = conductanceP(i) * (gas_Ca - cons)
      else
        cons = gasP(i) / (por * geom%dzP(i))
        root_transportP(i) = conductanceP(i) * (gas_Ca - cons)
      end if
    end do

    gas_flux = sum(root_transportP)
  end subroutine root_transport


  subroutine methane_production(met_rateP,met_prod_tot, &
       co2_rateP,co2_prod_tot,oxy_rateP,oxy_prod_tot,nz,geom, &
       rootsP,oxygenP,anoxic_respi_rate,fm, &
       index2m)
 
    ! This subroutine takes anoxic_respi_rate
    ! and distributes it along the root distribution in the layers below water=
    ! anoxic conditions. 'fm' fraction, modified by oxygen inhibition, is converted
    ! to methane. (The rest is converted to CO2.)
    ! If the peat depth >2m, i.e. there's peat also below the rooting zone,
    ! we 1) distribute the anoxic_resp_rate only in the rooted layers, 2) record what
    ! was the rate in the bottom rooted layer, 3) allocate 50% of that rate to all 
    ! deeper layers, and 4) re-allocate the rooted layers with what is left.

    use methane_parameters, only: ny ! oxygen inhibition parameter
    real(dp), intent(out)      :: met_rateP(nz),met_prod_tot, &
                                  co2_rateP(nz),co2_prod_tot, &
                                  oxy_rateP(nz),oxy_prod_tot
    integer,  intent(in)       :: nz, index2m
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: rootsP(nz),oxygenP(nz),anoxic_respi_rate, &
                                  fm
    integer  :: i
    real(dp) :: roots_anoxic, max_rateP(nz), respiratio
    real(dp) :: c_below2m, & ! carbon allocated below the rooted zone
                anoxic_respi_rate2    ! new input rate to rooted zone
 
    met_rateP = 0
    co2_rateP = 0
    oxy_rateP = 0
    c_below2m = 0  
    roots_anoxic = sum(rootsP(geom%w1:geom%w2))

    ! first carbon goes only in the anoxic rooted layers
    ! (below them rootsP = 0)
    if (geom%waterQ) then

      do i = geom%w1, geom%w2
        max_rateP(i) = (rootsP(i)/roots_anoxic) * anoxic_respi_rate * fm
        met_rateP(i) = max_rateP(i) / (1._dp + ny*oxygenP(i))
        co2_rateP(i) = (max_rateP(i) - met_rateP(i)) & 
                       + (rootsP(i)/roots_anoxic) * anoxic_respi_rate * (1._dp-fm)
        oxy_rateP(i) = - (max_rateP(i) - met_rateP(i))
      end do
   

    ! if peat depth > 2m, allocating input also in the non-rooted zone
     if (index2m<nz) then
       do i = index2m+1,nz
        max_rateP(i) = 0.5_dp*met_rateP(index2m)*(geom%dzP(i)/geom%dzP(index2m))
        met_rateP(i) = max_rateP(i)
        co2_rateP(i) = 0.5_dp*co2_rateP(index2m)*(geom%dzP(i)/geom%dzP(index2m))
        oxy_rateP(i) = - (max_rateP(i) - met_rateP(i))
        c_below2m    = c_below2m + met_rateP(i) + co2_rateP(i)
       end do

        anoxic_respi_rate2 = anoxic_respi_rate-c_below2m ! rate2 is allocated to rooted layers
        if(anoxic_respi_rate.eq.0.) then
        respiratio = 0.
        else
        respiratio = anoxic_respi_rate2/anoxic_respi_rate
        end if
                  
       do i=geom%w1, index2m
         max_rateP(i) = respiratio * max_rateP(i)
         met_rateP(i) = respiratio * met_rateP(i)
         co2_rateP(i) = respiratio * co2_rateP(i)
         oxy_rateP(i) = respiratio * oxy_rateP(i)
       end do
     end if
    end if

    met_prod_tot = sum(met_rateP)
    co2_prod_tot = sum(co2_rateP)
    oxy_prod_tot = sum(oxy_rateP)

  end subroutine methane_production

  pure subroutine diffusion(dgasdtP,flux,gasP,nz,geom,por,C_air,DcondP,alphaP)
    real(dp), intent(out)      :: dgasdtP(nz),flux
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: gasP(nz),por,C_air,DcondP(nz),alphaP(nz)
    real(dp)                   :: c1,c2,c3
    integer                    :: i
    ! flux: flux from atmosphere

    dgasdtP = 0
    if (geom%surfaceQ) then
      c1 = C_air*alphaP(1) ! corresponding concentration in water
      c2 = gasP(1)/geom%dzP(1) ! above peat water, no peat porosity
      c3 = gasP(2)/(por*geom%dzP(2))
      flux = DcondP(1)*(c1 - c2)
      dgasdtP(1) = flux - DcondP(2)*(c2 - c3)
      do i = 2,geom%w2 - 1
        if (i == 2) then
          c1 = gasP(i-1)/(geom%dzP(i-1)) ! layer is water above surface
        else
          c1 = gasP(i-1)/(por*geom%dzP(i-1))
        end if
        c2 = gasP( i )/(por*geom%dzP( i ))
        c3 = gasP(i+1)/(por*geom%dzP(i+1))
        dgasdtP(i) = DcondP(i)*(c1 - c2) - DcondP(i+1)*(c2 - c3)
      end do
      c1 = gasP(geom%w2-1)/(por*geom%dzP(geom%w2-1))
      c2 = gasP(geom%w2)/(por*geom%dzP(geom%w2))
      dgasdtP(geom%w2) = DcondP(geom%w2)*(c1 - c2)
      ! we are done
    else if (geom%airQ) then
      c1 = C_air
      c2 = gasP(1)/(por*geom%dzP(1)) ! above peat water, no peat porosity
      flux = DcondP(1)*(c1 - c2)
      dgasdtP(1) = flux
      if (geom%a2 >= 2) then ! also at least layer 2 is air, safe to go on
        c3 = gasP(2)/(por*geom%dzP(2))
        dgasdtP(1) = dgasdtP(1) - DcondP(2)*(c2 - c3)
        if (geom%a2 >= 3) then ! also at least layer 3 is air, safe to go on
          do i = 2, geom%a2 - 1
            c1 = gasP(i-1)/(por*geom%dzP(i-1))
            c2 = gasP( i )/(por*geom%dzP( i ))
            c3 = gasP(i+1)/(por*geom%dzP(i+1))
            dgasdtP(i) = DcondP(i)*(c1 - c2) - DcondP(i+1)*(c2 - c3)
          end do
        end if
        c1 = gasP(geom%a2-1)/(por*geom%dzP(geom%a2-1))
        c2 = gasP(geom%a2)/(por*geom%dzP(geom%a2))
        dgasdtP(geom%a2) = DcondP(geom%a2)*(c1 - c2)
      end if
      if (geom%waterQ) then
        ! first deal with air-water interface
        c1 = alphaP(geom%a2)*gasP(geom%a2)/(por*geom%dzP(geom%a2))
        c2 = gasP(geom%w1)/(por*geom%dzP(geom%w1))
        dgasdtP(geom%a2) = dgasdtP(geom%a2) - DcondP(geom%w1)*(c1 - c2)
        if (geom%w2 == geom%w1) then
          dgasdtP(geom%w1) = DcondP(geom%w1)*(c1 - c2)
          ! we are done
        else if (geom%w2 > geom%w1) then
          c3 = gasP(geom%w1+1)/(por*geom%dzP(geom%w1+1))
          dgasdtP(geom%w1) = &
            DcondP(geom%w1)*(c1 - c2) - DcondP(geom%w1+1)*(c2 - c3)
          if (geom%w2 > geom%w1 + 1) then
            do i = geom%w1 + 1, geom%w2 - 1
              c1 = gasP(i-1)/(por*geom%dzP(i-1))
              c2 = gasP( i )/(por*geom%dzP( i ))
              c3 = gasP(i+1)/(por*geom%dzP(i+1))
              dgasdtP(i) = DcondP(i)*(c1 - c2) - DcondP(i+1)*(c2 - c3)
            end do
          end if
          c1 = gasP(geom%w2-1)/(por*geom%dzP(geom%w2-1))
          c2 = gasP(geom%w2)/(por*geom%dzP(geom%w2))
          dgasdtP(geom%w2) = DcondP(geom%w2)*(c1 - c2)
          ! we are done
        end if
      end if
    else ! so water is exactly at peat surface level
      ! dzP(1) is 0
      ! here we assume nz >= 3, pretty much the only place where we assume this
      c1 = C_air*alphaP(2) ! corresponding concentration in water
      c2 = gasP(2)/(por*geom%dzP(2))
      c3 = gasP(3)/(por*geom%dzP(3))
      flux = DcondP(2)*(c1 - c2)
      dgasdtP(2) = flux - DcondP(3)*(c2 - c3)
      if (geom%w2 > geom%w1 + 1) then
        do i = geom%w1 + 1, geom%w2 - 1
          c1 = gasP(i-1)/(por*geom%dzP(i-1))
          c2 = gasP( i )/(por*geom%dzP( i ))
          c3 = gasP(i+1)/(por*geom%dzP(i+1))
          dgasdtP(i) = DcondP(i)*(c1 - c2) - DcondP(i+1)*(c2 - c3)
        end do
      end if
      c1 = gasP(geom%w2-1)/(por*geom%dzP(geom%w2-1))
      c2 = gasP(geom%w2)/(por*geom%dzP(geom%w2))
      dgasdtP(geom%w2) = DcondP(geom%w2)*(c1 - c2)
    end if
  end subroutine diffusion

  pure function find_max_diffusion_dt(nz,geom,DcondP) result(max_dt)
    ! For time stepping stability, apply Von Neumann stability condition:
    ! dt <= 1/2 * dx/cond, where dx and centerpoint-to-centerpoit donductances
    ! are arrays, and the smallest ratio needs to be found.

    use methane_parameters, only: time_scale

    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: DcondP(nz)
    real(dp)                   :: max_dt
    integer                    :: i
    real(dp)                   :: valP(nz)
    ! DcondP: array (nz elements) of diffusion conductances between boxes in
    ! the peat gas profile. DcondP(1) is the conductance from box 1 to
    ! atmosphere. DconP(i) is the conductance between boxes i-1 and i. In case
    ! of geom%dz(i) = 0 for layer i, both DcondP(i-1) and Dcond(i) are assumed
    ! to give the same conductance from layer i+1 to layer i+1.

    do i = 1,nz-1
      if (geom%dzP(i) == 0) then
        valP(i) = huge(1.0_dp)
      else
        valP(i) = 1 / (max(DcondP(i),DcondP(i+1)) / geom%dzP(i))
      end if
    end do
    if (geom%dzP(nz) == 0) then
      valP(nz) = huge(1.0_dp)
    else
      valP(nz) = 1 / (DcondP(nz) / geom%dzP(nz))
    end if

    max_dt = time_scale * minval(valP)
  end function find_max_diffusion_dt

  pure function diffusion_conductances(nz,geom,DaP,Dwp,alphaP,peat_coeff_w) &
                result(DcondP)
    ! prepare diffusion conductances between box centerpoints. DcondP(1) for
    ! the dzP(1)/2 distance from the top box center to atmosphere.
    real(dp)                   :: DcondP(nz)
    integer, intent(in)        :: nz
    real(dp), intent(in)       :: peat_coeff_w
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: DaP(nz),DwP(nz),alphaP(nz)
    integer                    :: i,i1,i2
    real(dp)                   :: r1,r2

    ! The conductance over air-water interface, when used above in 'diffusion',
    ! assumes concentrations in water are used in the calculation then.

    if (geom%surfaceQ) then
      DcondP(1) = (1/peat_coeff_w) * DwP(1) / (0.5 * geom%dzP(1))
      r1 = (0.5*geom%dzP(1)) / DwP(1) / (1/peat_coeff_w)
      r2 = (0.5*geom%dzP(2)) / DwP(1)
      DcondP(2) = 1/(r1 + r2)
      do i = geom%w1 + 1, geom%w2
        r1 = (0.5*geom%dzP(i-1)) / DwP(i-1)
        r2 = (0.5*geom%dzP(i)) / DwP(i)
        DcondP(i) = 1/(r1 + r2)
      end do
    else if (geom%airQ) then
      DcondP(1) = DaP(1) / (0.5 * geom%dzP(1))
      do i = geom%a1 + 1, geom%a2
        r1 = (0.5*geom%dzP(i-1)) / DaP(i-1)
        r2 = (0.5*geom%dzP(i)) / DaP(i)
        DcondP(i) = 1/(r1 + r2)
      end do
      if (geom%waterQ) then
        i1 = geom%a2
        i2 = geom%w1
        r1 = 0.5*geom%dzP(i2) / DwP(i2)
        r2 = 0.5*geom%dzP(i1)*alphaP(i2) / DaP(i1)
        DcondP(i1+1) = 1/(r1+r2)
        if (i1 + 1 < i2) then
          DcondP(i2) = DcondP(i1+1)
        end if
        do i = geom%w1 + 1, geom%w2
          r1 = (0.5*geom%dzP(i-1)) / DwP(i-1)
          r2 = (0.5*geom%dzP(i)) / DwP(i)
          DcondP(i) = 1/(r1 + r2)
        end do
      end if
    else ! so water is at peat surface level
      DcondP(1) = 0 ! dzP(1) must be 0, so this should never be used
      i = geom%w1
      DcondP(i) = DwP(i) / (0.5 * geom%dzP(i))
      do i = geom%w1 + 1, geom%w2
        r1 = (0.5*geom%dzP(i-1)) / DwP(i-1)
        r2 = (0.5*geom%dzP(i)) / DwP(i)
        DcondP(i) = 1/(r1 + r2)
      end do
    end if
  end function diffusion_conductances

  pure function root_conductances(nz,geom,DaP,rootsP,root_km,root_mass, &
                                   root_tortuosity) &
                result(conductanceP)
    integer, intent(in)        :: nz
    type(geometry), intent(in) :: geom
    real(dp), intent(in)       :: DaP(nz),rootsP(nz),root_km,root_mass
    real(dp)                   :: conductanceP(nz)
    real(dp)                   :: resistanceP(nz)
    integer                    :: i
    real(dp), intent(in)       :: root_tortuosity
    ! DaP: air diffusion coefficients
    ! resistanceP: box-to-box resistances

    ! - We assume that surface water does not resist aerenchyma transport
    ! - Special care to deal with the possible dz=0 layer

    i = 1
    if (geom%surfaceQ .or. geom%dzP(i) == 0) then
      resistanceP(i) = 0
    else
      resistanceP(i) = 0.5*geom%dzP(i)/DaP(i)
    end if
    do i = 2,nz
      if (geom%dzP(i) == 0) then
        resistanceP(i) = 0.5*geom%dzP(i-1)/DaP(i-1)
      else
        if (resistanceP(i-1) == 0) then
          resistanceP(i) = 0.5*geom%dzP(i)/DaP(i)
        else
          resistanceP(i) = 0.5*geom%dzP(i)/DaP(i) + 0.5*geom%dzP(i-1)/DaP(i-1)
        end if
      end if
    end do

    i = 1
    if (resistanceP(i) == 0) then
      conductanceP(i) = 0 ! should never be used, just have put something
    else
      conductanceP(i) = 1.0 / resistanceP(i)
    end if
    do i = 2,nz
      conductanceP(i) = 1.0 / sum(resistanceP(1:i))
    end do

    conductanceP = conductanceP * rootsP * root_km * root_mass / root_tortuosity
  end function root_conductances

  pure elemental function methane_D_air(temperature, peat_coeff_a) result(Da)
    ! methane diffusion coefficient for medium consisting of xx% peat,
    ! yy% air, depending on temperature
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: Da
    real(dp), parameter  :: Da_273 = 1.9e-5_dp ! [m2 s-1]
    real(dp), intent(in) :: peat_coeff_a

    ! Arah and Stephen (1998), Eq. 11
    Da = peat_coeff_a * Da_273 * (temperature/273._dp)**1.82_dp
  end function methane_D_air

  pure elemental function methane_D_water(temperature, peat_coeff_w) result(Dw)
    ! methane diffusion coefficient for medium consisting of xx% peat,
    ! yy% water, depending on temperature
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: Dw
    real(dp), parameter  :: Dw_298 = 1.5e-9_dp ! [m2 s-1]
    real(dp), intent(in) :: peat_coeff_w

    ! Arah and Stephen (1998), Eq. 11
    Dw = peat_coeff_w * Dw_298 * (temperature/298._dp)
    ! if (temperature < 273.15) Dw = 0
  end function methane_D_water

  pure elemental function methane_solubility(temperature, Rgas) result(alpha)
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: alpha ! [mol(CH4,water) mol(CH4,air)]
    real(dp), parameter  :: a = 1.3e-3_dp, b = 1700._dp, c = 298.0_dp
    real(dp), intent(in) :: Rgas
    real(dp) :: Hcp ! [mol(CH4) m(H2O)-3 Pa-1]
    ! Tang et al. (2010), eq. A4
    ! see also: Sander (2015), doi:10.5194/acp-15-4399-2015.
    Hcp = a * exp(b * (1._dp/temperature - 1._dp/c) ) * 9.86923266716e-3_dp
    alpha = Rgas * temperature * Hcp
  end function methane_solubility

  pure elemental function co2_D_air(temperature, peat_coeff_a) result(Da)
    ! CO2 diffusion coefficient for medium consisting of xx% peat,
    ! yy% air, depending on temperature
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: Da
    real(dp), parameter  :: Da_273 = 1.47e-5_dp ! [m2 s-1]
    real(dp), intent(in) :: peat_coeff_a

    ! Tang et al. (2010), eq. A8
    Da = peat_coeff_a * Da_273 * (temperature/273.15_dp)**1.792_dp
  end function co2_D_air

  pure elemental function co2_D_water(temperature, peat_coeff_w) result(Dw)
    ! CO2 diffusion coefficient for medium consisting of xx% peat,
    ! yy% water, depending on temperature
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: Dw
    real(dp), parameter  :: Dw_298 = 1.81e-6_dp ! [m2 s-1]
    real(dp), intent(in) :: peat_coeff_w

    ! Tang et al. (2010), eq. A12
    Dw = peat_coeff_w * Dw_298 * exp(-2032.6_dp/temperature)
  end function co2_D_water

  pure elemental function co2_solubility(temperature,Rgas) result(alpha)
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: alpha ! [mol(CO2,water) mol(CO2,air)]
    real(dp), parameter  :: a = 3.4e-2_dp, b = 2400._dp, c = 298.0_dp
    real(dp), intent(in) :: Rgas
    real(dp) :: Hcp ! [mol(CO2) m(H2O)-3 Pa-1]
    ! Tang et al. (2010), eq. A3
    ! see also: Sander (2015), doi:10.5194/acp-15-4399-2015.
    Hcp = a * exp(b * (1._dp/temperature - 1._dp/c) ) * 9.86923266716e-3_dp
    alpha = Rgas * temperature * Hcp
  end function co2_solubility

  pure elemental function oxygen_D_air(temperature,peat_coeff_a) result(Da)
    ! oxygen diffusion coefficient for medium consisting of xx% peat,
    ! yy% air, depending on temperature
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: Da
    real(dp), parameter  :: Da_273 = 1.8e-5_dp ! [m2 s-1]
    real(dp), intent(in) :: peat_coeff_a

    ! Arah and Stephen (1998), Eq. 11
    Da = peat_coeff_a * Da_273 * (temperature/273._dp)**1.82_dp
  end function oxygen_D_air

  pure elemental function oxygen_D_water(temperature,peat_coeff_w) result(Dw)
    ! methane diffusion coefficient for medium consisting of xx% peat,
    ! yy% water, depending on temperature
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: Dw
    real(dp), parameter  :: Dw_298 = 2.4e-9_dp ! [m2 s-1]
    real(dp), intent(in) :: peat_coeff_w

    ! Arah and Stephen (1998), Eq. 11
    Dw = peat_coeff_w * Dw_298 * (temperature/298._dp)
    ! if (temperature < 273.15) Dw = 0
  end function oxygen_D_water

  pure elemental function oxygen_solubility(temperature, Rgas) result(alpha)
    real(dp), intent(in) :: temperature ! [K]
    real(dp)             :: alpha ! [mol(O2,water) mol(O2,air)]
    real(dp), parameter  :: a = 1.3e-3_dp, b = 1500._dp, c = 298.0_dp
    real(dp), intent(in) :: Rgas
    real(dp) :: Hcp ! [mol(O2) m(H2O)-3 Pa-1]
    ! Tang et al. (2010), eq. A2
    ! see also: Sander (2015), doi:10.5194/acp-15-4399-2015.
    Hcp = a * exp(b * (1._dp/temperature - 1._dp/c) ) * 9.86923266716e-3_dp
    alpha = Rgas * temperature * Hcp
  end function oxygen_solubility

  pure function calculate_root_fraction(lambda_root,nz,levelsP,peat_surface,&
                  index2m) &
                result(rootsP)
    real(dp), intent(in)  :: lambda_root,levelsP(nz+1),peat_surface
    integer,  intent(in)  :: nz, index2m
    real(dp)              :: rootsP(nz),top,bottom
    integer               :: i,i_start
    ! roots are distributed according to:
    !     root_fraction(z) = C exp(-z/lambda_root)
    ! where C is a constant to normalize the integral of the distribution to 1.
    ! The integral from a to b of exp(-z/lambda_root) is
    !     (exp(-a/lambda_root) - exp(-b/lambda_root)) * lambda_root
    ! and then we just normalize numerically.
    !
    ! The z's are positive numbers here, even though we are underground.

    i_start = 1
    if (levelsP(1) < peat_surface) then ! layer is is water above surface
      rootsP(1) = 0
      i_start = 2
    end if
 
   ! The max rooting depth is, according to literature, 2 m.

    if (index2m == nz) then
      do i = i_start,nz
       top    = levelsP(i)   - peat_surface
       bottom = levelsP(i+1) - peat_surface
       rootsP(i) = &
         (exp(-top/lambda_root) - exp(-bottom/lambda_root)) * lambda_root
      end do
    else
      do i = i_start, index2m
       top = levelsP(i) - peat_surface
       bottom = levelsP(i+1) - peat_surface
       rootsP(i) = &
         (exp(-top/lambda_root) - exp(-bottom/lambda_root)) * lambda_root
      end do
     rootsP(index2m+1:nz) = 0.0
   end if

    rootsP = rootsP / sum(rootsP)

  end function calculate_root_fraction

end program methane_main
