module methane_geometry

  use methane_parameters, only: dp, min_dz

  implicit none

  private
  public geometry,new_geom,gas_move_wtd,find_wtd_index

  type geometry
    integer :: s1,s2,a1,a2,w1,w2
    logical :: surfaceQ,airQ,waterQ
    real(dp), allocatable :: levelsP(:),dzP(:)

    ! Terminology: Levels are z-coordinates, and layers are between levels. The
    ! array fixed_levelsP (in the main code) defines the default levels we want
    ! to use for the numerics. Levels can be of different thickness. Then we
    ! have one extra level to track the location of the WTD. In this way,
    ! whatever the WTD, we can always arrange a level at the WTD, so all the
    ! layers (boxes) will always be totally air (above WTD) or water (below
    ! WTD), which simplifies the main transport code.
    !
    ! levelsP: z-coordinates of the levels, after the extra, moving, level for
    ! the WTD has been added.
    !
    ! dzP: thicknesses of the layers between the levels.
    !
    ! surfaceQ : if surface water layer is present?
    ! airQ     : if air-peat layer is present?
    ! waterQ   : if water-peat layer is present?
    !
    ! s1: Index of the first box in levelsP, where surface water is. If this is
    !     0, then there is no surface water and s2 has no meaning. So this is
    !     either 0 or 1.
    ! s2: Last box of surface water.
    ! a1: First box of peat-air. If 0, there is no peat-air, a2 has no meaning.
    ! a2: Last box of peat-air.
    ! w1: First box of peat-water. If 0, no peat-water, w2 has no meaning.
    ! w2: Last box of peat-water.
    !
    ! So, compared to fixed_levelsP, levelsP here usually has one extra layer.
    ! This extra layer is for following the WTD. If WTD is negative, so above
    ! peat surface, then this extra layer is the water above peat surface. (If
    ! WTD were to have large negative values, for numerical reasons it might be
    ! needed to add several surface water layers, but currently this is not
    ! implemented, because then we would need to add boxes to all the depth
    ! profile arrays.)
    !
    ! But, if WTD is too close to one of the fixed levels in fixed_levelsP
    ! then, for reasons related to numerical solution of the diffusion
    ! equation, we would have one very small dz, leading to the need for a very
    ! small dt. For this reason, if WTD is closer than min_dz (currently 0.01 m
    ! i.e. 1 cm) to a fixed level, we "force" it to exactly to the level
    ! instead. This then causes the extra layer to be of 0 thickness.
    !
    ! So, if s1 /= 0, then a1 = 0 and w1 = 2 and w2 = nz.
    !
    ! If WTD is nearer than min_dz to the peat surface (assumed to be at
    ! fixed_levelsP(1)), then s1 = 0, a1 = 0, w1 = 2.
    !
    ! If WTD is nearer than min_dz to some other level (except the last), then
    ! s1 = 0, a1 = 1, w1 = a2+2, and the dz=0 layer is the box a2+1.
    !
    ! If WTD is larger than fixed_levelsP(nz) (the last level), then s1 = 0,
    ! a1 = 1, a2 = nz-1, w1 = 0, and the dz=0 layer is the box nz.
    !
    ! If WTD is nicely between two levels, then s1 = 0, a1 = 1, w1 = a2+1.
  end type geometry

contains

  subroutine gas_move_wtd(gas2P,flux,nz,wtd1,wtd2,gas1P, &
    CwaterP,alphaP,fixed_levelsP)
    ! Modify the gas content of boxes in 'gas2P', when Water Table Depth moves
    ! from 'wtd1' to 'wtd2'. Part of the gas may need to be released into
    ! atmosphere, this is returned in 'flux'.
    real(dp), intent(out)      :: gas2P(nz),flux
    integer, intent(in)        :: nz
    real(dp), intent(in)       :: wtd1,wtd2,gas1P(nz),CwaterP(nz), &
                                  alphaP(nz),fixed_levelsP(nz)

    if (wtd1 == wtd2) then
      gas2P = gas1P
      flux = 0
    else if (wtd1 > wtd2) then ! wtd1 is deeper than wtd2
      call gas_rise_wtd(gas2P,flux,nz,wtd1,wtd2,gas1P,alphaP,fixed_levelsP)
    else
      call gas_lower_wtd(gas2P,flux,nz,wtd1,wtd2,gas1P,fixed_levelsP)
    end if
  end subroutine gas_move_wtd

  subroutine gas_lower_wtd(gas_outP,flux,nz,wtd1,wtd2,gas_inP,fixed_levelsP)
    real(dp), intent(out) :: gas_outP(nz),flux
    integer, intent(in)   :: nz
    real(dp), intent(in)  :: wtd1,wtd2,gas_inP(nz),fixed_levelsP(nz)
    integer               :: i1
    real(dp)              :: wtd_eff,wtd_tmp,gas_tmpP(nz),gone_dz,gas_layer, &
                             top_limit
    type(geometry)        :: geom

    allocate(geom%levelsP(nz+1),geom%dzP(nz))
    call new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd1)
    if (wtd_eff /= wtd1) write(*,*) 'Caller did this wrong!'

    wtd_tmp = wtd1
    gas_tmpP = gas_inP
    flux = 0

    do while (wtd_tmp < wtd2)
      if (wtd_tmp < fixed_levelsP(1)) then ! water is above peat top surface
        top_limit = fixed_levelsP(1) - min_dz
        if (wtd2 <= top_limit) then
          gone_dz = wtd2 - wtd_tmp
          gas_layer = gone_dz / geom%dzP(1) * gas_inP(1)
          gas_tmpP(1) = gas_tmpP(1) - gas_layer
          flux = gas_layer
          wtd_tmp = wtd2
          exit ! now we are done
        else ! whole above-peat water layer will disappear
          gas_layer = gas_inP(1)
          gas_tmpP(1) = 0
          flux = gas_layer
          wtd_tmp = fixed_levelsP(1)
          call new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd_tmp)
        end if
      else
        i1 = find_wtd_index(geom%levelsP,wtd_tmp)
        if (geom%dzP(i1+1) == 0) i1 = i1 + 1
        if (wtd2 <= geom%levelsP(i1+2) - min_dz) then
          ! Lower level inside same background layer, this is last step
          gone_dz = wtd2 - wtd_tmp
          gas_layer = gone_dz / geom%dzP(i1+1) * gas_inP(i1+1)
          gas_tmpP(i1) = gas_tmpP(i1) + gas_layer
          gas_tmpP(i1+1) = gas_tmpP(i1+1) - gas_layer
          wtd_tmp = wtd2
          exit
        else
          ! Lower level until background layer bottom, then next iteration
          gas_layer = gas_inP(i1+1)
          gas_tmpP(i1) = gas_tmpP(i1) + gas_layer
          gas_tmpP(i1+1) = 0
          wtd_tmp = geom%levelsP(i1+2)
          call new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd_tmp)
        end if
      end if
    end do

    gas_outP = gas_tmpP
    ! and 'flux' was already set above
  end subroutine gas_lower_wtd

  subroutine gas_rise_wtd(gas_outP,flux,nz,wtd1,wtd2,gas_inP,alphaP, &
                         fixed_levelsP)
    real(dp), intent(out) :: gas_outP(nz),flux
    integer, intent(in)   :: nz
    real(dp), intent(in)  :: wtd1,wtd2,gas_inP(nz),alphaP(nz),fixed_levelsP(nz)
    integer               :: i1
    real(dp)              :: top_limit,gone_dz,gas_layer,gas_stays,gas_moves, &
                             wtd_tmp,wtd_tmp_new,wtd_eff,gas_tmpP(nz),alpha, &
                             gas_moves_total
    type(geometry)        :: geom

    ! gas_stays: Stays at the physical location it was (not at the cell of same
    ! index)
    ! gas_moves: moves up to stay above water level
    ! alphaP: Solubility of this gas at each level. (Solubility depends only on
    ! temperature, but let the caller do the calculation.)

    allocate(geom%levelsP(nz+1),geom%dzP(nz))
    call new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd1)
    if (wtd_eff /= wtd1) write(*,*) 'Caller did this wrong!'

    gas_moves_total = 0
    wtd_tmp = wtd1
    gas_tmpP = gas_inP

    do while (wtd2 < wtd_tmp)
      i1 = find_wtd_index(geom%levelsP,wtd_tmp)
      if (i1 > 0) then ! gas1P(i1) is air cell
        top_limit = geom%levelsP(i1) + min_dz
        if (top_limit <= wtd2) then
          ! Rise level inside same background layer, this is last step
          ! 
          ! 'gas2P(i1)' is still air. Its volume decreases. The water that
          ! fills the air space, gets a concentration of gas that is in
          ! solubility equilibrium (Henry's law) with the air-space that was
          ! filled with water. If dz(i1) is small, gas concentration can be
          ! momentarily high, but diffusion should even this out in an hour or
          ! two. If WTD moves a lot, this can be funny, but still physically
          ! somewhat accurate if we think the WTD as a piston that moved
          ! awfully fast. It would be better if the user did not move the WTD
          ! too many cm per hour.
          gone_dz = wtd_tmp - wtd2 ! this much air is filled with water
          alpha = min(alphaP(i1), 1._dp)
          gas_layer = gone_dz / geom%dzP(i1) * gas_tmpP(i1)
          gas_stays = gas_layer * alpha
          gas_moves = gas_layer - gas_stays
          gas_tmpP(i1+1) = gas_tmpP(i1+1) + gas_stays ! now dissolved in water
          gas_tmpP(i1) = gas_tmpP(i1) - gas_layer
          gas_moves_total = gas_moves_total + gas_moves
          wtd_tmp = wtd2
          exit ! now we are done
        else
          ! rise until 'geom1%levelsP(i1)', then new iteration
          wtd_tmp_new = geom%levelsP(i1)
          gone_dz = wtd_tmp - wtd_tmp_new ! this filled with water at this step
          alpha = min(alphaP(i1), 1._dp)
          gas_layer = gone_dz / geom%dzP(i1) * gas_tmpP(i1)
          gas_stays = gas_layer * alpha
          gas_moves = gas_layer - gas_stays
          gas_tmpP(i1+1) = gas_tmpP(i1+1) + gas_stays
          gas_tmpP(i1) = 0 ! dz(i1) is now zero
          gas_moves_total = gas_moves_total + gas_moves
          wtd_tmp = wtd_tmp_new
          call new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd_tmp)
          if (wtd_eff /= wtd_tmp) write(*,*) 'This is not good.'
        end if
      else ! gasP(1) is water cell

        ! Case 1: First, WTD is at peat surface level, then rises. So in the
        ! beginning dz(1) = 0 and then levelsP(1) rises (decreases in value).
        ! The initial dz=0 layer will be the above-peat water layer, but it
        ! just gets gas content of 0, since the added water (at least for now)
        ! does not carry any gas.

        ! Case 2: WTD is already above levelsP(1), and rises more. Again the
        ! added water contain no gas, so while the layer thickens, it's gas
        ! concentration does not change.

        ! In both cases, no operation on gas_tmpP is needed.
        wtd_tmp = wtd2
        exit
      end if
    end do

    gas_outP = gas_tmpP

    i1 = find_wtd_index(geom%levelsP,wtd_tmp)
    if (i1 > 0) then ! gasP(i1) is air cell 
      gas_outP(i1) = gas_outP(i1) + gas_moves_total
      flux = 0
    else
      flux = gas_moves_total
    end if

  end subroutine gas_rise_wtd
 
  !pure
  subroutine new_geom(geom,wtd_eff,nz,fixed_levelsP,wtd_in)
    ! Construct new geometry (levels and coordinates), given new wtd ('wtd_in')
    ! and the constant 'fixed_levelsP'. If 'wtd_in' is closer than 1 cm (i.e.
    ! 'min_dz') to a level in 'fixed_levelsP', then we just force it to be
    ! exactly that closest level in 'fixed_levelsP'. This way there will never
    ! be a layer thinner than 1 cm.
 
    type(geometry), intent(inout) :: geom
    real(dp), intent(out)         :: wtd_eff
    integer, intent(in)           :: nz
      ! geom%levelsP has nz+1 levels, nz layers
      ! fixed_levelsP has nz levels, nz-1 layers
    real(dp), intent(in)          :: fixed_levelsP(nz)
    real(dp), intent(in)          :: wtd_in
    integer :: i_wtd
    real(dp) :: z1,z2

    z1 = fixed_levelsP(1) - min_dz
    z2 = fixed_levelsP(1) + min_dz


    if (wtd_in <= z1) then
      ! WTD is above surface, the extra box is this box, being water. Also,
      ! usually fixed_levelsP(1) is 0, but let's try to be general.
      geom%s1 = 1 ; geom%a1 = 0 ; geom%w1 = 2
      geom%s2 = 1 ; geom%a2 = 0 ; geom%w2 = nz      
      geom%surfaceQ = .true. ; geom%airQ = .false. ; geom%waterQ = .true.
      geom%levelsP(1) = wtd_in
      geom%levelsP(2:nz+1) = fixed_levelsP(1:nz)
      wtd_eff = wtd_in ! no need to change
    else if ( (z1 < wtd_in) .and. (wtd_in < z2) ) then
      ! WTD is at peat surface level
      geom%s1 = 0 ; geom%a1 = 0 ; geom%w1 = 2
      geom%s2 = 0 ; geom%a2 = 0 ; geom%w2 = nz
      geom%surfaceQ = .false. ; geom%airQ = .false. ; geom%waterQ = .true.
      geom%levelsP(1) = fixed_levelsP(1)
      geom%levelsP(2:nz+1) = fixed_levelsP(1:nz)
      wtd_eff = fixed_levelsP(1)
    else
      ! WTD is below peat surface
      i_wtd = find_wtd_index(fixed_levelsP,wtd_in)
      z2 = fixed_levelsP(nz) - min_dz
      if (i_wtd == 0) then
        write(*,*) 'new_geom: something is very wrong.'
      else if (z2 < wtd_in) then
        ! WTD is near, at, or below lowest level
        geom%s1 = 0 ; geom%a1 = 1    ; geom%w1 = 0
        geom%s2 = 0 ; geom%a2 = nz-1 ; geom%w2 = 0
        geom%surfaceQ = .false. ; geom%airQ = .true. ; geom%waterQ = .false.
        geom%levelsP(1:nz) = fixed_levelsP(1:nz)
        geom%levelsP(nz+1) = fixed_levelsP(nz)
        wtd_eff = fixed_levelsP(nz)
      else
        z1 = fixed_levelsP(i_wtd) + min_dz
        z2 = fixed_levelsP(i_wtd + 1) - min_dz
        if (wtd_in < z1) then
          ! WTD is very close to fixed_levelsP(i_wtd)
          geom%s1 = 0 ; geom%a1 = 1       ; geom%w1 = i_wtd+1
          geom%s2 = 0 ; geom%a2 = i_wtd-1 ; geom%w2 = nz
          geom%surfaceQ = .false. ; geom%airQ = .true. ; geom%waterQ = .true.
          geom%levelsP(1:i_wtd) = fixed_levelsP(1:i_wtd)
          geom%levelsP(i_wtd+1) = fixed_levelsP(i_wtd)
          geom%levelsP(i_wtd+2:nz+1) = fixed_levelsP(i_wtd+1:nz)
          wtd_eff = fixed_levelsP(i_wtd)
        else if (z2 < wtd_in) then
          ! WTD is very close to fixed_levelsP(i_wtd+1)
          geom%s1 = 0 ; geom%a1 = 1     ; geom%w1 = i_wtd+2
          geom%s2 = 0 ; geom%a2 = i_wtd ; geom%w2 = nz
          geom%surfaceQ = .false. ; geom%airQ = .true. ; geom%waterQ = .true.
          geom%levelsP(1:i_wtd+1) = fixed_levelsP(1:i_wtd+1)
          geom%levelsP(i_wtd+2) = fixed_levelsP(i_wtd+1)
          geom%levelsP(i_wtd+3:nz+1) = fixed_levelsP(i_wtd+2:nz)
          wtd_eff = fixed_levelsP(i_wtd+1)
        else
          ! WTD is nicely between fixed_levelsP(i_wtd) and fixed_levelsP(i_wtd+1)
          geom%s1 = 0 ; geom%a1 = 1     ; geom%w1 = i_wtd+1
          geom%s2 = 0 ; geom%a2 = i_wtd ; geom%w2 = nz
          geom%surfaceQ = .false. ; geom%airQ = .true. ; geom%waterQ = .true.
          geom%levelsP(1:i_wtd) = fixed_levelsP(1:i_wtd)
          geom%levelsP(i_wtd+1) = wtd_in
          geom%levelsP(i_wtd+2:nz+1) = fixed_levelsP(i_wtd+1:nz)
          wtd_eff = wtd_in ! no need to change
        end if
      end if
    end if

    geom%dzP = geom%levelsP(2:nz+1) - geom%levelsP(1:nz)
  end subroutine new_geom

  pure function find_wtd_index(levelsP,wtd) result(index)
    ! Search for the index of the layer in which wtd is in. The bottom level
    ! (boundary) is part of the layer, the top level is not. If wtd is smaller
    ! than, or equal to, levelsP(1), return 0. If wtd is larger than
    ! levelsP(nz+1), return nz+1. Will not return the index of dz=0 layer, but
    ! the one physically above.
    real(dp), intent(in) :: levelsP(:),wtd
    integer              :: nz,i,index

    nz = size(levelsP) - 1 ; index = nz + 1
    do i = nz,0,-1
      if (wtd <= levelsP(i+1)) index = i
    end do
  end function find_wtd_index

end module methane_geometry
