!||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||

module ecosys_nd_mod

!BOP
! !MODULE: ecosys_nd_mod
!
! !DESCRIPTION:Module for Nd 
!  The concentration of Nd143 and Nd144 are treated as tracers.
!  Unit for Nd143 and Nd144 is pmol/m^3
!
! !REVISION HISTORY:
!  SVN:$Id: 

! !USES:

   use POP_KindsMod
   use POP_IOUnitsMod
   use POP_ErrorMod

   use blocks, only: nx_block, ny_block, block
   use domain_size, only: max_blocks_clinic, km, nx_global, ny_global
   use domain, only: nblocks_clinic
   use exit_mod, only: sigAbort, exit_POP
   use communicate, only: my_task, master_task
   use prognostic, only: tracer_field
   use kinds_mod
   use constants, only: c0,c1000,c10000, c1, p5, char_blank, delim_fmt, field_type_scalar
   use io, only: data_set
   use io_types, only: stdout, nml_in, nml_filename, datafile, io_dim,  &
       io_field_desc, rec_type_dbl, construct_file, construct_io_dim,   &
       construct_io_field, destroy_file, destroy_io_field
   use io_tools, only: document
   use tavg, only: define_tavg_field, accumulate_tavg_field, accumulate_tavg_now
   use passive_tracer_tools, only: ind_name_pair, tracer_read, &
       rest_read_tracer_block, file_read_tracer_block, read_field
   
   
   use ecosys_share
   use ecosys_parms
   
   
   implicit none
   private

! !PUBLIC MEMBER FUNCTIONS:

   public :: ecosys_nd_tracer_cnt,        &
             ecosys_nd_init,              &
             ecosys_nd_set_interior,      &
             ecosys_nd_tavg_forcing

!EOP
!BOC

!-----------------------------------------------------------------------
!  module variables required by passive_tracer
!-----------------------------------------------------------------------

   integer(int_kind), parameter :: &
       ecosys_nd_tracer_cnt = 2

!-----------------------------------------------------------------------
!  relative tracer indices
!-----------------------------------------------------------------------

   integer(int_kind), parameter :: &
       ecosys_nd143_ind = 1,      &     ! nd143 index
       ecosys_nd144_ind = 2          ! nd144 index

!-----------------------------------------------------------------------
!  derived type & parameter for tracer index lookup
!-----------------------------------------------------------------------

   type(ind_name_pair), dimension(ecosys_nd_tracer_cnt) :: &
       ind_name_table = (/ ind_name_pair(ecosys_nd143_ind, 'eco_ND143'),ind_name_pair(ecosys_nd144_ind, 'eco_ND144') /)

!-----------------------------------------------------------------------
!  tavg ids for non-standard tavg variables (3d fields)
!-----------------------------------------------------------------------

   integer (int_kind) :: &
      
       tavg_eco_ND143_D,                &
       tavg_eco_ND143_P,                &
       tavg_eco_ND144_D,                &
       tavg_eco_ND144_P,                &
       tavg_eco_ND143_SOURCE,           &
       tavg_eco_ND143_SINK,             &
       tavg_eco_ND144_SOURCE,           &
       tavg_eco_ND144_SINK,             &
       tavg_eco_ND143_RESET_TEND,        &
       tavg_eco_ND144_RESET_TEND,		&
       tavg_c_POC, 						&
       tavg_c_calcite,					&
       tavg_c_SiO2,						&
       tavg_c_dust,                     &
       tavg_eco_PRODK,                  &
       tavg_eco_PRODK_in,                  &
       tavg_eco_PRODK_out,                  &
       tavg_eco_SURFACE_DUST,        &       ! tavg id for surface dust deposition			
       tavg_eco_DUST_IR,				&
      tavg_eco_RIVER_CON,			&
      tavg_eco_RIVER_IR,			&
      tavg_eco_RIVER_SOURCE,		&
      tavg_eco_RIVER_SOURCE143,     &
      tavg_eco_RIVER_SOURCE144,     &
      tavg_eco_MARGIN_IR,			&		! tavg id for margin IR
      tavg_eco_DUST_SOURCE,			&       ! tavg id for dust source term
      tavg_eco_DUST_SOURCE143,      &       ! tavg id for dust source 143
      tavg_eco_DUST_SOURCE144,		&       ! tavg id for dust source 144
      tavg_eco_MARGIN_SOURCE,       &   	! tavg id for margin source term
      tavg_eco_MARGIN_SOURCE143,    &
      tavg_eco_MARGIN_SOURCE144,    &
      tavg_eco_ND143_cur,           &
      tavg_eco_ND144_cur,					&
      tavg_eco_ND143_TOP,			&
      tavg_eco_ND143_BOT,			&
      tavg_eco_ND144_TOP,			&
      tavg_eco_ND144_BOT,           &
      tavg_eco_POC_flux,            &
      tavg_eco_OPAL_flux,           &
      tavg_eco_calcite_flux,        &
      tavg_eco_dust_flux,           &
      tavg_eco_POC_flux_in,         &
      tavg_eco_POC_flux_out,		&
      tavg_eco_OPAL_flux_in,		&
      tavg_eco_OPAL_flux_out,		&
      tavg_eco_calcite_flux_in,		&
      tavg_eco_calcite_flux_out,	&
      tavg_eco_dust_flux_in,		&
      tavg_eco_dust_flux_out,		&
      tavg_poc_mass,			&
      tavg_w


!-----------------------------------------------------------------------
!  mask the continental margin area
!-----------------------------------------------------------------------
   logical(log_kind),dimension(:,:,:,:),allocatable :: &
       eco_MARGIN_MASK

!-----------------------------------------------------------------------
!  Surface dust deposition surf_dust: read from dust.nc unit: gm^-2s^-1
!-----------------------------------------------------------------------
   real(r8),dimension(:,:,:),allocatable :: &
       eco_surf_dust,           & !surface dust deposition 
       eco_ir_dust,             &
       eco_river_con,           &
       eco_ir_river,            &
       eco_river_source,        &
       eco_river_source143,     &
       eco_river_source144,     &
       eco_ir_margin,           &
       eco_dust_source,        &
       eco_dust_source144,  &
       eco_dust_source143,   &
       w,&
       poc_mass
 
    
   real(r8),dimension(:,:,:,:),allocatable :: &
   	   c_POC,								  & !POC concentration
   	   c_calcite,							  & !calcite concentration
   	   c_SiO2,								  & !SiO2 concentration
   	   c_dust,								  & !dust concentration
       eco_margin_source,                         &
       eco_margin_source144,                      &
       eco_margin_source143,                      &
       eco_prodk_cndk,                            &  ! denominator in equation 14(rempfer2011)
       eco_prodk_cndk_in,						  &
       eco_prodk_cndk_out,						  &
       eco_nd143_d,                               &
       eco_nd143_p,                               &
       eco_nd144_d,                               &
       eco_nd144_p,                               &
       eco_nd143_source,                          &
       eco_nd143_sink,                            &
       eco_nd144_source,                          &
       eco_nd144_sink,                            &
       nd143_cur,           &
       nd144_cur,   &
       eco_nd143_top,							  &
	   eco_nd143_bot,					   		  &
	   eco_nd144_top,							  &
	   eco_nd144_bot,      &
	   eco_poc_flux,   &
	   eco_opal_flux,   &
	   eco_calcite_flux,   &
	   eco_dust_flux,      &
	   eco_poc_flux_in,		&
	   eco_poc_flux_out,	&
	   eco_opal_flux_in,	&
	   eco_opal_flux_out,   &
	   eco_calcite_flux_in,	&
	   eco_calcite_flux_out,&
	   eco_dust_flux_in,	&
	   eco_dust_flux_out

  
   real(r8)          :: &
       eco_nd_w,         &
       eco_k_poc,                    &
       eco_k_calcite,                &
       eco_k_opal,                   &
       eco_k_dust,       & 
	   eco_cnd_dust,                 & ! global concentration of Nd in dust
       eco_beta_dust,                & ! fraction of Nd released
       eco_nd_mass                 ! Nd mass g/pmol

!EOC
!*****************************************************************************

contains

!*****************************************************************************
!BOP
! !IROUTINE: ecosys_nd_init
! !INTERFACE:
subroutine ecosys_nd_init(init_ts_file_fmt, read_restart_filename, &
                      tracer_d_module, TRACER_MODULE, errorCode)

! !DESCRIPTION:
!  Initialize nd tracer module. This involves setting metadata, reading
!  the module namelist and setting initial conditions.
!
! !REVISION HISTORY:
!  same as module

! !USES:

   use broadcast, only: broadcast_scalar
   use prognostic, only: curtime, oldtime
   use grid, only: KMT, n_topo_smooth, fill_points, dz
   use time_management, only: seconds_in_year

! !INPUT PARAMETERS:

   character (*), intent(in) ::  &
       init_ts_file_fmt,       &   ! format (bin or nc) for input file
       read_restart_filename       ! file name for restart file

    
! !INPUT/OUTPUT PARAMETERS:

   type (tracer_field), dimension(ecosys_nd_tracer_cnt), intent(inout) :: &
       tracer_d_module   ! descriptors for each tracer

   real(r8), dimension(nx_block,ny_block,km,ecosys_nd_tracer_cnt,3,max_blocks_clinic), &
       intent(inout) :: TRACER_MODULE

! !OUTPUT PARAMETERS:

   integer (POP_i4), intent(out) :: &
       errorCode            ! returned error code

!EOP
!BOC
!-----------------------------------------------------------------------
!  local variables
!-----------------------------------------------------------------------

   character(*), parameter :: subname = 'ecosys_nd_mod:ecosys_Nd_init'

   character(char_len) :: &
       init_ecosys_nd_option,           & ! option for initialization of nd
       init_ecosys_nd_init_file,        & ! filename for option 'file'
       init_ecosys_nd_init_file_fmt,    & ! file format for option 'file'
  
       eco_prescribed_filename,       & ! fielname for dust deposition
       eco_surf_dust_varname,        & ! variale name for dust deposition
       eco_dust_ir_varname,          &
       eco_river_con_varname,        & ! variable name for Nd concentration in river runoff
       eco_river_ir_varname,         &
       eco_margin_ir_varname         ! variable name for river ir
 !      calcite_r_varname,        & ! variable name for calcite R
 !      poc_r_varname,            & ! variable name for poc R
 !      opal_r_varname,           & ! variable name for opal R
 !      dust_r_varname              ! variable name for dust R

   real(r8)          :: &
  !     eco_cnd_dust,                 & ! global concentration of Nd in dust
   !    eco_beta_dust,                & ! fraction of Nd released
   !    eco_nd_mass,                  & ! Nd mass g/pmol
       eco_a_boundary_tot,           & ! total area for boundary
       eco_f_boundary                ! total source from boundary	
  
   logical(log_kind) :: &
       lnml_found             ! Was nd_nml found ?

   integer(int_kind) :: &
       n,                   & ! index for looping over trlsacers
       k,                   & ! index for looping over depth levels
       nx,                  & ! index for looping over x direction
       ny,                  & ! index for looping over y direction
       iblock,              & ! index for looping over blocks
       nml_error              ! namelist i/o error flag

!     l,                   & ! index for looping over time levels
   type(tracer_read), dimension(ecosys_nd_tracer_cnt) :: &
       ecosys_nd_init_ext        ! namelist variable for initializing tracers

   namelist /ecosys_nd_nml/ &
       init_ecosys_nd_option, init_ecosys_nd_init_file, ecosys_nd_init_ext, &
       init_ecosys_nd_init_file_fmt, eco_prescribed_filename, eco_surf_dust_varname, &
        eco_dust_ir_varname, eco_river_con_varname,&
       eco_river_ir_varname, eco_margin_ir_varname,&
       eco_cnd_dust, eco_beta_dust, eco_nd_mass,eco_a_boundary_tot, eco_f_boundary, eco_k_poc,         &
       eco_k_calcite,eco_k_opal,eco_k_dust,eco_nd_w


   character (char_len) ::  &
       nd_eco_restart_filename  ! modified file name for restart file
      
  
!-----------------------------------------------------------------------
!  initialize tracer_d values
!-----------------------------------------------------------------------

   errorCode = POP_Success
  
   tracer_d_module(ecosys_nd143_ind)%short_name = 'eco_ND143'
   tracer_d_module(ecosys_nd143_ind)%long_name  = 'eco_ND143'
   tracer_d_module(ecosys_nd143_ind)%units      = 'pmol/m^3'
   tracer_d_module(ecosys_nd143_ind)%tend_units = 'pmol/(m^3s)'
!   tracer_d_module(nd143_ind)%flux_units = 'cm years/s'


   tracer_d_module(ecosys_nd144_ind)%short_name = 'eco_ND144'
   tracer_d_module(ecosys_nd144_ind)%long_name  = 'eco_ND144'
   tracer_d_module(ecosys_nd144_ind)%units      = 'pmol/m^3'
   tracer_d_module(ecosys_nd144_ind)%tend_units = 'pmol/(m^3s)'
!  tracer_d_module(nd144_ind)%flux_units = 'cm years/s'
!-----------------------------------------------------------------------
!  default namelist settings
!-----------------------------------------------------------------------

   init_ecosys_nd_option = 'unknown'
   init_ecosys_nd_init_file = 'unknown'
   init_ecosys_nd_init_file_fmt = 'bin'

   do n = 1,ecosys_nd_tracer_cnt
       ecosys_nd_init_ext(n)%mod_varname  = 'unknown'
       ecosys_nd_init_ext(n)%filename     = 'unknown'
       ecosys_nd_init_ext(n)%file_varname = 'unknown'
       ecosys_nd_init_ext(n)%scale_factor = c1
       ecosys_nd_init_ext(n)%default_val  = c0
       ecosys_nd_init_ext(n)%file_fmt     = 'nc'
   end do
   
   eco_prescribed_filename = 'unknown'
   eco_surf_dust_varname = 'unknown'
   eco_dust_ir_varname = 'unknown'
   eco_river_con_varname = 'unknown'
   eco_river_ir_varname = 'unknown'
   eco_margin_ir_varname = 'unknown'
   
  
   eco_cnd_dust = c0
   eco_beta_dust = c0
   eco_nd_mass = c0
   eco_a_boundary_tot = c0
   eco_f_boundary = c0
   eco_k_poc = c0
   eco_k_calcite = c0
   eco_k_opal = c0
   eco_k_dust = c0
   eco_nd_w = c0
  
   if (my_task == master_task) then
       open (nml_in, file=nml_filename, status='old',iostat=nml_error)
       if (nml_error /= 0) then  
           nml_error = -1
           call document(subname,'nml_error = -1')
       else
           nml_error =  1      
           call document(subname,'nml_error = 1')
       endif
       do while (nml_error > 0)
           read(nml_in, nml=ecosys_nd_nml,iostat=nml_error)
       end do
       if (nml_error == 0) close(nml_in)
   endif

   call broadcast_scalar(nml_error, master_task)
   if (nml_error /= 0) then
       call document(subname, 'ecosys_nd_nml not found')
       call exit_POP(sigAbort, 'stopping in ' /&
                           &/ subname)
   endif

!-----------------------------------------------------------------------
!  broadcast all namelist variables
!-----------------------------------------------------------------------

   call broadcast_scalar(init_ecosys_nd_option , master_task)
   call broadcast_scalar(init_ecosys_nd_init_file, master_task)
   call broadcast_scalar(init_ecosys_nd_init_file_fmt, master_task)

   do n = 1,ecosys_nd_tracer_cnt
      call broadcast_scalar(ecosys_nd_init_ext(n)%mod_varname, master_task)
      call broadcast_scalar(ecosys_nd_init_ext(n)%filename, master_task)
      call broadcast_scalar(ecosys_nd_init_ext(n)%file_varname, master_task)
      call broadcast_scalar(ecosys_nd_init_ext(n)%scale_factor, master_task)
      call broadcast_scalar(ecosys_nd_init_ext(n)%default_val, master_task)
      call broadcast_scalar(ecosys_nd_init_ext(n)%file_fmt, master_task)
   end do


   call broadcast_scalar(eco_prescribed_filename, master_task)
   call broadcast_scalar(eco_surf_dust_varname, master_task)
   call broadcast_scalar(eco_dust_ir_varname, master_task)
   call broadcast_scalar(eco_river_con_varname, master_task)
   call broadcast_scalar(eco_river_ir_varname, master_task)
   call broadcast_scalar(eco_margin_ir_varname, master_task)
   call broadcast_scalar(eco_cnd_dust, master_task)
   call broadcast_scalar(eco_beta_dust, master_task)
   call broadcast_scalar(eco_nd_mass, master_task)
   call broadcast_scalar(eco_a_boundary_tot, master_task)
   call broadcast_scalar(eco_f_boundary, master_task)
   call broadcast_scalar(eco_k_poc, master_task)
   call broadcast_scalar(eco_k_calcite, master_task)
   call broadcast_scalar(eco_k_opal, master_task)
   call broadcast_scalar(eco_k_dust, master_task)
   call broadcast_scalar(eco_nd_w, master_task)
   

   
!-----------------------------------------------------------------------
!  initialize tracers
!-----------------------------------------------------------------------

   select case (init_ecosys_nd_option)

   case ('ccsm_startup', 'zero', 'ccsm_startup_spunup')
      TRACER_MODULE = c0
      if (my_task == master_task) then
          write(stdout,delim_fmt)
          write(stdout,*) ' Initial 3-d  isotope ratio set to all zeros' 
          write(stdout,delim_fmt)
          call POP_IOUnitsFlush(POP_stdout) ; call POP_IOUnitsFlush(stdout)
      endif
       
   case ('restart', 'ccsm_continue', 'ccsm_branch', 'ccsm_hybrid' )
      nd_eco_restart_filename = char_blank
      if (init_ecosys_nd_init_file == 'same_as_TS') then
        if (read_restart_filename == 'undefined') then
            call document(subname, 'no restart file to read eco Nd from')
            call exit_POP(sigAbort, 'stopping in ' /&
                                 &/ subname)
        endif
        nd_eco_restart_filename = read_restart_filename
        init_ecosys_nd_init_file_fmt = init_ts_file_fmt

      else  ! do not read from TS restart file
        nd_eco_restart_filename = trim(init_ecosys_nd_init_file)

      endif
      call rest_read_tracer_block(init_ecosys_nd_init_file_fmt, &
                                  nd_eco_restart_filename,   &
                                  tracer_d_module,         &
                                  TRACER_MODULE)

   case ('file')
      call document(subname, 'eco Nd being read from separate file')

      call file_read_tracer_block(init_ecosys_nd_init_file_fmt, &
                                  init_ecosys_nd_init_file,     &
                                  tracer_d_module,         &
                                  ind_name_table,          &
                                  ecosys_nd_init_ext,         &
                                  TRACER_MODULE)
 
      if (n_topo_smooth > 0) then
        do n = 1,ecosys_nd_tracer_cnt
         do k=1,km
            call fill_points(k,TRACER_MODULE(:,:,k,n,curtime,:), errorCode)

            if (errorCode /= POP_Success) then
               call POP_ErrorSet(errorCode, &
                  'ecosys_nd_init: error in fill_points for eco Nd (curtime)')
               return
            endif
            
            call fill_points(k,TRACER_MODULE(:,:,k,n,oldtime,:), errorCode)

            if (errorCode /= POP_Success) then
               call POP_ErrorSet(errorCode, &
                  'ecosys_nd_init: error in fill_points for eco Nd (oldime)')
               return
            endif
            
          end do
         end do
         
      endif
    case default
      call document(subname, 'init_ecosys_nd_option = ', init_ecosys_nd_option)
      call exit_POP(sigAbort, 'stopping in ' /&
                           &/ subname)

   end select

!-----------------------------------------------------------------------
!  apply land mask to tracers
!-----------------------------------------------------------------------

   do iblock=1,nblocks_clinic
      do n = 1,ecosys_nd_tracer_cnt
         do k = 1,km
            where (k > KMT(:,:,iblock))
                 TRACER_MODULE(:,:,k,n,curtime,iblock) = c0
                 TRACER_MODULE(:,:,k,n,oldtime,iblock) = c0
            end where
         end do
      end do
   enddo

!-----------------------------------------------------------------------

   call define_tavg_field(tavg_eco_ND143_RESET_TEND, 'eco_ND143_RESET_TEND',2,  &
                          long_name='surface reset tendency of eco ND', &
                          units='years/s', grid_loc='2110',           &
                          coordinates='TLONG TLAT time')
   call define_tavg_field(tavg_eco_ND144_RESET_TEND, 'eco_ND144_RESET_TEND',2,  &
                          long_name='surface reset tendency of eco ND', &
                          units='years/s', grid_loc='2110',           &
                          coordinates='TLONG TLAT time')

    
   call define_tavg_field(tavg_eco_ND143_D, 'eco_ND143_D',3,&
                          long_name='eco ND143 disolved',&
                          units='pmol/m^3', grid_loc='3111',coordinates='TLONG TLAT z_t time')
     
     
   call define_tavg_field(tavg_eco_ND143_P, 'eco_ND143_P',3,&
                          long_name='eco ND143 particle related',&
                          units='pmol/m^3', grid_loc='3111',coordinates='TLONG TLAT z_t time')
     
     
   call define_tavg_field(tavg_eco_ND144_D, 'eco_ND144_D',3,&
                          long_name='eco ND144 disolved',&
                          units='pmol/m^3', grid_loc='3111',coordinates='TLONG TLAT z_t time')
     
   call define_tavg_field(tavg_eco_ND144_P, 'eco_ND144_P',3,&
                          long_name='eco ND144 particle related',&
                           units='pmol/m^3', grid_loc='3111',coordinates='TLONG TLAT z_t time')
     
  
     
   call define_tavg_field(tavg_eco_ND143_SOURCE, 'eco_ND143_SOURCE',3,&
                          long_name='source term: dust and boundary',&
                          grid_loc='3111',coordinates='TLONG TLAT z_t time')
  
   call define_tavg_field(tavg_eco_ND143_SINK, 'eco_ND143_SINK',3,&
                          long_name='sink term',&
                          grid_loc='3111',coordinates='TLONG TLAT z_t time')
     

   call define_tavg_field(tavg_eco_ND144_SOURCE, 'eco_ND144_SOURCE',3,&
                          long_name='source term: dust and boundary',&
                          grid_loc='3111',coordinates='TLONG TLAT z_t time')
  
   call define_tavg_field(tavg_eco_ND144_SINK, 'eco_ND144_SINK',3,&
                          long_name='sink term',&
                          grid_loc='3111',coordinates='TLONG TLAT z_t time')


   call define_tavg_field(tavg_c_POC,'eco_POC_con',3,&
   						  long_name='POC concentration',&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')
   						  
   						  
   call define_tavg_field(tavg_c_calcite,'eco_calcite_con',3,&
   						  long_name='Calcite concentration',&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')
   						  
   call define_tavg_field(tavg_c_SiO2,'eco_SiO2_con',3,&
   						  long_name='SiO2 concentration',&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')
   						  
   						  
   call define_tavg_field(tavg_c_dust,'eco_dust_con',3,&
   						  long_name='dust concentration',&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')					  
     
     
    call define_tavg_field(tavg_eco_PRODK, 'eco_PRODK',3,				&
   						  long_name='denominator in equation 14',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')	
   						  

    call define_tavg_field(tavg_eco_PRODK_in, 'eco_PRODK_in',3,				&
   						  long_name='denominator in equation 14',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')	


    call define_tavg_field(tavg_eco_PRODK_out, 'eco_PRODK_out',3,				&
   						  long_name='denominator in equation 14',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')	


			  
   	call define_tavg_field(tavg_eco_SURFACE_DUST, 'eco_SURFACE_DUST',2,         &
   						  long_name='surface dust deposition',&
   						  units='g/(m^2s)',grid_loc='2110')

   call define_tavg_field(tavg_eco_DUST_IR, 'eco_DUST_IR',2,                   &
   						  long_name='isotopic ratio for surface dust depositon',&
   						  grid_loc='2110')
   
   
   call define_tavg_field(tavg_eco_RIVER_CON, 'eco_RIVER_CON',2,         &
   						  long_name='Nd concentration in river runoff',&
   						  units='pmol/kg',grid_loc='2110')						  
   
   
   
   call define_tavg_field(tavg_eco_RIVER_IR, 'eco_RIVER_IR',2,                   &
   						  long_name='isotopic ratio for river runoff',&
   						  grid_loc='2110')
   						  
   call define_tavg_field(tavg_eco_RIVER_SOURCE, 'eco_RIVER_SOURCE',2,                   &
   						  long_name='river source total',&
   						  grid_loc='2110')
   						  
   call define_tavg_field(tavg_eco_RIVER_SOURCE143, 'eco_RIVER_SOURCE143',2,                   &
   						  long_name='river source143',&
   						  grid_loc='2110')
   						  
   call define_tavg_field(tavg_eco_RIVER_SOURCE144, 'eco_RIVER_SOURCE144',2,                   &
   						  long_name='river source 144',&
   						  grid_loc='2110')						  						  
   						  
   call define_tavg_field(tavg_eco_MARGIN_IR, 'eco_MARGIN_IR',2,                   &
   						  long_name='isotopic ratio for boundary source',&
   						  grid_loc='2110')
   						         
   						         
   						         
    call define_tavg_field(tavg_eco_DUST_SOURCE, 'eco_DUST_SOURCE',2,				&
   						  long_name='dust source for total Nd', 			&
   						  units='pmol/(m^3s)', grid_loc='2110')
   
   call define_tavg_field(tavg_eco_DUST_SOURCE143, 'eco_DUST_SOURCE143',2,				&
   						  long_name='dust source for Nd143', 			&
   						  units='pmol/(m^3s)', grid_loc='2110')
   						  
   call define_tavg_field(tavg_eco_DUST_SOURCE144, 'eco_DUST_SOURCE144',2,				&
   						  long_name='dust source for Nd144', 			&
   						  units='pmol/(m^3s)', grid_loc='2110')
   						     						  						  						  
   call define_tavg_field(tavg_eco_MARGIN_SOURCE, 'eco_MARGIN_SOURCE',3,			&
   						  long_name='margin source for total Nd',			&
   						  units='pmol/(m^3s)', grid_loc='3111')
   						  
   call define_tavg_field(tavg_eco_MARGIN_SOURCE143, 'eco_MARGIN_SOURCE143',3,			&
   						  long_name='margin source for Nd143',			&
   						  units='pmol/(m^3s)', grid_loc='3111')						  
  
   call define_tavg_field(tavg_eco_MARGIN_SOURCE144, 'eco_MARGIN_SOURCE144',3,			&
   						  long_name='margin source for Nd144',			&
   						  units='pmol/(m^3s)', grid_loc='3111')
   
   
   call define_tavg_field(tavg_eco_ND143_cur, 'eco_ND143_cur',3,			&
   						  long_name='current ND 143',			&
   						  units='', grid_loc='3111')
   						  
   call define_tavg_field(tavg_eco_ND144_cur, 'eco_ND144_cur',3,			&
   						  long_name='current ND 144',			&
   						  units='', grid_loc='3111')
	
	
	
   call define_tavg_field(tavg_eco_ND143_TOP, 'eco_ND143_TOP',3,				&
   						  long_name='ND143_TOP',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')
   call define_tavg_field(tavg_eco_ND143_BOT, 'eco_ND143_BOT',3,				&
   						  long_name='ND143_BOT',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')
   call define_tavg_field(tavg_eco_ND144_TOP, 'eco_ND144_TOP',3,				&
   						  long_name='ND144_TOP',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')
   call define_tavg_field(tavg_eco_ND144_BOT, 'eco_ND144_BOT',3,				&
   						  long_name='ND144_BOT',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time')		
   
   
   
   call define_tavg_field(tavg_eco_POC_flux, 'eco_POC_flux',3,				&
   						  long_name=' ',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	
   						  						  
   call define_tavg_field(tavg_eco_OPAL_flux, 'eco_OPAL_flux',3,				&
   						  long_name='eco_OPAL_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	

   call define_tavg_field(tavg_eco_calcite_flux, 'eco_calcite_flux',3,				&
   						  long_name='eco_calcite_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	

   call define_tavg_field(tavg_eco_dust_flux, 'eco_dust_flux',3,				&
   						  long_name='eco_dust_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	
   						  
   	
   	
   call define_tavg_field(tavg_eco_POC_flux_in, 'eco_POC_flux_in',3,				&
   						  long_name=' ',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	
   						  						  
   call define_tavg_field(tavg_eco_OPAL_flux_in, 'eco_OPAL_flux_in',3,				&
   						  long_name='eco_OPAL_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	

   call define_tavg_field(tavg_eco_calcite_flux_in, 'eco_calcite_flux_in',3,				&
   						  long_name='eco_calcite_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	

   call define_tavg_field(tavg_eco_dust_flux_in, 'eco_dust_flux_in',3,				&
   						  long_name='eco_dust_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	
   						  
   
   
   call define_tavg_field(tavg_eco_POC_flux_out, 'eco_POC_flux_out',3,				&
   						  long_name=' ',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	
   						  						  
   call define_tavg_field(tavg_eco_OPAL_flux_out, 'eco_OPAL_flux_out',3,				&
   						  long_name='eco_OPAL_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	

   call define_tavg_field(tavg_eco_calcite_flux_out, 'eco_calcite_flux_out',3,				&
   						  long_name='eco_calcite_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	

   call define_tavg_field(tavg_eco_dust_flux_out, 'eco_dust_flux_out',3,				&
   						  long_name='eco_dust_flux',		&
   						  grid_loc='3111',coordinates='TLONG TLAT z_t time', units = 'base unit/cm^2/s')	
   						  
   						  					  
   						  
   						  
   						  
   	call define_tavg_field(tavg_poc_mass, 'poc_mass',2,                   &
   						  long_name='poc_mass',&
   						  grid_loc='2110')

  	call define_tavg_field(tavg_w, 'w_nde',2,                   &
   						  long_name='w',&
   						  grid_loc='2110')

   						  				
   write(stdout,delim_fmt)
   write(stdout,*) ' successful define tavg_SURFACE_DUST' 
   write(stdout,delim_fmt)

!-----------------------------------------------------------------------
!  allocate and initialize MARGIN_MASK (true for margin points)
!-----------------------------------------------------------------------
   allocate(eco_MARGIN_MASK(nx_block,ny_block,km,max_blocks_clinic))

   eco_MARGIN_MASK(:,:,:,:) = .false.

   do k = 1,51
       where (KMT .eq. k)
            eco_MARGIN_MASK(:,:,k,:) = .true.
       end where
   end do


!-----------------------------------------------------------------------
!  read in surface dust deposition
!-----------------------------------------------------------------------

   allocate(eco_surf_dust(nx_block,ny_block,max_blocks_clinic))
   eco_surf_dust = c0
   call read_field('nc',eco_prescribed_filename,eco_surf_dust_varname,eco_surf_dust)

   allocate(eco_ir_dust(nx_block,ny_block,max_blocks_clinic))
   eco_ir_dust = c0
   call read_field('nc',eco_prescribed_filename,eco_dust_ir_varname,eco_ir_dust)

   allocate(eco_river_con(nx_block,ny_block,max_blocks_clinic))
   eco_river_con = c0
   call read_field('nc',eco_prescribed_filename,eco_river_con_varname,eco_river_con)

   allocate(eco_ir_river(nx_block,ny_block,max_blocks_clinic))
   eco_ir_river = c0
   call read_field('nc',eco_prescribed_filename,eco_river_ir_varname,eco_ir_river)

   allocate(eco_river_source(nx_block,ny_block,max_blocks_clinic))
   eco_river_source = c0

   allocate(eco_river_source143(nx_block,ny_block,max_blocks_clinic))
   eco_river_source143 = c0
   allocate(eco_river_source144(nx_block,ny_block,max_blocks_clinic))
   eco_river_source144 = c0

   allocate(eco_ir_margin(nx_block,ny_block,max_blocks_clinic))
   eco_ir_margin = c0
   call read_field('nc',eco_prescribed_filename,eco_margin_ir_varname,eco_ir_margin)



allocate(poc_mass(nx_block,ny_block,max_blocks_clinic))
allocate(w(nx_block,ny_block,max_blocks_clinic))

poc_mass = c0
w = c0
!-----------------------------------------------------------------------
!  read in particle concentration ratio 
!-----------------------------------------------------------------------

   allocate(c_POC(nx_block,ny_block,km,max_blocks_clinic))
   c_POC = c0

   allocate(c_calcite(nx_block,ny_block,km,max_blocks_clinic))
   c_calcite = c0

   allocate(c_SiO2(nx_block,ny_block,km,max_blocks_clinic))
   c_SiO2 = c0

   allocate(c_dust(nx_block,ny_block,km,max_blocks_clinic))
   c_dust = c0

!-----------------------------------------------------------------------
!  calculate prodk_cndk
!-----------------------------------------------------------------------
   allocate(eco_prodk_cndk(nx_block,ny_block,km,max_blocks_clinic))
   eco_prodk_cndk = c0
     
   allocate(eco_prodk_cndk_in(nx_block,ny_block,km,max_blocks_clinic))
   eco_prodk_cndk_in = c0
   
   allocate(eco_prodk_cndk_out(nx_block,ny_block,km,max_blocks_clinic))
   eco_prodk_cndk_out = c0
   
!-----------------------------------------------------------------------
!  calculate dust source term for total Nd, use ir_dust to calculate 
!  dust source for Nd143 and Nd144
!-----------------------------------------------------------------------

   allocate(eco_dust_source(nx_block,ny_block,max_blocks_clinic))
   allocate(eco_dust_source144(nx_block,ny_block,max_blocks_clinic))
   allocate(eco_dust_source143(nx_block,ny_block,max_blocks_clinic))

 !  eco_dust_source = eco_surf_dust*eco_cnd_dust*eco_beta_dust/(dz(1)*0.01_r8*eco_nd_mass)
	eco_dust_source = c0
	eco_dust_source143 = c0
	eco_dust_source144 = c0


!-----------------------------------------------------------------------
!  calculate margin source term for total Nd, use ir_dust to calculate 
!  dust source for Nd143 and Nd144
!-----------------------------------------------------------------------
   allocate(eco_margin_source(nx_block,ny_block,km,max_blocks_clinic))
   allocate(eco_margin_source144(nx_block,ny_block,km,max_blocks_clinic))
   allocate(eco_margin_source143(nx_block,ny_block,km,max_blocks_clinic))


   do iblock = 1,nblocks_clinic
     do nx = 1,nx_block
       do ny = 1,ny_block 

            do k = 1,km
                eco_margin_source(nx,ny,k,iblock) = eco_f_boundary/(eco_a_boundary_tot*dz(k)*0.01_r8*seconds_in_year*eco_nd_mass)
                eco_margin_source144(nx,ny,k,iblock) = 0.36_r8*eco_margin_source(nx,ny,k,iblock)/(eco_ir_margin(nx,ny,iblock)+c1)
                eco_margin_source143(nx,ny,k,iblock) = 0.36_r8*eco_margin_source(nx,ny,k,iblock)-eco_margin_source144(nx,ny,k,iblock)
            end do
      end do
     end do
   end do

   where(eco_MARGIN_MASK)
   elsewhere
      eco_margin_source = c0
      eco_margin_source144 = c0
      eco_margin_source143 = c0
   end where


!-----------------------------------------------------------------------
!  allocate space for nd143_d,nd143_p,nd144_d,nd144_p
!-----------------------------------------------------------------------
allocate(eco_nd143_d(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd143_p(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd144_d(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd144_p(nx_block,ny_block,km,max_blocks_clinic))

allocate(eco_nd143_source(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd143_sink(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd144_source(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd144_sink(nx_block,ny_block,km,max_blocks_clinic))

allocate(nd143_cur(nx_block,ny_block,km,max_blocks_clinic))
allocate(nd144_cur(nx_block,ny_block,km,max_blocks_clinic))


allocate(eco_nd143_top(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd143_bot(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd144_top(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_nd144_bot(nx_block,ny_block,km,max_blocks_clinic))



allocate(eco_poc_flux(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_opal_flux(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_calcite_flux(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_dust_flux(nx_block,ny_block,km,max_blocks_clinic))


allocate(eco_poc_flux_in(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_opal_flux_in(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_calcite_flux_in(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_dust_flux_in(nx_block,ny_block,km,max_blocks_clinic))


allocate(eco_poc_flux_out(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_opal_flux_out(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_calcite_flux_out(nx_block,ny_block,km,max_blocks_clinic))
allocate(eco_dust_flux_out(nx_block,ny_block,km,max_blocks_clinic))



eco_nd143_top = c0
eco_nd144_top = c0
eco_nd143_bot = c0
eco_nd144_bot = c0

eco_nd143_source = c0
eco_nd144_source = c0
eco_nd143_sink = c0
eco_nd144_sink = c0

eco_nd143_d = c0
eco_nd143_p = c0
eco_nd144_d = c0
eco_nd144_p = c0

nd143_cur = c0
nd144_cur = c0


eco_poc_flux = c0
eco_opal_flux = c0
eco_calcite_flux = c0
eco_dust_flux = c0


eco_poc_flux_in = c0
eco_opal_flux_in = c0
eco_calcite_flux_in = c0
eco_dust_flux_in = c0


eco_poc_flux_out = c0
eco_opal_flux_out = c0
eco_calcite_flux_out = c0
eco_dust_flux_out = c0

!EOC

 end subroutine ecosys_nd_init

!***********************************************************************
!BOP
! !IROUTINE: nd_set_interior
! !INTERFACE:

 
subroutine ecosys_nd_set_interior(k, TRACER_MODULE_OLD, TRACER_MODULE_CUR, &
                           DTRACER_MODULE,this_block)

! !DESCRIPTION:
!  set interior source/sink term for nd isotope ratio tracer
!
! !REVISION HISTORY:
!  same as module

! !USES:

   use time_management, only: seconds_in_year
   use grid, only : dz,KMT,REGION_MASK,dzwr
   
   use forcing_fields, only : ROFF_F
! !INPUT PARAMETERS:

   integer(int_kind), intent(in) :: &
      k                   ! vertical level index
      
   real (r8), dimension(:,:,:,:), intent(in) :: &
      TRACER_MODULE_OLD, &! old tracer values
      TRACER_MODULE_CUR   ! current tracer values
      
   type (block), intent(in) :: &
      this_block          ! block info for the current block

! !OUTPUT PARAMETERS:

   real(r8), dimension(nx_block,ny_block,ecosys_nd_tracer_cnt), intent(out) :: &
      DTRACER_MODULE      ! computed source/sink term
    
!-----------------------------------------------------------------------
!  local variables
!-----------------------------------------------------------------------
   
      
   integer (int_kind)       :: &
      bid  ,        &               ! local_block id
      nx,           &
      ny,           &
      ntracer
   
   logical(log_kind),dimension(nx_block,ny_block) :: &
      mask
  
   real(r8), dimension(nx_block,ny_block) :: &
      p143_k_upper,        &
      p143_k,              &
      p143_k_lower,        &
      p144_k_upper,        &
      p144_k,              &
      p144_k_lower,        &
      p143_top,            &
      p143_bot,            &
      p144_top,            &
      p144_bot,            &
!      nd143_cur,           &
!      nd144_cur,           &
      nd143_upper,         &
      nd144_upper,         &
      nd143_lower,         &
      nd144_lower  
   
   
!EOP
!BOC

    bid = this_block%local_id
    p143_k_upper = c0
    p143_k_lower = c0
    
!----------------------------------------------------------------------
!  calculate source from river input
!---------------------------------------------------------------------- 
    eco_river_source(:,:,bid) = ROFF_F(:,:,bid)* eco_river_con(:,:,bid)/(dz(1)*0.01_r8)
    
!-----------------------------------------------------------------------
!  get sinking particle concentration
!-----------------------------------------------------------------------  


	poc_mass = POC%mass
	w = eco_nd_w


  where((KMT(:,:,bid))>k .and. k.eq.1)
    eco_poc_flux(:,:,k,bid) = (POC%sflux_out(:,:,bid) + POC%hflux_out(:,:,bid))
    eco_opal_flux(:,:,k,bid) = (P_SiO2%sflux_out(:,:,bid) + P_SiO2%hflux_out(:,:,bid))
    eco_calcite_flux(:,:,k,bid) = (P_CaCO3%sflux_out(:,:,bid) + P_CaCO3%hflux_out(:,:,bid))
    eco_dust_flux(:,:,k,bid) = p5*(dust%sflux_in(:,:,bid) + dust%hflux_in(:,:,bid) + dust%sflux_out(:,:,bid) + dust%hflux_out(:,:,bid))
    
    eco_dust_source(:,:,bid) = (dust%sflux_in(:,:,bid) + dust%hflux_in(:,:,bid))*c10000*eco_cnd_dust*eco_beta_dust/(dz(1)*0.01_r8*eco_nd_mass)
    eco_dust_source144(:,:,bid) = 0.36_r8*eco_dust_source(:,:,bid)/(eco_ir_dust(:,:,bid)+c1)
    eco_dust_source143(:,:,bid) = 0.36_r8*eco_dust_source(:,:,bid) - eco_dust_source144(:,:,bid)
   end where
   
   where((KMT(:,:,bid))>k .and. k.gt.1)
    eco_poc_flux(:,:,k,bid) = p5*(POC%sflux_in(:,:,bid) + POC%hflux_in(:,:,bid) + POC%sflux_out(:,:,bid) + POC%hflux_out(:,:,bid))
    eco_opal_flux(:,:,k,bid) = p5*(P_SiO2%sflux_in(:,:,bid) + P_SiO2%hflux_in(:,:,bid) + P_SiO2%sflux_out(:,:,bid) + P_SiO2%hflux_out(:,:,bid))
    eco_calcite_flux(:,:,k,bid) = p5*(P_CaCO3%sflux_in(:,:,bid) + P_CaCO3%hflux_in(:,:,bid) + P_CaCO3%sflux_out(:,:,bid) + P_CaCO3%hflux_out(:,:,bid))
    eco_dust_flux(:,:,k,bid) = p5*(dust%sflux_in(:,:,bid) + dust%hflux_in(:,:,bid) + dust%sflux_out(:,:,bid) + dust%hflux_out(:,:,bid))
   end where
   
   
   where((KMT(:,:,bid)).eq.k)
   eco_poc_flux(:,:,k,bid) = POC%sflux_in(:,:,bid) + POC%hflux_in(:,:,bid)
    eco_opal_flux(:,:,k,bid) = P_SiO2%sflux_in(:,:,bid) + P_SiO2%hflux_in(:,:,bid)
    eco_calcite_flux(:,:,k,bid) = P_CaCO3%sflux_in(:,:,bid) + P_CaCO3%hflux_in(:,:,bid) 
    eco_dust_flux(:,:,k,bid) = dust%sflux_in(:,:,bid) + dust%hflux_in(:,:,bid)
   end where
   
    eco_poc_flux_in(:,:,k,bid) = (POC%sflux_in(:,:,bid) + POC%hflux_in(:,:,bid))
    eco_opal_flux_in(:,:,k,bid) = (P_SiO2%sflux_in(:,:,bid) + P_SiO2%hflux_in(:,:,bid))
    eco_calcite_flux_in(:,:,k,bid) = (P_CaCO3%sflux_in(:,:,bid) + P_CaCO3%hflux_in(:,:,bid))
    eco_dust_flux_in(:,:,k,bid) = (dust%sflux_in(:,:,bid) + dust%hflux_in(:,:,bid))
    
    
    eco_poc_flux_out(:,:,k,bid) = (POC%sflux_out(:,:,bid) + POC%hflux_out(:,:,bid))
    eco_opal_flux_out(:,:,k,bid) = (P_SiO2%sflux_out(:,:,bid) + P_SiO2%hflux_out(:,:,bid))
    eco_calcite_flux_out(:,:,k,bid) = (P_CaCO3%sflux_out(:,:,bid) + P_CaCO3%hflux_out(:,:,bid))
    eco_dust_flux_out(:,:,k,bid) = (dust%sflux_out(:,:,bid) + dust%hflux_out(:,:,bid))
   
    c_POC(:,:,k,bid) =  eco_poc_flux(:,:,k,bid)*POC%mass*1.0e-3_r8/eco_nd_w
    c_calcite(:,:,k,bid) = eco_calcite_flux(:,:,k,bid)*P_CaCO3%mass*1.0e-3_r8/eco_nd_w
    c_SiO2(:,:,k,bid) =  eco_opal_flux(:,:,k,bid)*P_SiO2%mass*1.0e-3_r8/eco_nd_w
    c_dust(:,:,k,bid) = eco_dust_flux(:,:,k,bid)*1.0e6_r8/eco_nd_w


	eco_prodk_cndk(:,:,k,bid) = c1 + eco_k_poc*c_POC(:,:,k,bid)/(1024.5_r8*c1000)+eco_k_calcite*c_calcite(:,:,k,bid)/(1024.5_r8*c1000) + &
						    eco_k_opal*c_SiO2(:,:,k,bid)/(1024.5_r8*c1000) + eco_k_dust*c_dust(:,:,k,bid)/(1024.5_r8*c1000)
	

    eco_prodk_cndk_in(:,:,k,bid) = c1 + eco_k_poc*(eco_poc_flux_in(:,:,k,bid)*POC%mass*1.0e-3_r8/eco_nd_w)/(1024.5_r8*c1000)+ &
    									eco_k_calcite*(eco_calcite_flux_in(:,:,k,bid)*P_CaCO3%mass*1.0e-3_r8/eco_nd_w)/(1024.5_r8*c1000) + &
						    			eco_k_opal*(eco_opal_flux_in(:,:,k,bid)*P_SiO2%mass*1.0e-3_r8/eco_nd_w)/(1024.5_r8*c1000) + &
						    			eco_k_dust*(eco_dust_flux_in(:,:,k,bid)*1.0e6_r8/eco_nd_w)/(1024.5_r8*c1000)
						    						    
	eco_prodk_cndk_out(:,:,k,bid) = c1 + eco_k_poc*(eco_poc_flux_out(:,:,k,bid)*POC%mass*1.0e-3_r8/eco_nd_w)/(1024.5_r8*c1000)+ &
    									eco_k_calcite*(eco_calcite_flux_out(:,:,k,bid)*P_CaCO3%mass*1.0e-3_r8/eco_nd_w)/(1024.5_r8*c1000) + &
						    			eco_k_opal*(eco_opal_flux_out(:,:,k,bid)*P_SiO2%mass*1.0e-3_r8/eco_nd_w)/(1024.5_r8*c1000) + &
						    			eco_k_dust*(eco_dust_flux_out(:,:,k,bid)*1.0e6_r8/eco_nd_w)/(1024.5_r8*c1000)

!-----------------------------------------------------------------------
    DTRACER_MODULE = c0
!-----------------------------------------------------------------------
!  apply dust source and river source
!-----------------------------------------------------------------------
    if (k==1) then
        eco_river_source(:,:,bid) = 0.3_r8*ROFF_F(:,:,bid)* eco_river_con(:,:,bid)/(dz(1)*0.01_r8)
        eco_river_source144(:,:,bid) = 0.36_r8*eco_river_source(:,:,bid)/(eco_ir_river(:,:,bid)+c1)
        eco_river_source143(:,:,bid) = 0.36_r8*eco_river_source(:,:,bid)-eco_river_source144(:,:,bid)
        DTRACER_MODULE(:,:,ecosys_nd143_ind) = DTRACER_MODULE(:,:,ecosys_nd143_ind) + eco_dust_source143(:,:,bid) +eco_river_source143(:,:,bid)
        DTRACER_MODULE(:,:,ecosys_nd144_ind) = DTRACER_MODULE(:,:,ecosys_nd144_ind) + eco_dust_source144(:,:,bid) +eco_river_source144(:,:,bid)
    endif
!-----------------------------------------------------------------------
!  apply boundary source
!-----------------------------------------------------------------------   
    mask= eco_MARGIN_MASK(:,:,k,bid)
  
    where(mask)
        DTRACER_MODULE(:,:,ecosys_nd143_ind) = DTRACER_MODULE(:,:,ecosys_nd143_ind) + eco_margin_source143(:,:,k,bid)
        DTRACER_MODULE(:,:,ecosys_nd144_ind) = DTRACER_MODULE(:,:,ecosys_nd144_ind) + eco_margin_source144(:,:,k,bid) 
    end where
    
!-----------------------------------------------------------------------
!  sink term
!-----------------------------------------------------------------------  

!  calculate nd143_d,nd143_p,nd144_d,nd144p and p143_top, p143_bot,p144_top,p144_bot
    p143_top = c0
    p143_bot = c0
    p144_top = c0
    p144_bot = c0
    

    
    nd143_cur(:,:,k,bid) = p5*(TRACER_MODULE_OLD(:,:,k,ecosys_nd143_ind)+TRACER_MODULE_CUR(:,:,k,ecosys_nd143_ind))
    nd144_cur(:,:,k,bid) = p5*(TRACER_MODULE_OLD(:,:,k,ecosys_nd144_ind)+TRACER_MODULE_CUR(:,:,k,ecosys_nd144_ind))

    where(KMT(:,:,bid)>=k)
        eco_nd143_d(:,:,k,bid) = nd143_cur(:,:,k,bid)/eco_prodk_cndk(:,:,k,bid)
        eco_nd143_p(:,:,k,bid) = nd143_cur(:,:,k,bid) - eco_nd143_d(:,:,k,bid)
        eco_nd144_d(:,:,k,bid) = nd144_cur(:,:,k,bid)/eco_prodk_cndk(:,:,k,bid)
        eco_nd144_p(:,:,k,bid) = nd144_cur(:,:,k,bid) - eco_nd144_d(:,:,k,bid)
    end where  
    
    
    
    where(eco_nd143_d(:,:,k,bid).lt.c0)
        eco_nd143_d(:,:,k,bid) = c0
    end where
    
    where(eco_nd143_p(:,:,k,bid).lt.c0)
        eco_nd143_p(:,:,k,bid) = c0
    end where
    
    where(eco_nd144_d(:,:,k,bid).lt.c0)
        eco_nd144_d(:,:,k,bid) = c0
    end where
    
    where(eco_nd144_p(:,:,k,bid).lt.c0)
        eco_nd144_p(:,:,k,bid) = c0
    end where
                  
    if(k==1) then
    
        	nd143_lower =  p5*(TRACER_MODULE_OLD(:,:,k+1,ecosys_nd143_ind)+ TRACER_MODULE_CUR(:,:,k+1,ecosys_nd143_ind))
        	nd144_lower =  p5*(TRACER_MODULE_OLD(:,:,k+1,ecosys_nd144_ind)+ TRACER_MODULE_CUR(:,:,k+1,ecosys_nd144_ind))              
        	p143_bot     = (nd143_cur(:,:,k,bid) + (nd143_lower-nd143_cur(:,:,k,bid))*dzwr(k)*p5*dz(k))*(1-c1/eco_prodk_cndk_out(:,:,k,bid))    
        	p144_bot     = (nd144_cur(:,:,k,bid) + (nd144_lower-nd144_cur(:,:,k,bid))*dzwr(k)*p5*dz(k))*(1-c1/eco_prodk_cndk_out(:,:,k,bid))    
                   
            eco_nd143_top(:,:,k,bid) = p143_top  !right
    		eco_nd143_bot(:,:,k,bid) = p143_bot
    		eco_nd144_top(:,:,k,bid) = p144_top   !right
    		eco_nd144_bot(:,:,k,bid) = p144_bot

		
			
    else if(k >1) then
       where((KMT(:,:,bid))>k)
    
            nd143_upper  = p5*(TRACER_MODULE_OLD(:,:,k-1,ecosys_nd143_ind)+ TRACER_MODULE_CUR(:,:,k-1,ecosys_nd143_ind))
            nd143_lower =  p5*(TRACER_MODULE_OLD(:,:,k+1,ecosys_nd143_ind)+ TRACER_MODULE_CUR(:,:,k+1,ecosys_nd143_ind))

    		p143_top    = (nd143_upper + (nd143_cur(:,:,k,bid) - nd143_upper)*dzwr(k-1)*p5*dz(k-1))*(1-c1/eco_prodk_cndk_in(:,:,k,bid))  
    		p143_bot    = (nd143_cur(:,:,k,bid) + (nd143_lower-nd143_cur(:,:,k,bid))*dzwr(k)*p5*dz(k))*(1-c1/eco_prodk_cndk_out(:,:,k,bid)) 
    
            nd144_upper  = p5*(TRACER_MODULE_OLD(:,:,k-1,ecosys_nd144_ind)+ TRACER_MODULE_CUR(:,:,k-1,ecosys_nd144_ind))
            nd144_lower =  p5*(TRACER_MODULE_OLD(:,:,k+1,ecosys_nd144_ind)+ TRACER_MODULE_CUR(:,:,k+1,ecosys_nd144_ind))
          
    		p144_top    = (nd144_upper + (nd144_cur(:,:,k,bid) - nd144_upper)*dzwr(k-1)*p5*dz(k-1))*(1-c1/eco_prodk_cndk_in(:,:,k,bid))  
    		p144_bot    = (nd144_cur(:,:,k,bid) + (nd144_lower-nd144_cur(:,:,k,bid))*dzwr(k)*p5*dz(k))*(1-c1/eco_prodk_cndk_out(:,:,k,bid)) 
    
    
            eco_nd143_top(:,:,k,bid) = p143_top  !right
    		eco_nd143_bot(:,:,k,bid) = p143_bot
    		eco_nd144_top(:,:,k,bid) = p144_top   !right
    		eco_nd144_bot(:,:,k,bid) = p144_bot
    	
    		
       end where
       where((KMT(:,:,bid)).eq.k)
            nd143_upper  = p5*(TRACER_MODULE_OLD(:,:,k-1,ecosys_nd143_ind)+ TRACER_MODULE_CUR(:,:,k-1,ecosys_nd143_ind))
      
    		p143_top    = (nd143_upper + (nd143_cur(:,:,k,bid) - nd143_upper)*dzwr(k-1)*p5*dz(k-1))*(1-c1/eco_prodk_cndk_in(:,:,k,bid))  
            p143_bot     = eco_nd143_p(:,:,k,bid)
    	
    	
            nd144_upper  = p5*(TRACER_MODULE_OLD(:,:,k-1,ecosys_nd144_ind)+ TRACER_MODULE_CUR(:,:,k-1,ecosys_nd144_ind))    
 
            p144_bot     = eco_nd144_p(:,:,k,bid)
    		p144_top    = (nd144_upper + (nd144_cur(:,:,k,bid) - nd144_upper)*dzwr(k-1)*p5*dz(k-1))*(1-c1/eco_prodk_cndk_in(:,:,k,bid))  
    
    		
            eco_nd143_top(:,:,k,bid) = p143_top  !right
    		eco_nd143_bot(:,:,k,bid) = p143_bot  !right
    		eco_nd144_top(:,:,k,bid) = p144_top  !right
    		eco_nd144_bot(:,:,k,bid) = p144_bot  !right
   
       end where
   
   

    end if
    
    eco_nd143_source(:,:,k,bid) =  DTRACER_MODULE(:,:,ecosys_nd143_ind)
    eco_nd144_source(:,:,k,bid) =  DTRACER_MODULE(:,:,ecosys_nd144_ind)
    
    !eco_nd143_source(:,:,k,bid) =   eco_dust_source143(:,:,bid) +eco_river_source143(:,:,bid)
    !eco_nd144_source(:,:,k,bid) =   eco_dust_source144(:,:,bid) +eco_river_source144(:,:,bid)
   
    eco_nd143_sink(:,:,k,bid) = eco_nd_w*(p143_bot - p143_top)/dz(k)
    eco_nd144_sink(:,:,k,bid) = eco_nd_w*(p144_bot - p144_top)/dz(k)
    

    
    DTRACER_MODULE(:,:,ecosys_nd143_ind) = DTRACER_MODULE(:,:,ecosys_nd143_ind) - eco_nd143_sink(:,:,k,bid)
    DTRACER_MODULE(:,:,ecosys_nd144_ind) = DTRACER_MODULE(:,:,ecosys_nd144_ind) - eco_nd144_sink(:,:,k,bid)
 
    where(REGION_MASK(:,:,bid).eq.-14 .or. REGION_MASK(:,:,bid).eq.-13 .or. REGION_MASK(:,:,bid).eq.-12 .or.    &
           REGION_MASK(:,:,bid).eq.-5 .or. REGION_MASK(:,:,bid).eq.4 .or. REGION_MASK(:,:,bid).eq.11) 
              DTRACER_MODULE(:,:,ecosys_nd143_ind) = c0
              DTRACER_MODULE(:,:,ecosys_nd144_ind) = c0
    end where
      
    
!-----------------------------------------------------------------------
!EOC

 end subroutine ecosys_nd_set_interior

!***********************************************************************

!***********************************************************************
!BOP
! !IROUTINE: nd_tavg_forcing
! !INTERFACE:

 subroutine ecosys_nd_tavg_forcing
 


! !DESCRIPTION:
!  Make accumulation calls for forcing related tavg fields. This is
!  necessary because the forcing routines are called before tavg flags
!  are set.

! !REVISION HISTORY:
!  same as module

!EOP
!BOC
!-----------------------------------------------------------------------
!  local variables
!-----------------------------------------------------------------------

   integer (int_kind) :: &
      iblock,   &      ! block loop index
      k

!-----------------------------------------------------------------------

   !$OMP PARALLEL DO PRIVATE(iblock)

   do iblock = 1, nblocks_clinic
   
         call accumulate_tavg_field(eco_surf_dust(:,:,iblock),tavg_eco_SURFACE_DUST,iblock,1)       
         call accumulate_tavg_field(eco_ir_dust(:,:,iblock),tavg_eco_DUST_IR,iblock,1)
   	     call accumulate_tavg_field(eco_river_con(:,:,iblock),tavg_eco_RIVER_CON,iblock,1)
         call accumulate_tavg_field(eco_ir_river(:,:,iblock),tavg_eco_RIVER_IR,iblock,1)
         call accumulate_tavg_field(eco_river_source(:,:,iblock),tavg_eco_RIVER_SOURCE,iblock,1)
         call accumulate_tavg_field(eco_river_source143(:,:,iblock),tavg_eco_RIVER_SOURCE143,iblock,1)
         call accumulate_tavg_field(eco_river_source144(:,:,iblock),tavg_eco_RIVER_SOURCE144,iblock,1)
 		 call accumulate_tavg_field(eco_ir_margin(:,:,iblock),tavg_eco_MARGIN_IR,iblock,1)
 		 call accumulate_tavg_field(eco_dust_source(:,:,iblock),tavg_eco_DUST_SOURCE,iblock,1)
 		 call accumulate_tavg_field(eco_dust_source143(:,:,iblock),tavg_eco_DUST_SOURCE143,iblock,1)
 		 call accumulate_tavg_field(eco_dust_source144(:,:,iblock),tavg_eco_DUST_SOURCE144,iblock,1)
 		 call accumulate_tavg_field(poc_mass(:,:,iblock),tavg_poc_mass,iblock,1)
 		 call accumulate_tavg_field(w(:,:,iblock),tavg_w,iblock,1)
       
       do k = 1,km
          call accumulate_tavg_field(eco_nd143_d(:,:,k,iblock),tavg_eco_ND143_D,iblock,k)
          call accumulate_tavg_field(eco_nd143_p(:,:,k,iblock),tavg_eco_ND143_P,iblock,k)
          call accumulate_tavg_field(eco_nd144_d(:,:,k,iblock),tavg_eco_ND144_D,iblock,k)
          call accumulate_tavg_field(eco_nd144_p(:,:,k,iblock),tavg_eco_ND144_P,iblock,k)
          call accumulate_tavg_field(eco_nd143_source(:,:,k,iblock),tavg_eco_ND143_SOURCE,iblock,k)
          call accumulate_tavg_field(eco_nd143_sink(:,:,k,iblock),tavg_eco_ND143_SINK,iblock,k)
          call accumulate_tavg_field(eco_nd144_source(:,:,k,iblock),tavg_eco_ND144_SOURCE,iblock,k)
          call accumulate_tavg_field(eco_nd144_sink(:,:,k,iblock),tavg_eco_ND144_SINK,iblock,k)
          call accumulate_tavg_field(c_POC(:,:,k,iblock),tavg_c_POC,iblock,k)
          call accumulate_tavg_field(c_calcite(:,:,k,iblock),tavg_c_calcite,iblock,k)
          call accumulate_tavg_field(c_SiO2(:,:,k,iblock),tavg_c_SiO2,iblock,k)
          call accumulate_tavg_field(c_dust(:,:,k,iblock),tavg_c_dust,iblock,k)
          
          call accumulate_tavg_field(eco_prodk_cndk(:,:,k,iblock),tavg_eco_PRODK,iblock,k)
          call accumulate_tavg_field(eco_prodk_cndk_in(:,:,k,iblock),tavg_eco_PRODK_in,iblock,k)
          call accumulate_tavg_field(eco_prodk_cndk_out(:,:,k,iblock),tavg_eco_PRODK_out,iblock,k)
          
          call accumulate_tavg_field(eco_margin_source(:,:,k,iblock),tavg_eco_MARGIN_SOURCE,iblock,k)
    	  call accumulate_tavg_field(eco_margin_source143(:,:,k,iblock),tavg_eco_MARGIN_SOURCE143,iblock,k)
    	  call accumulate_tavg_field(eco_margin_source144(:,:,k,iblock),tavg_eco_MARGIN_SOURCE144,iblock,k)
		  call accumulate_tavg_field(nd143_cur(:,:,k,iblock),tavg_eco_ND143_cur,iblock,k)
   	      call accumulate_tavg_field(nd144_cur(:,:,k,iblock),tavg_eco_ND144_cur,iblock,k)
   	      call accumulate_tavg_field(eco_nd143_top(:,:,k,iblock),tavg_eco_ND143_TOP,iblock,k)
    	  call accumulate_tavg_field(eco_nd143_bot(:,:,k,iblock),tavg_eco_ND143_BOT,iblock,k)
    	  call accumulate_tavg_field(eco_nd144_top(:,:,k,iblock),tavg_eco_ND144_TOP,iblock,k)
    	  call accumulate_tavg_field(eco_nd144_bot(:,:,k,iblock),tavg_eco_ND144_BOT,iblock,k)
    	  call accumulate_tavg_field(eco_poc_flux(:,:,k,iblock),tavg_eco_POC_flux,iblock,k)
    	  call accumulate_tavg_field(eco_opal_flux(:,:,k,iblock),tavg_eco_OPAL_flux,iblock,k)
    	  call accumulate_tavg_field(eco_calcite_flux(:,:,k,iblock),tavg_eco_calcite_flux,iblock,k)
    	  call accumulate_tavg_field(eco_dust_flux(:,:,k,iblock),tavg_eco_dust_flux,iblock,k)
    	  
    	  call accumulate_tavg_field(eco_poc_flux_in(:,:,k,iblock),tavg_eco_POC_flux_in,iblock,k)
    	  call accumulate_tavg_field(eco_opal_flux_in(:,:,k,iblock),tavg_eco_OPAL_flux_in,iblock,k)
    	  call accumulate_tavg_field(eco_calcite_flux_in(:,:,k,iblock),tavg_eco_calcite_flux_in,iblock,k)
    	  call accumulate_tavg_field(eco_dust_flux_in(:,:,k,iblock),tavg_eco_dust_flux_in,iblock,k)
    	  
    	  call accumulate_tavg_field(eco_poc_flux_out(:,:,k,iblock),tavg_eco_POC_flux_out,iblock,k)
    	  call accumulate_tavg_field(eco_opal_flux_out(:,:,k,iblock),tavg_eco_OPAL_flux_out,iblock,k)
    	  call accumulate_tavg_field(eco_calcite_flux_out(:,:,k,iblock),tavg_eco_calcite_flux_out,iblock,k)
    	  call accumulate_tavg_field(eco_dust_flux_out(:,:,k,iblock),tavg_eco_dust_flux_out,iblock,k)

      end do  
   
   end do

   !$OMP END PARALLEL DO

!-----------------------------------------------------------------------
!EOC

 end subroutine ecosys_nd_tavg_forcing

!***********************************************************************

!***********************************************************************
!BOP
! !IROUTINE: read_field_3D
! !INTERFACE:

 subroutine read_field_3D(fmt, filename, fieldname, FIELD, record_length)

! !DESCRIPTION:
!  read 3D field from a file
!  Assumes the field is (nx_global,ny_global), cell centered, and scalar.
!  The length of the 3rd dimension is determined by the dimension of FIELD.
!  For binary files, the default external precision is double precision.
!  This can be overridden by passing the desired precision into record_length.
!
! !REVISION HISTORY:
!  same as module

! !INPUT PARAMETERS:

   character (*), intent(in) ::  &
      fmt,                 & ! format (bin or nc)
      filename,            & ! file to read from
      fieldname              ! field to be read

   integer(int_kind), intent(in), optional :: &
      record_length          ! record length type for binary files

! !INPUT/OUTPUT PARAMETERS:

   real(r8), dimension(:,:,:,:), intent(inout), target :: &
      FIELD                  ! field to be read in

!EOP
!BOC
!-----------------------------------------------------------------------
!  local variables
!-----------------------------------------------------------------------

   character(*), parameter :: &
      subname = 'passive_tracer_tools:read_field_3D'

   integer(int_kind) :: &
      record_length_loc    ! record length type for binary files

   type (io_field_desc) :: &
      FIELD_DESC           ! IO field descriptors for FIELD

   type (datafile) :: &
      restart_file         ! io file descriptor

   type (io_dim) :: &
      i_dim, j_dim, k_dim  ! dimension descriptors

!-----------------------------------------------------------------------

   call document(subname, 'reading ' /&
                       &/ trim(fieldname) /&
                       &/ ' from ' /&
                       &/ trim(filename))

   if (present(record_length)) then
      record_length_loc = record_length
   else
      record_length_loc = rec_type_dbl
   endif

   restart_file =                                     &
      construct_file(fmt,                             &
                     full_name=trim(filename),        &
                     record_length=record_length_loc, &
                     recl_words=nx_global*ny_global)

   call data_set(restart_file, 'open_read')

   i_dim = construct_io_dim('i', nx_global)
   j_dim = construct_io_dim('j', ny_global)
   k_dim = construct_io_dim('k', size(FIELD,3))

   FIELD_DESC =                                       &
      construct_io_field(trim(fieldname),             &
                         dim1=i_dim,                  &
                         dim2=j_dim,                  &
                         dim3=k_dim,                  &
                         grid_loc ='3111',            &
                         d3d_array = FIELD)

   call data_set (restart_file, 'define', FIELD_DESC)

   call data_set (restart_file, 'read', FIELD_DESC)

   call destroy_io_field (FIELD_DESC)

   call data_set (restart_file, 'close')

   call destroy_file (restart_file)

!-----------------------------------------------------------------------
!EOC

 end subroutine read_field_3D


end module ecosys_nd_mod

!|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
