function palaeotripfluxes(modeloutput,metadata)
% Write out model output fluxes into the PALAEOTRIP format. 
% 
% Syntax:
% palaeotripfluxes(modeloutput,metadata)
%
% Inputs: 
%  modeloutput : Structure array containing model output. See below for
%                instructions on how to fill this. 
%  metadata    : Structure array containing user and model run metadata.
%                See below for instructions on how to fill this.
% Example:
%  palaeotripfluxes(modeloutput,metadata)
% 
% Information on structures
%  - If you are unfamiliar with matlab stuctures: these are a really
%  useful and easy to use matlab data type. Have a quick look at the help
%  at http://www.mathworks.com/help/matlab/structures.html
% 
% - modeloutput should contain the following fields (e.g. modeloutput.plevel). 
%    Quantities on levels (those marked * are required). 
%    plevel*  : pressure on levels (layer bountaries) (Pa)
%    Fswdndir : direct solar flux down (W/m/m)
%    Fswdndif : difuse solar flux down (W/m/m)
%    Fswdn    : total solar flux down (Fswdndir+Fswdndif) (W/m/m)
%    Fswup    : solar flux up (W/m/m)
%    Fswnet*  : net solar flux (Fswdn-Fswup) (W/m/m)
%    Flwdn    : thermal flux down (W/m/m)
%    Flwup    : thermal flux up (W/m/m)
%    Flwnet*  : net thermal flux (Flwdn-Flwup) (W/m/m)
%    Quantities on layers (all should be included if any are)
%    player   : pressure at layer centre (Pa)
%    Qsolar   : solar heating rate (K/day)
%    Qtherm   : thermal heating rate (K/day)
%   If the model you are running does not supply data for a field, leave
%   that field empty. 
% 
% - metadata should contain the following fields (e.g. metadata.runcode)
%    runcode*  : String containing the code name of run. See protocol for codes.
%    modelname*: String containing the name (and version number) of model. 
%    username* : String containing your name  (e.g. 'Colin Goldblatt')
%    useremail*: String containing your email (e.g. 'czg@uvic.ca')
%    usernotes*: String containing any notes about this run

Quantlevel = {'plevel','Fswdndir','Fswdndif','Fswdn','Fswup','Fswnet','Flwdn','Flwup','Flwnet'};
Quantlayer = {'player', 'Qsolar', 'Qtherm'};
Quantreq = {'plevel', 'Fswnet', 'Flwnet'};
Quantmeta = {'runcode', 'modelname', 'username', 'useremail', 'usernotes'};
nQuantlevel = length(Quantlevel);
nQuantlayer = length(Quantlayer);
nQuantreq =  length(Quantreq);
nQuantmeta = length(Quantmeta);

% check for required quantities
for ijk = 1:nQuantreq
    if ~isfield(modeloutput, Quantreq{ijk})
        error(['Required quantity ',Quantreq{ijk},' not supplied']);
    end
end

% check whether layers are included
layeroutput = isfield(modeloutput,'player');

% check vectors are all the right way up, flip if not
if modeloutput.plevel(1) > modeloutput.plevel(end)
    for ijk = 1:nQuantlevel
        modeloutput.(Quantlevel{ijk}) = flip(modeloutput.(Quantlevel{ijk}));
    end
end
if layeroutput
    if modeloutput.player(1) > modeloutput.player(end)
        for ijk = 1:nQuantlayer
            modeloutput.(Quantlayer{ijk}) = flip(modeloutput.(Quantlayer{ijk}));
        end
    end
end

% Fill empty fields with null value
nullval = -999.999999;
nullvectorlevel = nullval*ones(size(modeloutput.plevel));
if layeroutput
    nullvectorlayer = nullval*ones(size(modeloutput.player));
end

for ijk = 1:nQuantlevel
    if ~isfield(modeloutput,Quantlevel{ijk})
        modeloutput.(Quantlevel{ijk}) = nullvectorlevel;
    end
end

if layeroutput
    for ijk = 1:nQuantlayer
        if ~isfield(modeloutput,Quantlayer{ijk})
            modeloutput.(Quantlayer{ijk}) = nullvectorlayer;
        end 
    end
end

% write out the model output
outputformat = '%12.4f';
%outputstringlevel = repmat(outputformat,1,nQuantlevel);
%outputstringlayer = repmat(outputformat,1,nQuantlayer);
filenamelevel = ['palaeotrip_',metadata.runcode,'_level.txt'];
filenamelayer = ['palaeotrip_',metadata.runcode,'_layer.txt'];
fid = fopen(filenamelevel, 'w+'); 
% levels
% fileheader:
fprintf(fid,'P A L A E O T R I P   M O D E L   O U T P U T   F O R M A T');
fprintf(fid,'\n');
fprintf(fid,'palaeotrip.org');
fprintf(fid,'\n \n');

% metadata:
fprintf(fid,['runcode: ', metadata.runcode, '\n']);
fprintf(fid,['modelname: ', metadata.modelname, '\n']);
fprintf(fid,['username: ', metadata.username, '\n']);
fprintf(fid,['usernotes: ', metadata.usernotes, '\n \n']);

% dataheader:
fprintf(fid,' pressure    Fswdndir    Fswdndif    Fswdn       Fswup       Fswnet      Flwnd       Flwup       Flwnet     \n');
fprintf(fid,' (Pa)        (W/m/m)     (W/m/m)     (W/m/m)     (W/m/m)     (W/m/m)     (W/m/m)     (W/m/m)     (W/m/m)  \n');

for ii = 1:length(modeloutput.plevel);
    for jj = 1:nQuantlevel
        fprintf(fid,outputformat,modeloutput.(Quantlevel{jj})(ii));
    end
        fprintf(fid,'\n');
end
fclose(fid);

if layeroutput
    fid = fopen(filenamelayer); 
        % layers
        % fileheader:
        fprintf(fid,'P A L A E O T R I P   M O D E L   O U T P U T   F O R M A T');
        fprintf(fid,'\n');
        fprintf(fid,'palaeotrip.org');
        fprintf(fid,'\n \n');

        % metadata:
        fprintf(fid,['runcode: ', metadata.runcode, '\n']);
        fprintf(fid,['modelname: ', metadata.modelname, '\n']);
        fprintf(fid,['username: ', metadata.username, '\n']);
        fprintf(fid,['usernotes: ', metadata.usernotes, '\n \n']);

        % dataheader:
        fprintf(fid,' player      Qsolar      Qtherm     \n');
        fprintf(fid,' (Pa)        (K/day)     (K/day)    \n');
    
    for ii = 1:length(modeloutput.player);
        for jj = 1:nQuantlayer
            fprintf(fid,outputformat,modeloutput.(Quantlayer{jj})(ii));
        end
            fprintf(fid,'\n');
    end
        fclose(fid);
end
end