% function [Map_**km, Stack_**km, StackN_**km, nMap_**km, nStack_**km, nStackN_**km, Overflows] = Fill_**km(...
%   C, nMap_**km, nStack_**km, nStackN_**km, Hb_**km, SL_**km, Hi_**km, lmx_**km, lmy_**km, Depth, S_1km, IsSea)
% 
% Tijn Berends, 11-02-2016
%
% This function fills a depression at **km resolution. It uses the given
% Map "nMap_**km" and Stack "nStack_**km" as a starting point, to enable
% the use of the "shoreline memory" optimization.
% The input variables "lmx_**km", "lmy_**km" and "Depth" determine the
% absolute water level lying at [Depth] meters above the bedrock
% elevation Hb_**km(lmx_**km, lmy_**km).
%
% A "Map" array is a mask overlay of the region with the following possible values:
%    0 - non-filled pixel
%    1 - shoreline pixel
%    2 - filled pixel
% 
% A "Stack" array row contains the X- and Y-indices of a Map element, as
% well as a flag denoting whether or not it requires inspection:
% Stack_**km(i,:) = [X_index, Y_index, flag]
% flag = 1 means it requires inspection, flag = 0 means it's been checked
%
% In each WHILE loop, all elements of the Stack are inspected. If an
% element is found to lie below the water level, its corresponding Map
% element is marked as filled (i.e. its values is changed from 1 to 2), 
% it is removed from the Stack and all its non-filled, non-Stack neighbours
% are added to the Stack. If it does not lie below the water level, it is
% kept in the Stack but flagged as "inspected".
%
% The "PixelsAdded" variable counts how many pixels are filled in each
% WHILE loop. If, after checking the whole Stack, this value is equal to
% zero, it means no more pixels were added and the algorithm is therefor
% finished. The resulting Map and Stack are returned (nMap_**km,
% nStack_**km, nStackN_**km) as well as the starting Map and Stack
% (Map_**km, Stack_**km, StackN_**km).
%
% The "Overflows" condition is met when a newly added shoreline
% element is already part of the sea. This is only done when the "IsSea"
% flag is turned off - when filling the sea, obviously this condition should
% not apply.

function [Map_1km, Stack_1km, StackN_1km, nMap_1km, nStack_1km, nStackN_1km, Overflows] = Fill_1km(...
  C, nMap_1km, nStack_1km, nStackN_1km, Hb_1km, SL_1km, Hi_1km, lmx_1km, lmy_1km, Depth, S_1km, IsSea)

ice_density      = 960;
seawater_density = 1018;

% Move old Map and Stack arrays to memory arrays. If the lake is found to
% Overflow, these old ones should describe the deepest non-overflowing
% lake.

Map_1km       = nMap_1km;
Stack_1km     = nStack_1km;
StackN_1km    = nStackN_1km;

% The main WHILE loop.
PixelsAdded  = 1;
Overflows    = false;

while (PixelsAdded > 0 && (~Overflows))

  PixelsAdded = 0;
  
  % Check all elements in the Stack. The variable 'nStackN_**km' tracks how
  % many elements are listed in the Stack, so that the FOR-loop does not
  % have to cover the complete array every time. Note that nStackN_**km = 1
  % denotes an empty stack, so the FOR-loop goes from 1 to nStackN_**km-1
  for i = 1 : nStackN_1km-1

    % Check if the Stack element has already been inspected. If not,
    % inspect it.
    if (nStack_1km(i,3) == 1)

      % X index and Y index of Map element.
      xi = nStack_1km(i,1);
      yi = nStack_1km(i,2);

      % IF (Stack element lies below water level at that location). The
      % condition accounts for ice flotation and geoid perturbation
      % relative to the given local minimum.
      if ((Hb_1km(xi,yi) + (Hi_1km(xi,yi) * ice_density / seawater_density) - SL_1km(xi,yi)) < ...
                (Hb_1km(lmx_1km,lmy_1km) - SL_1km(lmx_1km,lmy_1km) + Depth))

        % Mark the corresponding Map element as filled.
        nMap_1km(xi,yi) = 2;

        % Note that a pixel has been added.
        PixelsAdded = PixelsAdded + 1;

        % The X and Y indices of all 8 neighbours of the newly filled
        % element.
        NeighbourX = [xi-1, xi-1, xi-1, xi,   xi,   xi+1, xi+1, xi+1];
        NeighbourY = [yi-1, yi,   yi+1, yi-1, yi+1, yi-1, yi,   yi+1];

        % We always remove 1 Stack element, but we can add 0 new ones (if
        % no Neighbours are not yet shoreline or filled elements), 1 new
        % one (if only 1 Neighbour satisfies this) or more than 1. The
        % 'IsFirst' variable is needed to make sure this is done properly.
        IsFirst  = true;
        for NeighbourI = 1: 8

          % X and Y index of the Neighbour in question.
          nx = NeighbourX(NeighbourI);
          ny = NeighbourY(NeighbourI);

          % Check if the Neighbour element is on the map.
          if (nx >= 1 && nx <= C.NX1 && ny >= 1 && ny <= C.NY1)
            % Check if it's not already a shoreline or filled element.
            if (nMap_1km(nx,ny) == 0)

              % Mark it as a shoreline element on the Map.
              nMap_1km(nx,ny) = 1;

              if (IsFirst)
                % First overwrite the current Stack element with the Neighbour.
                nStack_1km(i,:) = [nx, ny, 1];
                IsFirst = false;
              else
                % Addd any other Neighbours to the end of the Stack.
                nStack_1km(nStackN_1km,:) = [nx, ny, 1];
                nStackN_1km = nStackN_1km + 1;
              end

              % Check if the lake overflows, either by touching the edge of
              % the map or by touching the sea.
              if (~IsSea)
                if (nx == 2 || nx == C.NX1-1 || ny == 2 || ny == C.NY1-1)
                  Overflows = true;
                else
                  % This is the only line that is really different for
                  % different resolution fill routines - it always checks
                  % against the high-resolution sea depth field, so make
                  % sure the change the block indices accordingly.
                  if (S_1km(nx,ny) > 0)
                    Overflows = true;
                  end
                end
              end

            end
          end     
        end

        if (IsFirst)
          % If no neighbours were added, move the last element of the Stack to
          % the newly available slot at [i].
          nStack_1km(i,:) = nStack_1km(nStackN_1km-1,:);
          nStack_1km(nStackN_1km-1,:) = [0, 0, 0];
          nStackN_1km = nStackN_1km - 1;
        end

      else
        % The inspected Stack element does not lie below the water level.
        % Keep it in the Stack but flag it as "inspected".
        nStack_1km(i,3) = 0;
      end

    end

    % If the lake Overflows, stop filling it
    if (Overflows)
      break
    end

  end
end