PROGRAM EVUOT95
 !Based on EVUOT.f written in FORTRAN 77 by PV-JUL87 and others at the Finnish Meteorological Institute (FMI)
 !this program was rewritten in FORTRAN 95 by John Backman (FMI) Oct-2015
 !comments added to aid the readability of the program by John Backman
 IMPLICIT NONE

 !declaration variables
 INTEGER :: i,j,stat
 INTEGER, PARAMETER :: columns = 27

 !variables that are read from the control file
 !cla = latitude, clo = longitude, cz0 = roughness length, chu = wind speed measurement height (as calculated by sermod.f)
 REAL :: CLA,CLO,CZ0,CHU
 REAL, DIMENSION (10) :: ALB
 !number of albedo classes in the file
 INTEGER, PARAMETER :: numberOfAlbedos = 10

 !declaration of input variables that are read from the data file
 REAL :: CFNR,CFST,CFVU,CFKU,CFPV,CFKL,CFPR,CFT2, &
         CFUU,CFSU,CFNN,CFTD,CFTW,CFRR,CFVV,CFWW, &
         CFW1,CFW2,CFNH,CFCL,CFHH,CFCM,CFCH,CFDD, &
         CFUA,CFAP,CFAS

 !declaration of variables returned by the main subroutine DO_EVUOT
 !the results are returned to this vector
 REAL, DIMENSION (10) :: res 

 !vector that stores the data for writing
 REAL, DIMENSION (36) :: writeBuffer
 
 !test variables
 REAL :: CFAL


 !read control file inputs
 OPEN(100,file='control.evuot.kumpula',action='read')
 !cla = latitude, clo = longitude, cz0 = roughness length, 
 !chu = wind speed measurement height (as calculated by sermod.f)
 READ(100,*) CLA,CLO,CZ0,CHU,(ALB(i),i=1,numberOfAlbedos)

 !read the input data
 OPEN(200,file='input.txt',action='read')

 !the results go into this file
 OPEN(300,file='evuot.res',action='write',status='replace')
 !format of the data to be written
 301 FORMAT(3I5,2I3,I5,3F8.2,I3,F6.3,2F8.2,F7.1,I6, &
            I4,2I3,F6.3,I3,I5,2I3,F6.1,F7.2,F6.1, &
            F7.2,F8.2,F9.2,F5.1,6F12.3)


 !read the file, one line at the time untill the end of the file (not the fastest way, I know)
 DO 
	 !read the data for the calculations
	 READ(200,*,iostat=stat) CFNR,CFST,CFVU,CFKU,CFPV,CFKL,CFPR,CFT2, &
	             CFUU,CFSU,CFNN,CFTD,CFTW,CFRR,CFVV,CFWW, &
	             CFW1,CFW2,CFNH,CFCL,CFHH,CFCM,CFCH,CFDD, &
	             CFUA,CFAP,CFAS

	 !Added for differentiation of the moisture parameter
	 CFAL = 1
        
         !if you are at the end, then exit
         IF (stat /= 0) EXIT

	 !this subroutine does the actual work and returns the 
	 !results in the vector named res with the following variables
         !(/ CFAB,CFAK,CFAS,CFAL,CFPL,CFTT,CFUT,CFH0,CFQT,CFLE /)
	 CALL DO_EVUOT(CFNR,CFST,CFVU,CFKU,CFPV,CFKL,CFPR,CFT2, &
	               CFUU,CFSU,CFNN,CFTD,CFTW,CFRR,CFVV,CFWW, &
	               CFW1,CFW2,CFNH,CFCL,CFHH,CFCM,CFCH,CFDD, &
	               CFUA,CFAP,CFAS,CFAL,ALB,CLA,CLO,CZ0,CHU,res)
	
	 writeBuffer = (/ CFNR,CFST,CFVU,CFKU,CFPV,CFKL,CFPR,CFT2, &
                          CFUU,CFSU,CFNN,CFTD,CFTW,CFRR,CFVV,CFWW, &
                          CFW1,CFW2,CFNH,CFCL,CFHH,CFCM,CFCH,CFDD, &
                          CFUA,CFAP,res /)

         !write the data and the results to the file
	 write(300,301) (INT(writeBuffer(j)),j=1,6),(writeBuffer(j),j=7,9),INT(writeBuffer(10)), &
                        (writeBuffer(j),j=11,14),(INT(writeBuffer(j)),j=15,18),writeBuffer(19),  &
                        (INT(writeBuffer(j)),j=20,23),(writeBuffer(j),j=24,36) 
         
 END DO
 !close the files
 CLOSE(100)
 CLOSE(200)
 CLOSE(300)

END PROGRAM EVUOT95

!the acutal subroutine that does the work ----------------------------->
SUBROUTINE DO_EVUOT(CFNR,CFST,CFVU,CFKU,CFPV,CFKL,CFPR,CFT2, &
                    CFUU,CFSU,CFNN,CFTD,CFTW,CFRR,CFVV,CFWW, &
                    CFW1,CFW2,CFNH,CFCL,CFHH,CFCM,CFCH,CFDD, &
                    CFUA,CFAP,CFAS,CFAL,ALB,CLA,CLO,CZ0,CHU,results)
 IMPLICIT NONE
 !input data
 REAL, INTENT(IN) :: CFNR,CFST,CFVU,CFKU,CFPV,CFKL,CFPR,CFT2, & 
                     CFUU,CFSU,CFNN,CFTD,CFTW,CFRR,CFVV,CFWW, &
                     CFW1,CFW2,CFNH,CFCL,CFHH,CFCM,CFCH,CFDD, &
                     CFUA,CFAP,CFAS,CFAL,CLA,CLO,CZ0,CHU
 REAL, DIMENSION (10), INTENT(IN) :: ALB
 !output data
 REAL, DIMENSION (10), INTENT(OUT) :: results
 !local variables 
 REAL :: CFAB,CFAK,CFPL,CFTT,CFUT,CFH0,CFQT,CFLE,CFH1, &
         CFH2,CFWT,CFGR,CF0,CF1,CF2,CF3,CF4,CLRAD,UN
 REAL :: SG,C1,PP,SI,SN,CS,S,TR,T0,A,RV,D,PL,deltaTheta, &
         deltaTheta_dx,X,DX,E,TC,B0,GR,X1,X2
 INTEGER :: IV,IK,IP,IL,i,n

 !match the surface albedo with the ground surface quality/state (CFSU 0-9) to the ten albedo values in the control file 
 CFAB=ALB(INT(CFSU)+1)
 !Stefan Boltsman's constant 5.67e-8 W/(m^2 K)
 SG=.567/10000000.0
 !Emprirical constant from van Ulden and Holtslag 1985 (Journal of Climate an Applied Meteorology) 9.35e-6 K⁻²
 C1=.935/100000.0
 IV=INT(CFVU)
 IK=INT(CFKU)
 IP=INT(CFPV)
 IL=INT(CFKL)

 !SALFA returns the sun elevation angle in degrees and calculates various help variables such as azimuth and zenith angles
 !Input: IV=year (yy), IK=Month (mm), IP=Day (dd), IL=clock (HHMM), CLA=lat, CLO=lon, 
 !Output("dummy arguments"): PP=solar midday difference in hours, SI=sin of the elevation angle, SI= sin of the azimut angle,
 !CS=cos of the azimut angle
 !CFAK=solar elevation in degrees
 !CALL to get the value for CFAK
 CALL SALFA(IV,IK,IP,IL,CLA,CLO,PP,SI,SN,CS,CFAK)

 !The subroutine returns the ground moisture parameters according to precipitation codes for the present 
 !(CFWW), past 3 hours (CFW2) and past hour (CFW1). 
 !If the CFW1 and CFW2 values are missing --> moisture parameter seems to become 1.
 !CALL to get the value for CFAL
 !UNCOMMENTED TO TEST SENSITIVITY OF MOISTURE PARAMETER
 !CALL ALFM(CFAK,CFSU,CFWW,CFW1,CFW2,CFRR,CFAL)

 !SI=sinus of the solar elevation angle produced by the subroutine SALFA()
 !CFAP = hourly amount of sunshine
 !get the value for CFAS --> presenty prohibited by declaring CFAS as INTENT(IN)
 !if you want to use CFAS declare CFAS as INTENT(INOUT) for the subroutine
 !CALL GLOB(SI,CFAP,CFAS)

 !The term CFAS*(1.0-CFAB) = K* (Net shortwave radiation) = (1-albedo)*R_S(K_R) (Eq. 10 in Karppinen et al. 1997)
 !CF2 = K* + L_c
 CF2=CFAS*(1.0-CFAB)+CLRAD(CFT2,CFNH,CFHH,CFSU)

 !S is the slope of the saturation entalphy curve (sat vap pres curve). CFT2 is the temperature at 2 metres.  
 !Karppinen et al. 1997 (Int. J. Env and Poll.) between Eq. 5 and Table 1
 !The equation is not exactly the same though: T2^2 here and T2 in Karppinen et al. 1997 
 !Equation calculated as suggested by Dop et al. 1982:
 S=51371.0*EXP(5423.0*(1.0/273.16-1.0/CFT2))/(CFT2*CFT2)

 !Eq. 6 in Karppinen et al. 1997 (Int. J. Env and Poll.) but the density of air is scaled to the temperature CFT2
 !349477.0 = c_p*rho_ref*288 K
 !The density of air at the temperature T2 is rho(T2) = rho_ref*T_ref/T2 --> into Equation below 
 !--> (aS/(S+1)-1)/c_p * (T2/T_ref*rho_ref) where T_ref ~ 288 K and rho_ref ~ 1.2 kg/m³ and c_p ~ 1011 kJ/(kg*K)
 !Then CF3 = (aS/(S+1)-1)/(rho(T)*c_p)
 CF3=CFT2*(CFAL*S/(S+1.0)-1.0)/349477.0

 !ln(z/z_0) 
 !CHU = wind speed measurement height, CZ0 = roughness parameter
 CF0=ALOG(CHU/CZ0)

 !u* friction velocity (UN) calculated without stability functions, Eq. 1 Karppinen et al. 1997
 !CFUA = wind speed at 10 metres
 UN=.4*CFUA/CF0
 !Subtract the temperature change from 2 meters to 50 meters --> 0.01K/m * (50m-2m) = 0.48K
 TR=CFT2-.48
 !This is the temperature at 0 meters using the lapse rate of 0.01K/m --> +0.02C for a drop in height by 2 meters
 T0=CFT2+0.02

 !Eq. 21 in van Ulden and Holtslag 1985 (Journal of Climate an Applied Meteorology)
 !A = L_out - L_in = SG*T_0⁴ - SG*c_1*T_r⁶ (positive is upwards)
 !SG*T_0⁴ is the outgoing longwave radiation. SG*c_1*T_r⁶ is the incomming longwave radiation from the atmosphere.
 !A=-L_net  + first aprrox for G (~2.5 Wm⁻²) !could at least be changed to a diurnal variation of some sort
 !T0**4 and TR**6 for easier reading. This produces the same result as in the F77 version of EVUOT to the 4th decimal
 A=SG*(T0**4-C1*TR**6)+2.5

 !CF4=-Lnet+G-thetad*alfa*UN*rho*cp/(alfa(s/(s+1))-1)
 CF4=A-0.033*CFAL*UN/CF3

 !3.924 = 9.81*0.4 = g*k
 !RV = g*k*CF3/(T_R*u*³) when CF3 = (aS/(S+1)-1)/(rho(T)*c_p) then RV = g*k*(aS/(S+1)-1)/(Tr*u*³*rho*c_p)
 RV=3.924*CF3/(TR*UN*UN*UN)

 !NOTE: more important : (CF2-CF4)*alfa(s/(s+1)-1) ~ -lambdaE0 +(Q*-G) ~ H0 -> stability can be estimated from the sign 
 !CF2-CF4: CF2 = K*+L_c, CF4 = L_out - L_in + G - thetad*a*u**rho*c_p/(a*S/(S+1)-1)
 !CF2-CF4 = K*+L_c - L_out + L_in - G + thetad*a*u**rho*c_p/(a*S/(S+1)-1)
 !K* + L_c + L_in-L_out = Q*
 !CF2-CF4 = Q* - G  + thetad*a*u**rho*c_p/(a*S/(S+1)-1) = -H_0/(a*S/(S+1)-1)
 !alternatively a more clear form would be: (CF2 - CF4)*(a*S(S+1)-1) = (a*S(S+1)-1)*(Q*-G)+thetad*a*u**rho*c_p 
 !---> = (a*S(S+1)-1)(Q*-G)-(Q*-G)+thetad*a*u**rho*c_p = (a*S(S+1)-1)(Q*-G)+thetad*a*u**rho*c_p-(Q*-G) = lambdaE-(Q*-G) = -H_0
 !Note that aS/(S+1) < 1 and therefore aS/(S+1)-1 < 0. The sign does not change with aS/(S+1)-1
 
 !stable
 IF (CF2-CF4 .LT. 0) THEN
        !D = 1/L --> CF2-CF4 = -H_0/(aS/(S+1)-1), RV = g*k*(aS/(S+1)-1)/(T*u*³*rho*c_p). When L=T*u*²/(k*g*theta*) and 
        !theta* = -H_0/(rho*c_p*u*) --> 1/L = -k*g*H_0/(T*u*³*rho*c_p) = RV*(CF2-CF4)
        !D = 1/L inverse M-O length
	D=RV*(CF2-CF4)

	DO i=1,10
                !increase the inverse M-O length untill delta_theta* = T*u*²/(k*g*L) - (aS/(S+1)-1)(Q*-G)/(rho*c_p*u*)-theta_d*a
                !changes sign i.e. becomes positive
		PL=D*i
		CALL DPLS(PL,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL,CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta)
		IF (deltaTheta .GT. 0) THEN
			EXIT
		END IF
        END DO

	!increments and tolerance for the loop
	X=PL
	DX=0.00001
	E=0.000001

	!iterative loop
        DO n=1,20
		!iterative loop --> change 1/L untill the temperature scale difference as returned in DPLS is less than E 
		!with no more than 20 iterations. The variables CFUT, CFQT, CFTT, CFH0, and CFLE get their values here.
		!check the subroutine to see which wariables are inputs and outputs
		CALL DPLS(X,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL,CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta)
		IF (n .GT. 20 .OR. ABS(deltaTheta) .LT. E) THEN
			EXIT
		END IF
		CALL DPLS(X+DX,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL,CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta_dx)
		!change 1/L, here X, in the right direction X2 = X1-DX*deltaTheta/(deltaTheta_dx-deltaTheta). 
		!If the difference deltaTheta increases, decrease X, otherwise increase X. 
		!Note that both deltaTheta = DPLS(X) and deltaTheta_dx = DPLS(X+DX) decreases with each iteration.
		X=X-DX*deltaTheta/(deltaTheta_dx-deltaTheta)
	END DO
	!Finaly set the value for the inverse M-O length...
	CFPL=X
 !Unstable
 ELSE IF (CF2-CF4 .GT. 0) THEN
	!temperature in celcius	
	TC=CFT2-273.16

	!Empirical regression equations (B0 and GR, reference?) to find where to start the iteration to find 1/L
	!CFAL = moisture parameter, CF0 = ln(z/z_0), TC = temperature in Celsius
	!Does B0 refere to boyant production of TKE?
	B0=.0429*CFAL-2.1235-(.012685+.046081/(CFAL-4.2765))* &
           TC+(.19806+.00962*CFAL-(.00093463+.0013998/(CFAL- &
           2.1069))*TC)*CF0

	!Does GR refere to flux into the ground?
        GR=.49957+.0002*TC-(.001465+.000237*TC)*CFAL+(.0029854 &
           -.0007136*CFAL+(.000065522+.000077531/(CFAL-1.828)) &
           *TC)/CZ0 

	!X = 1/L --> CF2-CF4 = -H_0/(aS/(S+1)-1), RV = g*k*(aS/(S+1)-1)/(T*u*³*rho*c_p). When L=T*u*²/(k*g*theta*) and 
	!theta* = -H_0/(rho*c_p*u*) --> 1/L = -k*g*H_0/(T*u*³*rho*c_p) = RV*(CF2-CF4)
	!X = 1/L inverse M-O length
	X=RV*(CF2-CF4)
	!RV is always negative because of CF3 in RV since (aS/(S+1)-1) < 0 and therefore X1 is always negative ---> 
	!needs to be so because it is in a logarithm in the following equation
	X1=1000.0*RV
	X2=X/X1

	!empirical formula to find the best guess for 1/L based on moisture parameter (CFAL), temperature (TC), 
	!roughness length (CZ0) and CF0
	X=X/(1.0-X/(EXP(B0+GR*ALOG(-X1))*X2*(.45042*X2+ &
          .81505)/(X2+.26547)))

	!initiation of the loop
	DX=-0.00001
        E=0.000001
	!iterative loop
	DO n=1,20
		!check the subroutine DPLL to see which variables are inputs and outputs
		!DPLL returns the difference between theta* from the profile method and theta* from the energy balance equations.
		CALL DPLL(X,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL,CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta)
		IF (n .GT. 20 .OR. ABS(deltaTheta) .LT. E) THEN
			EXIT
		END IF
		!change 1/L, here X, until the temperature scale difference as returned in DPLL is less than E with no more than
		!20 iterations. Note that DX is negative here.
		CALL DPLL(X+DX,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL,CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta_dx)
		X=X-DX*deltaTheta/(deltaTheta_dx-deltaTheta)
	END DO
	CFPL=X
 !None of the above holds true
 ELSE	
	!Set the value for the inverse M-O length = 0
	CFPL=0.0
	!sensible heat flux H0 = 0	
	CFH0=0
	!CF2 = K* + L_c and -A = L_net - G --> CFLE = K* + L_c + L_net - G  = Q* - G ----> 
	!H0 + lambdaE = Q* - G ---> when H0=0 then lambdaE = Q* - G
	CFLE=CF2-A
	!Q* = lambdaE-G + 0.05*z_0
	CFQT=CFLE-2.5+0.05*CZ0
	!+ other variables?! CFTT, CFUT
 END IF

 !return the variables
 results(1:10) = (/ CFAB,CFAK,CFAS,CFAL,CFPL,CFTT,CFUT,CFH0,CFQT,CFLE /)
 

END SUBROUTINE DO_EVUOT
!END OF the acutal subroutine that does the work <---------------------

!OTHER SUBROUTINES

SUBROUTINE SALFA(IVV,IKK,IPA,IKLO,PLAT,PPIT,PPP,SINAL, & 
           SINFII,COSFII,CFAK)
 IMPLICIT NONE
 !input variables
 INTEGER, INTENT(IN) :: IVV,IKK,IPA,IKLO
 REAL, INTENT(IN) :: PLAT,PPIT
 !output variables
 REAL, INTENT(OUT) :: PPP,SINAL,SINFII,COSFII,CFAK
 !local variables
 INTEGER, DIMENSION(12) :: IPAIV
 INTEGER :: IPV
 REAL :: RAD,RLAT,RPV,S,DECL,RDECL,PP,RL,DKLO,H,HH

 !days passed when the month changes
 IPAIV = (/ 0,31,59,90,120,151,181,212,243,273,304,334 /)
 !radians / degrees
 RAD=0.0174522925
 !latitude to radians
 RLAT=RAD*PLAT
 !leap year check
 IPV=IPA+IPAIV(IKK)
 IF (IKK .GE. 3 .AND. MOD(IVV,4) .EQ. 0) THEN
	IPV=IPV+1
 END IF
 !day to float
 RPV=REAL(IPV)
 S=0.0171672
 !declination in degrees
 DECL=0.33281+3.7872*SIN(S*RPV)-22.984*COS(S*RPV)
 !declination in radians
 RDECL=RAD*DECL
 PP=-0.1229*SIN(S*RPV)+0.0072*COS(S*RPV) &
    -0.1565*SIN(2.0*S*RPV)-0.0528*COS(2.0*S*RPV)
 RL=(24.0/360.0)*(30.0-PPIT)
 PPP=PP-RL
 DKLO=(REAL(IKLO-IKLO/100*100)/60.0)+REAL(IKLO/100)
 H=(360.0/24.0)*(DKLO-12.0+PPP)
 HH=RAD*H
 SINAL=SIN(RDECL)*SIN(RLAT)+COS(RDECL)*COS(RLAT)*COS(HH)
 SINFII=(COS(RDECL)/SQRT(1.0-SINAL**2))*SIN(HH)
 COSFII=(SINAL*SIN(RLAT)-SIN(RDECL))/(SQRT(1-SINAL**2)*COS(RLAT))
 !solar elevation in degrees
 CFAK=ASIN(SINAL)*57.29578
END SUBROUTINE SALFA


SUBROUTINE ALFM(CFAK,CFSU,CFWW,CFW1,CFW2,CFRR,CFAL)
!The subroutine calculates the moisture parameter CFAL from:
!CFAK solar elevation angle
!CFSU state of the ground (dry, humid, wet, snow, ....)
!CFWW weater now
!CFW1 weather previous hour
!CFW2 weather previous 3 hours
!CFRR rain in mm/12h
!ALFM = Priestley-Taylor parameter Table 1 in Karppinen et al. 1997 (Int. J. Env and Poll.)
!The logic of this function requires one to understand how the weather codes are dealth with in 
!sermod.f, which is tricky.

!CFSU is the state of the ground: 
!0=dry, 1=humid, 2=wet, 3=dry icy, 4=wet icy, 5=snow or melting snow <50%, 
!6=snow or melting snow >50%, 7=snow or melting snow 100% snow 
!cover, 8= 50% < x <100% dry snow cover, 9=100% dry snow cover

 IMPLICIT NONE
 !input
 REAL, INTENT(IN) :: CFAK,CFSU,CFWW,CFW1,CFW2,CFRR
 !output
 REAL, INTENT(OUT) ::  CFAL
 IF (CFAK .LE. 0.0 .OR. CFSU .GT. 4 .OR. CFSU .EQ. 2) THEN
	!if solar elevation < 0, ground state covered by snow or the ground is wet, 
	!i.e. continue with dry, humid or dry ice, 
	!otherwise let the moisture parameter equal unity
	CFAL = 1.0
 ELSE
	!CFWW is the weather synop code (http://weather.unisys.com/wxp/Appendices/Formats/SYNOP.html)
	!if the weather code (0-99) is in the range 50 - 99 (drizzle to heavy rain) let the moisture 
	!parameter equal 1
	IF (CFWW .GT. 49.0 .AND. CFWW .LT. 100.0) THEN
		CFAL = 1.0
	ELSE	
		!Continue on and set the value for presipitation that has been observed earlier as 
		!stated by the present weather synop code. If the weather code is in the range of 
		!20-29 (presipitation in some form or theother during the last hour but not at present) 
		!set the moisture parameter to 0.9
		IF (CFWW .GT. 19.0 .AND. CFWW .LT. 30.0) THEN 
		      CFAL=0.9
		ELSE	
			!You've ended up here if there is no presipitation at the moment, not indicated 
			!that there has been in the last hour. The weather of the past 3 hours = CFW2 
			!as produced by sermod.f. The code is a weather code in the range 20<W2<30 - 19 
			!= 1-10). If 4<W2<10 is equivalent to synop code 23-29 (rain and snow to fog 
			!during the last 3 hours) set the moisture parameter to 0.8
			IF (CFW2 .GT. 4.0 .AND. CFW2 .LT. 10.0) THEN
				CFAL=0.8
			ELSE	
				!If W2<4 then you will end up here and that means that the weather in 
				!the past 3 hours has been drizzle, rain or snow (synop codes 20,21,22)
				!Then, do the same for CFW1 as for CFW2. If the weather of the past hour 
				!(CFW1) (from sermod.f weather code in the range 20<W2<30 - 19) has 
				!involved precipitation but not during the entire 3 hour period. You are 
				!there only if the sky is and has no trace of rain at the monent (CFWW) 
				!or during the whole 3 hour period i.e. a brief shower of some sort. 
				IF (CFW1 .GT. 4.0 .AND. CFW1 .LT. 10.0) THEN
					CFAL=0.7
				ELSE
					!If the amount of rain during the last 12 hours exceeds 5 mm set
					!the moisture parameter to 0.6
					IF (CFRR .GT. 5.0) THEN
						CFAL=0.6
					ELSE
						!Else let the moisture parameter be 0.5
						CFAL=0.5
					END IF
				END IF
			END IF
		END IF
	END IF
 END IF
END SUBROUTINE ALFM

!ASIN replaced by SI to avoid confusion with inherent arcus sinus a.k.a. ASIN
SUBROUTINE GLOB(SI,AP,CFAS)
!Calculates the incoming solar radiatio (W/M**2) from:
!SI which is the sinus of the solar elevation angle
!AP (passed to the routine as CFAP) is the hourly amount of sunshine [0-1]
!This subroutine is not called at the moment because measurements of incoming 
!solar radiation are available
 IMPLICIT NONE
 !input
 REAL, INTENT(IN) :: SI,AP
 !output
 REAL, INTENT(OUT) :: CFAS
 !local
 REAL :: R
 IF (SI .LE. 0.0) THEN 
 	CFAS = 0
 ELSE
	!Eq 11 in Karppinen et al. 1997 --> clear sky radiation (K_R) as a fucntion of solar 
	!elevation angle (SI)
	R=1372.0*(.987-.0909/(SI+.118))*SI
	!Eq. 12 in Karppinen et al. 1997 -> R_S(K_R). At high latitudes the net radiation 
	!correlates better with sunshine duration than cloud cover
	CFAS=R*((.66013*AP*AP-.92003*AP-.043089)/ &
             (AP*AP-1.1928*AP-.14903)-.001*R* &
             (.33472*AP*AP-.38351*AP-.0070213)/ &
             (AP*AP-1.6121*AP-.073352))
 END IF
END SUBROUTINE GLOB

!stable
SUBROUTINE DPLS(PL,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL,CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta)
 IMPLICIT NONE
 !input
 REAL, INTENT(IN) :: PL,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL
 !output
 REAL, INTENT(OUT) :: CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta
 !local
 REAL :: A,B,C,D,TR,T0,G

 !Below Eq. 1 in Karppinen et al. 1997
 !A=psi_m(z_0/L)-psi_m(z/L)
 A=17.0*(EXP(-.29*CZ0*PL)-EXP(-.29*CHU*PL))

 !Equation 9b in van Ulden and Holtslag 1985
 !B = -psi_h(z=48,L) = 5*z/L = 5*48/L = 240*1/L
 B=240.0*PL

 !Equation 9b in van Ulden and Holtslag 1985
 !C = -psi_h(z=50,L) = 5*z/L = 5*50/L = 250*1/L
 C=250.0*PL

 !Equation 8 in van Ulden and Holtslag 1985 for u*
 !u* = k*U(z)/(ln(z/z0)+psi_m(z_0/L)-psi_m(z/L))
 CFUT=.4*CFUA/(CF0+A)

 !D = u*²/(L*g*k*k) = theta*/(k*TR), Note: PL = 1/L
 D=CFUT*CFUT*PL/1.5696

 TR=(CFT2-0.48)/(1.0-D*(3.218875825+B))

 T0=TR-TR*D*(12.+C)+0.5

 !Eq. 21 from van Ulden and Holtslag 1985
 !T0**4 and TR**6 for easier reading. This produces the same result as in the F77 version of EVUOT to the 4th decimal
 CFQT=CF2-.567*(T0**4-.935*TR**6/100000.0)/10000000.0

 !ground flux estimated from the temperature difference between TR and T0
 G=-5.0*(TR-T0)

 !solve theta* from van Ulden and Holtslag 1985 Eq. 6 -->
 !theta* = TR*u*²/(k*g*L), CFTT = k * u*²/(L*g*k*k) * TR = TR*u*²/(L*g*k) = theta* (temperature scale)
 CFTT=.4*D*TR

 !rho(T2) = rho_ref*T_ref/T2
 !349477.0 = T_ref ~ 288 K and rho ref ~ 1.2 kg/m³ and c_p ~ 1011 kJ/(kg*K)
 !CFH0 = rho(T) * c_p * theta* * u*, Eq. 5 in van Ulden and Holtslag 1985
 CFH0=-349477.0*CFTT*CFUT/CFT2

 !lambdaE = Q_E = Q*-G-H_0, Eq. 10 in van Ulden and Holtslag 1985
 CFLE=CFQT-G-CFH0

 !deltaTheta = theta* - (aS/(S+1)-1)(Q*-G)/(rho*c_p*u*)-theta_d*a: the right hand term is Eq. 6 in Karppinen 
 !et al. 1997. CF3 = (aS/(S+1)-1)/(rho(T)*c_p)
 !deltaTheta is the difference between the temperature scale calculated from the energy budget equations 
 !and using the "profile method". The iteration is done by changing 1/L.
 deltaTheta=CFTT-CF3*(CFQT-G)/CFUT-.033*CFAL
END SUBROUTINE DPLS

!unstable
SUBROUTINE DPLL(PL,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL,CFUT,CFQT,CFTT,CFH0,CFLE,deltaTheta)
 IMPLICIT NONE
 !input
 REAL, INTENT(IN) :: PL,CZ0,CHU,CFUA,CF0,CF2,CF3,CFT2,CFAL
 !output
 REAL, INTENT(OUT) :: CFUT, CFQT, CFTT, CFH0, CFLE, deltaTheta
 !local
 REAL :: A,B,C,CC,D,TR,T0,G

 !Eq. 2 in Karppinen et al. 1997: A=-psi_m(z/L)+psi_m(z_0/L)
 !PL = 1/L, CZ0 roughness length, CHU = wind speed mesasurement height
 A=(1.0-16.0*CZ0*PL)**.25-(1.0-16.0*CHU*PL)**.25

 !psi_H = 2*alog([1+y²]/2)
 !C = 1+y² where y=(1-16*z/L)^(1/4) from van Ulden and Holtslag 1985 Eqs 9 and 9a: z3=50 meters and PL = 1/L
 C=1.0+SQRT(1.0-800.0*PL)

 !B = -psi_H(z3/L)+psi_H(z2/L) where psi_H = 2*log([1+y²]/2) and y=(1-16*z/L)^(1/4), z3=50 meters, z2 = 2 meters
 B=2.0*ALOG((1.0+SQRT(1.0-32.0*PL))/C)

 !CC = -2*log([1+y²]/2) = 2*log(2/[1+sqrt(1-16*50/L)])
 CC=2.0*ALOG(2.0/C)

 !CFUT = friction velocity from Eq. 1 in Karppinen et al. 1997
 !u* = U(z)*k/(ln(z/z_0)-psi_m(z/L)+psi_m(z_0/L)) where CF0=ln(z/z_0)
 CFUT=.4*CFUA/(CF0+A)

 !D = u*²/(L*g*k*k) = theta*/k, Note: PL = 1/L
 D=CFUT*CFUT*PL/1.5696 

 TR=(CFT2-0.48)/(1.0-D*(3.218875825+B))

 T0=TR-TR*D*(12.+CC)+0.5

 !Eq. 21 from van Ulden and Holtslag 1985
 !T0**4 and TR**6 for easier reading. This produces the same result as in the F77 version of EVUOT to the 4th decimal
 CFQT=CF2-.567*(T0**4-.935*TR**6/100000.0)/10000000.0

 !ground flux estimated from the temperature difference between TR and T0
 G=-5.0*(TR-T0)

 !solve theta* from van Ulden and Holtslag 1985 Eq. 6 -->
 !theta* = T*u*²/(k*g*L), CFTT = k * u*²/(L*g*k*k) * T = T*u*²/(L*g*k) = theta* (temperature scale)
 CFTT=.4*D*TR

 !rho(T2) = rho_ref*T_ref/T2
 !349477.0 = T_ref ~ 288 K and rho ref ~ 1.2 kg/m³ and c_p ~ 1011 kJ/(kg*K)
 !CFH0 = rho(T) * c_p * theta* * u*, Eq. 5 in van Ulden and Holtslag 1985
 CFH0=-349477.0*CFTT*CFUT/CFT2

 !lambdaE = Q_E = Q*-G-H_0, Eq. 10 in van Ulden and Holtslag 1985
 CFLE=CFQT-G-CFH0

 !DPLL = theta* - (aS/(S+1)-1)(Q*-G)/(rho*c_p*u*)-theta_d*a: the right hand term is Eq. 6 in Karppinen et al. 1997
 !CF3 = (aS/(S+1)-1)/(rho(T)*c_p)
 !deltaTheta is the difference between the temperature scale calculated from the energy budget equations and 
 !using the "profile method". The iteration is done by changing 1/L.
 deltaTheta=CFTT-CF3*(CFQT-G)/CFUT-.033*CFAL
END SUBROUTINE DPLL

!FUNCTION(S)

!CLRAD(CFT2,CFNH,CFHH,CFSU)
REAL FUNCTION CLRAD(T,C,H,SU)
 !Calculates the longwave radiation from clouds (W/m**2)
 !T (passed as CFT2) is the temperature at Z=2M
 !C (passed as CFNH) is the amount of low clouds [0-1]
 !H (passed as CFHH) is the height of the cloud base (metres)
 !SU (passed as CFSU) is the state of the ground INT([0 - 9])

 IMPLICIT NONE
 !input
 REAL, INTENT(IN) :: T,C,H,SU
 !local
 REAL, PARAMETER :: SG=.0000000567, C1=.00000935
 REAL :: F
 !No clouds, no radiation from them
 IF (C .LE. 0.0) THEN
  CLRAD=0.0
 ELSE
   !Cloud base temperature T_c = T-0.01*H where T is the temperature at 2 meters and H is the cloud base height
   !Eq. 14 in Karppinen et al. 1997 (Int. J. Env and Poll.)
   !F here is L'_c in Eq. 14
   F=C*SG*(1.0-C1*T*T)*(T-.01*H)**4
   IF (SU .LT. 5.0) THEN 
	!For no snowcover on the ground Eq. 15a in Karppinen et al. 1997 (Int. J. Env and Poll.)
	CLRAD=.64*F
   ELSE
	!For snow covered ground Eq. 15b in Karppinen et al. 1997 (Int. J. Env and Poll.)
	CLRAD=.56*F
   END IF
 END IF
END FUNCTION CLRAD
