# ======================================================================
# PURPOSE
# =========
# To load observations vector for a particular season and region. Optionally,
# to get pseudo-observations (for observation system simulation experiments).

# STATUS
# ========
# Working

# INPUT
# =======
# season => Season. String. e.g., "JAN", "DJF". 5 seasons and "JAN" are
#           currently implemented.
# region => NSW planning region. String. e.g., "Ill" for Illawarra.
# osse   => If TRUE, observation system simulation experiments are performed.
#           Pseudo-observations from model #truemodel are fetched. Default:
#           FALSE.
# truemodel => Model number to use for pseudo-observations. Ignored if
#              osse=FALSE. Integer. Default: NULL.

# CALLS
# =======
# get_regions from BMAutils.R
# load.reg.models from load_models.R

# OUTPUT
# ========
# myobs => list with the following components:
#   [[obs]] => Observations of JAN tasmean for a given  (or pseudo-obs.).
#              Vector of doubles. Units: deg. C. 
#   [[region]] => Region name. Character.
#   [[years]] => Vector of years corresponding to [[obs]]. Vector of integers.
#                The years correspond to the years where the most of the months
#                in a requested season belong. 

# HISTORY
# ========
# Sep 11 2015 => Written by Roman Olson, CCRC, UNSW.
# Jan 22 2016 => Added season.
# Feb 10 2016 => Implemented OSSEs.
# ======================================================================

load.obs <- function(season, region, osse=FALSE, truemodel=NULL) { 

   # SOURCE FUNCTIONS
   source("load_models.R")
   source("BMAutils.R")
    
   # PRELIMINARIES
   if (season == "DJF") {
      years <- seq(1991, 2009)
   } else {
      years <- seq(1990, 2009)
   }
   if (!osse && !is.null(truemodel)) {
       cat("osse=FALSE, ignoring truemodel\n")
   }
   if (osse && is.null(truemodel)) {
       stop("osse is TRUE: truemodel must be specified!\n")
   }
   
   # READ REGION INFORMATION
   inputdir  <- "input/tasmean/"
   regions   <- get.regions()
   regind    <- which(regions == region)[1]
   if (!is.finite(regind)) {
      stop("Oops! Invalid region name")
   }

   # EXTRACT OBSERVATIONS FOR THAT SEASON AND REGION
   if (!osse) { #Load actual observations
      obs.fname <- paste(inputdir, "Obs_1990_2009_", season,
                         "_tasmean_clim.txt", sep="")
      obsall    <- as.matrix(read.table(obs.fname))
      obs       <- unname(obsall[regind,])
   } else {#Load pseudo-observations
      obs <- load.reg.models(season, "hindcast", region)$allmodels[truemodel,]
   }
       

   # RETURN OUTPUT
   myobs <- list(obs=obs, region=region, years=years)
   myobs
}

#To test function
#library(RUnit)
#runTestFile("runit.load.obs.R")
