# ======================================================================
# PURPOSE
# =========
# To return state-average temperature data [row, col] = [model index, 
# time index]

# STATUS
# ========
# Working

# WARNING! If 'incomplete final line' warning appears go to states.txt, go
# to the very end of the file and press "ENTER".

# INPUT
# =======
# season => Season. String. e.g., "DJF". Currently four seasons and "JAN" are
#           implemented.
# period => Period. String. Either "hindcast" for 1990-2009 or "forecast" for
#           2060-2079
# region => NSW Planning Region. String. e.g., "Ill" for Illawara. 


# CALLS
# =======
# No externals

# OUTPUT
# ========
# out => list with the following components:
#   [[allmodels]] => Model output for given period and region for all models.
#                Matrix of floats.[row, col] = [model index, time index].
#                Units: deg. C. 
#   [[region]] => Region name. Character.
#   [[years]] => Vector of years corresponding to columns of [[model]].
#                Vector of integers. For DJF, these are years where the
#                majority of months in a season lie. It excludes partial
#                seasons.

# HISTORY
# ========
# Sep 11 2015 => Written by Roman Olson, CCRC, UNSW.
# Jan 22 2015 => Now works for different seasons.
# ======================================================================
load.reg.models <- function(season, period, region) {

   # PRELIMINARIES
   if (period == "hindcast") {
       yearschar <- "1990_2009"
       if (season == "DJF") { 
         years     <- seq(1991, 2009)
       } else {
         years     <- seq(1990, 2009)
       }
   } else if (period == "forecast") {
       yearschar <- "2060_2079"
       if (season == "DJF") { 
         years     <- seq(2061, 2079)
       } else {
         years     <- seq(2060, 2079)
       }
   } else {
       stop("Oops! Illegal period name")
   }
   if (season == "DJF") { 
      allmodels <- matrix(nrow=12, ncol=19)
   } else {
      allmodels <- matrix(nrow=12, ncol=20)
   }      
   inputdir  <- paste(Sys.getenv("BMADIR"), "input/tasmean/", sep="")
   regfile   <- paste(inputdir, "states.txt", sep="")
   regions   <- as.vector(as.matrix(read.table(regfile, sep=",",
                         colClasses="character", nrows=1, strip.white=TRUE)))
   regind    <- which(regions == region)[1]
   if (!is.finite(regind)) {
      stop("Oops! Invalid region name")
   }

   
   # ITERATE THROUGH MODELS AND FILL IN THE MATRIX
   for (mnum in 1:12) {
      cur.fname <- paste(inputdir, "M", mnum, "_", yearschar,
                         "_pp_", season, "_tasmean_clim.txt", sep="")
      cur.mall  <- as.matrix(read.table(cur.fname))
      allmodels[mnum, ] <- cur.mall[regind,]
   }

   # RETURN OUTPUT
   out <- list(allmodels=allmodels, region=region, years=years)
   out
}

# To test function
#library(RUnit)
#runTestFile(paste(Sys.getenv("BMADIR"),"runit.load.reg.models.R",
#                 sep=""))
