                            %% SCRIPT OBJECTIVES

% Estimate the flux of the gas across the air-water interface.

                        %% FLUX NUMERICAL OPTIONS

%OBJECTIVES: declares the numerical options to the flux calculus functions  

% Flux numerical scheme: 'Mol' or 'ppm'
NSflux = 'Mol';

% Surface Layer numerical scheme: 'Double' or 'Single'
NSlayer = 'Single';

% Flux units: (1) mol*m^-2*s^-1; (2) mmol*m^-2*h^-1; (3) mmol*m^-2*d^-1
NSfluxunits = 3;

% Henry constant correction for temperature and salinity
% 'Sar13': Sarmiento (2013) adapted from Weiss(1974), Weiss and Price(1980)
% 'Joh10': Johnson (2010)
NSkH = 'Joh10';

% Air transfer velocity (ka)
% 'Joh10': Johnson (2010)
% 'Dea91': Duce et al (1991)
% 'Lis73': Liss (1973)
% 'M&Y83': Mackay & Yeun (1983), for wind speeds between 5 and 22 m/s
% 'Sea02': Shahin et al 2002
% 'Jef10': COARE's Jeffrey et al (2010)
NSka = 'Jef10';

% Drag coefficient
% 'Dea91': Duce et al (1991);
% 'M&Y83': Mackay & Yeun (1983)
% 'Smi80': Smith (1980);
% 'T&Y01': Taylor and Yelland (2001)
NSdragcoef ='Smi80';

% method to estimate friction velocity (u*)
% 'WLLP': from Wind Log-Linear Profile
% 'iWLP': from Wind Log-Linear Profile iterated to convergence following COARE
% 'VHFx': from Vertical Heat Flux term in Monin-Obhukov Similarity theory
NSu_='iWLP';

% number of iterations in case of iterative Wind Log-Linear Profile method
niWLLP = 4;

% method to estimate the Monin-Obhukov Stability theory:
% 'L97Bu' : Lee (1997) adapted to Businger et al (1971)
% 'L97Dy' : Lee (1997) adapted to Dyer (1974)
% 'L97DG' : Lee (1997) adapted to Delage and Girard (1992)
% 'Lea04' : Lange et al(2004)
% 'Stu88' : Stull (1988)
NSmos = 'Stu88';

% method to estimate the bulk Richardson number:
% 'G&F97' : Grachev and Fairall (1997)
% 'Lee97' : Lee (1997)
% 'Stu88' : Stull (1988)
NSrib ='Stu88';
    
% Water transfer velocity (kwind)
% 'Bea04': Borges et al (2004)
% 'Cea96': Carini et al (1996)
% 'Jea87': Jhne et al (1987), dependent on u_
% 'L&M86': Liss & Merlivat (1986)
% 'M&Y83': Mackay & Yeun (1983), dependent on u_
% 'MY83W': Mackay & Yeun (1983) adapted from Wolff and Heijde (1982), dependent on u_
% 'McG01': McGillis et al (2001)
% 'Nea00': Nightingale et al (2000)
% 'R&C01': Raymond & Cole (2001)
% 'Sea07': Sweeney et al (2007)
% 'Wan92': Wanninkhof (1992)
% 'WMG99': Wanninkhof & McGillis (1999)
% 'Wea07': Weiss et al (2007)
% 'Zea03': Zhao et al (2003), dependent on u_
% 'ZoJ03': Zhao et al (2003) on data from Jhne et al (1985), dependent on u_
% 'none.': no kwind term
NSkwind = 'none.';

% Water-side transfer velocity bubble term (kbubble)
% 'W05va': Woolf (2005) with kinematic viscosity of air
% 'W05vw': Woolf (2005) with kinematic viscosity of water
% 'Zhg06': Zhang et al (2006)
% 'ZRb03': Zhao et al (2003) dependent on u_ and whitecap
% 'none.': no kbubble term
NSkbubble = 'ZRb03';

% Water-side transfer velocity current term (kcurrent)
% 'OCD58': O'Connor and Dobbins (1958) dependent on current and depth
NSkcurrent = 'OCD58';

% Use chemical enhancement from Wanninkhof (1992) ('yes' or 'no.')
chemen ='no.';

% Chemical enhancement of water transfer velocity (alfa)
% 'Bea04': Borges et al (2004)
% 'Cea96': Carini et al (1996)
% 'Joh10': Johnson (2010)
% 'R&C01': Raymond Cole (2001)
NSalfa = 'Joh10';

% Water difusivity scheme (Dw)
% 'H&L74': Hayduk & Laudie (1974)
% 'H&M82': Hayduk & Minhas (1982)
% 'W&C55': Wilkie & Chang  (1955)
NSdw = 'W&C55';

%Stores numerical options in a structure array (not to edit)
NumericScheme.alfa      = NSalfa;
NumericScheme.chem_enh  = chemen;
NumericScheme.Dw        = NSdw;
NumericScheme.DragCoef  = NSdragcoef;
NumericScheme.Flux      = NSflux;
NumericScheme.FluxUnits = NSfluxunits;
NumericScheme.kH        = NSkH;
NumericScheme.ka        = NSka;
NumericScheme.kbubble   = NSkbubble;
NumericScheme.kcurrent  = NSkcurrent;
NumericScheme.kwind     = NSkwind;
NumericScheme.Layer     = NSlayer;
NumericScheme.MOS       = NSmos;
NumericScheme.Rib       = NSrib;
NumericScheme.u_        = NSu_;

                              %% CONSTANTS

%OBJECTIVES: provides the constants for the gas flux model.

%--------------------------------------------------------------------------
                             %Air constants

% Specific Heat of air                                              J/(g*K)
cp = 1.005;                                                          

% gravitational acceleration                                          m/s^2
g = 9.80665;                                                           

% von Krmn's constant for friction velocity estimates
Karman = 0.4;

% Relative molecular mass of air
Ma = 28.97;

% Rho air parameters (in Johnson 2010)
Sd = [1.293393662 -5.538444326*10^-3 3.860201577*10^-5 -5.2536065*10^-7];

% Eta air parameters (in Johnson 2010)
Sv = [1.715747771*10^-5 4.722402075*10^-8 -3.663027156*10^-10 ...
      1.873236686*10^-12 -8.050218737*10^-14];

% Liquid molar volume of air                                       cm^3/mol
Va = 20.1;                                                          

%--------------------------------------------------------------------------
                            %Water constants

% A and B constants for roughness length (z0) estimated from wave height
%(Hs) and wave length (Lp) according to Taylor and Yelland (2001)
Aw=1200;
Bw=4.5;

% Relative molecular mass of H20
Ms=18.01;

% Mass fraction (MF) of solutes in standard sea water as a proportion of
% total salinity.
MassFraction= ...
{'Compound' 'MF'  'v1'     'v2'      'v3'     'v4'       'v5'      'v6'   ;
  'NaCl'    0.798  16.22    1.3229    1.4849   0.0074691  30.78    2.0583 ;
  'KCl'     0.022   6.4883  1.3175   -0.7785   0.09272    -1.3     2.0811 ;
  'CaCl2'   0.033  32.028   0.787792 -1.1495   0.0026995  780860   5.8442 ;
  'MgCl2'   0.047  24.032   2.2694    3.7108   0.021853   -1.1236  0.14474;
  'MgSO4'   0.100  72.269   2.2238    6.6037   0.0079004  3340.1   6.1304};

% Association factor of H20
phiw = 2.6;

% Rhow0 (pure water) parameters (in Johnson 2010)
r = [999.842594 0.06793952 -0.00909529 0.0001001685 -0.000001120083 ...
                                                        0.000000006536332];

% Rhow (marine water) parameters (in Johnson 2010)
a = [0.824493 -0.0040899 0.000076438 -0.00000082467 0.0000000053875];
b = [-0.00572466 0.00010277 -0.0000016546];
c = 0.00048314;

%--------------------------------------------------------------------------
                            %Gas constants
                               
%'a' and 'b' constants for temperature and salinity dependence of Bunsen
% solubility coeficient taken from Sarimiento and Gruber (2013)
as = [-160.7373   215.4152  89.892    -1.477592]; %for CO2
bs = [ 0.029941  -0.027455  0.0053407          ]; %for CO2
%as = [-68.8862    101.4956  28.7314   0        ]; %for CH4
%bs = [-0.076146   0.04397  -0.0068672          ]; %for CH4
%as = [-165.8806   222.8743  92.0792   -1.48425 ]; %for N2O
%bs = [-0.056235   0.031619 -0.0048472          ]; %for N2O
                            
% Henry's law constant in pure water at 25Celsius. K=c/p form.
kHcp = 0.034;      %(for CO2)                                 % mol/(L.atm)
%kHcp = 0.0014;    %(for CH4)                                  % mol/(L.atm)
%kHcp = 0.025;     %(for N2O)                                  % mol/(L.atm)

% Relative molecular mass
Mb = 44.01;           %(for CO2)
%Mb = 16.043;           %(for CH4)
%Mb = 44.013;           %(for N2O)

% Universal ideal gas law constant
R = 8.314472;                                               %Pa.m^3/(mol.K)

% Temperature dependence of the solution: -delta_soln*H/R (in Johnson 2010)
tds = 2400;         %(for CO2)
%tds = 1700;         %(for CH4)
%tds = 2600;         %(for N2O)

% Liquid molar volume at its boiling point using the additive Schroeder method:
% increments of C=7,O=7, N=7, H=7, double bound=7 and triple bound=14 (see Johnson 2010)
% for CO2: molecular structure of CO2 is O=C=O
Vb = 1*7+2*7+2*7;                                                 %cm^3/mol
% for CH4: molecular structure of CH4 is 1C to 4H with 4 single bounds
%Vb = 1*7+4*7;
% for N2O: molecular structure of N2O is N=N=O
%Vb = 1*7+2*7+2*7;                                                 %cm^3/mol

%Real gas molar volume
Vm = 22.3;  %for CO2 and CH4                                         %L/mol
%Vm = 22.2432;  %for N2O                                              %L/mol
                    
                              
                            %% Variables

%OBJECTIVES: provides the environmental variables to run the analysis
                            
dir='C:\Users\Vasco Vieira\Dropbox\interface gas flux\Dados\ms2\'; %directory with the data file
file='Field_validation';                             %name of the data file
load([dir file]);
   
% Gas concentration in the air in                                 (mol/m^3)
Ca = Ca_mmol(ibounds)/1000;

% Gas concentration in the air in                                     (ppm)
Cappm = Ca_ppm(ibounds);

% Air temperature at height 0                                          (C)
Ta0 = ta(ibounds);

% Air temperature at height h                                          (C)
Tah = T12m(ibounds);

% Air pressure at height 0                                            (atm)
P0  = pa(ibounds)/1000;

% Air pressure at height h                                            (atm)
Ph  = pa(ibounds)/1000;

% relative humidity at height h                                    (scalar)
hrh = rh(ibounds);

% specific humidity at height h
qh  = nan(size(Ca));

% liquid-water saturation mixing ratio at height 0
rL0 = zeros(size(Ca));

% liquid-water saturation mixing ratio at height h
rLh = zeros(size(Ca));

% height h for T, P and q measures                                      (m)
hTP = 12*ones(size(Ca));

% Wind velocity h meters above the surface                            (m/s)
U = ws12m(ibounds);

% heigth of wind measurement                                            (m)
zu = 12*ones(size(Ca));

% Roughness length                                                      (m)
z0 = NaN(1);

% friction velocity (wind generating shear stress at the sea surface) (m/s)
u_ = NaN(1);

% Vertical Heat Flux                                          (J.m^-2.s^-1)
vhf = NaN(1);

% Gas concentration in the water                                  (mol/m^3)
Cw = nan(size(Ca));

% Gas concentration in the water in equivalent air partial pressure when in
%equilibrium. Optionals: fixed temperature and/or salinity.           (ppm)
Cwppm = Cw_ppm(ibounds);

% Significant wave heigth (wind seas)                                   (m)
Hs = Hs(ibounds);

% Wave length of waves at the peak wave spectrum (wind seas)            (m)
Lp = Lp(ibounds);

% Wave frequency (wind seas)                                       %(rad/s)
fp = fp(ibounds)*2*pi;

% psi_m (from Atmospheric Stability)                               (scalar)
psi_m = NaN(1);

% Salinity                                                            (ppt)
S = S_MyO(ibounds);

% Water skin temperature                                               (C)
Tw = Twaverider(ibounds);

% Water current velocity                                              (m/s)
w = 0.001*ones(size(Ca));

% Water depth                                                           (m)
zw = 100*ones(size(Ca));

%clear useless arrays
clear dir file

                               %% CALCULUS

%The sections below run the calculus. Do not change them as it may stop the
%software from working properly.

%OBJECTIVES: check for errors----------------------------------------------
i = sum(find(zu~=10));
if sum(strcmp(NSka,{'Lis73' 'Dea91' 'Sea02'}))>=1 && i>0
    'error: ka numerical scheme requires 10m wind measurement heigth'
end
    
%OBJECTIVES: estimates solubility and converts gas concentration units-----

Tka    = 273.15+Ta0;
Pascal = P0*101325.01;
Tkw    = 273.15+Tw;

%Conversion of gas concentration in the air from partial pressure (ppm) to
% mol/m^3 if originally given in ppm.
i      = find(isnan(Ca)==1);
Ca(i)  = Cappm(i).*Pascal(i)./(10^6*R*Tka(i));                     %mol/m^3

%Conversion of gas concentration in the water from equivalent atmospheric
%partial pressure in equilibrium (ppm) to mol/m^3 if originally given in ppm.
i     = find(isnan(Cw)==1);
kH    = estimatekH(P0,S,Tkw,as,bs,kHcp,R,tds,Vb,Vm,NSkH);
kHcp  = 12.1866./(Tka.*kH);                                    %mol/(L*atm)
Cw(i) = Cwppm(i).*kHcp(i).*P0(i)*10^-3;                            %mol/m^3

%OBJECTIVES: Estimates friction velocity-----------------------------------
if (sum(strcmp(NSka,{'Joh10' 'M&Y83' 'Jef10'}))==1 || ...
    sum(strcmp(NSkwind,{'M&Y83' 'MY83w' 'Zea03' 'ZoJ03'}))==1 || ...
    sum(strcmp(NSkbubble,{'Zhg06' 'ZRb03'}))==1)
    
    %Estimates roughness length
    if numel(Hs)>1 && numel(Lp)>1
        z0 = estimatez0(z0,Hs,Lp,Aw,Bw);
    end

    %estimates Bulk Richardson number
    i = find(isnan(z0)==0);
    if numel(i)>1
        Rib = estimateRib(P0,Ph,Tw,Tah,hrh,rL0,rLh,qh,U,hTP,g,NSrib,i);
    else
        Rib = [];
    end

    %Estimates Monin-Obhukov Stability theory (MOS)
    if numel(Rib)>1
        [psi_m,MO_L] = estimateMOS(z0,zu,Rib,Karman,NSmos);
    end
    
    %Estimates Schimdt number and kinematic viscosity of air
    [Sca,Upsa] = estimateSca(P0,Ta0,Tka,Ma,Mb,Sv,Sd,Va,Vb);
    
    %Iteratively estimates u_(star) and z0
    if numel(u_)==1
        u_=nan(size(U));
    end
    
    if numel(Rib)>1
        switch NSu_
            case 'WLLP'
                u_ = U.*Karman./(log(zu)-log(z0)+psi_m);
                
            case 'iWLP'
                u_ = U.*Karman./(log(zu)-log(z0)+psi_m);
                for i=1:niWLLP
                    z0 = 1200*Hs.*(Hs./(Lp)).^4.5+0.11.*Upsa./u_;
                    u_ = U.*0.4./(log(zu)-log(z0)+psi_m);
                    
                end
                %u_(u_<0)=0;
            case 'VHFx'
                %u_;
        end
    end
    u_(u_<0)=0;

    %estimates U10 from wind log-linear profile (for refference use)
    U10 = U.*(log(10)-log(z0)+psi_m)./(log(zu)-log(z0)+psi_m);
    
    %Estimates friction velocity from Drag Coefficient (CD)
    i  = find(isnan(u_)==1);
    if sum(find(zu(i)~=10))>=1
        'invalid CD: wind not estimated at 10m. Check zu(i)'
    else
        u_ = estimateCD2ustar(U,u_,NSdragcoef,i);
    end
end

%OBJECTIVES: estimates air-side transfer velocity--------------------------
try
    numel(psi_m);
catch
    psi_m=[];
end

ka = estimateka(psi_m,Sca,U,u_,z0,Karman,Mb,NSka);

clear Aw Bw cp g hTP hr0 hrh i Karman Ma Mb NSdragcoef NSmos NSka NSrib 
clear NSu_ niWLLP P0 Ph q0 qh rL0 rLh Sd Sv Ta0 Tah tds

%% Estimates water-side transfer velocity and gas fluxes
i = find(zu~=10);
if sum(strcmp(NSkwind,{'Bea04' 'Cea96' 'L&M86' 'McG01' 'Nea00' 'R&C01' ...
                                  'Sea07' 'Wan92' 'W92en'}))==1 && sum(i)>0
    'error: kw numerical scheme requires 10m wind measurement heigth'
    {'cell'   'height'}
    [i(1:20) zu(i(1:20))]
    if numel(i)>20
        '... and more'
    end
end

%Estimates the chemical enhancement factor (alfa) and the Schmidt number of
%the water (Scw)
[Alfa,Scw,Upsw] = estimatealfa(S,Tw,Tkw,NSalfa,NSdw,NSkwind,a,b,c,r,MassFraction,Ms,phiw,Vb);

%Estimates the water-side transfer velocity
[kwind]     = estimatekwind(U,u_,NSkwind);
[kbubble,W] = estimatekbubble(R,u_,Hs,fp,kHcp,Upsa,Upsw,Scw,NSkbubble);
[kcurrent]  = estimatekcurrent(w,zw,NSkcurrent);
[kw]        = estimatekw(Tw,chemen,kwind,kbubble,kcurrent,Alfa);


%Estimates the gas flux and total transfer velocity
if strcmp(NSflux,'Mol')==1
    [Fa,Fw,Ka,Kw] = estimateFluxMol(Ca,Cw,kH,ka,kw,NSlayer,NSfluxunits);
elseif strcmp(NSflux,'ppm')==1
    [Fa,Fw,Ka,Kw] = estimateFluxPPM(Cappm,Cwppm,kH,ka,kw,Pascal,R,Tka,NSlayer,NSfluxunits);
end

clear a as b bs c chemen fp i kHcp MassFraction Ms nc Pascal phiw R r
clear -regexp '^C|^NS|^T|^V'

%% stores the results and clears remaining data

Results.kH           = kH;
try
    Results.z0       = z0;
    Results.Rib      = Rib;
    Results.psi_m    = psi_m;
    Results.MO_L     = MO_L;
catch
    'no atmospheric stability implementation'
end
Results.u_           = u_;
Results.U10          = U10;
Results.W            = W;
Results.Upsa         = Upsa;
Results.kwind        = kwind;
Results.kbubble      = kbubble;
Results.kcurrent     = kcurrent;
Results.Upsw         = Upsw;
Results.Scw          = Scw;
Results.kw           = kw;
Results.Alfa         = Alfa;
try
    Results.Kw       = Kw;
    Results.ka       = ka;
    Results.Sca      = Sca;
    Results.Ka       = Ka;
    Results.Flux.air = Fa;
catch
    'single layer transfer velocity: no Ka'
end
Results.Flux.water   = Fw;

clear Alfa Hs Lp MO_L psi_m Rib S u_ vhf Wpw w
clear -regexp '^F|^K|^k|^N|^Sc|^U|^z'

