%% Authors: R. Raj, N.A.S. Hamm, C. van der Tol, and A. Stein
%% Affiliation: ITC, University of Twente, The Netherlands
%% Date: 22 August 2016
%% Version: 1 

%%% This script runs the BIOME-BGC simulator and returns the simulated gross primary production (GPP).

function [SimGPP] = BIOMErunScript(x);

%% Enter start and end of the Julian day to subset simulated GPP to be used in the calibration
JulS = 91;       
JulE = 243;       

ppc  = 20;      %% Define which column contains daily simulated GPP in the output file produced by BIOME-BGC
                  

%% Read mean values of BIOME-BGC parameters from the file "Meanpm.epc". See Table 1 of the main text for more details about the parameters.
epcMean = load('Meanpm.epc');

%% Define the sequence of BIOME-BGC parameters (LFRT, FRC:LC, CNleaf, Wint, FLNR) as appeared in "Meanpm.epc" file. 
seqEpc = [9,13,18,31,36];

%% Replace the mean values of LFRT, FRC:LC, CNleaf, Wint, and FLNR in "epcMean" variable (defined above) with the new values obtained from each generation of DREAM. 
m = 0;
for i = seqEpc
    m = m+1;
   epcMean(i) = x(m);
end  

%% Check condition C:N litter (row 19)  >= C:N leaf (row 18)
%% Assign C:N leaf to C:N litter, if C:N leaf > C:N litter
if (epcMean(19) < epcMean(18))
epcMean(19) = epcMean(18);
end

%% Generate new input parameter file ("pm.epc") including new values of LFRT, FRC:LC, CNleaf, Wint, and FLNR
dlmwrite('pm.epc','ECOPHYS','delimiter',''); %% 'ECOPHYS' header is added so that BIOME-BGC could read input parameters file
dlmwrite('pm.epc',epcMean,'delimiter',',','-append');

%% Soil depth (SD) is defined in the input initialization (ini) file of BIOME-BGC. Therefore, the new value of SD obtained from each generation of DREAM should be added in the ini file.  
%% Read existing ini file ("enf_speuld_Main.ini"), replace the existing value of SD with the new value, and generate new ini file ("enf_speuld.ini") including new value of SD.
rep = num2str(x(6));             %% Give the value to be replaced
sdN =  34;                  %%Set the line number of soil depth value in ini file
fileID = fopen('enf_speuld.ini', 'w');
iniFile = textread('enf_speuld_Main.ini','%s','delimiter', '\n');
test = iniFile(sdN);
iniFile(sdN) = strrep(test,'0.4',rep);
fprintf(fileID, '%s \r\n', iniFile{:});
fclose(fileID);

%% Run BIOME-BGC with new ini file ("enf_speuld.ini") and new input parameter file ("pm.epc")
dos('pointbgc.exe -a -g enf_speuld.ini');

% Load the output of the model  
SimGPP = load('Speuld_pm.dayout.ascii'); 
SimGPP = SimGPP(JulS:JulE,ppc)*1000; %% Unit of simulated GPP in kg C m-2 s-1 is converted to g C m-2 s-1 (unit of flux tower GPP)