################################################
#                                              #
# Sub-grid air quality model                   #
# Author: Mark Theobald, CIEMAT, Madrid Spain  #
# Version: 21 June 2016                        #
#                                              #
################################################

# Input files:

emis_file <- "x_y_emis_by_SNAP.csv"
# CSV file with the following columns: x_index[1 to kernel_x], y_index[1 to kernel_y], emis_SNAP1, emis_SNAP2,...,emis_SNAPn

short_range_disp_file <- "kernel_50km_101x101_radial_source_h_"   
# CSV file with the following columns: x_index[1 to emis_x], y_index[1 to emis_y], conc
# complete filename has suffix with source height used e.g. "kernel_50km_101x101_radial_source_h_200m.csv"

# Output file:
out_file <- "sub_grid_out.csv"

# Model parameters

kernel_x <- 101 # No. of kernel grid cells in x direction
kernel_y <- 101 # No. of kernel grid cells in y direction
emis_x <- 320 # No. of emission grid cells in x direction
emis_y <- 370 # No. of emission grid cells in y direction
cell_dim <- 1000 # Grid cell side length (m) (must be the same for the kernel and the emis file)
SNAP_list <- c(1,2,3,4,7,8,9) # List of SNAP emission sectors used
src_h_by_SNAP <- c(400,0,400,50,0,0,200) # Effective emission heights (m) for the sectors in SNAP_list

# Start of calculations

margin <- (kernel_x-1)/2 # Calculate margin (buffer) around 
grid_x <- emis_x+(2*margin) # Calculate domain dimension in x direction (emis domain + margin around the edge)
grid_y <- emis_y+(2*margin) # Calculate domain dimension in y direction (emis domain + margin around the edge)
emis_matrix <- array(NA, dim=c(emis_y,emis_x))
kernel_matrix <- array(NA, dim=c(kernel_y,kernel_x))
conc_array <- array(0, dim=c(grid_y,grid_x,length(SNAP_list))) # Define array for concentrations for full domain (emis domain + margin)
domain_conc <- array(NA, dim=c(grid_y-(2*margin),grid_x-(2*margin),length(SNAP_list))) # Define array for concentrations for output domain (same as emission domain)
output <- array(NA, dim=c((grid_y-(2*margin))*(grid_x-(2*margin)),length(SNAP_list)+2)) # Define array for concentrations (for file output)
colnames(output) <- c("x_index", "y_index", paste("SNAP_",SNAP_list,sep=""))

emis_data <- read.table(emis_file, sep=",", header=TRUE) # Read emission data

num_SNAP <- length(emis_data[1,])-2 # Number of sectors in input emission data

emis_data[,3:(2+num_SNAP)] <- emis_data[,3:(2+num_SNAP)]/1000 # convert emissions from mg/m2 to Mg/km2

for (SNAP_num in 1:length(SNAP_list)){  # Loop through each emission sector

	print(paste("Processing SNAP: ", SNAP_list[SNAP_num]))
	
	krnel <- read.table(paste(short_range_disp_file, src_h_by_SNAP[SNAP_num],"m.csv", sep=""), sep=",", header=TRUE) # Read in concentration field for the relevant source height
	
	# Convert 1D emission data to 2D matrix
	
	for (row_num in 1:(emis_x*emis_y)){
	
		x_index <- emis_data[row_num,1]
		y_index <- emis_data[row_num,2]
		emis_matrix[y_index,x_index] <- emis_data[row_num,2+SNAP_list[SNAP_num]]
		
	}
	
	emis_matrix[is.na(emis_matrix)] <- 0 # Set NA values to zero
	
	# Convert 1D kernel data to 2D matrix
	
	for (row_num in 1:(kernel_x*kernel_y)){
	
		x_index <- krnel[row_num,1]
		y_index <- krnel[row_num,2]
		kernel_matrix[y_index,x_index] <- krnel[row_num,3]
		
	}
	
	kernel_matrix[is.na(kernel_matrix)] <- 0 # Set NA values to zero
		
	
	for (src_x in 1:(grid_x-(2*margin))){ # Loop through each x coordinate of emissions
	
	
		# Define x indices for the "moving window"
		ind3 <- src_x
		ind4 <- src_x+kernel_x-1

		for (src_y in 1:(grid_y-(2*margin))){ # Loop through each y coordinate of emissions
		
			# Define y indices for the "moving window"
			ind1 <- src_y
			ind2 <- src_y+kernel_y-1
		
			if (emis_matrix[src_y,src_x]>0) { # If emissions are non-zero
			
				
				conc_array[ind1:ind2,ind3:ind4,SNAP_num] <- conc_array[ind1:ind2,ind3:ind4,SNAP_num] + emis_matrix[src_y,src_x] * kernel_matrix # Multiply emission at (src_x,src_y) by the kernel concentrations and add the result to the concentration array
				
			}
		}
	}
	
	domain_conc[,,SNAP_num] <- conc_array[(margin+1):(grid_y-margin),(margin+1):(grid_x-margin),SNAP_num]	# Update the domain concentration array for SNAP_num
				
	# Convert the domain concentration array from 2D to 1D (for output to file)
	
	row_num <- 1
	
	for (x in 1:emis_x){
	
		for (y in 1:emis_y){
			
			output[row_num,1] <- x
			output[row_num,2] <- y
			output[row_num,(SNAP_num+2)] <- domain_conc[y,x,SNAP_num]
			
			row_num <- row_num+1
		
		}
		
	}
	
 } # End of loop "SNAP_num"

# Output results to file
write.table(output, file = out_file, sep = ",", col.names = TRUE, row.names = FALSE, quote=FALSE)

closeAllConnections()