function SedimentationRateEstimation(fileName,windowsize,Porder,Tnominal)
% InstantaneousFrequencyEstimation - Estimates the sedimentation rate from 
% the estimated waveforms of the orbital components. 
%
% About:
%   This script is part of the complementary material for the manuscript
%   entitled "Astronomical component estimation (ACE v.1) by time-variant
%   sinusoidal modeling", submitted to Geoscientific Model Development.
%
% Input:
%   fileName          *.mat file containing the estimated orbital
%                     component waveforms
%   windowsize        length of the analysis frame in meters
%                     See Chapter 2.3 in Sinnesael et al., 2016, 
%                     Geoscientific Model Development for instructions
%   Porder            number of terms for the polynomial approximation.
%                     See Chapter 2.3 in Sinnesael et al., 2016, 
%                     Geoscientific Model Development for instructions%
%   Tnominal          nominal orbital component period in kyr
%                     Predicted periods for precession and obliquity can be
%                     found in Berger et al., (1992) or Laskar et al.,
%                     (2004)
%
% Description:
%   From spatial-domain waveforms it estimates the instantaneous frequency
%   and sedimentation rate for orbital components.
% Example:
%   SedimentationRateEstimation('EstimatedComponents',60,2,[41 23])


ss = load(fileName);
signal = ss.orbitalComp;
fs = ss.fs;

[~,Nsources] = size(signal);

N = length(signal);
windowsize = round(windowsize*fs); % in samples

inc = round(windowsize/4);
Position = 1:inc:round(length(signal)-windowsize+1);
f0est = zeros(length(Position),Nsources);

ooff = 0;
normo = zeros(N,1);
win = hanning(windowsize);
winHarm = repmat(win,1,Nsources);

freqInst = zeros(N,Nsources);
f0inst = zeros(windowsize,Nsources);

for i = 1:length(Position)
    seg = signal(Position(i):Position(i)+windowsize-1,:);
    
    fprintf('processing frame %d of %d \r',i,length(Position));
        
    for k = 1:Nsources
        x = hilbert(seg(:,k));       
        F = 8*4096; % FFT size
        X = abs(fft(x,F)).^2/N; % periodogram
        [~,maxInd] = max(X);
        f0est(i,k) = (maxInd-1)*fs/F; % estimated mean frequency
        
                                    
        [segEst,a,b] = PolyEstimation(seg(:,k),f0est(i,k),fs,Porder); % component waveform estimate
        A = sqrt(a.^2 + b.^2); % estimated envelope of an orbital component
        segFM = segEst./A; % remove envelope to obtain pure phase modulated signal
    
        [~,c,d] = PolyEstimation(segFM,f0est(i,k),fs,Porder); % waveform estimate on pure phase modulated signal
        argEst = unwrap(-atan2(c,d)); % unwrap the instantaneous phase
        F0 = f0est(i,k) + (1/(2*pi))*diff(argEst)/(1/fs); % estimate instantaneous frequency
        f0inst(:,k) = [F0; F0(end)]';
    end
    
    freqInst(round(ooff)+windowsize,:) = 0; % combine the estimated segments
    freqInst((1:windowsize)+round(ooff),:) = freqInst((1:windowsize)+round(ooff),:) + f0inst.*winHarm;
    
    normo(round(ooff)+windowsize) = 0; % overlap-add procedure to account on the record edges
    normo((1:windowsize)+round(ooff)) = normo((1:windowsize)+round(ooff)) + win;
    ooff = ooff + inc;
end
freqInst = freqInst./repmat(normo,1,Nsources);
save('InstanteousFrequecyEstimates','freqInst','fs')

% ************** SEDIMENTATION RATE ESTIMATION ***************************
SR = zeros(N,Nsources); % sedimentation rate in cm/kyr
for i = 1:Nsources % SR for Nsources orbital components
    T = 100./freqInst(:,i); % x100 is to get centimeters
    SR(:,i) = T/Tnominal(i);
end
save('SedimentationRateEstimates','SR','fs')
return





function [signalEst,a,b,THETA] = PolyEstimation(signal,f0,fs,Porder)

N = length(signal);
t = (0:N-1)'/fs;

T = zeros(N,Porder); % time-basis for the polynomials
for i = 1:Porder
    T(:,i) = t.^(i-1);
end

X1 = repmat(sin(2*pi*f0*t),1,Porder); % model matrix - sine
X2 = repmat(cos(2*pi*f0*t),1,Porder); % model matrix - cosine

X1 = X1 .* T;
X2 = X2 .* T;
FI = [X1 X2];

THETA = FI\signal; % linear least-squares
signalEst = FI*THETA;

a = T*THETA(1:Porder); % sine polynomial
b = T*THETA(Porder+1:end); % cosine polynomial

return

