/* -*-  mode: c++; c-default-style: "google"; indent-tabs-mode: nil -*- */

/*
  Interface for a thermal conductivity model with three phases.

  License: BSD
  Authors: Ethan Coon (ecoon@lanl.gov)
           Satish Karra (satkarra@lanl.gov)
*/

#include "dbc.hh"
#include "thermal_conductivity_threephase_factory.hh"
#include "thermal_conductivity_threephase_evaluator.hh"

namespace Amanzi {
namespace Energy {
namespace EnergyRelations {

ThermalConductivityThreePhaseEvaluator::ThermalConductivityThreePhaseEvaluator(
      Teuchos::ParameterList& plist) :
    SecondaryVariableFieldEvaluator(plist) {
  if (my_key_ == std::string("")) {
    my_key_ = plist_.get<std::string>("thermal conductivity key", "thermal_conductivity");
  }
  
  poro_key_ = plist_.get<std::string>("porosity key", "porosity");
  dependencies_.insert(poro_key_);

  temp_key_ = plist_.get<std::string>("temperature key", "temperature");
  dependencies_.insert(temp_key_);

  sat_key_ = plist_.get<std::string>("saturation key", "saturation_liquid");
  dependencies_.insert(sat_key_);

  sat2_key_ = plist_.get<std::string>("second saturation key", "saturation_ice");
  dependencies_.insert(sat2_key_);

  ASSERT(plist_.isSublist("thermal conductivity parameters"));
  Teuchos::ParameterList tc_sublist = plist_.sublist("thermal conductivity parameters");
  
  ThermalConductivityThreePhaseFactory fac;
  
  for (Teuchos::ParameterList::ConstIterator lcv=tc_sublist.begin();
       lcv!=tc_sublist.end(); ++lcv){
    std::string name = lcv->first;
    if (tc_sublist.isSublist(name)){
      Teuchos::ParameterList& tcp_sublist = tc_sublist.sublist(name);
      std::string region_name = tcp_sublist.get<std::string>("region");
      Teuchos::RCP<ThermalConductivityThreePhase> tc = fac.createThermalConductivityModel(tcp_sublist);
      tcs_.push_back(std::make_pair(region_name,tc));
    } else {
      Errors::Message message("ThermalConductivityThreePhaseEvaluator: region-based lists.  (Perhaps you have an old-style input file?)");
      Exceptions::amanzi_throw(message);
    }
  }
}


ThermalConductivityThreePhaseEvaluator::ThermalConductivityThreePhaseEvaluator(
      const ThermalConductivityThreePhaseEvaluator& other) :
    SecondaryVariableFieldEvaluator(other),
    poro_key_(other.poro_key_),
    temp_key_(other.temp_key_),
    sat_key_(other.sat_key_),
    sat2_key_(other.sat2_key_),
    tcs_(other.tcs_) {}

Teuchos::RCP<FieldEvaluator>
ThermalConductivityThreePhaseEvaluator::Clone() const {
  return Teuchos::rcp(new ThermalConductivityThreePhaseEvaluator(*this));
}


void ThermalConductivityThreePhaseEvaluator::EvaluateField_(
      const Teuchos::Ptr<State>& S,
      const Teuchos::Ptr<CompositeVector>& result) {
  // pull out the dependencies
  Teuchos::RCP<const CompositeVector> poro = S->GetFieldData(poro_key_);
  Teuchos::RCP<const CompositeVector> temp = S->GetFieldData(temp_key_);
  Teuchos::RCP<const CompositeVector> sat = S->GetFieldData(sat_key_);
  Teuchos::RCP<const CompositeVector> sat2 = S->GetFieldData(sat2_key_);
  Teuchos::RCP<const AmanziMesh::Mesh> mesh = result->Mesh();

  for (CompositeVector::name_iterator comp = result->begin();
       comp!=result->end(); ++comp) {
    // much more efficient to pull out vectors first

    ASSERT(*comp == "cell"); 
    const Epetra_MultiVector& poro_v = *poro->ViewComponent(*comp,false);
    const Epetra_MultiVector& temp_v = *temp->ViewComponent(*comp,false);
    const Epetra_MultiVector& sat_v = *sat->ViewComponent(*comp,false);
    const Epetra_MultiVector& sat2_v = *sat2->ViewComponent(*comp,false);
    Epetra_MultiVector& result_v = *result->ViewComponent(*comp,false);

    for (std::vector<RegionModelPair>::const_iterator lcv = tcs_.begin();
         lcv != tcs_.end(); ++lcv){
         std::string region_name = lcv->first;
         if (mesh->valid_set_name(region_name, AmanziMesh::CELL)) {
            // get the indices of the domain.
            AmanziMesh::Entity_ID_List id_list;
            mesh->get_set_entities(region_name, AmanziMesh::CELL, AmanziMesh::OWNED, &id_list);

            // loop over indices
            for (AmanziMesh::Entity_ID_List::const_iterator id=id_list.begin();
                 id!=id_list.end(); ++id) {
                 result_v[0][*id] = lcv->second->ThermalConductivity(poro_v[0][*id],
                 sat_v[0][*id], sat2_v[0][*id], temp_v[0][*id]);
            }
          } else {
            std::stringstream m;
            m << "Thermal conductivity evaluator: unknown region on cells: \"" << region_name << "\"";
            Errors::Message message(m.str());
            Exceptions::amanzi_throw(message);
          }         
    }
  }
}


void ThermalConductivityThreePhaseEvaluator::EvaluateFieldPartialDerivative_(
      const Teuchos::Ptr<State>& S, Key wrt_key,
      const Teuchos::Ptr<CompositeVector>& result) {
  ASSERT(0); // not implemented, not yet needed
}


} //namespace
} //namespace
} //namespace
