/*
  This is the Nonlinear Solver component of the Amanzi code.

  Very simple continuation method.

  Authors: Ethan Coon (ecoon@lanl.gov)
*/

#ifndef AMANZI_CONTINUATION_SOLVER_
#define AMANZI_CONTINUATION_SOLVER_

#include "Teuchos_RCP.hpp"
#include "Teuchos_ParameterList.hpp"

#include "VerboseObject.hh"

#include "Solver.hh"
#include "SolverFnBase.hh"
#include "SolverDefs.hh"

namespace Amanzi {
namespace AmanziSolvers {

template<class Vector, class VectorSpace>
class SolverContinuation : public Solver<Vector,VectorSpace> {
 public:
  SolverContinuation(Teuchos::ParameterList& plist) :
      plist_(plist) {};

  SolverContinuation(Teuchos::ParameterList& plist,
            const Teuchos::RCP<SolverFnBase<Vector> >& fn,
            const VectorSpace& map) :
      plist_(plist) {
    Init(fn, map);
  }

  void Init(const Teuchos::RCP<SolverFnBase<Vector> >& fn,
            const VectorSpace& map);

  int Solve(const Teuchos::RCP<Vector>& u) {
    returned_code_ = Solve_(u);
    return (returned_code_ >= 0) ? 0 : 1;
  }

  // mutators
  void set_tolerance(double tol) {
    tol_ = tol;
    solver_->set_tolerance(tol);
  }
  void set_pc_lag(double pc_lag) {
    solver_->set_pc_lag(pc_lag);
  }

  // access
  double tolerance() { return tol_; }
  double residual() { return solver_->residual(); }
  int num_itrs() { return num_itrs_; }
  int returned_code() { return returned_code_; }

 private:
  void Init_();
  int Solve_(const Teuchos::RCP<Vector>& u);

 protected:
  Teuchos::ParameterList plist_;
  Teuchos::RCP<SolverFnBase<Vector> > fn_;
  Teuchos::RCP<Solver<Vector, VectorSpace> > solver_;

  Teuchos::RCP<VerboseObject> vo_;

 private:
  double tol_;
  int num_itrs_;
  double n_cont_steps_;
  int returned_code_;
};

} // namespace
} // namespace

#include "SolverFactory.hh"

namespace Amanzi {
namespace AmanziSolvers {

/* ******************************************************************
* Public Init method.
****************************************************************** */
template<class Vector, class VectorSpace>
void
SolverContinuation<Vector,VectorSpace>::Init(const Teuchos::RCP<SolverFnBase<Vector> >& fn,
        const VectorSpace& map)
{
  fn_ = fn;
  Init_();
  solver_->Init(fn, map);
}


/* ******************************************************************
* Initialization of the NKA solver.
****************************************************************** */
template<class Vector, class VectorSpace>
void SolverContinuation<Vector, VectorSpace>::Init_()
{
  tol_ = plist_.get<double>("nonlinear tolerance", 1.e-6);
  n_cont_steps_ = plist_.get<int>("number of continuation steps", 5);

  SolverFactory<Vector,VectorSpace> fac;
  solver_ = fac.Create(plist_.sublist("inner solver"));
  
  // update the verbose options
  vo_ = Teuchos::rcp(new VerboseObject("Solver::Continuation", plist_));
}


/* ******************************************************************
* The body of NKA solver
****************************************************************** */
template<class Vector, class VectorSpace>
int SolverContinuation<Vector, VectorSpace>::Solve_(const Teuchos::RCP<Vector>& u) {
  Teuchos::OSTab tab = vo_->getOSTab();

  // initialize the iteration counter
  num_itrs_ = 0;

  int itr = 0;
  do {
    if (vo_->os_OK(Teuchos::VERB_MEDIUM)) {
      *vo_->os() << "taking continuation step " << itr << " of "
                 << n_cont_steps_ << std::endl;
    }

    // continuation parameter ranges from 1 (smooth problem) to 0 (true problem)
    double lambda = (n_cont_steps_ - itr) / n_cont_steps_;
    fn_->UpdateContinuationParameter(lambda);

    // solve
    int ierr = solver_->Solve(u);

    num_itrs_ += solver_->num_itrs();

    if (ierr) {
      return solver_->returned_code();
    }

    itr++;
  } while (itr <= n_cont_steps_);
  return itr;
}


}  // namespace AmanziSolvers
}  // namespace Amanzi

#endif
