/*
  This is the flow component of the Amanzi code. 

  Copyright 2010-201x held jointly by LANS/LANL, LBNL, and PNNL. 
  Amanzi is released under the three-clause BSD License. 
  The terms of use and "as is" disclaimer for this license are 
  provided in the top-level COPYRIGHT file.

  Authors: Neil Carlson (nnc@lanl.gov), 
           Konstantin Lipnikov (lipnikov@lanl.gov)
           Daniil Svyatskiy (dasvyat@lanl.gov)

  The routine implements interface to the BDF1 time integrator.  
*/

#include <algorithm>
#include <string>
#include <vector>

#include "LinearOperatorFactory.hh"
#include "Richards_PK.hh"

namespace Amanzi {
namespace Flow {

/* ******************************************************************
* Calculate f(u, du/dt) = a d(s(u))/dt + A*u - rhs.
****************************************************************** */
void Richards_PK::Functional(double T0, double T1, 
                             Teuchos::RCP<CompositeVector> u_old, Teuchos::RCP<CompositeVector> u_new, 
                             Teuchos::RCP<CompositeVector> f)
{ 
  double dTp(T1 - T0);

  const Epetra_MultiVector& uold_cell = *u_old->ViewComponent("cell");
  const Epetra_MultiVector& unew_cell = *u_new->ViewComponent("cell");

  // update coefficients
  darcy_flux_copy->ScatterMasterToGhosted("face");
  rel_perm_->Compute(*u_new); 

  RelativePermeabilityUpwindFn func1 = &RelativePermeability::Value;
  upwind_->Compute(*darcy_flux_upwind, *u_new, bc_model, bc_value, 
                   *rel_perm_->Krel(), *rel_perm_->Krel(), func1);

  RelativePermeabilityUpwindFn func2 = &RelativePermeability::Derivative;
  upwind_->Compute(*darcy_flux_upwind, *u_new, bc_model, bc_value,
                   *rel_perm_->dKdP(), *rel_perm_->dKdP(), func2);
  UpdateSourceBoundaryData(T0, T1, *u_new);
  
  // assemble residual for diffusion operator
  op_matrix_->Init();
  op_matrix_diff_->UpdateMatrices(darcy_flux_copy.ptr(), solution.ptr());
  op_matrix_diff_->ApplyBCs(true);

  Teuchos::RCP<CompositeVector> rhs = op_matrix_->rhs();
  if (src_sink != NULL) AddSourceTerms(*rhs);

  op_matrix_->ComputeNegativeResidual(*u_new, *f);

  // add accumulation term 
  const Epetra_MultiVector& phi = *S_->GetFieldData("porosity")->ViewComponent("cell");
  Epetra_MultiVector& f_cell = *f->ViewComponent("cell");

  functional_max_norm = 0.0;
  functional_max_cell = 0;

  std::vector<Teuchos::RCP<WRM> >& WRM = rel_perm_->wrm();  
  for (int mb = 0; mb < WRM.size(); mb++) {
    std::string region = WRM[mb]->region();
    AmanziMesh::Entity_ID_List block;
    mesh_->get_set_entities(region, AmanziMesh::CELL, AmanziMesh::OWNED, &block);

    double s1, s2, volume;
    for (int i = 0; i < block.size(); i++) {
      int c = block[i];
      s1 = WRM[mb]->saturation(atm_pressure_ - unew_cell[0][c]);
      s2 = WRM[mb]->saturation(atm_pressure_ - uold_cell[0][c]);

      double factor = rho_ * phi[0][c] * mesh_->cell_volume(c) / dTp;
      f_cell[0][c] += (s1 - s2) * factor;

      double tmp = fabs(f_cell[0][c]) / factor;  // calculate errors
      if (tmp > functional_max_norm) {
        functional_max_norm = tmp;
        functional_max_cell = c;        
      }
    }
  }

  // add vapor diffusion 
  if (vapor_diffusion_) {
    Functional_AddVaporDiffusion_(f);
  }
}


/* ******************************************************************
* Calculate additional conribution to Richards functional:
*  f += -div (\phi s_g \tau_g n_g D_g grad \omega).
****************************************************************** */
void Richards_PK::Functional_AddVaporDiffusion_(Teuchos::RCP<CompositeVector> f)
{
  CompositeVector g(*f);
  g.PutScalar(0.0);

  // Extract fields
  Teuchos::RCP<const CompositeVector> pres = S_->GetFieldData("pressure");
  Teuchos::RCP<const CompositeVector> temp = S_->GetFieldData("temperature");

  // Populate vapor matrix
  CalculateVaporDiffusionTensor_();

  op_vapor_matrix_->Init();
  op_vapor_matrix_diff_->UpdateMatrices(Teuchos::null, Teuchos::null);
  op_vapor_matrix_diff_->ApplyBCs(false);

  // Calculate the residual
  CompositeVector omega(*f);
  omega.PutScalar(1.0);
  op_vapor_matrix_->ComputeNegativeResidual(omega, g);

  f->Update(1.0, g, 1.0);
}


/* ******************************************************************
* Calculation of diffusion coefficient for vapor diffusion operator.
****************************************************************** */
void Richards_PK::CalculateVaporDiffusionTensor_()
{
   S_->GetFieldEvaluator("molar_density_liquid")->HasFieldChanged(S_.ptr(), passwd_);
   const Epetra_MultiVector& n_l = *S_->GetFieldData("molar_density_liquid")->ViewComponent("cell");

   S_->GetFieldEvaluator("molar_density_gas")->HasFieldChanged(S_.ptr(), passwd_);
   const Epetra_MultiVector& n_g = *S_->GetFieldData("molar_density_gas")->ViewComponent("cell");

   S_->GetFieldEvaluator("porosity")->HasFieldChanged(S_.ptr(), passwd_);
   const Epetra_MultiVector& phi = *S_->GetFieldData("porosity")->ViewComponent("cell");

   S_->GetFieldEvaluator("saturation_gas")->HasFieldChanged(S_.ptr(), passwd_);
   const Epetra_MultiVector& s_g = *S_->GetFieldData("saturation_gas")->ViewComponent("cell");

   S_->GetFieldEvaluator("molar_fraction_gas")->HasFieldChanged(S_.ptr(), passwd_);
   const Epetra_MultiVector& mlf_g = *S_->GetFieldData("molar_fraction_gas")->ViewComponent("cell");

   std::string key("temperature");
   S_->GetFieldEvaluator("molar_fraction_gas")->HasFieldDerivativeChanged(S_.ptr(), passwd_, key);
   const Epetra_MultiVector& dmlf_g_dt = *S_->GetFieldData("dmol_frac_gas_dtemperature")->ViewComponent("cell");

   const Epetra_MultiVector& temp = *S_->GetFieldData("temperature")->ViewComponent("cell");
   const Epetra_MultiVector& pres = *S_->GetFieldData("pressure")->ViewComponent("cell");

   double R = 8.3144621;
   double a = 4./3.;
   double b = 10./3.;
   double Dref = 0.282;
   double Pref = atm_pressure_;
   double Tref = 298;  // Kelvins

   K_vapor.clear();
   WhetStone::Tensor Kc(dim, 1);

   for (int c = 0; c != ncells_owned; ++c) {
     double tmp = Dref * (Pref / atm_pressure_) * pow(temp[0][c] / Tref, 1.8);

     tmp *= pow(phi[0][c], a) * pow(s_g[0][c], b) * n_g[0][c];
     tmp *= exp(-(atm_pressure_ - pres[0][c]) / (n_l[0][c] * R * temp[0][c]));

     Kc(0, 0) = tmp;
     K_vapor.push_back(Kc);
   }

   /*
   if (var_name == "pressure"){
     for (unsigned int c=0; c!=ncells; ++c){
       diff_coef[0][c] *= mlf_g[0][c] * (1./ (n_l[0][c]*R*temp[0][c]));
     }
   }
   else if (var_name == "temperature"){
     for (unsigned int c=0; c!=ncells; ++c){
       diff_coef[0][c] *= (1./Patm)*dmlf_g_dt[0][c] + mlf_g[0][c]* (Patm - pressure[0][c])/ (n_l[0][c]*R*temp[0][c]*temp[0][c]);
     }
   }
   */
}


/* ******************************************************************
* Apply preconditioner inv(B) * X.                                                 
****************************************************************** */
void Richards_PK::ApplyPreconditioner(Teuchos::RCP<const CompositeVector> X, 
                                      Teuchos::RCP<CompositeVector> Y)
{
  Y->PutScalar(0.);
  op_pc_solver_->ApplyInverse(*X, *Y);
}


/* ******************************************************************
* Update new preconditioner B(p, dT_prec).                                   
****************************************************************** */
void Richards_PK::UpdatePreconditioner(double Tp, Teuchos::RCP<const CompositeVector> u, double dTp)
{
  // update coefficients
  if (update_upwind == FLOW_UPWIND_UPDATE_ITERATION) {
    op_matrix_diff_->UpdateFlux(*solution, *darcy_flux_copy);
  }
  darcy_flux_copy->ScatterMasterToGhosted("face");
  rel_perm_->Compute(*u);

  RelativePermeabilityUpwindFn func1 = &RelativePermeability::Value;
  upwind_->Compute(*darcy_flux_upwind, *u, bc_model, bc_value,
                   *rel_perm_->Krel(), *rel_perm_->Krel(), func1);

  RelativePermeabilityUpwindFn func2 = &RelativePermeability::Derivative;
  upwind_->Compute(*darcy_flux_upwind, *u, bc_model, bc_value,
                   *rel_perm_->dKdP(), *rel_perm_->dKdP(), func2);

  double T0 = Tp - dTp;
  UpdateSourceBoundaryData(T0, Tp, *u);

  // create diffusion operators
  op_preconditioner_->Init();
  op_preconditioner_diff_->UpdateMatrices(darcy_flux_copy.ptr(), solution.ptr());
  op_preconditioner_diff_->ApplyBCs(true);

  // add time derivative
  CompositeVectorSpace cvs;
  cvs.SetMesh(mesh_);
  cvs.SetGhosted(false);
  cvs.SetComponent("cell", AmanziMesh::CELL, 1);

  CompositeVector dSdP(cvs);
  rel_perm_->DerivedSdP(*u->ViewComponent("cell"), *dSdP.ViewComponent("cell"));

  const CompositeVector& phi = *S_->GetFieldData("porosity");
  dSdP.Multiply(rho_, phi, dSdP, 0.0);

  if (dTp > 0.0) {
    op_acc_->AddAccumulationTerm(*u, dSdP, dTp, "cell");
  }

  // finalize preconditioner
  op_preconditioner_->AssembleMatrix();
  op_preconditioner_->InitPreconditioner(preconditioner_name_, *preconditioner_list_);
}


/* ******************************************************************
* Modify preconditior as needed.
****************************************************************** */
bool Richards_PK::ModifyPredictor(double dT, Teuchos::RCP<const CompositeVector> u0,
                                  Teuchos::RCP<CompositeVector> u)
{
  Teuchos::RCP<CompositeVector> du = Teuchos::rcp(new CompositeVector(*u));
  du->Update(-1.0, *u0, 1.0);
 
  ModifyCorrection(dT, Teuchos::null, u0, du);

  *u = *u0;
  u->Update(1.0, *du, 1.0);
  return true;
}


/* ******************************************************************
* Check difference du between the predicted and converged solutions.
* This is a wrapper for various error control methods. 
****************************************************************** */
double Richards_PK::ErrorNorm(Teuchos::RCP<const CompositeVector> u, 
                              Teuchos::RCP<const CompositeVector> du)
{
  double error;
  error = ErrorNormSTOMP(*u, *du);

  return error;
}


/* ******************************************************************
* Error control a-la STOMP.
****************************************************************** */
double Richards_PK::ErrorNormSTOMP(const CompositeVector& u, const CompositeVector& du)
{
  const Epetra_MultiVector& uc = *u.ViewComponent("cell");
  const Epetra_MultiVector& duc = *du.ViewComponent("cell");

  double error, error_p, error_r;
  int cell_p(0), cell_r(0);

  if (error_control_ & FLOW_TI_ERROR_CONTROL_PRESSURE) {
    error_p = 0.0;
    for (int c = 0; c < ncells_owned; c++) {
      double tmp = fabs(duc[0][c]) / (fabs(uc[0][c] - atm_pressure_) + atm_pressure_);
      if (tmp > error_p) {
        error_p = tmp;
        cell_p = c;
      } 
    }
  } else {
    error_p = 0.0;
  }

  if (error_control_ & FLOW_TI_ERROR_CONTROL_RESIDUAL) {
    error_r = functional_max_norm;
  } else {
    error_r = 0.0;
  }

  error = std::max(error_r, error_p);

#ifdef HAVE_MPI
  double buf = error;
  du.Comm().MaxAll(&buf, &error, 1);  // find the global maximum
#endif

  // maximum error is printed out only on one processor
  if (vo_->getVerbLevel() >= Teuchos::VERB_EXTREME) {
    const Epetra_IntVector& map = rel_perm_->map_c2mb();

    if (error == buf) {
      int c = functional_max_cell;
      const AmanziGeometry::Point& xp = mesh_->cell_centroid(c);

      Teuchos::OSTab tab = vo_->getOSTab();
      *vo_->os() << "residual=" << functional_max_norm << " at point";
      for (int i = 0; i < dim; i++) *vo_->os() << " " << xp[i];
      *vo_->os() << std::endl;
 
      c = cell_p;
      const AmanziGeometry::Point& yp = mesh_->cell_centroid(c);

      *vo_->os() << "pressure err=" << error_p << " at point";
      for (int i = 0; i < dim; i++) *vo_->os() << " " << yp[i];
      *vo_->os() << std::endl;

      int mb = map[c];
      double s = (rel_perm_->wrm())[mb]->saturation(atm_pressure_ - uc[0][c]);
      *vo_->os() << "saturation=" << s << " pressure=" << uc[0][c] << std::endl;
    }
  }

  return error;
}


/********************************************************************
* Modifies nonlinear update du based on the maximum allowed change
* of saturation.
****************************************************************** */
AmanziSolvers::FnBaseDefs::ModifyCorrectionResult
    Richards_PK::ModifyCorrection(double dT, Teuchos::RCP<const CompositeVector> f,
                                  Teuchos::RCP<const CompositeVector> u,
                                  Teuchos::RCP<CompositeVector> du)
{
  const Epetra_MultiVector& uc = *u->ViewComponent("cell");
  const Epetra_MultiVector& duc = *du->ViewComponent("cell");
  AmanziGeometry::Point face_centr, cell_cntr;
  double max_sat_pert = 0.25;
  double damping_factor = 0.5;
  double reference_pressure = 101325.0;
  
  if (rp_list_->isSublist("clipping parameters")) {
    Teuchos::ParameterList& clip_list = rp_list_->sublist("clipping parameters");
    max_sat_pert = clip_list.get<double>("maximum saturation change", 0.25);
    damping_factor = clip_list.get<double>("pressure damping factor", 0.5);
  }

  int nsat_clipped(0), npre_clipped(0);

  std::vector<Teuchos::RCP<WRM> >& WRM = rel_perm_->wrm(); 
  const Epetra_IntVector& map = rel_perm_->map_c2mb();
 
  for (int c = 0; c < ncells_owned; c++) {
    int mb = map[c];
    double pc = atm_pressure_ - uc[0][c];
    double sat = WRM[mb]->saturation(pc);
    double sat_pert;
    if (sat >= 0.5) sat_pert = sat - max_sat_pert;
    else sat_pert = sat + max_sat_pert;
    
    double press_pert = atm_pressure_ - WRM[mb]->capillaryPressure(sat_pert);
    double du_pert_max = fabs(uc[0][c] - press_pert); 
    double tmp = duc[0][c];

    if ((fabs(duc[0][c]) > du_pert_max) && (1 - sat > 1e-5)) {
      if (vo_->getVerbLevel() >= Teuchos::VERB_EXTREME) {
        Teuchos::OSTab tab = vo_->getOSTab();
        *vo_->os() << "clip saturation: c=" << c 
                   << " p=" << uc[0][c]
                   << " dp: " << duc[0][c] << " -> " << du_pert_max << std::endl;
      }

      if (duc[0][c] >= 0.0) duc[0][c] = du_pert_max;
      else duc[0][c] = -du_pert_max;
      
      nsat_clipped++;
    }    
  }

  for (int c = 0; c < ncells_owned; c++) {
    double unew = uc[0][c] - duc[0][c];
    double tmp = duc[0][c];

    if ((unew > atm_pressure_) && (uc[0][c] < atm_pressure_)) {
      if (vo_->getVerbLevel() >= Teuchos::VERB_EXTREME) {
	 *vo_->os() << "pressure change: " << uc[0][c] << " -> " << unew << std::endl;
      }
      duc[0][c] = tmp * damping_factor;
      npre_clipped++;
    }
  }

  // output statistics
  if (vo_->getVerbLevel() >= Teuchos::VERB_HIGH) {
    int nsat_tmp = nsat_clipped, npre_tmp = npre_clipped;
    mesh_->get_comm()->SumAll(&nsat_tmp, &nsat_clipped, 1);
    mesh_->get_comm()->SumAll(&npre_tmp, &npre_clipped, 1);

    if (nsat_clipped > 0 || npre_clipped > 0) {
      Teuchos::OSTab tab = vo_->getOSTab();
      *vo_->os() << vo_->color("green") << "saturation/pressure clipped in " 
                 << nsat_clipped << "/" << npre_clipped << " cells" << vo_->reset() << std::endl;
    }
  }

  return (nsat_clipped + npre_clipped) > 0 ? AmanziSolvers::FnBaseDefs::CORRECTION_MODIFIED :
      AmanziSolvers::FnBaseDefs::CORRECTION_NOT_MODIFIED;
}

}  // namespace Flow
}  // namespace Amanzi



