/*
  This is the Operator component of the Amanzi code.

  Copyright 2010-2013 held jointly by LANS/LANL, LBNL, and PNNL. 
  Amanzi is released under the three-clause BSD License. 
  The terms of use and "as is" disclaimer for this license are 
  provided in the top-level COPYRIGHT file.

  Authors: Konstantin Lipnikov (lipnikov@lanl.gov)
           Ethan Coon (ecoon@lanl.gov)
*/

#ifndef AMANZI_OPERATOR_HH_
#define AMANZI_OPERATOR_HH_

#include <vector>

#include "Teuchos_RCP.hpp"
#include "Teuchos_ParameterList.hpp"
#include "Epetra_CrsMatrix.h"

#include "Mesh.hh"
#include "CompositeVectorSpace.hh"
#include "CompositeVector.hh"

#include "BCs.hh"

/* ******************************************************************
1. Operator is a linear operator acting from linear space X to linear
space Y. These spaces are described by CompositeVectors (CV). A few
maps X->Y is supported. 

At the moment X = Y.

2. An Operator consists of, potentially, both a single, global, assembled
matrix AND an un-ordered, additive collection of lower-rank (or equal) local
operators, here called 'Op's. During its construction, an operator can grow by
assimilating more Ops.  An Operator knows how to Apply and Assemble all of its
local Ops.

3. Typically the forward operator is applied using only local Ops -- the
inverse operator typically requires assembling a matrix, which may represent
the entire operator or may be a Schur complement.

4. In all Operators, Ops, and matrices, a key concept is the schema.  A schema
includes, at the least, an enum representing the dofs associated with the
Operator/matrix's domain (and implied equivalent range, X=Y).  A schema may
also, in the case of Ops, include information on the base entity on which the
local matrix lives.

5. Note on the construction of Operators: For simple operations
(i.e. diffusion), Operators are not constructed directly.  Instead, a helper
class that contains methods for creating and populating the Ops within the
Operator is created.  This helper class can create the appropriate Operator
itself.  More complex operations, i.e. advection-diffusion, can be generated by
creating an Operator that is the union of all dof requirements, and then
passing this Operator into the helper's constructor.  When this is done, the
helper simply checks to make sure the Operator contains the necessary dofs and
adds local Ops to the Operator's list of Ops.

6. Note on implementation for developers: Ops work via a visitor pattern.
Assembly (resp Apply, BCs, and SymbolicAssembly) are implemented by the (base
class) Operator calling a dispatch to the (base virtual class) Op, which then
dispatches back to the (derived class) Operator so that type information of
both the Operator (i.e. global matrix info) and the Op (i.e. local matrix
info) are kown.

****************************************************************** */ 

namespace Amanzi {

namespace AmanziPreconditioners { class Preconditioner; }
class CompositeVector;

namespace Operators {

class SuperMap;
class MatrixFE;
class GraphFE;
class Op;
class Op_Cell_FaceCell;
class Op_Cell_Face;
class Op_Cell_Cell;
class Op_Cell_Node;
class Op_Face_Cell;
class Op_Node_Node;


class Operator {
 public:
  // constuctors
  // main constructor
  //   The CVS is the domain and range of the operator
  Operator() {}
  Operator(const Teuchos::RCP<const CompositeVectorSpace>& cvs,
           Teuchos::ParameterList& plist,
           int schema);

  void Init();

  // main members
  // -- virtual methods potentially altered by the schema
  virtual int Apply(const CompositeVector& X, CompositeVector& Y, double scalar=0.) const;
  virtual int ApplyInverse(const CompositeVector& X, CompositeVector& Y) const;

  // symbolic assembly:
  // -- wrapper
  virtual void SymbolicAssembleMatrix();
  // -- first dispatch
  virtual void SymbolicAssembleMatrix(const SuperMap& map,
          GraphFE& graph, int my_block_row, int my_block_col) const;
  
  virtual void AssembleMatrix();
  virtual void AssembleMatrix(const SuperMap& map,
          MatrixFE& matrix, int my_block_row, int my_block_col) const;

  virtual void SetBCs(const Teuchos::RCP<BCs>& bc) { bc_ = bc; }
  virtual void UpdateRHS(const CompositeVector& source, bool volume_included=true);
  virtual void Rescale(const CompositeVector& scaling);

  // -- default functionality
  const CompositeVectorSpace& DomainMap() const { return *cvs_; }
  const CompositeVectorSpace& RangeMap() const { return *cvs_; }

  int ComputeResidual(const CompositeVector& u, CompositeVector& r);
  int ComputeNegativeResidual(const CompositeVector& u, CompositeVector& r);

  void InitPreconditioner(const std::string& prec_name, const Teuchos::ParameterList& plist);

  void CreateCheckPoint();
  void RestoreCheckPoint();

  // -- supporting members

  // access
  int schema() const { return schema_; }
  const std::string& schema_string() const { return schema_string_; }
  void set_schema_string(const std::string& schema_string) { schema_string_ = schema_string; }

  Teuchos::RCP<Epetra_CrsMatrix> A() { return A_; }
  Teuchos::RCP<const Epetra_CrsMatrix> A() const { return A_; }
  Teuchos::RCP<CompositeVector> rhs() { return rhs_; }
  Teuchos::RCP<const CompositeVector> rhs() const { return rhs_; }
  bool data_validity() { return data_validity_; }

  // block access
  typedef std::vector<Teuchos::RCP<Op> >::const_iterator const_op_iterator;
  const_op_iterator OpBegin() const { return ops_.begin(); }
  const_op_iterator OpEnd() const { return ops_.end(); }
  const_op_iterator FindMatrixOp(int schema_dofs, int matching_rule, bool action) const;

  typedef std::vector<Teuchos::RCP<Op> >::iterator op_iterator;
  op_iterator OpBegin() { return ops_.begin(); }
  op_iterator OpEnd() { return ops_.end(); }
  op_iterator FindMatrixOp(int schema_dofs, int matching_rule, bool action);

  // block mutate
  void OpPushBack(const Teuchos::RCP<Op>& block);
  void OpExtend(op_iterator begin, op_iterator end);

 public:

  // visit methods for Apply
  virtual int ApplyMatrixFreeOp(const Op_Cell_FaceCell& op,
      const CompositeVector& X, CompositeVector& Y) const;
  virtual int ApplyMatrixFreeOp(const Op_Cell_Face& op,
      const CompositeVector& X, CompositeVector& Y) const;
  virtual int ApplyMatrixFreeOp(const Op_Cell_Node& op,
      const CompositeVector& X, CompositeVector& Y) const;
  virtual int ApplyMatrixFreeOp(const Op_Cell_Cell& op,
      const CompositeVector& X, CompositeVector& Y) const;
  virtual int ApplyMatrixFreeOp(const Op_Face_Cell& op,
      const CompositeVector& X, CompositeVector& Y) const;
  virtual int ApplyMatrixFreeOp(const Op_Node_Node& op,
      const CompositeVector& X, CompositeVector& Y) const;

  // visit methods for RHS
  virtual void AssembleRHSOp(const Op_Cell_FaceCell& op,
                            CompositeVector& rhs) const;
  virtual void AssembleRHSOp(const Op_Cell_Face& op,
                            CompositeVector& rhs) const;
  virtual void AssembleRHSOp(const Op_Cell_Node& op,
          CompositeVector& rhs) const;
  virtual void AssembleRHSOp(const Op_Cell_Cell& op,
          CompositeVector& rhs) const;
  virtual void AssembleRHSOp(const Op_Face_Cell& op,
          CompositeVector& rhs) const;
  virtual void AssembleRHSOp(const Op_Node_Node& op,
          CompositeVector& rhs) const;

  // visit methods for symbolic assemble
  virtual void SymbolicAssembleMatrixOp(const Op_Cell_FaceCell& op,
          const SuperMap& map, GraphFE& graph,
          int my_block_row, int my_block_col) const;
  virtual void SymbolicAssembleMatrixOp(const Op_Cell_Face& op,
          const SuperMap& map, GraphFE& graph,
          int my_block_row, int my_block_col) const;
  virtual void SymbolicAssembleMatrixOp(const Op_Cell_Node& op,
          const SuperMap& map, GraphFE& graph,
          int my_block_row, int my_block_col) const;
  virtual void SymbolicAssembleMatrixOp(const Op_Cell_Cell& op,
          const SuperMap& map, GraphFE& graph,
          int my_block_row, int my_block_col) const;
  virtual void SymbolicAssembleMatrixOp(const Op_Face_Cell& op,
          const SuperMap& map, GraphFE& graph,
          int my_block_row, int my_block_col) const;
  virtual void SymbolicAssembleMatrixOp(const Op_Node_Node& op,
          const SuperMap& map, GraphFE& graph,
          int my_block_row, int my_block_col) const;
  
  // visit methods for assemble
  virtual void AssembleMatrixOp(const Op_Cell_FaceCell& op,
          const SuperMap& map, MatrixFE& mat,
          int my_block_row, int my_block_col) const;
  virtual void AssembleMatrixOp(const Op_Cell_Face& op,
          const SuperMap& map, MatrixFE& mat,
          int my_block_row, int my_block_col) const;
  virtual void AssembleMatrixOp(const Op_Cell_Node& op,
          const SuperMap& map, MatrixFE& mat,
          int my_block_row, int my_block_col) const;
  virtual void AssembleMatrixOp(const Op_Cell_Cell& op,
          const SuperMap& map, MatrixFE& mat,
          int my_block_row, int my_block_col) const;
  virtual void AssembleMatrixOp(const Op_Face_Cell& op,
          const SuperMap& map, MatrixFE& mat,
          int my_block_row, int my_block_col) const;
  virtual void AssembleMatrixOp(const Op_Node_Node& op,
          const SuperMap& map, MatrixFE& mat,
          int my_block_row, int my_block_col) const;

 protected:
  Teuchos::RCP<const AmanziMesh::Mesh> mesh_;
  Teuchos::RCP<const CompositeVectorSpace> cvs_;
  mutable bool data_validity_;

  mutable std::vector<Teuchos::RCP<Op> > ops_;
  Teuchos::RCP<CompositeVector> rhs_, rhs_checkpoint_;
  Teuchos::RCP<BCs> bc_;

  int ncells_owned, nfaces_owned, nnodes_owned;
  int ncells_wghost, nfaces_wghost, nnodes_wghost;
 
  Teuchos::RCP<Epetra_CrsMatrix> A_;
  Teuchos::RCP<MatrixFE> Amat_;
  Teuchos::RCP<SuperMap> smap_;

  Teuchos::RCP<AmanziPreconditioners::Preconditioner> preconditioner_;

  Teuchos::RCP<VerboseObject> vo_;

  int schema_;
  std::string schema_string_;
  bool symbolic_assembled_;
  bool assembled_;

 private:
  Operator(const Operator& op);
  Operator& operator=(const Operator& op);
  
};

}  // namespace Operators
}  // namespace Amanzi


#endif
