/*
  This is the operator component of the Amanzi code. 

  Copyright 2010-2012 held jointly by LANS/LANL, LBNL, and PNNL. 
  Amanzi is released under the three-clause BSD License. 
  The terms of use and "as is" disclaimer for this license are 
  provided in the top-level COPYRIGHT file.

  Author: Ethan Coon (ecoon@lanl.gov)
*/

#include <cstdlib>
#include <cmath>
#include <iostream>
#include <string>
#include <vector>

#include "UnitTest++.h"

#include "Teuchos_RCP.hpp"
#include "Teuchos_ParameterList.hpp"
#include "Teuchos_ParameterXMLFileReader.hpp"

#include "MeshFactory.hh"

#include "Teuchos_TimeMonitor.hpp"
#include "Epetra_IntSerialDenseVector.h"
#include "Epetra_SerialDenseVector.h"
#include "Epetra_SerialDenseMatrix.h"
#include "Epetra_CrsMatrix.h"
#include "Epetra_FECrsGraph.h"
#include "Epetra_FECrsMatrix.h"
#include "GraphFE.hh"
#include "MatrixFE.hh"

/* *****************************************************************
 * this test is a null test -- all entries are local
* **************************************************************** */

TEST(FE_MATRIX_NEAREST_NEIGHBOR_TPFA_Epetra_FECrs) {
  using namespace Amanzi;
  using namespace Amanzi::AmanziMesh;
  using namespace Amanzi::AmanziGeometry;
  using namespace Amanzi::Operators;

  Epetra_MpiComm comm(MPI_COMM_WORLD);
  int MyPID = comm.MyPID();

  if (MyPID == 0) std::cout << "Test: FD like matrix, null off-proc assembly" << std::endl;

  // read parameter list
  std::string xmlFileName = "test/operator_convergence.xml";
  Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
  Teuchos::ParameterList plist = xmlreader.getParameters();
  Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("Cell FECrsMatrix");

  Amanzi::VerboseObject::hide_line_prefix = true;

  // create a mesh 
  Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
  GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

  FrameworkPreference pref;
  pref.clear();
  pref.push_back(MSTK);

  MeshFactory meshfactory(&comm);
  meshfactory.preference(pref);
  Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
  //Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

  // grab the maps
  int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
  Teuchos::RCP<Epetra_Map> cell_map = Teuchos::rcp(new Epetra_Map(mesh->cell_map(false)));
  Teuchos::RCP<Epetra_Map> cell_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->cell_map(true)));

  // create the graph
  int ierr(0);
  Teuchos::RCP<GraphFE> graph =
    Teuchos::rcp(new GraphFE(cell_map, cell_map_ghosted, cell_map_ghosted, 5));
  
  Entity_ID_List faces;
  Entity_ID_List face_cells;
  std::vector<int> neighbor_cells;
  for (int c=0; c!=ncells; ++c) {
    neighbor_cells.resize(0);
    neighbor_cells.push_back(c);
    
    mesh->cell_get_faces(c, &faces);
    for (int n=0; n!=faces.size(); ++n) {
      mesh->face_get_cells(faces[n], AmanziMesh::USED, &face_cells);
      if (face_cells.size() > 1) {
        neighbor_cells.push_back(c == face_cells[0] ? face_cells[1] : face_cells[0]);
      }	
    }

    ierr |= graph->InsertMyIndices(c, neighbor_cells.size(), &neighbor_cells[0]);
    CHECK(!ierr);
  }

  ierr |= graph->FillComplete(cell_map, cell_map);
  CHECK(!ierr);

  // and the control matrix
  Epetra_FECrsMatrix control(Copy, graph->Graph());
  {
    Teuchos::TimeMonitor monitor(*timer);
  for (int c=0; c!=ncells; ++c) {
    neighbor_cells.resize(0);
    neighbor_cells.push_back(c);
    
    mesh->cell_get_faces(c, &faces);
    for (int n=0; n!=faces.size(); ++n) {
      mesh->face_get_cells(faces[n], AmanziMesh::USED, &face_cells);
      if (face_cells.size() > 1) {
        neighbor_cells.push_back(c == face_cells[0] ? face_cells[1] : face_cells[0]);
      }
    }

    std::vector<double> vals(neighbor_cells.size(), 1);
    std::vector<int> neighbor_cell_gids(neighbor_cells.size());
    for (int n=0; n!=neighbor_cells.size(); ++n) {
      neighbor_cell_gids[n] = cell_map_ghosted->GID(neighbor_cells[n]);
      ASSERT(neighbor_cell_gids[n] >= 0);
    }
    ASSERT(cell_map_ghosted->GID(c) >= 0);
    ierr |= control.SumIntoGlobalValues(cell_map_ghosted->GID(c), neighbor_cells.size(), &vals[0], &neighbor_cell_gids[0]);
    ASSERT(!ierr);
  }

  ierr |= control.GlobalAssemble();
  }

  CHECK(!ierr);
}

TEST(FE_MATRIX_NEAREST_NEIGHBOR_TPFA_Epetra_FECrs_Nonlocal) {
  using namespace Amanzi;
  using namespace Amanzi::AmanziMesh;
  using namespace Amanzi::AmanziGeometry;
  using namespace Amanzi::Operators;

  Epetra_MpiComm comm(MPI_COMM_WORLD);
  int MyPID = comm.MyPID();

  if (MyPID == 0) std::cout << "Test: FD like matrix, null off-proc assembly" << std::endl;

  // read parameter list
  std::string xmlFileName = "test/operator_convergence.xml";
  Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
  Teuchos::ParameterList plist = xmlreader.getParameters();
  Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("Cell FECrsMatrix offproc");

  Amanzi::VerboseObject::hide_line_prefix = true;

  // create a mesh 
  Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
  GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

  FrameworkPreference pref;
  pref.clear();
  pref.push_back(MSTK);

  MeshFactory meshfactory(&comm);
  meshfactory.preference(pref);
  Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
  //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

  // grab the maps
  int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
  Teuchos::RCP<Epetra_Map> cell_map = Teuchos::rcp(new Epetra_Map(mesh->cell_map(false)));
  Teuchos::RCP<Epetra_Map> cell_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->cell_map(true)));

  // create the graph
  int ierr(0);
  Teuchos::RCP<Epetra_FECrsGraph> graph =
      Teuchos::rcp(new Epetra_FECrsGraph(Copy, *cell_map, *cell_map_ghosted, 5, false, true));
  
  Entity_ID_List faces;
  Entity_ID_List face_cells;
  std::vector<int> neighbor_cells;
  for (int c=0; c!=ncells; ++c) {
    neighbor_cells.resize(0);
    neighbor_cells.push_back(c);
    
    mesh->cell_get_faces(c, &faces);
    for (int n=0; n!=faces.size(); ++n) {
      mesh->face_get_cells(faces[n], AmanziMesh::USED, &face_cells);
      if (face_cells.size() > 1) {
        neighbor_cells.push_back(c == face_cells[0] ? face_cells[1] : face_cells[0]);
      }	
    }

    std::vector<int> neighbor_cell_gids(neighbor_cells.size());
    for (int n=0; n!=neighbor_cells.size(); ++n)
      neighbor_cell_gids[n] = cell_map_ghosted->GID(neighbor_cells[n]);

    ierr |= graph->InsertGlobalIndices(cell_map_ghosted->GID(c), neighbor_cells.size(), &neighbor_cell_gids[0]);
    CHECK(!ierr);
  }

  ierr |= graph->FillComplete(*cell_map, *cell_map);
  CHECK(!ierr);

  // and the control matrix
  Epetra_FECrsMatrix control(Copy, *graph);

  {
    Teuchos::TimeMonitor monitor(*timer);
  
  for (int c=0; c!=ncells; ++c) {
    neighbor_cells.resize(0);
    neighbor_cells.push_back(c);
    
    mesh->cell_get_faces(c, &faces);
    for (int n=0; n!=faces.size(); ++n) {
      mesh->face_get_cells(faces[n], AmanziMesh::USED, &face_cells);
      if (face_cells.size() > 1) {
        neighbor_cells.push_back(c == face_cells[0] ? face_cells[1] : face_cells[0]);
      }
    }

    std::vector<int> neighbor_cell_gids(neighbor_cells.size());
    for (int n=0; n!=neighbor_cells.size(); ++n) {
      neighbor_cell_gids[n] = cell_map_ghosted->GID(neighbor_cells[n]);
      ASSERT(neighbor_cell_gids[n] >= 0);
    }

    std::vector<double> vals(neighbor_cells.size(), 1);
    ASSERT(cell_map_ghosted->GID(c) >= 0);
    ierr |= control.SumIntoGlobalValues(cell_map_ghosted->GID(c), neighbor_cells.size(), &vals[0], &neighbor_cell_gids[0]);
    ASSERT(!ierr);
  }

  ierr |= control.GlobalAssemble();
  }
  CHECK(!ierr);
}

TEST(FE_MATRIX_NEAREST_NEIGHBOR_TPFA_MatrixFE) {
  using namespace Amanzi;
  using namespace Amanzi::AmanziMesh;
  using namespace Amanzi::AmanziGeometry;
  using namespace Amanzi::Operators;

  Epetra_MpiComm comm(MPI_COMM_WORLD);
  int MyPID = comm.MyPID();

  if (MyPID == 0) std::cout << "Test: FD like matrix, null off-proc assembly" << std::endl;

  // read parameter list
  std::string xmlFileName = "test/operator_convergence.xml";
  Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
  Teuchos::ParameterList plist = xmlreader.getParameters();
  Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("Cell MatrixFE");

  Amanzi::VerboseObject::hide_line_prefix = true;

  // create a mesh 
  Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
  GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

  FrameworkPreference pref;
  pref.clear();
  pref.push_back(MSTK);

  MeshFactory meshfactory(&comm);
  meshfactory.preference(pref);
  Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
  //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

  // grab the maps
  int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
  Teuchos::RCP<Epetra_Map> cell_map = Teuchos::rcp(new Epetra_Map(mesh->cell_map(false)));
  Teuchos::RCP<Epetra_Map> cell_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->cell_map(true)));

  // create the graph
  int ierr(0);
  Teuchos::RCP<GraphFE> graph =
    Teuchos::rcp(new GraphFE(cell_map, cell_map_ghosted, cell_map_ghosted, 5));
  
  Entity_ID_List faces;
  Entity_ID_List face_cells;
  std::vector<int> neighbor_cells;
  for (int c=0; c!=ncells; ++c) {
    neighbor_cells.resize(0);
    neighbor_cells.push_back(c);
    
    mesh->cell_get_faces(c, &faces);
    for (int n=0; n!=faces.size(); ++n) {
      mesh->face_get_cells(faces[n], AmanziMesh::USED, &face_cells);
      if (face_cells.size() > 1) {
        neighbor_cells.push_back(c == face_cells[0] ? face_cells[1] : face_cells[0]);
      }	
    }

    ierr |= graph->InsertMyIndices(c, neighbor_cells.size(), &neighbor_cells[0]);
    CHECK(!ierr);
  }

  ierr |= graph->FillComplete(cell_map, cell_map);
  CHECK(!ierr);

  // create the test matrix
  MatrixFE matrix(graph);
  {
    Teuchos::TimeMonitor monitor(*timer);
  
  for (int c=0; c!=ncells; ++c) {
    neighbor_cells.resize(0);
    neighbor_cells.push_back(c);
    
    mesh->cell_get_faces(c, &faces);
    for (int n=0; n!=faces.size(); ++n) {
      mesh->face_get_cells(faces[n], AmanziMesh::USED, &face_cells);
      if (face_cells.size() > 1) {
        neighbor_cells.push_back(c == face_cells[0] ? face_cells[1] : face_cells[0]);
      }
    }

    std::vector<double> vals(neighbor_cells.size(), 1);
    ierr |= matrix.SumIntoMyValues(c, neighbor_cells.size(), &vals[0], &neighbor_cells[0]);
    ASSERT(!ierr);
  }

  ierr |= matrix.FillComplete();

  }
  CHECK(!ierr);

}


/* *****************************************************************
 * this test is a real test with FE-like assembly of face-face system
* **************************************************************** */
// TEST(FE_MATRIX_FACE_FACE_Epetra_FECrsMatrix) {
//   using namespace Amanzi;
//   using namespace Amanzi::AmanziMesh;
//   using namespace Amanzi::AmanziGeometry;
//   using namespace Amanzi::Operators;

//   Epetra_MpiComm comm(MPI_COMM_WORLD);
//   int MyPID = comm.MyPID();

//   if (MyPID == 0) std::cout << "Test: FE like matrix, off-proc assembly" << std::endl;

//   // read parameter list
//   std::string xmlFileName = "test/operator_convergence.xml";
//   Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
//   Teuchos::ParameterList plist = xmlreader.getParameters();
//   Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("FF FECrsMatrix");

//   Amanzi::VerboseObject::hide_line_prefix = true;

//   // create a mesh 
//   Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
//   GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

//   FrameworkPreference pref;
//   pref.clear();
//   pref.push_back(MSTK);

//   MeshFactory meshfactory(&comm);
//   meshfactory.preference(pref);
//   Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
//   //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

//   // grab the maps
//   int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
//   Teuchos::RCP<Epetra_Map> face_map = Teuchos::rcp(new Epetra_Map(mesh->face_map(false)));
//   Teuchos::RCP<Epetra_Map> face_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->face_map(true)));

//   // create the graph
//   int ierr(0);
//   Teuchos::RCP<GraphFE> graph =
//       Teuchos::rcp(new GraphFE(face_map, face_map_ghosted, face_map_ghosted, 5));
  
//   Entity_ID_List faces;
//   Entity_ID_List face_cells;
//   for (int c=0; c!=ncells; ++c) {
//     mesh->cell_get_faces(c, &faces);

//     for (int n=0; n!=faces.size(); ++n) {
//       ierr |= graph->InsertMyIndices(faces[n], faces.size(), &faces[0]);
//       CHECK(!ierr);
//     }
//   }

//   ierr |= graph->FillComplete(face_map, face_map);
//   CHECK(!ierr);

//   // and the control matrix
//   Epetra_FECrsMatrix control(Copy, graph->Graph());

//   {
//     Teuchos::TimeMonitor monitor(*timer);
//   for (int c=0; c!=ncells; ++c) {
//     mesh->cell_get_faces(c, &faces);
//     std::vector<int> face_gids(faces.size());
//     for (int n=0; n!=faces.size(); ++n) {
//       face_gids[n] = face_map_ghosted->GID(faces[n]);
//     }

//     for (int n=0; n!=faces.size(); ++n) {
//       std::vector<double> vals(faces.size(), -1.);
//       vals[n] = faces.size()-1;

//       ierr |= control.SumIntoGlobalValues(face_gids[n], face_gids.size(), &vals[0], &face_gids[0]);
//     }
//   }
//   ierr |= control.GlobalAssemble();
//   }
//   CHECK(!ierr);
// }

// TEST(FE_MATRIX_FACE_FACE_Epetra_FECrsMatrix_offproc) {
//   using namespace Amanzi;
//   using namespace Amanzi::AmanziMesh;
//   using namespace Amanzi::AmanziGeometry;
//   using namespace Amanzi::Operators;

//   Epetra_MpiComm comm(MPI_COMM_WORLD);
//   int MyPID = comm.MyPID();

//   if (MyPID == 0) std::cout << "Test: FE like matrix, off-proc assembly" << std::endl;

//   // read parameter list
//   std::string xmlFileName = "test/operator_convergence.xml";
//   Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
//   Teuchos::ParameterList plist = xmlreader.getParameters();
//   Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("FF FECrsMatrix offproc");

//   Amanzi::VerboseObject::hide_line_prefix = true;

//   // create a mesh 
//   Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
//   GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

//   FrameworkPreference pref;
//   pref.clear();
//   pref.push_back(MSTK);

//   MeshFactory meshfactory(&comm);
//   meshfactory.preference(pref);
//   Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
//   //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

//   // grab the maps
//   int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
//   Teuchos::RCP<Epetra_Map> face_map = Teuchos::rcp(new Epetra_Map(mesh->face_map(false)));
//   Teuchos::RCP<Epetra_Map> face_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->face_map(true)));

//   // create the graph
//   int ierr(0);
//   Teuchos::RCP<Epetra_FECrsGraph> graph =
//       Teuchos::rcp(new Epetra_FECrsGraph(Copy, *face_map, *face_map_ghosted, 8, false, true));
  
//   Entity_ID_List faces;
//   for (int c=0; c!=ncells; ++c) {
//     mesh->cell_get_faces(c, &faces);

//     std::vector<int> face_gids(faces.size());
//     for (int n=0; n!=faces.size(); ++n)
//       face_gids[n] = face_map_ghosted->GID(faces[n]);
//     for (int n=0; n!=faces.size(); ++n) {
//       ierr |= graph->InsertGlobalIndices(face_gids[n], face_gids.size(), &face_gids[0]);
//       ASSERT(!ierr);
//     }
//   }

//   ierr |= graph->GlobalAssemble();
//   CHECK(!ierr);

//   // and the control matrix
//   Epetra_FECrsMatrix control(Copy, *graph);

//   {
//     Teuchos::TimeMonitor monitor(*timer);
//   for (int c=0; c!=ncells; ++c) {
//     mesh->cell_get_faces(c, &faces);
//     std::vector<int> face_gids(faces.size());
//     for (int n=0; n!=faces.size(); ++n) {
//       face_gids[n] = face_map_ghosted->GID(faces[n]);
//     }

//     for (int n=0; n!=faces.size(); ++n) {
//       std::vector<double> vals(faces.size(), -1.);
//       vals[n] = faces.size()-1;

//       ierr |= control.SumIntoGlobalValues(face_gids[n], face_gids.size(), &vals[0], &face_gids[0]);
//     }
//   }

//   ierr |= control.GlobalAssemble();
//   }
//   CHECK(!ierr);
//   Teuchos::TimeMonitor::summarize(std::cout);
// }

// TEST(FE_MATRIX_FACE_FACE_MatrixFE) {
//   using namespace Amanzi;
//   using namespace Amanzi::AmanziMesh;
//   using namespace Amanzi::AmanziGeometry;
//   using namespace Amanzi::Operators;

//   Epetra_MpiComm comm(MPI_COMM_WORLD);
//   int MyPID = comm.MyPID();

//   if (MyPID == 0) std::cout << "Test: FE like matrix, off-proc assembly" << std::endl;

//   // read parameter list
//   std::string xmlFileName = "test/operator_convergence.xml";
//   Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
//   Teuchos::ParameterList plist = xmlreader.getParameters();
//   Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("FF MatrixFE");

//   Amanzi::VerboseObject::hide_line_prefix = true;

//   // create a mesh 
//   Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
//   GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

//   FrameworkPreference pref;
//   pref.clear();
//   pref.push_back(MSTK);

//   MeshFactory meshfactory(&comm);
//   meshfactory.preference(pref);
//   Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
//   //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

//   // grab the maps
//   int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
//   Teuchos::RCP<Epetra_Map> face_map = Teuchos::rcp(new Epetra_Map(mesh->face_map(false)));
//   Teuchos::RCP<Epetra_Map> face_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->face_map(true)));

//   // create the graph
//   int ierr(0);
//   Teuchos::RCP<GraphFE> graph =
//       Teuchos::rcp(new GraphFE(face_map, face_map_ghosted, face_map_ghosted, 5));
  
//   Entity_ID_List faces;
//   Entity_ID_List face_cells;
//   for (int c=0; c!=ncells; ++c) {
//     mesh->cell_get_faces(c, &faces);

//     for (int n=0; n!=faces.size(); ++n) {
//       ierr |= graph->InsertMyIndices(faces[n], faces.size(), &faces[0]);
//       CHECK(!ierr);
//     }
//   }

//   ierr |= graph->FillComplete(face_map, face_map);
//   CHECK(!ierr);

//   // create the test matrix
//   MatrixFE matrix(graph);

//   {
//     Teuchos::TimeMonitor monitor(*timer);

//   for (int c=0; c!=ncells; ++c) {
//     mesh->cell_get_faces(c, &faces);

//     for (int n=0; n!=faces.size(); ++n) {
//       std::vector<double> vals(faces.size(), -1.);
//       vals[n] = faces.size()-1;

//       ierr |= matrix.SumIntoMyValues(faces[n], faces.size(), &vals[0], &faces[0]);
//     }
//   }

//   ierr |= matrix.FillComplete();
//   }
//   CHECK(!ierr);
// }

TEST(FE_MATRIX_FACE_FACE_Epetra_FECrsMatrix2) {
  using namespace Amanzi;
  using namespace Amanzi::AmanziMesh;
  using namespace Amanzi::AmanziGeometry;
  using namespace Amanzi::Operators;

  Epetra_MpiComm comm(MPI_COMM_WORLD);
  int MyPID = comm.MyPID();

  if (MyPID == 0) std::cout << "Test: FE like matrix, off-proc assembly" << std::endl;

  // read parameter list
  std::string xmlFileName = "test/operator_convergence.xml";
  Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
  Teuchos::ParameterList plist = xmlreader.getParameters();
  Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("Block FECrsMatrix");

  Amanzi::VerboseObject::hide_line_prefix = true;

  // create a mesh 
  Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
  GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

  FrameworkPreference pref;
  pref.clear();
  pref.push_back(MSTK);

  MeshFactory meshfactory(&comm);
  meshfactory.preference(pref);
  Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
  //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

  // grab the maps
  int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
  Teuchos::RCP<Epetra_Map> face_map = Teuchos::rcp(new Epetra_Map(mesh->face_map(false)));
  Teuchos::RCP<Epetra_Map> face_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->face_map(true)));

  // create the graph
  int ierr(0);
  Teuchos::RCP<GraphFE> graph =
      Teuchos::rcp(new GraphFE(face_map, face_map_ghosted, face_map_ghosted, 5));
  
  Entity_ID_List faces;
  Entity_ID_List face_cells;
  for (int c=0; c!=ncells; ++c) {
    mesh->cell_get_faces(c, &faces);

    for (int n=0; n!=faces.size(); ++n) {
      ierr |= graph->InsertMyIndices(faces[n], faces.size(), &faces[0]);
      CHECK(!ierr);
    }
  }

  ierr |= graph->FillComplete(face_map, face_map);
  CHECK(!ierr);

  // and the control matrix
  Epetra_FECrsMatrix control(Copy, graph->Graph());

  {
    Teuchos::TimeMonitor monitor(*timer);
  for (int c=0; c!=ncells; ++c) {
    mesh->cell_get_faces(c, &faces);
    Epetra_IntSerialDenseVector face_gids(faces.size());
    for (int n=0; n!=faces.size(); ++n) {
      face_gids[n] = face_map_ghosted->GID(faces[n]);
    }

    Epetra_SerialDenseMatrix vals(faces.size(), faces.size());
    ierr |= control.SumIntoGlobalValues(face_gids, vals);
  }
  ierr |= control.GlobalAssemble();
  }
  CHECK(!ierr);
}

TEST(FE_MATRIX_FACE_FACE_Epetra_FECrsMatrix_offproc2) {
  using namespace Amanzi;
  using namespace Amanzi::AmanziMesh;
  using namespace Amanzi::AmanziGeometry;
  using namespace Amanzi::Operators;

  Epetra_MpiComm comm(MPI_COMM_WORLD);
  int MyPID = comm.MyPID();

  if (MyPID == 0) std::cout << "Test: FE like matrix, off-proc assembly" << std::endl;

  // read parameter list
  std::string xmlFileName = "test/operator_convergence.xml";
  Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
  Teuchos::ParameterList plist = xmlreader.getParameters();
  Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("Block FECrsMatrix offproc");

  Amanzi::VerboseObject::hide_line_prefix = true;

  // create a mesh 
  Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
  GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

  FrameworkPreference pref;
  pref.clear();
  pref.push_back(MSTK);

  MeshFactory meshfactory(&comm);
  meshfactory.preference(pref);
  Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
  //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

  // grab the maps
  int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
  Teuchos::RCP<Epetra_Map> face_map = Teuchos::rcp(new Epetra_Map(mesh->face_map(false)));
  Teuchos::RCP<Epetra_Map> face_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->face_map(true)));

  // create the graph
  int ierr(0);
  Teuchos::RCP<Epetra_FECrsGraph> graph =
      Teuchos::rcp(new Epetra_FECrsGraph(Copy, *face_map, *face_map_ghosted, 8, false, true));
  
  Entity_ID_List faces;
  for (int c=0; c!=ncells; ++c) {
    mesh->cell_get_faces(c, &faces);

    std::vector<int> face_gids(faces.size());
    for (int n=0; n!=faces.size(); ++n)
      face_gids[n] = face_map_ghosted->GID(faces[n]);

    ierr |= graph->InsertGlobalIndices(face_gids.size(), &face_gids[0], face_gids.size(), &face_gids[0]);
    ASSERT(!ierr);
  }

  ierr |= graph->FillComplete(*face_map, *face_map);
  CHECK(!ierr);

  // and the control matrix
  Epetra_FECrsMatrix control(Copy, *graph);

  {
    Teuchos::TimeMonitor monitor(*timer);
  for (int c=0; c!=ncells; ++c) {
    mesh->cell_get_faces(c, &faces);
    Epetra_IntSerialDenseVector face_gids(faces.size());
    for (int n=0; n!=faces.size(); ++n) {
      face_gids[n] = face_map_ghosted->GID(faces[n]);
    }

    Epetra_SerialDenseMatrix vals(faces.size(), faces.size());
    ierr |= control.SumIntoGlobalValues(face_gids, vals);
  }

  ierr |= control.GlobalAssemble();
  }
  CHECK(!ierr);
}

TEST(FE_MATRIX_FACE_FACE_MatrixFE2) {
  using namespace Amanzi;
  using namespace Amanzi::AmanziMesh;
  using namespace Amanzi::AmanziGeometry;
  using namespace Amanzi::Operators;

  Epetra_MpiComm comm(MPI_COMM_WORLD);
  int MyPID = comm.MyPID();

  if (MyPID == 0) std::cout << "Test: FE like matrix, off-proc assembly" << std::endl;

  // read parameter list
  std::string xmlFileName = "test/operator_convergence.xml";
  Teuchos::ParameterXMLFileReader xmlreader(xmlFileName);
  Teuchos::ParameterList plist = xmlreader.getParameters();
  Teuchos::RCP<Teuchos::Time> timer = Teuchos::TimeMonitor::getNewTimer("Block MatrixFE");

  Amanzi::VerboseObject::hide_line_prefix = true;

  // create a mesh 
  Teuchos::ParameterList region_list = plist.get<Teuchos::ParameterList>("Regions");
  GeometricModelPtr gm = new GeometricModel(2, region_list, &comm);

  FrameworkPreference pref;
  pref.clear();
  pref.push_back(MSTK);

  MeshFactory meshfactory(&comm);
  meshfactory.preference(pref);
  Teuchos::RCP<Mesh> mesh = meshfactory(0.0, 0.0, 1.0, 1.0, 100, 1000, gm);
  //  Teuchos::RCP<const Mesh> mesh = meshfactory("test/median32x33.exo", gm);

  // grab the maps
  int ncells = mesh->num_entities(AmanziMesh::CELL, AmanziMesh::OWNED);
  Teuchos::RCP<Epetra_Map> face_map = Teuchos::rcp(new Epetra_Map(mesh->face_map(false)));
  Teuchos::RCP<Epetra_Map> face_map_ghosted = Teuchos::rcp(new Epetra_Map(mesh->face_map(true)));

  // create the graph
  int ierr(0);
  Teuchos::RCP<GraphFE> graph =
      Teuchos::rcp(new GraphFE(face_map, face_map_ghosted, face_map_ghosted, 5));
  
  Entity_ID_List faces;
  Entity_ID_List face_cells;
  for (int c=0; c!=ncells; ++c) {
    mesh->cell_get_faces(c, &faces);

    for (int n=0; n!=faces.size(); ++n) {
      ierr |= graph->InsertMyIndices(faces[n], faces.size(), &faces[0]);
      CHECK(!ierr);
    }
  }

  ierr |= graph->FillComplete(face_map, face_map);
  CHECK(!ierr);

  // create the test matrix
  MatrixFE matrix(graph);

  {
    Teuchos::TimeMonitor monitor(*timer);

  for (int c=0; c!=ncells; ++c) {
    mesh->cell_get_faces(c, &faces);

    Epetra_SerialDenseMatrix vals(faces.size(), faces.size());
    ierr |= matrix.SumIntoMyValues(&faces[0], vals);
  }

  ierr |= matrix.FillComplete();
  }
  CHECK(!ierr);
}

