module mod_read_netcdf

use netcdf

implicit none

public :: read_netcdf_hybrid

character(len=512) :: string1, string2, string3

!----------------------------------------------------------------------------
contains
!----------------------------------------------------------------------------

subroutine read_netcdf_hybrid

use mod_adm, only : r8, orig, ifile_orog, ifile_netcdf, numoffile, u_input, &
                    v_input, t_input, q_input, ppp, orog, nx, ny, nz
use mod_time

integer :: inum, idm, ivr, J

! Input field requirements
integer                          :: Nf=6
character(len=256), dimension(4) :: Req_dmnms
character(len=256), dimension(6) :: Req_vrnms
character(len=256), dimension(6) :: Req_longvrnms
logical, dimension(6)            :: inp_check(6) = .false.

logical :: CF_compliant

! NetCDF info storage
integer            :: nFiles, ncid, nDim, nVar, nAtt, unlimDimID
character(len=256), allocatable  :: dimname(:), vrname(:)  ! All dimension names, All variable names per NetCDF file
integer, allocatable :: ldim(:), varNdim(:), varNatt(:) ! # elements per dimension, # dimensions per variable, # attributes per variable
integer, allocatable :: vardimIDs(:,:) ! All variables with their dimension IDs (e.g. lon(lon), u(x,y,z),lnsp(x,y),...)

! Input fields
real(r8) :: inp_lon(nx), inp_lat(ny), inp_lev(nz)
real(r8), allocatable :: inp_values(:,:,:,:)
real(r8), allocatable :: inp_values3D(:,:,:)
real(r8), allocatable :: inp_values2D(:,:)

! Input time
integer :: time_offset
character(len=256) :: time_units
!character(len=256) :: dataTime
!character(len=256) :: dataStep

! Variables to extract: long names
Req_longvrnms(1) = "U component of wind"
Req_longvrnms(2) = "V component of wind"
Req_longvrnms(3) = "Temperature"
Req_longvrnms(4) = "Specific humidity"
Req_longvrnms(5) = "Logarithm of surface pressure"
Req_longvrnms(6) = "Surface geopotential (orography)"

! Variable names as given in NetCDF files per orig
if (orig == "ECMWF") then
     Req_dmnms(1) = "lon"
     Req_dmnms(2) = "lat"
     Req_dmnms(3) = "lev"
     Req_dmnms(4) = "time"
     Req_vrnms(1) = "u"
     Req_vrnms(2) = "v"
     Req_vrnms(3) = "t"
     Req_vrnms(4) = "q"
     Req_vrnms(5) = "lnsp"
     Req_vrnms(6) = "z"
endif

if (orig == "EC-EARTH") then
     Req_dmnms(1) = "lon"
     Req_dmnms(2) = "lat"
     Req_dmnms(3) = "mlev"
     Req_dmnms(4) = "time"
     Req_vrnms(1) = "var131"
     Req_vrnms(2) = "var132"
     Req_vrnms(3) = "var130"
     Req_vrnms(4) = "var133"
     Req_vrnms(5) = "var134"
     Req_vrnms(6) = "var129"
endif

if (orig == "CMIP5") then
     Req_dmnms(1) = "lon"
     Req_dmnms(2) = "lat"
     Req_dmnms(3) = "lev"
     Req_dmnms(4) = "time"
     Req_vrnms(1) = "ua"
     Req_vrnms(2) = "va"
     Req_vrnms(3) = "ta"
     Req_vrnms(4) = "hus"
     Req_vrnms(5) = "ps"
     Req_vrnms(6) = "orog"
endif

! We recommend that each variable carry a "units" attribute.
!  character (len = *), parameter :: UNITS = "units"
!  character (len = *), parameter :: PRES_UNITS = "hPa"
!  character (len = *), parameter :: TEMP_UNITS = "celsius"
!  character (len = *), parameter :: LAT_UNITS = "degrees_north"
!  character (len = *), parameter :: LON_UNITS = "degrees_east"

! TJH len(trim( has proven to be non-portable ...

if (len(trim(ifile_orog)) == 0) Nfiles = numoffile
! Include orography nc file in iteration process when specified by user
if (len(trim(ifile_orog)) /= 0) Nfiles = numoffile+1

FILELOOP : do inum = 1, Nfiles

   write(string1,*) 'NetCDF file: [',trim(ifile_netcdf(inum)),']'
   write(*,*)trim(string1)
   call nc_check( nf90_open(ifile_netcdf(inum), nf90_nowrite, ncid), 'opening '//trim(string1))

   !If file is CF-compliant, we know what the coordinate variable names should be.
   call IsFileCF_compliant(ncid, ifile_netcdf(inum), Req_dmnms, Req_vrnms, Req_longvrnms, CF_compliant)

   !write(*,*) "1. NetCDF file: ", trim(ifile_netcdf(inum)), " with ID ", ncid, " has been opened"

   ! <2> Examine NetCDF file (# dimensions, # variables, # attributes)
   call nc_check( nf90_inquire(ncid, nDim, nVar, nAtt, unlimDimID), &
                     "inquire "//trim(string1))

   !write(*,*) "2. # dimensions, # variables and # attributes"
   !write(*,*) nDim, nVar, nAtt
   !write(*,*) "   unlimited dimension ID = ", unlimDimID

   ! - allocate arrays
   allocate(dimname(nDim))
   allocate(   ldim(nDim))
   allocate( vrname(nVar))
   allocate(varNdim(nVar))
   allocate(varNatt(nVar))
   allocate(vardimIDs(nVar,nf90_max_var_dims))

   ! - for dimensions
   !write(*,*) "3. Examine dimensions"
   do idm = 1, nDim
       call nc_check( nf90_inquire_dimension(ncid, idm, dimname(idm), ldim(idm)), string2)
       write(string2,*) "Dimension ",idm, "name=", trim(dimname(idm)), 'length=',ldim(idm)
   enddo
   ! result: lon(256), lat(256), lev(60), nhym(60), nhyi(61), lev_2(1), time(1)

   ! - for variables
   !write(*,*) "4. Examine variables"
   do ivr = 1, nVar
      call nc_check( nf90_inquire_variable(ncid, ivr, vrname(ivr), &
                         ndims=varNdim(ivr), dimids=vardimIDs(ivr,:),  &
                         natts=varNatt(ivr)),'inquire variable'//trim(vrname(ivr)) )
   enddo

   ! <3> Overview of variables found in NetCDF:
   write(*,*) "========= NetCDF file content overview =========="
   write(*,*) ">> Variables found <<"
   do ivr = 1, nVar
      ! TJH This bombs out on scalar variables ... 0D
      ! write(*,*) trim(vrname(ivr)),' with dimensions: ', ldim(vardimIDs(ivr,1))
   enddo
   write(*,*) ""

   ! <4> Extract grid data
   write(*,*) "============ Start extracting grid data ============"

   do ivr = 1, nVar

      write(string1,*)'get_var ['//trim(vrname(ivr))//']'

      ! - lon
      if (trim(vrname(ivr)) == trim(Req_dmnms(1))) then
         call nc_check( nf90_get_var(ncid, ivr, inp_lon), string1 )
         write(*,*) "inp_lon:   from", inp_lon(1), "till", inp_lon(ldim(vardimIDs(ivr,1)))
      endif

      ! - lat
      if (trim(vrname(ivr)) == trim(Req_dmnms(2))) then
         call nc_check( nf90_get_var(ncid, ivr, inp_lat), string1 )
         write(*,*) "inp_lat:   from", inp_lat(1), "till", inp_lat(ldim(vardimIDs(ivr,1)))
      endif

      ! - lev
      ! FIXME ... could check the size of the variable against what was
      ! specified in the namelist. 
      if (trim(vrname(ivr)) == trim(Req_dmnms(3))) then
         call nc_check( nf90_get_var(ncid, ivr, inp_lev), string1 )
         write(*,*) "inp_lev:   from", inp_lev(1), "till", inp_lev(ldim(vardimIDs(ivr,1)))
      endif

      ! - time
      if (trim(vrname(ivr)) == trim(Req_dmnms(4))) then
        if ( CF_compliant ) then
           call Get_CF_time(ncid, vrname(ivr), ifile_netcdf(inum), time_units, time_offset)
        else
           call nc_check( nf90_get_att(ncid, ivr, "units" , time_units) )
           call nc_check( nf90_get_var(ncid, ivr, time_offset) )
           write(*,*) "time = ", trim(time_units), " + ", time_offset
        endif
      endif
   enddo

   ! <5> Extract data per variable
   write(*,*) "============ Start extracting fields ============"

   do ivr = 1, nVar

      ! - u
      if (trim(vrname(ivr)) == trim(Req_vrnms(1))) then
         allocate( inp_values(ldim(vardimIDs(ivr,1)),ldim(vardimIDs(ivr,2)), &
                              ldim(vardimIDs(ivr,3)),ldim(vardimIDs(ivr,4))) )
         call nc_check( nf90_get_var(ncid, ivr, inp_values) )
         u_input = inp_values(:,:,:,1)
         inp_check(1) = .true.
         write(*,*) "Zonal wind field extracted"
         deallocate(inp_values)
      endif

      ! - v
      if (trim(vrname(ivr)) == trim(Req_vrnms(2))) then
         allocate( inp_values(ldim(vardimIDs(ivr,1)),ldim(vardimIDs(ivr,2)), &
                              ldim(vardimIDs(ivr,3)),ldim(vardimIDs(ivr,4))) )
         call nc_check( nf90_get_var(ncid, ivr, inp_values) )
         v_input = inp_values(:,:,:,1)
         inp_check(2) = .true.
         write(*,*) "Meridional wind field extracted"
         deallocate(inp_values)
      endif

      ! - t
      if (trim(vrname(ivr)) == trim(Req_vrnms(3))) then
         allocate( inp_values(ldim(vardimIDs(ivr,1)),ldim(vardimIDs(ivr,2)), &
                              ldim(vardimIDs(ivr,3)),ldim(vardimIDs(ivr,4))) )
         call nc_check( nf90_get_var(ncid, ivr, inp_values) )
         t_input = inp_values(:,:,:,1)
         inp_check(3) = .true.
         write(*,*) "Temperature field extracted"
         deallocate(inp_values)
      endif

       ! - q
       if (trim(vrname(ivr)) == trim(Req_vrnms(4))) then
          allocate( inp_values(ldim(vardimIDs(ivr,1)),ldim(vardimIDs(ivr,2)), &
                               ldim(vardimIDs(ivr,3)),ldim(vardimIDs(ivr,4))) )
          call nc_check( nf90_get_var(ncid, ivr, inp_values) )
          q_input = inp_values(:,:,:,1)
          inp_check(4) = .true.
          write(*,*) "Specific humidity field extracted"
          deallocate(inp_values)
       endif

       ! - lnsp (or sp)
       if (trim(vrname(ivr)) == trim(Req_vrnms(5))) then
          allocate( inp_values3D(ldim(vardimIDs(ivr,1)),ldim(vardimIDs(ivr,2)), &
                                 ldim(vardimIDs(ivr,3))) )
          call nc_check( nf90_get_var(ncid, ivr, inp_values3D) )
          ppp = inp_values3D(:,:,1)
          inp_check(5) = .true.
          write(*,*) "(Log.) surface pressure field extracted"
          deallocate(inp_values3D)
       endif

       ! - z
       if (trim(vrname(ivr)) == trim(Req_vrnms(6))) then
          allocate( inp_values2D(ldim(vardimIDs(ivr,1)),ldim(vardimIDs(ivr,2))) )
          call nc_check( nf90_get_var(ncid, ivr, inp_values2D) )
          orog = inp_values2D(:,:)
          inp_check(6) = .true.
          write(*,*) "Orography (surface geopotential) extracted"
          deallocate(inp_values2D)
       endif
   enddo

   ! deallocate arrays for this NetCDF file
   deallocate(dimname)
   deallocate(ldim)
   deallocate(vrname)
   deallocate(varNdim)
   deallocate(varNatt)
   deallocate(vardimIDs)

   ! close NetCDF file
   call nc_check( nf90_close(ncid) )

enddo FILELOOP

! <6> Check if all required variables (u,v,t,q,lnsp,z) are extracted from the user-given NetCDF files.
! If orography file (var = z) is not found yet, then read it from "ifile_orog" parameter in namelist
call check_fields(Nf, inp_check, Req_vrnms, Req_longvrnms)

! <7> Check grid data on their order per dimension to make sure:
! - Latitude: North pole to south pole
! - Altitude: Bottom to top >> also correct for negative orography here
call check_lat( inp_lat )
call check_lvl( inp_lev )
! - No negative orography, and check surface pressure field too (and save it)
call check_surface

! If we got this far, everything worked as expected. Yipee!
write(*,*) "*** SUCCESSFULLY read and processed all NetCDF files."
write(*,*) ""

!----------------------------------------------------------------------------
contains
!----------------------------------------------------------------------------


   subroutine check_fields(Nch, arrcheck, varnms, varlng)
   ! This subroutine checks if all fields are imported
   ! - and imports orography from separate file if necessary

   use mod_adm

   integer,          intent(in) :: Nch
   logical,          intent(in) :: arrcheck(Nch)
   character(len=*), intent(in) :: varnms(Nch)
   character(len=*), intent(in) :: varlng(Nch)

   integer :: ic

   do ic=1, Nch
      if (.NOT. arrcheck(ic)) then
         write(*,*) "Variable ", trim(varnms(ic)), " (", trim(varlng(ic)), &
                      " ) not found in the provided NetCDF files. Program aborted."
         if (trim(varnms(ic)) == 'z') then
            write(*,*) "Please supply the orography file in the namelist ", &
            "(i) as a single netCDF file (without date-time string) through 'ifile_orog' or ", &
            "(ii) for each time step through 'ifile_head' parameter."
         endif
         stop
      endif
   enddo

   end subroutine check_fields




   subroutine check_lat(latin)
   ! This subroutine checks and reorders the input data such that it starts from the north pole to south pole

   use mod_adm

   real(r8), intent(inout) :: latin(ny)

   real(r8) :: templat(ny)
   logical  :: revlat = .false.

   real(r8) :: x(nx,ny,nz)
   real(r8) :: xs(nx,ny)

   ! lat array should be from NP down to SP
   if (latin(1) < latin(ny)) then
      templat = latin(ny:1:-1)
      latin = templat
      revlat = .true.
      write(*,*) "NOTE: Input data will be latitudally reversed (NP to SP)"
   endif

   !Now, reorder the input data too
   if (revlat) then
      x(:,:,:) = 0.0_r8
      x(:,:,:) = u_input(:,ny:1:-1,:)
      u_input(:,:,:) = x(:,:,:)
      x(:,:,:) = 0.0_r8
      x(:,:,:) = v_input(:,ny:1:-1,:)
      v_input(:,:,:) = x(:,:,:)
      x(:,:,:) = 0.0_r8
      x(:,:,:) = t_input(:,ny:1:-1,:)
      t_input(:,:,:) = x(:,:,:)
      x(:,:,:) = 0.0_r8
      x(:,:,:) = q_input(:,ny:1:-1,:)
      q_input(:,:,:) = x(:,:,:)
      xs(:,:) = 0.0_r8
      xs(:,:) = ppp(:,ny:1:-1)
      ppp(:,:) = xs(:,:)
      xs(:,:) = 0.0_r8
      xs(:,:) = orog(:,ny:1:-1)
      orog(:,:) = xs(:,:)
   endif
   end subroutine check_lat




   subroutine check_lvl(levin)
   ! This subroutine checks and reorders the input data such that the level 1 is the top level
   use mod_adm

   real(r8), intent( inout)  :: levin(nz)

   real(r8) :: templev(nz)
   logical  :: revlev = .false.
   real(r8) :: x(nx,ny,nz)

   if (levin(1) > levin(nz)) then
      templev = levin(nz:1:-1)
      levin = templev
      revlev = .true.
      write(*,*) "NOTE: Input data will be altitudally reversed (TOP to BOTTOM)"
   endif

   !Now, reorder the input data too
   if (revlev) then
      x(:,:,:) = 0.0_r8
      x(:,:,:) = u_input(:,:,nz:1:-1)
      u_input(:,:,:) = x(:,:,:)
      x(:,:,:) = 0.0_r8
      x(:,:,:) = v_input(:,:,nz:1:-1)
      v_input(:,:,:) = x(:,:,:)
      x(:,:,:) = 0.0_r8
      x(:,:,:) = t_input(:,:,nz:1:-1)
      t_input(:,:,:) = x(:,:,:)
      x(:,:,:) = 0.0_r8
      x(:,:,:) = q_input(:,:,nz:1:-1)
      q_input(:,:,:) = x(:,:,:)
   endif
   end subroutine check_lvl



   subroutine check_surface
   ! This subroutine puts negative orography to zero
   use mod_adm

   integer ::  iy, ix
   real(r8) :: flilla
   real(r8) :: xs(nx,ny)
   character(len=256) :: fname, tmp_fname

   ! TJH Question ... is flilla 14.0 or 1.0_d0E-15
   ! Remove negative orography here
   flilla = 1.0d0-15
   do ix = 1, nx
      do iy = 1, ny
        if ( orog(ix,iy) < 0.0_r8 )  orog(ix,iy) = flilla
      enddo
   enddo

   ! Surface pressure values given in logarithmic form or not?
   ! If given in log(sp) form then averaged field value wouldn't exceed 50
   ! If it does exceed, derive lnsp = dlog(sp)
   if (sum(ppp)/(nx*ny) > 50.0) then
      write(*,*) "Average surface pressure: ", sum(ppp)/(nx*ny)
      write(*,*) "Too high values >> ln(Ps) values are expected."
      xs(:,:) = dlog(ppp(:,:))
      ppp(:,:) = xs(:,:)
      write(*,*) "New values of log(surface pressure): ", sum(ppp)/(nx*ny)
   endif

   ! save surface pressure for later use during the inversion
   if(saveps) then
      fname="Ps_"
      tmp_fname=trim(ps_fname)//trim(cdate)
      call output_2d(trim(tmp_fname), ppp, nx, ny, 1, .true., 8)
   endif

   end subroutine check_surface



end subroutine read_netcdf_hybrid



!>
!> IsFileCF_compliant() checks to see if the netCDF file uses the CF convention.
!>

subroutine IsFileCF_compliant(ncid, fname, Req_dmnms, Req_vrnms, Req_longvrnms, CF_compliant)

integer,                        intent(inout)    :: ncid   ! TJH FIXME after error test
character(len=*),               intent(in)    :: fname
character(len=*), dimension(*), intent(inout) :: Req_dmnms
character(len=*), dimension(*), intent(inout) :: Req_vrnms
character(len=*), dimension(*), intent(inout) :: Req_longvrnms
logical,                        intent(out)   :: CF_compliant

integer :: nDimensions, nVariables, nAttributes, unlimitedDimId, formatNum
character(len=NF90_MAX_NAME) :: att_name
character(len=NF90_MAX_NAME) :: att_value
integer :: iatt, att_type, att_len , att_num

CF_compliant = .false.   ! until proven otherwise

! Determine how many global attributes there are.
call nc_check(nf90_inquire(ncid, nDimensions, nVariables, nAttributes, &
                          unlimitedDimId, formatNum), 'nf90_inquire '//trim(fname))

if (nAttributes == 0) return   ! no way it can be CF-compliant

ATTRIBUTELOOP : do iatt = 1,nAttributes

   write(string1,*)iatt,trim(fname)

   call nc_check(nf90_inq_attname(ncid, NF90_GLOBAL, iatt, att_name), &
                     'inq_attname '//trim(string1))

   call nc_check(nf90_inquire_attribute(ncid, NF90_GLOBAL, att_name, att_type, att_len, att_num), &
                     'inquire_attribute '//trim(string1))

   write(*,*)'Checking attribute [',trim(att_name),'] of file [',trim(fname),']'

   if (att_type /= NF90_CHAR) cycle ATTRIBUTELOOP

   if ( att_name(1:10) == 'Convention' ) then

      call nc_check(nf90_get_att(ncid, NF90_GLOBAL, att_name, att_value), &
                     'Convention inq_get_att '//trim(string1))

      write(*,*)'TJH attribute was ',trim(att_value)

      if ( att_value(1:3) == 'CF-' ) then
         CF_compliant = .true.
         write(*,*)'TJH ',trim(fname),' uses CF convention.'
         exit ATTRIBUTELOOP
      endif

   endif

enddo ATTRIBUTELOOP

end subroutine IsFileCF_compliant



!> IsFileCF_compliant() checks to see if the netCDF file uses the CF convention.

subroutine Get_CF_time(ncid, varname, fname, time_units, time_offset)
integer,          intent(in)  :: ncid
character(len=*), intent(in)  :: varname
character(len=*), intent(in)  :: fname
character(len=*), intent(out) :: time_units
integer,          intent(out) :: time_offset

integer :: iatt, nAttributes
character(len=NF90_MAX_NAME) :: att_name
character(len=NF90_MAX_NAME) :: att_value

integer :: dimid, dimlen, varid, ierr, i
integer :: oyear, omonth, oday, ohour, omin, osec ! time origin
integer :: iyear, imonth, iday, ihour, imin, isec, rem
integer, allocatable, dimension(:) :: time, datetmp, datesec

time_units = 'none'
time_offset = -8888

! All the caminput.nc files I have seen have two variables of
! length 'time' (the unlimited dimension): date, datesec
! The rest of the routine presumes there is but one time in the file -
! print warning message if this is not the case.

call nc_check(nf90_inq_dimid(ncid, 'time', dimid), &
                  'inq_dimid time '//trim(fname))
call nc_check(nf90_inquire_dimension(ncid, dimid, len=dimlen), &
                  'inquire_dimension time '//trim(fname))

allocate(time(dimlen), datetmp(dimlen), datesec(dimlen))

call nc_check(nf90_inq_varid(ncid, 'time', varid), &
       'inq_varid time '//trim(fname))
call nc_check(nf90_get_var(ncid, varid, values=time), &
       'get_var time '//trim(fname))

call nc_check( nf90_inquire_variable(ncid, varid, natts=nAttributes), &
                   'inquire_variable '//trim(varname) )

ATTRIBUTELOOP : do iatt = 1,nAttributes

   write(string1,*)trim(fname),'time',iatt

   call nc_check(nf90_inq_attname(ncid, varid, iatt, att_name), &
                     'inq_attname '//trim(string1))

   write(*,*)'Checking attribute [',trim(att_name),'] of file [',trim(fname),']'

   ! FIXME ... could check to make sure they are using the gregorian calendar.

   ! Find the time base of the variable.
   if ( att_name(1:5) == 'units' ) then

      call nc_check(nf90_get_att(ncid, varid, att_name, att_value), &
                     'time inq_get_att '//trim(string1))

      ! time:units = "days since 2000-09-01 00:00:00" ;
      !               12345678901

      read(att_value,'(11x,i4,5(1x,i2))',iostat=ierr)oyear,omonth,oday,ohour,omin,osec

      if (ierr /= 0) then
         write(*,*)'Unable to parse time origin. Exiting with an error.'
         write(*,*)'time origin string is [',trim(att_value),']'
         stop
      endif

      write(*,*)'TJH FIXME YYYY MM DD HH MM SS',oyear,omonth,oday,ohour,omin,osec

      exit ATTRIBUTELOOP
   endif

enddo ATTRIBUTELOOP

! FIXME ... could be unique to CAM ... not sure.
call nc_check(nf90_inq_varid(ncid, 'date', varid), &
       'inq_varid date '//trim(fname))
call nc_check(nf90_get_var(ncid, varid, values=datetmp), &
       'get_var date '//trim(fname))

call nc_check(nf90_inq_varid(ncid, 'datesec', varid), &
       'inq_varid datesec '//trim(fname))
call nc_check(nf90_get_var(ncid, varid, values=datesec), &
       'get_var datesec '//trim(fname))

! FIXME NEDJELJKA here is the calendar work.
do i=1,size(time)
   write(*,*)'total time is ',oyear, omonth, oday+time(i), ohour, omin, osec
enddo 

! The 'date' is YYYYMMDD ... datesec is 'current seconds of current day'
iyear  = datetmp(dimlen) / 10000
rem    = datetmp(dimlen) - iyear*10000
imonth = rem / 100
iday   = rem - imonth*100

ihour  = datesec(dimlen) / 3600
rem    = datesec(dimlen) - ihour*3600
imin   = rem / 60
isec   = rem - imin*60

! some cam files are from before the start of the gregorian calendar.
! since these are 'arbitrary' years, just change the offset.

if (iyear < 1601) then
   write(*,*)' '
   write(*,*)'WARNING - ',trim(fname),' changing year from ',iyear,'to',iyear+1601
   write(*,*)'WARNING - ',trim(fname),' changing year from ',iyear,'to',iyear+1601
   write(*,*)' '
   iyear = iyear + 1601
endif

! TJH ... Hurrah

write(*,*)'TJH Debug time is ',time
write(*,*)'TJH Debug date is ',datetmp
write(*,*)'TJH Debug datesec is ',datesec
write(*,*)'TJH interpreted date ',iyear, imonth, iday, ihour, imin, isec

deallocate(time, datetmp, datesec)

end subroutine Get_CF_time


subroutine nc_check(mystatus, message)
   integer,                    intent(in) :: mystatus
   character(len=*), optional, intent(in) :: message

   if(mystatus /= nf90_noerr) then

      if (present(message)) then
         print *, trim(nf90_strerror(mystatus))//" "//trim(message)
      else
         print *, trim(nf90_strerror(mystatus))
      endif
      stop "Stopped"

   endif
end subroutine nc_check


end module mod_read_netcdf
