module mod_interpolation
  implicit none

  public  :: interpolation_hybrid2sigma
  public  :: interpolation_sigma2hybrid
  public  :: linear_interpolation
  public  :: sigma2hybrid_ecmwf
  private :: find_closest_grid
  private :: find_closest_ecmwf
  
  !----------------------------------------------------------------------------
contains
  !----------------------------------------------------------------------------
  subroutine interpolation_hybrid2sigma(n, y1, m, y2, lnpres)
    use mod_adm, only: & 
        vgrid, aa, bb
    implicit none
    integer :: n ! Num. of vertical points in the original data
    integer :: m ! Num. of vertical points of the grid you want to interpolate
    double precision :: y1(n) ! value of original data
    double precision, intent(out) :: y2(m) ! value of interpolated data
    double precision :: lnpres !input value of ln(ps)
    double precision :: Slev(m)
    double precision :: phalf(0:n)
    double precision :: pfull(n), sfull(m)
 
!    double precision :: vgrid1(60)
            
    integer :: iz, ibelow, iabove
    double precision :: pbelow, pabove, pslev, deltap

     
    do iz = 1,m
      Slev(iz) = vgrid(m-iz+1) ! since the input file with sigma levels is sorted bottom-up
    end do
    do iz = 1,m
      sfull(iz) = Slev(iz)*dexp(lnpres) ! full sigma levels pressure
    end do
    
    phalf(0) = aa(0) + bb(0) * dexp(lnpres)! hybrid top half-level pressure
    do iz = 1,n
      phalf(iz) = aa(iz) + bb(iz) * dexp(lnpres)!
      pfull(iz) = (phalf(iz-1)+phalf(iz))*0.5d0 ! hybrid full levels pressure
    end do
          
    do iz = m,1,-1
      pslev = sfull(iz)
      
      call find_closest_ecmwf(pslev,n,pfull,ibelow,iabove)
      pbelow = dlog(pfull(ibelow))
      pabove = dlog(pfull(iabove)) 
      deltap=pbelow-pabove
     
      if(ibelow==iabove) then
        y2(iz)=y1(ibelow)
      elseif(ibelow > iabove) then
        y2(iz) = y1(ibelow)*(dlog(pslev)-pabove)/deltap + & 
                         y1(iabove)*(pbelow-dlog(pslev))/deltap
      else
        write(*,*) 'Hybrid2sigma: level below ',ibelow,' not below level above ',iabove
        stop
      end if
    end do

  end subroutine interpolation_hybrid2sigma 
  !----------------------------------------------------------------------------
  subroutine interpolation_sigma2hybrid(n, y1, m, y2, lnpres)
    use mod_adm, only: & 
        vgrid, aa, bb
    implicit none
    integer :: n ! Num. of vertical points in the original data
    integer :: m ! Num. of vertical points of the grid you want to interpolate
    double precision, intent(in) :: y1(n) ! value of original data, sigma levels
    double precision, intent(out) :: y2(m) ! value of interpolated data, hybrid levels
    double precision, intent(in) :: lnpres !input value of ln(ps)
    double precision :: Slev(n)
    double precision :: phalf(0:m)
    double precision :: pfull(m)
    double precision :: sfull(n)
        
    integer :: iz, ibelow, iabove
    double precision :: pbelow, pabove, pslev, deltap

    do iz = 1,n
      Slev(iz) = vgrid(n-iz+1)
    end do
    do iz = 1,n
      sfull(iz) = Slev(iz)*dexp(lnpres) ! full sigma levels pressure
    end do

    phalf(0) = aa(0) + bb(0) * dexp(lnpres)! hybrid top half-level pressure
    do iz = 1,m
!      phalf(iz-1) = aa(iz-1) + bb(iz-1) * dexp(lnpres)! hybrid half levels pressure
      phalf(iz) = aa(iz) + bb(iz) * dexp(lnpres)!
      pfull(iz) = (phalf(iz-1)+phalf(iz))*0.5d0 ! hybrid full levels pressure
    end do
         
    do iz = m,1,-1
      pslev = pfull(iz)
      call find_closest_ecmwf(pslev,n,sfull,ibelow,iabove)
!!      call find_closest_grid(pslev,n,sfull,ibelow,iabove)
      pbelow = dlog(sfull(ibelow))
      pabove = dlog(sfull(iabove)) 
      deltap=pbelow-pabove
     
      if(ibelow==iabove) then
        y2(iz)=y1(ibelow)
      elseif(ibelow > iabove) then
        y2(iz) = y1(ibelow)*(dlog(pslev)-pabove)/deltap + & 
                         y1(iabove)*(pbelow-dlog(pslev))/deltap
      else
        write(*,*) 'Sigma2hybrid: level below ',ibelow,' not below level above ',iabove
        stop
      end if  
    end do
    
  end subroutine interpolation_sigma2hybrid
  !----------------------------------------------------------------------------
  subroutine find_closest_ecmwf(psigma,n,pfull,ibelow,iabove)
    implicit none
    integer, intent(in) :: n
    integer, intent(out) :: ibelow, iabove
    double precision, intent(in) :: pfull(n)
    double precision, intent(in) :: psigma

    integer :: ibe, iab, k
    double precision :: pabove, pbelow, pz, zlittle

    pz = 0.0d0
    ibelow = 0
    iabove = n
    zlittle = 1.0d-15

    ! Find a nearest grid point above a given point
    pabove = psigma - pfull(1)  
    iab    = 1
    if (pabove < 0.0d0) then
     pabove = 0.0d0
     iab = 1
    end if
    do k= 2, n
     pz = psigma - pfull(k)
     if (pz > 0.0d0 .and. pz<pabove) then
      pabove = pz
      iab = k
     end if
    end do
    iabove = iab
    
    ! Find a nearest grid point below a given point
    pbelow = pfull(n) - psigma
    ibe = n
    do k= n-1,1,-1
     pz = pfull(k) - psigma
     if (pz > 0.0 .and. pz < pbelow) then
      pbelow = pz
      ibe = k
     end if
    end do
   ibelow = ibe
   
  end subroutine find_closest_ecmwf
  !----------------------------------------------------------------------------
  subroutine linear_interpolation(n, m, x1, y1, x2, y2)
    implicit none
    integer :: n ! Num. of points for original data
    integer :: m ! Num. of points which you want to interpolate
    double precision :: x1(n) ! value of grid for original data
    double precision :: y1(n) ! value of original data
    double precision :: x2(m) ! value of grid which you want to interpolate
    double precision, intent(out) :: y2(m) ! value of interpolated data

    integer :: iz, ibelow, iabove
    double precision :: pbelow, pabove

    do iz = 1,m
      call find_closest_grid(x2(iz), n, x1(:), ibelow, iabove)
      if(ibelow==iabove) then
        y2(iz)=y1(ibelow)
      else
        y2(iz)=( y1(ibelow)*(x1(iabove)-x2(iz)) +  &
                 y1(iabove)*(x2(iz)-x1(ibelow) ) ) / (x1(iabove)-x1(ibelow))
      end if
    end do

  end subroutine linear_interpolation 
  !----------------------------------------------------------------------------
  subroutine find_closest_grid(x1,m,x2,ibelow,iabove)
  ! assumes that points are sorted top to bottom, bottom is largest value
  ! to be done for a more general sorting 
    implicit none
    integer, intent(in) :: m
    integer, intent(inout) :: ibelow, iabove
    double precision, intent(inout) :: x2(m)
    double precision, intent(inout) :: x1

    integer :: ibe, iab, k
    double precision :: pabove, pbelow, pz, zlittle

    pz = 0.0d0
    ibelow = m
    iabove = 1
    zlittle = 1.0d-15

    ! Find a nearest grid point below a given point
    pbelow = x1 - x2(m) 
    ibe = m
    if(pbelow >= 0.0d0) then ! input point above the lowest point of the new grid
      pbelow = x2(m)
      ibe    = m
    end if
    do k = m-1, 1, -1
      pz = x1 - x2(k)
      if(pz<0.0d0 .and. pz>pbelow) then
        pbelow = pz
        ibe    = k
      end if
    end do
    ibelow = ibe

    ! Find a nearest grid point above a given point
    pabove = x2(1) - x1
    iab    = 1
    do k = 1, m
      pz = x2(k) - x1
      if(pz>0.0d0 .and. pz>pabove) then
        pabove = pz
        iab    = k
      end if
    end do
    iabove = iab

  end subroutine find_closest_grid
  !----------------------------------------------------------------------------
  subroutine sigma2hybrid_ecmwf(ctime)  
    use mod_adm
!    use mod_time
!    use mod_time, only: &
!        cdate
    implicit none
    integer :: ix, iy
    double precision, allocatable :: u_tmp(:,:,:)
    double precision, allocatable :: z_tmp(:,:,:)
    double precision, allocatable :: v_tmp(:,:,:)
    double precision :: Tm(nz)
    double precision :: ps(nx,ny)
    character(15), intent(in) :: ctime
    character(256) :: tmp_fname
    allocate(u_tmp(nx,ny,nz))
    allocate(v_tmp(nx,ny,nz))
    allocate(z_tmp(nx,ny,nz))
!
    tmp_fname=trim(meant_fname)//trim(ctime)
    call input_1d(trim(tmp_fname), Tm, nz, 1, .true., 8) 
    tmp_fname=trim(ps_fname)//trim(ctime) 
    call input_2d(trim(tmp_fname), ps, nx, ny, 1, .true., 8) 
!    call output_2d_gmt('p_surface.dat', ps, nx, ny)
 
    ! Interpolate from sigma levels back to hybrid levels   
    do iy = 1,ny
      do ix = 1,nx
        call interpolation_sigma2hybrid(nz,u_input(ix,iy,:),nz,u_tmp(ix,iy,:), &
                                        ps(ix,iy))
        call interpolation_sigma2hybrid(nz,v_input(ix,iy,:),nz,v_tmp(ix,iy,:), &
                                        ps(ix,iy))
        call interpolation_sigma2hybrid(nz,z_input(ix,iy,:),nz,z_tmp(ix,iy,:), & 
                                        ps(ix,iy))
      end do
    end do
    do iy = 1,ny
      do ix = 1,nx
        u_input(ix,iy,:)=u_tmp(ix,iy,:)
        v_input(ix,iy,:)=v_tmp(ix,iy,:)
        z_input(ix,iy,:)=z_tmp(ix,iy,:)
      end do
    end do
    deallocate(u_tmp)
    deallocate(v_tmp)
    deallocate(z_tmp)
    
  end subroutine sigma2hybrid_ecmwf   
  !----------------------------------------------------------------------------
       
end module mod_interpolation
