/** @file
  * @copyright University of Warsaw
  * @section LICENSE
  * GPLv3+ (see the COPYING file or http://www.gnu.org/licenses/)
  */

#pragma once

#if (!defined(__FAST_MATH__) || !defined(NDEBUG)) && !defined(BZ_DEBUG)
#  warning neither __FAST_MATH__ && NDEBUG nor BZ_DEBUG defined
#  warning   -ffast-math (Clang and GCC) and -DNDEBUG are recomended for release-mode builds
#  warning   -DBZ_DEBUG is recommended for debug-mode builds
#endif

#if defined(BZ_THREADSAFE)
#  error libmpdata++ uses blitz::neverDeleteData, please unset BZ_THREADSAFE
#endif

// force use of #pragma ivdep even if Blitz thinks the compiler does not support it
// (as of gcc 20140212, it gives an ICE: http://gcc.gnu.org/bugzilla/show_bug.cgi?id=60198) - TODO: check in CMake
//#define BZ_USE_ALIGNMENT_PRAGMAS  

#include <blitz/tv2fastiter.h> // otherwise Clang fails in debug mode
#include <blitz/array.h>

  
#include <libmpdata++/kahan_reduction.hpp>

//////////////////////////////////////////////////////////
#include <boost/ptr_container/ptr_vector.hpp>

// C++11 auto return type macro
#define return_macro(init,expr)          \
  -> decltype(blitz::safeToReturn(expr)) \
{                                        \
  init                                   \
  return safeToReturn(expr);             \
} 

namespace libmpdataxx
{
  template <int n_dims> using idx_t = blitz::RectDomain<n_dims>;
  using rng_t = blitz::Range;

  // Boost ptr_vector 
  template <class arr_t>
  struct arrvec_t : boost::ptr_vector<arr_t> 
  {
    using parent_t = boost::ptr_vector<arr_t>;

    const arr_t &operator[](const int i) const 
    {   
      return this->at(
	(i + this->size()) % this->size()
      );  
    }

    void push_back(arr_t *arr)
    {
      parent_t::push_back(arr);

#if !defined(NDEBUG)
      // filling the array with NaNs to ease debugging
      *arr = blitz::has_signalling_NaN(*arr->dataFirst())
	? blitz::signalling_NaN(*arr->dataFirst())
	: blitz::quiet_NaN(*arr->dataFirst());
#endif
    }
  };
}; // namespace libmpdataxx
