
program gen_be_nc2gsi

!------------------------------------------------------------------------
!  Purpose: converter, read be.nc and write binary for GSI applications 
!           gen_be (new features)
!
!  Auothor: Gael Descombes (MMM/NESL/NCAR)   Date: 30/12/2013
!
!  Note: Please acknowledge author/institute in work that uses this code.
!------------------------------------------------------------------------

   implicit none

   character (len=1024) :: filename
   character (len=32) :: varname
   character (len=5) :: variable
   character (len=3) :: cne, cinterval
   integer, parameter :: StrLen = 32
   integer :: StrLen0
   real, allocatable ,dimension(:,:) :: mapfac_m
   real, allocatable ,dimension(:) :: field1d
   real :: field0d

   integer :: nferr, VarID, rd_ncid
   integer, dimension(1) :: start1, count1
   integer, dimension(2) :: start2, count2

   integer :: rdVarIDvarnd
   integer :: ii, vv, kk

   integer :: bin_type,  num_bins, num_bins2d, nvar, nlon, nlat, nlev
   real :: binwidth_lat

   ! read the namelist parameter
   integer, allocatable, dimension(:,:) :: covarID
   integer, allocatable, dimension(:) :: vardimlist
   character (len=StrLen), allocatable, dimension(:) :: cvlist, cvlistu

   !==========================================================================================

   real, allocatable, dimension(:,:,:) :: corz3d, hwll3d, vztdq3d, corqq3d
   real, allocatable, dimension(:,:) :: cor2d, hwll2d, corz3d2
   integer :: qoption
   integer :: vdim, vind, nrf, nc3d, vind3d, vind2d
   character (len=10) :: cvar

   integer :: nsig, mlat
   real, allocatable, dimension(:,:)    :: lat2d 
   real, allocatable, dimension(:,:,:)    :: agvi
   real, allocatable, dimension(:,:)      :: wgvi
   real, allocatable, dimension(:,:)      :: bvi
   real, allocatable, dimension(:,:)      :: varq2
   real, allocatable, dimension(:)        :: znu 
   integer :: mlat_min, mlat_max
   real*4  ::  lat_max0, lat_min0 
   logical :: nf_err0
   integer :: bin

   !--------------------------------------------------------------------
   ! real simple
   integer :: nz
   real*4, allocatable  :: sigl_s(:),clat_s(:)
   real*4, allocatable  :: agv_s(:,:,:),wgv_s(:,:),bv_s(:,:)
   real*4, allocatable, dimension(:,:,:) :: corz3d_s, hwll3d_s, vztdq3d_s, corqq3d_s
   real*4, allocatable, dimension(:,:) :: cor2d_s, hwll2d_s,corz3d2_s

   !---------------------------------------------------------------------

   integer :: vardim, numarg
#ifdef IARGC
   integer, intrinsic    :: iargc
#else
   integer, external     :: iargc
#endif
   integer :: ne, interval
   character*10 :: start_date, end_date

!--------------------------------------------------------------------

include 'netcdf.inc'


!----------------------------------------------------------------------------------------------------
! Read the variable in argument
!----------------------------------------------------------------------------------------------------
   numarg = iargc()
   if ( numarg /= 4 ) then
      write(UNIT=6,FMT='(a)') "the converter without any argument will select qoption=1"
      write(UNIT=6,FMT='(a)') "qoption=2 needs 4 arguments start_date end_date interval ne"
      qoption=1
   else
      ! Initialise to stop false Cray compiler warnings
      start_date=""
      end_date=""
      cinterval=""
      cne=""
      qoption=2
      call getarg( 1, start_date )
      call getarg( 2, end_date )
      call getarg( 3, cinterval )
      call getarg( 4, cne )
      read(cne,'(i3)')ne
      read(cinterval,'(i3)')interval
   end if




!---------- GSI -----------------------------------------------------
nc3d = 1
nvar = 3
nrf  = 1
mlat = 4
nsig = 27
cvar = 'sf'
vdim = 3
vind = 1
mlat_min = 0
mlat_max = mlat + 1


!--------------------------------------------------------
! read netcdf file and allocate
!--------------------------------------------------------

filename = 'be.nc'

call io_input_init(filename, rd_ncid)

call read_dimensions(rd_ncid, nlev, nlat, nlon, nvar, num_bins, num_bins2d, StrLen0)

allocate(vardimlist(nvar))
varname='vardim_list'
call read_field1dint(rd_ncid, nvar, varname, vardimlist)
nc3d = 0
do ii=1, nvar
   if (vardimlist(ii).eq.3) then
      nc3d = nc3d + 1
   end if
end do


! binning
!-----------------------------------------------
varname='bin_type'
call read_field0dint(rd_ncid, varname, bin_type)
if ( bin_type.eq.5 ) then
  write(6,*)'binning option 5, binning by uniform vertical level'
else if ( bin_type.eq.3 ) then
  write(6,*)'binning option 3, binning by uniform vertical level and latitude'
else
  write(6,*)'binning option not supported into GSI'  
  stop 
end if

varname='binwidth_lat'
call read_field0d(rd_ncid, varname, binwidth_lat)

allocate(lat2d(nlon,nlat))
varname='lat'
call read_field2d(rd_ncid, nlon, nlat, varname, lat2d)

lat_max0 = maxval(lat2d)
lat_min0 = minval(lat2d)
!binwidth_lat = 4. !!GD HARD CODED to test
write(*,*)'lat_max0,lat_min0 ',lat_max0,lat_min0
mlat = floor (abs((lat_max0 - lat_min0) / binwidth_lat) ) + 1
write(6,*)'mlat   ',mlat
allocate(clat_s(mlat))
clat_s(1) = lat_min0 + binwidth_lat/2.
do ii=2, mlat
    clat_s(ii) = clat_s(1) + binwidth_lat*(ii-1)
end do
write(6,*)'clat_s ',clat_s
mlat_min = 0
mlat_max = mlat + 1


nsig = nlev
allocate(corz3d(1:mlat,nsig,nc3d))
allocate(corz3d2(1:mlat,nsig))
allocate(corqq3d(1:mlat,nsig,nc3d))
allocate(hwll3d(mlat_min:mlat_max,nsig,nc3d))
allocate(vztdq3d(nsig,mlat_min:mlat_max,nc3d))
allocate(cor2d(1:mlat,nvar-nc3d))
!allocate(hwll2d(mlat_min:mlat_max,nvar-nc3d))
allocate(hwll2d(mlat_min:mlat_max,1))

allocate(agvi(mlat_min:mlat_max,nsig,nsig))
allocate(wgvi(mlat_min:mlat_max,nsig))
allocate(bvi(mlat_min:mlat_max,nsig))
allocate(znu(nsig))

!! simple array
nz=nlev
allocate(sigl_s(nz))
allocate(agv_s(0:mlat+1,nz,nz),wgv_s(0:mlat+1,nz))
allocate(bv_s(0:mlat+1,nz))

allocate(corz3d_s(1:mlat,nsig,nc3d))
allocate(corz3d2_s(1:mlat,nsig))
allocate(corqq3d_s(1:mlat,nsig,nc3d))
allocate(hwll3d_s(mlat_min:mlat_max,nsig,nc3d))
allocate(vztdq3d_s(nsig,mlat_min:mlat_max,nc3d))
allocate(cor2d_s(1:mlat,nvar-nc3d))
!allocate(hwll2d_s(mlat_min:mlat_max,nvar-nc3d))
allocate(hwll2d_s(mlat_min:mlat_max,1))
hwll2d = 0.0
allocate(varq2(1:mlat,nlev))
varq2 = 0.0


!----------------------------------------------
! read namelist parameter
!----------------------------------------------
allocate(cvlist(nvar))
allocate(cvlistu(nvar))

varname='cv_list'
call read_field1dstr(rd_ncid, nvar, StrLen0, varname, cvlist)
varname='cv_listu'
call read_field1dstr(rd_ncid, nvar, StrLen0, varname, cvlistu)
varname='vardim_list'
call read_field1dint(rd_ncid, nvar, varname, vardimlist)
   
!----------------------------------------------
! read regcoeff
!----------------------------------------------
call  read_regcoeff_gsi(rd_ncid, nsig, num_bins2d, mlat, mlat_min, mlat_max, agvi, bvi, wgvi, znu)


!-----------------------------------------------
! if qoption = 2 
!-----------------------------------------------

if ( qoption.eq.2) then
   call specific_qoption2(varq2, ne, start_date, end_date, interval, nlon, nlat,nlev, num_bins2d)
end if


!----------------------------------------------
! Open binary and write in simple 
!----------------------------------------------

nz = nlev
sigl_s = znu
!rlat4 = 

agv_s = agvi
wgv_s = wgvi
bv_s  = bvi

do bin = 1, 2

if ( bin.eq.1) then
   open(333,file='wrf-arw-gsi_be_big_endian.gcv',convert='big_endian',form='unformatted')
else
   open(333,file='wrf-arw-gsi_be_little_endian.gcv',convert='little_endian',form='unformatted')
end if
rewind 333
write(333)nz,mlat
write(333)clat_s,sigl_s
write(333)agv_s,bv_s,wgv_s



!---------------------------------------------------------------
! write variable
!---------------------------------------------------------------
nf_err0 = .false.
vind3d = 0
vind2d = 0


do vv=1, nvar

      vdim = vardimlist(vv)
      if (vdim .eq. 3 ) then
          vind3d = vind3d + 1
          vind = vind3d
          kk = nsig
      else
          vind2d = vind2d + 1
          vind = vind2d
          kk = 1
      end if

      
      nf_err0 = .false.
      write(*,*)'***********************************************************'
      write(*,*)' ',trim(cvlistu(vv))
      write(*,*)'***********************************************************'
      call read_be_gsi(rd_ncid, cvlistu(vv), variable, vdim, vind, nf_err0, hwll2d, cor2d, &
                hwll3d, vztdq3d, corz3d, corz3d2, varq2, znu, qoption, num_bins2d, mlat, mlat_min, mlat_max, nsig, nvar, nc3d)

      hwll2d_s  = hwll2d
      hwll3d_s  = hwll3d
      corz3d_s  = corz3d
      corz3d2_s = corz3d2
      vztdq3d_s = vztdq3d
      cor2d_s   = cor2d

      !write(*,*)'hwll2d_s ',hwll2d_s
      !write(*,*)'hwll2d ',hwll2d
      if ( nf_err0 ) then 
         
         write(*,*)'========================================'
         write(6,*)'write variable ,nsig ',trim(variable),kk
         write(*,*)'========================================'
         write(333)variable,kk

         if ( trim(variable) .eq. 'q' ) then
            if ( qoption .eq.2 ) then
               write(333)corz3d_s(:,:,vind), corz3d2_s(:,:)
               write(6,*)'------------------------------------------------------'
               write(*,*)'stdq',maxval(corz3d_s(:,:,vind)),minval(corz3d2_s(:,:))
               write(6,*)'------------------------------------------------------'
            else
               write(333)corz3d_s(:,:,vind), corz3d_s(:,:,vind)
               write(6,*)'------------------------------------------------------'
               write(*,*)'stdq',maxval(corz3d_s(:,:,vind)),minval(corz3d_s(:,:,vind))
               write(6,*)'------------------------------------------------------'
            end if

            write(6,*)'------------------------------------------------------'
            write(*,*)'stdq',maxval(corz3d_s(:,:,vind)),minval(corz3d_s(:,:,vind))
            write(6,*)'------------------------------------------------------'
         else if ( trim(variable).eq.'ps' ) then
            write(333)cor2d_s(:,vind)
            write(6,*)'------------------------------------------------------'
            write(*,*)'std ps',cor2d_s(:,vind)
            write(6,*)'------------------------------------------------------'
         else
            write(6,*)'------------------------------------------------------'
            write(*,*)'std',maxval(corz3d_s(:,:,vind)),minval(corz3d_s(:,:,vind))
            write(6,*)'------------------------------------------------------'
            write(333)corz3d_s(:,:,vind)
         endif

         write(*,*)'vind ',vind
         if ( vardimlist(vv) .eq. 2 ) then
            write(6,*)'------------------------------------------------------'
            write(*,*)'write 2d  horizontal length scale ',hwll2d_s
            write(6,*)'------------------------------------------------------'
            write(333) hwll2d_s  ! From the hlt array
            !write(333) hwll2d_s(:,1)  ! From the hlt array
         else
            write(6,*)'------------------------------------------------------'
            write(6,*)'horizontal length scale :',maxval(hwll3d_s(:,:,vind)),minval(hwll3d_s(:,:,vind))
            write(6,*)'------------------------------------------------------'
            write(333) hwll3d_s(:,:,vind)  ! From the hlt array 
            write(6,*)'------------------------------------------------------'
            write(6,*)'vertical length scale :',maxval(vztdq3d_s(:,:,vind)),minval(vztdq3d_s(:,:,vind))
            write(6,*)'------------------------------------------------------'
            write(333) vztdq3d_s(:,:,vind)
         endif
      write(*,*)'after writting' 
      end if

end do

close(333)

write(*,*)'after to close bin ',bin

end do

stop
!----------------------------------------------------------------


contains

!================================================================================
! subroutines to read be.nc netcdf file, form gen_be (NCAR, gael DESCOMBES /MMM)
!================================================================================

subroutine read_be_gsi(rd_ncid, variable, cvar, vdim, vind, nf_err0, hwll2d, cor2d, &
                hwll3d, vztdq3d, corz3d, corz3d2, varq2 , znu, qoption, num_bins2d, mlat, mlat_min, mlat_max, nsig, nvars, nc3d)

   implicit none

   real, dimension(mlat_min:mlat_max,1:nsig,1:nc3d), intent(inout) ::  hwll3d
   real, dimension(1:mlat,1:nsig,1:nc3d), intent(inout) :: corz3d
   real, dimension(1:mlat,1:nsig), intent(inout) ::  corz3d2
   real, dimension(1:nsig,mlat_min:mlat_max,1:nc3d), intent(inout) :: vztdq3d
   real, dimension(mlat_min:mlat_max, nvars-nc3d), intent(inout) :: hwll2d
   real, dimension(1:mlat, nvars-nc3d), intent(inout) :: cor2d
   real, dimension(1:nsig), intent(in)                :: znu
   real, dimension(1:mlat,1:nsig), intent(in)         :: varq2
   integer, intent(inout) :: qoption
   integer, intent(in) :: mlat, nsig, nvars, nc3d, mlat_min, mlat_max,num_bins2d
   integer, intent(in) :: vdim, vind, rd_ncid
   logical, intent(inout) :: nf_err0
   character (len=*), intent(inout) :: cvar
   character (len=*), intent(in) :: variable 

   character (len=64) :: filename, varname, variable2
 
   real, allocatable, dimension(:,:) :: vertvarce2d, vertvarce2d_2
   real, allocatable, dimension(:,:) :: horlen2d, verlen2d
   real, allocatable, dimension(:)   :: field1d
   real :: field0d
   integer :: i, j, k, b, dim3, nferr, i0, rdVarIDvarnd  
   integer :: nlev, nlat, nlon, nvar, ncovar, num_bins
   integer, dimension(1) :: start1, count1
   real, allocatable    :: rsig(:), dsig(:)
   real(4) :: vlen 

   include 'netcdf.inc'


!    write(*,*)'cvar, vdim, vind, mlat, nsig, ',cvar, vdim, vind, mlat, nsig, nrf 

   !-------------------------------------
   ! before reading variable into be.nc
   !-------------------------------------

   nf_err0 = .true.
   select case( trim(adjustl(variable)) )
      case ('psi')
         cvar = 'sf'
      case ('chi_u')
         cvar = 'vp'
      case ('t_u')
         cvar = 't'
      case ('rh')
         cvar = 'q'
         variable2 = 'rhm'
      case ('ps_u')
         cvar = 'ps'
   case default;
        write(*,*)' The selected variable is not in the pre-defined list in read_be_gsi '
        nf_err0 = .false.
   end select

 
  if ( nf_err0  ) then

  ! allocation 
   if ( vdim .eq. 3 ) then
      dim3 = nsig
   else
      dim3 = 1
   end if
   !!num_bins2d = 1 !! = mlat

   allocate(horlen2d(dim3,num_bins2d))
   allocate(verlen2d(dim3,num_bins2d))
   allocate(vertvarce2d(dim3,num_bins2d))
   allocate(vertvarce2d_2(dim3,num_bins2d))
   horlen2d = 0.0
   verlen2d = 0.0
   vertvarce2d   = 0.0
   vertvarce2d_2 = 0.0
   vztdq3d       = 0.0

   !------------------------------------
   ! read the variables
   !------------------------------------
   if ( vdim .eq. 3 ) then

      ! read horizontal length scale
      varname = 'lenscale_'//trim(variable)
      call read_field2d(rd_ncid, nsig, num_bins2d,  trim(varname), horlen2d)
      ! read vertical length scale
      varname = 'vert_lenscale_'//trim(variable)
      call read_field2d(rd_ncid, nsig, num_bins2d, trim(varname), verlen2d)
      ! read verticale variance
      varname = 'vert_varce_'//trim(variable)
      call read_field2d(rd_ncid, nsig, num_bins2d, trim(varname), vertvarce2d)
      vertvarce2d = sqrt(vertvarce2d)
      if (cvar == 'q') then
         !varname = 'vert_varce_'//trim(variable2)
         !nferr   = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
         if ( qoption .eq.2 ) then
            vertvarce2d_2 = sqrt(varq2)
         end if
      end if   

   else  !  vdim = 2
      
      ! read horizontal length scale
      varname = 'lenscale_'//trim(variable)
      !call read_field0d(rd_ncid, varname, field0d)
      !horlen2d(1,1) = field0d
      !call read_field0d(rd_ncid, varname, field1d)
      allocate(field1d(num_bins2d))
      field1d = 0.0 
      call read_field1d(rd_ncid, num_bins2d, trim(varname), field1d)
      horlen2d(1,:) = field1d
      ! read verticale variance
      varname = 'vert_varce_'//trim(variable)
      field1d = 0.0 
      call read_field1d(rd_ncid, num_bins2d, trim(varname), field1d)
      write(*,*)'field1d ',field1d(:) 
      vertvarce2d(1,:) = sqrt(field1d(:))
      write(*,*)'vertvarce2d PS ',vertvarce2d(1,:)
      deallocate(field1d)
      write(*,*)'after read 2d'
   end if
  

   !--------------------------------------------------------
   ! transform vertical length scale
   !--------------------------------------------------------
   !write(*,*)'verlen2d grid_point 1 ',verlen2d

   if ( vdim .eq. 3 ) then
      write(*,*)' compute  transform vertical length scale'

      allocate(rsig(1:nsig))
      allocate(dsig(1:nsig))

      do b=1, num_bins2d
      ! Definition of vertical scale in log(sigma) unit
      do k=1,nsig
         rsig(k)=log(znu(k))
       end do

       dsig(1)=rsig(1)-rsig(2)
       do k=2,nsig-1
          dsig(k) = 0.5*(rsig(k-1)-rsig(k+1))
       end do
       dsig(nsig)=rsig(nsig-1)-rsig(nsig)
       !write(*,*)' vertvarce2d 1 ',vertvarce2d 

       ! compute vertical length scale
       do k=1, nsig
          if (verlen2d(k,b).ne.0) then
             verlen2d(k,b) = 1./verlen2d(k,b)/dsig(k)
          end if
       end do

       end do

       deallocate(rsig)
       deallocate(dsig)

    end if

  ! write(*,*)'verlen2d grid_point 2',verlen2d

   !--------------------------------------------------------
   ! transform dimensions for GSI
   !--------------------------------------------------------

   if ( vdim .eq. 3 ) then
        write(*,*)'transform dimensions for GSI'
            do k = 1, nsig
               do i=1, mlat
                  if ( num_bins2d.eq.1  ) then
                      i0 = 1
                  else
                      i0 = i
                  end if
                  
                  if (cvar=='q' .and. qoption==2) then
                     corz3d2(i,k) = varq2(k,i0)
                  end if
                  corz3d(i,k,vind) = vertvarce2d(k,i0)
               end do
               do i=1, mlat
                  if ( num_bins2d.eq.1  ) then
                      i0 = 1
                  else
                      i0 = i
                  end if
                  vztdq3d(k,i,vind) =  verlen2d(k,i0)
                  hwll3d(i,k,vind)  =  horlen2d(k,i0)
           !       write(*,*)'vztdq3d(k,i,vind)',k,i,vztdq3d(k,i,vind)
               end do
            end do
            hwll3d(mlat_min,:,vind)  =  horlen2d(:,1)
            hwll3d(mlat_max,:,vind)  =  horlen2d(:,mlat)
            vztdq3d(:,mlat_min,vind) = verlen2d(:,1)
            vztdq3d(:,mlat_max,vind) = verlen2d(:,mlat)
!write(*,*)'vztdq3d ',vztdq3d(:,0:mlat+1,vind)
!write(*,*)'------------------------------------'
!write(*,*)'vztdq3d2',vztdq3d(:,:,vind)
   else  ! vdim = 2
            do i=1, mlat
               if ( num_bins2d.eq.1  ) then
                   i0 = 1
               else
                   i0 = i
               end if
               cor2d(i,vind) = vertvarce2d(1,i0)
            end do
            do i=mlat_min, mlat_max
               if ( num_bins2d.eq.1  ) then
                   i0 = 1
               else
                   i0 = i
               end if
               hwll2d(i,vind) = horlen2d(1,i0)
            end do
            hwll2d(mlat_min,vind) = horlen2d(1,1)
            !write(*,*)'GD hwll2d(mlat_max,vind) ',horlen2d(1,mlat)
            hwll2d(mlat_max,vind) = horlen2d(1,mlat)
   end if

!      write(*,*)'--------------------------------------------------'  
!      write(*,*)' variable  ',cvar
!      write(*,*)'--------------------------------------------------'  
!
!      if ( vdim .eq. 3 ) then
!         write(*,*)'vertvarce2d corz ',vertvarce2d
!         write(*,*)'horlen2d ', horlen2d
!         write(*,*)'verlen2d ', verlen2d
!      else
!         write(*,*)'vertvarce2d ',vertvarce2d(1,1)
!         write(*,*)'horlen2d  ', horlen2d(1,1)
!      end if


   deallocate(horlen2d)
   deallocate(verlen2d)
   deallocate(vertvarce2d)
   deallocate(vertvarce2d_2)

   end if ! nf_err0

   !---------------------------------------------------------
   ! Close NetCdF file 
   !--------------------------------------------------------
   ! nferr = nf_close(rd_ncid)

end subroutine read_be_gsi

subroutine read_regcoeff_gsi(rd_ncid, nsig, num_bins2d, mlat, mlat_min, mlat_max, agvi, bvi, wgvi, sigma)

   implicit none

   integer ,intent(in) :: nsig, mlat, mlat_min, mlat_max,num_bins2d
   real, dimension(mlat_min:mlat_max,1:nsig,1:nsig),intent(inout) :: agvi
   real*8, dimension(mlat_min:mlat_max,1:nsig,1:nsig) :: agvi8
   real, dimension(mlat_min:mlat_max,1:nsig)     ,intent(inout) :: wgvi
   real, dimension(mlat_min:mlat_max,1:nsig)     ,intent(inout) :: bvi
   real, dimension(1:nsig)     ,intent(inout) :: sigma
   integer, intent(in) :: rd_ncid

   real, allocatable, dimension(:,:,:) :: field3d
   real, allocatable, dimension(:,:) :: field2d
   real, allocatable, dimension(:) :: field1d
   integer :: ii, jj, kk, k2, nferr
   integer :: num_bins
   character (len=1024) :: filename, varname

   include 'netcdf.inc'

   write(*,*)'mlat, nsig : ',mlat, nsig

   num_bins = nsig*num_bins2d
   !num_bins2d = 1   
  
   ! Read regcoeff bvi
   allocate(field1d(num_bins)) 
   varname = 'regcoeff_psi_chi'
   call read_field1d(rd_ncid, num_bins, trim(varname), field1d)
   do ii = 1, num_bins2d
   do kk = 1, nsig
             jj = ii + (kk-1)*num_bins2d
             bvi(ii,kk) = field1d(jj)
             write(*,*)'bvi,1d',ii,kk,jj
             write(*,*)'',bvi(ii,kk),field1d(jj)
      end do
   end do
!write(*,*)'field1d ',field1d
!stop

   bvi(mlat_min,:) = bvi(1,:)
   bvi(mlat_max,:) = bvi(num_bins2d,:)
   write(6,*)'',trim(varname)
   write(6,*)' bvi ',bvi
   deallocate(field1d)

   ! Read regcoeff wgvi
   allocate(field2d(nsig,num_bins2d))
   varname = 'regcoeff_psi_ps'
   wgvi = 0.0
   call read_field2d(rd_ncid, nsig, num_bins2d, trim(varname), field2d)
   do ii = 1, num_bins2d
      do kk = 1, nsig
         if ( num_bins2d .eq. 1 ) then
            wgvi(ii,kk) = field2d(kk,1)
         else
            wgvi(ii,kk) = field2d(kk,ii)
         end if
         write(*,*)'ii, kk ',ii,kk
      write(*,*)'wgvi(ii,kk) ',wgvi(ii,kk),field2d(kk,ii)
      end do
   end do
   wgvi(mlat_min,:) = wgvi(1,:)
   wgvi(mlat_max,:) = wgvi(num_bins2d,:)
   write(6,*)'',trim(varname)
   write(6,*)'wgvi ',wgvi
   deallocate(field2d)

   ! Read regcoeff agvi
   allocate(field3d(nsig,nsig,num_bins2d))
   varname = 'regcoeff_psi_t'
   field3d = 0.0
   agvi = 0.0
   call read_field3d(rd_ncid, nsig, nsig, num_bins2d, trim(varname), field3d)
   do ii = 1, num_bins2d 
      do kk = 1, nsig
         do k2 = 1, nsig
            if ( num_bins2d .eq. 1 ) then
               agvi(ii,kk,k2) = field3d(kk,k2,1)
            else
               agvi(ii,kk,k2) = field3d(kk,k2,ii)
               write(6,*)'ii,kk,k2 ',ii,kk,k2
               write(6,*)'agvi 3d ',agvi(ii,kk,k2),field3d(kk,k2,ii)
               !agvi(ii,kk,k2) = field3d(kk,k2,ii)
            end if
         end do
      end do
   end do
   agvi(mlat_min,:,:) = agvi(1,:,:)
   agvi(mlat_max,:,:) = agvi(num_bins2d,:,:)

   write(6,*)'',trim(varname)
   !write(6,*)'agvi ',field3d
   deallocate(field3d)
   ! Read sigma
   varname = 'znu'
   call read_field1d(rd_ncid, nsig, trim(varname), sigma)
   write(6,*)'',trim(varname)
   write(6,*)'sigma ',sigma

end subroutine read_regcoeff_gsi


subroutine io_input_init(filename, rd_ncid)

      implicit none

      character (len=*), intent(in) :: filename
      integer :: rd_ncid
      integer :: nferr

      include 'netcdf.inc'

      !nferr = nf_open(trim(filename), 0, rd_ncid)
      nferr = nf_open(trim(filename), ior(NF_SHARE,NF_64BIT_OFFSET), rd_ncid)
      if (nferr /= NF_NOERR) then
         write(0,*) ' '
         write(0,*) 'Error opening input file ''', trim(filename), ''''
         write(0,*) ' '
         stop
      end if
      write(*,*)'open ',trim(filename)

end subroutine io_input_init


subroutine read_dimensions(rd_ncid, nlev, nlat, nlon, nvar, num_bins, num_bins2d, StrLen)

   implicit none
   integer, intent(in)    :: rd_ncid
   integer, intent(out)   :: nlev, nlat, nlon, nvar, num_bins, num_bins2d, StrLen
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   nferr = nf_inq_dimid(rd_ncid, 'num_bins2d', rdVarIDvarnd)
   nferr = nf_inq_dimlen(rd_ncid, rdVarIDvarnd, num_bins2d)
   nferr = nf_inq_dimid(rd_ncid, 'num_bins', rdVarIDvarnd)
   nferr = nf_inq_dimlen(rd_ncid, rdVarIDvarnd, num_bins)
   nferr = nf_inq_dimid(rd_ncid, 'lev', rdVarIDvarnd)
   nferr = nf_inq_dimlen(rd_ncid, rdVarIDvarnd, nlev)
   nferr = nf_inq_dimid(rd_ncid, 'lat', rdVarIDvarnd)
   nferr = nf_inq_dimlen(rd_ncid, rdVarIDvarnd, nlat)
   nferr = nf_inq_dimid(rd_ncid, 'lon', rdVarIDvarnd)
   nferr = nf_inq_dimlen(rd_ncid, rdVarIDvarnd, nlon)
   nferr = nf_inq_dimid(rd_ncid, 'nvar', rdVarIDvarnd)
   nferr = nf_inq_dimlen(rd_ncid, rdVarIDvarnd, nvar)
   nferr = nf_inq_dimid(rd_ncid, 'StrLen', rdVarIDvarnd)
   nferr = nf_inq_dimlen(rd_ncid, rdVarIDvarnd, StrLen)

   write(*,*)'read dimensions of binning, num_bins, num_bins2d : ',num_bins, num_bins2d
   write(*,*)'read dimensions of grid, lev, lat, lon : ',nlev, nlat, nlon
   write(*,*)'read dimensions of the number of control variables nvar : ',nvar

end subroutine read_dimensions

!! read character

subroutine read_field1dstr(rd_ncid, dim1, StrLen, varname, field1d)
  
   implicit none
   integer, intent(in) :: rd_ncid, StrLen, dim1
   character (len=*), intent(in) :: varname
   character(len=*), dimension(1:dim1), intent(inout) :: field1d
   integer, dimension(2) :: start2, count2
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   nferr = nf_inq_varid(rd_ncid, varname, rdVarIDvarnd)
   start2(2) = 1
   count2(2) = dim1
   start2(1) = 1
   count2(1) = StrLen 
   nferr = nf_get_vara_text(rd_ncid, rdVarIDvarnd, start2, count2, field1d )
   call check_err(nferr,varname)

end subroutine  read_field1dstr

!! integer

subroutine read_field0dint(rd_ncid, varname, field0d)

   implicit none
   integer, intent(in) :: rd_ncid
   integer, intent(inout) :: field0d
   character (len=*), intent(in) :: varname
   integer, dimension(1) :: start1, count1
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   start1(1) = 1
   count1(1) = 1
   nferr = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
   nferr = nf_get_vara_int( rd_ncid, rdVarIDvarnd, start1, count1, field0d)
   call check_err(nferr,varname)

end subroutine read_field0dint


subroutine read_field1dint(rd_ncid, dim1, varname, field1d)

   implicit none
   integer, intent(in) :: dim1, rd_ncid
   integer, dimension(1:dim1), intent(inout) :: field1d
   character (len=*), intent(in) :: varname
   integer, dimension(1) :: start1, count1
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   start1(1) = 1
   count1(1) = dim1
   nferr = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
   nferr = nf_get_vara_int( rd_ncid, rdVarIDvarnd, start1, count1, field1d)
   call check_err(nferr,varname)

end subroutine read_field1dint

subroutine read_field2dint(rd_ncid, dim1, dim2, varname, field2d)

   implicit none
   integer, intent(in) :: dim1, dim2, rd_ncid
   integer, dimension(1:dim1,1:dim2), intent(inout) :: field2d
   character (len=*), intent(in) :: varname
   integer, dimension(2) :: start2, count2
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   start2(1) = 1
   count2(1) = dim1
   start2(2) = 1
   count2(2) = dim2
   nferr = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
   nferr = nf_get_vara_int( rd_ncid, rdVarIDvarnd, start2, count2, field2d)
   call check_err(nferr,varname)

end subroutine read_field2dint


!! Real

subroutine read_field0d(rd_ncid, varname, field0d)

   implicit none
   integer, intent(in) :: rd_ncid
   real, intent(inout) :: field0d
   real(kind=8) :: field0d8
   character (len=*), intent(in) :: varname
   integer, dimension(1) :: start1, count1
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   start1(1) = 1
   count1(1) = 1
   nferr = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
   nferr = nf_get_vara_double( rd_ncid, rdVarIDvarnd, start1, count1, field0d8)
   field0d = field0d8
   call check_err(nferr,varname)

end subroutine read_field0d

subroutine read_field1d(rd_ncid, dim1, varname, field1d)

   implicit none
   integer, intent(in) :: dim1, rd_ncid
   real, dimension(1:dim1), intent(inout) :: field1d
   real*8, dimension(1:dim1) :: field1d8
   character (len=*), intent(in) :: varname
   integer, dimension(1) :: start1, count1
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   start1(1) = 1
   count1(1) = dim1
   nferr = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
   nferr = nf_get_vara_double( rd_ncid, rdVarIDvarnd, start1, count1, field1d8)
   field1d = field1d8
   call check_err(nferr,varname)

end subroutine read_field1d


subroutine read_field2d(rd_ncid, dim1, dim2, varname, field2d)

   implicit none 
   integer, intent(in) :: dim1, dim2, rd_ncid
   real, dimension(1:dim1,1:dim2), intent(inout) :: field2d
   real*8, dimension(1:dim1,1:dim2) :: field2d8
   character (len=*), intent(in) :: varname
   integer, dimension(2) :: start2, count2
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'
  
   start2(1) = 1
   count2(1) = dim1
   start2(2) = 1
   count2(2) = dim2
   nferr = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
   nferr = nf_get_vara_double( rd_ncid, rdVarIDvarnd, start2, count2, field2d8)
   field2d = field2d8
   call check_err(nferr,varname)

end subroutine read_field2d


subroutine read_field3d(rd_ncid, dim1, dim2, dim3, varname, field3d)

   implicit none
   integer, intent(in) :: dim1, dim2, dim3, rd_ncid
   real, dimension(1:dim1,1:dim2,1:dim3), intent(inout) :: field3d
   real*8, dimension(1:dim1,1:dim2,1:dim3) :: field3d8
   character (len=*), intent(in) :: varname
   integer, dimension(3) :: start3, count3
   integer :: nferr, rdVarIDvarnd

   include 'netcdf.inc'

   start3(1) = 1
   count3(1) = dim1
   start3(2) = 1
   count3(2) = dim2
   start3(3) = 1
   count3(3) = dim3
   nferr = nf_inq_varid(rd_ncid, trim(varname), rdVarIDvarnd)
   nferr = nf_get_vara_double( rd_ncid, rdVarIDvarnd, start3, count3, field3d8)
   field3d = field3d8
   call check_err(nferr,varname)

end subroutine read_field3d

subroutine check_err(nferr,varname)

   implicit none
   integer, intent(in) :: nferr
   character(len=*), optional, intent(in) :: varname
   include "netcdf.inc"

   if (nferr /= NF_NOERR) then
         write(0,*) ' '
         write(0,*) 'Error nferr ', nferr
         write(0,*) 'nf_strerror ', nf_strerror(nferr)
         if (present(varname)) then
            write(0,*) ' problem ',trim(varname)
         end if
         write(0,*) ' '
         stop
   else
        if (present(varname)) then
            write(0,*) ' read ',trim(varname)
         end if
   end if

end subroutine check_err

!=============================================================

subroutine da_advance_cymdh (start_date, dh, end_date)

   !-----------------------------------------------------------------------
   ! Purpose: TBD
   !-----------------------------------------------------------------------

   implicit none

   character (len=10), intent(in)  :: start_date ! In date (ccyymmddhh).
   integer, intent(in)             :: dh         ! Period to advance (-ve forpast).
   character (len=10), intent(out) :: end_date   ! Out date (ccyymmddhh).

   integer :: ccyy, mm, dd, hh

   read(start_date(1:10), fmt='(i4, 3i2)')  ccyy, mm, dd, hh

   hh = hh + dh

   do while (hh < 0)
      hh = hh + 24
      call da_change_date  (ccyy, mm, dd, -1)
   end do

   do while (hh > 23)
      hh = hh - 24
      call da_change_date  (ccyy, mm, dd, 1)
   end do

   write(unit=end_date(1:10), fmt='(i4.4, 3i2.2)')  ccyy, mm, dd, hh

end subroutine da_advance_cymdh

subroutine da_change_date(ccyy, mm, dd, delta)

   !-----------------------------------------------------------------------
   ! Purpose: TBD
   !-----------------------------------------------------------------------

   implicit none

   integer, intent(inout) :: ccyy, mm, dd
   integer, intent(in)    :: delta

   integer, dimension(12) :: mmday

   mmday = (/31,28,31,30,31,30,31,31,30,31,30,31/)

   mmday(2) = 28

   if (mod(ccyy,4) == 0) then
      mmday(2) = 29

      if (mod(ccyy,100) == 0) then
         mmday(2) = 28
      end if

      if (mod(ccyy,400) == 0) then
         mmday(2) = 29
      end if
   end if

   dd = dd + delta

   if (dd == 0) then
      mm = mm - 1

      if (mm == 0) then
         mm = 12
         ccyy = ccyy - 1
      end if

      dd = mmday(mm)
   elseif (dd .gt. mmday(mm)) then
      dd = 1
      mm = mm + 1
      if (mm > 12) then
         mm = 1
         ccyy = ccyy + 1
      end if
   end if
end subroutine da_change_date


!=============================================================

subroutine specific_qoption2(varq2, ne, start_date, end_date, interval, dim1, dim2, dim3, bins2d0)

  implicit none

  integer, intent(in)                     :: dim1, dim2, dim3, bins2d0
  real, intent(inout), dimension(1:bins2d0,1:dim3)   :: varq2
  integer, intent(in)                     :: ne, interval
  character*10, intent(in)                :: start_date, end_date

  integer                                 :: n, ii, jj, kk, counter, member, ee
  character(len=1024)                     :: filename, filename2
  character*3                             :: ce
  logical                                 :: file_here, file_here2
  real, dimension(1:bins2d0,1:dim3)     :: varce8
  real, dimension(1:bins2d0,1:dim3)     :: qcount
  real, dimension(1:dim1,1:dim2,1:dim3) :: field_rh, field_bin 
  character*10                            :: new_date, date, varname
  integer                                 :: cdate, edate        ! Starting, current ending dates.   
  integer                                 :: rd_ncid, rd_ncid2

  counter = 0
  qcount(:,:) = 0
  read(start_date(1:10), fmt='(i10)')cdate
  read(end_date(1:10), fmt='(i10)')edate

  if ( bins2d0 .lt. 20  ) then
     write(6,*)'!!!! WARNING, qoption2 form GSI should work with number of bins > 20  !!!!' 
  end if

  do while ( cdate <= edate )
      do member = 1, ne

        write(UNIT=ce,FMT='(i3.3)')ee 

        varname   = 'rh'
        filename  = trim(varname)//'/trim(varname).'//date(1:10)//'.e'//trim(ce)
        file_here = .false.
        inquire(file=trim(filename),exist=file_here)

        varname   = 'temp'
        filename2  = trim(varname)//'/trim(varname).'//date(1:10)//'.e'//trim(ce)
        file_here = .false.
        inquire(file=trim(filename2),exist=file_here2)

        if (  (file_here).and. (file_here2) ) then
        
           ! read data
           varname='rh'
           call io_input_init(filename, rd_ncid)
           call io_input_init(filename2, rd_ncid2)
           call read_field3d(rd_ncid, dim3, dim2, dim1, trim(varname),field_rh)
           call read_field3d(rd_ncid2, dim3, dim2, dim1, trim(varname),field_bin)
    
           ! compute variance
           do kk=1,dim3
              do jj=1,dim2
                 do ii=1,dim1
                    n = int(field_bin(ii,jj,kk)*20.+1 )
                    if ( n< bins2d0 .and. n> 0 ) then
                       varce8(n,kk)  = varce8(n,kk) + field_rh(ii,jj,kk)*field_rh(ii,jj,kk) 
                       qcount(n,kk)  = qcount(n,kk)+1.
                    end if
                 end do
              end do
           end do
           counter = counter + 1

        else if (.not.(file_here)) then
           write(*,*)'File is missing ',trim(filename)
        else if (.not.(file_here2)) then
           write(*,*)'File is missing ',trim(filename2)
        end if

     end do

     ! Calculate next date:
     call da_advance_cymdh( date, interval, new_date )
     date = new_date
     read(date(1:10), fmt='(i10)')cdate

   end do


   do n = 1, bins2d0
      do kk = 1, dim3
         if ( qcount(n,kk).gt.10. ) then
            varq2(n,kk) = varce8(n,kk) / qcount(n,kk)
         else 
            varq2(n,kk) = 0.
         end if
      end do
  end do


end subroutine specific_qoption2

!=============================================================

end program gen_be_nc2gsi 
