#include <meteoio/MeteoIO.h>

using namespace mio; //The MeteoIO namespace is called mio

const Date d1(2008, 12, 15, 12, 0, 1.);

void generate_dem(const size_t& n, DEMObject& dem)
{
	Coords llcorner("CH1903", "");
	llcorner.setLatLon(46., 9.8, 1000.);
	const double cellsize = 10.;

	dem.set(n, n, cellsize, llcorner);
	for(size_t jj=0; jj<n; jj++) {
		for(size_t ii=0; ii<n; ii++) {
			const double dist = Optim::fastSqrt_Q3( (ii*ii + jj*jj)*cellsize*cellsize);
			dem(ii,jj) = dist;
		}
	}
}

void write_dem(const Config& cfg, const size_t& n)
{
	IOManager io(cfg);
	DEMObject dem;

	generate_dem(n, dem);

	std::ostringstream ss;
	ss << n << "x" << n;
	io.write2DGrid(dem, ss.str());
}

double run_interpol(const Config& cfg, const DEMObject& dem)
{
	IOManager io(cfg);

	//get some data to make sure the buffer is filled
	std::vector<MeteoData> vecMeteo;
	io.getMeteoData(d1, vecMeteo);

	//now do the interpolation and time it
	Grid2DObject param;
	Timer timer;
	timer.start();
	io.getMeteoData(d1, dem, MeteoData::TA, param);
	timer.stop();

	return timer.getElapsed();
}

void time_algo(Config &cfg, const std::string& algo, const size_t& n, const DEMObject& dem)
{
	cfg.addKey("TA::algorithms", "Interpolations2D", algo);

	std::cout << n*n << " " << run_interpol(cfg, dem) << " " << algo << "\n";
}

void time_dem_size(Config &cfg, const size_t& n)
{
	std::ostringstream ss;
	ss << n << "x" << n;

	DEMObject dem;
	generate_dem(n, dem);
	std::cout << ss.str() << "\n";

	time_algo(cfg, "None", n, dem);
	time_algo(cfg, "CST", n, dem);
	time_algo(cfg, "CST_LAPSE", n, dem);
	time_algo(cfg, "IDW", n, dem);
	time_algo(cfg, "IDW_LAPSE", n, dem);
	time_algo(cfg, "ODKRIG_LAPSE", n, dem);

	std::cout << "\n";
}

int main(void) {
	Config cfg("io.ini");

	time_dem_size(cfg, 1);
	time_dem_size(cfg, 10);
	time_dem_size(cfg, 100);
	time_dem_size(cfg, 1000);
	time_dem_size(cfg, 2000);
	time_dem_size(cfg, 5000);

	return 0;
}
